# -*- coding: utf-8 -*-
################################################################################
#
#  pyLOCSMS -- Python Interface to LOC Store Management Suite
#  Copyright © 2013-2014 Lance Edgar
#
#  This file is part of pyLOCSMS.
#
#  pyLOCSMS is free software: you can redistribute it and/or modify it under
#  the terms of the GNU General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  pyLOCSMS is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
#  FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
#  details.
#
#  You should have received a copy of the GNU General Public License along with
#  pyLOCSMS.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################

"""
Database Schema

The data models contained here represent the SMS database tables which have
been mapped thus far.  Certainly there are more tables which have not yet been
mapped.

For convenience, here is a list of the models:

* :class:`AccountLink` (``CLL_TAB``)
* :class:`AlternateCode` (``ALT_TAB``)
* :class:`Cost` (``COST_TAB``)
* :class:`Customer` (``CLT_TAB``)
* :class:`CustomerGroup` (``CLG_TAB``)
* :class:`Department` (``DEPT_TAB``)
* :class:`Operator` (``CLK_TAB``)
* :class:`PerpetualInventory` (``RPT_ITM_N``)
* :class:`POS` (``POS_TAB``)
* :class:`Price` (``PRICE_TAB``)
* :class:`Product` (``OBJ_TAB``)
* :class:`Purchase` (``REC_HDR``)
* :class:`PurchaseDetail` (``REC_REG``)
* :class:`ReportCode` (``RPC_TAB``)
* :class:`Sale` (``SAL_HDR``)
* :class:`SaleDetail` (``SAL_REG``)
* :class:`Store` (``STD_TAB``)
* :class:`Subdepartment` (``SDP_TAB``)
* :class:`Vendor` (``VENDOR_TAB``)
"""

import datetime

import sqlalchemy as sa
from sqlalchemy.ext.declarative import declarative_base
from sqlalchemy import Column, String, Integer, Float, Numeric, DateTime, Date
from sqlalchemy import and_
from sqlalchemy.orm import relationship, object_session
from sqlalchemy.orm.collections import attribute_mapped_collection


__all__ = [u'Store', u'Department', u'Subdepartment', u'ReportCode',
           u'Product', u'AlternateCode', u'POS', u'Price', u'Cost', u'PerpetualInventory',
           u'Operator', u'CustomerGroup', u'Customer', u'AccountLink',
           u'Vendor', u'Purchase', u'PurchaseDetail',
           u'Sale', u'SaleDetail']


Base = declarative_base()


def get_datetime(date, time):
    """
    Convert a date and time field pair to a standard ``datetime`` instance.

    :type date: date
    :param date: An internal date field value.

    :type time: string
    :param time: An internal time field value.

    :returns: A ``datetime.datetime`` instance, or ``None``
    """
    if date is None:
        return None
    if time is None:
        time = datetime.time(0) # assume midnight
    else:
        time = datetime.datetime.strptime(time, u'%H%M').time()
    return datetime.datetime.combine(date, time)


class Store(Base):
    """
    Represents a store (``STD_TAB``).
    """
    __tablename__ = u'STD_TAB'

    F1056 = Column(
        String(length=4), primary_key=True, nullable=False,
        doc=u"Terminal store")

    F1530 = Column(
        String(length=20), nullable=True,
        doc=u"Store long number")

    F1531 = Column(
        String(length=50), nullable=True,
        doc=u"Store name")

    F1532 = Column(
        String(length=50), nullable=True,
        doc=u"Store address 1")

    F1533 = Column(
        String(length=50), nullable=True,
        doc=u"Store address 2")

    F1534 = Column(
        String(length=50), nullable=True,
        doc=u"Store city")

    F1535 = Column(
        String(length=15), nullable=True,
        doc=u"Store zip code")

    F1536 = Column(
        String(length=20), nullable=True,
        doc=u"Store phone")

    F1537 = Column(
        String(length=20), nullable=True,
        doc=u"Store fax")

    F1538 = Column(
        String(length=50), nullable=True,
        doc=u"Store email")

    F1539 = Column(
        String(length=6), nullable=True,
        doc=u"Store class")

    F1540 = Column(
        String(length=6), nullable=True,
        doc=u"Store group")

    F1541 = Column(
        String(length=6), nullable=True,
        doc=u"Store location type")

    F1542 = Column(
        String(length=6), nullable=True,
        doc=u"Store region")

    F1543 = Column(
        Integer(), nullable=True,
        doc=u"Store manager")

    F1544 = Column(
        Integer(), nullable=True,
        doc=u"Store interim")

    F1545 = Column(
        Float(), nullable=True,
        doc=u"Store man/hour week1")

    F1546 = Column(
        Date(), nullable=True,
        doc=u"Store opening date")

    F1547 = Column(
        Date(), nullable=True,
        doc=u"Store closing date")

    F1548 = Column(
        String(length=6), nullable=True,
        doc=u"Store type")

    F1549 = Column(
        String(length=15), nullable=True,
        doc=u"Store province")

    F1551 = Column(
        String(length=4), nullable=True,
        doc=u"Store elible")

    F1579 = Column(
        Float(), nullable=True,
        doc=u"Store man/hour week2")

    F1580 = Column(
        Float(), nullable=True,
        doc=u"Store man/hour week3")

    F2688 = Column(
        String(length=20), nullable=True,
        doc=u"Store long number 2")

    F2689 = Column(
        String(length=20), nullable=True,
        doc=u"Accounting prefix")

    F2690 = Column(
        String(length=20), nullable=True,
        doc=u"Accounting prefix 2")

    F2698 = Column(
        Float(), nullable=True,
        doc=u"Backstore capacity")

    F2774 = Column(
        String(length=20), nullable=True,
        doc=u"Bank account")

    F2775 = Column(
        String(length=20), nullable=True,
        doc=u"Bank account #2")

    F2776 = Column(
        String(length=60), nullable=True,
        doc=u"Camera server address")

    F2777 = Column(
        String(length=240), nullable=True,
        doc=u"Camera terminal link")

    def __repr__(self):
        return u"Store(F1056={0})".format(repr(self.F1056)).encode(u'utf_8')

    def __unicode__(self):
        return unicode(self.F1531 or u'')


class Department(Base):
    """
    Represents a department (``DEPT_TAB``).
    """
    __tablename__ = u'DEPT_TAB'

    F03 = Column(
        Integer(), primary_key=True, autoincrement=False, nullable=False,
        doc=u"Department code")

    F49 = Column(
        Float(), nullable=True,
        doc=u"Price margin")

    F78 = Column(
        String(length=1), nullable=True,
        doc=u"Require validation")

    F79 = Column(
        String(length=1), nullable=True,
        doc=u"Food stamp")

    F80 = Column(
        String(length=1), nullable=True,
        doc=u"Trading stamp*")

    F81 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 1")

    F82 = Column(
        String(length=1), nullable=True,
        doc=u"Scalable item")

    F88 = Column(
        String(length=1), nullable=True,
        doc=u"Store coupon")

    F96 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 2")

    F97 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 3")

    F98 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 4")

    F99 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 5")

    F100 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 6")

    F101 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 7")

    F104 = Column(
        String(length=1), nullable=True,
        doc=u"Vendor coupon")

    F108 = Column(
        String(length=1), nullable=True,
        doc=u"Coupon restricted")

    F114 = Column(
        String(length=1), nullable=True,
        doc=u"Allow price override")

    F115 = Column(
        String(length=1), nullable=True,
        doc=u"Not in net sale")

    F121 = Column(
        String(length=1), nullable=True,
        doc=u"Allow manuel weight")

    F124 = Column(
        String(length=1), nullable=True,
        doc=u"Prohibit returns*")

    F150 = Column(
        String(length=1), nullable=True,
        doc=u"Prohibit discount")

    F170 = Column(
        Integer(), nullable=True,
        doc=u"Restriction code")

    F171 = Column(
        Integer(), nullable=True,
        doc=u"Minimum age customer")

    F172 = Column(
        String(length=1), nullable=True,
        doc=u"Prohibit refund*")

    F177 = Column(
        String(length=1), nullable=True,
        doc=u"Not in admissible spending")

    F178 = Column(
        String(length=1), nullable=True,
        doc=u"WIC Eligible")

    F238 = Column(
        String(length=30), nullable=True,
        doc=u"Departement Description")

    F239 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Maximum amount")

    F240 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Minimum amount")

    F241 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Maximum void")

    F242 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Maximum refund")

    F1120 = Column(
        String(length=12), nullable=True,
        doc=u"POS specific flags")

    F1123 = Column(
        Float(), nullable=True,
        doc=u"Known shrink factor")

    F1124 = Column(
        Float(), nullable=True,
        doc=u"Commission rate")

    F1132 = Column(
        String(length=20), nullable=True,
        doc=u"Department group")

    F1139 = Column(
        Integer(), nullable=True,
        doc=u"Minimum age operator")

    F1147 = Column(
        Integer(), nullable=True,
        doc=u"Sequence number")

    F1168 = Column(
        Integer(), nullable=True,
        doc=u"Operator responsible")

    F1256 = Column(
        Float(), nullable=True,
        doc=u"Department discount")

    F1785 = Column(
        String(length=12), nullable=True,
        doc=u"Behavior")

    F1894 = Column(
        String(length=30), nullable=True,
        doc=u"Alt desc Dept")

    F1965 = Column(
        Integer(), nullable=True,
        doc=u"Show priority")

    F1966 = Column(
        String(length=20), nullable=True,
        doc=u"Show filter")

    F1967 = Column(
        String(length=1), nullable=True,
        doc=u"SPARE")

    F2660 = Column(
        String(length=250), nullable=True,
        doc=u"URL")

    def __repr__(self):
        return u"Department(F03={0})".format(repr(self.F03)).encode(u'utf_8')

    def __unicode__(self):
        return unicode(self.F238 or u'')


class Subdepartment(Base):
    """
    Represents a subdepartment (``SDP_TAB``).
    """
    __tablename__ = u'SDP_TAB'

    F04 = Column(
        Integer(), primary_key=True, autoincrement=False, nullable=False,
        doc=u"Sub-Department code")

    F03 = Column(
        Integer(), nullable=True,
        doc=u"Department code")

    F49 = Column(
        Float(), nullable=True,
        doc=u"Price margin")

    F78 = Column(
        String(length=1), nullable=True,
        doc=u"Require validation")

    F79 = Column(
        String(length=1), nullable=True,
        doc=u"Food stamp")

    F80 = Column(
        String(length=1), nullable=True,
        doc=u"Trading stamp*")

    F81 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 1")

    F82 = Column(
        String(length=1), nullable=True,
        doc=u"Scalable item")

    F88 = Column(
        String(length=1), nullable=True,
        doc=u"Store coupon")

    F96 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 2")

    F97 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 3")

    F98 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 4")

    F99 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 5")

    F100 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 6")

    F101 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 7")

    F104 = Column(
        String(length=1), nullable=True,
        doc=u"Vendor coupon")

    F108 = Column(
        String(length=1), nullable=True,
        doc=u"Coupon restricted")

    F114 = Column(
        String(length=1), nullable=True,
        doc=u"Allow price override")

    F115 = Column(
        String(length=1), nullable=True,
        doc=u"Not in net sale")

    F121 = Column(
        String(length=1), nullable=True,
        doc=u"Allow manuel weight")

    F124 = Column(
        String(length=1), nullable=True,
        doc=u"Prohibit returns*")

    F150 = Column(
        String(length=1), nullable=True,
        doc=u"Prohibit discount")

    F170 = Column(
        Integer(), nullable=True,
        doc=u"Restriction code")

    F171 = Column(
        Integer(), nullable=True,
        doc=u"Minimum age customer")

    F172 = Column(
        String(length=1), nullable=True,
        doc=u"Prohibit refund*")

    F177 = Column(
        String(length=1), nullable=True,
        doc=u"Not in admissible spending")

    F178 = Column(
        String(length=1), nullable=True,
        doc=u"WIC Eligible")

    F239 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Maximum amount")

    F240 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Minimum amount")

    F241 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Maximum void")

    F242 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Maximum refund")

    F1022 = Column(
        String(length=30), nullable=True,
        doc=u"Sub-Dept descriptor")

    F1120 = Column(
        String(length=12), nullable=True,
        doc=u"POS specific flags")

    F1123 = Column(
        Float(), nullable=True,
        doc=u"Known shrink factor")

    F1124 = Column(
        Float(), nullable=True,
        doc=u"Commission rate")

    F1139 = Column(
        Integer(), nullable=True,
        doc=u"Minimum age operator")

    F1147 = Column(
        Integer(), nullable=True,
        doc=u"Sequence number")

    F1168 = Column(
        Integer(), nullable=True,
        doc=u"Operator responsible")

    F1256 = Column(
        Float(), nullable=True,
        doc=u"Department discount")

    F1785 = Column(
        String(length=12), nullable=True,
        doc=u"Behavior")

    F1893 = Column(
        String(length=30), nullable=True,
        doc=u"Alt desc SDP")

    F1938 = Column(
        Float(), nullable=True,
        doc=u"Cost plus percent")

    F1965 = Column(
        Integer(), nullable=True,
        doc=u"Show priority")

    F1966 = Column(
        String(length=20), nullable=True,
        doc=u"Show filter")

    F1967 = Column(
        String(length=1), nullable=True,
        doc=u"SPARE")

    F2660 = Column(
        String(length=250), nullable=True,
        doc=u"URL")

    def __repr__(self):
        return u"Subdepartment(F04={0})".format(repr(self.F04)).encode(u'utf_8')

    def __unicode__(self):
        return unicode(self.F1022 or u'')


Department.subdepartments = relationship(
    Subdepartment,
    primaryjoin=Subdepartment.F03 == Department.F03,
    foreign_keys=[Subdepartment.F03],
    uselist=True,
    backref=u'department',
    doc=u"""
List-like collection of :class:`Subdepartment` instances which belong to the
department.
""")


class ReportCode(Base):
    """
    Represents a report code (``RPC_TAB``).
    """
    __tablename__ = u'RPC_TAB'
    __table_args__ = (
        sa.PrimaryKeyConstraint(u'F18', name=u'rpc_tab_pk_f18'),
        )

    F18 = sa.Column(
        sa.Integer(), autoincrement=False, nullable=False,
        doc=u"Report code")

    F49 = sa.Column(
        sa.Float(), nullable=True,
        doc=u"Price margin")

    F1024 = sa.Column(
        sa.String(length=30), nullable=True,
        doc=u"Report code descriptor")

    F1123 = sa.Column(
        sa.Float(), nullable=True,
        doc=u"Known shrink factor")

    F1124 = sa.Column(
        sa.Float(), nullable=True,
        doc=u"Commission rate")

    F1168 = sa.Column(
        sa.Integer(), nullable=True,
        doc=u"Operator responsible")

    F1965 = sa.Column(
        sa.Integer(), nullable=True,
        doc=u"Show priority")

    F1966 = sa.Column(
        sa.String(length=20), nullable=True,
        doc=u"Show filter")

    F1967 = sa.Column(
        sa.String(length=1), nullable=True,
        doc=u"SPARE")

    def __repr__(self):
        return u"ReportCode(F18={0})".format(repr(self.F18)).encode(u'utf_8')

    def __unicode__(self):
        return unicode(self.F1024 or u'')


class Vendor(Base):
    """
    Represents a vendor (``VENDOR_TAB``).
    """
    __tablename__ = u'VENDOR_TAB'

    F27 = Column(
        String(length=14), primary_key=True, nullable=False,
        doc=u"Vendor id")

    F902 = Column(
        String(length=8), nullable=True,
        doc=u"Batch indentifier")

    F1001 = Column(
        Integer(), nullable=True,
        doc=u"Record status")

    F253 = Column(
        Date(), nullable=True,
        doc=u"Date start")

    F334 = Column(
        String(length=40), nullable=True,
        doc=u"Vendor name")

    F335 = Column(
        String(length=40), nullable=True,
        doc=u"Vendor contact name")

    F336 = Column(
        String(length=40), nullable=True,
        doc=u"Vendor address line 1")

    F337 = Column(
        String(length=40), nullable=True,
        doc=u"Vendor address line 2")

    F338 = Column(
        String(length=40), nullable=True,
        doc=u"Vendor address city")

    F339 = Column(
        String(length=15), nullable=True,
        doc=u"Vendor address state")

    F340 = Column(
        String(length=15), nullable=True,
        doc=u"Vendor address zip")

    F341 = Column(
        String(length=20), nullable=True,
        doc=u"Vendor phone voice")

    F342 = Column(
        String(length=15), nullable=True,
        doc=u"Vendor phone fax")

    F343 = Column(
        String(length=30), nullable=True,
        doc=u"Vendor product categories")

    F344 = Column(
        String(length=20), nullable=True,
        doc=u"Freight/broker company")

    F345 = Column(
        String(length=20), nullable=True,
        doc=u"FOB Point")

    F346 = Column(
        Float(), nullable=True,
        doc=u"Vendor limit")

    F347 = Column(
        Float(), nullable=True,
        doc=u"Special discount percent.")

    F348 = Column(
        Integer(), nullable=True,
        doc=u"Terms days")

    F349 = Column(
        Float(), nullable=True,
        doc=u"Terms discount percent")

    F350 = Column(
        String(length=9), nullable=True,
        doc=u"DUNS number")

    F351 = Column(
        String(length=6), nullable=True,
        doc=u"Supplier location number")

    F352 = Column(
        String(length=10), nullable=True,
        doc=u"VendorAccount")

    F353 = Column(
        String(length=1), nullable=True,
        doc=u"Deliver monday flag")

    F354 = Column(
        String(length=1), nullable=True,
        doc=u"Deliver tuesday flag")

    F355 = Column(
        String(length=1), nullable=True,
        doc=u"Deliver wednesday")

    F356 = Column(
        String(length=1), nullable=True,
        doc=u"Deliver thursday")

    F357 = Column(
        String(length=1), nullable=True,
        doc=u"Deliver Friday")

    F358 = Column(
        String(length=1), nullable=True,
        doc=u"Deliver saturday")

    F359 = Column(
        String(length=1), nullable=True,
        doc=u"Deliver sunday")

    F1000 = Column(
        String(length=5), nullable=True,
        doc=u"Target Identifier")

    F1168 = Column(
        Integer(), nullable=True,
        doc=u"Operator responsible")

    F1642 = Column(
        String(length=14), nullable=True,
        doc=u"Ship id number")

    F1654 = Column(
        Float(), nullable=True,
        doc=u"Freight rate weight")

    F1655 = Column(
        Float(), nullable=True,
        doc=u"Freight rate volume")

    F1656 = Column(
        String(length=30), nullable=True,
        doc=u"Vendor term descriptor")

    F1685 = Column(
        Integer(), nullable=True,
        doc=u"Back order days")

    F1759 = Column(
        Integer(), nullable=True,
        doc=u"Maintenance operator level")

    F1760 = Column(
        Float(), nullable=True,
        doc=u"Minimum cases")

    F1761 = Column(
        Float(), nullable=True,
        doc=u"Minimum cubic")

    F1779 = Column(
        String(length=120), nullable=True,
        doc=u"Vendor internal comment")

    F1793 = Column(
        Integer(), nullable=True,
        doc=u"Delivery days")

    F1794 = Column(
        Integer(), nullable=True,
        doc=u"Delivery cycle days")

    F1882 = Column(
        String(length=120), nullable=True,
        doc=u"Vendor comment")

    F1883 = Column(
        Float(), nullable=True,
        doc=u"Minimum weight")

    F1884 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Minimum dollars")

    F1887 = Column(
        String(length=2), nullable=True,
        doc=u"Buying format")

    F1889 = Column(
        String(length=40), nullable=True,
        doc=u"Vendor contact 2")

    F1890 = Column(
        String(length=20), nullable=True,
        doc=u"Vendor contact 2 phone")

    F1891 = Column(
        String(length=15), nullable=True,
        doc=u"Vendor contact 2 fax")

    F1948 = Column(
        String(length=30), nullable=True,
        doc=u"Vendor country")

    F1949 = Column(
        String(length=6), nullable=True,
        doc=u"Vendor status")

    F1964 = Column(
        String(length=4), nullable=True,
        doc=u"Store responsible")

    F1965 = Column(
        Integer(), nullable=True,
        doc=u"Show priority")

    F1966 = Column(
        String(length=20), nullable=True,
        doc=u"Show filter")

    F2597 = Column(
        String(length=20), nullable=True,
        doc=u"Profil")

    F2602 = Column(
        String(length=3), nullable=True,
        doc=u"Currency code")

    F2603 = Column(
        String(length=60), nullable=True,
        doc=u"Vendor email")

    F2630 = Column(
        String(length=4), nullable=True,
        doc=u"SPARE")

    F2631 = Column(
        Integer(), nullable=True,
        doc=u"SPARE")

    F2633 = Column(
        Float(), nullable=True,
        doc=u"SPARE")

    F2635 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"SPARE")

    F2658 = Column(
        String(length=1), nullable=True,
        doc=u"Invisible on PDA")

    F2660 = Column(
        String(length=250), nullable=True,
        doc=u"URL")

    F2666 = Column(
        Float(), nullable=True,
        doc=u"Default quantity")

    F2700 = Column(
        Float(), nullable=True,
        doc=u"Maximum Cases")

    F2701 = Column(
        Float(), nullable=True,
        doc=u"Maximum Cubic")

    F2702 = Column(
        Float(), nullable=True,
        doc=u"Maximum Weight")

    F2710 = Column(
        String(length=10), nullable=True,
        doc=u"Associated tax rates")

    def __repr__(self):
        return u"Vendor(F27={0})".format(repr(self.F27)).encode(u'utf_8')

    def __unicode__(self):
        return unicode(self.F334 or u'')


class Product(Base):
    """
    Represents a product's main item record (``OBJ_TAB``).
    """
    __tablename__ = u'OBJ_TAB'

    F01 = Column(
        String(length=13), primary_key=True, nullable=False,
        doc=u"UPC code")

    F16 = Column(
        Integer(), nullable=True,
        doc=u"Family code")

    F17 = Column(
        Integer(), nullable=True,
        doc=u"Category code")

    F18 = Column(
        Integer(), nullable=True,
        doc=u"Report code")

    F93 = Column(
        String(length=16), nullable=True,
        doc=u"Account code")

    F193 = Column(
        String(length=16), nullable=True,
        doc=u"Reporting departement")

    F902 = Column(
        String(length=8), nullable=True,
        doc=u"Batch indentifier")

    F1001 = Column(
        Integer(), nullable=True,
        doc=u"Record status")

    F07 = Column(
        Integer(), nullable=True,
        doc=u"UPC code format")

    F11 = Column(
        Integer(), nullable=True,
        doc=u"Measurement system")

    F12 = Column(
        Float(), nullable=True,
        doc=u"Size height")

    F13 = Column(
        Float(), nullable=True,
        doc=u"Size width")

    F14 = Column(
        Float(), nullable=True,
        doc=u"Size depth")

    F21 = Column(
        Float(), nullable=True,
        doc=u"Measure sell pack")

    F22 = Column(
        String(length=30), nullable=True,
        doc=u"Size description")

    F23 = Column(
        String(length=10), nullable=True,
        doc=u"Measure description")

    F29 = Column(
        String(length=60), nullable=True,
        doc=u"Expanded description")

    F155 = Column(
        String(length=30), nullable=True,
        doc=u"Brand description")

    F180 = Column(
        String(length=20), nullable=True,
        doc=u"Manufacture code")

    F213 = Column(
        String(length=13), nullable=True,
        doc=u"DUNS number plus suffix")

    F214 = Column(
        String(length=13), nullable=True,
        doc=u"Alias code")

    F215 = Column(
        Integer(), nullable=True,
        doc=u"Alias code format")

    F218 = Column(
        String(length=4), nullable=True,
        doc=u"Comparable size unit of measure description")

    F253 = Column(
        Date(), nullable=True,
        doc=u"Date start")

    F255 = Column(
        String(length=120), nullable=True,
        doc=u"Long description")

    F270 = Column(
        Float(), nullable=True,
        doc=u"Weight net")

    F1000 = Column(
        String(length=5), nullable=True,
        doc=u"Target Identifier")

    F1002 = Column(
        Float(), nullable=True,
        doc=u"Size cubic")

    F1004 = Column(
        Integer(), nullable=True,
        doc=u"Container type")

    F1118 = Column(
        String(length=9), nullable=True,
        doc=u"Manifacture id")

    F1119 = Column(
        String(length=30), nullable=True,
        doc=u"Graphic file")

    F1168 = Column(
        Integer(), nullable=True,
        doc=u"Operator responsible")

    F1699 = Column(
        Float(), nullable=True,
        doc=u"Shipping piece count")

    F1736 = Column(
        Integer(), nullable=True,
        doc=u"Best before days")

    F1737 = Column(
        String(length=13), nullable=True,
        doc=u"Medical product code")

    F1738 = Column(
        String(length=13), nullable=True,
        doc=u"NDC/DIN Number")

    F1744 = Column(
        Float(), nullable=True,
        doc=u"Measure weight or volume")

    F1759 = Column(
        Integer(), nullable=True,
        doc=u"Maintenance operator level")

    F1781 = Column(
        String(length=12), nullable=True,
        doc=u"Spare field OBJ 1")

    F1782 = Column(
        String(length=12), nullable=True,
        doc=u"Spare field OBJ 2")

    F1783 = Column(
        Integer(), nullable=True,
        doc=u"Spare field OBJ 3")

    F1784 = Column(
        Integer(), nullable=True,
        doc=u"Spare field OBJ 4")

    F1939 = Column(
        String(length=30), nullable=True,
        doc=u"Alt Brand desc")

    F1940 = Column(
        String(length=60), nullable=True,
        doc=u"Alt Expended desc")

    F1941 = Column(
        String(length=30), nullable=True,
        doc=u"Alt Size desc")

    F1942 = Column(
        String(length=120), nullable=True,
        doc=u"Alt Long desc")

    F1957 = Column(
        String(length=100), nullable=True,
        doc=u"Classification")

    F1958 = Column(
        String(length=20), nullable=True,
        doc=u"Target customer type")

    F1959 = Column(
        String(length=20), nullable=True,
        doc=u"Target store type")

    F1960 = Column(
        String(length=10), nullable=True,
        doc=u"Handling type")

    F1962 = Column(
        String(length=250), nullable=True,
        doc=u"Marketing justification")

    F1964 = Column(
        String(length=4), nullable=True,
        doc=u"Store responsible")

    F2600 = Column(
        String(length=2), nullable=True,
        doc=u"Item substitution policy")

    F2693 = Column(
        String(length=13), nullable=True,
        doc=u"Competitive code")

    def __repr__(self):
        return u"Product(F01={0})".format(repr(self.F01)).encode(u'utf_8')

    @property
    def subdepartment(self):
        """
        Reference to the product's subdepartment.

        This is for convenience only.  Since the schema does not provide a
        direct relationship for this, the following logic is employed here:

        If the product has a :class:`POS` record with a ``F1000`` value of
        ``'PAL'``, that record's :attr:`POS.subdepartment` reference is used.
        I.e., this is the equivalent of::

           product.pos[u'PAL'].subdepartment

        If the product does *not* have such a record, this will be ``None``.
        """
        if u'PAL' in self.pos:
            return self.pos[u'PAL'].subdepartment
        return None

    @property
    def department(self):
        """
        Reference to the product's department.

        This is for convenience only.  Since the schema does not provide a
        direct relationship for this, the following logic is employed here:

        If the product has a :attr:`subdepartment`, that record's
        :attr:`Subdepartment.department` reference is used.  If not, this will
        be ``None``.
        """
        sub = self.subdepartment
        if sub:
            return sub.department
        return None

    @property
    def cost(self):
        """
        The "authorized" cost for the product.

        This uses the same logic as SMS in determining which of the associated
        :class:`Cost` instances is considered authorized: namely, the one with
        :attr:`Cost.F90` set to either ``u'1'`` or ``NULL``.

        However, as a convenience, it also considers a cost record as
        authorized if the product only *has* one cost record, regardless of its
        :attr:`Cost.F90` value.

        If no cost records exist for the product, or if multiple records exist
        and all are explicitly marked *unauthorized* (``F90 == u'0'``), then
        this will be ``None``.
        """
        for cost in self.costs:
            if cost.F90 == u'1' or cost.F90 is None:
                return cost
        if len(self.costs) == 1:
            return self.costs[0]
        return None
        
    def last_purchase_base_cost(self, F27=None):
        """
        Fetch the last-paid base cost for the product, if available.  This
        calls :func:`locsms.db.api.products.last_purchase_base_cost()` under
        the hood; see that function for more info.
        """
        from locsms.db.api.products import last_purchase_base_cost
        return last_purchase_base_cost(object_session(self), self.F01, F27)


Product.report_code = relationship(
    ReportCode,
    primaryjoin=ReportCode.F18 == Product.F18,
    foreign_keys=[ReportCode.F18],
    doc=u"""\
Reference to a :class:`ReportCode` instance to which the product belongs, if
any.
""")


class AlternateCode(Base):
    """
    Represents an alternate code for a product (ALT_TAB).
    """
    __tablename__ = u'ALT_TAB'

    F154 = Column(
        String(length=20), primary_key=True, nullable=False,
        doc=u"Alternate code")

    F1000 = Column(
        String(length=5), primary_key=True, nullable=False,
        doc=u"Target Identifier")

    F01 = Column(
        String(length=13), nullable=True,
        doc=u"UPC code")

    F902 = Column(
        String(length=8), nullable=True,
        doc=u"Batch indentifier")

    F126 = Column(
        Integer(), nullable=True,
        doc=u"Price level")

    F253 = Column(
        Date(), nullable=True,
        doc=u"Date start")

    F1001 = Column(
        Integer(), nullable=True,
        doc=u"Record status")

    F1759 = Column(
        Integer(), nullable=True,
        doc=u"Maintenance operator level")

    F1874 = Column(
        Integer(), nullable=True,
        doc=u"Select pkg price")

    F1898 = Column(
        String(length=1), nullable=True,
        doc=u"Main alt code")

    F1964 = Column(
        String(length=4), nullable=True,
        doc=u"Store responsible")

    F2637 = Column(
        String(length=1), nullable=True,
        doc=u"Main code by format")

    def __repr__(self):
        return u"AlternateCode(F154={0}, F1000={1})".format(
            repr(self.F154), repr(self.F1000)).encode(u'utf_8')

    def __unicode__(self):
        return unicode(self.F154 or u'')


Product.alternate_codes = relationship(
    AlternateCode,
    primaryjoin=AlternateCode.F01 == Product.F01,
    foreign_keys=[AlternateCode.F01],
    backref=u'product',
    doc=u"""
List-like collection of :class:`AlternateCode` instances associated with the
product.
""")

Product.alternate_code = relationship(
    AlternateCode,
    primaryjoin=and_(
        AlternateCode.F01 == Product.F01,
        AlternateCode.F1000 == u'PAL',
        AlternateCode.F1898 == u'1'),
    foreign_keys=[AlternateCode.F01],
    uselist=False,
    viewonly=True,
    doc=u"""
Main :class:`AlternateCode` instance associated with the product, or ``None``.
""")


class POS(Base):
    """
    Represents various POS flags for a product (``POS_TAB``).
    """
    __tablename__ = u'POS_TAB'

    F01 = Column(
        String(length=13), primary_key=True, nullable=False,
        doc=u"UPC code")

    F1000 = Column(
        String(length=5), primary_key=True, nullable=False,
        doc=u"Target Identifier")

    F04 = Column(
        Integer(), nullable=True,
        doc=u"Sub-Department code")

    F383 = Column(
        String(length=20), nullable=True,
        doc=u"Promotion code")

    F902 = Column(
        String(length=8), nullable=True,
        doc=u"Batch indentifier")

    F1001 = Column(
        Integer(), nullable=True,
        doc=u"Record status")

    F02 = Column(
        String(length=40), nullable=True,
        doc=u"POS description")

    F03 = Column(
        Integer(), nullable=True,
        doc=u"Department code")

    F05 = Column(
        Integer(), nullable=True,
        doc=u"Bottle deposit link")

    F06 = Column(
        Integer(), nullable=True,
        doc=u"Tare weight link")

    F07 = Column(
        Integer(), nullable=True,
        doc=u"UPC code format")

    F08 = Column(
        String(length=4), nullable=True,
        doc=u"Status code")

    F09 = Column(
        Date(), nullable=True,
        doc=u"Status indicator date")

    F24 = Column(
        Float(), nullable=True,
        doc=u"Weight divisor")

    F40 = Column(
        String(length=1), nullable=True,
        doc=u"Item pricing required")

    F50 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Bottle deposit value")

    F60 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Excise tax amount*")

    F61 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Tax exempt amount*")

    F66 = Column(
        Integer(), nullable=True,
        doc=u"Sales activity level")

    F77 = Column(
        Integer(), nullable=True,
        doc=u"Coupon family code")

    F78 = Column(
        String(length=1), nullable=True,
        doc=u"Require validation")

    F79 = Column(
        String(length=1), nullable=True,
        doc=u"Food stamp")

    F80 = Column(
        String(length=1), nullable=True,
        doc=u"Trading stamp*")

    F81 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 1")

    F82 = Column(
        String(length=1), nullable=True,
        doc=u"Scalable item")

    F83 = Column(
        String(length=1), nullable=True,
        doc=u"Require price entry")

    F84 = Column(
        String(length=1), nullable=True,
        doc=u"Require visual verify")

    F85 = Column(
        String(length=1), nullable=True,
        doc=u"Require quantity")

    F86 = Column(
        String(length=1), nullable=True,
        doc=u"Not for sale in store")

    F87 = Column(
        String(length=1), nullable=True,
        doc=u"Restrited sale")

    F88 = Column(
        String(length=1), nullable=True,
        doc=u"Store coupon")

    F92 = Column(
        String(length=1), nullable=True,
        doc=u"Deposit container code")

    F96 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 2")

    F97 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 3")

    F98 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 4")

    F99 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 5")

    F100 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 6")

    F101 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 7")

    F102 = Column(
        String(length=1), nullable=True,
        doc=u"Prohibit quantity")

    F104 = Column(
        String(length=1), nullable=True,
        doc=u"Vendor coupon")

    F106 = Column(
        String(length=1), nullable=True,
        doc=u"Follow subdept status")

    F107 = Column(
        String(length=1), nullable=True,
        doc=u"Record item sale")

    F108 = Column(
        String(length=1), nullable=True,
        doc=u"Coupon restricted")

    F110 = Column(
        String(length=1), nullable=True,
        doc=u"Electronic coupon")

    F114 = Column(
        String(length=1), nullable=True,
        doc=u"Allow price override")

    F115 = Column(
        String(length=1), nullable=True,
        doc=u"Not in net sale")

    F121 = Column(
        String(length=1), nullable=True,
        doc=u"Allow manuel weight")

    F123 = Column(
        String(length=13), nullable=True,
        doc=u"PLU code")

    F124 = Column(
        String(length=1), nullable=True,
        doc=u"Prohibit returns*")

    F125 = Column(
        String(length=1), nullable=True,
        doc=u"Prohibit repeat key*")

    F141 = Column(
        Float(), nullable=True,
        doc=u"Link quantity limit")

    F149 = Column(
        String(length=1), nullable=True,
        doc=u"Coupon multiplication")

    F150 = Column(
        String(length=1), nullable=True,
        doc=u"Prohibit discount")

    F153 = Column(
        Integer(), nullable=True,
        doc=u"Link reason code")

    F158 = Column(
        String(length=1), nullable=True,
        doc=u"Deposit item")

    F159 = Column(
        String(length=1), nullable=True,
        doc=u"Refund item")

    F160 = Column(
        String(length=1), nullable=True,
        doc=u"Bottle return")

    F161 = Column(
        String(length=1), nullable=True,
        doc=u"Misc. receipt")

    F162 = Column(
        String(length=1), nullable=True,
        doc=u"Misc. payout")

    F163 = Column(
        Integer(), nullable=True,
        doc=u"Coupon family code (previous)")

    F170 = Column(
        Integer(), nullable=True,
        doc=u"Restriction code")

    F171 = Column(
        Integer(), nullable=True,
        doc=u"Minimum age customer")

    F172 = Column(
        String(length=1), nullable=True,
        doc=u"Prohibit refund*")

    F173 = Column(
        String(length=1), nullable=True,
        doc=u"Prohibit multiple coupon")

    F174 = Column(
        String(length=1), nullable=True,
        doc=u"Tax included")

    F176 = Column(
        String(length=1), nullable=True,
        doc=u"Keyed department override")

    F178 = Column(
        String(length=1), nullable=True,
        doc=u"WIC Eligible")

    F188 = Column(
        String(length=1), nullable=True,
        doc=u"POS Valid item")

    F189 = Column(
        String(length=1), nullable=True,
        doc=u"Send to scale")

    F209 = Column(
        Float(), nullable=True,
        doc=u"Competitive price qty")

    F210 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Competitive price")

    F211 = Column(
        Date(), nullable=True,
        doc=u"Competitive price start date")

    F217 = Column(
        Float(), nullable=True,
        doc=u"Comparable size")

    F253 = Column(
        Date(), nullable=True,
        doc=u"Date start")

    F302 = Column(
        String(length=8), nullable=True,
        doc=u"Coupon offer code")

    F303 = Column(
        Date(), nullable=True,
        doc=u"Coupon expiration date")

    F304 = Column(
        String(length=8), nullable=True,
        doc=u"Coupon Household ID")

    F306 = Column(
        Float(), nullable=True,
        doc=u"Coupon redemption multiple")

    F388 = Column(
        String(length=1), nullable=True,
        doc=u"Price multiple as quantity")

    F397 = Column(
        String(length=1), nullable=True,
        doc=u"Prompt for SKU")

    F1099 = Column(
        Integer(), nullable=True,
        doc=u"Rent able quantity")

    F1120 = Column(
        String(length=12), nullable=True,
        doc=u"POS specific flags")

    F1124 = Column(
        Float(), nullable=True,
        doc=u"Commission rate")

    F1136 = Column(
        String(length=8), nullable=True,
        doc=u"Ticket template")

    F1138 = Column(
        Float(), nullable=True,
        doc=u"Tare weight proportional")

    F1139 = Column(
        Integer(), nullable=True,
        doc=u"Minimum age operator")

    F1236 = Column(
        String(length=1), nullable=True,
        doc=u"Allow raincheck ticket")

    F1237 = Column(
        Float(), nullable=True,
        doc=u"Scale divisor see F24")

    F1735 = Column(
        String(length=4), nullable=True,
        doc=u"Controlled product indicator")

    F1759 = Column(
        Integer(), nullable=True,
        doc=u"Maintenance operator level")

    F1785 = Column(
        String(length=12), nullable=True,
        doc=u"Behavior")

    F1786 = Column(
        String(length=12), nullable=True,
        doc=u"SPARE")

    F1787 = Column(
        Integer(), nullable=True,
        doc=u"Replace adding function")

    F1788 = Column(
        Integer(), nullable=True,
        doc=u"Receipt copy count")

    F1789 = Column(
        Float(), nullable=True,
        doc=u"Store coupon count")

    F1790 = Column(
        Float(), nullable=True,
        doc=u"SPARE")

    F1892 = Column(
        String(length=40), nullable=True,
        doc=u"Alt desc POS")

    F1926 = Column(
        Float(), nullable=True,
        doc=u"SPARE")

    F1929 = Column(
        Integer(), nullable=True,
        doc=u"SPARE")

    F1933 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"SPARE")

    F1964 = Column(
        String(length=4), nullable=True,
        doc=u"Store responsible")

    F2608 = Column(
        String(length=4), nullable=True,
        doc=u"SPARE")

    F2660 = Column(
        String(length=250), nullable=True,
        doc=u"URL")

    def __repr__(self):
        return u"POS(F01={0}, F1000={1})".format(
            repr(self.F01), repr(self.F1000)).encode(u'utf_8')


POS.subdepartment = relationship(
    Subdepartment,
    primaryjoin=Subdepartment.F04 == POS.F04,
    foreign_keys=[Subdepartment.F04],
    uselist=False,
    doc=u"""
Reference to the :class:`Subdepartment` to which the product belongs.

Note that the relationship between a product and a subdepartment only exists by
way of the ``POS`` record.  Therefore the product does not (necessarily)
"exclusively" belong to any particular subdepartment.  However the "default"
subdepartment may be had via the :attr:`Product.subdepartment` attribute.
""")


Product.pos = relationship(
    POS,
    primaryjoin=POS.F01 == Product.F01,
    foreign_keys=[POS.F01],
    uselist=True,
    collection_class=attribute_mapped_collection('F1000'),
    backref=u'product',
    doc=u"""
Dictionary-like collection of :class:`POS` records, keyed by :attr:`POS.F1000`.

Typical access will be something like::

   pos = product.pos[u'PAL']
""")


class Price(Base):
    """
    Represents a price for a product (``PRICE_TAB``).
    """

    __tablename__ = u'PRICE_TAB'

    F01 = Column(
        String(length=13), primary_key=True, nullable=False,
        doc=u"UPC code")

    F1000 = Column(
        String(length=5), primary_key=True, nullable=False,
        doc=u"Target Identifier")

    F126 = Column(
        Integer(), primary_key=True, autoincrement=False, nullable=False,
        doc=u"Price level")

    F32 = Column(
        Integer(), nullable=True,
        doc=u"Price mix/match")

    F902 = Column(
        String(length=8), nullable=True,
        doc=u"Batch indentifier")

    F1001 = Column(
        Integer(), nullable=True,
        doc=u"Record status")

    F1014 = Column(
        Date(), nullable=True,
        doc=u"Next price changed date")

    F07 = Column(
        Integer(), nullable=True,
        doc=u"UPC code format")

    F21 = Column(
        Float(), nullable=True,
        doc=u"Measure sell pack")

    F30 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Price")

    F31 = Column(
        Float(), nullable=True,
        doc=u"Price qty")

    F33 = Column(
        String(length=8), nullable=True,
        doc=u"Price method")

    F34 = Column(
        String(length=2), nullable=True,
        doc=u"Change price reason")

    F35 = Column(
        Date(), nullable=True,
        doc=u"Price start date")

    F36 = Column(
        String(length=6), nullable=True,
        doc=u"Price start time")

    F37 = Column(
        String(length=1), nullable=True,
        doc=u"Change price originator")

    F41 = Column(
        String(length=1), nullable=True,
        doc=u"Price audit")

    F42 = Column(
        String(length=1), nullable=True,
        doc=u"Ad flag")

    F43 = Column(
        String(length=1), nullable=True,
        doc=u"Apply item discount*")

    F49 = Column(
        Float(), nullable=True,
        doc=u"Price margin")

    F59 = Column(
        String(length=1), nullable=True,
        doc=u"Instore over regular price")

    F62 = Column(
        Float(), nullable=True,
        doc=u"Price limited qty")

    F63 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Price limited")

    F109 = Column(
        String(length=1), nullable=True,
        doc=u"Discount print/detailtrack")

    F111 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Price discount amount")

    F112 = Column(
        Float(), nullable=True,
        doc=u"Price discount percent")

    F113 = Column(
        String(length=8), nullable=True,
        doc=u"Active price description")

    F119 = Column(
        String(length=1), nullable=True,
        doc=u"Hold retail price")

    F122 = Column(
        String(length=13), nullable=True,
        doc=u"Like code")

    F129 = Column(
        Date(), nullable=True,
        doc=u"Price end date")

    F130 = Column(
        String(length=6), nullable=True,
        doc=u"Price end time")

    F133 = Column(
        Date(), nullable=True,
        doc=u"Target gross margin start date")

    F135 = Column(
        Float(), nullable=True,
        doc=u"Sale price qty")

    F136 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Sale price")

    F137 = Column(
        Date(), nullable=True,
        doc=u"Sale price start date")

    F138 = Column(
        Date(), nullable=True,
        doc=u"Sale price end date")

    F139 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Sale package price")

    F140 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Price package")

    F142 = Column(
        Float(), nullable=True,
        doc=u"Price package qty")

    F143 = Column(
        Float(), nullable=True,
        doc=u"Sale package qty")

    F144 = Column(
        String(length=6), nullable=True,
        doc=u"Sale price start time")

    F145 = Column(
        String(length=6), nullable=True,
        doc=u"Sale price end time")

    F146 = Column(
        String(length=8), nullable=True,
        doc=u"Sale price method")

    F147 = Column(
        Float(), nullable=True,
        doc=u"Sale limited price qty")

    F148 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Sale limited price")

    F164 = Column(
        String(length=13), nullable=True,
        doc=u"Item link code")

    F166 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Alternate price")

    F167 = Column(
        Float(), nullable=True,
        doc=u"Alternate price quantity")

    F168 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Suggested price")

    F169 = Column(
        Integer(), nullable=True,
        doc=u"Price shopper points")

    F175 = Column(
        String(length=1), nullable=True,
        doc=u"Prohibit item markdown*")

    F177 = Column(
        String(length=1), nullable=True,
        doc=u"Not in admissible spending")

    F179 = Column(
        String(length=8), nullable=True,
        doc=u"Flyer impact")

    F181 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"TPR price")

    F182 = Column(
        Float(), nullable=True,
        doc=u"TPR price qty")

    F183 = Column(
        Date(), nullable=True,
        doc=u"TPR start date")

    F184 = Column(
        Date(), nullable=True,
        doc=u"TPR end date")

    F205 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Target dollar markup")

    F221 = Column(
        String(length=8), nullable=True,
        doc=u"TPR Price method")

    F232 = Column(
        Float(), nullable=True,
        doc=u"Override number")

    F253 = Column(
        Date(), nullable=True,
        doc=u"Date start")

    F903 = Column(
        String(length=16), nullable=True,
        doc=u"Batch creator")

    F1005 = Column(
        String(length=4), nullable=True,
        doc=u"Price rounding method")

    F1006 = Column(
        Float(), nullable=True,
        doc=u"Active price quantity")

    F1007 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Active price")

    F1008 = Column(
        Date(), nullable=True,
        doc=u"Active price changed date")

    F1009 = Column(
        String(length=6), nullable=True,
        doc=u"Active price changed time")

    F1010 = Column(
        String(length=8), nullable=True,
        doc=u"Next price method")

    F1011 = Column(
        String(length=8), nullable=True,
        doc=u"Next price source")

    F1012 = Column(
        Float(), nullable=True,
        doc=u"Next price quantity")

    F1013 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Next price")

    F1015 = Column(
        String(length=6), nullable=True,
        doc=u"Next price changed time")

    F1133 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Instore price")

    F1134 = Column(
        Float(), nullable=True,
        doc=u"Instore price qty")

    F1135 = Column(
        String(length=1), nullable=True,
        doc=u"Auto calculate store price*")

    F1186 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"TPR package price")

    F1187 = Column(
        Float(), nullable=True,
        doc=u"TPR package price qty")

    F1188 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"TPR Limited qty price")

    F1189 = Column(
        Float(), nullable=True,
        doc=u"TPR Limited qty")

    F1190 = Column(
        Integer(), nullable=True,
        doc=u"TPR Mix & match")

    F1191 = Column(
        Integer(), nullable=True,
        doc=u"TPR shopper points")

    F1192 = Column(
        Integer(), nullable=True,
        doc=u"Sale shopper points")

    F1193 = Column(
        Integer(), nullable=True,
        doc=u"Sale mix&match")

    F1194 = Column(
        String(length=1), nullable=True,
        doc=u"Instore over TPR")

    F1195 = Column(
        String(length=1), nullable=True,
        doc=u"Instore over Sale")

    F1196 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Instore package price")

    F1197 = Column(
        Float(), nullable=True,
        doc=u"Instore package price qty")

    F1198 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Instore limited qty price")

    F1199 = Column(
        Float(), nullable=True,
        doc=u"Instore limited qty")

    F1200 = Column(
        Integer(), nullable=True,
        doc=u"Instore mix&match")

    F1201 = Column(
        String(length=8), nullable=True,
        doc=u"Instore price method")

    F1202 = Column(
        Integer(), nullable=True,
        doc=u"Instore shopper points")

    F1203 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Active package price")

    F1204 = Column(
        Float(), nullable=True,
        doc=u"Active package qty")

    F1205 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Active limited price")

    F1206 = Column(
        Float(), nullable=True,
        doc=u"Active limited qty")

    F1207 = Column(
        Integer(), nullable=True,
        doc=u"Active mix&match")

    F1208 = Column(
        String(length=8), nullable=True,
        doc=u"Active price method")

    F1209 = Column(
        Integer(), nullable=True,
        doc=u"Active shopper points")

    F1210 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Next package price")

    F1211 = Column(
        Float(), nullable=True,
        doc=u"Next pachage price qty")

    F1212 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Next limited qty price")

    F1213 = Column(
        Float(), nullable=True,
        doc=u"Next limited qty")

    F1214 = Column(
        Integer(), nullable=True,
        doc=u"Next mix&match")

    F1215 = Column(
        Integer(), nullable=True,
        doc=u"Next shopper points")

    F1216 = Column(
        Date(), nullable=True,
        doc=u"Instore start date")

    F1217 = Column(
        Date(), nullable=True,
        doc=u"Instore stop date")

    F1218 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"TPR discount amount")

    F1219 = Column(
        Float(), nullable=True,
        doc=u"TPR discount percent")

    F1220 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Sale discount amount")

    F1221 = Column(
        Float(), nullable=True,
        doc=u"Sale discount percent")

    F1222 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Instore discount amount")

    F1223 = Column(
        Float(), nullable=True,
        doc=u"Instore discount percent")

    F1224 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Active discount amount")

    F1225 = Column(
        Float(), nullable=True,
        doc=u"Active discount percent")

    F1226 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Next discount amount")

    F1227 = Column(
        Float(), nullable=True,
        doc=u"Next discount percent")

    F1228 = Column(
        Integer(), nullable=True,
        doc=u"Redeem discount points")

    F1229 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Redeem discount amount")

    F1230 = Column(
        Integer(), nullable=True,
        doc=u"SPARE")

    F1231 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Last label regular price")

    F1232 = Column(
        Integer(), nullable=True,
        doc=u"Redeem free points")

    F1233 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"SPARE")

    F1234 = Column(
        Date(), nullable=True,
        doc=u"Redeem start date")

    F1235 = Column(
        Date(), nullable=True,
        doc=u"Redeem end date")

    F1713 = Column(
        Float(), nullable=True,
        doc=u"Suggested price margin")

    F1714 = Column(
        String(length=4), nullable=True,
        doc=u"Suggested price rounding")

    F1759 = Column(
        Integer(), nullable=True,
        doc=u"Maintenance operator level")

    F1767 = Column(
        Float(), nullable=True,
        doc=u"Price package margin")

    F1768 = Column(
        String(length=4), nullable=True,
        doc=u"Price package rounding")

    F1769 = Column(
        Float(), nullable=True,
        doc=u"Price limited margin")

    F1770 = Column(
        String(length=4), nullable=True,
        doc=u"Price limited rounding")

    F1799 = Column(
        String(length=12), nullable=True,
        doc=u"Spare field PRI 1")

    F1800 = Column(
        String(length=12), nullable=True,
        doc=u"Spare field PRI 2")

    F1801 = Column(
        Integer(), nullable=True,
        doc=u"Flyer page")

    F1802 = Column(
        Integer(), nullable=True,
        doc=u"Adding totalizer/function")

    F1803 = Column(
        Float(), nullable=True,
        doc=u"Time included")

    F1804 = Column(
        Float(), nullable=True,
        doc=u"Spare field PRI 6")

    F1805 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Label price")

    F1806 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Normalized price")

    F1885 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Commission dollars")

    F1927 = Column(
        Float(), nullable=True,
        doc=u"SPARE")

    F1928 = Column(
        Float(), nullable=True,
        doc=u"SPARE")

    F1930 = Column(
        Integer(), nullable=True,
        doc=u"SPARE")

    F1934 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"SPARE")

    F1935 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"SPARE")

    F1964 = Column(
        String(length=4), nullable=True,
        doc=u"Store responsible")

    F2569 = Column(
        String(length=8), nullable=True,
        doc=u"Label price method")

    F2570 = Column(
        Float(), nullable=True,
        doc=u"Label price quantity")

    F2571 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Label package price")

    F2572 = Column(
        Float(), nullable=True,
        doc=u"Label package price qty")

    F2573 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Label limited qty price")

    F2574 = Column(
        Float(), nullable=True,
        doc=u"Label limited qty")

    F2575 = Column(
        Integer(), nullable=True,
        doc=u"Label mix&match")

    F2576 = Column(
        Integer(), nullable=True,
        doc=u"Label shopper points")

    F2577 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Label discount amount")

    F2578 = Column(
        Float(), nullable=True,
        doc=u"Label discount percent")

    F2579 = Column(
        String(length=8), nullable=True,
        doc=u"Label price source")

    F2580 = Column(
        Integer(), nullable=True,
        doc=u"Label quantity")

    F2667 = Column(
        String(length=12), nullable=True,
        doc=u"TPR number")

    F2668 = Column(
        String(length=12), nullable=True,
        doc=u"Sale number")

    F2694 = Column(
        String(length=12), nullable=True,
        doc=u"Price 1 description")

    F2695 = Column(
        String(length=12), nullable=True,
        doc=u"Price 2 description")

    F2696 = Column(
        String(length=12), nullable=True,
        doc=u"Price 3 description")

    F2744 = Column(
        String(length=4), nullable=True,
        doc=u"Point redeem program")

    def __repr__(self):
        return u"Price(F01={0}, F1000={1}, F126={2})".format(
            repr(self.F01), repr(self.F1000), repr(self.F126)).encode(u'utf_8')

    @property
    def regular_start(self):
        """
        The regular price start time as a standard ``datetime`` instance, or ``None``.
        """
        return get_datetime(self.F35, self.F36)

    @property
    def regular_stop(self):
        """
        The regular price end time as a standard ``datetime`` instance, or ``None``.
        """
        return get_datetime(self.F129, self.F130)

    @property
    def tpr_start(self):
        """
        The TPR price start time as a standard ``datetime`` instance, or ``None``.
        """
        return get_datetime(self.F183, None)

    @property
    def tpr_stop(self):
        """
        The TPR price end time as a standard ``datetime`` instance, or ``None``.
        """
        return get_datetime(self.F184, None)

    @property
    def sale_start(self):
        """
        The sale price start time as a standard ``datetime`` instance, or ``None``.
        """
        return get_datetime(self.F137, self.F144)

    @property
    def sale_stop(self):
        """
        The sale price end time as a standard ``datetime`` instance, or ``None``.
        """
        return get_datetime(self.F138, self.F145)

    def guess_active_price(self, now=None):
        """
        Calculates what "ought" to be the current active price value.

        This method was added in order to provide a means for double-checking
        the active price (F1007) value within SMS.  Under normal circumstances
        that field value should be preferred to whatever this returns.

        .. warning::
           In particular, the current implementation here is *very* basic, and
           does not even inspect the TPR or Store price values.  It is
           therefore only useful for determining if the Sale or Regular price
           should be considered active.
        """
        if not now:
            now = datetime.datetime.now()

        # Sale
        sale_start = self.sale_start
        sale_stop = self.sale_stop
        if sale_start and sale_stop and sale_start <= now <= sale_stop:
            return self.F136

        # Regular
        return self.F30


Product.prices = relationship(
    Price,
    primaryjoin=Price.F01 == Product.F01,
    foreign_keys=[Price.F01],
    backref=u'product',
    doc=u"""
List-like collection of :class:`Price` instances associated with the product.
""")


Product.price = relationship(
    Price,
    primaryjoin=and_(
        Price.F01 == Product.F01,
        Price.F1000 == u'PAL',
        Price.F126 == 1,
        ),
    foreign_keys=[Price.F01],
    uselist=False,
    doc=u"""
Reference to the :class:`Price` instance which could be considered "primary".

If it exists, this price will have a :attr:`Price.F1000` value of ``u'PAL'``
and a :attr:`Price.F126` value of ``1``.  However it is possible for no such
price to exist, in which case this will be ``None``.
""")


class Cost(Base):
    """
    Represents a cost record for a product (COST_TAB).
    """
    __tablename__ = u'COST_TAB'

    F01 = Column(
        String(length=13), primary_key=True, nullable=False,
        doc=u"UPC code")

    F1000 = Column(
        String(length=5), primary_key=True, nullable=False,
        doc=u"Target Identifier")

    F27 = Column(
        String(length=14), primary_key=True, nullable=False,
        doc=u"Vendor id")

    F1184 = Column(
        String(length=12), primary_key=True, nullable=False,
        doc=u"Buying format")

    F26 = Column(
        String(length=15), nullable=True,
        doc=u"Vendor code")

    F902 = Column(
        String(length=8), nullable=True,
        doc=u"Batch indentifier")

    F1001 = Column(
        Integer(), nullable=True,
        doc=u"Record status")

    F07 = Column(
        Integer(), nullable=True,
        doc=u"UPC code format")

    F08 = Column(
        String(length=4), nullable=True,
        doc=u"Status code")

    F19 = Column(
        Float(), nullable=True,
        doc=u"Case size")

    F20 = Column(
        Float(), nullable=True,
        doc=u"Receiving pack size")

    F28 = Column(
        String(length=13), nullable=True,
        doc=u"Case UPC code")

    F38 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Base cost")

    F39 = Column(
        Date(), nullable=True,
        doc=u"Base cost date")

    F90 = Column(
        String(length=1), nullable=True,
        doc=u"Vendor authorized item")

    F120 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Split case cost")

    F127 = Column(
        String(length=15), nullable=True,
        doc=u"Alternate item code")

    F131 = Column(
        Integer(), nullable=True,
        doc=u"Sequence number")

    F134 = Column(
        Date(), nullable=True,
        doc=u"Split case cost start date")

    F151 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Next cost net")

    F152 = Column(
        Date(), nullable=True,
        doc=u"Next cost start date")

    F156 = Column(
        String(length=120), nullable=True,
        doc=u"Allowance qualifier")

    F165 = Column(
        String(length=8), nullable=True,
        doc=u"Vendor broker number")

    F185 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Half case cost")

    F194 = Column(
        Float(), nullable=True,
        doc=u"Last receiving pack size")

    F195 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Last base cost")

    F196 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Case net cost")

    F201 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Allowance amount")

    F202 = Column(
        Date(), nullable=True,
        doc=u"Allowance start date")

    F203 = Column(
        Date(), nullable=True,
        doc=u"Allowance end date")

    F204 = Column(
        String(length=1), nullable=True,
        doc=u"Allowance code")

    F212 = Column(
        String(length=6), nullable=True,
        doc=u"Base cost time")

    F216 = Column(
        Date(), nullable=True,
        doc=u"Availability date")

    F219 = Column(
        Date(), nullable=True,
        doc=u"Cost break end date")

    F220 = Column(
        String(length=13), nullable=True,
        doc=u"Split item code")

    F222 = Column(
        String(length=12), nullable=True,
        doc=u"Allowance number")

    F223 = Column(
        String(length=1), nullable=True,
        doc=u"Allowance off invoice")

    F224 = Column(
        Float(), nullable=True,
        doc=u"Cost break min qty")

    F225 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Cost break amount")

    F226 = Column(
        Float(), nullable=True,
        doc=u"Cost break percent")

    F227 = Column(
        Date(), nullable=True,
        doc=u"Cost break start date")

    F228 = Column(
        Float(), nullable=True,
        doc=u"Discount percent")

    F229 = Column(
        Date(), nullable=True,
        doc=u"Discount start date")

    F230 = Column(
        Date(), nullable=True,
        doc=u"Discount end date")

    F231 = Column(
        String(length=12), nullable=True,
        doc=u"Discount number")

    F233 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Rebate amount")

    F234 = Column(
        Date(), nullable=True,
        doc=u"Rebate start date")

    F235 = Column(
        Date(), nullable=True,
        doc=u"Rebate end date")

    F236 = Column(
        String(length=12), nullable=True,
        doc=u"Rebate number")

    F237 = Column(
        Float(), nullable=True,
        doc=u"Rebate percent")

    F253 = Column(
        Date(), nullable=True,
        doc=u"Date start")

    F325 = Column(
        Float(), nullable=True,
        doc=u"Pallet cases")

    F326 = Column(
        Float(), nullable=True,
        doc=u"Pallet layers")

    F327 = Column(
        Float(), nullable=True,
        doc=u"Freight case weight")

    F328 = Column(
        Float(), nullable=True,
        doc=u"Freight case cube")

    F329 = Column(
        Float(), nullable=True,
        doc=u"Allowance max qty")

    F1037 = Column(
        String(length=1), nullable=True,
        doc=u"Active vendor")

    F1038 = Column(
        String(length=8), nullable=True,
        doc=u"Next cost event")

    F1121 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Cost fee amount")

    F1122 = Column(
        Float(), nullable=True,
        doc=u"Cost fee percent")

    F1140 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Unit net cost")

    F1168 = Column(
        Integer(), nullable=True,
        doc=u"Operator responsible")

    F1657 = Column(
        Float(), nullable=True,
        doc=u"Allowance percent")

    F1658 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Discount amount")

    F1659 = Column(
        String(length=1), nullable=True,
        doc=u"Discount off invoice")

    F1660 = Column(
        String(length=1), nullable=True,
        doc=u"Rebate off invoice")

    F1661 = Column(
        String(length=1), nullable=True,
        doc=u"Cost break off invoice")

    F1662 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"BillBack1 Amount")

    F1663 = Column(
        Float(), nullable=True,
        doc=u"BillBack1 Percent")

    F1664 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"BillBack2 Amount")

    F1665 = Column(
        Float(), nullable=True,
        doc=u"BillBack2 Percent")

    F1666 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"BillBack3 Amount")

    F1667 = Column(
        Float(), nullable=True,
        doc=u"BillBack3 Percent")

    F1668 = Column(
        String(length=16), nullable=True,
        doc=u"BillBack4 Descriptor")

    F1669 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"BillBack4 Amount")

    F1670 = Column(
        Float(), nullable=True,
        doc=u"BillBack4 Percent")

    F1685 = Column(
        Integer(), nullable=True,
        doc=u"Back order days")

    F1759 = Column(
        Integer(), nullable=True,
        doc=u"Maintenance operator level")

    F1760 = Column(
        Float(), nullable=True,
        doc=u"Minimum cases")

    F1761 = Column(
        Float(), nullable=True,
        doc=u"Minimum cubic")

    F1766 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Return fees")

    F1791 = Column(
        String(length=12), nullable=True,
        doc=u"Cost method")

    F1792 = Column(
        String(length=12), nullable=True,
        doc=u"Spare field COS 2")

    F1793 = Column(
        Integer(), nullable=True,
        doc=u"Delivery days")

    F1794 = Column(
        Integer(), nullable=True,
        doc=u"Delivery cycle days")

    F1795 = Column(
        Float(), nullable=True,
        doc=u"Split item qty")

    F1796 = Column(
        Float(), nullable=True,
        doc=u"Last average cost")

    F1797 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Spare COS/RECREG")

    F1798 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Cost for supplier")

    F1875 = Column(
        String(length=2), nullable=True,
        doc=u"Inventory mode")

    F1887 = Column(
        String(length=2), nullable=True,
        doc=u"Buying format")

    F1961 = Column(
        String(length=2), nullable=True,
        doc=u"Return policy")

    F1964 = Column(
        String(length=4), nullable=True,
        doc=u"Store responsible")

    F1973 = Column(
        String(length=120), nullable=True,
        doc=u"Discount qualifier")

    F1974 = Column(
        String(length=120), nullable=True,
        doc=u"Rebate qualifer")

    F1975 = Column(
        String(length=120), nullable=True,
        doc=u"Cost break qualifier")

    F1976 = Column(
        String(length=120), nullable=True,
        doc=u"Cost comment")

    F1977 = Column(
        Float(), nullable=True,
        doc=u"Allowance min qty")

    F1978 = Column(
        Float(), nullable=True,
        doc=u"Discount min qty")

    F1979 = Column(
        Float(), nullable=True,
        doc=u"Rebate min qty")

    F2566 = Column(
        Float(), nullable=True,
        doc=u"Discount max qty")

    F2567 = Column(
        Float(), nullable=True,
        doc=u"Rebate max qty")

    F2568 = Column(
        Float(), nullable=True,
        doc=u"Cost break max qty")

    F2601 = Column(
        String(length=2), nullable=True,
        doc=u"Vendor substitution policy")

    F2624 = Column(
        Integer(), nullable=True,
        doc=u"SPARE")

    F2626 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"SPARE")

    F2628 = Column(
        Float(), nullable=True,
        doc=u"SPARE")

    F2666 = Column(
        Float(), nullable=True,
        doc=u"Default quantity")

    F2699 = Column(
        Date(), nullable=True,
        doc=u"Cost method date")

    def __repr__(self):
        return u"Cost(F01={0}, F1000={1}, F27={2}, F1184={3})".format(
            repr(self.F01), repr(self.F1000), repr(self.F27), repr(self.F1184)).encode(u'utf_8')


Cost.vendor = relationship(
    Vendor,
    primaryjoin=Vendor.F27 == Cost.F27,
    foreign_keys=[Vendor.F27],
    uselist=False,
    doc=u"""
Reference to the :class:`Vendor` for which the cost is valid.
""")


Product.costs = relationship(
    Cost,
    primaryjoin=Cost.F01 == Product.F01,
    foreign_keys=[Cost.F01],
    backref=u'product',
    doc=u"""
List-like collection of :class:`Cost` instances associated with the product.
""")


class PerpetualInventory(Base):
    """
    Represents the perpetual inventory (running total) for a product (``RPT_ITM_N``).
    """
    __tablename__ = u'RPT_ITM_N'

    F01 = Column(
        String(length=13), primary_key=True, nullable=False,
        doc=u"UPC code")

    F1034 = Column(
        Integer(), primary_key=True, autoincrement=False, nullable=False,
        doc=u"Totalizer number")

    F1056 = Column(
        String(length=4), primary_key=True, nullable=False,
        doc=u"Terminal store")

    F64 = Column(
        Float(), nullable=False,
        doc=u"Total units")

    F65 = Column(
        Numeric(precision=8, scale=4), nullable=False,
        doc=u"Total dollars")

    F67 = Column(
        Float(), nullable=False,
        doc=u"Total weight")

    F253 = Column(
        Date(), nullable=False,
        doc=u"Date start")

    F381 = Column(
        Date(), nullable=True,
        doc=u"Movement reset date")

    F1301 = Column(
        Float(), nullable=True,
        doc=u"Soft field 1")

    F1302 = Column(
        Float(), nullable=True,
        doc=u"Soft field 2")

    def __repr__(self):
        return u"PerpetualInventory(F01={0}, F1056={1})".format(
            repr(self.F01), repr(self.F1056)).encode(u'utf_8')


PerpetualInventory.product = relationship(
    Product,
    primaryjoin=Product.F01 == PerpetualInventory.F01,
    foreign_keys=[Product.F01],
    uselist=False,
    doc=u"""
Reference to the :class:`Product` to which the inventory record applies.
""")

PerpetualInventory.store = relationship(
    Store,
    primaryjoin=Store.F1056 == PerpetualInventory.F1056,
    foreign_keys=[Store.F1056],
    uselist=False,
    doc=u"""
Reference to the :class:`Store` to which the inventory record applies.
""")


class Operator(Base):
    """
    Represents an employee /  user (``CLK_TAB``).
    """
    __tablename__ = u'CLK_TAB'

    F1185 = Column(
        Integer(), primary_key=True, autoincrement=False, nullable=False,
        doc=u"Operator order number")

    F1001 = Column(
        Integer(), nullable=True,
        doc=u"Record status")

    F1126 = Column(
        Integer(), nullable=True,
        doc=u"Operator number")

    F27 = Column(
        String(length=14), nullable=True,
        doc=u"Vendor id")

    F170 = Column(
        Integer(), nullable=True,
        doc=u"Restriction code")

    F253 = Column(
        Date(), nullable=True,
        doc=u"Date start")

    F902 = Column(
        String(length=8), nullable=True,
        doc=u"Batch indentifier")

    F1000 = Column(
        String(length=5), nullable=False,
        doc=u"Target Identifier")

    F1056 = Column(
        String(length=4), nullable=True,
        doc=u"Terminal store")

    F1127 = Column(
        String(length=30), nullable=True,
        doc=u"Operator short name")

    F1141 = Column(
        Integer(), nullable=True,
        doc=u"Operator secret number")

    F1142 = Column(
        Integer(), nullable=True,
        doc=u"Operator restriction level")

    F1143 = Column(
        String(length=30), nullable=True,
        doc=u"Operator First name")

    F1144 = Column(
        String(length=30), nullable=True,
        doc=u"Operator Last name")

    F1145 = Column(
        Date(), nullable=True,
        doc=u"Operator birth date")

    F1146 = Column(
        String(length=10), nullable=True,
        doc=u"Oper Language")

    F1148 = Column(
        String(length=14), nullable=True,
        doc=u"Customer Id")

    F1176 = Column(
        String(length=3), nullable=True,
        doc=u"Operator active on node")

    F1552 = Column(
        String(length=4), nullable=True,
        doc=u"Operator status")

    F1553 = Column(
        Date(), nullable=True,
        doc=u"Oper. hiring date")

    F1554 = Column(
        Date(), nullable=True,
        doc=u"Oper. Break date")

    F1555 = Column(
        Date(), nullable=True,
        doc=u"Oper. last promotion date")

    F1556 = Column(
        Integer(), nullable=True,
        doc=u"Oper. Superior")

    F1557 = Column(
        String(length=40), nullable=True,
        doc=u"Oper. Address 1")

    F1558 = Column(
        String(length=40), nullable=True,
        doc=u"Oper. Address 2")

    F1559 = Column(
        String(length=40), nullable=True,
        doc=u"Oper. City")

    F1560 = Column(
        String(length=20), nullable=True,
        doc=u"Oper. Province/State")

    F1561 = Column(
        String(length=20), nullable=True,
        doc=u"Oper. Country")

    F1562 = Column(
        String(length=15), nullable=True,
        doc=u"Oper. Zip/Postal code")

    F1563 = Column(
        String(length=20), nullable=True,
        doc=u"Oper. Telephone 1")

    F1564 = Column(
        String(length=20), nullable=True,
        doc=u"Oper. Telephone 2")

    F1565 = Column(
        String(length=20), nullable=True,
        doc=u"Oper. Insurence number")

    F1566 = Column(
        Float(), nullable=True,
        doc=u"Oper. Holidays")

    F1567 = Column(
        Integer(), nullable=True,
        doc=u"Oper. Payroll table link")

    F1568 = Column(
        Integer(), nullable=True,
        doc=u"Oper. Assign drawer num")

    F1569 = Column(
        Integer(), nullable=True,
        doc=u"Oper. Department number")

    F1570 = Column(
        Float(), nullable=True,
        doc=u"Oper. Added experience")

    F1571 = Column(
        String(length=60), nullable=True,
        doc=u"Oper. Email address")

    F1585 = Column(
        String(length=20), nullable=True,
        doc=u"Oper. Job description")

    F1586 = Column(
        String(length=4), nullable=True,
        doc=u"Oper. Gender")

    F1587 = Column(
        String(length=20), nullable=True,
        doc=u"Oper. Break reason")

    F1588 = Column(
        String(length=4), nullable=True,
        doc=u"Oper. Civil state")

    F1589 = Column(
        String(length=15), nullable=True,
        doc=u"Oper. Employee number")

    F1590 = Column(
        String(length=80), nullable=True,
        doc=u"Oper. Note")

    F1964 = Column(
        String(length=4), nullable=True,
        doc=u"Store responsible")

    F2587 = Column(
        Date(), nullable=True,
        doc=u"Clock in/out")

    F2597 = Column(
        String(length=20), nullable=True,
        doc=u"Profil")

    F2692 = Column(
        String(length=1), nullable=True,
        doc=u"Show filter operator")

    def __repr__(self):
        return u"Operator(F1185={0}, F1126={1})".format(
            repr(self.F1185), repr(self.F1126)).encode(u'utf_8')

    def __unicode__(self):
        return unicode(self.F1127 or u'')


class CustomerGroup(Base):
    """
    Represents a customer group (CLG_TAB).
    """
    __tablename__ = u'CLG_TAB'

    F1154 = Column(
        String(length=10), primary_key=True, nullable=False,
        doc=u"Cust. group")

    F1268 = Column(
        String(length=30), nullable=True,
        doc=u"Cust. group descptor")

    def __repr__(self):
        return u"CustomerGroup(F1154={0})".format(repr(self.F1154)).encode(u'utf_8')
    
    def __unicode__(self):
        return unicode(self.F1268 or u'')


class Customer(Base):
    """
    Represents a customer (CLT_TAB).
    """
    __tablename__ = u'CLT_TAB'

    F1148 = Column(
        String(length=14), primary_key=True, nullable=False,
        doc=u"Customer Id")

    F902 = Column(
        String(length=8), nullable=True,
        doc=u"Batch indentifier")

    F1000 = Column(
        String(length=5), nullable=True,
        doc=u"Target Identifier")

    F1001 = Column(
        Integer(), nullable=True,
        doc=u"Record status")

    F1150 = Column(
        String(length=40), nullable=True,
        doc=u"Cust. last name")

    F1152 = Column(
        Integer(), nullable=True,
        doc=u"Cust. shopper level")

    F1153 = Column(
        Integer(), nullable=True,
        doc=u"Cust. risk level")

    F1154 = Column(
        String(length=10), nullable=True,
        doc=u"Cust. group")

    F1172 = Column(
        String(length=15), nullable=True,
        doc=u"Cust. phone number")

    F1581 = Column(
        String(length=20), nullable=True,
        doc=u"Secret number")

    F253 = Column(
        Date(), nullable=True,
        doc=u"Date start")

    F1149 = Column(
        String(length=40), nullable=True,
        doc=u"Cust. first name")

    F1151 = Column(
        String(length=40), nullable=True,
        doc=u"Cust. pos comment")

    F1155 = Column(
        String(length=40), nullable=True,
        doc=u"Cust. company name")

    F1156 = Column(
        String(length=40), nullable=True,
        doc=u"Cust. address 1")

    F1157 = Column(
        String(length=40), nullable=True,
        doc=u"Cust. address 2")

    F1158 = Column(
        String(length=40), nullable=True,
        doc=u"Cust. city")

    F1159 = Column(
        String(length=15), nullable=True,
        doc=u"Cust. zip/postal code")

    F1160 = Column(
        String(length=20), nullable=True,
        doc=u"Cust. Tax1-3 exempt")

    F1161 = Column(
        String(length=20), nullable=True,
        doc=u"Cust. Tax2-4 exempt")

    F1168 = Column(
        Integer(), nullable=True,
        doc=u"Operator responsible")

    F1170 = Column(
        String(length=20), nullable=True,
        doc=u"Cust. province/state")

    F1171 = Column(
        String(length=30), nullable=True,
        doc=u"Cust. country")

    F1173 = Column(
        String(length=15), nullable=True,
        doc=u"Cust. fax number")

    F1264 = Column(
        Date(), nullable=True,
        doc=u"Date creation")

    F1265 = Column(
        Date(), nullable=True,
        doc=u"Cust. valid until")

    F1501 = Column(
        String(length=14), nullable=True,
        doc=u"Customer alias")

    F1502 = Column(
        String(length=6), nullable=True,
        doc=u"Cust. Alias reason")

    F1504 = Column(
        String(length=2), nullable=True,
        doc=u"Customer language")

    F1520 = Column(
        Date(), nullable=True,
        doc=u"Birth date")

    F1521 = Column(
        Date(), nullable=True,
        doc=u"Update date")

    F1522 = Column(
        Date(), nullable=True,
        doc=u"Membership renewal date")

    F1523 = Column(
        String(length=4), nullable=True,
        doc=u"Gender")

    F1524 = Column(
        String(length=4), nullable=True,
        doc=u"Creating store")

    F1525 = Column(
        String(length=4), nullable=True,
        doc=u"Calling store")

    F1526 = Column(
        Date(), nullable=True,
        doc=u"Membership closing date")

    F1527 = Column(
        String(length=4), nullable=True,
        doc=u"Mailing content code")

    F1528 = Column(
        String(length=4), nullable=True,
        doc=u"Calling reason code")

    F1529 = Column(
        String(length=1), nullable=True,
        doc=u"Verify address")

    F1550 = Column(
        String(length=10), nullable=True,
        doc=u"Civic number")

    F1573 = Column(
        String(length=60), nullable=True,
        doc=u"Email address")

    F1574 = Column(
        String(length=25), nullable=True,
        doc=u"Credit card number")

    F1575 = Column(
        Date(), nullable=True,
        doc=u"Credit card expiration")

    F1576 = Column(
        String(length=40), nullable=True,
        doc=u"Credit card signature")

    F1582 = Column(
        String(length=10), nullable=True,
        doc=u"Credit card type")

    F1583 = Column(
        Date(), nullable=True,
        doc=u"Last calling date")

    F1584 = Column(
        Date(), nullable=True,
        doc=u"Last mailing date")

    F1642 = Column(
        String(length=14), nullable=True,
        doc=u"Ship id number")

    F1643 = Column(
        String(length=40), nullable=True,
        doc=u"Ship name")

    F1644 = Column(
        String(length=40), nullable=True,
        doc=u"Ship contact name")

    F1645 = Column(
        String(length=40), nullable=True,
        doc=u"Ship address line 1")

    F1646 = Column(
        String(length=40), nullable=True,
        doc=u"Ship address line 2")

    F1647 = Column(
        String(length=40), nullable=True,
        doc=u"Ship address city")

    F1648 = Column(
        String(length=15), nullable=True,
        doc=u"Ship address state")

    F1649 = Column(
        String(length=15), nullable=True,
        doc=u"Ship address zip")

    F1650 = Column(
        String(length=15), nullable=True,
        doc=u"Ship phone voice")

    F1651 = Column(
        String(length=15), nullable=True,
        doc=u"Ship phone fax")

    F1654 = Column(
        Float(), nullable=True,
        doc=u"Freight rate weight")

    F1655 = Column(
        Float(), nullable=True,
        doc=u"Freight rate volume")

    F1685 = Column(
        Integer(), nullable=True,
        doc=u"Back order days")

    F1697 = Column(
        String(length=40), nullable=True,
        doc=u"Freight route")

    F1743 = Column(
        Integer(), nullable=True,
        doc=u"Temporary shopper level")

    F1759 = Column(
        Integer(), nullable=True,
        doc=u"Maintenance operator level")

    F1777 = Column(
        String(length=10), nullable=True,
        doc=u"Customer group 2")

    F1778 = Column(
        String(length=80), nullable=True,
        doc=u"Customer internal comment")

    F1809 = Column(
        String(length=12), nullable=True,
        doc=u"Spare CLT")

    F1810 = Column(
        Integer(), nullable=True,
        doc=u"Special adding function")

    F1950 = Column(
        String(length=2), nullable=True,
        doc=u"Customer status")

    F1964 = Column(
        String(length=4), nullable=True,
        doc=u"Store responsible")

    F2597 = Column(
        String(length=20), nullable=True,
        doc=u"Profil")

    F2602 = Column(
        String(length=3), nullable=True,
        doc=u"Currency code")

    F2615 = Column(
        String(length=4), nullable=True,
        doc=u"SPARE")

    F2616 = Column(
        Integer(), nullable=True,
        doc=u"SPARE")

    F2617 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"SPARE")

    F2619 = Column(
        Float(), nullable=True,
        doc=u"SPARE")

    F2660 = Column(
        String(length=250), nullable=True,
        doc=u"URL")

    def __repr__(self):
        return u"Customer(F1148={0})".format(repr(self.F1148)).encode(u'utf_8')
    
    def __unicode__(self):
        first_name = self.F1149
        last_name = self.F1150
        if first_name and last_name:
            return unicode(first_name) + u' ' + unicode(last_name)
        if first_name:
            return unicode(first_name)
        if last_name:
            return unicode(last_name)
        return u''


class AccountLink(Base):
    """
    Represents an account link (CLL_TAB).
    """
    __tablename__ = u'CLL_TAB'

    F1577 = Column(
        String(length=25), primary_key=True, nullable=False,
        doc=u"Alternate cust number")

    F1578 = Column(
        String(length=6), primary_key=True, nullable=False,
        doc=u"Aternate cust type")

    F902 = Column(
        String(length=8), nullable=True,
        doc=u"Batch indentifier")

    F1000 = Column(
        String(length=5), nullable=True,
        doc=u"Target Identifier")

    F1001 = Column(
        Integer(), nullable=True,
        doc=u"Record status")

    F1148 = Column(
        String(length=14), nullable=True,
        doc=u"Customer Id")

    F253 = Column(
        Date(), nullable=True,
        doc=u"Date start")

    F1759 = Column(
        Integer(), nullable=True,
        doc=u"Maintenance operator level")

    F1964 = Column(
        String(length=4), nullable=True,
        doc=u"Store responsible")

    def __repr__(self):
        return u"AccountLink(F1577={0}, F1578={1})".format(
            repr(self.F1577), repr(self.F1578)).encode(u'utf_8')


class Purchase(Base):
    """
    Represents a purchase in any of its phases (``REC_HDR``).

    Although this model is named ``Purchase``, it can represent any of the
    following:

    * purchasing document
    * receiving document
    * billing document
    * direct delivery document
    """
    __tablename__ = u'REC_HDR'

    F1032 = Column(
        Integer(), primary_key=True, autoincrement=False, nullable=False,
        doc=u"Transaction number")

    F27 = Column(
        String(length=14), nullable=True,
        doc=u"Vendor id")

    F76 = Column(
        Date(), nullable=True,
        doc=u"Date order")

    F91 = Column(
        String(length=8), nullable=True,
        doc=u"Purchase order number")

    F253 = Column(
        Date(), nullable=True,
        doc=u"Date start")

    F254 = Column(
        Date(), nullable=True,
        doc=u"Date end")

    F334 = Column(
        String(length=40), nullable=True,
        doc=u"Vendor name")

    F335 = Column(
        String(length=40), nullable=True,
        doc=u"Vendor contact name")

    F336 = Column(
        String(length=40), nullable=True,
        doc=u"Vendor address line 1")

    F337 = Column(
        String(length=40), nullable=True,
        doc=u"Vendor address line 2")

    F338 = Column(
        String(length=40), nullable=True,
        doc=u"Vendor address city")

    F339 = Column(
        String(length=15), nullable=True,
        doc=u"Vendor address state")

    F340 = Column(
        String(length=15), nullable=True,
        doc=u"Vendor address zip")

    F341 = Column(
        String(length=20), nullable=True,
        doc=u"Vendor phone voice")

    F342 = Column(
        String(length=15), nullable=True,
        doc=u"Vendor phone fax")

    F343 = Column(
        String(length=30), nullable=True,
        doc=u"Vendor product categories")

    F344 = Column(
        String(length=20), nullable=True,
        doc=u"Freight/broker company")

    F345 = Column(
        String(length=20), nullable=True,
        doc=u"FOB Point")

    F346 = Column(
        Float(), nullable=True,
        doc=u"Vendor limit")

    F347 = Column(
        Float(), nullable=True,
        doc=u"Special discount percent.")

    F348 = Column(
        Integer(), nullable=True,
        doc=u"Terms days")

    F349 = Column(
        Float(), nullable=True,
        doc=u"Terms discount percent")

    F352 = Column(
        String(length=10), nullable=True,
        doc=u"VendorAccount")

    F902 = Column(
        String(length=8), nullable=True,
        doc=u"Batch indentifier")

    F1035 = Column(
        String(length=6), nullable=True,
        doc=u"Movement start time")

    F1036 = Column(
        String(length=6), nullable=True,
        doc=u"Movement end time")

    F1056 = Column(
        String(length=4), nullable=True,
        doc=u"Terminal store")

    F1057 = Column(
        String(length=4), nullable=True,
        doc=u"Terminal number")

    F1067 = Column(
        String(length=7), nullable=True,
        doc=u"Registration mode")

    F1068 = Column(
        String(length=6), nullable=True,
        doc=u"Transaction mode")

    F1101 = Column(
        Integer(), nullable=True,
        doc=u"Line number")

    F1126 = Column(
        Integer(), nullable=True,
        doc=u"Operator number")

    F1127 = Column(
        String(length=30), nullable=True,
        doc=u"Operator short name")

    F1148 = Column(
        String(length=14), nullable=True,
        doc=u"Customer Id")

    F1168 = Column(
        Integer(), nullable=True,
        doc=u"Operator responsible")

    F1245 = Column(
        String(length=20), nullable=True,
        doc=u"Reference number")

    F1246 = Column(
        Date(), nullable=True,
        doc=u"Date delivery")

    F1254 = Column(
        String(length=500), nullable=True,
        doc=u"Transaction note")

    F1255 = Column(
        String(length=40), nullable=True,
        doc=u"Trans. Batch file name")

    F1264 = Column(
        Date(), nullable=True,
        doc=u"Date creation")

    F1642 = Column(
        String(length=14), nullable=True,
        doc=u"Ship id number")

    F1643 = Column(
        String(length=40), nullable=True,
        doc=u"Ship name")

    F1644 = Column(
        String(length=40), nullable=True,
        doc=u"Ship contact name")

    F1645 = Column(
        String(length=40), nullable=True,
        doc=u"Ship address line 1")

    F1646 = Column(
        String(length=40), nullable=True,
        doc=u"Ship address line 2")

    F1647 = Column(
        String(length=40), nullable=True,
        doc=u"Ship address city")

    F1648 = Column(
        String(length=15), nullable=True,
        doc=u"Ship address state")

    F1649 = Column(
        String(length=15), nullable=True,
        doc=u"Ship address zip")

    F1650 = Column(
        String(length=15), nullable=True,
        doc=u"Ship phone voice")

    F1651 = Column(
        String(length=15), nullable=True,
        doc=u"Ship phone fax")

    F1652 = Column(
        String(length=30), nullable=True,
        doc=u"Ship product categories")

    F1653 = Column(
        Date(), nullable=True,
        doc=u"Date shipping")

    F1654 = Column(
        Float(), nullable=True,
        doc=u"Freight rate weight")

    F1655 = Column(
        Float(), nullable=True,
        doc=u"Freight rate volume")

    F1656 = Column(
        String(length=30), nullable=True,
        doc=u"Vendor term descriptor")

    F1685 = Column(
        Integer(), nullable=True,
        doc=u"Back order days")

    F1686 = Column(
        Integer(), nullable=True,
        doc=u"Operator order printed")

    F1687 = Column(
        Integer(), nullable=True,
        doc=u"Operator receipt printed")

    F1688 = Column(
        Integer(), nullable=True,
        doc=u"Operator invoice printed")

    F1689 = Column(
        Integer(), nullable=True,
        doc=u"Operator export made")

    F1692 = Column(
        String(length=80), nullable=True,
        doc=u"Shipping note")

    F1693 = Column(
        Integer(), nullable=True,
        doc=u"Operator order entered")

    F1694 = Column(
        Integer(), nullable=True,
        doc=u"Operator receipt confirm")

    F1695 = Column(
        Integer(), nullable=True,
        doc=u"Operator invoice entered")

    F1696 = Column(
        Integer(), nullable=True,
        doc=u"Operator trucker")

    F1760 = Column(
        Float(), nullable=True,
        doc=u"Minimum cases")

    F1761 = Column(
        Float(), nullable=True,
        doc=u"Minimum cubic")

    F1793 = Column(
        Integer(), nullable=True,
        doc=u"Delivery days")

    F1883 = Column(
        Float(), nullable=True,
        doc=u"Minimum weight")

    F1884 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Minimum dollars")

    F1887 = Column(
        String(length=2), nullable=True,
        doc=u"Buying format")

    F1889 = Column(
        String(length=40), nullable=True,
        doc=u"Vendor contact 2")

    F1890 = Column(
        String(length=20), nullable=True,
        doc=u"Vendor contact 2 phone")

    F2596 = Column(
        Date(), nullable=True,
        doc=u"Date payable")

    F2598 = Column(
        Float(), nullable=True,
        doc=u"Currency exchange now")

    F2599 = Column(
        Float(), nullable=True,
        doc=u"Currency exchange order")

    F2630 = Column(
        String(length=4), nullable=True,
        doc=u"SPARE")

    F2631 = Column(
        Integer(), nullable=True,
        doc=u"SPARE")

    F2632 = Column(
        Integer(), nullable=True,
        doc=u"Operator pickup printed")

    F2633 = Column(
        Float(), nullable=True,
        doc=u"SPARE")

    F2634 = Column(
        Float(), nullable=True,
        doc=u"SPARE")

    F2635 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"SPARE")

    F2636 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"SPARE")

    F2655 = Column(
        String(length=6), nullable=True,
        doc=u"Order time")

    F2656 = Column(
        String(length=6), nullable=True,
        doc=u"Delivery time")

    F2657 = Column(
        Date(), nullable=True,
        doc=u"Event date")

    F2658 = Column(
        String(length=1), nullable=True,
        doc=u"Invisible on PDA")

    F2659 = Column(
        String(length=6), nullable=True,
        doc=u"Document sub type")

    F2710 = Column(
        String(length=10), nullable=True,
        doc=u"Associated tax rates")

    def __repr__(self):
        return u"Purchase(F1032={0})".format(repr(self.F1032)).encode(u'utf_8')


Purchase.vendor = relationship(
    Vendor,
    primaryjoin=Vendor.F27 == Purchase.F27,
    foreign_keys=[Vendor.F27],
    uselist=False,
    doc=u"""
Reference to the :class:`Vendor` from which the order was (or will be) purchased.
""")

Purchase.operator_order_entered = relationship(
    Operator,
    primaryjoin=Operator.F1126 == Purchase.F1693,
    foreign_keys=[Operator.F1126],
    uselist=False,
    doc=u"""
Reference to the :class:`Operator` who initially entered the order into SMS.
""")


class PurchaseDetail(Base):
    """
    Represents a detail line within a purchase transaction (REC_REG).
    """
    __tablename__ = u'REC_REG'

    F1032 = Column(
        Integer(), primary_key=True, autoincrement=False, nullable=False,
        doc=u"Transaction number")

    F1101 = Column(
        Integer(), primary_key=True, autoincrement=False, nullable=False,
        doc=u"Line number")

    F01 = Column(
        String(length=13), nullable=True,
        doc=u"UPC code")

    F03 = Column(
        Integer(), nullable=True,
        doc=u"Department code")

    F04 = Column(
        Integer(), nullable=True,
        doc=u"Sub-Department code")

    F05 = Column(
        Integer(), nullable=True,
        doc=u"Bottle deposit link")

    F08 = Column(
        String(length=4), nullable=True,
        doc=u"Status code")

    F19 = Column(
        Float(), nullable=True,
        doc=u"Case size")

    F20 = Column(
        Float(), nullable=True,
        doc=u"Receiving pack size")

    F26 = Column(
        String(length=15), nullable=True,
        doc=u"Vendor code")

    F38 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Base cost")

    F50 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Bottle deposit value")

    F64 = Column(
        Float(), nullable=True,
        doc=u"Total units")

    F65 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Total dollars")

    F67 = Column(
        Float(), nullable=True,
        doc=u"Total weight")

    F70 = Column(
        Float(), nullable=True,
        doc=u"Unit quantity")

    F75 = Column(
        Float(), nullable=True,
        doc=u"Case on order")

    F76 = Column(
        Date(), nullable=True,
        doc=u"Date order")

    F81 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 1")

    F82 = Column(
        String(length=1), nullable=True,
        doc=u"Scalable item")

    F90 = Column(
        String(length=1), nullable=True,
        doc=u"Vendor authorized item")

    F92 = Column(
        String(length=1), nullable=True,
        doc=u"Deposit container code")

    F96 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 2")

    F97 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 3")

    F98 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 4")

    F99 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 5")

    F100 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 6")

    F101 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 7")

    F120 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Split case cost")

    F131 = Column(
        Integer(), nullable=True,
        doc=u"Sequence number")

    F151 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Next cost net")

    F156 = Column(
        String(length=120), nullable=True,
        doc=u"Allowance qualifier")

    F186 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Case deposit value")

    F201 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Allowance amount")

    F202 = Column(
        Date(), nullable=True,
        doc=u"Allowance start date")

    F203 = Column(
        Date(), nullable=True,
        doc=u"Allowance end date")

    F204 = Column(
        String(length=1), nullable=True,
        doc=u"Allowance code")

    F219 = Column(
        Date(), nullable=True,
        doc=u"Cost break end date")

    F220 = Column(
        String(length=13), nullable=True,
        doc=u"Split item code")

    F222 = Column(
        String(length=12), nullable=True,
        doc=u"Allowance number")

    F223 = Column(
        String(length=1), nullable=True,
        doc=u"Allowance off invoice")

    F224 = Column(
        Float(), nullable=True,
        doc=u"Cost break min qty")

    F225 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Cost break amount")

    F226 = Column(
        Float(), nullable=True,
        doc=u"Cost break percent")

    F227 = Column(
        Date(), nullable=True,
        doc=u"Cost break start date")

    F228 = Column(
        Float(), nullable=True,
        doc=u"Discount percent")

    F229 = Column(
        Date(), nullable=True,
        doc=u"Discount start date")

    F230 = Column(
        Date(), nullable=True,
        doc=u"Discount end date")

    F231 = Column(
        String(length=12), nullable=True,
        doc=u"Discount number")

    F233 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Rebate amount")

    F234 = Column(
        Date(), nullable=True,
        doc=u"Rebate start date")

    F235 = Column(
        Date(), nullable=True,
        doc=u"Rebate end date")

    F236 = Column(
        String(length=12), nullable=True,
        doc=u"Rebate number")

    F237 = Column(
        Float(), nullable=True,
        doc=u"Rebate percent")

    F253 = Column(
        Date(), nullable=True,
        doc=u"Date start")

    F254 = Column(
        Date(), nullable=True,
        doc=u"Date end")

    F270 = Column(
        Float(), nullable=True,
        doc=u"Weight net")

    F327 = Column(
        Float(), nullable=True,
        doc=u"Freight case weight")

    F328 = Column(
        Float(), nullable=True,
        doc=u"Freight case cube")

    F329 = Column(
        Float(), nullable=True,
        doc=u"Allowance max qty")

    F1003 = Column(
        Float(), nullable=True,
        doc=u"Case quantity")

    F1034 = Column(
        Integer(), nullable=True,
        doc=u"Totalizer number")

    F1041 = Column(
        String(length=60), nullable=True,
        doc=u"Description registration")

    F1063 = Column(
        Integer(), nullable=True,
        doc=u"Function code")

    F1067 = Column(
        String(length=7), nullable=True,
        doc=u"Registration mode")

    F1069 = Column(
        String(length=1), nullable=True,
        doc=u"Void previous flag")

    F1070 = Column(
        String(length=1), nullable=True,
        doc=u"Refund key flag")

    F1071 = Column(
        String(length=1), nullable=True,
        doc=u"Correction key flag")

    F1079 = Column(
        String(length=30), nullable=True,
        doc=u"Reference number")

    F1081 = Column(
        String(length=500), nullable=True,
        doc=u"Alpha parameter")

    F1086 = Column(
        String(length=1), nullable=True,
        doc=u"Void prohibited")

    F1121 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Cost fee amount")

    F1122 = Column(
        Float(), nullable=True,
        doc=u"Cost fee percent")

    F1140 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Unit net cost")

    F1184 = Column(
        String(length=12), nullable=True,
        doc=u"Buying format")

    F1246 = Column(
        Date(), nullable=True,
        doc=u"Date delivery")

    F1247 = Column(
        String(length=1), nullable=True,
        doc=u"Item free flag")

    F1248 = Column(
        String(length=1), nullable=True,
        doc=u"Item return flag")

    F1249 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Allowance total")

    F1250 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Rebate total")

    F1251 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Discount total")

    F1252 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Cost break total")

    F1266 = Column(
        Float(), nullable=True,
        doc=u"Unit on order")

    F1267 = Column(
        Float(), nullable=True,
        doc=u"Weight on order")

    F1595 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Envir.fees value")

    F1596 = Column(
        Integer(), nullable=True,
        doc=u"Envir.fees tlz")

    F1657 = Column(
        Float(), nullable=True,
        doc=u"Allowance percent")

    F1658 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Discount amount")

    F1659 = Column(
        String(length=1), nullable=True,
        doc=u"Discount off invoice")

    F1660 = Column(
        String(length=1), nullable=True,
        doc=u"Rebate off invoice")

    F1661 = Column(
        String(length=1), nullable=True,
        doc=u"Cost break off invoice")

    F1662 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"BillBack1 Amount")

    F1663 = Column(
        Float(), nullable=True,
        doc=u"BillBack1 Percent")

    F1664 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"BillBack2 Amount")

    F1665 = Column(
        Float(), nullable=True,
        doc=u"BillBack2 Percent")

    F1666 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"BillBack3 Amount")

    F1667 = Column(
        Float(), nullable=True,
        doc=u"BillBack3 Percent")

    F1668 = Column(
        String(length=16), nullable=True,
        doc=u"BillBack4 Descriptor")

    F1669 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"BillBack4 Amount")

    F1670 = Column(
        Float(), nullable=True,
        doc=u"BillBack4 Percent")

    F1671 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"BillBack1 Total")

    F1672 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"BillBack2 Total")

    F1673 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"BillBack3 Total")

    F1674 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"BillBack4 Total")

    F1675 = Column(
        String(length=8), nullable=True,
        doc=u"Event regular type")

    F1676 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Event regular discount")

    F1677 = Column(
        String(length=8), nullable=True,
        doc=u"Event promo type")

    F1678 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Event promo discount")

    F1679 = Column(
        Integer(), nullable=True,
        doc=u"Event regular days")

    F1680 = Column(
        Date(), nullable=True,
        doc=u"Event promo date")

    F1681 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Bottle deposit total")

    F1682 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Envir.fees total")

    F1683 = Column(
        Float(), nullable=True,
        doc=u"Freight weight total")

    F1684 = Column(
        Float(), nullable=True,
        doc=u"Freight cubic total")

    F1685 = Column(
        Integer(), nullable=True,
        doc=u"Back order days")

    F1687 = Column(
        Integer(), nullable=True,
        doc=u"Operator receipt printed")

    F1691 = Column(
        String(length=250), nullable=True,
        doc=u"Registration note")

    F1732 = Column(
        Float(), nullable=True,
        doc=u"Back order filled")

    F1771 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Retail total")

    F1772 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Cost fee amount total")

    F1773 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Cost fee percent total")

    F1774 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Total match up/down")

    F1775 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Retail price")

    F1776 = Column(
        Integer(), nullable=True,
        doc=u"Spare RECREG")

    F1791 = Column(
        String(length=12), nullable=True,
        doc=u"Cost method")

    F1793 = Column(
        Integer(), nullable=True,
        doc=u"Delivery days")

    F1795 = Column(
        Float(), nullable=True,
        doc=u"Split item qty")

    F1797 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Spare COS/RECREG")

    F1815 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Open amount")

    F1816 = Column(
        Integer(), nullable=True,
        doc=u"Open totalizer")

    F1887 = Column(
        String(length=2), nullable=True,
        doc=u"Buying format")

    F1899 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"SO matching")

    F1918 = Column(
        Float(), nullable=True,
        doc=u"Back order case")

    F1919 = Column(
        Float(), nullable=True,
        doc=u"Back order unit")

    F1920 = Column(
        Float(), nullable=True,
        doc=u"Back order weight")

    F1973 = Column(
        String(length=120), nullable=True,
        doc=u"Discount qualifier")

    F1974 = Column(
        String(length=120), nullable=True,
        doc=u"Rebate qualifer")

    F1975 = Column(
        String(length=120), nullable=True,
        doc=u"Cost break qualifier")

    F1976 = Column(
        String(length=120), nullable=True,
        doc=u"Cost comment")

    F1977 = Column(
        Float(), nullable=True,
        doc=u"Allowance min qty")

    F1978 = Column(
        Float(), nullable=True,
        doc=u"Discount min qty")

    F1979 = Column(
        Float(), nullable=True,
        doc=u"Rebate min qty")

    F2551 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Fee 1")

    F2552 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Fee 2")

    F2553 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Fee 3")

    F2554 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Fee 4")

    F2555 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Fee 5")

    F2556 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Fee 6")

    F2557 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Fee 7")

    F2558 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Fee 8")

    F2559 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Fee 9")

    F2560 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Fee 10")

    F2561 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Fee 11")

    F2562 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Fee 12")

    F2563 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Fee 13")

    F2564 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Fee 14")

    F2565 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Fee 15")

    F2566 = Column(
        Float(), nullable=True,
        doc=u"Discount max qty")

    F2567 = Column(
        Float(), nullable=True,
        doc=u"Rebate max qty")

    F2568 = Column(
        Float(), nullable=True,
        doc=u"Cost break max qty")

    F2624 = Column(
        Integer(), nullable=True,
        doc=u"SPARE")

    F2625 = Column(
        Integer(), nullable=True,
        doc=u"SPARE")

    F2626 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"SPARE")

    F2627 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Gross purchase")

    F2628 = Column(
        Float(), nullable=True,
        doc=u"SPARE")

    F2629 = Column(
        Float(), nullable=True,
        doc=u"Backstore pickup")

    F2630 = Column(
        String(length=4), nullable=True,
        doc=u"SPARE")

    F2631 = Column(
        Integer(), nullable=True,
        doc=u"SPARE")

    F2660 = Column(
        String(length=250), nullable=True,
        doc=u"URL")

    F2666 = Column(
        Float(), nullable=True,
        doc=u"Default quantity")

    F2709 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Total Tax included")

    F2711 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Tax 1 paid")

    F2712 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Tax 2 paid")

    F2713 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Tax 3 paid")

    F2714 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Tax 4 paid")

    F2715 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Tax 5 paid")

    F2716 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Tax 6 paid")

    F2717 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Tax 7 paid")

    F2718 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Tax 8 paid")

    def __repr__(self):
        return u"PurchaseDetail(F1032={0}, F1101={1})".format(
            repr(self.F1032), repr(self.F1101)).encode(u'utf_8')


Purchase.details = relationship(
    PurchaseDetail,
    primaryjoin=PurchaseDetail.F1032 == Purchase.F1032,
    foreign_keys=[PurchaseDetail.F1032],
    doc=u"""
List-like collection of :class:`PurchaseDetail` instances which belong to the
purchasing document.
""")


class Sale(Base):
    """
    Represents a selling transaction (SAL_HDR).
    """
    __tablename__ = u'SAL_HDR'

    F1032 = Column(
        Integer(), primary_key=True, autoincrement=False, nullable=False,
        doc=u"Transaction number")

    F1148 = Column(
        String(length=14), nullable=True,
        doc=u"Customer Id")

    F76 = Column(
        Date(), nullable=True,
        doc=u"Date order")

    F91 = Column(
        String(length=8), nullable=True,
        doc=u"Purchase order number")

    F253 = Column(
        Date(), nullable=True,
        doc=u"Date start")

    F254 = Column(
        Date(), nullable=True,
        doc=u"Date end")

    F902 = Column(
        String(length=8), nullable=True,
        doc=u"Batch indentifier")

    F1035 = Column(
        String(length=6), nullable=True,
        doc=u"Movement start time")

    F1036 = Column(
        String(length=6), nullable=True,
        doc=u"Movement end time")

    F1056 = Column(
        String(length=4), nullable=True,
        doc=u"Terminal store")

    F1057 = Column(
        String(length=4), nullable=True,
        doc=u"Terminal number")

    F1067 = Column(
        String(length=7), nullable=True,
        doc=u"Registration mode")

    F1068 = Column(
        String(length=6), nullable=True,
        doc=u"Transaction mode")

    F1101 = Column(
        Integer(), nullable=True,
        doc=u"Line number")

    F1126 = Column(
        Integer(), nullable=True,
        doc=u"Operator number")

    F1127 = Column(
        String(length=30), nullable=True,
        doc=u"Operator short name")

    F1137 = Column(
        Integer(), nullable=True,
        doc=u"Transaction void count")

    F1149 = Column(
        String(length=40), nullable=True,
        doc=u"Cust. first name")

    F1150 = Column(
        String(length=40), nullable=True,
        doc=u"Cust. last name")

    F1151 = Column(
        String(length=40), nullable=True,
        doc=u"Cust. pos comment")

    F1152 = Column(
        Integer(), nullable=True,
        doc=u"Cust. shopper level")

    F1153 = Column(
        Integer(), nullable=True,
        doc=u"Cust. risk level")

    F1154 = Column(
        String(length=10), nullable=True,
        doc=u"Cust. group")

    F1155 = Column(
        String(length=40), nullable=True,
        doc=u"Cust. company name")

    F1156 = Column(
        String(length=40), nullable=True,
        doc=u"Cust. address 1")

    F1157 = Column(
        String(length=40), nullable=True,
        doc=u"Cust. address 2")

    F1158 = Column(
        String(length=40), nullable=True,
        doc=u"Cust. city")

    F1159 = Column(
        String(length=15), nullable=True,
        doc=u"Cust. zip/postal code")

    F1160 = Column(
        String(length=20), nullable=True,
        doc=u"Cust. Tax1-3 exempt")

    F1161 = Column(
        String(length=20), nullable=True,
        doc=u"Cust. Tax2-4 exempt")

    F1163 = Column(
        Integer(), nullable=True,
        doc=u"Shopper price level")

    F1164 = Column(
        Float(), nullable=True,
        doc=u"Shopper discount")

    F1165 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Cust. risk max credit")

    F1167 = Column(
        String(length=20), nullable=True,
        doc=u"Cust. risk term")

    F1168 = Column(
        Integer(), nullable=True,
        doc=u"Operator responsible")

    F1170 = Column(
        String(length=20), nullable=True,
        doc=u"Cust. province/state")

    F1171 = Column(
        String(length=30), nullable=True,
        doc=u"Cust. country")

    F1172 = Column(
        String(length=15), nullable=True,
        doc=u"Cust. phone number")

    F1173 = Column(
        String(length=15), nullable=True,
        doc=u"Cust. fax number")

    F1185 = Column(
        Integer(), nullable=True,
        doc=u"Operator order number")

    F1238 = Column(
        Integer(), nullable=True,
        doc=u"Client points beginning")

    F1242 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Client balance begining")

    F1245 = Column(
        String(length=20), nullable=True,
        doc=u"Reference number")

    F1246 = Column(
        Date(), nullable=True,
        doc=u"Date delivery")

    F1254 = Column(
        String(length=500), nullable=True,
        doc=u"Transaction note")

    F1255 = Column(
        String(length=40), nullable=True,
        doc=u"Trans. Batch file name")

    F1271 = Column(
        String(length=1), nullable=True,
        doc=u"Shopper get Dsc $")

    F1272 = Column(
        String(length=1), nullable=True,
        doc=u"Shopper get Dsc %")

    F1273 = Column(
        String(length=1), nullable=True,
        doc=u"Shopper get linked item")

    F1274 = Column(
        String(length=1), nullable=True,
        doc=u"Shopper get coupon link")

    F1277 = Column(
        String(length=1), nullable=True,
        doc=u"Shopper get dsc dept")

    F1287 = Column(
        String(length=1), nullable=True,
        doc=u"Shopper get package price")

    F1288 = Column(
        String(length=1), nullable=True,
        doc=u"Shopper get limited price")

    F1295 = Column(
        String(length=8), nullable=True,
        doc=u"Multiple promotion method")

    F1504 = Column(
        String(length=2), nullable=True,
        doc=u"Customer language")

    F1520 = Column(
        Date(), nullable=True,
        doc=u"Birth date")

    F1642 = Column(
        String(length=14), nullable=True,
        doc=u"Ship id number")

    F1643 = Column(
        String(length=40), nullable=True,
        doc=u"Ship name")

    F1644 = Column(
        String(length=40), nullable=True,
        doc=u"Ship contact name")

    F1645 = Column(
        String(length=40), nullable=True,
        doc=u"Ship address line 1")

    F1646 = Column(
        String(length=40), nullable=True,
        doc=u"Ship address line 2")

    F1647 = Column(
        String(length=40), nullable=True,
        doc=u"Ship address city")

    F1648 = Column(
        String(length=15), nullable=True,
        doc=u"Ship address state")

    F1649 = Column(
        String(length=15), nullable=True,
        doc=u"Ship address zip")

    F1650 = Column(
        String(length=15), nullable=True,
        doc=u"Ship phone voice")

    F1651 = Column(
        String(length=15), nullable=True,
        doc=u"Ship phone fax")

    F1652 = Column(
        String(length=30), nullable=True,
        doc=u"Ship product categories")

    F1653 = Column(
        Date(), nullable=True,
        doc=u"Date shipping")

    F1654 = Column(
        Float(), nullable=True,
        doc=u"Freight rate weight")

    F1655 = Column(
        Float(), nullable=True,
        doc=u"Freight rate volume")

    F1685 = Column(
        Integer(), nullable=True,
        doc=u"Back order days")

    F1686 = Column(
        Integer(), nullable=True,
        doc=u"Operator order printed")

    F1687 = Column(
        Integer(), nullable=True,
        doc=u"Operator receipt printed")

    F1688 = Column(
        Integer(), nullable=True,
        doc=u"Operator invoice printed")

    F1689 = Column(
        Integer(), nullable=True,
        doc=u"Operator export made")

    F1692 = Column(
        String(length=80), nullable=True,
        doc=u"Shipping note")

    F1693 = Column(
        Integer(), nullable=True,
        doc=u"Operator order entered")

    F1694 = Column(
        Integer(), nullable=True,
        doc=u"Operator receipt confirm")

    F1695 = Column(
        Integer(), nullable=True,
        doc=u"Operator invoice entered")

    F1696 = Column(
        Integer(), nullable=True,
        doc=u"Operator trucker")

    F1697 = Column(
        String(length=40), nullable=True,
        doc=u"Freight route")

    F1699 = Column(
        Float(), nullable=True,
        doc=u"Shipping piece count")

    F1711 = Column(
        Integer(), nullable=True,
        doc=u"Price level fall back")

    F1763 = Column(
        String(length=8), nullable=True,
        doc=u"Sale order number")

    F1764 = Column(
        String(length=8), nullable=True,
        doc=u"Invoice number")

    F1938 = Column(
        Float(), nullable=True,
        doc=u"Cost plus percent")

    F2596 = Column(
        Date(), nullable=True,
        doc=u"Date payable")

    F2598 = Column(
        Float(), nullable=True,
        doc=u"Currency exchange now")

    F2599 = Column(
        Float(), nullable=True,
        doc=u"Currency exchange order")

    F2613 = Column(
        Integer(), nullable=True,
        doc=u"SPARE")

    F2614 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"SPARE")

    F2615 = Column(
        String(length=4), nullable=True,
        doc=u"SPARE")

    F2616 = Column(
        Integer(), nullable=True,
        doc=u"SPARE")

    F2617 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"SPARE")

    F2618 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"SPARE")

    F2619 = Column(
        Float(), nullable=True,
        doc=u"SPARE")

    F2620 = Column(
        Float(), nullable=True,
        doc=u"SPARE")

    F2621 = Column(
        Integer(), nullable=True,
        doc=u"SPARE")

    F2622 = Column(
        Integer(), nullable=True,
        doc=u"SPARE")

    F2623 = Column(
        String(length=4), nullable=True,
        doc=u"SPARE")

    def __repr__(self):
        return u"Sale(F1032={0})".format(repr(self.F1032)).encode(u'utf_8')


Sale.operator = relationship(
    Operator,
    primaryjoin=Operator.F1126 == Sale.F1126,
    foreign_keys=[Operator.F1126],
    uselist=False,
    doc=u"""
Reference to the :class:`Operator` who rang the transaction.
""")


Sale.customer = relationship(
    Customer,
    primaryjoin=Customer.F1148 == Sale.F1148,
    foreign_keys=[Customer.F1148],
    uselist=False,
    doc=u"""
Reference to the :class:`Customer` whose transaction it was.
""")


class SaleDetail(Base):
    """
    Represents a detail line within a sale transaction (SAL_REG).
    """
    __tablename__ = u'SAL_REG'

    F1032 = Column(
        Integer(), primary_key=True, autoincrement=False, nullable=False,
        doc=u"Transaction number")

    F1101 = Column(
        Integer(), primary_key=True, autoincrement=False, nullable=False,
        doc=u"Line number")

    F01 = Column(
        String(length=13), nullable=True,
        doc=u"UPC code")

    F03 = Column(
        Integer(), nullable=True,
        doc=u"Department code")

    F04 = Column(
        Integer(), nullable=True,
        doc=u"Sub-Department code")

    F05 = Column(
        Integer(), nullable=True,
        doc=u"Bottle deposit link")

    F06 = Column(
        Integer(), nullable=True,
        doc=u"Tare weight link")

    F08 = Column(
        String(length=4), nullable=True,
        doc=u"Status code")

    F24 = Column(
        Float(), nullable=True,
        doc=u"Weight divisor")

    F30 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Price")

    F31 = Column(
        Float(), nullable=True,
        doc=u"Price qty")

    F43 = Column(
        String(length=1), nullable=True,
        doc=u"Apply item discount*")

    F50 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Bottle deposit value")

    F60 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Excise tax amount*")

    F61 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Tax exempt amount*")

    F64 = Column(
        Float(), nullable=True,
        doc=u"Total units")

    F65 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Total dollars")

    F67 = Column(
        Float(), nullable=True,
        doc=u"Total weight")

    F77 = Column(
        Integer(), nullable=True,
        doc=u"Coupon family code")

    F79 = Column(
        String(length=1), nullable=True,
        doc=u"Food stamp")

    F80 = Column(
        String(length=1), nullable=True,
        doc=u"Trading stamp*")

    F81 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 1")

    F82 = Column(
        String(length=1), nullable=True,
        doc=u"Scalable item")

    F83 = Column(
        String(length=1), nullable=True,
        doc=u"Require price entry")

    F88 = Column(
        String(length=1), nullable=True,
        doc=u"Store coupon")

    F96 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 2")

    F97 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 3")

    F98 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 4")

    F99 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 5")

    F100 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 6")

    F101 = Column(
        String(length=1), nullable=True,
        doc=u"Tax flag 7")

    F102 = Column(
        String(length=1), nullable=True,
        doc=u"Prohibit quantity")

    F104 = Column(
        String(length=1), nullable=True,
        doc=u"Vendor coupon")

    F106 = Column(
        String(length=1), nullable=True,
        doc=u"Follow subdept status")

    F108 = Column(
        String(length=1), nullable=True,
        doc=u"Coupon restricted")

    F109 = Column(
        String(length=1), nullable=True,
        doc=u"Discount print/detailtrack")

    F110 = Column(
        String(length=1), nullable=True,
        doc=u"Electronic coupon")

    F113 = Column(
        String(length=8), nullable=True,
        doc=u"Active price description")

    F114 = Column(
        String(length=1), nullable=True,
        doc=u"Allow price override")

    F115 = Column(
        String(length=1), nullable=True,
        doc=u"Not in net sale")

    F124 = Column(
        String(length=1), nullable=True,
        doc=u"Prohibit returns*")

    F125 = Column(
        String(length=1), nullable=True,
        doc=u"Prohibit repeat key*")

    F126 = Column(
        Integer(), nullable=True,
        doc=u"Price level")

    F149 = Column(
        String(length=1), nullable=True,
        doc=u"Coupon multiplication")

    F150 = Column(
        String(length=1), nullable=True,
        doc=u"Prohibit discount")

    F160 = Column(
        String(length=1), nullable=True,
        doc=u"Bottle return")

    F168 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Suggested price")

    F169 = Column(
        Integer(), nullable=True,
        doc=u"Price shopper points")

    F170 = Column(
        Integer(), nullable=True,
        doc=u"Restriction code")

    F171 = Column(
        Integer(), nullable=True,
        doc=u"Minimum age customer")

    F172 = Column(
        String(length=1), nullable=True,
        doc=u"Prohibit refund*")

    F173 = Column(
        String(length=1), nullable=True,
        doc=u"Prohibit multiple coupon")

    F175 = Column(
        String(length=1), nullable=True,
        doc=u"Prohibit item markdown*")

    F177 = Column(
        String(length=1), nullable=True,
        doc=u"Not in admissible spending")

    F178 = Column(
        String(length=1), nullable=True,
        doc=u"WIC Eligible")

    F253 = Column(
        Date(), nullable=True,
        doc=u"Date start")

    F254 = Column(
        Date(), nullable=True,
        doc=u"Date end")

    F270 = Column(
        Float(), nullable=True,
        doc=u"Weight net")

    F383 = Column(
        String(length=20), nullable=True,
        doc=u"Promotion code")

    F903 = Column(
        String(length=16), nullable=True,
        doc=u"Batch creator")

    F1002 = Column(
        Float(), nullable=True,
        doc=u"Size cubic")

    F1006 = Column(
        Float(), nullable=True,
        doc=u"Active price quantity")

    F1007 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Active price")

    F1034 = Column(
        Integer(), nullable=True,
        doc=u"Totalizer number")

    F1041 = Column(
        String(length=60), nullable=True,
        doc=u"Description registration")

    F1063 = Column(
        Integer(), nullable=True,
        doc=u"Function code")

    F1067 = Column(
        String(length=7), nullable=True,
        doc=u"Registration mode")

    F1069 = Column(
        String(length=1), nullable=True,
        doc=u"Void previous flag")

    F1070 = Column(
        String(length=1), nullable=True,
        doc=u"Refund key flag")

    F1071 = Column(
        String(length=1), nullable=True,
        doc=u"Correction key flag")

    F1072 = Column(
        Integer(), nullable=True,
        doc=u"Sales person")

    F1078 = Column(
        Float(), nullable=True,
        doc=u"Ordered quantity")

    F1079 = Column(
        String(length=30), nullable=True,
        doc=u"Reference number")

    F1080 = Column(
        Float(), nullable=True,
        doc=u"Elapse time")

    F1081 = Column(
        String(length=500), nullable=True,
        doc=u"Alpha parameter")

    F1086 = Column(
        String(length=1), nullable=True,
        doc=u"Void prohibited")

    F1120 = Column(
        String(length=12), nullable=True,
        doc=u"POS specific flags")

    F1126 = Column(
        Integer(), nullable=True,
        doc=u"Operator number")

    F1136 = Column(
        String(length=8), nullable=True,
        doc=u"Ticket template")

    F1164 = Column(
        Float(), nullable=True,
        doc=u"Shopper discount")

    F1178 = Column(
        Float(), nullable=True,
        doc=u"Tare link value")

    F1185 = Column(
        Integer(), nullable=True,
        doc=u"Operator order number")

    F1203 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Active package price")

    F1204 = Column(
        Float(), nullable=True,
        doc=u"Active package qty")

    F1205 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Active limited price")

    F1206 = Column(
        Float(), nullable=True,
        doc=u"Active limited qty")

    F1207 = Column(
        Integer(), nullable=True,
        doc=u"Active mix&match")

    F1208 = Column(
        String(length=8), nullable=True,
        doc=u"Active price method")

    F1209 = Column(
        Integer(), nullable=True,
        doc=u"Active shopper points")

    F1224 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Active discount amount")

    F1225 = Column(
        Float(), nullable=True,
        doc=u"Active discount percent")

    F1239 = Column(
        Integer(), nullable=True,
        doc=u"Item discount fct")

    F1240 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Item discount amount")

    F1241 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Client discount total")

    F1256 = Column(
        Float(), nullable=True,
        doc=u"Department discount")

    F1263 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Total rounded tax")

    F1595 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Envir.fees value")

    F1596 = Column(
        Integer(), nullable=True,
        doc=u"Envir.fees tlz")

    F1683 = Column(
        Float(), nullable=True,
        doc=u"Freight weight total")

    F1684 = Column(
        Float(), nullable=True,
        doc=u"Freight cubic total")

    F1691 = Column(
        String(length=250), nullable=True,
        doc=u"Registration note")

    F1693 = Column(
        Integer(), nullable=True,
        doc=u"Operator order entered")

    F1694 = Column(
        Integer(), nullable=True,
        doc=u"Operator receipt confirm")

    F1699 = Column(
        Float(), nullable=True,
        doc=u"Shipping piece count")

    F1712 = Column(
        Float(), nullable=True,
        doc=u"Commited qty")

    F1715 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Bottle deposit total")

    F1716 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Envir.fees total")

    F1717 = Column(
        Float(), nullable=True,
        doc=u"Tare total")

    F1718 = Column(
        Integer(), nullable=True,
        doc=u"Store coupon fct")

    F1719 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Elect.store coupon amt")

    F1720 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Store coupon amt")

    F1721 = Column(
        Integer(), nullable=True,
        doc=u"Vendor coupon fct")

    F1722 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Elect.vendor coupon amt")

    F1723 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Vendor coupon amt")

    F1724 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Double coupon amt")

    F1725 = Column(
        Float(), nullable=True,
        doc=u"Item discount qty")

    F1726 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Item discount $")

    F1727 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Item discount %")

    F1728 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Proportinal discount amt")

    F1729 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Department discount total")

    F1730 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Gross sale")

    F1731 = Column(
        Integer(), nullable=True,
        doc=u"Proportinal discount fct")

    F1732 = Column(
        Float(), nullable=True,
        doc=u"Back order filled")

    F1733 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Pakage discount")

    F1734 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Limited discount")

    F1739 = Column(
        Float(), nullable=True,
        doc=u"Entry unit")

    F1740 = Column(
        Float(), nullable=True,
        doc=u"Entry Weight")

    F1741 = Column(
        String(length=8), nullable=True,
        doc=u"Envir.fees flags")

    F1742 = Column(
        String(length=8), nullable=True,
        doc=u"User field EJ")

    F1785 = Column(
        String(length=12), nullable=True,
        doc=u"Behavior")

    F1787 = Column(
        Integer(), nullable=True,
        doc=u"Replace adding function")

    F1789 = Column(
        Float(), nullable=True,
        doc=u"Store coupon count")

    F1802 = Column(
        Integer(), nullable=True,
        doc=u"Adding totalizer/function")

    F1803 = Column(
        Float(), nullable=True,
        doc=u"Time included")

    F1805 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Label price")

    F1815 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Open amount")

    F1816 = Column(
        Integer(), nullable=True,
        doc=u"Open totalizer")

    F1831 = Column(
        Integer(), nullable=True,
        doc=u"Points on store e-coupons")

    F1832 = Column(
        Integer(), nullable=True,
        doc=u"Redeemed points")

    F1833 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Discount from points")

    F1834 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"PO matching")

    F1835 = Column(
        Float(), nullable=True,
        doc=u"Reg last cost")

    F1860 = Column(
        Integer(), nullable=True,
        doc=u"Elect.vendor coupon fct")

    F1861 = Column(
        Float(), nullable=True,
        doc=u"Elect.vendor coupon cnt")

    F1862 = Column(
        Integer(), nullable=True,
        doc=u"Elect.store coupon fct")

    F1863 = Column(
        Float(), nullable=True,
        doc=u"Elect.store coupon cnt")

    F1864 = Column(
        Float(), nullable=True,
        doc=u"Vendor coupon cnt")

    F1874 = Column(
        Integer(), nullable=True,
        doc=u"Select pkg price")

    F1888 = Column(
        Float(), nullable=True,
        doc=u"Double cpn cnt")

    F1924 = Column(
        Float(), nullable=True,
        doc=u"Proportional disc percent")

    F1925 = Column(
        Float(), nullable=True,
        doc=u"Compare price qty")

    F1926 = Column(
        Float(), nullable=True,
        doc=u"SPARE")

    F1927 = Column(
        Float(), nullable=True,
        doc=u"SPARE")

    F1928 = Column(
        Float(), nullable=True,
        doc=u"SPARE")

    F1929 = Column(
        Integer(), nullable=True,
        doc=u"SPARE")

    F1930 = Column(
        Integer(), nullable=True,
        doc=u"SPARE")

    F1931 = Column(
        Integer(), nullable=True,
        doc=u"Redeem on store e-coupon")

    F1932 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Compare price")

    F1933 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"SPARE")

    F1934 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"SPARE")

    F1935 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"SPARE")

    F1936 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Passive discount info")

    F1938 = Column(
        Float(), nullable=True,
        doc=u"Cost plus percent")

    F2551 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Fee 1")

    F2552 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Fee 2")

    F2553 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Fee 3")

    F2554 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Fee 4")

    F2555 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Fee 5")

    F2608 = Column(
        String(length=4), nullable=True,
        doc=u"SPARE")

    F2609 = Column(
        Integer(), nullable=True,
        doc=u"Redeem on store coupon")

    F2610 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"PO Distribution match")

    F2611 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"Price modification")

    F2612 = Column(
        String(length=4), nullable=True,
        doc=u"SPARE")

    F2613 = Column(
        Integer(), nullable=True,
        doc=u"SPARE")

    F2614 = Column(
        Numeric(precision=8, scale=4), nullable=True,
        doc=u"SPARE")

    F2660 = Column(
        String(length=250), nullable=True,
        doc=u"URL")

    F2744 = Column(
        String(length=4), nullable=True,
        doc=u"Point redeem program")

    F2745 = Column(
        String(length=8), nullable=True,
        doc=u"Store E-Coupon method")

    F2746 = Column(
        String(length=8), nullable=True,
        doc=u"Vendor E-Coupon method")

    F2747 = Column(
        Integer(), nullable=True,
        doc=u"Points on vendor e-coupons")

    F2748 = Column(
        Integer(), nullable=True,
        doc=u"Redeem on vendor e-coupon")

    F2749 = Column(
        Integer(), nullable=True,
        doc=u"Redeem on vendor coupon")

    F2750 = Column(
        Integer(), nullable=True,
        doc=u"Points on store coupon")

    F2751 = Column(
        Integer(), nullable=True,
        doc=u"Points on vendor coupon")

    F2752 = Column(
        String(length=8), nullable=True,
        doc=u"Store Coupon method")

    F2753 = Column(
        String(length=8), nullable=True,
        doc=u"Vendor Coupon method")

    def __repr__(self):
        return u"SaleDetail(F1032={0}, F1101={1})".format(
            repr(self.F1032), repr(self.F1101)).encode(u'utf_8')


Sale.details = relationship(
    SaleDetail,
    primaryjoin=SaleDetail.F1032 == Sale.F1032,
    foreign_keys=[SaleDetail.F1032],
    order_by=SaleDetail.F1101,
    backref=u'transaction',
    doc=u"""
List-like collection of :class:`SaleDetail` instances which comprise the
transaction.
""")
