# -*- coding: utf-8 -*-
################################################################################
#
#  pyLOCSMS -- Python Interface to LOC Store Management Suite
#  Copyright © 2013-2014 Lance Edgar
#
#  This file is part of pyLOCSMS.
#
#  pyLOCSMS is free software: you can redistribute it and/or modify it under
#  the terms of the GNU General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  pyLOCSMS is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
#  FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
#  details.
#
#  You should have received a copy of the GNU General Public License along with
#  pyLOCSMS.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################

"""
Data Load Utilities
"""

import os
import sys
import argparse
import tempfile
import zipfile
import shutil

from locsms.deploy import Node


default_names = [
    u'ALT',
    u'BTL',
    u'CAT',
    u'CLF',
    u'CLG',
    u'CLK',
    u'CLL',
    u'CLR',
    u'CLT',
    u'COST',
    u'DEPT',
    u'ECL',
    u'FAM',
    u'KIT',
    u'OBJ',
    u'POS',
    u'PRICE',
    u'RPC',
    u'SCL',
    u'SCL_NUT',
    u'SCL_TXT',
    u'SDP',
    u'VENDOR',
    ]


def generate_load_backup(*args):
    """
    Command to generate a load data backup file.

    It is invoked as ``locsms-load-backup``:

    .. program-output:: locsms-load-backup --help
    """
    parser = argparse.ArgumentParser(prog=u'locsms-load-backup', description=u"""
Generate a zipped backup file containing load data from a SMS node.
""")
    parser.add_argument(u'-n', u'--node', default=ur'C:\Storeman',
                        help=ur"Path to the SMS instance node; defaults to 'C:\Storeman'")
    parser.add_argument(u'zip_file', metavar=u'ZIP_FILE',
                        help=u"Path to output ZIP file")
    args = parser.parse_args(args or sys.argv[1:])

    # TODO: Add command line parameters to include and/or exclude names from
    # the default list.
    load_names = list(default_names)

    node = Node(args.node)
    fd, temp_path = tempfile.mkstemp()
    os.close(fd)

    zip_file = zipfile.ZipFile(temp_path, u'w')
    for name in load_names:
        name = u'{0}_LOAD.SQL'.format(name)
        load_path = node.get_load_path(name)
        zip_file.write(load_path, name)
    zip_file.close()
    shutil.move(temp_path, args.zip_file)


def restore_load_backup(*args):
    """
    Command to restore a load data backup file.

    It is invoked as ``locsms-load-restore``:

    .. program-output:: locsms-load-restore --help
    """
    parser = argparse.ArgumentParser(prog=u'locsms-load-restore', description=u"""
Restore a zipped backup file containing load data into a SMS node.
""")
    parser.add_argument(u'-n', u'--node', metavar=u'PATH', default=ur'C:\Storeman',
                        help=ur"Path to the SMS instance node; defaults to 'C:\Storeman'")
    parser.add_argument(u'-i', u'--inbox', metavar=u'PATH',
                        help=u"Surrogate inbox path to which load files should be deployed.  "
                        u"By default, load files will be deployed to the node's true inbox.")
    parser.add_argument(u'zip_file', metavar=u'ZIP_FILE',
                        help=u"Path to ZIP file which contains load data")
    args = parser.parse_args(args or sys.argv[1:])

    node = Node(args.node, read_config=True)
    surrogate = False
    if args.inbox:
        node.surrogate_inbox_path = args.inbox
        surrogate = True

    temp_path = tempfile.mkdtemp()
    zip_file = zipfile.ZipFile(args.zip_file, u'r')
    for name in zip_file.namelist():
        zip_file.extract(name, temp_path)
        load_path = os.path.join(temp_path, name)
        node.deploy_to_inbox(load_path, surrogate=surrogate)
        os.remove(load_path)
    zip_file.close()
    os.rmdir(temp_path)
