#!/usr/bin/env python
# -*- coding: utf-8  -*-
################################################################################
#
#  pyLOCSMS -- Python Interface to LOC Store Management Suite
#  Copyright © 2013-2014 Lance Edgar
#
#  This file is part of pyLOCSMS.
#
#  pyLOCSMS is free software: you can redistribute it and/or modify it under
#  the terms of the GNU General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  pyLOCSMS is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
#  FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
#  details.
#
#  You should have received a copy of the GNU General Public License along with
#  pyLOCSMS.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################

"""
Deployment Interface
"""

import os
import sys
import shutil
import ConfigParser


class Node(object):
    """
    Represents a node within the SMS system.

    A "node" in this sense refers to any installed "primary instance" of the
    software, e.g. a host or store server, or a lane.  It does *not* refer to a
    backoffice workstation or any other type of "secondary instance".

    :type path: string
    :param path: Root path for the node, e.g. ``u'\\\\server\\storeman'``.

    :type read_config: boolean
    :param read_config: Whether basic configuration should be read from the
       root path upon instantiation.  Note that no read will be attempted
       unless a root path is also provided via the ``path`` parameter.  See
       :meth:`read_config()` for details of what is read.
    """
    store = None
    terminal = None
    path = None
    surrogate_inbox_path = None

    def __init__(self, path=None, read_config=False):
        self.path = path
        if self.path and read_config:
            self.read_config()

    @property
    def name(self):
        """
        The node's "name" in ``STORE:TER`` format, e.g. ``u'001:901'``.
        """
        return u'{0}:{1}'.format(self.store, self.terminal)

    def __unicode__(self):
        return self.name

    def __repr__(self):
        return u"Node(store={0}, terminal={1})".format(
            repr(self.store), repr(self.terminal)).encode(u'utf_8')

    def read_config(self):
        """
        Read the SMS configuration file for the node.

        This method reads the actual SMS configuration directly from the node
        root path, and sets the following attributes on the (Python-side) node
        instance based on the config file contents:

        * :attr:`store`
        * :attr:`terminal`
        """
        if not self.path:
            raise ValueError(u"Node has no root path: {0}".format(unicode(repr(self))))

        ini_path = os.path.join(self.path, u'SMSStart.ini')
        with open(ini_path, u'rb') as ini_file:
            config = ConfigParser.SafeConfigParser()
            config.readfp(ini_file)

        self.store = config.get(u'SMSSTART', u'STORE')
        self.terminal = config.get(u'SMSSTART', u'TER')

    def get_inbox_path(self, filename=None, surrogate=False):
        """
        Returns the full path to the node's "inbox" folder.

        :type filename: string
        :param filename: Optional name of a file which should be appended to
           the path.

        :type surrogate: boolean
        :param surrogate: Optional flag indicating whether the node's
           "surrogate" inbox path should be used instead of its "true" inbox
           path.  Defaults to ``False``.

        :rtype: string
        :returns: If no ``filename`` is provided, the absolute path to the
           node's inbox (true or surrogate) folder will be returned.  If a
           ``filename`` *is* provided, the path returned will include it.
        """
        if surrogate:
            if self.surrogate_inbox_path:
                inbox_path = self.surrogate_inbox_path
            else:
                raise ValueError(u"Node has no surrogate inbox path defined: {0}".format(unicode(repr(self))))
        else:
            if not self.path:
                raise ValueError(u"Node has no root path defined: {0}".format(unicode(repr(self))))
            if not self.store:
                raise ValueError(u"Node has no store defined: {0}".format(unicode(repr(self))))
            if not self.terminal:
                raise ValueError(u"Node has no terminal defined: {0}".format(unicode(repr(self))))
            inbox_path = os.path.join(
                self.path, u'Office', u'XF{0}{1}'.format(self.store, self.terminal))

        if filename:
            return os.path.join(inbox_path, filename)
        else:
            return inbox_path

    def deploy_to_inbox(self, path, surrogate=False):
        """
        Deploys a file to the node's inbox folder.

        If ``surrogate`` is ``False`` (the default), then in addition to
        copying the file to the node inbox, the archive bit on the destination
        file will be turned off (via :func:`flip_archive_bit()`).  This will
        cause the node's Launchpad to process the file immediately (assuming it
        is running).

        .. note::
           Fiddling with the archive bit is not (currently?) possible if this
           call is made from a Linux system, as the Windows API is required.
           Therefore if ``surrogate`` *is* ``False``, and the calling system is
           Linux, the file will be deployed to the true inbox folder but no
           attempt will be made to flip the bit.

        :type path: string
        :param path: Path to a file which is to be deployed to the node inbox.

        :type surrogate: boolean
        :param surrogate: Optional flag which determines if the destination
           should be the node's surrogate inbox instead of its true inbox.
           Defaults to ``False``.
        """
        dest = self.get_inbox_path(filename=os.path.basename(path), surrogate=surrogate)
        shutil.copy(path, dest)
        if not surrogate and sys.platform == u'win32':
            flip_archive_bit(dest)

    def get_load_path(self, filename=None):
        """
        Returns the full path to the node's "load" folder.

        :type filename: string
        :param filename: Optional name of a file which should be appended to
           the path.

        :rtype: string
        :returns: If no ``filename`` is provided, the absolute path to the
           node's load folder will be returned.  If a ``filename`` *is*
           provided, the path returned will include it.
        """
        if not self.path:
            raise ValueError(u"Node does not have a root path defined: {0}".format(unicode(repr(self))))
        load_path = os.path.join(self.path, u'Office', u'Load')
        if filename:
            return os.path.join(load_path, filename)
        else:
            return load_path


def flip_archive_bit(path): # pragma no cover
    """
    Turn off the "File is ready for archiving" attribute for a file.

    .. note::
       This function will only work if called from a Windows system, because
       the current implementation uses the Windows API.

    :type path: string
    :param path: Path to the file whose archive bit should be turned off.
    """
    import win32api
    import win32con

    attrs = win32api.GetFileAttributes(path)
    win32api.SetFileAttributes(path, attrs &~ win32con.FILE_ATTRIBUTE_ARCHIVE)


def deploy_to_inbox(path, node_path):
    """
    Deploys a file to a SMS node's inbox folder.

    This is a convenience function, intended for use with the Rattail File
    Monitor.  It merely copies a file a SMS node inbox folder, and then flips
    the file's archive bit so that SMS Launchpad will process it.

    :type path: string
    :param path: Path to a file which is to be deployed to the node inbox.

    :type node_path: string
    :param node_path: Full path to the node's root folder,
       e.g. ``'C:\storeman'``.
    """
    node = Node(node_path, read_config=True)
    node.deploy_to_inbox(path)
