#!/usr/bin/env python
# -*- coding: utf-8  -*-
################################################################################
#
#  pyLOCSMS -- Python Interface to LOC Store Management Suite
#  Copyright © 2013-2014 Lance Edgar
#
#  This file is part of pyLOCSMS.
#
#  pyLOCSMS is free software: you can redistribute it and/or modify it under
#  the terms of the GNU General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  pyLOCSMS is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
#  FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
#  details.
#
#  You should have received a copy of the GNU General Public License along with
#  pyLOCSMS.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################

"""
Fields CSV Utilities
"""

import csv


def get_field_map(csv_path):
    """
    Parse a ``Fields.csv`` file to obtain its contents.

    This parses a fields CSV file (such as is typically found at
    ``C:\\Storeman\\Fields.csv``), and returns a dictionary keyed by field
    name.  Each element in turn contains data for a particular field, keyed by
    the headers within the CSV file itself.  The structure will look something
    like this::

       {
           u'F01' : {
               u'HOT'           : u'1',
               u'FIELD'         : u'F01',
               u'TYPE'          : u'C',
               u'SIZE'          : u'13',
               u'DEC'           : u'',
               u'DESCRIPTOR'    : u'UPC code',
               # etc...
               },

           # etc...
           }

    :type csv_path: string
    :param csv_path: Path to the fields CSV file.

    :rtype: dict
    :returns: Contents of the CSV file, structured as shown above.
    """
    fields = {}
    with open(csv_path, u'rb') as csv_file:
        reader = csv.DictReader(csv_file)
        for row in reader:
            fields[row[u'FIELD']] = row
    return fields
