"""
Simulator class providing the top-level interface to all the methods in the
framework.

@author: Jerker Nordh
"""

import numpy
from filter import ParticleTrajectory

class Simulator():
    """
    Class interfacing filters/smoothers to assisst in solving estimation problem

    Args:
     - model: object of class describing problem type
     - u (array-like): inputs, first dimension is the time index, the rest is specific
       to the particlar model class being used
     - y (array-like):  measurements, first dimension is the time index, the rest is specific
       to the particlar model class being used
    """

    def __init__(self, model, u, y):
        if (u != None):
            self.u = u
        else:
            self.u = [None] * len(y)
        self.y = y
        self.pt = None
        self.straj = None
        self.params = None
        self.model = model

    def set_params(self, params):
        """
        Set the parameters of the model (if any)

        Args:
         - params (array-like): Model specific paremeters
        """
        self.params = numpy.copy(params)
        self.model.set_params(self.params)

    def simulate(self, num_part, num_traj, filter='PF', smoother='full',
                 smoother_options=None, res=0.67, meas_first=False):
        """
        Solve the estimation problem

        Args:
         - num_part (int): Number of particles used in the forward filter.
         - num_traj (int): Number of backward trajectories generated by the smoother.
         - filter (string): The filter algorithm to use
         - smooter (string): The smoothing algorithm to use
         - smoother_options (dict): algorithm specific smoother options
         - res (float): resampling threshold for the forward filter
         - meas_first (bool): Is the first measurement of the initial state
           (true) or after the first time update? (false)

        Supported filters:
            - 'pf': regular particle filter
            - 'apf': auxilliary particle filter

        Supported smoothers:
            - 'ancestor': return forward trajectories from particle filtier
              (no extra smoothing step)
            - 'full': Backward simulation evaluating all particle weights
            - 'rs': Rejection sampling
            - 'rses': Rejection sampling with early stopping.
               Options:
                - R: number of rejection sampling steps before
                  falling back to 'full'
            - 'rsas': Rejection sampling with early stopping.
               Options:
                - x1 (float): (default is 1.0)
                - P1 (float): (default is 1.0)
                - sv (float): (default is 1.0)
                - sw (float): (default is 1.0)
                - ratio (float): (default is 1.0)
            - 'mcmc': Metropolis-Hastings FFBSi
               Options:
                - R: number of iterations to run the Markov chain
            - 'mhips': Metropolis-Hastings Improved Particle Smoother
               Options:
                - R: number of passes of the dataset to run the algortithms
            - 'mhbp': Metropolis-Hastings Backward Proposer
               Options:
                - R: the number of iterations to run the Markov chain for each
                  time step
        """
        resamplings = 0

        # Initialise a particle filter with our particle approximation of the initial state,
        # set the resampling threshold to 0.67 (effective particles / total particles )
        self.pt = ParticleTrajectory(self.model, num_part, res, filter=filter)

        offset = 0
        # Run particle filter
        if (meas_first):
            self.pt.measure(self.y[0])
            offset = 1
        for i in range(offset, len(self.y)):
            # Run PF using noise corrupted input signal
            if (self.pt.forward(self.u[i - offset], self.y[i])):
                resamplings = resamplings + 1

        # Use the filtered estimates above to created smoothed estimates
        if (smoother != None):
            self.straj = self.pt.perform_smoothing(num_traj, method=smoother,
                                                   smoother_options=smoother_options)
        return resamplings

    def get_filtered_estimates(self):
        """
        Returns type (est, w) (must first have called 'simulate')
         - est: (T, N, D) array containing all particles
         - w: (T,D) array containing all particle weights

        T is the length of the dataset, N is the number of particles and
        D is the dimension of each particle
        """

        T = len(self.pt.traj)
        N = self.pt.traj[0].pa.part.shape[0]
        D = self.pt.traj[0].pa.part.shape[1]

        est = numpy.empty((T, N, D))

        w = numpy.empty((T, N))

        for t in xrange(T):
            wtmp = numpy.exp(self.pt.traj[t].pa.w)
            w[t] = wtmp / numpy.sum(wtmp)
            est[t] = self.pt.traj[t].pa.part

        return (est, w)

    def get_filtered_mean(self):
        """
        Calculate mean of filtered estimates (must first have
        called 'simulate')

        Returns:
         - (T, D) array

        T is the length of the dataset, N is the number of particles and
        D is the dimension of each particle
        """
        (est, w) = self.get_filtered_estimates()

        T = len(self.pt.traj)
        D = self.pt.traj[0].pa.part.shape[1]

        mean = numpy.empty((T, D))
        for t in xrange(T):
            mean[t] = numpy.sum((w[t].ravel() * est[t].T).T, 0)

        return mean
    def get_smoothed_estimates(self):
        """
        Return smoothed estimates (must first have called 'simulate')

        Returns:
         - (T, N, D) array

        T is the length of the dataset,
        N is the number of particles
        D is the dimension of each particle
        """
        return self.straj.traj

    def get_smoothed_mean(self):
        """
        Calculate mean of smoothed estimates (must first have
        called 'simulate')

        Returns:
         - (T, D) array

        T is the length of the dataset, N is the number of particles and
        D is the dimension of each particle
        """
        return numpy.mean(self.straj.traj, 1)
