#!/usr/bin/env python
# -*- coding: utf-8  -*-
################################################################################
#
#  pyScanMaster -- Python Interface to ScanMaster POS
#  Copyright © 2013 Sacramento Natural Foods Co-op, Inc
#
#  This file is part of pyScanMaster.
#
#  pyScanMaster is free software: you can redistribute it and/or modify it
#  under the terms of the GNU General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  pyScanMaster is distributed in the hope that it will be useful, but WITHOUT
#  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
#  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
#  more details.
#
#  You should have received a copy of the GNU General Public License along with
#  pyScanMaster.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################

"""
Database Models
"""

from sqlalchemy.ext.declarative import declarative_base
from sqlalchemy import Column
from sqlalchemy import (
    String, Integer, SmallInteger, Date, Time, Numeric, Boolean)
from sqlalchemy.orm import relationship
from sqlalchemy.ext.associationproxy import association_proxy

import datetime
from decimal import Decimal


__all__ = ['Customer', 'EJTransaction', 'EJCustomer', 'EJItem', 'EJPromotion',
           'EJDiscount', 'EJTax', 'ChargeHistory']


Base = declarative_base()


# NOTE: The following comments and code appeared to be necessary at some point,
# but I have since been able to access a ScanMaster database just fine on Linux
# without it.  I used the Pervasive `TranslationOption` setting which may have
# made the difference (?).

# # There seems to be some sort of Unicode issue with (at least) the Linux /
# # Python 2.7 / pyodbc / Pervasive combination.  Not sure "whose fault" it is
# # exactly but I was seeing some nasty premature application death.  The only
# # thing I found to resolve the issue was to force SQLAchemy to convert to/from
# # Unicode and keep things "safe" on the database side.
# def String(*args, **kwargs):
#     kwargs.setdefault('convert_unicode', 'force')
#     return SA_String(*args, **kwargs)


class Customer(Base):
    """
    Represents a customer.
    """

    __tablename__ = 'Customer'

    account_number = Column(
        'AccountNumber', String(length=12), primary_key=True, nullable=False)

    last_name = Column(
        'LastName', String(length=25), nullable=False, default='')

    first_name = Column(
        'FirstName', String(length=25), nullable=False, default='')

    address = Column(
        'Address', String(length=25), nullable=False, default='')

    city = Column(
        'City', String(length=15), nullable=False, default='')

    state = Column(
        'State', String(length=2), nullable=False, default='')

    zipcode = Column(
        'Zipcode', String(length=9), nullable=False, default='')

    phone_number = Column(
        'PhoneNumber', String(length=10), nullable=False, default='')

    social_security_number = Column(
        'SSNumber', String(length=9), nullable=False, default='000000000')

    _DateOpened = Column(
        'DateOpened', String(length=6), nullable=False, default='000000')

    def _get_date_opened(self):
        if self._DateOpened is None:
            return None
        if self._DateOpened == '000000':
            return None
        return datetime.datetime.strptime(self._DateOpened, '%m%d%y').date()

    def _set_date_opened(self, date):
        if date is None:
            self._DateOpened = '000000'
        else:
            self._DateOpened = date.strftime('%m%d%y')

    date_opened = property(_get_date_opened, _set_date_opened)

    tax_exempt_code = Column(
        'TaxExemptCode', String(length=12), nullable=False, default='000000000000')

    checking_account_number = Column(
        'CheckingAcctNum', String(length=12), nullable=False, default='000000000000')

    number_visits_to_date = Column(
        'NumVisitsTD', String(length=3), nullable=False, default='000')

    number_visits_period_to_date = Column(
        'NumVisitsPTD', String(length=3), nullable=False, default='000')

    purchase_amount_to_date = Column(
        'PurchaseAmountTD', String(length=9), nullable=False, default='000000000')

    purchase_amount_period_to_date = Column(
        'PurchaseAmountPTD', String(length=9), nullable=False, default='000000000')

    _DOLPurchase = Column(
        'DOLPurchase', String(length=6), nullable=False, default='000000')

    def _get_date_of_last_purchase(self):
        if self._DOLPurchase is None:
            return None
        if self._DOLPurchase == '000000':
            return None
        return datetime.datetime.strptime(self._DOLPurchase, '%m%d%y').date()

    def _set_date_of_last_purchase(self, date):
        if date is None:
            self._DOLPurchase = '000000'
        else:
            self._DOLPurchase = date.strftime('%m%d%y')

    date_of_last_purchase = property(_get_date_of_last_purchase, _set_date_of_last_purchase)

    comments = Column(
        'Comments', String(length=30), nullable=False, default='')

    override_count = Column(
        'OverrideCount', String(length=3), nullable=False, default='000')

    account_status = Column(
        'AccountStatus', String(length=1), nullable=False, default='0')

    frequent_shopper_dollars_to_date = Column(
        'FSDollarsTD', String(length=9), nullable=False, default='000000000')

    frequent_shopper_dollars_period_to_date = Column(
        'FSDollarsPTD', String(length=9), nullable=False, default='000000000')

    number_checks_today = Column(
        'NumChecksToday', String(length=1), nullable=False, default='0')

    number_checks_this_week = Column(
        'NumChecksThisWeek', String(length=1), nullable=False, default='0')

    amount_checks_today = Column(
        'AmtChecksToday', String(length=9), nullable=False, default='000000000')

    amount_checks_this_week = Column(
        'AmtChecksThisWeek', String(length=9), nullable=False, default='000000000')

    record_changed = Column(
        'RecordChanged', String(length=1), nullable=False, default='0')

    # TODO: Give this column a default value?  Not sure what its meaning is yet.
    update_flag = Column(
        'UpdateFlag', String(length=1), nullable=False)

    # TODO: Give this column a default value?  Might need to be business-specific?
    frequent_shopper_level = Column(
        'FSLevel', String(length=1), nullable=False)

    points_period_to_date = Column(
        'PointsPTD', String(length=9), nullable=False, default='000000000')

    points_to_date = Column(
        'PointsTD', String(length=9), nullable=False, default='000000000')

    bonus_points_period_to_date = Column(
        'BonusPointsPTD', String(length=9), nullable=False, default='000000000')

    bonus_points_to_date = Column(
        'BonusPointsTD', String(length=9), nullable=False, default='000000000')

    special_promotion_points = Column(
        'SpecialPromoPoints', String(length=9), nullable=False, default='000000000')

    frequent_shopper_discounts_to_date = Column(
        'FSDiscountsTD', String(length=9), nullable=False, default='000000000')

    frequent_shopper_discounts_period_to_date = Column(
        'FSDiscountsPTD', String(length=9), nullable=False, default='000000000')

    electronic_coupon_to_date = Column(
        'ElectronicCouponTD', String(length=9), nullable=False, default='000000000')

    electronic_coupon_period_to_date = Column(
        'ElectronicCouponPTD', String(length=9), nullable=False, default='000000000')

    auto_discount1_flag = Column(
        'AutoDisc1Flag', String(length=1), nullable=False, default='0')

    auto_discount2_flag = Column(
        'AutoDisc2Flag', String(length=1), nullable=False, default='0')

    auto_discount4_flag = Column(
        'AutoDisc4Flag', String(length=1), nullable=False, default='0')

    auto_discount5_flag = Column(
        'AutoDisc5Flag', String(length=1), nullable=False, default='0')

    cash_only = Column(
        'CashOnly', String(length=1), nullable=False, default='0')

    check_account_number = Column(
        'ChkAcctNum', String(length=20), nullable=False, default='00000000000000000000')

    transit_number = Column(
        'TransitNum', String(length=9), nullable=False, default='000000000')

    blanks = Column(
        'Blanks', String(length=12), nullable=True, default='000000000000')

    def __repr__(self):
        return "Customer(account_number={0})".format(
            repr(self.account_number))


class CustomerExtended(Base):
    """
    Represents extended information about a customer.
    """

    __tablename__ = 'CustomerExtended'

    account_number = Column(
        'AccountNumber', String(length=12), primary_key=True, nullable=True)

    address2 = Column(
        'Address2', String(length=25), nullable=True, default='')

    email = Column(
        'Email1', String(length=50), nullable=True, default='')

    email2 = Column(
        'Email2', String(length=50), nullable=True, default='')

    ship_to_address = Column(
        'ShipTo1', String(length=25), nullable=True, default='')

    ship_to_address2 = Column(
        'ShipTo2', String(length=25), nullable=True, default='')

    ship_to_city = Column(
        'City', String(length=15), nullable=True, default='')

    ship_to_state = Column(
        'State', String(length=2), nullable=True, default='')

    ship_to_zipcode = Column(
        'Zip', String(length=9), nullable=True, default='')

    contact_last_name = Column(
        'ContactLastName', String(length=25), nullable=True, default='')

    contact_first_name = Column(
        'ContactFirstName', String(length=25), nullable=True, default='')

    phone_number2 = Column(
        'Phone2', String(length=10), nullable=True, default='')

    limit_checks_today = Column(
        'LimitChecksToday', String(length=1), nullable=True, default='0')

    limit_checks_this_week = Column(
        'LimitChecksWeek', String(length=1), nullable=True, default='0')

    limit_dollars_today = Column(
        'LimitDollarDay', String(length=9), nullable=True, default='000000000')

    limit_dollars_this_week = Column(
        'LimitDollarWeek', String(length=9), nullable=True, default='000000000')

    memo = Column(
        'Memo', String(length=100), nullable=True, default='')

    drivers_license_number = Column(
        'DriversLicNum', String(length=18), nullable=True, default='')

    drivers_license_state = Column(
        'DriversLicState', String(length=2), nullable=True, default='')

    alternate_lookup = Column(
        'AltLookUp', String(length=20), nullable=True, default='00000000000000000000')

    blanks = Column(
        'Blanks', String(length=58), nullable=True, default='0000000000000000000000000000000000000000000000000000000000')

    def __repr__(self):
        return "CustomerExtended(account_number={0})".format(
            repr(self.account_number))


Customer.extended = relationship(
    CustomerExtended,
    primaryjoin=CustomerExtended.account_number == Customer.account_number,
    foreign_keys=[CustomerExtended.account_number],
    uselist=False,
    backref='customer')

def getset_factory(collection_class, proxy):
    def getter(obj):
        if obj is None:
            return ''
        return getattr(obj, proxy.value_attr)
    def setter(obj, val):
        setattr(obj, proxy.value_attr, val)
    return getter, setter

def extended_proxy(attr):
    def creator(val):
        obj = CustomerExtended()
        setattr(obj, attr, val)
        return obj
    proxy = association_proxy(
        'extended', attr, creator=creator,
        getset_factory=getset_factory)
    setattr(Customer, attr, proxy)

extended_proxy('address2')
extended_proxy('email')
extended_proxy('email2')
extended_proxy('ship_to_address')
extended_proxy('ship_to_address2')
extended_proxy('ship_to_city')
extended_proxy('ship_to_state')
extended_proxy('ship_to_zipcode')
extended_proxy('contact_last_name')
extended_proxy('contact_first_name')
extended_proxy('phone_number2')
extended_proxy('limit_checks_today')
extended_proxy('limit_checks_this_week')
extended_proxy('limit_dollars_today')
extended_proxy('limit_dollars_this_week')
extended_proxy('memo')
extended_proxy('drivers_license_number')
extended_proxy('drivers_license_state')
extended_proxy('alternate_lookup')


class EJTransaction(Base):
    """
    Represents an electronic journal transaction.
    """

    __tablename__ = 'tabEJTransaction'

    transaction_id = Column(
        'TransactionID', Integer(), primary_key=True, nullable=False)

    transaction_number = Column(
        'TransactionNum', String(length=8), nullable=False)

    cashier_id = Column(
        'CashierID', SmallInteger(), nullable=False)

    cashier_name = Column(
        'CashierName', String(length=20), nullable=True)

    register_number = Column(
        'RegisterNum', SmallInteger(), nullable=False)

    start_date = Column(
        'StartDate', Date(), nullable=False)

    end_date = Column(
        'EndDate', Date(), nullable=False)

    start_time = Column(
        'StartTime', Time(), nullable=False)

    end_time = Column(
        'EndTime', Time(), nullable=False)

    @property
    def started(self):
        """
        The date and time when the transaction started.
        """
        if self.start_date is None:
            return self.start_time
        if self.start_time is None:
            return self.start_date
        return datetime.datetime.combine(self.start_date, self.start_time)

    @property
    def ended(self):
        """
        The date and time when the transaction ended.
        """
        if self.end_date is None:
            return self.end_time
        if self.end_time is None:
            return self.end_date
        return datetime.datetime.combine(self.end_date, self.end_time)

    transaction_type = Column(
        'TransactionType', SmallInteger(), nullable=False)

    shoppers_hotline = Column(
        'ShoppersHotline', String(length=12), nullable=True)

    transaction_amount = Column(
        'TransactionAmt', Numeric(precision=19, scale=4), nullable=True)

    item_subtotal_amount = Column(
        'ItemSubtotalAmt', Numeric(precision=19, scale=4), nullable=True)

    total_discount_amount = Column(
        'TotalDiscAmt', Numeric(precision=19, scale=4), nullable=True)

    total_void_amount = Column(
        'TotalVoidAmt', Numeric(precision=19, scale=4), nullable=True)

    total_return_amount = Column(
        'TotalReturnAmt', Numeric(precision=19, scale=4), nullable=True)

    total_tt_discount_amount = Column(
        'TotalTTDiscAmt', Numeric(precision=19, scale=4), nullable=True)

    number_of_items = Column(
        'NumberOfItems', SmallInteger(), nullable=True)

    original_transaction_number = Column(
        'OrigTransactionNum', String(length=8), nullable=True)

    retrieved_flag = Column(
        'RetrievedFlag', Boolean(), nullable=False)

    pf_recovery_flag = Column(
        'PFRecoveryFlag', Boolean(), nullable=False)

    cancel_flag = Column(
        'CancelFlag', Boolean(), nullable=False)

    suspend_flag = Column(
        'SuspendFlag', Boolean(), nullable=False)

    online_flag = Column(
        'OnlineFlag', SmallInteger(), nullable=True)

    register_version = Column(
        'RegisterVer', String(length=15), nullable=True)

    transaction_seconds = Column(
        'TransSeconds', Integer(), nullable=True)

    idle_seconds = Column(
        'IdleSeconds', Integer(), nullable=True)

    ring_seconds = Column(
        'RingSeconds', Integer(), nullable=True)

    tender_seconds = Column(
        'TenderSeconds', Integer(), nullable=True)

    scan_rate = Column(
        'ScanRate', Numeric(precision=10, scale=2), nullable=True)

    scanned_entries = Column(
        'ScannedEntries', SmallInteger(), nullable=True)

    keyed_entries = Column(
        'KeyedEntries', SmallInteger(), nullable=True)

    open_department_entries = Column(
        'OpenDeptEntries', SmallInteger(), nullable=True)

    aff_flag = Column(
        'AFFFlag', SmallInteger(), nullable=True)

    def __repr__(self):
        return "EJTransaction(transaction_id={0}, transaction_number={1})".format(
            repr(self.transaction_id), repr(self.transaction_number))


class EJCustomer(Base):
    """
    Represents a customer association with an EJ transaction.
    """

    __tablename__ = 'tabEJCustomer'

    transaction_id = Column(
        'TransactionID', Integer(), primary_key=True, nullable=False)

    customer_number = Column(
        'CustomerNum', String(length=12), primary_key=True, nullable=False)

    last_name = Column(
        'LastName', String(length=25), nullable=True)

    first_name = Column(
        'FirstName', String(length=25), nullable=True)

    restricted_action_flag = Column(
        'RestrictedActionFlag', SmallInteger(), nullable=True)

    approval_mode = Column(
        'ApprovalMode', SmallInteger(), nullable=True)

    birthdate = Column(
        'Birthdate', Date(), nullable=True)

    verification_id = Column(
        'VerificationID', String(length=20), nullable=True)

    origin = Column(
        'Origin', SmallInteger(), nullable=True)

    frequent_shopper_level = Column(
        'FreqShopLevel', SmallInteger(), nullable=True)

    def __repr__(self):
        return "EJCustomer(transaction_id={0}, customer_number={1})".format(
            repr(self.transaction_id), repr(self.customer_number))


EJCustomer.customer = relationship(
    Customer,
    primaryjoin=Customer.account_number == EJCustomer.customer_number,
    foreign_keys=[Customer.account_number],
    uselist=False)

EJTransaction.customers = relationship(
    EJCustomer,
    primaryjoin=EJCustomer.transaction_id == EJTransaction.transaction_id,
    foreign_keys=[EJCustomer.transaction_id],
    backref='transaction')


class EJItem(Base):
    """
    Represents a line item within an EJ transaction.
    """

    __tablename__ = 'tabEJItem'

    transaction_id = Column(
        'TransactionID', Integer(), primary_key=True, nullable=False)

    sequence_number = Column(
        'SequenceNum', Integer(), primary_key=True, nullable=False)

    item_type = Column(
        'ItemType', SmallInteger(), nullable=False)

    upc = Column(
        'UPC', String(length=20), nullable=False)

    item_description = Column(
        'ItemDesc', String(length=20), nullable=True)

    department_number = Column(
        'DepartmentNum', SmallInteger(), nullable=False)

    subdepartment_number = Column(
        'SubDeptNum', SmallInteger(), nullable=True)

    quantity = Column(
        'Quantity', Integer(), nullable=False)

    adjective_quantity = Column(
        'AdjectiveQty', Integer(), nullable=True)

    adjective_level = Column(
        'AdjectiveLevel', SmallInteger(), nullable=True)

    sold_at_price = Column(
        'SoldAtPrice', Numeric(precision=19, scale=4), nullable=True)

    discount_amount = Column(
        'DiscountAmt', Numeric(precision=19, scale=4), nullable=True)

    bottle_deposit_amount = Column(
        'BottleDepAmt', Numeric(precision=19, scale=4), nullable=True)

    original_quantity = Column(
        'OriginalQty', Integer(), nullable=True)

    original_price = Column(
        'OriginalPrice', Numeric(precision=19, scale=4), nullable=True)

    regular_price = Column(
        'RegularPrice', Numeric(precision=19, scale=4), nullable=True)

    food_stamp_status = Column(
        'FoodStampStatus', Boolean(), nullable=False)

    tax1_status = Column(
        'Tax1Status', Boolean(), nullable=False)

    tax2_status = Column(
        'Tax2Status', Boolean(), nullable=False)

    tax3_status = Column(
        'Tax3Status', Boolean(), nullable=False)

    discount1_status = Column(
        'Disc1Status', Boolean(), nullable=False)

    discount2_status = Column(
        'Disc2Status', Boolean(), nullable=False)

    discount4_status = Column(
        'Disc4Status', Boolean(), nullable=False)

    discount5_status = Column(
        'Disc5Status', Boolean(), nullable=False)

    frequent_shopper_status = Column(
        'FreqShopperStatus', Boolean(), nullable=False)

    sale_level = Column(
        'SaleLevel', SmallInteger(), nullable=True)

    bottle_link_department_number = Column(
        'BottleLinkDeptNum', SmallInteger(), nullable=True)

    scale_weight = Column(
        'ScaleWeight', Numeric(precision=5, scale=3), nullable=True)

    split_weight = Column(
        'SplitWeight', Numeric(precision=6, scale=3), nullable=True)

    tare_weight = Column(
        'TareWeight', Numeric(precision=5, scale=3), nullable=True)

    tare_code = Column(
        'TareCode', SmallInteger(), nullable=True)

    commodity_code = Column(
        'CommodityCode', String(length=8), nullable=True)

    reason_code = Column(
        'ReasonCode', SmallInteger(), nullable=True)

    report_code = Column(
        'ReportCode', String(length=4), nullable=True)

    online_flag = Column(
        'OnlineFlag', Boolean(), nullable=False)

    void_flag = Column(
        'VoidFlag', Boolean(), nullable=False)

    error_correct_flag = Column(
        'ErrorCorrectFlag', Boolean(), nullable=False)

    scan_flag = Column(
        'ScanFlag', Boolean(), nullable=False)

    override_flag = Column(
        'OverrideFlag', Boolean(), nullable=False)

    return_flag = Column(
        'ReturnFlag', Boolean(), nullable=False)

    refund_flag = Column(
        'RefundFlag', Boolean(), nullable=False)

    not_found_flag = Column(
        'NotFoundFlag', Boolean(), nullable=False)

    suspended_flag = Column(
        'SuspendedFlag', Boolean(), nullable=False)

    retrieved_flag = Column(
        'RetrievedFlag', Boolean(), nullable=False)

    canceled_flag = Column(
        'CanceledFlag', Boolean(), nullable=False)

    unit_price = Column(
        'UnitPrice', Numeric(precision=19, scale=4), nullable=True)

    case_cost = Column(
        'CaseCost', Numeric(precision=19, scale=4), nullable=True)

    case_quantity = Column(
        'CaseQty', Integer(), nullable=True)

    mix_match_code = Column(
        'MixMatchCode', Integer(), nullable=True)

    coupon_code = Column(
        'CouponCode', Integer(), nullable=True)

    fsa_rx_flag = Column(
        'FSARxFlag', Boolean(), nullable=False)

    fsa_non_rx_flag = Column(
        'FSANonRxFlag', Boolean(), nullable=False)

    promotion_id = Column(
        'PromoID', String(length=20), nullable=True)

    def __repr__(self):
        return "EJItem(transaction_id={0}, sequence_number={1})".format(
            repr(self.transaction_id), repr(self.sequence_number))


EJTransaction.items = relationship(
    EJItem,
    primaryjoin=EJItem.transaction_id == EJTransaction.transaction_id,
    foreign_keys=[EJItem.transaction_id],
    order_by=EJItem.sequence_number,
    backref='transaction')


class EJPromotion(Base):
    """
    Represents a promotion within an EJ transaction.
    """

    __tablename__ = 'tabEJPromotions'

    transaction_id = Column(
        'TransactionID', Integer(), primary_key=True, nullable=False)

    sequence_number = Column(
        'SequenceNum', Integer(), primary_key=True, nullable=False)

    promotion_number = Column(
        'PromotionNum', String(length=20), nullable=True)

    promotion_type = Column(
        'PromoType', SmallInteger(), nullable=True)

    quantity = Column(
        'Quantity', Integer(), nullable=True)

    amount = Column(
        'Amount', Numeric(precision=19, scale=4), nullable=True)

    department_number = Column(
        'DepartmentNum', SmallInteger(), nullable=True)

    frequent_shopper_level = Column(
        'FreqShopperLevel', SmallInteger(), nullable=True)

    coupon_trigger = Column(
        'CouponTrigger', SmallInteger(), nullable=True)

    points = Column(
        'Points', Integer(), nullable=True)

    bonus_points = Column(
        'BonusPoints', Integer(), nullable=True)

    promotion_description = Column(
        'PromoDesc', String(length=30), nullable=True)

    multimedia_filename = Column(
        'MultiMediaFN', String(length=35), nullable=True)

    void_flag = Column(
        'VoidFlag', Boolean(), nullable=False)

    error_correct_flag = Column(
        'ErrorCorrectFlag', Boolean(), nullable=False)

    def __repr__(self):
        return "EJPromotion(transaction_id={0}, sequence_number={1})".format(
            repr(self.transaction_id), repr(self.sequence_number))


EJTransaction.promotions = relationship(
    EJPromotion,
    primaryjoin=EJPromotion.transaction_id == EJTransaction.transaction_id,
    foreign_keys=[EJPromotion.transaction_id],
    order_by=EJPromotion.sequence_number,
    backref='transaction')


class EJDiscount(Base):
    """
    Represents a discount on an EJ transaction.
    """

    __tablename__ = 'tabEJDiscount'

    transaction_id = Column(
        'TransactionID', Integer(), primary_key=True, nullable=False)

    sequence_number = Column(
        'SequenceNum', Integer(), primary_key=True, nullable=False)

    discount_number = Column(
        'DiscountNum', SmallInteger(), nullable=False)

    discount_type = Column(
        'DiscountType', SmallInteger(), nullable=True)

    discount_amount = Column(
        'DiscountAmt', Numeric(precision=19, scale=4), nullable=True)

    def __repr__(self):
        return "EJDiscount(transaction_id={0}, sequence_number={1})".format(
            repr(self.transaction_id), repr(self.sequence_number))


EJTransaction.discounts = relationship(
    EJDiscount,
    primaryjoin=EJDiscount.transaction_id == EJTransaction.transaction_id,
    foreign_keys=[EJDiscount.transaction_id],
    order_by=EJDiscount.sequence_number,
    backref='transaction')


class EJTax(Base):
    """
    Represents the tax on an EJ transaction.
    """

    __tablename__ = 'tabEJTax'

    transaction_id = Column(
        'TransactionID', Integer(), primary_key=True, nullable=False)

    tax_number = Column(
        'TaxNum', SmallInteger(), primary_key=True, nullable=False)

    tax_amount = Column(
        'TaxAmt', Numeric(precision=19, scale=4), nullable=True)

    taxable_sales_amount = Column(
        'TaxableSalesAmt', Numeric(precision=19, scale=4), nullable=True)

    food_stamp_taxable_sales_amount = Column(
        'FSTaxableSalesAmt', Numeric(precision=19, scale=4), nullable=True)

    food_stamp_tax_amount = Column(
        'FSTaxAmt', Numeric(precision=19, scale=4), nullable=True)

    tax_exempt_amount = Column(
        'TaxExemptAmt', Numeric(precision=19, scale=4), nullable=True)

    tax_exempt_code = Column(
        'TaxExemptCode', String(length=12), nullable=True)

    def __repr__(self):
        return "EJTax(transaction_id={0}, tax_number={1})".format(
            repr(self.transaction_id), repr(self.tax_number))


EJTransaction.taxes = relationship(
    EJTax,
    primaryjoin=EJTax.transaction_id == EJTransaction.transaction_id,
    foreign_keys=[EJTax.transaction_id],
    order_by=EJTax.tax_number,
    backref='transaction')


class ChargeHistory(Base):
    """
    Represents a charge event within a customer account's history.
    """

    __tablename__ = 'ChargeHistory'

    account_number = Column(
        'ACCOUNTNUMBER', String(length=12), primary_key=True, nullable=True)

    transaction_number = Column(
        'TRANSACTIONNUMBER', String(length=8), primary_key=True, nullable=True)

    _BUSINESSDATE = Column(
        'BUSINESSDATE', String(length=6), nullable=True)

    def _get_business_date(self):
        if self._BUSINESSDATE is None:
            return None
        return datetime.datetime.strptime(self._BUSINESSDATE, '%y%m%d').date()

    def _set_business_date(self, date):
        if date is None:
            self._BUSINESSDATE = None
        else:
            self._BUSINESSDATE = date.strftime('%y%m%d')

    business_date = property(_get_business_date, _set_business_date)

    _BUSINESSTIME = Column(
        'BUSINESSTIME', String(length=4), nullable=True)

    def _get_business_time(self):
        if self._BUSINESSTIME is None:
            return None
        return datetime.datetime.strptime(self._BUSINESSTIME, '%H%M').time()

    def _set_business_time(self, time):
        if time is None:
            self._BUSINESSTIME = None
        else:
            self._BUSINESSTIME = time.strftime('%H%M')

    business_time = property(_get_business_time, _set_business_time)

    @property
    def occurred(self):
        """
        The date and time when the charge occurred.
        """

        if self.business_date is None:
            return self.business_time
        if self.business_time is None:
            return self.business_date
        return datetime.datetime.combine(self.business_date, self.business_time)

    store_number = Column(
        'STORENUMBER', String(length=4), nullable=True)

    register_number = Column(
        'REGISTERNUMBER', String(length=2), nullable=True)

    cashier_number = Column(
        'CASHIERNUMBER', String(length=3), nullable=True)

    activity_code = Column(
        'ACTIVITYCODE', String(length=1), nullable=True)

    _AMOUNT = Column(
        'AMOUNT', String(length=8), nullable=True)

    def _get_amount(self):
        if self._AMOUNT is None:
            return None
        return Decimal('{0}.{1}'.format(self._AMOUNT[:-2], self._AMOUNT[-2:]))

    def _set_amount(self, amount):
        if amount is None:
            self._AMOUNT = None
        else:
            self._AMOUNT = '{0:08d}'.format(int(amount * 100))

    amount = property(_get_amount, _set_amount)

    export_flag = Column(
        'EXPORTFLAG', String(length=1), nullable=True)

    update_flag = Column(
        'UPDATEFLAG', String(length=1), nullable=True)

    def __repr__(self):
        return "ChargeHistory(account_number={0}, transaction_number={1})".format(
            repr(self.account_number), repr(self.transaction_number))


EJTransaction.charges = relationship(
    ChargeHistory,
    primaryjoin=ChargeHistory.transaction_number == EJTransaction.transaction_number,
    foreign_keys=[ChargeHistory.transaction_number],
    backref='transaction')
