#!/usr/bin/env python
# -*- coding: utf-8  -*-
################################################################################
#
#  pyScanMaster -- Python Interface to ScanMaster POS
#  Copyright © 2013 Sacramento Natural Foods Co-op, Inc
#
#  This file is part of pyScanMaster.
#
#  pyScanMaster is free software: you can redistribute it and/or modify it
#  under the terms of the GNU General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  pyScanMaster is distributed in the hope that it will be useful, but WITHOUT
#  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
#  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
#  more details.
#
#  You should have received a copy of the GNU General Public License along with
#  pyScanMaster.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################

"""
Host File Utilities
"""

import os.path


def write_customer_import_file(customers, path=None, progress=None):
    """
    Write a customer data file which ScanMaster can import.

    :param customers: Sequence of tuples.  Each tuple should include a
       :class:`scanmaster.db.Customer` instance as its first element, and a
       single character as its second element.  The character must be one of
       the following (the meaning of which is interpreted by ScanMaster):

       * ``'A'`` - Add
       * ``'U'`` - Update
       * ``'D'`` - Delete
       * ``'H'`` - Header
       * ``'I'`` - Ignore

    :param path: Optional path to which the file should be written.  If none is
       specified, defaults to ``'SSXCUSP`` in the current directory.
    :type path: string

    :returns: Full path of the file which was written.
    :rtype: string
    """

    assert customers, "Nothing to do!"

    data_path = path or 'SSXCUSP'
    data_file = open(data_path, 'wb')

    prog = None
    if progress:
        prog = progress("Writing customer import file", len(customers))

    for i, (customer, action_code) in enumerate(customers, 1):

        assert action_code in 'AUDHI', "Invalid action code: {0}".format(repr(action_code))

        data_file.write(''.join((
                    action_code,
                    '{0:12.12s}'.format('{0:012d}'.format(int(customer.account_number))),
                    '{0:25.25s}'.format(customer.last_name),
                    '{0:25.25s}'.format(customer.first_name),
                    '{0:25.25s}'.format(customer.address),
                    '{0:15.15s}'.format(customer.city),
                    '{0:2.2s}'.format(customer.state),
                    '{0:9.9s}'.format(customer.zipcode),
                    '{0:10.10s}'.format(customer.phone_number),
                    '{0:9.9s}'.format('{0:09d}'.format(int(customer.social_security_number))),
                    '{0:6.6s}'.format(customer._date_opened),
                    '{0:12.12s}'.format(customer.tax_exempt_code),
                    '{0:12.12s}'.format(customer.checking_account_number),
                    '{0:3.3s}'.format('{0:03d}'.format(int(customer.number_visits_to_date))),
                    '{0:3.3s}'.format('{0:03d}'.format(int(customer.number_visits_period_to_date))),
                    '{0:9.9s}'.format('{0:09d}'.format(int(customer.purchase_amount_to_date))),
                    '{0:9.9s}'.format('{0:09d}'.format(int(customer.purchase_amount_period_to_date))),
                    '{0:6.6s}'.format(customer._date_of_last_purchase),
                    '{0:30.30s}'.format(customer.comments),
                    '{0:3.3s}'.format('{0:03d}'.format(int(customer.override_count))),
                    '{0:1.1s}'.format('{0:01d}'.format(int(customer.account_status))),
                    '{0:9.9s}'.format('{0:09d}'.format(int(customer.frequent_shopper_dollars_to_date * 100))),
                    '{0:9.9s}'.format('{0:09d}'.format(int(customer.frequent_shopper_dollars_period_to_date * 100))),
                    '{0:1.1s}'.format(customer.number_checks_today),
                    '{0:1.1s}'.format(customer.number_checks_this_week),
                    '{0:9.9s}'.format(customer.amount_checks_today),
                    '{0:9.9s}'.format(customer.amount_checks_this_week),
                    '{0:1.1s}'.format(customer.record_changed),
                    '{0:1.1s}'.format(customer.update_flag),
                    '{0:1.1s}'.format(customer.frequent_shopper_level),
                    '{0:9.9s}'.format(customer.points_period_to_date),
                    '{0:9.9s}'.format(customer.points_to_date),
                    '{0:9.9s}'.format(customer.bonus_points_period_to_date),
                    '{0:9.9s}'.format(customer.bonus_points_to_date),
                    '{0:9.9s}'.format(customer.special_promotion_points),
                    '{0:9.9s}'.format(customer.frequent_shopper_discounts_to_date),
                    '{0:9.9s}'.format(customer.frequent_shopper_discounts_period_to_date),
                    '{0:9.9s}'.format(customer.electronic_coupon_to_date),
                    '{0:9.9s}'.format(customer.electronic_coupon_period_to_date),
                    '{0:1.1s}'.format(customer.auto_discount1_flag),
                    '{0:1.1s}'.format(customer.auto_discount2_flag),
                    '{0:1.1s}'.format(customer.auto_discount4_flag),
                    '{0:1.1s}'.format(customer.auto_discount5_flag),
                    '{0:1.1s}'.format(customer.cash_only),
                    '{0:20.20s}'.format(customer.check_account_number),
                    '{0:9.9s}'.format(customer.transit_number),
                    '{0:013d}'.format(0), # customer.blanks
                    '{0:25.25s}'.format(customer.address2),
                    '{0:50.50s}'.format(customer.email),
                    '{0:50.50s}'.format(customer.email2),
                    '{0:25.25s}'.format(customer.ship_to_address),
                    '{0:25.25s}'.format(customer.ship_to_address2),
                    '{0:15.15s}'.format(customer.ship_to_city),
                    '{0:2.2s}'.format(customer.ship_to_state),
                    '{0:9.9s}'.format(customer.ship_to_zipcode),
                    '{0:25.25s}'.format(customer.contact_last_name),
                    '{0:25.25s}'.format(customer.contact_first_name),
                    '{0:10.10s}'.format(customer.phone_number2),
                    '{0:1.1s}'.format(customer.limit_checks_today),
                    '{0:1.1s}'.format(customer.limit_checks_this_week),
                    '{0:9.9s}'.format(customer.limit_dollars_today),
                    '{0:9.9s}'.format(customer.limit_dollars_this_week),
                    '{0:100.100s}'.format(customer.memo),
                    '{0:18.18s}'.format(customer.drivers_license_number),
                    '{0:2.2s}'.format(customer.drivers_license_state),
                    '{0:20.20s}'.format(customer.alternate_lookup),
                    '{0:058d}'.format(0), # customer.extended.blanks
                    '\r\n'
                    )))

        if prog:
            prog.update(i)

    if prog:
        prog.destroy()
    data_file.close()
    return os.path.abspath(data_path)
