
import datetime
from decimal import Decimal
from unittest import TestCase
from mock import Mock

from scanmaster.db import model


class CustomerTests(TestCase):

    def test_repr(self):
        customer = model.Customer(account_number='123456789012')
        self.assertEqual(repr(customer), "Customer(account_number='123456789012')")

    def test_get_date_opened(self):
        customer = model.Customer()
        self.assertIsNone(customer.date_opened)
        customer._DateOpened = '000000'
        self.assertIsNone(customer.date_opened)
        customer._DateOpened = '102113'
        self.assertEqual(customer.date_opened, datetime.date(2013, 10, 21))

    def test_set_date_opened(self):
        customer = model.Customer()
        self.assertIsNone(customer._DateOpened)
        customer.date_opened = datetime.date(2013, 10, 21)
        self.assertEqual(customer._DateOpened, '102113')
        customer.date_opened = None
        self.assertEqual(customer._DateOpened, '000000')

    def test_get_date_of_last_purchase(self):
        customer = model.Customer()
        self.assertIsNone(customer.date_of_last_purchase)
        customer._DOLPurchase = '000000'
        self.assertIsNone(customer.date_of_last_purchase)
        customer._DOLPurchase = '102113'
        self.assertEqual(customer.date_of_last_purchase, datetime.date(2013, 10, 21))

    def test_set_date_of_last_purchase(self):
        customer = model.Customer()
        self.assertIsNone(customer._DOLPurchase)
        customer.date_of_last_purchase = datetime.date(2013, 10, 21)
        self.assertEqual(customer._DOLPurchase, '102113')
        customer.date_of_last_purchase = None
        self.assertEqual(customer._DOLPurchase, '000000')


class CustomerExtendedTests(TestCase):

    def test_repr(self):
        customer = model.CustomerExtended(account_number='123456789012')
        self.assertEqual(repr(customer), "CustomerExtended(account_number='123456789012')")

    def test_getset_factory(self):
        # This doesn't directly test CustomerExtended; maybe should go somewhere else.
        getter, setter = model.getset_factory(None, Mock(value_attr='foo'))
        self.assertEqual(getter(None), '')
        self.assertEqual(getter(Mock(foo='bar')), 'bar')
        dummy = Mock(foo='bar')
        self.assertEqual(dummy.foo, 'bar')
        setter(dummy, 'baz')
        self.assertEqual(dummy.foo, 'baz')

    def test_extended_proxy(self):
        # This doesn't directly test CustomerExtended; maybe should go somewhere else.
        # (Also this test isn't very "clean" in that it adds instrumentation to Customer...)
        self.assertFalse(hasattr(model.Customer, 'something'))
        model.extended_proxy('something')
        self.assertTrue(hasattr(model.Customer, 'something'))
        customer = model.Customer()
        self.assertIsNone(customer.extended)
        # getter returns empty string when no extended object is present
        self.assertEqual(customer.something, '')
        customer.something = 'whatever'
        self.assertIsNotNone(customer.extended)
        self.assertEqual(customer.something, 'whatever')


class EJTransactionTests(TestCase):

    def test_repr(self):
        tran = model.EJTransaction(transaction_id=12345, transaction_number='12345678')
        self.assertEqual(repr(tran), "EJTransaction(transaction_id=12345, transaction_number='12345678')")

    def test_started(self):
        tran = model.EJTransaction()
        self.assertIsNone(tran.started)
        tran.start_time = datetime.time(8, 0)
        self.assertEqual(tran.started, datetime.time(8, 0))
        tran.start_time = None
        tran.start_date = datetime.date(2013, 10, 21)
        self.assertEqual(tran.started, datetime.date(2013, 10, 21))
        tran.start_time = datetime.time(8, 0)
        self.assertEqual(tran.started, datetime.datetime(2013, 10, 21, 8, 0))

    def test_ended(self):
        tran = model.EJTransaction()
        self.assertIsNone(tran.ended)
        tran.end_time = datetime.time(8, 0)
        self.assertEqual(tran.ended, datetime.time(8, 0))
        tran.end_time = None
        tran.end_date = datetime.date(2013, 10, 21)
        self.assertEqual(tran.ended, datetime.date(2013, 10, 21))
        tran.end_time = datetime.time(8, 0)
        self.assertEqual(tran.ended, datetime.datetime(2013, 10, 21, 8, 0))


class EJCustomerTests(TestCase):

    def test_repr(self):
        customer = model.EJCustomer(transaction_id=12345, customer_number='123456789012')
        self.assertEqual(repr(customer), "EJCustomer(transaction_id=12345, customer_number='123456789012')")


class EJItemTests(TestCase):

    def test_repr(self):
        item = model.EJItem(transaction_id=12345, sequence_number=1)
        self.assertEqual(repr(item), "EJItem(transaction_id=12345, sequence_number=1)")


class EJPromotionTests(TestCase):

    def test_repr(self):
        promotion = model.EJPromotion(transaction_id=12345, sequence_number=1)
        self.assertEqual(repr(promotion), "EJPromotion(transaction_id=12345, sequence_number=1)")


class EJDiscountTests(TestCase):

    def test_repr(self):
        discount = model.EJDiscount(transaction_id=12345, sequence_number=1)
        self.assertEqual(repr(discount), "EJDiscount(transaction_id=12345, sequence_number=1)")


class EJTaxTests(TestCase):

    def test_repr(self):
        tax = model.EJTax(transaction_id=12345, tax_number=1)
        self.assertEqual(repr(tax), "EJTax(transaction_id=12345, tax_number=1)")


class ChargeHistoryTests(TestCase):

    def test_repr(self):
        charge = model.ChargeHistory(account_number='123456789012', transaction_number='12345678')
        self.assertEqual(repr(charge), "ChargeHistory(account_number='123456789012', transaction_number='12345678')")

    def test_get_business_date(self):
        charge = model.ChargeHistory()
        self.assertIsNone(charge.business_date)
        charge = model.ChargeHistory(_BUSINESSDATE='131009')
        self.assertEqual(charge.business_date, datetime.date(2013, 10, 9))

    def test_set_business_date(self):
        charge = model.ChargeHistory()
        self.assertIsNone(charge._BUSINESSDATE)
        charge.business_date = datetime.date(2013, 10, 9)
        self.assertEqual(charge._BUSINESSDATE, '131009')
        charge.business_date = None
        self.assertIsNone(charge._BUSINESSDATE)

    def test_get_business_time(self):
        charge = model.ChargeHistory()
        self.assertIsNone(charge.business_time)
        charge = model.ChargeHistory(_BUSINESSTIME='1453')
        self.assertEqual(charge.business_time, datetime.time(14, 53))

    def test_set_business_time(self):
        charge = model.ChargeHistory()
        self.assertIsNone(charge._BUSINESSTIME)
        charge.business_time = datetime.time(14, 53)
        self.assertEqual(charge._BUSINESSTIME, '1453')
        charge.business_time = None
        self.assertIsNone(charge._BUSINESSTIME)

    def test_occurred(self):
        charge = model.ChargeHistory()
        self.assertIsNone(charge.occurred)
        charge.business_date = datetime.date(2013, 10, 9)
        self.assertIsInstance(charge.occurred, datetime.date)
        self.assertEqual(charge.occurred, datetime.date(2013, 10, 9))
        charge.business_date = None
        charge.business_time = datetime.time(14, 56)
        self.assertIsInstance(charge.occurred, datetime.time)
        self.assertEqual(charge.occurred, datetime.time(14, 56))
        charge.business_date = datetime.date(2013, 10, 9)
        self.assertIsInstance(charge.occurred, datetime.datetime)
        self.assertEqual(charge.occurred, datetime.datetime(2013, 10, 9, 14, 56))

    def test_get_amount(self):
        charge = model.ChargeHistory()
        self.assertIsNone(charge.amount)
        charge._AMOUNT = '00004242'
        self.assertEqual(charge.amount, Decimal('42.42'))

    def test_set_amount(self):
        charge = model.ChargeHistory()
        self.assertIsNone(charge._AMOUNT)
        charge.amount = Decimal('42.42')
        self.assertEqual(charge._AMOUNT, '00004242')
        charge.amount = None
        self.assertIsNone(charge._AMOUNT)
