#!/usr/bin/env python
# -*- coding: utf-8  -*-
################################################################################
#
#  pyScanMaster -- Python Interface to ScanMaster POS
#  Copyright © 2013 Sacramento Natural Foods Co-op, Inc
#
#  This file is part of pyScanMaster.
#
#  pyScanMaster is free software: you can redistribute it and/or modify it
#  under the terms of the GNU General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  pyScanMaster is distributed in the hope that it will be useful, but WITHOUT
#  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
#  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
#  more details.
#
#  You should have received a copy of the GNU General Public License along with
#  pyScanMaster.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################

"""
Data Hosting Utilities
"""

from .db import model


ssxcusp_record_basic_format = [
    (0,         12,             'account_number'),
    (12,        25,             'last_name'),
    (37,        25,             'first_name'),
    (62,        25,             'address'),
    (87,        15,             'city'),
    (102,       2,              'state'),
    (104,       9,              'zipcode'),
    (113,       10,             'phone_number'),
    (123,       9,              'social_security_number'),
    (132,       6,              'date_opened'),
    (138,       12,             'tax_exempt_code'),
    (150,       12,             'checking_account_number'),
    (162,       3,              'number_visits_to_date'),
    (165,       3,              'number_visits_period_to_date'),
    (168,       9,              'purchase_amount_to_date'),
    (177,       9,              'purchase_amount_period_to_date'),
    (186,       6,              'date_of_last_purchase'),
    (192,       30,             'comments'),
    (222,       3,              'override_count'),
    (225,       1,              'account_status'),
    (226,       9,              'frequent_shopper_dollars_to_date'),
    (235,       9,              'frequent_shopper_dollars_period_to_date'),
    (244,       1,              'number_checks_today'),
    (245,       1,              'number_checks_this_week'),
    (246,       9,              'amount_checks_today'),
    (255,       9,              'amount_checks_this_week'),
    (264,       1,              'record_changed'),
    (265,       1,              'update_flag'),
    (266,       1,              'frequent_shopper_level'),
    (267,       9,              'points_period_to_date'),
    (276,       9,              'points_to_date'),
    (285,       9,              'bonus_points_period_to_date'),
    (294,       9,              'bonus_points_to_date'),
    (303,       9,              'special_promotion_points'),
    (312,       9,              'frequent_shopper_discounts_to_date'),
    (321,       9,              'frequent_shopper_discounts_period_to_date'),
    (330,       9,              'electronic_coupon_to_date'),
    (339,       9,              'electronic_coupon_period_to_date'),
    (348,       1,              'auto_discount1_flag'),
    (349,       1,              'auto_discount2_flag'),
    (350,       1,              'auto_discount4_flag'),
    (351,       1,              'auto_discount5_flag'),
    (352,       1,              'cash_only'),
    (353,       20,             'check_account_number'),
    (373,       9,              'transit_number'),
    (382,       13,             'blanks'),
    ]

# ssxcusp_record_extended_format = [
#     (395,       25,             'address2'),
#     (420,       50,             'email'),
#     (470,       50,             'email2'),
#     (520,       25,             'ship_to_address'),
#     (545,       25,             'ship_to_address2'),
#     (570,       15,             'ship_to_city'),
#     (585,       2,              'ship_to_state'),
#     (587,       9,              'ship_to_zipcode'),
#     (596,       25,             'contact_last_name'),
#     (621,       25,             'contact_first_name'),
#     (646,       10,             'phone_number2'),
#     (656,       1,              'limit_checks_today'),
#     (657,       1,              'limit_checks_this_week'),
#     (658,       9,              'limit_dollars_today'),
#     (667,       9,              'limit_dollars_this_week'),
#     (676,       100,            'memo'),
#     (776,       18,             'drivers_license_number'),
#     (794,       2,              'drivers_license_state'),
#     (796,       20,             'alternate_lookup'),
#     (816,       58,             'blanks'),
#     ]

ssxcusp_record_format = ssxcusp_record_basic_format


def parse_ssxcusp_record(record):
    """
    Parse a line from a "Customer File Import" data file.

    :param record: Line from the data file.
    :type record: string

    :returns: A :class:`.db.model.Customer` instance.
    """
    last_field = ssxcusp_record_format[-1]
    record_length = last_field[0] + last_field[1]
    if len(record) != record_length:
        raise ValueError("Record should be {0} characters long, but is {1}: {2}".format(
                record_length, len(record), repr(record)))

    customer = model.Customer()
    for offset, length, field in ssxcusp_record_format:
        data = record[offset:offset + length]
        if field == 'date_opened':
            customer._DateOpened = data
        elif field == 'date_of_last_purchase':
            customer._DOLPurchase = data
        else:
            setattr(customer, field, data.strip())
    return customer


def write_ssxcusp_record(ssxcusp_file, customer, action_code, extended=False):
    """
    Write a customer record to a "Customer File Import" data file.

    :param ssxcusp_file: File-like object which as been opened for writing.

    :param customer: Reference to a :class:`.db.model.Customer` instance, whose
       attributes will constitute the data record.

    :param action_code: One of the action codes defined by ScanMaster:

       * ``'A'`` - Add
       * ``'U'`` - Update
       * ``'D'`` - Delete
       * ``'H'`` - Header
       * ``'I'`` - Ignore

    :param extended: Boolean indicating whether extended attributes should be
       included in the output.  Defaults to ``False``.
    """
    record = ''.join((
            action_code,
            '{0:12.12s}'.format('{0:012d}'.format(int(customer.account_number))),
            '{0:25.25s}'.format(customer.last_name),
            '{0:25.25s}'.format(customer.first_name),
            '{0:25.25s}'.format(customer.address),
            '{0:15.15s}'.format(customer.city),
            '{0:2.2s}'.format(customer.state),
            '{0:9.9s}'.format(customer.zipcode),
            '{0:10.10s}'.format(customer.phone_number),
            '{0:9.9s}'.format('{0:09d}'.format(int(customer.social_security_number))),
            '{0:6.6s}'.format(customer._DateOpened),
            '{0:12.12s}'.format(customer.tax_exempt_code),
            '{0:12.12s}'.format(customer.checking_account_number),
            '{0:3.3s}'.format('{0:03d}'.format(customer.number_visits_to_date)),
            '{0:3.3s}'.format('{0:03d}'.format(customer.number_visits_period_to_date)),
            '{0:9.9s}'.format('{0:09d}'.format(int(customer.purchase_amount_to_date))),
            '{0:9.9s}'.format('{0:09d}'.format(int(customer.purchase_amount_period_to_date))),
            '{0:6.6s}'.format(customer._DOLPurchase),
            '{0:30.30s}'.format(customer.comments),
            '{0:3.3s}'.format('{0:03d}'.format(int(customer.override_count))),
            '{0:1.1s}'.format('{0:01d}'.format(int(customer.account_status))),
            '{0:9.9s}'.format('{0:09d}'.format(int(customer.frequent_shopper_dollars_to_date * 100))),
            '{0:9.9s}'.format('{0:09d}'.format(int(customer.frequent_shopper_dollars_period_to_date * 100))),
            '{0:1.1s}'.format(customer.number_checks_today),
            '{0:1.1s}'.format(customer.number_checks_this_week),
            '{0:9.9s}'.format(customer.amount_checks_today),
            '{0:9.9s}'.format(customer.amount_checks_this_week),
            '{0:1.1s}'.format(customer.record_changed),
            '{0:1.1s}'.format(customer.update_flag),
            '{0:1.1s}'.format(customer.frequent_shopper_level),
            '{0:9.9s}'.format(customer.points_period_to_date),
            '{0:9.9s}'.format(customer.points_to_date),
            '{0:9.9s}'.format(customer.bonus_points_period_to_date),
            '{0:9.9s}'.format(customer.bonus_points_to_date),
            '{0:9.9s}'.format(customer.special_promotion_points),
            '{0:9.9s}'.format(customer.frequent_shopper_discounts_to_date),
            '{0:9.9s}'.format(customer.frequent_shopper_discounts_period_to_date),
            '{0:9.9s}'.format(customer.electronic_coupon_to_date),
            '{0:9.9s}'.format(customer.electronic_coupon_period_to_date),
            '{0:1.1s}'.format(customer.auto_discount1_flag),
            '{0:1.1s}'.format(customer.auto_discount2_flag),
            '{0:1.1s}'.format(customer.auto_discount4_flag),
            '{0:1.1s}'.format(customer.auto_discount5_flag),
            '{0:1.1s}'.format(customer.cash_only),
            '{0:20.20s}'.format(customer.check_account_number),
            '{0:9.9s}'.format(customer.transit_number),
            '{0:013d}'.format(0), # customer.blanks
            ))

    # if extended:
    #     record += ''.join((
    #             '{0:25.25s}'.format(customer.address2),
    #             '{0:50.50s}'.format(customer.email),
    #             '{0:50.50s}'.format(customer.email2),
    #             '{0:25.25s}'.format(customer.ship_to_address),
    #             '{0:25.25s}'.format(customer.ship_to_address2),
    #             '{0:15.15s}'.format(customer.ship_to_city),
    #             '{0:2.2s}'.format(customer.ship_to_state),
    #             '{0:9.9s}'.format(customer.ship_to_zipcode),
    #             '{0:25.25s}'.format(customer.contact_last_name),
    #             '{0:25.25s}'.format(customer.contact_first_name),
    #             '{0:10.10s}'.format(customer.phone_number2),
    #             '{0:1.1s}'.format(customer.limit_checks_today),
    #             '{0:1.1s}'.format(customer.limit_checks_this_week),
    #             '{0:9.9s}'.format(customer.limit_dollars_today),
    #             '{0:9.9s}'.format(customer.limit_dollars_this_week),
    #             '{0:100.100s}'.format(customer.memo),
    #             '{0:18.18s}'.format(customer.drivers_license_number),
    #             '{0:2.2s}'.format(customer.drivers_license_state),
    #             '{0:20.20s}'.format(customer.alternate_lookup),
    #             '{0:058d}'.format(0), # customer.extended.blanks
    #             ))

    ssxcusp_file.write('{0}\r\n'.format(record))
