#!/usr/bin/env python
# -*- coding: utf-8  -*-
################################################################################
#
#  pyScanMaster -- Python Interface to ScanMaster POS
#  Copyright © 2013 Sacramento Natural Foods Co-op, Inc
#
#  This file is part of pyScanMaster.
#
#  pyScanMaster is free software: you can redistribute it and/or modify it
#  under the terms of the GNU General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  pyScanMaster is distributed in the hope that it will be useful, but WITHOUT
#  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
#  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
#  more details.
#
#  You should have received a copy of the GNU General Public License along with
#  pyScanMaster.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################

"""
Host Data Utilities
"""

from . import ssxcusp
from . import ssxcrgp
from . import ssxncfp


# Data file type constants.
SSXCUSP = u'SSXCUSP'
SSXCRGP = u'SSXCRGP'
SSXNCFP = u'SSXNCFP'


def write_data_file(fileobj, filetype, records):
    """
    Write a series of records to a host data file.  This function exists
    primarily for convenience, since the data file should *not* have a newline
    character(s) following the last data record.  It will delegate most of the
    writing logic to the relevant ``write_ssx*_record()`` function, i.e. one
    of:

    * :func:`ssxcusp.write_ssxcusp_record()`
    * :func:`ssxcrgp.write_ssxcrgp_record()`
    * :func:`ssxncfp.write_ssxncfp_record()`

    :param fileobj: File-like object which as been opened for writing.

    :param filetype: The type of file which is to be written.  This determines
       which record-writing function will be leveraged.  It should be one of
       the following values:

       * ``'SSXCUSP'``
       * ``'SSXCRGP'``
       * ``'SSXNCFP'``

    :param records: A sequence of 2-tuples of the form ``(instance,
       action_code)``.  These values will be unpacked and passed directly to
       the relevant ``write_ssx*_record()`` function.
    """
    if filetype == SSXCUSP:
        write_record = ssxcusp.write_ssxcusp_record
    elif filetype == SSXCRGP:
        write_record = ssxcrgp.write_ssxcrgp_record
    elif filetype == SSXNCFP:
        write_record = ssxncfp.write_ssxncfp_record
    else:
        raise ValueError(u"Unknown file type: {0}".format(repr(filetype)))

    first = True
    for instance, action_code in records:
        if not first:
            fileobj.write(u'\r\n')
        write_record(fileobj, instance, action_code)
        first = False
