#!/usr/bin/env python
# -*- coding: utf-8  -*-
################################################################################
#
#  pyScanMaster -- Python Interface to ScanMaster POS
#  Copyright © 2013 Sacramento Natural Foods Co-op, Inc
#
#  This file is part of pyScanMaster.
#
#  pyScanMaster is free software: you can redistribute it and/or modify it
#  under the terms of the GNU General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  pyScanMaster is distributed in the hope that it will be useful, but WITHOUT
#  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
#  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
#  more details.
#
#  You should have received a copy of the GNU General Public License along with
#  pyScanMaster.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################

"""
SSXCRGP Data Utilities
"""

from ..db import model


__all__ = ['parse_ssxcrgp_record', 'write_ssxcrgp_record']


ssxcrgp_record_format = [
    # offset    # length        # attribute
    (0,         12,             'account_number'),
    (12,        6,              'date_opened'),
    (18,        6,              'credit_limit'),
    (24,        1,              'credit_status'),
    (25,        6,              'date_of_last_payment'),
    (31,        8,              'amount_of_last_payment'),
    (39,        8,              'current_balance'),
    (47,        8,              'previous_balance'),
    (55,        2,              'page_number'),
    (57,        2,              'line_number'),
    (59,        1,              'finance_charges'),
    (60,        4,              'apr'),
    (64,        35,             'blanks'),
    (99,        1,              'update_flag'),
    ]


def parse_ssxcrgp_record(record):
    """
    Parse a line from a "Instore Charge Import" data file.

    :param record: Line from the data file.
    :type record: string

    :returns: A :class:`..db.model.InstoreCharge` instance.
    """
    last_field = ssxcrgp_record_format[-1]
    record_length = last_field[0] + last_field[1]
    if len(record) != record_length:
        raise ValueError("Record should be {0} characters long, but is {1}: {2}".format(
                record_length, len(record), repr(record)))

    charge = model.InstoreCharge()
    for offset, length, field in ssxcrgp_record_format:
        data = record[offset:offset + length]
        data = data.strip()
        data = data.replace('\x00', '')
        setattr(charge, field, data)
    return charge


def write_ssxcrgp_record(ssxcrgp_file, charge, action_code):
    """
    Write a record to a "Instore Charge Import" data file.

    :param ssxcrgp_file: File-like object which as been opened for writing.

    :param charge: Reference to a :class:`.db.model.InstoreCharge` instance,
       whose attributes will constitute the data record.

    :param action_code: One of the action codes defined by ScanMaster:

       * ``'A'`` - Add
       * ``'U'`` - Update
       * ``'D'`` - Delete
    """
    # Hack due to difference between ScanMaster database schema and file format
    # specification (former uses 34 characters; latter 35).
    blanks = charge.blanks
    if blanks == '0' * 34:
        blanks += '0'
    record = ''.join((
            action_code,
            '{0:12.12s}'.format(charge.account_number),
            '{0:6.6s}'.format(charge._DATEOPENED),
            '{0:6.6s}'.format(charge._CREDITLIMIT),
            '{0:1.1s}'.format(charge.credit_status),
            '{0:6.6s}'.format(charge._DATEOFLASTPAYMENT),
            '{0:8.8s}'.format(charge._AMOUNTOFLASTPAYMENT),
            '{0:8.8s}'.format(charge._CURRENTBALANCE),
            '{0:8.8s}'.format(charge._PREVIOUSBALANCE),
            '{0:2.2s}'.format(charge._PAGENUMBER),
            '{0:2.2s}'.format(charge._LINENUMBER),
            '{0:1.1s}'.format(charge.finance_charges),
            '{0:4.4s}'.format(charge.apr),
            '{0:35.35s}'.format(blanks),
            '{0:1.1s}'.format(charge.update_flag),
            ))
    ssxcrgp_file.write(record)
