# -*- coding: utf-8 -*-
################################################################################
#
#  pyScanMaster -- Python Interface to ScanMaster POS
#  Copyright © 2013-2014 Sacramento Natural Foods Co-op, Inc
#
#  This file is part of pyScanMaster.
#
#  pyScanMaster is free software: you can redistribute it and/or modify it
#  under the terms of the GNU General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  pyScanMaster is distributed in the hope that it will be useful, but WITHOUT
#  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
#  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
#  more details.
#
#  You should have received a copy of the GNU General Public License along with
#  pyScanMaster.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################

"""
Database Models
"""

from sqlalchemy.ext.declarative import declarative_base
from sqlalchemy import Column
from sqlalchemy import (
    String, Integer, SmallInteger, Date, Time, Numeric, Boolean)
from sqlalchemy.orm import relationship, object_mapper
from sqlalchemy.ext.associationproxy import association_proxy

import datetime
from decimal import Decimal


__all__ = [
    u'Department', u'Item', u'ElectronicCoupon',
    u'Customer', u'NegativeCheck', u'InstoreCharge', u'ChargeHistory',
    u'EJTransaction', u'EJCustomer', u'EJItem', u'EJPromotion', u'EJDiscount', u'EJTax',
    ]


class BaseObject(object):
    """
    Superclass for the declarative base.
    """

    def __init__(self, **kwargs):
        """
        Custom constructor to provide default attribute values.

        This leverages column defaults in order to initially populate the
        object instance.
        """
        for attr in object_mapper(self).column_attrs:
            if attr.key not in kwargs:
                column = attr.columns[0]
                if column.default:
                    setattr(self, attr.key, column.default.arg)
        # Kwargs always overrides default values.  This is especially necessary
        # since some attributes are "raw" columns and others use property
        # function wrappers.  Therefore the property will be set after the
        # column has provided the default value.
        for key, value in kwargs.iteritems():
            setattr(self, key, value)
    

Base = declarative_base(cls=BaseObject, constructor=None)


# NOTE: The following comments and code appeared to be necessary at some point,
# but I have since been able to access a ScanMaster database just fine on Linux
# without it.  I used the Pervasive `TranslationOption` setting which may have
# made the difference (?).

# # There seems to be some sort of Unicode issue with (at least) the Linux /
# # Python 2.7 / pyodbc / Pervasive combination.  Not sure "whose fault" it is
# # exactly but I was seeing some nasty premature application death.  The only
# # thing I found to resolve the issue was to force SQLAchemy to convert to/from
# # Unicode and keep things "safe" on the database side.
# def String(*args, **kwargs):
#     kwargs.setdefault('convert_unicode', 'force')
#     return SA_String(*args, **kwargs)


def date_property(column, fmt):
    """
    Establish a date property for the given column.
    """
    
    def getter(self):
        value = getattr(self, column)
        if value is None:
            return None
        if value == '000000':
            return None
        return datetime.datetime.strptime(value, fmt).date()

    def setter(self, value):
        if value is None:
            setattr(self, column, '000000')
        elif isinstance(value, basestring):
            setattr(self, column, value)
        else:
            setattr(self, column, value.strftime(fmt))

    return property(getter, setter)


def decimal_property(column, length):
    """
    Establish a decimal property for the given column.
    """

    def getter(self):
        value = getattr(self, column)
        if value is None:
            return None
        return Decimal('{0}.{1}'.format(value[:-2], value[-2:]))

    def setter(self, value):
        if value is None:
            setattr(self, column, None)
        elif isinstance(value, basestring):
            setattr(self, column, value)
        else:
            fmt = '{{0:0{0}d}}'.format(length)
            setattr(self, column, fmt.format(int(value * 100)))

    return property(getter, setter)


def int_property(column, length):
    """
    Establish an integer property for the given column.
    """

    def getter(self):
        value = getattr(self, column)
        if value is None:
            return 0
        return int(value)

    def setter(self, value):
        if value is None:
            value = 0
        else:
            value = int(value)
        fmt = '{{0:0{0}d}}'.format(length)
        setattr(self, column, fmt.format(value))

    return property(getter, setter)


def time_property(column, fmt):
    """
    Establish a time property for the given column.
    """
    
    def getter(self):
        value = getattr(self, column)
        if value is None:
            return None
        return datetime.datetime.strptime(value, fmt).time()

    def setter(self, value):
        if value is None:
            setattr(self, column, None)
        else:
            setattr(self, column, value.strftime(fmt))

    return property(getter, setter)


class Department(Base):
    """
    Represents a department.
    """
    __tablename__ = u'Department'

    Department = Column(
        u'Department', String(length=4), primary_key=True, nullable=False)

    Description = Column(
        u'Description', String(length=15), nullable=False)

    FoodStamp = Column(
        u'FoodStamp', String(length=1), nullable=False)

    Tax1 = Column(
        u'Tax1', String(length=1), nullable=False)

    Tax2 = Column(
        u'Tax2', String(length=1), nullable=False)

    Tax3 = Column(
        u'Tax3', String(length=1), nullable=False)

    Discount1 = Column(
        u'Discount1', String(length=1), nullable=False)

    Discount2 = Column(
        u'Discount2', String(length=1), nullable=False)

    FrequentShopper = Column(
        u'FrequentShopper', String(length=1), nullable=False)

    HALO = Column(
        u'HALO', String(length=6), nullable=False)

    LALO = Column(
        u'LALO', String(length=6), nullable=False)

    QuantityLimit = Column(
        u'QuantityLimit', String(length=2), nullable=False)

    GroupLink = Column(
        u'GroupLink', String(length=2), nullable=False)

    ReceiptCouponNum = Column(
        u'ReceiptCouponNum', String(length=4), nullable=False)

    CouponUnderOver = Column(
        u'CouponUnderOver', String(length=1), nullable=False)

    UnderOverAmount = Column(
        u'UnderOverAmount', String(length=6), nullable=False)

    DepartmentType = Column(
        u'DepartmentType', String(length=1), nullable=False)

    GLAccountNumber = Column(
        u'GLAccountNumber', String(length=10), nullable=False)

    ExcludeFromMinPur = Column(
        u'ExcludeFromMinPur', String(length=1), nullable=False)

    Rx = Column(
        u'Rx', String(length=1), nullable=False)

    NonRxHC = Column(
        u'NonRxHC', String(length=1), nullable=False)

    Blanks = Column(
        u'Blanks', String(length=8), nullable=True)


class Item(Base):
    """
    Represents an item (product).
    """
    __tablename__ = u'Item'

    UPC = Column(
        u'UPC', String(length=20), primary_key=True, nullable=False)

    POSDescription = Column(
        u'POSDescription', String(length=16), nullable=False)

    Department = Column(
        u'Department', String(length=4), nullable=False)

    SubDepartment = Column(
        u'SubDepartment', String(length=4), nullable=False)

    ItemType = Column(
        u'ItemType', String(length=1), nullable=False)

    UnitPrice = Column(
        u'UnitPrice', String(length=6), nullable=False)

    SplitQuantity = Column(
        u'SplitQuantity', String(length=2), nullable=False)

    SplitPrice = Column(
        u'SplitPrice', String(length=6), nullable=False)

    FoodStamp = Column(
        u'FoodStamp', String(length=1), nullable=False)

    Tax1 = Column(
        u'Tax1', String(length=1), nullable=False)

    Tax2 = Column(
        u'Tax2', String(length=1), nullable=False)

    Tax3 = Column(
        u'Tax3', String(length=1), nullable=False)

    Discount1 = Column(
        u'Discount1', String(length=1), nullable=False)

    Discount2 = Column(
        u'Discount2', String(length=1), nullable=False)

    FrequentShopper = Column(
        u'FrequentShopper', String(length=1), nullable=False)

    EnforceQuantity = Column(
        u'EnforceQuantity', String(length=1), nullable=False)

    LimitedQuantity = Column(
        u'LimitedQuantity', String(length=2), nullable=False)

    LimitedPrice = Column(
        u'LimitedPrice', String(length=6), nullable=False)

    BottleLink = Column(
        u'BottleLink', String(length=2), nullable=False)

    MixMatch = Column(
        u'MixMatch', String(length=4), nullable=False)

    ReportCode = Column(
        u'ReportCode', String(length=4), nullable=False)

    AdLevel = Column(
        u'AdLevel', String(length=1), nullable=False)

    VisualVerify = Column(
        u'VisualVerify', String(length=1), nullable=False)

    ReceiptCouponNumber = Column(
        u'ReceiptCouponNumber', String(length=4), nullable=False)

    RestrictSale = Column(
        u'RestrictSale', String(length=1), nullable=False)

    UnitCost = Column(
        u'UnitCost', String(length=9), nullable=False)

    CaseCost = Column(
        u'CaseCost', String(length=9), nullable=False)

    CaseQuantity = Column(
        u'CaseQuantity', String(length=4), nullable=False)

    CommodityCode = Column(
        u'CommodityCode', String(length=8), nullable=False)

    DateTimeModified = Column(
        u'DateTimeModified', String(length=14), nullable=False)

    WIC = Column(
        u'WIC', String(length=1), nullable=False)

    ElectronicCoupon = Column(
        u'ElectronicCoupon', String(length=1), nullable=False)

    AuditFlag = Column(
        u'AuditFlag', String(length=1), nullable=False)

    VendorNumber = Column(
        u'VendorNumber', String(length=8), nullable=False)

    ItemNumber = Column(
        u'ItemNumber', String(length=13), nullable=False)

    ModifiedByBatch = Column(
        u'ModifiedByBatch', String(length=4), nullable=False)

    Seasonal = Column(
        u'Seasonal', String(length=1), nullable=False)

    AdFlag = Column(
        u'AdFlag', String(length=1), nullable=False)

    PackSize = Column(
        u'PackSize', String(length=4), nullable=False)

    UnitSize = Column(
        u'UnitSize', String(length=9), nullable=False)

    UnitMeasure = Column(
        u'UnitMeasure', String(length=4), nullable=False)

    Aisle = Column(
        u'Aisle', String(length=3), nullable=False)

    Shelf = Column(
        u'Shelf', String(length=3), nullable=False)

    Location = Column(
        u'Location', String(length=3), nullable=False)

    TagStockNumber = Column(
        u'TagStockNumber', String(length=3), nullable=False)

    TagQuantity = Column(
        u'TagQuantity', String(length=3), nullable=False)

    FamilyCode1 = Column(
        u'FamilyCode1', String(length=3), nullable=False)

    FamilyCode2 = Column(
        u'FamilyCode2', String(length=3), nullable=False)

    TareCode = Column(
        u'TareCode', String(length=2), nullable=False)

    Weight = Column(
        u'Weight', String(length=6), nullable=False)

    AdjectivePriceCode = Column(
        u'AdjectivePriceCode', String(length=4), nullable=False)

    Points = Column(
        u'Points', String(length=6), nullable=False)

    BonusPoints = Column(
        u'BonusPoints', String(length=6), nullable=False)

    FreeItem = Column(
        u'FreeItem', String(length=1), nullable=False)

    ExcludeFromMinPur = Column(
        u'ExcludeFromMinPur', String(length=1), nullable=False)

    BOFDescription = Column(
        u'BOFDescription', String(length=30), nullable=False)

    GiftCard = Column(
        u'GiftCard', String(length=1), nullable=False)

    PriceOverride = Column(
        u'PriceOverride', String(length=1), nullable=False)

    LikeCode = Column(
        u'LikeCode', String(length=20), nullable=False)

    PriceEndDate = Column(
        u'PriceEndDate', String(length=8), nullable=False)

    UnitSizeDescription = Column(
        u'UnitSizeDescription', String(length=9), nullable=False)

    BaseUnitPrice = Column(
        u'BaseUnitPrice', String(length=6), nullable=False)

    BaseSplitQuantity = Column(
        u'BaseSplitQuantity', String(length=2), nullable=False)

    BaseSplitPrice = Column(
        u'BaseSplitPrice', String(length=6), nullable=False)

    BaseTareCode = Column(
        u'BaseTareCode', String(length=2), nullable=False)

    BaseWeight = Column(
        u'BaseWeight', String(length=6), nullable=False)

    BaseLimitQuantity = Column(
        u'BaseLimitQuantity', String(length=2), nullable=False)

    BaseLimitPrice = Column(
        u'BaseLimitPrice', String(length=6), nullable=False)

    BaseLikeCode = Column(
        u'BaseLikeCode', String(length=20), nullable=False)

    BaseBottleLink = Column(
        u'BaseBottleLink', String(length=2), nullable=False)

    BaseMixMatch = Column(
        u'BaseMixMatch', String(length=4), nullable=False)

    BaseItemType = Column(
        u'BaseItemType', String(length=1), nullable=False)

    AllowManualWeight = Column(
        u'AllowManualWeight', String(length=1), nullable=False)

    PromoID = Column(
        u'PromoID', String(length=20), nullable=False)

    UOMCompareID = Column(
        u'UOMCompareID', String(length=4), nullable=False)

    Rx = Column(
        u'Rx', String(length=1), nullable=False)

    NonRxHC = Column(
        u'NonRxHC', String(length=1), nullable=False)

    CVV = Column(
        u'CVV', String(length=1), nullable=False)

    Blanks = Column(
        u'Blanks', String(length=173), nullable=True)

    def __repr__(self):
        return u'Item(UPC={0})'.format(repr(self.UPC))

    def __unicode__(self):
        return unicode(self.POSDescription)


class ElectronicCoupon(Base):
    """
    Represents an electronic coupon.
    """
    __tablename__ = u'ElectronicCoupon'

    Upc = Column(
        u'Upc', String(length=20), primary_key=True, nullable=False)

    Vendor = Column(
        u'Vendor', String(length=12), nullable=False)

    Commoditycode = Column(
        u'Commoditycode', String(length=12), nullable=False)

    Department = Column(
        u'Department', String(length=4), primary_key=True, nullable=False)

    Reportcode = Column(
        u'Reportcode', String(length=4), primary_key=True, nullable=False)

    Description = Column(
        u'Description', String(length=30), nullable=False)

    Coupontype = Column(
        u'Coupontype', String(length=1), nullable=False)

    Discounttype = Column(
        u'Discounttype', String(length=1), nullable=False)

    Discount = Column(
        u'Discount', String(length=6), nullable=False)

    Buys = Column(
        u'Buys', String(length=6), nullable=False)

    Gets = Column(
        u'Gets', String(length=6), nullable=False)

    Limitedquantity = Column(
        u'Limitedquantity', String(length=6), nullable=False)

    Startdate = Column(
        u'Startdate', String(length=6), nullable=False)

    Starttime = Column(
        u'Starttime', String(length=4), nullable=False)

    Enddate = Column(
        u'Enddate', String(length=6), nullable=False)

    Endtime = Column(
        u'Endtime', String(length=4), nullable=False)

    Points = Column(
        u'Points', String(length=6), nullable=False)

    Bonus = Column(
        u'Bonus', String(length=6), nullable=False)

    ReceiptCoupon = Column(
        u'ReceiptCoupon', String(length=4), nullable=False)

    PromoID = Column(
        u'PromoID', String(length=20), primary_key=True, nullable=False)

    CEcquantity = Column(
        u'CEcquantity', String(length=6), nullable=False)

    CEcamount = Column(
        u'CEcamount', String(length=9), nullable=False)

    CTpoints = Column(
        u'CTpoints', String(length=12), nullable=False)

    CTbonus = Column(
        u'CTbonus', String(length=12), nullable=False)

    PECquantity = Column(
        u'PECquantity', String(length=6), nullable=False)

    PECcamount = Column(
        u'PECcamount', String(length=9), nullable=False)

    PTpoints = Column(
        u'PTpoints', String(length=12), nullable=False)

    PTbonus = Column(
        u'PTbonus', String(length=12), nullable=False)

    CloseDate = Column(
        u'CloseDate', String(length=6), nullable=False)

    FrequentShopperLevel = Column(
        u'FrequentShopperLevel', String(length=1), primary_key=True, nullable=False)

    MinPurchase = Column(
        u'MinPurchase', String(length=7), nullable=False)

    MixMatch = Column(
        u'MixMatch', String(length=4), primary_key=True, nullable=False)

    TotalSale = Column(
        u'TotalSale', String(length=7), primary_key=True, nullable=False)

    DepartmentTotal = Column(
        u'DepartmentTotal', String(length=7), nullable=False)

    SortDepartment = Column(
        u'SortDepartment', String(length=4), nullable=False)

    RedeemValue = Column(
        u'RedeemValue', String(length=6), nullable=False)

    BundleCode = Column(
        u'BundleCode', String(length=2), primary_key=True, nullable=False)

    Weighted = Column(
        u'Weighted', String(length=1), nullable=False)

    FavorFlag = Column(
        u'FavorFlag', String(length=1), nullable=False)

    Mediafile = Column(
        u'Mediafile', String(length=35), nullable=False)

    UCIPromoID = Column(
        u'UCIPromoID', String(length=20), nullable=False)

    Unused = Column(
        u'Unused', String(length=10), nullable=False)

    MMCurrentQuantity = Column(
        u'MMCurrentQuantity', String(length=6), nullable=False)

    MMPTDQuantity = Column(
        u'MMPTDQuantity', String(length=6), nullable=False)

    MinimumQty = Column(
        u'MinimumQty', String(length=6), nullable=False)

    Blanks = Column(
        u'Blanks', String(length=98), nullable=True)


class Customer(Base):
    """
    Represents a customer.
    """

    __tablename__ = 'Customer'

    account_number = Column(
        'AccountNumber', String(length=12), primary_key=True, nullable=False)

    last_name = Column(
        'LastName', String(length=25), nullable=False, default='')

    first_name = Column(
        'FirstName', String(length=25), nullable=False, default='')

    address = Column(
        'Address', String(length=25), nullable=False, default='')

    city = Column(
        'City', String(length=15), nullable=False, default='')

    state = Column(
        'State', String(length=2), nullable=False, default='')

    zipcode = Column(
        'Zipcode', String(length=9), nullable=False, default='')

    phone_number = Column(
        'PhoneNumber', String(length=10), nullable=False, default='0000000000')

    social_security_number = Column(
        'SSNumber', String(length=9), nullable=False, default='000000000')

    _DateOpened = Column(
        'DateOpened', String(length=6), nullable=False, default='000000')

    date_opened = date_property('_DateOpened', '%m%d%y')

    tax_exempt_code = Column(
        'TaxExemptCode', String(length=12), nullable=False, default='000000000000')

    checking_account_number = Column(
        'CheckingAcctNum', String(length=12), nullable=False, default='000000000000')

    _NumVisitsTD = Column(
        'NumVisitsTD', String(length=3), nullable=False, default='000')

    number_visits_to_date = int_property('_NumVisitsTD', 3)

    _NumVisitsPTD = Column(
        'NumVisitsPTD', String(length=3), nullable=False, default='000')

    number_visits_period_to_date = int_property('_NumVisitsPTD', 3)

    purchase_amount_to_date = Column(
        'PurchaseAmountTD', String(length=9), nullable=False, default='000000000')

    purchase_amount_period_to_date = Column(
        'PurchaseAmountPTD', String(length=9), nullable=False, default='000000000')

    _DOLPurchase = Column(
        'DOLPurchase', String(length=6), nullable=False, default='000000')

    date_of_last_purchase = date_property('_DOLPurchase', '%m%d%y')

    comments = Column(
        'Comments', String(length=30), nullable=False, default='')

    override_count = Column(
        'OverrideCount', String(length=3), nullable=False, default='000')

    account_status = Column(
        'AccountStatus', String(length=1), nullable=False, default='0')

    _FSDollarsTD = Column(
        'FSDollarsTD', String(length=9), nullable=False, default='000000000')

    frequent_shopper_dollars_to_date = decimal_property('_FSDollarsTD', 9)

    _FSDollarsPTD = Column(
        'FSDollarsPTD', String(length=9), nullable=False, default='000000000')

    frequent_shopper_dollars_period_to_date = decimal_property('_FSDollarsPTD', 9)

    number_checks_today = Column(
        'NumChecksToday', String(length=1), nullable=False, default='0')

    number_checks_this_week = Column(
        'NumChecksThisWeek', String(length=1), nullable=False, default='0')

    amount_checks_today = Column(
        'AmtChecksToday', String(length=9), nullable=False, default='000000000')

    amount_checks_this_week = Column(
        'AmtChecksThisWeek', String(length=9), nullable=False, default='000000000')

    record_changed = Column(
        'RecordChanged', String(length=1), nullable=False, default='0')

    update_flag = Column(
        'UpdateFlag', String(length=1), nullable=False, default='0')

    # TODO: Give this column a default value?  Might need to be business-specific?
    frequent_shopper_level = Column(
        'FSLevel', String(length=1), nullable=False)

    points_period_to_date = Column(
        'PointsPTD', String(length=9), nullable=False, default='000000000')

    points_to_date = Column(
        'PointsTD', String(length=9), nullable=False, default='000000000')

    bonus_points_period_to_date = Column(
        'BonusPointsPTD', String(length=9), nullable=False, default='000000000')

    bonus_points_to_date = Column(
        'BonusPointsTD', String(length=9), nullable=False, default='000000000')

    special_promotion_points = Column(
        'SpecialPromoPoints', String(length=9), nullable=False, default='000000000')

    frequent_shopper_discounts_to_date = Column(
        'FSDiscountsTD', String(length=9), nullable=False, default='000000000')

    frequent_shopper_discounts_period_to_date = Column(
        'FSDiscountsPTD', String(length=9), nullable=False, default='000000000')

    electronic_coupon_to_date = Column(
        'ElectronicCouponTD', String(length=9), nullable=False, default='000000000')

    electronic_coupon_period_to_date = Column(
        'ElectronicCouponPTD', String(length=9), nullable=False, default='000000000')

    auto_discount1_flag = Column(
        'AutoDisc1Flag', String(length=1), nullable=False, default='0')

    auto_discount2_flag = Column(
        'AutoDisc2Flag', String(length=1), nullable=False, default='0')

    auto_discount4_flag = Column(
        'AutoDisc4Flag', String(length=1), nullable=False, default='0')

    auto_discount5_flag = Column(
        'AutoDisc5Flag', String(length=1), nullable=False, default='0')

    cash_only = Column(
        'CashOnly', String(length=1), nullable=False, default='0')

    check_account_number = Column(
        'ChkAcctNum', String(length=20), nullable=False, default='00000000000000000000')

    transit_number = Column(
        'TransitNum', String(length=9), nullable=False, default='000000000')

    blanks = Column(
        'Blanks', String(length=12), nullable=True, default='000000000000')

    def __repr__(self):
        return "Customer(account_number={0})".format(
            repr(self.account_number))


class CustomerExtended(Base):
    """
    Represents extended information about a customer.
    """

    __tablename__ = 'CustomerExtended'

    account_number = Column(
        'AccountNumber', String(length=12), primary_key=True, nullable=True)

    address2 = Column(
        'Address2', String(length=25), nullable=True, default='')

    email = Column(
        'Email1', String(length=50), nullable=True, default='')

    email2 = Column(
        'Email2', String(length=50), nullable=True, default='')

    ship_to_address = Column(
        'ShipTo1', String(length=25), nullable=True, default='')

    ship_to_address2 = Column(
        'ShipTo2', String(length=25), nullable=True, default='')

    ship_to_city = Column(
        'City', String(length=15), nullable=True, default='')

    ship_to_state = Column(
        'State', String(length=2), nullable=True, default='')

    ship_to_zipcode = Column(
        'Zip', String(length=9), nullable=True, default='')

    contact_last_name = Column(
        'ContactLastName', String(length=25), nullable=True, default='')

    contact_first_name = Column(
        'ContactFirstName', String(length=25), nullable=True, default='')

    phone_number2 = Column(
        'Phone2', String(length=10), nullable=True, default='')

    limit_checks_today = Column(
        'LimitChecksToday', String(length=1), nullable=True, default='0')

    limit_checks_this_week = Column(
        'LimitChecksWeek', String(length=1), nullable=True, default='0')

    limit_dollars_today = Column(
        'LimitDollarDay', String(length=9), nullable=True, default='000000000')

    limit_dollars_this_week = Column(
        'LimitDollarWeek', String(length=9), nullable=True, default='000000000')

    memo = Column(
        'Memo', String(length=100), nullable=True, default='')

    drivers_license_number = Column(
        'DriversLicNum', String(length=18), nullable=True, default='')

    drivers_license_state = Column(
        'DriversLicState', String(length=2), nullable=True, default='')

    alternate_lookup = Column(
        'AltLookUp', String(length=20), nullable=True, default='00000000000000000000')

    blanks = Column(
        'Blanks', String(length=58), nullable=True, default='0000000000000000000000000000000000000000000000000000000000')

    def __repr__(self):
        return "CustomerExtended(account_number={0})".format(
            repr(self.account_number))


Customer.extended = relationship(
    CustomerExtended,
    primaryjoin=CustomerExtended.account_number == Customer.account_number,
    foreign_keys=[CustomerExtended.account_number],
    uselist=False,
    backref='customer')

def getset_factory(collection_class, proxy):
    def getter(obj):
        if obj is None:
            return ''
        return getattr(obj, proxy.value_attr)
    def setter(obj, val):
        setattr(obj, proxy.value_attr, val)
    return getter, setter

def extended_proxy(attr):
    def creator(val):
        obj = CustomerExtended()
        setattr(obj, attr, val)
        return obj
    proxy = association_proxy(
        'extended', attr, creator=creator,
        getset_factory=getset_factory)
    setattr(Customer, attr, proxy)

extended_proxy('address2')
extended_proxy('email')
extended_proxy('email2')
extended_proxy('ship_to_address')
extended_proxy('ship_to_address2')
extended_proxy('ship_to_city')
extended_proxy('ship_to_state')
extended_proxy('ship_to_zipcode')
extended_proxy('contact_last_name')
extended_proxy('contact_first_name')
extended_proxy('phone_number2')
extended_proxy('limit_checks_today')
extended_proxy('limit_checks_this_week')
extended_proxy('limit_dollars_today')
extended_proxy('limit_dollars_this_week')
extended_proxy('memo')
extended_proxy('drivers_license_number')
extended_proxy('drivers_license_state')
extended_proxy('alternate_lookup')


class NegativeCheck(Base):
    """
    Represents a "negative check" record.
    """

    __tablename__ = 'NegativeCheck'

    account_number = Column(
        'AccountNumber', String(length=20), primary_key=True, nullable=False)

    last_name = Column(
        'LastName', String(length=25), nullable=False)

    first_name = Column(
        'FirstName', String(length=25), nullable=False)

    address = Column(
        'Address', String(length=25), nullable=False)

    city = Column(
        'City', String(length=15), nullable=False)

    state = Column(
        'State', String(length=2), nullable=False)

    zipcode = Column(
        'Zipcode', String(length=9), nullable=False)

    phone_number = Column(
        'PhoneNumber', String(length=10), nullable=False)

    social_security_number = Column(
        'SSNumber', String(length=9), nullable=False)

    _DateOpened = Column(
        'DateOpened', String(length=6), nullable=False)

    date_opened = date_property('_DateOpened', '%m%d%y')

    tax_exempt_code = Column(
        'TaxExemptCode', String(length=12), nullable=False)

    checking_account_number = Column(
        'CheckingAcctNum', String(length=20), nullable=False)

    _NumChecksTendered = Column(
        'NumChecksTendered', String(length=3), nullable=False)

    checks_tendered_count = int_property('_NumChecksTendered', 3)

    _NumNSFChecks = Column(
        'NumNSFChecks', String(length=3), nullable=False)

    checks_nsf_count = int_property('_NumNSFChecks', 3)

    _ChecksTenderedAmount = Column(
        'ChecksTenderedAmount', String(length=7), nullable=False)

    checks_tendered_amount = decimal_property('_ChecksTenderedAmount', 7)

    _NSFChecksAmount = Column(
        'NSFChecksAmount', String(length=7), nullable=False)

    checks_nsf_amount = decimal_property('_NSFChecksAmount', 7)

    _DateOfLastPurchase = Column(
        'DateOfLastPurchase', String(length=6), nullable=False)

    date_of_last_purchase = date_property('_DateOfLastPurchase', '%m%d%y')

    comments = Column(
        'Comments', String(length=30), nullable=False)

    _OverrideCount = Column(
        'OverrideCount', String(length=3), nullable=False)

    override_count = int_property('_OverrideCount', 3)

    account_status = Column(
        'AccountStatus', String(length=1), nullable=False)

    transit_routing_number = Column(
        'TransitRoutingNumber', String(length=9), nullable=False)

    blanks = Column(
        'Blanks', String(length=66), nullable=True)

    def __repr__(self):
        return "NegativeCheck(account_number={0})".format(
            repr(self.account_number))


NegativeCheck.customer = relationship(
    Customer,
    primaryjoin=Customer.account_number == NegativeCheck.account_number,
    foreign_keys=[Customer.account_number],
    uselist=False)


class EJTransaction(Base):
    """
    Represents an electronic journal transaction.
    """

    __tablename__ = 'tabEJTransaction'

    transaction_id = Column(
        'TransactionID', Integer(), primary_key=True, nullable=False)

    transaction_number = Column(
        'TransactionNum', String(length=8), nullable=False)

    cashier_id = Column(
        'CashierID', SmallInteger(), nullable=False)

    cashier_name = Column(
        'CashierName', String(length=20), nullable=True)

    register_number = Column(
        'RegisterNum', SmallInteger(), nullable=False)

    start_date = Column(
        'StartDate', Date(), nullable=False)

    end_date = Column(
        'EndDate', Date(), nullable=False)

    start_time = Column(
        'StartTime', Time(), nullable=False)

    end_time = Column(
        'EndTime', Time(), nullable=False)

    @property
    def started(self):
        """
        The date and time when the transaction started.
        """
        if self.start_date is None:
            return self.start_time
        if self.start_time is None:
            return self.start_date
        return datetime.datetime.combine(self.start_date, self.start_time)

    @property
    def ended(self):
        """
        The date and time when the transaction ended.
        """
        if self.end_date is None:
            return self.end_time
        if self.end_time is None:
            return self.end_date
        return datetime.datetime.combine(self.end_date, self.end_time)

    transaction_type = Column(
        'TransactionType', SmallInteger(), nullable=False)

    shoppers_hotline = Column(
        'ShoppersHotline', String(length=12), nullable=True)

    transaction_amount = Column(
        'TransactionAmt', Numeric(precision=19, scale=4), nullable=True)

    item_subtotal_amount = Column(
        'ItemSubtotalAmt', Numeric(precision=19, scale=4), nullable=True)

    total_discount_amount = Column(
        'TotalDiscAmt', Numeric(precision=19, scale=4), nullable=True)

    total_void_amount = Column(
        'TotalVoidAmt', Numeric(precision=19, scale=4), nullable=True)

    total_return_amount = Column(
        'TotalReturnAmt', Numeric(precision=19, scale=4), nullable=True)

    total_tt_discount_amount = Column(
        'TotalTTDiscAmt', Numeric(precision=19, scale=4), nullable=True)

    number_of_items = Column(
        'NumberOfItems', SmallInteger(), nullable=True)

    original_transaction_number = Column(
        'OrigTransactionNum', String(length=8), nullable=True)

    retrieved_flag = Column(
        'RetrievedFlag', Boolean(), nullable=False)

    pf_recovery_flag = Column(
        'PFRecoveryFlag', Boolean(), nullable=False)

    cancel_flag = Column(
        'CancelFlag', Boolean(), nullable=False)

    suspend_flag = Column(
        'SuspendFlag', Boolean(), nullable=False)

    online_flag = Column(
        'OnlineFlag', SmallInteger(), nullable=True)

    register_version = Column(
        'RegisterVer', String(length=15), nullable=True)

    transaction_seconds = Column(
        'TransSeconds', Integer(), nullable=True)

    idle_seconds = Column(
        'IdleSeconds', Integer(), nullable=True)

    ring_seconds = Column(
        'RingSeconds', Integer(), nullable=True)

    tender_seconds = Column(
        'TenderSeconds', Integer(), nullable=True)

    scan_rate = Column(
        'ScanRate', Numeric(precision=10, scale=2), nullable=True)

    scanned_entries = Column(
        'ScannedEntries', SmallInteger(), nullable=True)

    keyed_entries = Column(
        'KeyedEntries', SmallInteger(), nullable=True)

    open_department_entries = Column(
        'OpenDeptEntries', SmallInteger(), nullable=True)

    aff_flag = Column(
        'AFFFlag', SmallInteger(), nullable=True)

    def __repr__(self):
        return "EJTransaction(transaction_id={0}, transaction_number={1})".format(
            repr(self.transaction_id), repr(self.transaction_number))


class EJCustomer(Base):
    """
    Represents a customer association with an EJ transaction.
    """

    __tablename__ = 'tabEJCustomer'

    transaction_id = Column(
        'TransactionID', Integer(), primary_key=True, nullable=False)

    customer_number = Column(
        'CustomerNum', String(length=12), primary_key=True, nullable=False)

    last_name = Column(
        'LastName', String(length=25), nullable=True)

    first_name = Column(
        'FirstName', String(length=25), nullable=True)

    restricted_action_flag = Column(
        'RestrictedActionFlag', SmallInteger(), nullable=True)

    approval_mode = Column(
        'ApprovalMode', SmallInteger(), nullable=True)

    birthdate = Column(
        'Birthdate', Date(), nullable=True)

    verification_id = Column(
        'VerificationID', String(length=20), nullable=True)

    origin = Column(
        'Origin', SmallInteger(), nullable=True)

    frequent_shopper_level = Column(
        'FreqShopLevel', SmallInteger(), nullable=True)

    def __repr__(self):
        return "EJCustomer(transaction_id={0}, customer_number={1})".format(
            repr(self.transaction_id), repr(self.customer_number))


EJCustomer.customer = relationship(
    Customer,
    primaryjoin=Customer.account_number == EJCustomer.customer_number,
    foreign_keys=[Customer.account_number],
    uselist=False)

EJTransaction.customers = relationship(
    EJCustomer,
    primaryjoin=EJCustomer.transaction_id == EJTransaction.transaction_id,
    foreign_keys=[EJCustomer.transaction_id],
    backref='transaction')


class EJItem(Base):
    """
    Represents a line item within an EJ transaction.
    """

    __tablename__ = 'tabEJItem'

    transaction_id = Column(
        'TransactionID', Integer(), primary_key=True, nullable=False)

    sequence_number = Column(
        'SequenceNum', Integer(), primary_key=True, nullable=False)

    item_type = Column(
        'ItemType', SmallInteger(), nullable=False)

    upc = Column(
        'UPC', String(length=20), nullable=False)

    item_description = Column(
        'ItemDesc', String(length=20), nullable=True)

    department_number = Column(
        'DepartmentNum', SmallInteger(), nullable=False)

    subdepartment_number = Column(
        'SubDeptNum', SmallInteger(), nullable=True)

    quantity = Column(
        'Quantity', Integer(), nullable=False)

    adjective_quantity = Column(
        'AdjectiveQty', Integer(), nullable=True)

    adjective_level = Column(
        'AdjectiveLevel', SmallInteger(), nullable=True)

    sold_at_price = Column(
        'SoldAtPrice', Numeric(precision=19, scale=4), nullable=True)

    discount_amount = Column(
        'DiscountAmt', Numeric(precision=19, scale=4), nullable=True)

    bottle_deposit_amount = Column(
        'BottleDepAmt', Numeric(precision=19, scale=4), nullable=True)

    original_quantity = Column(
        'OriginalQty', Integer(), nullable=True)

    original_price = Column(
        'OriginalPrice', Numeric(precision=19, scale=4), nullable=True)

    regular_price = Column(
        'RegularPrice', Numeric(precision=19, scale=4), nullable=True)

    food_stamp_status = Column(
        'FoodStampStatus', Boolean(), nullable=False)

    tax1_status = Column(
        'Tax1Status', Boolean(), nullable=False)

    tax2_status = Column(
        'Tax2Status', Boolean(), nullable=False)

    tax3_status = Column(
        'Tax3Status', Boolean(), nullable=False)

    discount1_status = Column(
        'Disc1Status', Boolean(), nullable=False)

    discount2_status = Column(
        'Disc2Status', Boolean(), nullable=False)

    discount4_status = Column(
        'Disc4Status', Boolean(), nullable=False)

    discount5_status = Column(
        'Disc5Status', Boolean(), nullable=False)

    frequent_shopper_status = Column(
        'FreqShopperStatus', Boolean(), nullable=False)

    sale_level = Column(
        'SaleLevel', SmallInteger(), nullable=True)

    bottle_link_department_number = Column(
        'BottleLinkDeptNum', SmallInteger(), nullable=True)

    scale_weight = Column(
        'ScaleWeight', Numeric(precision=5, scale=3), nullable=True)

    split_weight = Column(
        'SplitWeight', Numeric(precision=6, scale=3), nullable=True)

    tare_weight = Column(
        'TareWeight', Numeric(precision=5, scale=3), nullable=True)

    tare_code = Column(
        'TareCode', SmallInteger(), nullable=True)

    commodity_code = Column(
        'CommodityCode', String(length=8), nullable=True)

    reason_code = Column(
        'ReasonCode', SmallInteger(), nullable=True)

    report_code = Column(
        'ReportCode', String(length=4), nullable=True)

    online_flag = Column(
        'OnlineFlag', Boolean(), nullable=False)

    void_flag = Column(
        'VoidFlag', Boolean(), nullable=False)

    error_correct_flag = Column(
        'ErrorCorrectFlag', Boolean(), nullable=False)

    scan_flag = Column(
        'ScanFlag', Boolean(), nullable=False)

    override_flag = Column(
        'OverrideFlag', Boolean(), nullable=False)

    return_flag = Column(
        'ReturnFlag', Boolean(), nullable=False)

    refund_flag = Column(
        'RefundFlag', Boolean(), nullable=False)

    not_found_flag = Column(
        'NotFoundFlag', Boolean(), nullable=False)

    suspended_flag = Column(
        'SuspendedFlag', Boolean(), nullable=False)

    retrieved_flag = Column(
        'RetrievedFlag', Boolean(), nullable=False)

    canceled_flag = Column(
        'CanceledFlag', Boolean(), nullable=False)

    unit_price = Column(
        'UnitPrice', Numeric(precision=19, scale=4), nullable=True)

    case_cost = Column(
        'CaseCost', Numeric(precision=19, scale=4), nullable=True)

    case_quantity = Column(
        'CaseQty', Integer(), nullable=True)

    mix_match_code = Column(
        'MixMatchCode', Integer(), nullable=True)

    coupon_code = Column(
        'CouponCode', Integer(), nullable=True)

    fsa_rx_flag = Column(
        'FSARxFlag', Boolean(), nullable=False)

    fsa_non_rx_flag = Column(
        'FSANonRxFlag', Boolean(), nullable=False)

    promotion_id = Column(
        'PromoID', String(length=20), nullable=True)

    def __repr__(self):
        return "EJItem(transaction_id={0}, sequence_number={1})".format(
            repr(self.transaction_id), repr(self.sequence_number))


EJTransaction.items = relationship(
    EJItem,
    primaryjoin=EJItem.transaction_id == EJTransaction.transaction_id,
    foreign_keys=[EJItem.transaction_id],
    order_by=EJItem.sequence_number,
    backref='transaction')


class EJPromotion(Base):
    """
    Represents a promotion within an EJ transaction.
    """

    __tablename__ = 'tabEJPromotions'

    transaction_id = Column(
        'TransactionID', Integer(), primary_key=True, nullable=False)

    sequence_number = Column(
        'SequenceNum', Integer(), primary_key=True, nullable=False)

    promotion_number = Column(
        'PromotionNum', String(length=20), nullable=True)

    promotion_type = Column(
        'PromoType', SmallInteger(), nullable=True)

    quantity = Column(
        'Quantity', Integer(), nullable=True)

    amount = Column(
        'Amount', Numeric(precision=19, scale=4), nullable=True)

    department_number = Column(
        'DepartmentNum', SmallInteger(), nullable=True)

    frequent_shopper_level = Column(
        'FreqShopperLevel', SmallInteger(), nullable=True)

    coupon_trigger = Column(
        'CouponTrigger', SmallInteger(), nullable=True)

    points = Column(
        'Points', Integer(), nullable=True)

    bonus_points = Column(
        'BonusPoints', Integer(), nullable=True)

    promotion_description = Column(
        'PromoDesc', String(length=30), nullable=True)

    multimedia_filename = Column(
        'MultiMediaFN', String(length=35), nullable=True)

    void_flag = Column(
        'VoidFlag', Boolean(), nullable=False)

    error_correct_flag = Column(
        'ErrorCorrectFlag', Boolean(), nullable=False)

    def __repr__(self):
        return "EJPromotion(transaction_id={0}, sequence_number={1})".format(
            repr(self.transaction_id), repr(self.sequence_number))


EJTransaction.promotions = relationship(
    EJPromotion,
    primaryjoin=EJPromotion.transaction_id == EJTransaction.transaction_id,
    foreign_keys=[EJPromotion.transaction_id],
    order_by=EJPromotion.sequence_number,
    backref='transaction')


class EJDiscount(Base):
    """
    Represents a discount on an EJ transaction.
    """

    __tablename__ = 'tabEJDiscount'

    transaction_id = Column(
        'TransactionID', Integer(), primary_key=True, nullable=False)

    sequence_number = Column(
        'SequenceNum', Integer(), primary_key=True, nullable=False)

    discount_number = Column(
        'DiscountNum', SmallInteger(), nullable=False)

    discount_type = Column(
        'DiscountType', SmallInteger(), nullable=True)

    discount_amount = Column(
        'DiscountAmt', Numeric(precision=19, scale=4), nullable=True)

    def __repr__(self):
        return "EJDiscount(transaction_id={0}, sequence_number={1})".format(
            repr(self.transaction_id), repr(self.sequence_number))


EJTransaction.discounts = relationship(
    EJDiscount,
    primaryjoin=EJDiscount.transaction_id == EJTransaction.transaction_id,
    foreign_keys=[EJDiscount.transaction_id],
    order_by=EJDiscount.sequence_number,
    backref='transaction')


class EJTax(Base):
    """
    Represents the tax on an EJ transaction.
    """

    __tablename__ = 'tabEJTax'

    transaction_id = Column(
        'TransactionID', Integer(), primary_key=True, nullable=False)

    tax_number = Column(
        'TaxNum', SmallInteger(), primary_key=True, nullable=False)

    tax_amount = Column(
        'TaxAmt', Numeric(precision=19, scale=4), nullable=True)

    taxable_sales_amount = Column(
        'TaxableSalesAmt', Numeric(precision=19, scale=4), nullable=True)

    food_stamp_taxable_sales_amount = Column(
        'FSTaxableSalesAmt', Numeric(precision=19, scale=4), nullable=True)

    food_stamp_tax_amount = Column(
        'FSTaxAmt', Numeric(precision=19, scale=4), nullable=True)

    tax_exempt_amount = Column(
        'TaxExemptAmt', Numeric(precision=19, scale=4), nullable=True)

    tax_exempt_code = Column(
        'TaxExemptCode', String(length=12), nullable=True)

    def __repr__(self):
        return "EJTax(transaction_id={0}, tax_number={1})".format(
            repr(self.transaction_id), repr(self.tax_number))


EJTransaction.taxes = relationship(
    EJTax,
    primaryjoin=EJTax.transaction_id == EJTransaction.transaction_id,
    foreign_keys=[EJTax.transaction_id],
    order_by=EJTax.tax_number,
    backref='transaction')


class InstoreCharge(Base):
    """
    Represents an in-store charge account.
    """

    __tablename__ = 'InstoreCharge'

    account_number = Column(
        'ACCOUNTNUMBER', String(length=12), primary_key=True, nullable=False)

    _DATEOPENED = Column(
        'DATEOPENED', String(length=6), nullable=False, default='000000')

    date_opened = date_property('_DATEOPENED', '%m%d%y')

    _CREDITLIMIT = Column(
        'CREDITLIMIT', String(length=6), nullable=False, default='000000')

    credit_limit = decimal_property('_CREDITLIMIT', 6)

    credit_status = Column(
        'CREDITSTATUS', String(length=1), nullable=False, default='0')

    _DATEOFLASTPAYMENT = Column(
        'DATEOFLASTPAYMENT', String(length=6), nullable=False, default='000000')

    date_of_last_payment = date_property('_DATEOFLASTPAYMENT', '%y%m%d')

    _AMOUNTOFLASTPAYMENT = Column(
        'AMOUNTOFLASTPAYMENT', String(length=8), nullable=False, default='00000000')

    amount_of_last_payment = decimal_property('_AMOUNTOFLASTPAYMENT', 8)

    _CURRENTBALANCE = Column(
        'CURRENTBALANCE', String(length=8), nullable=False, default='00000000')

    current_balance = decimal_property('_CURRENTBALANCE', 8)

    _PREVIOUSBALANCE = Column(
        'PREVIOUSBALANCE', String(length=8), nullable=False, default='00000000')

    previous_balance = decimal_property('_PREVIOUSBALANCE', 8)

    _PAGENUMBER = Column(
        'PAGENUMBER', String(length=2), nullable=False, default='00')

    page_number = int_property('_PAGENUMBER', 2)

    _LINENUMBER = Column(
        'LINENUMBER', String(length=2), nullable=False, default='00')

    line_number = int_property('_LINENUMBER', 2)

    finance_charges = Column(
        'FINANCECHARGES', String(length=1), nullable=False, default='0')

    apr = Column(
        'APR', String(length=4), nullable=False, default='0000')

    blanks = Column(
        'BLANKS', String(length=34), nullable=True, default='0000000000000000000000000000000000')

    update_flag = Column(
        'UPDATEFLAG', String(length=1), nullable=False, default='0')

    def __repr__(self):
        return "InstoreCharge(account_number={0})".format(
            repr(self.account_number))


InstoreCharge.customer = relationship(
    Customer,
    primaryjoin=Customer.account_number == InstoreCharge.account_number,
    foreign_keys=[Customer.account_number],
    uselist=False)


class ChargeHistory(Base):
    """
    Represents a charge event within a customer account's history.
    """

    __tablename__ = 'ChargeHistory'

    account_number = Column(
        'ACCOUNTNUMBER', String(length=12), primary_key=True, nullable=True)

    transaction_number = Column(
        'TRANSACTIONNUMBER', String(length=8), primary_key=True, nullable=True)

    _BUSINESSDATE = Column(
        'BUSINESSDATE', String(length=6), nullable=True)

    business_date = date_property('_BUSINESSDATE', '%y%m%d')

    _BUSINESSTIME = Column(
        'BUSINESSTIME', String(length=4), nullable=True)

    business_time = time_property('_BUSINESSTIME', '%H%M')

    @property
    def occurred(self):
        """
        The date and time when the charge occurred.
        """
        if self.business_date is None:
            return self.business_time
        if self.business_time is None:
            return self.business_date
        return datetime.datetime.combine(self.business_date, self.business_time)

    store_number = Column(
        'STORENUMBER', String(length=4), nullable=True)

    register_number = Column(
        'REGISTERNUMBER', String(length=2), nullable=True)

    cashier_number = Column(
        'CASHIERNUMBER', String(length=3), nullable=True)

    activity_code = Column(
        'ACTIVITYCODE', String(length=1), nullable=True)

    _AMOUNT = Column(
        'AMOUNT', String(length=8), nullable=True)

    amount = decimal_property('_AMOUNT', 8)

    export_flag = Column(
        'EXPORTFLAG', String(length=1), nullable=True)

    update_flag = Column(
        'UPDATEFLAG', String(length=1), nullable=True)

    def __repr__(self):
        return "ChargeHistory(account_number={0}, transaction_number={1})".format(
            repr(self.account_number), repr(self.transaction_number))


EJTransaction.charges = relationship(
    ChargeHistory,
    primaryjoin=ChargeHistory.transaction_number == EJTransaction.transaction_number,
    foreign_keys=[ChargeHistory.transaction_number],
    backref='transaction')
