#!/usr/bin/env python
# -*- coding: utf-8  -*-
################################################################################
#
#  pyScanMaster -- Python Interface to ScanMaster POS
#  Copyright © 2013 Sacramento Natural Foods Co-op, Inc
#
#  This file is part of pyScanMaster.
#
#  pyScanMaster is free software: you can redistribute it and/or modify it
#  under the terms of the GNU General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  pyScanMaster is distributed in the hope that it will be useful, but WITHOUT
#  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
#  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
#  more details.
#
#  You should have received a copy of the GNU General Public License along with
#  pyScanMaster.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################

"""
SSXNCFP Data Utilities
"""

from ..db import model


__all__ = ['parse_ssxncfp_record', 'write_ssxncfp_record']


ssxncfp_record_format = [
    # offset    # length        # attribute
    (0,         20,             'account_number'),
    (20,        25,             'last_name'),
    (45,        25,             'first_name'),
    (70,        25,             'address'),
    (95,        15,             'city'),
    (110,       2,              'state'),
    (112,       9,              'zipcode'),
    (121,       10,             'phone_number'),
    (131,       9,              'social_security_number'),
    (140,       6,              'date_opened'),
    (146,       12,             'tax_exempt_code'),
    (158,       20,             'checking_account_number'),
    (178,       3,              'checks_tendered_count'),
    (181,       3,              'checks_nsf_count'),
    (184,       7,              'checks_tendered_amount'),
    (191,       7,              'checks_nsf_amount'),
    (198,       6,              'date_of_last_purchase'),
    (204,       30,             'comments'),
    (234,       3,              'override_count'),
    (237,       1,              'account_status'),
    (238,       9,              'transit_routing_number'),
    (247,       67,             'blanks'),
    ]


def parse_ssxncfp_record(record):
    """
    Parse a line from a "Negative Check Import" data file.

    :param record: Line from the data file.
    :type record: string

    :returns: A :class:`..db.model.NegativeCheck` instance.
    """
    last_field = ssxncfp_record_format[-1]
    record_length = last_field[0] + last_field[1]
    if len(record) != record_length:
        raise ValueError("Record should be {0} characters long, but is {1}: {2}".format(
                record_length, len(record), repr(record)))

    check = model.NegativeCheck()
    for offset, length, field in ssxncfp_record_format:
        data = record[offset:offset + length]
        data = data.strip()
        setattr(check, field, data)
    return check


def write_ssxncfp_record(ssxncfp_file, check, action_code):
    """
    Write a record to a "Negative Check Import" data file.

    :param ssxncfp_file: File-like object which as been opened for writing.

    :param check: Reference to a :class:`.db.model.NegativeCheck` instance,
       whose attributes will constitute the data record.

    :param action_code: One of the action codes defined by ScanMaster:

       * ``'A'`` - Add
       * ``'U'`` - Update
       * ``'D'`` - Delete
    """
    # Hack due to difference between ScanMaster database schema and file format
    # specification (former uses 66 characters; latter 67).
    blanks = check.blanks
    if blanks == '0' * 66:
        blanks += '0'
    record = ''.join((
            action_code,
            '{0:20.20s}'.format(check.account_number),
            '{0:25.25s}'.format(check.last_name),
            '{0:25.25s}'.format(check.first_name),
            '{0:25.25s}'.format(check.address),
            '{0:15.15s}'.format(check.city),
            '{0:2.2s}'.format(check.state),
            '{0:9.9s}'.format(check.zipcode),
            '{0:10.10s}'.format(check.phone_number),
            '{0:9.9s}'.format(check.social_security_number),
            '{0:6.6s}'.format(check._DateOpened),
            '{0:12.12s}'.format(check.tax_exempt_code),
            '{0:20.20s}'.format(check.checking_account_number),
            '{0:3.3s}'.format(check._NumChecksTendered),
            '{0:3.3s}'.format(check._NumNSFChecks),
            '{0:7.7s}'.format(check._ChecksTenderedAmount),
            '{0:7.7s}'.format(check._NSFChecksAmount),
            '{0:6.6s}'.format(check._DateOfLastPurchase),
            '{0:30.30s}'.format(check.comments),
            '{0:3.3s}'.format(check._OverrideCount),
            '{0:1.1s}'.format(check.account_status),
            '{0:9.9s}'.format(check.transit_routing_number),
            '{0:67.67s}'.format(blanks),
            ))
    ssxncfp_file.write(record)
