#!/usr/bin/env python
# -*- coding: utf-8  -*-
################################################################################
#
#  pyScanMaster -- Python Interface to ScanMaster POS
#  Copyright © 2013 Sacramento Natural Foods Co-op, Inc
#
#  This file is part of pyScanMaster.
#
#  pyScanMaster is free software: you can redistribute it and/or modify it
#  under the terms of the GNU General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  pyScanMaster is distributed in the hope that it will be useful, but WITHOUT
#  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
#  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
#  more details.
#
#  You should have received a copy of the GNU General Public License along with
#  pyScanMaster.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################

"""
Transaction Log (Tlog) Utilities
"""

import sys
import datetime
import argparse
from decimal import Decimal
from collections import namedtuple
from struct import unpack


# Global registry/cache of generated `namedtuple` classes.
_consolidated_tuple_classes = {}
_nonconsolidated_tuple_classes = {}


class Record(object):
    """
    A record within a transaction, as represented within a Tlog file.
    """

    def __repr__(self):
        # Replace "FooTuple" with "Foo" but otherwise use tuple's repr() value.
        return '{0}{1}'.format(
            self.__class__.__name__,
            repr(self._tuple)[len(self._tuple.__class__.__name__):])

    def __getattr__(self, name):
        # Leverage underlying tuple fields.
        if name == '_tuple':
            return None
        return getattr(self._tuple, name)

    def data_format(self, consolidated=True):
        """
        The binary data format string, as used by ``struct``.

        This format is automatically generated from the ``format_legend`` which
        is defined on each record class.
        """
        fields = [f[1] for f in self.format_legend]
        if not consolidated:
            fields.append('129s')
        return '<{0}'.format(''.join(fields))

    def tuple_class(self, consolidated=True):
        """
        The tuple class used by this record type.

        This will be a ``namedtuple`` class, automatically generated from the
        ``format_legend`` which is defined on each record class.
        """
        if consolidated:
            registry = _consolidated_tuple_classes
        else:
            registry = _nonconsolidated_tuple_classes
        tuple_class_name = '{0}Tuple'.format(self.__class__.__name__)
        if tuple_class_name not in registry:
            fields = [f[2] for f in self.format_legend]
            if not consolidated:
                fields.append('realtime_cruft')
            registry[tuple_class_name] = namedtuple(tuple_class_name, fields)
        return registry[tuple_class_name]

    def initialize(self, line, consolidated=True):
        """
        Initialize a record's data.

        :param line: A string of data representing a line from a Tlog file.
           Note that this line of text must contain a specific number of bytes.
        :type line: string

        :param consolidated: Whether the Tlog line format is of the
           "consolidated" variety.  See :func:`parse_tlog()` for more
           information.
        :type consolidated: boolean
        """
        data_format = self.data_format(consolidated)
        tuple_class = self.tuple_class(consolidated)
        self._tuple = tuple_class._make(unpack(data_format, line))


class GenericRecord(Record):
    """
    Generic type which all records must match.

    This format is used primarily to determine the *actual* record type, which
    will then be used to further parse the record.
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '103s',         'record_data'),
        (124,           '1s',           'online_flag'),
        )


class PowerOn(Record):
    """
    A transaction record indicating power on (?).
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '4s',           'store_number'),
        (25,            '2s',           'lane_number'),
        (27,            '3s',           'manager_id'),
        (30,            '94s',          'blanks'),
        (124,           '1s',           'online_flag'),
        )


class Configuration(Record):
    """
    Active configuration for the transaction.
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '1s',           'add_double_mfg_coupon_to_sales'),
        (22,            '1s',           'add_store_coupon_to_sales'),
        (23,            '1s',           'add_double_store_coupon_to_sales'),
        (24,            '1s',           'accountability_by_lane'),
        (25,            '1s',           'add_line_item_discounts_to_sales'),
        (26,            '98s',          'blanks'),
        (124,           '1s',           'online_flag'),
        )


class SignOnOffRecord(Record):
    """
    A sign-on or sign-off event.
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '4s',           'store_number'),
        (25,            '2s',           'lane_number'),
        (27,            '3s',           'cashier_id'),
        (30,            '12s',          'customer_number'),
        (42,            '3s',           'manager_id'),
        (45,            '54s',          'blanks1'),
        (99,            '15s',          'pos_version'),
        (114,           '4s',           'blanks2'),
        (118,           '6s',           'idle_time'),
        (124,           '1s',           'online_flag'),
        )


class SignOn(SignOnOffRecord):
    """
    A sign-on event.
    """


class SignOff(SignOnOffRecord):
    """
    A sign-off event.
    """


class NonResetableTotalsRecord(Record):
    """
    A generic record containing non-resetable totals data.

    This class of record is further broken down into 5 types.  All 5 record
    types appear together, in sequence.
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '1s',           'record_number'),
        (22,            '103s',         'record_data'),
        )


class NonResetableTotalsOpen(NonResetableTotalsRecord):
    """
    Non-resetable totals upon open.
    """


class NonResetableTotalsClose(NonResetableTotalsRecord):
    """
    Non-resetable totals upon close.
    """


class NonResetableTotalsRecord1(Record):
    """
    Non-resetable totals record, type 1.
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '1s',           'record_number'),
        (22,            '15s',          'positive_sales'),
        (37,            '15s',          'negative_sales'),
        (52,            '72s',          'blanks'),
        (124,           '1s',           'online_flag'),
        )


class NonResetableTotalsOpen1(NonResetableTotalsRecord1):
    """
    Non-resetable totals upon open, type 1.
    """


class NonResetableTotalsClose1(NonResetableTotalsRecord1):
    """
    Non-resetable totals upon close, type 1.
    """


class NonResetableTotalsRecord2(Record):
    """
    Non-resetable totals record, type 2.
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '1s',           'record_number'),
        (22,            '15s',          'tax1_total'),
        (37,            '15s',          'tax2_total'),
        (52,            '15s',          'tax3_total'),
        (67,            '57s',          'blanks'),
        (124,           '1s',           'online_flag'),
        )


class NonResetableTotalsOpen2(NonResetableTotalsRecord2):
    """
    Non-resetable totals upon open, type 2.
    """


class NonResetableTotalsClose2(NonResetableTotalsRecord2):
    """
    Non-resetable totals upon close, type 2.
    """


class NonResetableTotalsRecord3(Record):
    """
    Non-resetable totals record, type 3.
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '1s',           'record_number'),
        (22,            '15s',          'discount1_total'),
        (37,            '15s',          'discount2_total'),
        (52,            '15s',          'frequent_shopper_discount'),
        (67,            '15s',          'discount4_total'),
        (82,            '42s',          'blanks'),
        (124,           '1s',           'online_flag'),
        )


class NonResetableTotalsOpen3(NonResetableTotalsRecord3):
    """
    Non-resetable totals upon open, type 3.
    """


class NonResetableTotalsClose3(NonResetableTotalsRecord3):
    """
    Non-resetable totals upon close, type 3.
    """


class NonResetableTotalsRecord4(Record):
    """
    Non-resetable totals record, type 4.
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '1s',           'record_number'),
        (22,            '15s',          'discount5_total'),
        (37,            '15s',          'store_coupon_total'),
        (52,            '15s',          'double_store_coupon_total'),
        (67,            '15s',          'double_mfgr_coupon_total'),
        (82,            '42s',          'blanks'),
        (124,           '1s',           'online_flag'),
        )


class NonResetableTotalsOpen4(NonResetableTotalsRecord4):
    """
    Non-resetable totals upon open, type 4.
    """


class NonResetableTotalsClose4(NonResetableTotalsRecord4):
    """
    Non-resetable totals upon close, type 4.
    """


class NonResetableTotalsRecord5(Record):
    """
    Non-resetable totals record, type 5.
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '1s',           'record_number'),
        (22,            '15s',          'gift_certificates_sold'),
        (37,            '15s',          'money_orders_sold'),
        (52,            '15s',          'money_order_fees_collected'),
        (67,            '15s',          'paid_out_total'),
        (82,            '15s',          'paid_in_total'),
        (97,            '27s',          'blanks'),
        (124,           '1s',           'online_flag'),
        )


class NonResetableTotalsOpen5(NonResetableTotalsRecord5):
    """
    Non-resetable totals upon open, type 5.
    """


class NonResetableTotalsClose5(NonResetableTotalsRecord5):
    """
    Non-resetable totals upon close, type 5.
    """


class SaleHeader(Record):
    """
    The sale header for a transaction.
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '4s',           'store_number'),
        (25,            '2s',           'lane_number'),
        (27,            '3s',           'cashier_id'),
        (30,            '12s',          'customer_number'),
        (42,            '3s',           'manager_id'),
        (45,            '12s',          'shopper_hotline_number'),
        (57,            '4s',           'blanks1'),
        (61,            '1s',           'retrieved_flag'),
        (62,            '8s',           'original_transaction'),
        (70,            '1s',           'customer_restricted_action_flag'),
        (71,            '1s',           'customer_approval_mode'),
        (72,            '6s',           'customer_birthdate'),
        (78,            '20s',          'customer_verification_id'),
        (98,            '1s',           'customer_origin'),
        (99,            '15s',          'blanks2'),
        (114,           '1s',           'customer_frequent_shopper_level'),
        (115,           '3s',           'blanks3'),
        (118,           '6s',           'idle_time'),
        (124,           '1s',           'online_flag'),
        )


class PickupLoanRecord(Record):
    """
    A money pickup or loan event.
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '2s',           'tender_code'),
        (23,            '8s',           'amount'),
        (31,            '5s',           'count'),
        (36,            '3s',           'cashier_id'),
        (39,            '2s',           'lane_number'),
        (41,            '3s',           'action_by'),
        (44,            '10s',          'native_amount'),
        (54,            '70s',          'blanks'),
        (124,           '1s',           'online_flag'),
        )


class Pickup(PickupLoanRecord):
    """
    A money pickup event.
    """


class Loan(PickupLoanRecord):
    """
    A money loan event.
    """


class PaymentRecord(Record):
    """
    A charge payment, or paid-in, or paid-out.
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '6s',           'amount'),
        (27,            '20s',          'account_number'),
        (47,            '3s',           'manager_id'),
        (50,            '2s',           'tender_code'),
        (52,            '72s',          'blanks'),
        (124,           '1s',           'online_flag'),
        )


class ChargePayment(PaymentRecord):
    """
    A charge payment.
    """


class PaidIn(PaymentRecord):
    """
    A paid-in event.
    """


class PaidOut(PaymentRecord):
    """
    A paid-out event.
    """


class UPCRecord(Record):
    """
    A transaction record containing a UPC (item).
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '20s',          'upc'),
        (41,            '2s',           'department_number'),
        (43,            '3s',           'subdepartment_number'),
        (46,            '6s',           'sold_at_price'),
        (52,            '2s',           'quantity'),
        (54,            '1s',           'food_stamp_fsa_status'),
        (55,            '1s',           'tax1_status'),
        (56,            '1s',           'tax2_status'),
        (57,            '1s',           'tax3_status'),
        (58,            '1s',           'discount14_status'),
        (59,            '1s',           'discount25_status'),
        (60,            '1s',           'frequent_shopper_status'),
        (61,            '6s',           'discount_amount'),
        (67,            '6s',           'original_price'),
        (73,            '6s',           'bottle_deposit_amount'),
        (79,            '2s',           'adjective_quantity'),
        (81,            '1s',           'item_type'),
        (82,            '1s',           'sale_level'),
        (83,            '8s',           'commodity_code'),
        (91,            '1s',           'reason_code'),
        (92,            '3s',           'manager_id'),
        (95,            '2s',           'bottle_link_department'),
        (97,            '4s',           'scale_weight'),
        (101,           '6s',           'split_weight'),
        (107,           '2s',           'tare_code'),
        (109,           '4s',           'tare_weight'),
        (113,           '1s',           'adjective_level'),
        (114,           '6s',           'regular_price'),
        (120,           '4s',           'report_code'),
        (124,           '1s',           'online_flag'),
        )

    @property
    def error_correct(self):
        """
        Boolean indicating whether or not the line item was error-corrected.
        """
        return isinstance(self, ErrorCorrect)

    @property
    def void(self):
        """
        Boolean indicating whether or not the line item was voided.
        """
        return isinstance(self, Void)


# TODO: Eventually remove this.
class UPC(UPCRecord):
    """
    A simple UPC (product) line item.
    """


class ScannedUPC(UPCRecord):
    """
    A scanned line item for product being purchased.
    """


class KeyedUPC(UPCRecord):
    """
    A keyed line item for product being purchased.
    """


class KeyedUPCOverride(UPCRecord):
    """
    A keyed line item for product being purchased, with price override.
    """


class OpenDepartmentUPC(UPCRecord):
    """
    An open department line item.
    """


class OpenDepartmentReturn(UPCRecord):
    """
    An open department return line item.
    """


class ScannedReturn(UPCRecord):
    """
    A scanned line item for product being returned.
    """


class KeyedReturn(UPCRecord):
    """
    A keyed line item for product being returned.
    """


class KeyedReturnOverride(UPCRecord):
    """
    A keyed line item for product being returned, with price override.
    """


class MissingUPC(UPCRecord):
    """
    A line item for a missing product.

    .. todo::
       This record type is not well known...a better description is in order.
    """


class ErrorCorrect(UPCRecord):
    """
    An "error correct" line item.
    """


class Void(UPCRecord):
    """
    Represents a voided line item.
    """


class FSATotals(Record):
    """
    Flexible spending account totals.
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '6s',           'rx_eligible_amount'),
        (27,            '6s',           'nonrx_eligible_amount'),
        (33,            '6s',           'rx_tender_amount'),
        (39,            '6s',           'nonrx_tender_amount'),
        (45,            '79s',          'blanks'),
        (124,           '1s',           'online_flag'),
        )


class DiscountRecord(Record):
    """
    A discount on either a line item, or the total sale.
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '7s',           'discount1_amount'),
        (28,            '7s',           'discount2_amount'),
        (35,            '7s',           'frequent_shopper_discount_amount'),
        (42,            '7s',           'discount4_amount'),
        (49,            '7s',           'discount5_amount'),
        (56,            '3s',           'discount1_manager_id'),
        (59,            '3s',           'discount2_manager_id'),
        (62,            '3s',           'frequent_shopper_discount_manager_id'),
        (65,            '3s',           'discount4_manager_id'),
        (68,            '3s',           'discount5_manager_id'),
        (71,            '53s',          'blanks'),
        (124,           '1s',           'online_flag'),
        )


class LineItemDiscount(DiscountRecord):
    """
    A discount on a line item.
    """


class TotalSaleDiscount(DiscountRecord):
    """
    A discount on a total sale.
    """


class SaleTotal(Record):
    """
    The sale totals.
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '6s',           'positive_sale_amount'),
        (27,            '6s',           'negative_sale_amount'),
        (33,            '6s',           'positive_tax1_amount'),
        (39,            '6s',           'negative_tax1_amount'),
        (45,            '6s',           'positive_tax2_amount'),
        (51,            '6s',           'negative_tax2_amount'),
        (57,            '6s',           'positive_tax3_amount'),
        (63,            '6s',           'negative_tax3_amount'),
        (69,            '6s',           'positive_discount_amount'),
        (75,            '6s',           'negative_discount_amount'),
        (81,            '4s',           'item_count'),
        (85,            '6s',           'taxable1_sales_amount'),
        (91,            '6s',           'taxable2_sales_amount'),
        (97,            '6s',           'taxable3_sales_amount'),
        (103,           '6s',           'fs_taxable1_sales_amount'),
        (109,           '6s',           'fs_taxable2_sales_amount'),
        (115,           '6s',           'fs_taxable3_sales_amount'),
        (121,           '3s',           'blanks'),
        (124,           '1s',           'online_flag'),
        )


class Tender(Record):
    """
    Tender received on the transaction.
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '2s',           'tender_code'),
        (23,            '6s',           'tender_amount'),
        (29,            '20s',          'control_number'),
        (49,            '4s',           'expiration_date'),
        (53,            '3s',           'manager_id'),
        (56,            '6s',           'cash_change'),
        (62,            '7s',           'food_stamp_tax1_amount'),
        (69,            '7s',           'food_stamp_tax2_amount'),
        (76,            '7s',           'food_stamp_tax3_amount'),
        (83,            '9s',           'transit_number'),
        (92,            '6s',           'food_stamp_change'),
        (98,            '7s',           'foreign_currency_tender_native_amount'),
        (105,           '6s',           'foreign_currench_change'),
        (111,           '6s',           'foreign_currench_change_native_amount'),
        (117,           '5s',           'check_number'),
        (122,           '1s',           'associated_file_flag'),
        (123,           '1s',           'fsa_flag'),
        (124,           '1s',           'online_flag'),
        )


class TaxExempted(Record):
    """
    Tax exemption status for the transaction.
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '12s',          'tax_code'),
        (33,            '1s',           'tax1_exempted_flag'),
        (34,            '7s',           'tax1_exempted_amount'),
        (41,            '1s',           'tax2_exempted_flag'),
        (42,            '7s',           'tax2_exempted_amount'),
        (49,            '1s',           'tax3_exempted_flag'),
        (50,            '7s',           'tax3_exempted_amount'),
        (59,            '3s',           'manager_id'),
        (60,            '64s',          'blanks'),
        (124,           '1s',           'online_flag'),
        )


class IdleModeStartStop(Record):
    """
    Beginning and/or end of "idle mode."
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '6s',           'start_date'),
        (27,            '6s',           'start_time'),
        (33,            '6s',           'end_date'),
        (39,            '6s',           'end_time'),
        (45,            '3s',           'manager_id'),
        (48,            '76s',          'blanks'),
        (124,           '1s',           'online_flag'),
        )


class CancelSuspendRecord(Record):
    """
    The canceling or suspending of a transaction.
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '4s',           'store_number'),
        (25,            '2s',           'lane_number'),
        (27,            '3s',           'cashier_id'),
        (30,            '12s',          'customer_number'),
        (42,            '3s',           'manager_id'),
        (45,            '6s',           'blanks1'),
        (51,            '10s',          'transaction_amount'),
        (61,            '1s',           'retrieved_flag'),
        (62,            '8s',           'original_transaction'),
        (70,            '1s',           'customer_restricted_action_flag'),
        (71,            '1s',           'customer_approval_mode'),
        (72,            '6s',           'customer_birthdate'),
        (78,            '20s',          'customer_verification_id'),
        (98,            '1s',           'customer_origin'),
        (99,            '19s',          'blanks2'),
        (118,           '6s',           'idle_time'),
        (124,           '1s',           'online_flag'),
        )


class Cancel(CancelSuspendRecord):
    """
    The canceling of a transaction.
    """


class Suspend(CancelSuspendRecord):
    """
    The suspending of a transaction.
    """


class NoSale(Record):
    """
    A "no sale" event.
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '3s',           'manager_id'),
        (24,            '94s',          'blanks'),
        (118,           '6s',           'idle_time'),
        (124,           '1s',           'online_flag'),
        )


class OpenDepartmentCouponRecord(Record):
    """
    An open department coupon, from the manufacturer or store.
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '2s',           'department_number'),
        (23,            '6s',           'coupon_amount'),
        (29,            '6s',           'old_coupon_amount'),
        (35,            '20s',          'coupon_number'),
        (55,            '2s',           'quantity'),
        (57,            '67s',          'blanks'),
        (124,           '1s',           'online_flag'),
        )


class OpenDepartmentCouponManufacturer(OpenDepartmentCouponRecord):
    """
    An open department coupon from the manufacturer.
    """


class OpenDepartmentCouponStore(OpenDepartmentCouponRecord):
    """
    An open department coupon from the store.
    """


class GiftCertificateMoneyOrderSaleRecord(Record):
    """
    The sale of a gift certificate or money order.
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '6s',           'amount'),
        (27,            '20s',          'document_number'),
        (47,            '3s',           'manager_id'),
        (50,            '74s',          'blanks'),
        (124,           '1s',           'online_flag'),
        )


class GiftCertificateSale(GiftCertificateMoneyOrderSaleRecord):
    """
    The sale of a gift certificate.
    """


class MoneyOrderSale(GiftCertificateMoneyOrderSaleRecord):
    """
    The sale of a money order.
    """


class CouponIssuedOnReceipt(Record):
    """
    The issuance of a coupon along with the receipt.
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '4s',           'coupon_number'),
        (25,            '99s',          'blanks'),
        (124,           '1s',           'online_flag'),
        )


class MediaCashing(Record):
    """
    The cashing of media, e.g. a check.
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '6s',           'amount'),
        (27,            '20s',          'account_number'),
        (47,            '3s',           'manager_id'),
        (50,            '2s',           'tender_code'),
        (52,            '6s',           'cash_back_amount'),
        (58,            '2s',           'check_cashing_media_fee_link'),
        (60,            '2s',           'check_cashing_department'),
        (62,            '6s',           'check_cashing_fee'),
        (68,            '1s',           'associated_file_flag'),
        (69,            '55s',          'blanks'),
        (124,           '1s',           'online_flag'),
        )


class CouponPaidOutRecord(Record):
    """
    A pay-out for a manufacturer or store coupon.
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '6s',           'amount'),
        (27,            '2s',           'department_number'),
        (29,            '18s',          'blanks'),
        (47,            '3s',           'manager_id'),
        (50,            '74s',          'blanks2'),
        (124,           '1s',           'online_flag'),
        )


class CouponPaidOutManufacturer(CouponPaidOutRecord):
    """
    A payout for a manufacturer coupon.
    """


class CouponPaidOutStore(CouponPaidOutRecord):
    """
    A payout for a store coupon.
    """


class MediaSwap(Record):
    """
    The swapping of media (tender).
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '6s',           'swap_amount'),
        (27,            '2s',           'from_tender_code'),
        (29,            '2s',           'to_tender_code'),
        (31,            '3s',           'manager_id'),
        (34,            '16s',          'blanks'),
        (50,            '1s',           'reason_code'),
        (51,            '10s',          'native_amount'),
        (61,            '63s',          'blanks2'),
        (124,           '1s',           'online_flag'),
        )


class ElectronicPromotionRecord(Record):
    """
    An electronic promotion, from manufacturer or store.
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '20s',          'promotion_number'),
        (41,            '6s',           'quantity'),
        (47,            '6s',           'amount'),
        (53,            '2s',           'department_number'),
        (55,            '1s',           'frequent_shopper_level'),
        (56,            '1s',           'type'),
        (57,            '9s',           'points'),
        (66,            '9s',           'bonus_points'),
        (75,            '6s',           'weight'),
        (81,            '43s',          'blanks'),
        (124,           '1s',           'online_flag'),
        )


class ElectronicPromotionManufacturer(ElectronicPromotionRecord):
    """
    An electronic promotion from the manufacturer.
    """


class ElectronicPromotionStore(ElectronicPromotionRecord):
    """
    An electronic promotion from the store.
    """


class MultimediaPromotion(Record):
    """
    A transaction record representing a multimedia promotion.
    """

    format_legend = (
        # offset        data            fieldname
        (0,             '8s',           'transaction_number'),
        (8,             '6s',           'date'),
        (14,            '6s',           'time'),
        (20,            '1s',           'record_code'),
        (21,            '20s',          'promotion_number'),
        (41,            '1s',           'type'),
        (42,            '1s',           'frequent_shopper_level'),
        (43,            '6s',           'quantity'),
        (49,            '35s',          'multimedia_filename'),
        (84,            '12s',          'vendor_number'),
        (96,            '28s',          'blanks'),
        (124,           '1s',           'online_flag'),
        )


# Mapping of record code values to corresponding record classes.
RECORD_REGISTRY = {
    '~':        PowerOn,
    'K':        Configuration,
    'A':        NonResetableTotalsOpen,
    'B':        NonResetableTotalsClose,
    'Y':        SignOn,
    'Z':        SignOff,
    'H':        SaleHeader,
    '<':        Pickup,
    '>':        Loan,
    'P':        ChargePayment,
    '$':        PaidIn,
    '@':        PaidOut,
    'U':        ScannedUPC,
    'W':        KeyedUPC,
    'R':        ScannedReturn,
    'X':        KeyedReturn,
    '{':        MissingUPC,
    '#':        UPC,
    'E':        UPC,
    '}':        UPC,
    '?':        UPC,
    'F':        UPC,
    '[':        UPC,
    '\\':       UPC,
    'I':        KeyedUPCOverride,
    ']':        UPC,
    '/':        UPC,
    'J':        KeyedReturnOverride,
    '+':        OpenDepartmentUPC,
    '-':        OpenDepartmentReturn,
    '(':        OpenDepartmentCouponManufacturer,
    ')':        OpenDepartmentCouponStore,
    'G':        GiftCertificateSale,
    'M':        MoneyOrderSale,
    ':':        ElectronicPromotionManufacturer,
    ';':        ElectronicPromotionStore,
    '|':        MultimediaPromotion,
    'L':        LineItemDiscount,
    'D':        TotalSaleDiscount,
    'Q':        TaxExempted,
    '£':        FSATotals,
    'O':        SaleTotal,
    'T':        Tender,
    'C':        CouponIssuedOnReceipt,
    '!':        CouponPaidOutManufacturer,
    ',':        CouponPaidOutStore,
    '`':        IdleModeStartStop,
    '&':        MediaCashing,
    '%':        MediaSwap,
    'V':        Void,
    '.':        ErrorCorrect,
    '*':        Cancel,
    'S':        Suspend,
    'N':        NoSale,
    }


def parse_tlog_file(tlog_path, consolidated=True):
    """
    Parse a Tlog file to obtain "raw" data records.

    This is a convenience function which wraps :func:`parse_tlog()`.

    :param tlog_path: Path to the Tlog file which should be parsed.
    :type tlog_path: string

    :param consolidated: Same meaning as for :func:`parse_tlog()`.
    :type consolidated: boolean
    """
    tlog_file = open(tlog_path, 'rb')
    for record in parse_tlog(tlog_file, consolidated):
        yield record
    tlog_file.close()


def parse_tlog(tlog_file, consolidated=True):
    """
    Parse a file-like object to obtain "raw" Tlog data records.

    This is a generator which yields :class:`Record` instances as they are read
    from the file.

    :param tlog_file: File (or file-like) object containing Tlog data.
    :type tlog_path: object

    :param consolidated: Whether the Tlog file format is of the "consolidated"
       variety.  The only Tlog files which are *not* consolidated (of which the
       author is aware) are the "real-time" files which may be provided by the
       ScanMaster EJ processor.  In any case this is a ScanMaster distinction;
       see that documentation for more information.
    :type consolidated: boolean
    """
    for line in tlog_file:

        # Parse generic record first, to determine actual record type.
        line = line.rstrip('\r\n')
        generic = GenericRecord()
        generic.initialize(line, consolidated)
        record_class = RECORD_REGISTRY[generic.record_code]
        record = record_class()
        record.initialize(line, consolidated)

        # Non-resetable totals record have multiple subtypes.
        if record_class is NonResetableTotalsOpen:
            record_class = eval('NonResetableTotalsOpen{0}'.format(record.record_number))
            record = record_class()
            record.initialize(line, consolidated)
        elif record_class is NonResetableTotalsClose:
            record_class = eval('NonResetableTotalsClose{0}'.format(record.record_number))
            record = record_class()
            record.initialize(line, consolidated)

        yield record


class TlogTransaction(object):
    """
    Simple aggregation class for Tlog transaction records.
    """

    def __init__(self, record):
        """
        Constructor; initial record establishes transaction number.
        """
        self.transaction_number = record.transaction_number
        self.records = [record]

    def __repr__(self):
        return "TlogTransaction(transaction_number={0})".format(
            repr(self.transaction_number))

    def has(self, record_class):
        """
        Whether the transaction contains a record of the given class.
        """
        for record in self.records:
            if isinstance(record, record_class):
                return True
        return False

    def get(self, record_class):
        """
        Returns a record of the given class.

        This method expects there to be one and only one record of the
        specified class present in the transaction's record collection.  If
        this is not the case, an exception will be raised.
        """
        found = None
        for record in self.records:
            if isinstance(record, record_class):
                if found:
                    raise ValueError("Found multiple records of class {0} for transaction {1}".format(
                            record_class.__name__, self.transaction_number))
                found = record
        if not found:
            raise ValueError("Found no records of class {0} for transaction {1}".format(
                    record_class.__name__, self.transaction_number))
        return found

    def get_all(self, record_class):
        """
        Returnsa a list of all records of the given class.
        """
        records = []
        for record in self.records:
            if isinstance(record, record_class):
                records.append(record)
        return records

    def decimal(self, amount):
        """
        Convert an amount string to decimal.
        """
        return Decimal('{0}.{1}'.format(amount[:-2], amount[-2:]))

    def timestamp(self, record):
        """
        Returns the date and time of a given transaction record.
        """
        return datetime.datetime.strptime(record.date + record.time, '%m%d%y%H%M%S')

    @property
    def regular(self):
        """
        Whether the transaction was "regular".

        I.e., if the transaction was a simple sale or return, as opposed to a
        no-sale or sign-on, etc.
        """
        return self.has(SaleHeader) and self.has(SaleTotal)

    @property
    def sale_header(self):
        """
        Returns the transaction's sale header record.
        """
        return self.get(SaleHeader)

    @property
    def items(self):
        """
        Returns all item (UPC) records in the transaction.
        """
        return self.get_all(UPCRecord)

    @property
    def open_department_coupons(self):
        """
        Returns all open department coupon records in the transaction.
        """
        return self.get_all(OpenDepartmentCouponRecord)

    @property
    def payments(self):
        """
        Returns all payment records in the transaction.
        """
        return self.get_all(PaymentRecord)

    @property
    def sale_total(self):
        """
        Returns the transaction's sale total record.
        """
        return self.get(SaleTotal)

    @property
    def line_item_discounts(self):
        """
        Returns all line item discount records in the transaction.
        """
        return self.get_all(LineItemDiscount)

    @property
    def total_sale_discounts(self):
        """
        Returns all total sale discount records in the transaction.
        """
        return self.get_all(TotalSaleDiscount)

    @property
    def electronic_promotions(self):
        """
        Returns all electronic promotion records in the transaction.
        """
        return self.get_all(ElectronicPromotionRecord)


def parse_tlog_transactions(tlog_path, consolidated=True):
    """
    Parse a Tlog file for complete transactions.

    This is a generator which yields :class:`TlogTransaction` instances, as
    complete transactions are encountered within the Tlog file.
    """
    current_transaction = None
    current_transaction_number = None

    for record in parse_tlog_file(tlog_path, consolidated):

        if record.transaction_number != current_transaction_number:
            if current_transaction:
                yield current_transaction
            current_transaction_number = record.transaction_number
            current_transaction = TlogTransaction(record)

        elif current_transaction:
            current_transaction.records.append(record)

    # Yield once more to account for EOF.
    if current_transaction:
        yield current_transaction


def decipher_tlog():
    """
    Decipher a Tlog file, with human-readable output.
    """
    parser = argparse.ArgumentParser(
        description="Decipher a Tlog file, with human-readable output.")
    parser.add_argument('tlog_file', metavar='TLOG_FILE',
                        help="Path to the ScanMaster Tlog file.")
    parser.add_argument('-C', '--consolidated', action='store_true',
                        help="Should be set if the Tlog file is consolidated.")
    parser.add_argument('-O', '--output', metavar='FILE',
                        help="Path to output file.  If not set, standard output is assumed.")
    parser.add_argument('-T', '--transactions', nargs='+', metavar='NUMBER',
                        help="One or more transaction numbers by which to filter the output.  "
                        "If set, *only* these transactions will be included in the output.  "
                        "Otherwise *all* transactions will be included.")

    args = parser.parse_args()
    stdout = sys.stdout
    if args.output:
        stdout = open(args.output, 'wb')
    for record in parse_tlog_file(args.tlog_file, args.consolidated):
        if not args.transactions or record.transaction_number in args.transactions:
            stdout.write('{0}\n'.format(repr(record)))
    if stdout is not sys.stdout:
        stdout.close()
