import logging
import random

from pyage.core.address import Addressable
from pyage.core.inject import Inject, InjectWithDefault


logger = logging.getLogger(__name__)


class EmasAgent(Addressable):
    @Inject("locator", "migration", "evaluation", "crossover", "mutation", "emas", "transferred_energy")
    def __init__(self, genotype, energy, name=None):
        self.name = name
        super(EmasAgent, self).__init__()
        self.genotype = genotype
        self.energy = energy
        self.steps = 0
        self.dead = False
        self.evaluation.process([genotype])

    def step(self):
        self.steps += 1
        try:
            neighbour = self.locator.get_neighbour(self)
            if neighbour:
                if self.emas.should_die(self):
                    self.death()
                elif self.emas.should_reproduce(self, neighbour):
                    self.emas.reproduce(self, neighbour)
                else:
                    self.meet(neighbour)
                if self.emas.can_migrate(self) and self.migration.migrate(self):
                    self.locator.remove_agent(self)
                elif self.emas.should_move(self):
                    self.move()
        except:
            logging.exception('')

    def get_fitness(self):
        return self.genotype.fitness

    def get_best_genotype(self):
        return self.genotype

    def add_energy(self, energy):
        self.energy += energy
        if self.emas.should_die(self):
            self.death()

    def get_energy(self):
        return self.energy

    def get_genotype(self):
        return self.genotype

    def meet(self, neighbour):
        logger.debug(str(self) + "meets" + str(neighbour))
        if self.get_fitness() > neighbour.get_fitness():
            transfered_energy = min(self.transferred_energy, neighbour.energy)
            self.energy += transfered_energy
            neighbour.add_energy(-transfered_energy)
        elif self.get_fitness() < neighbour.get_fitness():
            transfered_energy = min(self.transferred_energy, self.energy)
            self.energy -= transfered_energy
            neighbour.add_energy(transfered_energy)
        if self.emas.should_die(self):
            self.death()

    def death(self):
        self.distribute_energy()
        self.energy = 0
        self.dead = True
        self.parent.remove_agent(self)
        logger.debug(str(self) + "died!")

    def distribute_energy(self):
        logger.debug("%s is dying, energy level: %d" % (self, self.energy))
        if self.energy > 0:
            siblings = set(self.parent.get_agents())
            siblings.remove(self)
            portion = self.energy / len(siblings)
            if portion > 0:
                logger.debug("passing %d portion of energy to %d agents" % (portion, len(siblings)))
                for agent in siblings:
                    agent.add_energy(portion)
            left = self.energy % len(siblings)
            logger.debug("distributing %d left energy" % left)
            while left > 0:
                e = min(left, 1)
                siblings.pop().add_energy(e)
                left -= e

    def move(self):
        allowed_moves = self.locator.get_allowed_moves(self)
        if allowed_moves:
            self.locator.remove_agent(self)
            destination = get_random_move(allowed_moves)
            self.locator.add_agent(self, destination)
            logger.debug("%s moved to %s" % (self, destination))

    def __repr__(self):
        return "<EmasAgent@%s>" % self.get_address()


class EmasService(object):
    @Inject("minimal_energy", "reproduction_minimum", "migration_minimum", "newborn_energy", "locator")
    @InjectWithDefault(("move_probability", 0.1))
    def __init__(self):
        super(EmasService, self).__init__()

    def should_die(self, agent):
        return agent.get_energy() <= self.minimal_energy and not agent.dead

    def should_reproduce(self, a1, a2):
        return a1.get_energy() > self.reproduction_minimum and a2.get_energy() > self.reproduction_minimum \
               and self.locator.get_allowed_moves(a1)

    def can_migrate(self, agent):
        return agent.get_energy() > self.migration_minimum and len(agent.parent.get_agents()) > 10

    def should_move(self, agent):
        return random.random() < self.move_probability

    def reproduce(self, a1, a2):
        logger.debug(str(a1) + " " + str(a2) + " reproducing!")
        energy = self.newborn_energy / 2 * 2
        a1.energy -= self.newborn_energy / 2
        a2.add_energy(-self.newborn_energy / 2)
        genotype = a1.crossover.cross(a1.genotype, a2.get_genotype())
        a1.mutation.mutate(genotype)
        newborn = EmasAgent(genotype, energy)
        self.locator.add_agent(newborn, get_random_move(self.locator.get_allowed_moves(a1)))
        a1.parent.add_agent(newborn)


def get_random_move(allowed_moves):
    destination = random.sample(allowed_moves, 1)[0]
    return destination