#ifndef AMGCL_BACKEND_DETAIL_MATRIX_OPS_HPP
#define AMGCL_BACKEND_DETAIL_MATRIX_OPS_HPP

/*
The MIT License

Copyright (c) 2012-2014 Denis Demidov <dennis.demidov@gmail.com>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

/**
 * \file    amgcl/adapter/detail/matrix_ops.hpp
 * \author  Denis Demidov <dennis.demidov@gmail.com>
 * \brief   Sparse matrix operations for matrices that provide row_iterator.
 */

#include <boost/type_traits.hpp>
#include <amgcl/backend/interface.hpp>

namespace amgcl {
namespace backend {
namespace detail {

template <class Matrix, class Enable = void>
struct use_builtin_matrix_ops : boost::false_type {};

} // namespace detail

template <class Matrix, class Vector1, class Vector2>
struct spmv_impl<
    Matrix, Vector1, Vector2,
    typename boost::enable_if<
        typename detail::use_builtin_matrix_ops<Matrix>::type
        >::type
    >
{
    typedef typename value_type<Matrix>::type V;

    static void apply(
            V alpha, const Matrix &A, const Vector1 &x, V beta, Vector2 &y
            )
    {
        const ptrdiff_t n = static_cast<ptrdiff_t>( rows(A) );

        if (beta) {
#pragma omp parallel for
            for(ptrdiff_t i = 0; i < n; ++i) {
                V sum = 0;
                for(typename row_iterator<Matrix>::type a = row_begin(A, i); a; ++a)
                    sum += a.value() * x[ a.col() ];
                y[i] = alpha * sum + beta * y[i];
            }
        } else {
#pragma omp parallel for
            for(ptrdiff_t i = 0; i < n; ++i) {
                V sum = 0;
                for(typename row_iterator<Matrix>::type a = row_begin(A, i); a; ++a)
                    sum += a.value() * x[ a.col() ];
                y[i] = alpha * sum;
            }
        }
    }
};

template <class Matrix, class Vector1, class Vector2, class Vector3>
struct residual_impl<
    Matrix, Vector1, Vector2, Vector3,
    typename boost::enable_if<
        typename detail::use_builtin_matrix_ops<Matrix>::type
        >::type
    >
{
    static void apply(
            Vector1 const &rhs,
            Matrix  const &A,
            Vector2 const &x,
            Vector3       &res
            )
    {
        typedef typename value_type<Matrix>::type V;

        const ptrdiff_t n = static_cast<ptrdiff_t>( rows(A) );

#pragma omp parallel for
        for(ptrdiff_t i = 0; i < n; ++i) {
            V sum = 0;
            for(typename row_iterator<Matrix>::type a = row_begin(A, i); a; ++a)
                sum += a.value() * x[ a.col() ];
            res[i] = rhs[i] - sum;
        }
    }
};

} // namespace backend
} // namespace amgcl

#endif
