"""Pyblish Endpoint Server

| Endpoint                | Description
|-------------------------|--------------
| /processes              | List processes
| /processes/<id>         | Query and manipulate process
| /instances              | List instances
| /instances/<id>         | Query instance
| /instances/<id>/nodes   | List nodes
| /instances/<id>/data    | Query and manipulate data

"""

# Standard library
import os
import logging

# Dependencies
import flask
import flask.ext.restful

# Local library
import service
import resource

log = logging.getLogger("endpoint")

prefix = "/pyblish/v0.1"
resource_map = {
    "/processes": resource.ProcessesListApi,
    "/processes/<process_id>": resource.ProcessesApi,
    "/application": resource.ApplicationApi,
    "/application/shutdown": resource.ApplicationShutdownApi,
    "/instances": resource.InstancesListApi,
    "/instances/<instance_id>": resource.InstancesApi,
    "/instances/<instance_id>/nodes": resource.NodesListApi,
    "/instances/<instance_id>/data": resource.DataListApi,
    "/instances/<instance_id>/data/<data_id>": resource.DataApi,
}

endpoint_map = {
    "/processes/<process_id>":          "process",
    "/processes":                       "processes",
    "/application":                     "application",
    "/application/shutdown":            "application.shutdown",
    "/instances/<instance_id>":         "instance",
    "/instances":                       "instances",
    "/instances/<instance_id>/nodes":   "instance.nodes",
    "/instances/<instance_id>/data":    "instance.data"
}


def create_app():
    app = flask.Flask(__name__)
    api = flask.ext.restful.Api(app)
    resource.setup_message_queue()

    # Map resources to URIs
    for uri, _resource in resource_map.items():
        endpoint = endpoint_map.get(uri)
        api.add_resource(_resource, prefix + uri, endpoint=endpoint)

    return app, api


def start_debug_server(port, **kwargs):
    formatter = logging.Formatter("%(levelname)-8s %(message)s")
    handler = logging.StreamHandler()
    handler.setFormatter(formatter)

    log.addHandler(handler)
    log.setLevel(logging.DEBUG)

    os.environ["ENDPOINT_PORT"] = str(port)

    service.MockService.SLEEP = 3
    service.register_service(service.MockService)

    app, api = create_app()
    app.run(debug=True, port=port)


if __name__ == '__main__':
    import argparse

    parser = argparse.ArgumentParser()
    parser.add_argument("--port", type=int, default=6000, help="Port to use")

    args = parser.parse_args()

    start_debug_server(**args.__dict__)
