import unittest

import pycoind

class TestScriptTransactions(unittest.TestCase):

    class MockTransaction(pycoind.blockchain.transaction.Transaction):
      '''Script only requires the wrapper to include the raw Txn. Many
         operations will otherwise fail on this object, so do not try this
         in your day-to-day life. Badness may ensue.'''

      def __init__(self, txn_bytes):
          (dummy, self._transaction) = pycoind.protocol.Txn.parse(txn_bytes)

    @staticmethod
    def bv(value):
        'Create a stack literal sequence for an integer value.'

        v = pycoind.script.ByteVector.from_value(value).vector
        return chr(len(v)) + v

    def setUp(self):
        pass

    def check_transaction(self, txn_bytes, pk_script, input_index):
        txn = self.MockTransaction(txn_bytes)
        txio = pycoind.script.Script(txn)
        valid = txio.verify_input(input_index, pk_script)
        self.assertTrue(valid)

        # with self.assertRaises(exception):
        #self.assetRaises(exception, script.script, broken

    def test_hashall(self):
        # Block: bitcoin@40094
        # Txn: 854a3aaaee36be32e441cc8f86890b7eebe4710f30003dfff362cc2d90d163d4

        txn_bytes = '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'.decode('hex')
        pk_script = '76a9143f320f852a51643d3ffbaa1f49bfe521dd97764a88ac'.decode('hex')
        input_index = 70

        self.check_transaction(txn_bytes, pk_script, input_index)

    def test_hashall_anyonecanpay(self):
        # Block: bitcoin@276459
        # Txn: 33854f625c90e3287eae951103489a2449f91bfe039aa4d4c810bd66450edbf1

        txn_bytes = '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'.decode('hex')
        pk_script = '76a9149bc0bbdd3024da4d0c38ed1aecf5c68dd1d3fa1288ac'.decode('hex')
        input_index = 12

        self.check_transaction(txn_bytes, pk_script, input_index)

    def test_hashnone(self):
        # Block: bitcoin@
        # Txn: 

        txn_bytes = ''.decode('hex')
        pk_script = ''.decode('hex')
        input_index = 3

        #self.check_transaction(txn_bytes, pk_script, input_index)

    def test_hashnone_anyonecanpay(self):
        # Block: bitcoin@
        # Txn: 

        txn_bytes = ''.decode('hex')
        pk_script = ''.decode('hex')
        input_index = 3

        #self.check_transaction(txn_bytes, pk_script, input_index)

    def test_hashsingle(self):
        # Block: bitcoin@238797
        # Txn: afd9c17f8913577ec3509520bd6e5d63e9c0fd2a5f70c787993b097ba6ca9fae

        txn_bytes = '010000000370ac0a1ae588aaf284c308d67ca92c69a39e2db81337e563bf40c59da0a5cf63000000006a4730440220360d20baff382059040ba9be98947fd678fb08aab2bb0c172efa996fd8ece9b702201b4fb0de67f015c90e7ac8a193aeab486a1f587e0f54d0fb9552ef7f5ce6caec032103579ca2e6d107522f012cd00b52b9a65fb46f0c57b9b8b6e377c48f526a44741affffffff7d815b6447e35fbea097e00e028fb7dfbad4f3f0987b4734676c84f3fcd0e804010000006b483045022100c714310be1e3a9ff1c5f7cacc65c2d8e781fc3a88ceb063c6153bf950650802102200b2d0979c76e12bb480da635f192cc8dc6f905380dd4ac1ff35a4f68f462fffd032103579ca2e6d107522f012cd00b52b9a65fb46f0c57b9b8b6e377c48f526a44741affffffff3f1f097333e4d46d51f5e77b53264db8f7f5d2e18217e1099957d0f5af7713ee010000006c493046022100b663499ef73273a3788dea342717c2640ac43c5a1cf862c9e09b206fcb3f6bb8022100b09972e75972d9148f2bdd462e5cb69b57c1214b88fc55ca638676c07cfc10d8032103579ca2e6d107522f012cd00b52b9a65fb46f0c57b9b8b6e377c48f526a44741affffffff0380841e00000000001976a914bfb282c70c4191f45b5a6665cad1682f2c9cfdfb88ac80841e00000000001976a9149857cc07bed33a5cf12b9c5e0500b675d500c81188ace0fd1c00000000001976a91443c52850606c872403c0601e69fa34b26f62db4a88ac00000000'.decode('hex')
        pk_script = '76a914dcf72c4fd02f5a987cf9b02f2fabfcac3341a87d88ac'.decode('hex')
        input_index = 1

        self.check_transaction(txn_bytes, pk_script, input_index)

    def test_hashsingle_anyonecanpay(self):
        # Block: bitcoin@
        # Txn: 

        txn_bytes = ''.decode('hex')
        pk_script = ''.decode('hex')
        input_index = 3

        #self.check_transaction(txn_bytes, pk_script, input_index)

    def check_script(self, script, output):

        # for each output, add the literal and do checkverify
        if output:
            for o in reversed(output):
                if isinstance(o, int):
                    script += self.bv(o)
                elif isinstance(o, str):
                    script += chr(len(o)) + o
                else:
                    raise Exception()
                script += chr(pycoind.script.opcodes.OP_EQUALVERIFY)

        # make sure the stack depth is 0 and return true
        script += (
            chr(pycoind.script.opcodes.OP_DEPTH) +
            self.bv(0) +
            chr(pycoind.script.opcodes.OP_EQUALVERIFY) +
            chr(pycoind.script.opcodes.OP_TRUE)
        )

        # run the script
        result = pycoind.script.Script.process('', script, None, None)

        # check the output (None indicates expected failure)
        if output is None:
            self.assertFalse(result)
        else:
            self.assertTrue(result)

    def test_flow_control_ops(self):
        pass

    def test_literal_ops(self):
        tests = [
            # https://github.com/bitcoin/bitcoin/blob/master/src/test/script_tests.cpp#L110
            ((1, 0x5a), 0x5a),
            ((pycoind.script.opcodes.OP_PUSHDATA1, 1, 0x5a), 0x5a),
            ((pycoind.script.opcodes.OP_PUSHDATA2, 1, 0, 0x5a), 0x5a),
            ((pycoind.script.opcodes.OP_PUSHDATA4, 1, 0, 0, 0, 0x5a), 0x5a),
        ]

        for (inputs, output) in tests:
            script = "".join(chr(i) for i in inputs)
            self.check_script(script, [output])

    def test_stack_ops(self):
        xTop = 19
        (x, x0, x1, x2, x3, x4, x5, x6) = (41, 43, 47, 53, 59, 61, 67, 71)
        tests = [
            ('OP_IFDUP', [0], [0],),
            ('OP_IFDUP', [1], [1, 1],),

            ('OP_DEPTH', [], [0]),
            ('OP_DEPTH', [1,1], [1,1,2]),

            ('OP_DROP', [x], []),
            ('OP_DROP', [], None),

            ('OP_DUP', [x], [x, x]),
            ('OP_DUP', [x1, x], [x1, x, x]),
            ('OP_DUP', [], None),

            ('OP_NIP', [x1,x2], [x2]),
            ('OP_NIP', [x,x2,x3], [x,x3]),
            ('OP_NIP', [x1], None),
            ('OP_NIP', [], None),

            ('OP_OVER', [x1,x2], [x1,x2,x1]),
            ('OP_OVER', [x1], None),
            ('OP_OVER', [], None),

            ('OP_PICK', [x5, x4, x3, x2, x1, x0, 5], [x5, x4, x3, x2, x1, x0, x5]),
            ('OP_PICK', [x5, x4, x3, x2, x1, x0, 2], [x5, x4, x3, x2, x1, x0, x2]),
            # @TODO: pick error cases

            ('OP_ROLL', [x5, x4, x3, x2, x1, x0, 5], [x4, x3, x2, x1, x0, x5]),
            ('OP_ROLL', [x5, x4, x3, x2, x1, x0, 2], [x5, x4, x3, x1, x0, x2]),
            # @TODO: roll error cases (not enough arguments to start, len < n)

            ('OP_ROT', [x1, x2, x3], [x2, x3, x1]),
            ('OP_ROT', [x, x1, x2, x3], [x, x2, x3, x1]),
            # @TODO: rot error cases (ditto)

            ('OP_SWAP', [x1, x2], [x2, x1]),
            ('OP_SWAP', [x, x1, x2], [x, x2, x1]),
            ('OP_SWAP', [x1], None),
            ('OP_SWAP', [], None),

            ('OP_TUCK', [x1, x2], [x2, x1, x2]),
            ('OP_TUCK', [x, x1, x2], [x, x2, x1, x2]),
            # @TODO: tuck error cases

            ('OP_2DROP', [x1, x2], []),
            ('OP_2DROP', [x, x1, x2], [x]),
            ('OP_2DROP', [x1], None),
            ('OP_2DROP', [], None),

            ('OP_2DUP', [x1, x2], [x1, x2, x1, x2]),
            ('OP_2DUP', [x, x1, x2], [x, x1, x2, x1, x2]),
            ('OP_2DUP', [x1], None),
            ('OP_2DUP', [], None),

            ('OP_3DUP', [x1, x2, x3], [x1, x2, x3, x1, x2, x3]),
            ('OP_3DUP', [x, x1, x2, x3], [x, x1, x2, x3, x1, x2, x3]),
            ('OP_3DUP', [x1, x2], None),
            ('OP_3DUP', [x1], None),
            ('OP_3DUP', [], None),

            ('OP_2OVER', [x1, x2, x3, x4], [x1, x2, x3, x4, x1, x2]),
            ('OP_2OVER', [x, x1, x2, x3, x4], [x, x1, x2, x3, x4, x1, x2]),
            # @TODO: 2over error cases

            ('OP_2ROT', [x1, x2, x3, x4, x5, x6], [x3, x4, x5, x6, x1, x2]),
            ('OP_2ROT', [x, x1, x2, x3, x4, x5, x6], [x, x3, x4, x5, x6, x1, x2]),
            # @TODO: 2rot error cases

            ('OP_2SWAP', [x1, x2, x3, x4], [x3, x4, x1, x2]),
            ('OP_2SWAP', [x, x1, x2, x3, x4], [x, x3, x4, x1, x2]),
            # @TODO: 2swap error cases
        ]

        for opname, inputs, outputs in tests:
            # length(x1) + x1 [+ length(x1) + x2 ...] + opcode
            script = "".join(self.bv(i) for i in inputs) + chr(pycoind.script.opcodes.get_opcode(opname))
            self.check_script(script, outputs)

    def test_splice_ops(self):
        tests = [
            (['hello'], ['hello', 5]),
            (['hello', 'world!'], ['hello', 'world!', 6]),
            ([], None),
        ]

        for (inputs, outputs) in tests:
            script = ''.join((chr(len(i)) + i) for i in inputs) + chr(pycoind.script.opcodes.OP_SIZE)
            self.check_script(script, outputs)

    def test_logic_ops(self):
        pass

    def test_math_overflow_ops(self):
        pass

    def test_unary_math_ops(self):
        tests = [
            ('OP_1ADD', -2, -1),
            ('OP_1ADD', -1, 0),
            ('OP_1ADD', 0, 1),
            ('OP_1ADD', 1, 2),

            ('OP_1SUB', -1, -2),
            ('OP_1SUB', 0, -1),
            ('OP_1SUB', 1, 0),
            ('OP_1SUB', 2, 1),

            ('OP_NEGATE', 0, 0),
            ('OP_NEGATE', 1, -1),
            ('OP_NEGATE', -1, 1),

            ('OP_ABS', 0, 0),
            ('OP_ABS', 1, 1),
            ('OP_ABS', -1, 1),

            ('OP_NOT', 0, 1),
            ('OP_NOT', 1, 0),
            ('OP_NOT', -1, 0),
            ('OP_NOT', 200, 0),

            ('OP_0NOTEQUAL', 0, 0),
            ('OP_0NOTEQUAL', 1, 1),
            ('OP_0NOTEQUAL', -1, 1),
        ]

        for opname, input, output in tests:
            # length(input) + input + opcode
            script = self.bv(input) + chr(pycoind.script.opcodes.get_opcode(opname))
            self.check_script(script, [output])

    def test_binary_math_ops(self):
        tests = [
          ('OP_ADD', 1, 2, 3),
          ('OP_ADD', 1, -1, 0),
          ('OP_ADD', 1, -2, -1),
          ('OP_ADD', 1, -2, -1),
          ('OP_ADD', 1, 127, 128),
          ('OP_ADD', 129, 129, 258),
          ('OP_ADD', -1000, -1000, -2000),

          ('OP_SUB', 1, 1, 0),
          ('OP_SUB', 5, 1, 4),
          ('OP_SUB', 1, 5, -4),

          ('OP_BOOLAND', 0, 0, 0),
          ('OP_BOOLAND', 0, 1, 0),
          ('OP_BOOLAND', 1, 0, 0),
          ('OP_BOOLAND', 1, 1, 1),

          ('OP_BOOLOR', 0, 0, 0),
          ('OP_BOOLOR', 0, 1, 1),
          ('OP_BOOLOR', 1, 0, 1),
          ('OP_BOOLOR', 1, 1, 1),

          ('OP_NUMEQUAL', 1, 1, 1),
          ('OP_NUMEQUAL', 1, 5, 0),

          ('OP_NUMNOTEQUAL', 1, 1, 0),
          ('OP_NUMNOTEQUAL', 5, 1, 1),

          ('OP_LESSTHAN', 5, 6, 1),
          ('OP_LESSTHAN', 5, 5, 0),
          ('OP_LESSTHAN', 5, 4, 0),

          ('OP_LESSTHANOREQUAL', 5, 6, 1),
          ('OP_LESSTHANOREQUAL', 5, 5, 1),
          ('OP_LESSTHANOREQUAL', 5, 4, 0),

          ('OP_GREATERTHAN', 5, 6, 0),
          ('OP_GREATERTHAN', 5, 5, 0),
          ('OP_GREATERTHAN', 5, 4, 1),

          ('OP_GREATERTHANOREQUAL', 5, 6, 0),
          ('OP_GREATERTHANOREQUAL', 5, 5, 1),
          ('OP_GREATERTHANOREQUAL', 5, 4, 1),

          ('OP_MIN', 5, 6, 5),
          ('OP_MIN', 6, 5, 5),
          ('OP_MIN', 5, -6, -6),

          ('OP_MAX', 5, 6, 6),
          ('OP_MAX', 6, 5, 6),
          ('OP_MAX', -5, -6, -5),
        ]

        for opname, a, b, output in tests:
            # length(a) + a + length(b) + b + opcode
            script = self.bv(a) + self.bv(b) + chr(pycoind.script.opcodes.get_opcode(opname))
            self.check_script(script, [output])

    def test_multisig(self):
        pass
        # test 1/3 work for 2/3
        # test 2/3 work for 2/3
        # test 3/3 work for 2/3

        # 1 of 2 multi-sig, where one of the public keys is giberish
        # txid: 7aa7f9172660e38236b3bb97830c0b79a6e843ae83145d8707b9b8f249e7c470

    def test_ternary_math_ops(self):
        tests = [
            (pycoind.script.opcodes.OP_WITHIN, [2, 1, 3], [1]),
            (pycoind.script.opcodes.OP_WITHIN, [1, 1, 3], [1]),
            (pycoind.script.opcodes.OP_WITHIN, [3, 1, 3], [0]),

            (pycoind.script.opcodes.OP_WITHIN, [0, -5, 5], [1]),
            (pycoind.script.opcodes.OP_WITHIN, [0, 5, -5], [0]),

            (pycoind.script.opcodes.OP_WITHIN, [10, 12], None),
            (pycoind.script.opcodes.OP_WITHIN, [12, 10], None),
            (pycoind.script.opcodes.OP_WITHIN, [12], None),
            (pycoind.script.opcodes.OP_WITHIN, [], None),

        ]
        for (opname, inputs, outputs) in tests:
            # length(x1) + x1 [+ length(x1) + x2 ...] + opcode
            script = "".join(self.bv(i) for i in inputs) + chr(opname)
            self.check_script(script, outputs)

    def test_crypto_ops(self):
        tests = [
            # http://en.wikipedia.org/wiki/RIPEMD
            ('OP_RIPEMD160', 'The quick brown fox jumps over the lazy dog', '37f332f68db77bd9d7edd4969571ad671cf9dd3b'),
            ('OP_RIPEMD160', '', '9c1185a5c5e9fc54612808977ee8f548b2258d31'),

            # http://en.wikipedia.org/wiki/SHA_1
            ('OP_SHA1', 'The quick brown fox jumps over the lazy dog', '2fd4e1c67a2d28fced849ee1bb76e7391b93eb12'),
            ('OP_SHA1', '', 'da39a3ee5e6b4b0d3255bfef95601890afd80709'),

            # http://en.wikipedia.org/wiki/Sha256
            ('OP_SHA256', 'The quick brown fox jumps over the lazy dog', 'd7a8fbb307d7809469ca9abcb0082e4f8d5651e46d3cdb762d02d0bf37c9e592'),
            ('OP_SHA256', '', 'e3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855'),

            # https://en.bitcoin.it/wiki/Technical_background_of_Bitcoin_addresses
            ('OP_HASH160', '0450863AD64A87AE8A2FE83C1AF1A8403CB53F53E486D8511DAD8A04887E5B23522CD470243453A299FA9E77237716103ABC11A1DF38855ED6F2EE187E9C582BA6'.decode('hex'), '010966776006953D5567439E5E39F86A0D273BEE'),

            # https://en.bitcoin.it/wiki/Technical_background_of_Bitcoin_addresses
            ('OP_HASH256', '00010966776006953D5567439E5E39F86A0D273BEE'.decode('hex'), 'D61967F63C7DD183914A4AE452C9F6AD5D462CE3D277798075B107615C1A8A30'),
        ]

        for opname, input, output in tests:
            # length(input) + input + opcode
            script = chr(len(input)) + input + chr(pycoind.script.opcodes.get_opcode(opname))
            self.check_script(script, [output.decode('hex')])

    def test_reserved_ops(self):
        pass

suite = unittest.TestLoader().loadTestsFromTestCase(TestScriptTransactions)
unittest.TextTestRunner(verbosity=2).run(suite)
