#  Copyright (C) 2013 Rocky Bernstein
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.

from import_relative import import_relative
# Our local modules
Mbase_cmd = import_relative('base_cmd', top_name='pydbgr')

class AliasCommand(Mbase_cmd.DebuggerCommand):
    """**alias** *alias-name* *debugger-command*

Add alias *alias-name* for a debugger command *debugger-comand*.

Add an alias when you want to use a command abbreviation for a command
that would otherwise be ambigous. For example, by default we make `s`
be an alias of `step` to force it to be used. Without the alias, `s`
might be `step`, `show`, or `set` among others

**Example:**

    alias cat list   # "cat prog.py" is the same as "list prog.py"
    alias s   step   # "s" is now an alias for "step".
                     # The above example is done by default.

See also `unalias` and `show alias`."""
    
    category      = 'support'
    min_args      = 0
    max_args      = 2  # Need at most this many
    name          = 'alias'
    need_stack    = True
    short_help    = 'Add an alias for a debugger command'
  
    # Run command. 
    def run(self, args):
        if len(args) == 1:
            self.proc.commands['show'].run(['show', 'alias'])
        elif len(args) == 2:
            self.proc.commands['show'].run(['show', 'alias', args[1]])
        else:
            junk, al, command = args
            if command in self.proc.commands:
                if al in self.proc.aliases:
                    old_command = self.proc.aliases[al]
                    self.msg(("Alias '%s#' for command '%s'replaced old " + 
                              "alias for '%s'.") %
                             (al, command, od_command, old_command))
                else:
                    self.msg("New alias '%s' for command '%s' created." % 
                             (al, command))
                    pass
                self.proc.aliases[al] = command
            else:
                self.errmsg(("You must alias to a command name, and '%s' " +
                             'and is not one.') % command)  
                pass
            return
        pass
    pass

if __name__ == '__main__':
    # Demo it.
    cmdproc      = import_relative('cmdproc', '..')
    debugger     = import_relative('debugger', '...')
    d            = debugger.Debugger()
    cp           = d.core.processor
    command      = AliasCommand(cp)
    command.run(['alias', 'yy', 'foo'])
    command.run(['alias', 'yy',' foo'])
    command.run(['alias', 'yy' 'step'])
    command.run(['alias'])
    command.run(['alias', 'yy' 'next'])
    pass
