"""
Boolean Tables

Interface Classes:
    Table
        TruthTable
        ImplicantTable
"""

__copyright__ = "Copyright (c) 2012, Chris Drake"

from pyeda import __version__
from pyeda.common import bit_on
from pyeda.boolfunc import Function
from pyeda.expr import Expression

# Positional Cube Notation
PC_VOID, PC_ONE, PC_ZERO, PC_DC = range(4)

PC_VALS = [PC_ZERO, PC_ONE]

PC_STR = {
    PC_VOID : "?",
    PC_ZERO : "0",
    PC_ONE  : "1",
    PC_DC   : "-"
}

class Table(Function):
    def __repr__(self):
        return self.__str__()


class TruthTable(Table):

    def __init__(self, arg):
        self._data = bytearray()
        if isinstance(arg, Expression):
            # If the expression lists the variables in a, b, c order, 'a' will
            # be in the LSB. Conventionally we read left-to-right, so reverse
            # the inputs to put 'a' in the MSB.
            self._inputs = arg.inputs[::-1]
            pos = 0
            for n in range(2 ** self.degree):
                point = {v: bit_on(n, i) for i, v in enumerate(self._inputs)}
                output = arg.restrict(point)
                if pos == 0:
                    self._data.append(0)
                self._data[-1] += (PC_VALS[output] << pos)
                pos = (pos + 2) & 7
        else:
            raise ValueError("invalid input")

    def __str__(self):
        """Return the table in Espresso input file format.

        >>> from pyeda.expr import var
        >>> a, b = map(var, "ab")
        >>> TruthTable(a * -b)
        .i 2
        .o 1
        .ilb a b
        00 0
        01 0
        10 1
        11 0
        .e
        """
        #s = ["# auto-generated by pyeda version: ", __version__, "\n"]
        s = [".i ", str(self.degree), "\n"]
        s += [".o 1", "\n"]
        s.append(".ilb ")
        s.append(" ".join(str(v) for v in reversed(self._inputs)))
        s.append("\n")
        for n in range(2 ** self.degree):
            s += [str(bit_on(n, i)) for i in range(self.degree - 1, -1, -1)]
            # n >> 2 == n / 4; n & 3 == n % 4
            byte = self._data[n >> 2]
            output = (byte >> ((n & 3) << 1)) & 3
            s += [" ", PC_STR[output], "\n"]
        s.append(".e")
        return "".join(s)

    # From Function
    @property
    def support(self):
        return set(self._inputs)

    # Specific to TruthTable
    @property
    def inputs(self):
        """Return the support set in name/index order."""
        return self._inputs

    @property
    def degree(self):
        return len(self._inputs)
