# -*- coding: utf8 -*-
from __future__ import division  # confidence high
from __future__ import with_statement

import itertools
import sys
import warnings

import numpy as np

import pyfits

from pyfits.card import _pad
from pyfits.util import encode_ascii, _pad_length, BLOCK_SIZE, StringIO
from pyfits.tests import PyfitsTestCase
from pyfits.tests.util import catch_warnings, ignore_warnings, CaptureStdio

from nose.tools import assert_equal, assert_false, assert_raises, assert_true


PY3 = sys.version_info[0] >= 3


class TestOldApiHeaderFunctions(PyfitsTestCase):
    """
    Tests that specifically use attributes and methods from the old
    Header/CardList API from PyFITS 3.0 and prior.

    This tests backward compatibility support for those interfaces.
    """

    def test_ascardimage_verifies_the_comment_string_to_be_ascii_text(self):
        # the ascardimage() verifies the comment string to be ASCII text
        c = pyfits.Card.fromstring('abc     = +  2.1   e + 12 / abcde\0')
        assert_raises(Exception, c.ascardimage)

    def test_rename_key(self):
        """Test backwards compatibility support for Header.rename_key()"""
        header = pyfits.Header([('A', 'B', 'C'), ('D', 'E', 'F')])
        header.rename_key('A', 'B')
        assert_true('A' not in header)
        assert_true('B' in header)
        assert_equal(header[0], 'B')
        assert_equal(header['B'], 'B')
        assert_equal(header.comments['B'], 'C')

    def test_add_commentary(self):
        header = pyfits.Header([('A', 'B', 'C'), ('HISTORY', 1),
                                ('HISTORY', 2), ('HISTORY', 3), ('', '', ''),
                                ('', '', '')])
        header.add_history(4)
        # One of the blanks should get used, so the length shouldn't change
        assert_equal(len(header), 6)
        assert_equal(header.cards[4].value, 4)
        assert_equal(header['HISTORY'], [1, 2, 3, 4])

        header.add_history(0, after='A')
        assert_equal(len(header), 6)
        assert_equal(header.cards[1].value, 0)
        assert_equal(header['HISTORY'], [0, 1, 2, 3, 4])

        header = pyfits.Header([('A', 'B', 'C'), ('', 1), ('', 2), ('', 3),
                                ('', '', ''), ('', '', '')])
        header.add_blank(4)
        # This time a new blank should be added, and the existing blanks don't
        # get used... (though this is really kinda sketchy--there's a
        # distinction between truly blank cards, and cards with blank keywords
        # that isn't currently made int he code)
        assert_equal(len(header), 7)
        assert_equal(header.cards[6].value, 4)
        assert_equal(header[''], [1, 2, 3, '', '', 4])

        header.add_blank(0, after='A')
        assert_equal(len(header), 8)
        assert_equal(header.cards[1].value, 0)
        assert_equal(header[''], [0, 1, 2, 3, '', '', 4])

    def test_has_key(self):
        header = pyfits.Header([('A', 'B', 'C'), ('D', 'E', 'F')])
        assert_true(header.has_key('A'))
        assert_true(header.has_key('D'))
        assert_false(header.has_key('C'))

    def test_totxtfile(self):
        hdul = pyfits.open(self.data('test0.fits'))
        hdul[0].header.toTxtFile(self.temp('header.txt'))
        hdu = pyfits.ImageHDU()
        hdu.header.update('MYKEY', 'FOO', 'BAR')
        hdu.header.fromTxtFile(self.temp('header.txt'), replace=True)
        assert_equal(len(hdul[0].header.ascard), len(hdu.header.ascard))
        assert_false(hdu.header.has_key('MYKEY'))
        assert_false(hdu.header.has_key('EXTENSION'))
        assert_true(hdu.header.has_key('SIMPLE'))

        # Write the hdu out and read it back in again--it should be recognized
        # as a PrimaryHDU
        hdu.writeto(self.temp('test.fits'), output_verify='ignore')
        assert_true(isinstance(pyfits.open(self.temp('test.fits'))[0],
                               pyfits.PrimaryHDU))

        hdu = pyfits.ImageHDU()
        hdu.header.update('MYKEY', 'FOO', 'BAR')
        hdu.header.fromTxtFile(self.temp('header.txt'))
        # hdu.header should have MYKEY keyword, and also adds PCOUNT and
        # GCOUNT, giving it 3 more keywords in total than the original
        assert_equal(len(hdul[0].header.ascard), len(hdu.header.ascard) - 3)
        assert_true(hdu.header.has_key('MYKEY'))
        assert_false(hdu.header.has_key('EXTENSION'))
        assert_true(hdu.header.has_key('SIMPLE'))

        with ignore_warnings():
            hdu.writeto(self.temp('test.fits'), output_verify='ignore',
                        clobber=True)
        hdul2 = pyfits.open(self.temp('test.fits'))
        assert_true(len(hdul2), 2)
        assert_true(hdul2[1].header.has_key('MYKEY'))

    def test_update_comment(self):
        hdul = pyfits.open(self.data('arange.fits'))
        hdul[0].header.update('FOO', 'BAR', 'BAZ')
        assert_equal(hdul[0].header['FOO'], 'BAR')
        assert_equal(hdul[0].header.ascard['FOO'].comment, 'BAZ')

        hdul.writeto(self.temp('test.fits'))

        hdul = pyfits.open(self.temp('test.fits'), mode='update')
        hdul[0].header.ascard['FOO'].comment = 'QUX'
        hdul.close()

        hdul = pyfits.open(self.temp('test.fits'))
        assert_equal(hdul[0].header.ascard['FOO'].comment, 'QUX')

    def test_long_commentary_card(self):
        # Another version of this test using new API methods is found in
        # TestHeaderFunctions
        header = pyfits.Header()
        header.update('FOO', 'BAR')
        header.update('BAZ', 'QUX')
        longval = 'ABC' * 30
        header.add_history(longval)
        header.update('FRED', 'BARNEY')
        header.add_history(longval)

        assert_equal(len(header.ascard), 7)
        assert_equal(header.ascard[2].key, 'FRED')
        assert_equal(str(header.cards[3]), 'HISTORY ' + longval[:72])
        assert_equal(str(header.cards[4]).rstrip(), 'HISTORY ' + longval[72:])

        header.add_history(longval, after='FOO')
        assert_equal(len(header.ascard), 9)
        assert_equal(str(header.cards[1]), 'HISTORY ' + longval[:72])
        assert_equal(str(header.cards[2]).rstrip(), 'HISTORY ' + longval[72:])

    def test_wildcard_slice(self):
        """Test selecting a subsection of a header via wildcard matching."""

        header = pyfits.Header()
        header.update('ABC', 0)
        header.update('DEF', 1)
        header.update('ABD', 2)
        cards = header.ascard['AB*']
        assert_equal(len(cards), 2)
        assert_equal(cards[0].value, 0)
        assert_equal(cards[1].value, 2)

    def test_assign_boolean(self):
        """
        Regression test for #123. Tests assigning Python and Numpy boolean
        values to keyword values.
        """

        fooimg = _pad('FOO     =                    T')
        barimg = _pad('BAR     =                    F')
        h = pyfits.Header()
        h.update('FOO', True)
        h.update('BAR', False)
        assert_equal(h['FOO'], True)
        assert_equal(h['BAR'], False)
        assert_equal(h.ascard['FOO'].cardimage, fooimg)
        assert_equal(h.ascard['BAR'].cardimage, barimg)

        h = pyfits.Header()
        h.update('FOO', np.bool_(True))
        h.update('BAR', np.bool_(False))
        assert_equal(h['FOO'], True)
        assert_equal(h['BAR'], False)
        assert_equal(h.ascard['FOO'].cardimage, fooimg)
        assert_equal(h.ascard['BAR'].cardimage, barimg)

        h = pyfits.Header()
        h.ascard.append(pyfits.Card.fromstring(fooimg))
        h.ascard.append(pyfits.Card.fromstring(barimg))
        assert_equal(h['FOO'], True)
        assert_equal(h['BAR'], False)
        assert_equal(h.ascard['FOO'].cardimage, fooimg)
        assert_equal(h.ascard['BAR'].cardimage, barimg)

    def test_cardlist_list_methods(self):
        """Regression test for #190."""

        header = pyfits.Header()
        header.update('A', 'B', 'C')
        header.update('D', 'E', 'F')
        # The old header.update method won't let you append a duplicate keyword
        header.append(('D', 'G', 'H'))

        assert_equal(header.ascardlist().index(header.cards['A']), 0)
        assert_equal(header.ascardlist().index(header.cards['D']), 1)
        assert_equal(header.ascardlist().index(header.cards[('D', 1)]), 2)

        # Since the original CardList class really only works on card objects
        # the count method is mostly useless since cards didn't used to compare
        # equal sensibly
        assert_equal(header.ascardlist().count(header.cards['A']), 1)
        assert_equal(header.ascardlist().count(header.cards['D']), 1)
        assert_equal(header.ascardlist().count(header.cards[('D', 1)]), 1)
        assert_equal(header.ascardlist().count(pyfits.Card('A', 'B', 'C')), 0)


class TestHeaderFunctions(PyfitsTestCase):
    """Test PyFITS Header and Card objects."""

    def test_card_constructor_default_args(self):
        """Test Card constructor with default argument values."""

        c = pyfits.Card()
        assert_equal('', c.key)

    def test_string_value_card(self):
        """Test Card constructor with string value"""

        c = pyfits.Card('abc', '<8 ch')
        assert_equal(str(c),
                     "ABC     = '<8 ch   '                                                            ")
        c = pyfits.Card('nullstr', '')
        assert_equal(str(c),
                     "NULLSTR = ''                                                                    ")

    def test_boolean_value_card(self):
        """Test Card constructor with boolean value"""

        c = pyfits.Card("abc", True)
        assert_equal(str(c),
                     "ABC     =                    T                                                  ")

        c = pyfits.Card.fromstring('abc     = F')
        assert_equal(c.value, False)

    def test_long_integer_value_card(self):
        """Test Card constructor with long integer value"""

        c = pyfits.Card('long_int', -467374636747637647347374734737437)
        assert_equal(str(c),
                     "LONG_INT= -467374636747637647347374734737437                                    ")

    def test_floating_point_value_card(self):
        """Test Card constructor with floating point value"""

        c = pyfits.Card('floatnum', -467374636747637647347374734737437.)

        if (str(c) != "FLOATNUM= -4.6737463674763E+32                                                  " and
            str(c) != "FLOATNUM= -4.6737463674763E+032                                                 "):
            assert_equal(str(c),
                         "FLOATNUM= -4.6737463674763E+32                                                  ")

    def test_complex_value_card(self):
        """Test Card constructor with complex value"""

        c = pyfits.Card('abc',
                        1.2345377437887837487e88+6324767364763746367e-33j)

        if (str(c) != "ABC     = (1.23453774378878E+88, 6.32476736476374E-15)                          " and
            str(c) != "ABC     = (1.2345377437887E+088, 6.3247673647637E-015)                          "):
            assert_equal(str(c),
                         "ABC     = (1.23453774378878E+88, 6.32476736476374E-15)                          ")

    def test_card_image_constructed_too_long(self):
        """Test that over-long cards truncate the comment"""

        # card image constructed from key/value/comment is too long
        # (non-string value)
        with ignore_warnings():
            c = pyfits.Card('abc', 9, 'abcde'*20)
            assert_equal(str(c),
                         "ABC     =                    9 / abcdeabcdeabcdeabcdeabcdeabcdeabcdeabcdeabcdeab")
            c = pyfits.Card('abc', 'a'*68, 'abcdefg')
            assert_equal(str(c),
                         "ABC     = 'aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa'")

    def test_constructor_filter_illegal_data_structures(self):
        """Test that Card constructor raises exceptions on bad arguments"""

        assert_raises(ValueError, pyfits.Card, ('abc',), {'value': (2, 3)})
        assert_raises(ValueError, pyfits.Card, 'key', [], 'comment')

    def test_keyword_too_long(self):
        """Test that long Card keywords are allowed, but with a warning"""

        with catch_warnings():
            warnings.simplefilter('error')
            assert_raises(UserWarning, pyfits.Card, 'abcdefghi', 'long')

    def test_illegal_characters_in_key(self):
        """
        Test that Card constructor allows illegal characters in the keyword,
        but creates a HIERARCH card.
        """

        # This test used to check that a ValueError was raised, because a
        # keyword like 'abc+' was simply not allowed.  Now it should create a
        # HIERARCH card.

        with catch_warnings(record=True) as w:
            c = pyfits.Card('abc+', 9)
            assert_equal(len(w), 1)
            assert_equal(c.image, _pad('HIERARCH abc+ =                    9'))

    def test_commentary_cards(self):
        # commentary cards
        c = pyfits.Card("history",
                        "A commentary card's value has no quotes around it.")
        assert_equal(str(c),
                     "HISTORY A commentary card's value has no quotes around it.                      ")
        c = pyfits.Card("comment",
                        "A commentary card has no comment.", "comment")
        assert_equal(str(c),
                     "COMMENT A commentary card has no comment.                                       ")

    def test_commentary_card_created_by_fromstring(self):
        # commentary card created by fromstring()
        c = pyfits.Card.fromstring("COMMENT card has no comments. / text after slash is still part of the value.")
        assert_equal(c.value,
                     'card has no comments. / text after slash is still part of the value.')
        assert_equal(c.comment, '')

    def test_commentary_card_will_not_parse_numerical_value(self):
        # commentary card will not parse the numerical value
        c = pyfits.Card.fromstring("history  (1, 2)")
        assert_equal(str(c),
                     "HISTORY  (1, 2)                                                                 ")

    def test_equal_sign_after_column8(self):
        # equal sign after column 8 of a commentary card will be part ofthe
        # string value
        c = pyfits.Card.fromstring("history =   (1, 2)")
        assert_equal(str(c),
                     "HISTORY =   (1, 2)                                                              ")

    def test_blank_keyword(self):
        c = pyfits.Card('', '       / EXPOSURE INFORMATION')
        assert_equal(str(c),
                     '               / EXPOSURE INFORMATION                                           ')
        c = pyfits.Card.fromstring(str(c))
        assert_equal(c.keyword, '')
        assert_equal(c.value, '       / EXPOSURE INFORMATION')


    def test_specify_undefined_value(self):
        # this is how to specify an undefined value
        c = pyfits.Card("undef", pyfits.card.UNDEFINED)
        assert_equal(str(c),
                     "UNDEF   =                                                                       ")

    def test_complex_number_using_string_input(self):
        # complex number using string input
        c = pyfits.Card.fromstring('abc     = (8, 9)')
        assert_equal(str(c),
                     "ABC     =               (8, 9)                                                  ")

    def test_fixable_non_standard_fits_card(self):
        # fixable non-standard FITS card will keep the original format
        c = pyfits.Card.fromstring('abc     = +  2.1   e + 12')
        assert_equal(c.value, 2100000000000.0)
        with CaptureStdio():
            assert_equal(str(c),
                         "ABC     =             +2.1E+12                                                  ")

    def test_fixable_non_fsc(self):
        # fixable non-FSC: if the card is not parsable, it's value will be
        # assumed
        # to be a string and everything after the first slash will be comment
        c = pyfits.Card.fromstring("no_quote=  this card's value has no quotes / let's also try the comment")
        with CaptureStdio():
            assert_equal(str(c),
                         "NO_QUOTE= 'this card''s value has no quotes' / let's also try the comment       ")

    def test_undefined_value_using_string_input(self):
        # undefined value using string input
        c = pyfits.Card.fromstring('abc     =    ')
        assert_equal(str(c),
                     "ABC     =                                                                       ")

    def test_mislocated_equal_sign(self):
        # test mislocated "=" sign
        c = pyfits.Card.fromstring('xyz= 100')
        assert_equal(c.keyword, 'XYZ')
        assert_equal(c.value, 100)
        with CaptureStdio():
            assert_equal(str(c),
                         "XYZ     =                  100                                                  ")

    def test_equal_only_up_to_column_10(self):
        # the test of "=" location is only up to column 10

        # This test used to check if PyFITS rewrote this card to a new format,
        # something like "HISTO   = '=   (1, 2)".  But since ticket #109 if the
        # format is completely wrong we don't make any assumptions and the card
        # should be left alone
        with CaptureStdio():
            c = pyfits.Card.fromstring("histo       =   (1, 2)")
            assert_equal(str(c), _pad("histo       =   (1, 2)"))

            # Likewise this card should just be left in its original form and
            # we shouldn't guess how to parse it or rewrite it.
            c = pyfits.Card.fromstring("   history          (1, 2)")
            assert_equal(str(c), _pad("   history          (1, 2)"))

    def test_verify_invalid_equal_sign(self):
        # verification
        c = pyfits.Card.fromstring('abc= a6')
        with catch_warnings(record=True) as w:
            with CaptureStdio():
                c.verify()
            err_text1 = ("Card 'ABC' is not FITS standard (equal sign not at "
                         "column 8)")
            err_text2 = ("Card 'ABC' is not FITS standard (invalid value "
                         "string: a6")
            assert_equal(len(w), 4)
            assert_true(err_text1 in str(w[1].message))
            assert_true(err_text2 in str(w[2].message))

    def test_fix_invalid_equal_sign(self):
        c = pyfits.Card.fromstring('abc= a6')
        with catch_warnings(record=True) as w:
            with CaptureStdio():
                c.verify('fix')
            fix_text = "Fixed 'ABC' card to meet the FITS standard."
            assert_equal(len(w), 4)
            assert_true(fix_text in str(w[1].message))
        assert_equal(str(c),
                     "ABC     = 'a6      '                                                            ")

    def test_long_string_value(self):
        # test long string value
        c = pyfits.Card('abc', 'long string value '*10, 'long comment '*10)
        assert_equal(str(c),
            "ABC     = 'long string value long string value long string value long string &' "
            "CONTINUE  'value long string value long string value long string value long &'  "
            "CONTINUE  'string value long string value long string value &'                  "
            "CONTINUE  '&' / long comment long comment long comment long comment long        "
            "CONTINUE  '&' / comment long comment long comment long comment long comment     "
            "CONTINUE  '&' / long comment                                                    ")

    def test_long_unicode_string(self):
        """Regression test for
        https://github.com/spacetelescope/PyFITS/issues/1

        So long as a unicode string can be converted to ASCII it should have no
        different behavior in this regard from a byte string.
        """

        h1 = pyfits.Header()
        h1['TEST'] = 'abcdefg' * 30

        h2 = pyfits.Header()
        with catch_warnings(record=True) as w:
            h2['TEST'] = u'abcdefg' * 30
            assert_equal(len(w), 0)

        assert_equal(str(h1), str(h2))

    def test_long_string_repr(self):
        """Regression test for #193

        Ensure that the __repr__() for cards represented with CONTINUE cards is
        split across multiple lines (broken at each *physical* card).
        """

        header = pyfits.Header()
        header['TEST1'] = ('Regular value', 'Regular comment')
        header['TEST2'] = ('long string value ' * 10, 'long comment ' * 10)
        header['TEST3'] = ('Regular value', 'Regular comment')

        assert_equal(repr(header).splitlines(),
            [str(pyfits.Card('TEST1', 'Regular value', 'Regular comment')),
             "TEST2   = 'long string value long string value long string value long string &' ",
             "CONTINUE  'value long string value long string value long string value long &'  ",
             "CONTINUE  'string value long string value long string value &'                  ",
             "CONTINUE  '&' / long comment long comment long comment long comment long        ",
             "CONTINUE  '&' / comment long comment long comment long comment long comment     ",
             "CONTINUE  '&' / long comment                                                    ",
             str(pyfits.Card('TEST3', 'Regular value', 'Regular comment'))])

    def test_blank_keyword_long_value(self):
        """Regression test for #194

        Test that a blank keyword ('') can be assigned a too-long value that is
        continued across multiple cards with blank keywords, just like COMMENT
        and HISTORY cards.
        """

        value = 'long string value ' * 10
        header = pyfits.Header()
        header[''] = value

        assert_equal(len(header), 3)
        assert_equal(' '.join(header['']), value.rstrip())

        # Ensure that this works like other commentary keywords
        header['COMMENT'] = value
        header['HISTORY'] = value
        assert_equal(header['COMMENT'], header['HISTORY'])
        assert_equal(header['COMMENT'], header[''])

    def test_long_string_from_file(self):
        c = pyfits.Card('abc', 'long string value '*10, 'long comment '*10)
        hdu = pyfits.PrimaryHDU()
        hdu.header.append(c)
        hdu.writeto(self.temp('test_new.fits'))

        hdul = pyfits.open(self.temp('test_new.fits'))
        c = hdul[0].header.cards['abc']
        hdul.close()
        assert_equal(str(c),
            "ABC     = 'long string value long string value long string value long string &' "
            "CONTINUE  'value long string value long string value long string value long &'  "
            "CONTINUE  'string value long string value long string value &'                  "
            "CONTINUE  '&' / long comment long comment long comment long comment long        "
            "CONTINUE  '&' / comment long comment long comment long comment long comment     "
            "CONTINUE  '&' / long comment                                                    ")


    def test_word_in_long_string_too_long(self):
        # if a word in a long string is too long, it will be cut in the middle
        c = pyfits.Card('abc', 'longstringvalue'*10, 'longcomment'*10)
        assert_equal(str(c),
            "ABC     = 'longstringvaluelongstringvaluelongstringvaluelongstringvaluelongstr&'"
            "CONTINUE  'ingvaluelongstringvaluelongstringvaluelongstringvaluelongstringvalu&'"
            "CONTINUE  'elongstringvalue&'                                                   "
            "CONTINUE  '&' / longcommentlongcommentlongcommentlongcommentlongcommentlongcomme"
            "CONTINUE  '&' / ntlongcommentlongcommentlongcommentlongcomment                  ")

    def test_long_string_value_via_fromstring(self):
        # long string value via fromstring() method
        c = pyfits.Card.fromstring(
            _pad("abc     = 'longstring''s testing  &  ' / comments in line 1") +
            _pad("continue  'continue with long string but without the ampersand at the end' /") +
            _pad("continue  'continue must have string value (with quotes)' / comments with ''. "))
        assert_equal(str(c),
            "ABC     = 'longstring''s testing  continue with long string but without the &'  "
            "CONTINUE  'ampersand at the endcontinue must have string value (with quotes)&'  "
            "CONTINUE  '&' / comments in line 1 comments with ''.                            ")

    def test_continue_card_with_equals_in_value(self):
        """
        Regression test for #117.
        """

        c = pyfits.Card.fromstring(
            _pad("EXPR    = '/grp/hst/cdbs//grid/pickles/dat_uvk/pickles_uk_10.fits * &'") +
            _pad("CONTINUE  '5.87359e-12 * MWAvg(Av=0.12)&'") +
            _pad("CONTINUE  '&' / pysyn expression"))

        assert_equal(c.keyword, 'EXPR')
        assert_equal(c.value,
                     '/grp/hst/cdbs//grid/pickles/dat_uvk/pickles_uk_10.fits '
                     '* 5.87359e-12 * MWAvg(Av=0.12)')
        assert_equal(c.comment, 'pysyn expression')

    def test_hierarch_card_creation(self):
        # Test automatic upgrade to hierarch card
        with catch_warnings(record=True) as w:
            c = pyfits.Card('ESO INS SLIT2 Y1FRML',
                            'ENC=OFFSET+RESOL*acos((WID-(MAX+MIN))/(MAX-MIN)')
            assert_equal(len(w), 1)
            assert_true('HIERARCH card will be created' in str(w[0].message))
            assert_equal(str(c),
                         "HIERARCH ESO INS SLIT2 Y1FRML= "
                         "'ENC=OFFSET+RESOL*acos((WID-(MAX+MIN))/(MAX-MIN)'")

        # Test manual creation of hierarch card
        c = pyfits.Card('hierarch abcdefghi', 10)
        assert_equal(str(c),
            "HIERARCH abcdefghi = "
            "10                                                         ")
        c = pyfits.Card('HIERARCH ESO INS SLIT2 Y1FRML',
                        'ENC=OFFSET+RESOL*acos((WID-(MAX+MIN))/(MAX-MIN)')
        assert_equal(str(c),
                     "HIERARCH ESO INS SLIT2 Y1FRML= "
                     "'ENC=OFFSET+RESOL*acos((WID-(MAX+MIN))/(MAX-MIN)'")

    def test_hierarch_with_abbrev_value_indicator(self):
        """Regression test for
        https://github.com/spacetelescope/PyFITS/issues/5
        """

        c = pyfits.Card.fromstring("HIERARCH key.META_4='calFileVersion'")
        assert_equal(c.keyword, 'key.META_4')
        assert_equal(c.value, 'calFileVersion')
        assert_equal(c.comment, '')

    def test_hierarch_keyword_whitespace(self):
        """
        Regression test for
        https://github.com/spacetelescope/PyFITS/issues/6

        Make sure any leading or trailing whitespace around HIERARCH
        keywords is stripped from the actual keyword value.
        """

        c = pyfits.Card.fromstring(
                "HIERARCH  key.META_4    = 'calFileVersion'")
        assert_equal(c.keyword, 'key.META_4')
        assert_equal(c.value, 'calFileVersion')
        assert_equal(c.comment, '')

        # Test also with creation via the Card constructor
        c = pyfits.Card('HIERARCH  key.META_4', 'calFileVersion')
        assert_equal(c.keyword, 'key.META_4')
        assert_equal(c.value, 'calFileVersion')
        assert_equal(c.comment, '')

    def test_verify_mixed_case_hierarch(self):
        """Regression test for
        https://github.com/spacetelescope/PyFITS/issues/7

        Assures that HIERARCH keywords with lower-case characters and other
        normally invalid keyword characters are not considered invalid.
        """

        c = pyfits.Card('HIERARCH WeirdCard.~!@#_^$%&', 'The value',
                        'a comment')
        # This should not raise any exceptions
        c.verify('exception')
        assert_equal(c.keyword, 'WeirdCard.~!@#_^$%&')
        assert_equal(c.value, 'The value')
        assert_equal(c.comment, 'a comment')

        # Test also the specific case from the original bug report
        header = pyfits.Header([
            ('simple', True),
            ('BITPIX', 8),
            ('NAXIS', 0),
            ('EXTEND', True, 'May contain datasets'),
            ('HIERARCH key.META_0', 'detRow')
        ])
        hdu = pyfits.PrimaryHDU(header=header)
        hdu.writeto(self.temp('test.fits'))
        with pyfits.open(self.temp('test.fits')) as hdul:
            header2 = hdul[0].header
            assert_equal(str(header.cards[header.index('key.META_0')]),
                         str(header2.cards[header2.index('key.META_0')]))

    def test_missing_keyword(self):
        """Test that accessing a non-existent keyword raises a KeyError."""

        header = pyfits.Header()
        assert_raises(KeyError, lambda k: header[k], 'NAXIS')
        # Test the exception message
        try:
            header['NAXIS']
        except KeyError, e:
            assert_equal(e.args[0], "Keyword 'NAXIS' not found.")

    def test_hierarch_card_lookup(self):
        header = pyfits.Header()
        header['hierarch abcdefghi'] = 10
        assert_true('abcdefghi' in header)
        assert_equal(header['abcdefghi'], 10)
        # This used to be assert_false, but per ticket #155 hierarch keywords
        # should be treated case-insensitively when performing lookups
        assert_true('ABCDEFGHI' in header)

    def test_hierarch_create_and_update(self):
        """
        Regression test for #158.  Tests several additional use cases for
        working with HIERARCH cards.
        """

        msg = 'a HIERARCH card will be created'

        header = pyfits.Header()
        with catch_warnings(record=True) as w:
            header.update('HIERARCH BLAH BLAH', 'TESTA')
            assert_equal(len(w), 0)
            assert_true('BLAH BLAH' in header)
            assert_equal(header['BLAH BLAH'], 'TESTA')

            header.update('HIERARCH BLAH BLAH', 'TESTB')
            assert_equal(len(w), 0)
            assert_true(header['BLAH BLAH'], 'TESTB')

            # Update without explicitly stating 'HIERARCH':
            header.update('BLAH BLAH', 'TESTC')
            assert_equal(len(w), 0)
            assert_equal(len(header), 1)
            assert_true(header['BLAH BLAH'], 'TESTC')

            # Test case-insensitivity
            header.update('HIERARCH blah blah', 'TESTD')
            assert_equal(len(w), 0)
            assert_equal(len(header), 1)
            assert_true(header['blah blah'], 'TESTD')

            header.update('blah blah', 'TESTE')
            assert_equal(len(w), 0)
            assert_equal(len(header), 1)
            assert_true(header['blah blah'], 'TESTE')

            # Create a HIERARCH card > 8 characters without explicitly stating
            # 'HIERARCH'
            header.update('BLAH BLAH BLAH', 'TESTA')
            assert_equal(len(w), 1)
            assert_true(msg in str(w[0].message))

            header.update('HIERARCH BLAH BLAH BLAH', 'TESTB')
            assert_equal(len(w), 1)
            assert_true(header['BLAH BLAH BLAH'], 'TESTB')

            # Update without explicitly stating 'HIERARCH':
            header.update('BLAH BLAH BLAH', 'TESTC')
            assert_equal(len(w), 1)
            assert_true(header['BLAH BLAH BLAH'], 'TESTC')

            # Test case-insensitivity
            header.update('HIERARCH blah blah blah', 'TESTD')
            assert_equal(len(w), 1)
            assert_true(header['blah blah blah'], 'TESTD')

            header.update('blah blah blah', 'TESTE')
            assert_equal(len(w), 1)
            assert_true(header['blah blah blah'], 'TESTE')

    def test_short_hierarch_create_and_update(self):
        """
        Regression test for #158.  Tests several additional use cases for
        working with HIERARCH cards, specifically where the keyword is fewer
        than 8 characters, but contains invalid characters such that it can
        only be created as a HIERARCH card.
        """

        msg = 'a HIERARCH card will be created'

        header = pyfits.Header()
        with catch_warnings(record=True) as w:
            header.update('HIERARCH BLA BLA', 'TESTA')
            assert_equal(len(w), 0)
            assert_true('BLA BLA' in header)
            assert_equal(header['BLA BLA'], 'TESTA')

            header.update('HIERARCH BLA BLA', 'TESTB')
            assert_equal(len(w), 0)
            assert_true(header['BLA BLA'], 'TESTB')

            # Update without explicitly stating 'HIERARCH':
            header.update('BLA BLA', 'TESTC')
            assert_equal(len(w), 0)
            assert_true(header['BLA BLA'], 'TESTC')

            # Test case-insensitivity
            header.update('HIERARCH bla bla', 'TESTD')
            assert_equal(len(w), 0)
            assert_equal(len(header), 1)
            assert_true(header['bla bla'], 'TESTD')

            header.update('bla bla', 'TESTE')
            assert_equal(len(w), 0)
            assert_equal(len(header), 1)
            assert_true(header['bla bla'], 'TESTE')

        header = pyfits.Header()
        with catch_warnings(record=True) as w:
            # Create a HIERARCH card containing invalid characters without
            # explicitly stating 'HIERARCH'
            header.update('BLA BLA', 'TESTA')
            assert_equal(len(w), 1)
            assert_true(msg in str(w[0].message))

            header.update('HIERARCH BLA BLA', 'TESTB')
            assert_equal(len(w), 1)
            assert_true(header['BLA BLA'], 'TESTB')

            # Update without explicitly stating 'HIERARCH':
            header.update('BLA BLA', 'TESTC')
            assert_equal(len(w), 1)
            assert_true(header['BLA BLA'], 'TESTC')

            # Test case-insensitivity
            header.update('HIERARCH bla bla', 'TESTD')
            assert_equal(len(w), 1)
            assert_equal(len(header), 1)
            assert_true(header['bla bla'], 'TESTD')

            header.update('bla bla', 'TESTE')
            assert_equal(len(w), 1)
            assert_equal(len(header), 1)
            assert_true(header['bla bla'], 'TESTE')

    def test_header_setitem_invalid(self):
        header = pyfits.Header()
        def test():
            header['FOO'] = ('bar', 'baz', 'qux')
        assert_raises(ValueError, test)

    def test_header_setitem_1tuple(self):
        header = pyfits.Header()
        header['FOO'] = ('BAR',)
        assert_equal(header['FOO'], 'BAR')
        assert_equal(header[0], 'BAR')
        assert_equal(header.comments[0], '')
        assert_equal(header.comments['FOO'], '')

    def test_header_setitem_2tuple(self):
        header = pyfits.Header()
        header['FOO'] = ('BAR', 'BAZ')
        assert_equal(header['FOO'], 'BAR')
        assert_equal(header[0], 'BAR')
        assert_equal(header.comments[0], 'BAZ')
        assert_equal(header.comments['FOO'], 'BAZ')

    def test_header_set_value_to_none(self):
        """
        Setting the value of a card to None should simply give that card a
        blank value.
        """

        header = pyfits.Header()
        header['FOO'] = 'BAR'
        assert_equal(header['FOO'], 'BAR')
        header['FOO'] = None
        assert_equal(header['FOO'], '')

    def test_set_comment_only(self):
        header = pyfits.Header([('A', 'B', 'C')])
        header.set('A', comment='D')
        assert_equal(header['A'], 'B')
        assert_equal(header.comments['A'], 'D')

    def test_header_iter(self):
        header = pyfits.Header([('A', 'B'), ('C', 'D')])
        assert_equal(list(header), ['A', 'C'])

    def test_header_slice(self):
        header = pyfits.Header([('A', 'B'), ('C', 'D'), ('E', 'F')])
        newheader = header[1:]
        assert_equal(len(newheader), 2)
        assert_true('A' not in newheader)
        assert_true('C' in newheader)
        assert_true('E' in newheader)

        newheader = header[::-1]
        assert_equal(len(newheader), 3)
        assert_equal(newheader[0], 'F')
        assert_equal(newheader[1], 'D')
        assert_equal(newheader[2], 'B')

        newheader = header[::2]
        assert_equal(len(newheader), 2)
        assert_true('A' in newheader)
        assert_true('C' not in newheader)
        assert_true('E' in newheader)

    def test_header_slice_assignment(self):
        """
        Assigning to a slice should just assign new values to the cards
        included in the slice.
        """

        header = pyfits.Header([('A', 'B'), ('C', 'D'), ('E', 'F')])

        # Test assigning slice to the same value; this works similarly to numpy
        # arrays
        header[1:] = 1
        assert_equal(header[1], 1)
        assert_equal(header[2], 1)

        # Though strings are iterable they should be treated as a scalar value
        header[1:] = 'GH'
        assert_equal(header[1], 'GH')
        assert_equal(header[2], 'GH')

        # Now assign via an iterable
        header[1:] = ['H', 'I']
        assert_equal(header[1], 'H')
        assert_equal(header[2], 'I')

    def test_header_slice_delete(self):
        """Test deleting a slice of cards from the header."""

        header = pyfits.Header([('A', 'B'), ('C', 'D'), ('E', 'F')])
        del header[1:]
        assert_equal(len(header), 1)
        assert_equal(header[0], 'B')
        del header[:]
        assert_equal(len(header), 0)

    def test_wildcard_slice(self):
        """Test selecting a subsection of a header via wildcard matching."""

        header = pyfits.Header([('ABC', 0), ('DEF', 1), ('ABD', 2)])
        newheader = header['AB*']
        assert_equal(len(newheader), 2)
        assert_equal(newheader[0], 0)
        assert_equal(newheader[1], 2)

    def test_wildcard_slice_assignment(self):
        """Test assigning to a header slice selected via wildcard matching."""

        header = pyfits.Header([('ABC', 0), ('DEF', 1), ('ABD', 2)])

        # Test assigning slice to the same value; this works similarly to numpy
        # arrays
        header['AB*'] = 1
        assert_equal(header[0], 1)
        assert_equal(header[2], 1)

        # Though strings are iterable they should be treated as a scalar value
        header['AB*'] = 'GH'
        assert_equal(header[0], 'GH')
        assert_equal(header[2], 'GH')

        # Now assign via an iterable
        header['AB*'] = ['H', 'I']
        assert_equal(header[0], 'H')
        assert_equal(header[2], 'I')

    def test_wildcard_slice_deletion(self):
        """Test deleting cards from a header that match a wildcard pattern."""

        header = pyfits.Header([('ABC', 0), ('DEF', 1), ('ABD', 2)])
        del header['AB*']
        assert_equal(len(header), 1)
        assert_equal(header[0], 1)

    def test_header_history(self):
        header = pyfits.Header([('ABC', 0), ('HISTORY', 1), ('HISTORY', 2),
                                ('DEF', 3), ('HISTORY', 4), ('HISTORY', 5)])
        assert_equal(header['HISTORY'], [1, 2, 4, 5])

    def test_header_clear(self):
        header = pyfits.Header([('A', 'B'), ('C', 'D')])
        header.clear()
        assert_true('A' not in header)
        assert_true('C' not in header)
        assert_equal(len(header), 0)

    def test_header_fromkeys(self):
        header = pyfits.Header.fromkeys(['A', 'B'])
        assert_true('A' in header)
        assert_equal(header['A'], '')
        assert_equal(header.comments['A'], '')
        assert_true('B' in header)
        assert_equal(header['B'], '')
        assert_equal(header.comments['B'], '')

    def test_header_fromkeys_with_value(self):
        header = pyfits.Header.fromkeys(['A', 'B'], 'C')
        assert_true('A' in header)
        assert_equal(header['A'], 'C')
        assert_equal(header.comments['A'], '')
        assert_true('B' in header)
        assert_equal(header['B'], 'C')
        assert_equal(header.comments['B'], '')

    def test_header_fromkeys_with_value_and_comment(self):
        header = pyfits.Header.fromkeys(['A'], ('B', 'C'))
        assert_true('A' in header)
        assert_equal(header['A'], 'B')
        assert_equal(header.comments['A'], 'C')

    def test_header_fromkeys_with_duplicates(self):
        header = pyfits.Header.fromkeys(['A', 'B', 'A'], 'C')
        assert_true('A' in header)
        assert_true(('A', 0) in header)
        assert_true(('A', 1) in header)
        assert_true(('A', 2) not in header)
        assert_equal(header[0], 'C')
        assert_equal(header['A'], 'C')
        assert_equal(header[('A', 0)], 'C')
        assert_equal(header[2], 'C')
        assert_equal(header[('A', 1)], 'C')

    def test_header_items(self):
        header = pyfits.Header([('A', 'B'), ('C', 'D')])
        assert_equal(header.items(), list(header.iteritems()))

    def test_header_iterkeys(self):
        header = pyfits.Header([('A', 'B'), ('C', 'D')])
        for a, b in itertools.izip(header.iterkeys(), header):
            assert_equal(a, b)

    def test_header_itervalues(self):
        header = pyfits.Header([('A', 'B'), ('C', 'D')])
        for a, b in itertools.izip(header.itervalues(), ['B', 'D']):
            assert_equal(a, b)

    def test_header_keys(self):
        hdul = pyfits.open(self.data('arange.fits'))
        assert_equal(hdul[0].header.keys(),
                     ['SIMPLE', 'BITPIX', 'NAXIS', 'NAXIS1', 'NAXIS2',
                      'NAXIS3', 'EXTEND'])

    def test_header_list_like_pop(self):
        header = pyfits.Header([('A', 'B'), ('C', 'D'), ('E', 'F'),
                                ('G', 'H')])

        last = header.pop()
        assert_equal(last, 'H')
        assert_equal(len(header), 3)
        assert_equal(header.keys(), ['A', 'C', 'E'])

        mid = header.pop(1)
        assert_equal(mid, 'D')
        assert_equal(len(header), 2)
        assert_equal(header.keys(), ['A', 'E'])

        first = header.pop(0)
        assert_equal(first, 'B')
        assert_equal(len(header), 1)
        assert_equal(header.keys(), ['E'])

        assert_raises(IndexError, header.pop, 42)

    def test_header_dict_like_pop(self):
        header = pyfits.Header([('A', 'B'), ('C', 'D'), ('E', 'F'),
                                ('G', 'H')])
        assert_raises(TypeError, header.pop, 'A', 'B', 'C')

        last = header.pop('G')
        assert_equal(last, 'H')
        assert_equal(len(header), 3)
        assert_equal(header.keys(), ['A', 'C', 'E'])

        mid = header.pop('C')
        assert_equal(mid, 'D')
        assert_equal(len(header), 2)
        assert_equal(header.keys(), ['A', 'E'])

        first = header.pop('A')
        assert_equal(first, 'B')
        assert_equal(len(header), 1)
        assert_equal(header.keys(), ['E'])

        default = header.pop('X', 'Y')
        assert_equal(default, 'Y')
        assert_equal(len(header), 1)

        assert_raises(KeyError, header.pop, 'X')

    def test_popitem(self):
        header = pyfits.Header([('A', 'B'), ('C', 'D'), ('E', 'F')])
        keyword, value = header.popitem()
        assert_true(keyword not in header)
        assert_equal(len(header), 2)
        keyword, value = header.popitem()
        assert_true(keyword not in header)
        assert_equal(len(header), 1)
        keyword, value = header.popitem()
        assert_true(keyword not in header)
        assert_equal(len(header), 0)
        assert_raises(KeyError, header.popitem)

    def test_setdefault(self):
        header = pyfits.Header([('A', 'B'), ('C', 'D'), ('E', 'F')])
        assert_equal(header.setdefault('A'), 'B')
        assert_equal(header.setdefault('C'), 'D')
        assert_equal(header.setdefault('E'), 'F')
        assert_equal(len(header), 3)
        assert_equal(header.setdefault('G', 'H'), 'H')
        assert_equal(len(header), 4)
        assert_true('G' in header)
        assert_equal(header.setdefault('G', 'H'), 'H')
        assert_equal(len(header), 4)

    def test_update_from_dict(self):
        """
        Test adding new cards and updating existing cards from a dict using
        Header.update()
        """

        header = pyfits.Header([('A', 'B'), ('C', 'D')])
        header.update({'A': 'E', 'F': 'G'})
        assert_equal(header['A'], 'E')
        assert_equal(header[0], 'E')
        assert_true('F' in header)
        assert_equal(header['F'], 'G')
        assert_equal(header[-1], 'G')

        # Same as above but this time pass the update dict as keyword arguments
        header = pyfits.Header([('A', 'B'), ('C', 'D')])
        header.update(A='E', F='G')
        assert_equal(header['A'], 'E')
        assert_equal(header[0], 'E')
        assert_true('F' in header)
        assert_equal(header['F'], 'G')
        assert_equal(header[-1], 'G')

    def test_update_from_iterable(self):
        """
        Test adding new cards and updating existing cards from an iterable of
        cards and card tuples.
        """

        header = pyfits.Header([('A', 'B'), ('C', 'D')])
        header.update([('A', 'E'), pyfits.Card('F', 'G')])
        assert_equal(header['A'], 'E')
        assert_equal(header[0], 'E')
        assert_true('F' in header)
        assert_equal(header['F'], 'G')
        assert_equal(header[-1], 'G')

    def test_header_extend(self):
        """
        Test extending a header both with and without stripping cards from the
        extension header.
        """

        hdu = pyfits.PrimaryHDU()
        hdu2 = pyfits.ImageHDU()
        hdu2.header['MYKEY'] = ('some val', 'some comment')
        hdu.header += hdu2.header
        assert_equal(len(hdu.header), 5)
        assert_equal(hdu.header[-1], 'some val')

        # Same thing, but using + instead of +=
        hdu = pyfits.PrimaryHDU()
        hdu.header = hdu.header + hdu2.header
        assert_equal(len(hdu.header), 5)
        assert_equal(hdu.header[-1], 'some val')

        # Directly append the other header in full--not usually a desirable
        # operation when the header is coming from another HDU
        hdu.header.extend(hdu2.header, strip=False)
        assert_equal(len(hdu.header), 11)
        assert_equal(hdu.header.keys()[5], 'XTENSION')
        assert_equal(hdu.header[-1], 'some val')
        assert_true(('MYKEY', 1) in hdu.header)

    def test_header_extend_unique(self):
        """
        Test extending the header with and without unique=True.
        """
        hdu = pyfits.PrimaryHDU()
        hdu2 = pyfits.ImageHDU()
        hdu.header['MYKEY'] = ('some val', 'some comment')
        hdu2.header['MYKEY'] = ('some other val', 'some other comment')
        hdu.header.extend(hdu2.header)
        assert_equal(len(hdu.header), 6)
        assert_equal(hdu.header[-2], 'some val')
        assert_equal(hdu.header[-1], 'some other val')

        hdu = pyfits.PrimaryHDU()
        hdu2 = pyfits.ImageHDU()
        hdu.header['MYKEY'] = ('some val', 'some comment')
        hdu.header.extend(hdu2.header, unique=True)
        assert_equal(len(hdu.header), 5)
        assert_equal(hdu.header[-1], 'some val')

    def test_header_extend_update(self):
        """
        Test extending the header with and without update=True.
        """

        hdu = pyfits.PrimaryHDU()
        hdu2 = pyfits.ImageHDU()
        hdu.header['MYKEY'] = ('some val', 'some comment')
        hdu.header['HISTORY'] = 'history 1'
        hdu2.header['MYKEY'] = ('some other val', 'some other comment')
        hdu2.header['HISTORY'] = 'history 1'
        hdu2.header['HISTORY'] = 'history 2'
        hdu.header.extend(hdu2.header)
        assert_equal(len(hdu.header), 9)
        assert_true(('MYKEY', 0) in hdu.header)
        assert_true(('MYKEY', 1) in hdu.header)
        assert_equal(hdu.header[('MYKEY', 1)], 'some other val')
        assert_equal(len(hdu.header['HISTORY']), 3)
        assert_equal(hdu.header[-1], 'history 2')

        hdu = pyfits.PrimaryHDU()
        hdu.header['MYKEY'] = ('some val', 'some comment')
        hdu.header['HISTORY'] = 'history 1'
        hdu.header.extend(hdu2.header, update=True)
        assert_equal(len(hdu.header), 7)
        assert_true(('MYKEY', 0) in hdu.header)
        assert_false(('MYKEY', 1) in hdu.header)
        assert_equal(hdu.header['MYKEY'], 'some other val')
        assert_equal(len(hdu.header['HISTORY']), 2)
        assert_equal(hdu.header[-1], 'history 2')

    def test_header_extend_exact(self):
        """
        Test that extending an empty header with the contents of an existing
        header can exactly duplicate that header, given strip=False and
        end=True.
        """

        header = pyfits.getheader(self.data('test0.fits'))
        header2 = pyfits.Header()
        header2.extend(header, strip=False, end=True)
        assert_equal(header, header2)

    def test_header_count(self):
        header = pyfits.Header([('A', 'B'), ('C', 'D'), ('E', 'F')])
        assert_equal(header.count('A'), 1)
        assert_equal(header.count('C'), 1)
        assert_equal(header.count('E'), 1)
        header['HISTORY'] = 'a'
        header['HISTORY'] = 'b'
        assert_equal(header.count('HISTORY'), 2)
        assert_raises(KeyError, header.count, 'G')

    def test_header_append_use_blanks(self):
        """
        Tests that blank cards can be appended, and that future appends will
        use blank cards when available (unless useblanks=False)
        """

        header = pyfits.Header([('A', 'B'), ('C', 'D')])

        # Append a couple blanks
        header.append()
        header.append()
        assert_equal(len(header), 4)
        assert_equal(header[-1], '')
        assert_equal(header[-2], '')

        # New card should fill the first blank by default
        header.append(('E', 'F'))
        assert_equal(len(header), 4)
        assert_equal(header[-2], 'F')
        assert_equal(header[-1], '')

        # This card should not use up a blank spot
        header.append(('G', 'H'), useblanks=False)
        assert_equal(len(header), 5)
        assert_equal(header[-1], '')
        assert_equal(header[-2], 'H')

    def test_header_append_keyword_only(self):
        """
        Test appending a new card with just the keyword, and no value or
        comment given.
        """

        header = pyfits.Header([('A', 'B'), ('C', 'D')])
        header.append('E')
        assert_equal(len(header), 3)
        assert_equal(header.keys()[-1], 'E')
        assert_equal(header[-1], '')
        assert_equal(header.comments['E'], '')

        # Try appending a blank--normally this can be accomplished with just
        # header.append(), but header.append('') should also work (and is maybe
        # a little more clear)
        header.append('')
        assert_equal(len(header), 4)

        assert_equal(header.keys()[-1], '')
        assert_equal(header[''], '')
        assert_equal(header.comments[''], '')

    def test_header_insert_use_blanks(self):
        header = pyfits.Header([('A', 'B'), ('C', 'D')])

        # Append a couple blanks
        header.append()
        header.append()

        # Insert a new card; should use up one of the blanks
        header.insert(1, ('E', 'F'))
        assert_equal(len(header), 4)
        assert_equal(header[1], 'F')
        assert_equal(header[-1], '')
        assert_equal(header[-2], 'D')

        # Insert a new card without using blanks
        header.insert(1, ('G', 'H'), useblanks=False)
        assert_equal(len(header), 5)
        assert_equal(header[1], 'H')
        assert_equal(header[-1], '')

    def test_header_insert_before_keyword(self):
        """
        Test that a keyword name or tuple can be used to insert new keywords.

        Also tests the ``after`` keyword argument.

        Regression test for https://github.com/spacetelescope/PyFITS/issues/12
        """

        header = pyfits.Header([
            ('NAXIS1', 10), ('COMMENT', 'Comment 1'),
            ('COMMENT', 'Comment 3')])

        header.insert('NAXIS1', ('NAXIS', 2, 'Number of axes'))
        assert list(header.keys())[0] == 'NAXIS'
        assert header[0] == 2
        assert header.comments[0] == 'Number of axes'

        header.insert('NAXIS1', ('NAXIS2', 20), after=True)
        assert list(header.keys())[1] == 'NAXIS1'
        assert list(header.keys())[2] == 'NAXIS2'
        assert header[2] == 20

        header.insert(('COMMENT', 1), ('COMMENT', 'Comment 2'))
        assert header['COMMENT'] == ['Comment 1', 'Comment 2', 'Comment 3']

        header.insert(('COMMENT', 2), ('COMMENT', 'Comment 4'), after=True)
        assert header['COMMENT'] == ['Comment 1', 'Comment 2', 'Comment 3',
                                     'Comment 4']

        header.insert(-1, ('TEST1', True))
        assert list(header.keys())[-2] == 'TEST1'

        header.insert(-1, ('TEST2', True), after=True)
        assert list(header.keys())[-1] == 'TEST2'
        assert list(header.keys())[-3] == 'TEST1'

    def test_header_comments(self):
        header = pyfits.Header([('A', 'B', 'C'), ('DEF', 'G', 'H')])
        assert_equal(repr(header.comments),
                     '       A  C\n'
                     '     DEF  H')

    def test_comment_slices_and_filters(self):
        header = pyfits.Header([('AB', 'C', 'D'), ('EF', 'G', 'H'),
                                ('AI', 'J', 'K')])
        s = header.comments[1:]
        assert_equal(list(s), ['H', 'K'])
        s = header.comments[::-1]
        assert_equal(list(s), ['K', 'H', 'D'])
        s = header.comments['A*']
        assert_equal(list(s), ['D', 'K'])

    def test_comment_slice_filter_assign(self):
        header = pyfits.Header([('AB', 'C', 'D'), ('EF', 'G', 'H'),
                                ('AI', 'J', 'K')])
        header.comments[1:] = 'L'
        assert_equal(list(header.comments), ['D', 'L', 'L'])
        assert_equal(header.cards[header.index('AB')].comment, 'D')
        assert_equal(header.cards[header.index('EF')].comment, 'L')
        assert_equal(header.cards[header.index('AI')].comment, 'L')

        header.comments[::-1] = header.comments[:]
        assert_equal(list(header.comments), ['L', 'L', 'D'])

        header.comments['A*'] = ['M', 'N']
        assert_equal(list(header.comments), ['M', 'L', 'N'])

    def test_update_comment(self):
        hdul = pyfits.open(self.data('arange.fits'))
        hdul[0].header['FOO'] = ('BAR', 'BAZ')
        hdul.writeto(self.temp('test.fits'))

        hdul = pyfits.open(self.temp('test.fits'), mode='update')
        hdul[0].header.comments['FOO'] = 'QUX'
        hdul.close()

        hdul = pyfits.open(self.temp('test.fits'))
        assert_equal(hdul[0].header.comments['FOO'], 'QUX')

    def test_commentary_slicing(self):
        header = pyfits.Header()

        indices = range(5)

        for idx in indices:
            header['HISTORY'] = idx

        # Just a few sample slice types; this won't get all corner cases but if
        # these all work we should be in good shape
        assert_equal(header['HISTORY'][1:], indices[1:])
        assert_equal(header['HISTORY'][:3], indices[:3])
        assert_equal(header['HISTORY'][:6], indices[:6])
        assert_equal(header['HISTORY'][:-2], indices[:-2])
        assert_equal(header['HISTORY'][::-1], indices[::-1])
        assert_equal(header['HISTORY'][1::-1], indices[1::-1])
        assert_equal(header['HISTORY'][1:5:2], indices[1:5:2])

        # Same tests, but copy the values first; as it turns out this is
        # different from just directly doing an __eq__ as in the first set of
        # assertions
        header.insert(0, ('A', 'B', 'C'))
        header.append(('D', 'E', 'F'), end=True)
        assert_equal(list(header['HISTORY'][1:]), indices[1:])
        assert_equal(list(header['HISTORY'][:3]), indices[:3])
        assert_equal(list(header['HISTORY'][:6]), indices[:6])
        assert_equal(list(header['HISTORY'][:-2]), indices[:-2])
        assert_equal(list(header['HISTORY'][::-1]), indices[::-1])
        assert_equal(list(header['HISTORY'][1::-1]), indices[1::-1])
        assert_equal(list(header['HISTORY'][1:5:2]), indices[1:5:2])

    def test_update_commentary(self):
        header = pyfits.Header()
        header['FOO'] = 'BAR'
        header['HISTORY'] = 'ABC'
        header['FRED'] = 'BARNEY'
        header['HISTORY'] = 'DEF'
        header['HISTORY'] = 'GHI'

        assert_equal(header['HISTORY'], ['ABC', 'DEF', 'GHI'])

        # Single value update
        header['HISTORY'][0] = 'FOO'
        assert_equal(header['HISTORY'], ['FOO', 'DEF', 'GHI'])

        # Single value partial slice update
        header['HISTORY'][1:] = 'BAR'
        assert_equal(header['HISTORY'], ['FOO', 'BAR', 'BAR'])

        # Multi-value update
        header['HISTORY'][:] = ['BAZ', 'QUX']
        assert_equal(header['HISTORY'], ['BAZ', 'QUX', 'BAR'])

    def test_long_commentary_card(self):
        header = pyfits.Header()
        header['FOO'] = 'BAR'
        header['BAZ'] = 'QUX'
        longval = 'ABC' * 30
        header['HISTORY'] = longval
        header['FRED'] = 'BARNEY'
        header['HISTORY'] = longval

        assert_equal(len(header), 7)
        assert_equal(header.keys()[2], 'FRED')
        assert_equal(str(header.cards[3]), 'HISTORY ' + longval[:72])
        assert_equal(str(header.cards[4]).rstrip(), 'HISTORY ' + longval[72:])

        header.set('HISTORY', longval, after='FOO')
        assert_equal(len(header), 9)
        assert_equal(str(header.cards[1]), 'HISTORY ' + longval[:72])
        assert_equal(str(header.cards[2]).rstrip(), 'HISTORY ' + longval[72:])

    def test_header_fromtextfile(self):
        """Regression test for #122.

        Manually write a text file containing some header cards ending with
        newlines and ensure that fromtextfile can read them back in.
        """

        header = pyfits.Header()
        header['A'] = ('B', 'C')
        header['B'] = ('C', 'D')
        header['C'] = ('D', 'E')

        with open(self.temp('test.hdr'), 'w') as f:
            f.write('\n'.join(str(c).strip() for c in header.cards))

        header2 = pyfits.Header.fromtextfile(self.temp('test.hdr'))
        assert_equal(header, header2)

    def test_header_fromtextfile_with_end_card(self):
        """Regression test for #154.

        Make sure that when a Header is read from a text file that the END card
        is ignored.
        """

        header = pyfits.Header([('A', 'B', 'C'), ('D', 'E', 'F')])

        # We don't use header.totextfile here because it writes each card with
        # trailing spaces to pad them out to 80 characters.  But this bug only
        # presents itself when each card ends immediately with a newline, and
        # no trailing spaces
        with open(self.temp('test.hdr'), 'w') as f:
            f.write('\n'.join(str(c).strip() for c in header.cards))
            f.write('\nEND')

        new_header = pyfits.Header.fromtextfile(self.temp('test.hdr'))

        assert_false('END' in new_header)
        assert_equal(header, new_header)

    def test_append_end_card(self):
        """
        Regression test 2 for #154.

        Manually adding an END card to a header should simply result in a
        ValueError (as was the case in PyFITS 3.0 and earlier).
        """

        header = pyfits.Header([('A', 'B', 'C'), ('D', 'E', 'F')])

        def setitem(k, v):
            header[k] = v

        assert_raises(ValueError, setitem, 'END', '')
        assert_raises(ValueError, header.append, 'END')
        assert_raises(ValueError, header.append, 'END', end=True)
        assert_raises(ValueError, header.insert, len(header), 'END')
        assert_raises(ValueError, header.set, 'END')

    def test_invalid_end_cards(self):
        """
        Regression test for https://trac.assembla.com/pyfits/ticket/217

        This tests the case where the END card looks like a normal card like
        'END = ' and other similar oddities.  As long as a card starts with END
        and looks like it was intended to be the END card we allow it, but with
        a warning.
        """

        horig = pyfits.PrimaryHDU(data=np.arange(100)).header

        def invalid_header(end, pad):
            # Build up a goofy invalid header
            # Start from a seemingly normal header
            s = horig.tostring(sep='', endcard=False, padding=False)
            # append the bogus end card
            s += end
            # add additional padding if requested
            if pad:
                s += ' ' * _pad_length(len(s))

            return StringIO(s)

        # Basic case motivated by the original issue; it's as if the END card
        # was appened by software that doesn't know to treat it specially, and
        # it is given an = after it
        s = invalid_header('END =', True)

        with catch_warnings(record=True) as w:
            h = pyfits.Header.fromfile(s)
            assert h == horig
            assert len(w) == 1
            assert str(w[0].message).startswith(
                "Unexpected bytes trailing END keyword: ' ='")

        # A case similar to the last but with more spaces between END and the
        # =, as though the '= ' value indicator were placed like that of a
        # normal card
        s = invalid_header('END     = ', True)
        with catch_warnings(record=True) as w:
            h = pyfits.Header.fromfile(s)
            assert h == horig
            assert len(w) == 1
            assert str(w[0].message).startswith(
                "Unexpected bytes trailing END keyword: '     ='")

        # END card with trailing gibberish
        s = invalid_header('END$%&%^*%*', True)
        with catch_warnings(record=True) as w:
            h = pyfits.Header.fromfile(s)
            assert h == horig
            assert len(w) == 1
            assert str(w[0].message).startswith(
                "Unexpected bytes trailing END keyword: '$%&%^*%*'")

        # 'END' at the very end of a truncated file without padding; the way
        # the block reader works currently this can only happen if the 'END'
        # is at the very end of the file.
        s = invalid_header('END', False)
        with catch_warnings(record=True) as w:
            # Don't raise an exception on missing padding, but still produce a
            # warning that the END card is incomplete
            h = pyfits.Header.fromfile(s, padding=False)
            assert h == horig
            assert len(w) == 1
            assert str(w[0].message).startswith(
                "Missing padding to end of the FITS block")

    def test_unnecessary_move(self):
        """
        Regression test for #125.

        Ensures that a header is not modified when setting the position of a
        keyword that's already in its correct position.
        """

        header = pyfits.Header([('A', 'B'), ('B', 'C'), ('C', 'D')])

        header.set('B', before=2)
        assert_equal(header.keys(), ['A', 'B', 'C'])
        assert_false(header._modified)

        header.set('B', after=0)
        assert_equal(header.keys(), ['A', 'B', 'C'])
        assert_false(header._modified)

        header.set('B', before='C')
        assert_equal(header.keys(), ['A', 'B', 'C'])
        assert_false(header._modified)

        header.set('B', after='A')
        assert_equal(header.keys(), ['A', 'B', 'C'])
        assert_false(header._modified)

        header.set('B', before=2)
        assert_equal(header.keys(), ['A', 'B', 'C'])
        assert_false(header._modified)

        # 123 is well past the end, and C is already at the end, so it's in the
        # right place already
        header.set('C', before=123)
        assert_equal(header.keys(), ['A', 'B', 'C'])
        assert_false(header._modified)

        header.set('C', after=123)
        assert_equal(header.keys(), ['A', 'B', 'C'])
        assert_false(header._modified)

    def test_invalid_float_cards(self):
        """Regression test for #137."""

        # Create a header containing two of the problematic cards in the test
        # case where this came up:
        hstr = "FOCALLEN= +1.550000000000e+002\nAPERTURE= +0.000000000000e+000"
        h = pyfits.Header.fromstring(hstr, sep='\n')

        # First the case that *does* work prior to fixing this issue
        assert_equal(h['FOCALLEN'], 155.0)
        assert_equal(h['APERTURE'], 0.0)

        # Now if this were reserialized, would new values for these cards be
        # written with repaired exponent signs?
        with CaptureStdio():
            assert_equal(str(h.cards['FOCALLEN']),
                         _pad("FOCALLEN= +1.550000000000E+002"))
            assert_true(h.cards['FOCALLEN']._modified)
            assert_equal(str(h.cards['APERTURE']),
                         _pad("APERTURE= +0.000000000000E+000"))
            assert_true(h.cards['APERTURE']._modified)
            assert_true(h._modified)

        # This is the case that was specifically causing problems; generating
        # the card strings *before* parsing the values.  Also, the card strings
        # really should be "fixed" before being returned to the user
        h = pyfits.Header.fromstring(hstr, sep='\n')
        with CaptureStdio():
            assert_equal(str(h.cards['FOCALLEN']),
                         _pad("FOCALLEN= +1.550000000000E+002"))
            assert_true(h.cards['FOCALLEN']._modified)
            assert_equal(str(h.cards['APERTURE']),
                         _pad("APERTURE= +0.000000000000E+000"))
            assert_true(h.cards['APERTURE']._modified)

        assert_equal(h['FOCALLEN'], 155.0)
        assert_equal(h['APERTURE'], 0.0)
        assert_true(h._modified)

        # For the heck of it, try assigning the identical values and ensure
        # that the newly fixed value strings are left intact
        h['FOCALLEN'] = 155.0
        h['APERTURE'] = 0.0
        assert_equal(str(h.cards['FOCALLEN']),
                     _pad("FOCALLEN= +1.550000000000E+002"))
        assert_equal(str(h.cards['APERTURE']),
                     _pad("APERTURE= +0.000000000000E+000"))

    def test_invalid_float_cards2(self):
        """
        Regression test for #140.
        """

        # The example for this test requires creating a FITS file containing a
        # slightly misformatted float value.  I can't actually even find a way
        # to do that directly through PyFITS--it won't let me.
        hdu = pyfits.PrimaryHDU()
        hdu.header['TEST'] = 5.0022221e-07
        hdu.writeto(self.temp('test.fits'))

        # Here we manually make the file invalid
        with open(self.temp('test.fits'), 'rb+') as f:
            f.seek(346)  # Location of the exponent 'E' symbol
            f.write(encode_ascii('e'))

        hdul = pyfits.open(self.temp('test.fits'))
        with catch_warnings(record=True) as w:
            with CaptureStdio():
                hdul.writeto(self.temp('temp.fits'), output_verify='warn')
            assert_equal(len(w), 5)
            # The first two warnings are just the headers to the actual warning
            # message (HDU 0, Card 4).  I'm still not sure things like that
            # should be output as separate warning messages, but that's
            # something to think about...
            msg = str(w[3].message)
            assert_true('(invalid value string: 5.0022221e-07)' in msg)

    def test_leading_zeros(self):
        """
        Regression test for #137, part 2.

        Ticket #137 also showed that in float values like 0.001 the leading
        zero was unnecessarily being stripped off when rewriting the header.
        Though leading zeros should be removed from integer values to prevent
        misinterpretation as octal by python (for now PyFITS will still
        maintain the leading zeros if now changes are made to the value, but
        will drop them if changes are made).
        """

        c = pyfits.Card.fromstring("APERTURE= +0.000000000000E+000")
        assert_equal(str(c), _pad("APERTURE= +0.000000000000E+000"))
        assert_equal(c.value, 0.0)
        c = pyfits.Card.fromstring("APERTURE= 0.000000000000E+000")
        assert_equal(str(c), _pad("APERTURE= 0.000000000000E+000"))
        assert_equal(c.value, 0.0)
        c = pyfits.Card.fromstring("APERTURE= 017")
        assert_equal(str(c), _pad("APERTURE= 017"))
        assert_equal(c.value, 17)

    def test_assign_boolean(self):
        """
        Regression test for #123. Tests assigning Python and Numpy boolean
        values to keyword values.
        """

        fooimg = _pad('FOO     =                    T')
        barimg = _pad('BAR     =                    F')
        h = pyfits.Header()
        h['FOO'] = True
        h['BAR'] = False
        assert_equal(h['FOO'], True)
        assert_equal(h['BAR'], False)
        assert_equal(str(h.cards['FOO']), fooimg)
        assert_equal(str(h.cards['BAR']), barimg)

        h = pyfits.Header()
        h['FOO'] = np.bool_(True)
        h['BAR'] = np.bool_(False)
        assert_equal(h['FOO'], True)
        assert_equal(h['BAR'], False)
        assert_equal(str(h.cards['FOO']), fooimg)
        assert_equal(str(h.cards['BAR']), barimg)

        h = pyfits.Header()
        h.append(pyfits.Card.fromstring(fooimg))
        h.append(pyfits.Card.fromstring(barimg))
        assert_equal(h['FOO'], True)
        assert_equal(h['BAR'], False)
        assert_equal(str(h.cards['FOO']), fooimg)
        assert_equal(str(h.cards['BAR']), barimg)

    def test_header_method_keyword_normalization(self):
        """
        Regression test for #149.  Basically ensures that all public Header
        methods are case-insensitive w.r.t. keywords.

        Provides a reasonably comprehensive test of several methods at once.
        """

        h = pyfits.Header([('abC', 1), ('Def', 2), ('GeH', 3)])
        assert_equal(h.keys(), ['ABC', 'DEF', 'GEH'])
        assert_true('abc' in h)
        assert_true('dEf' in h)

        assert_equal(h['geh'], 3)

        # Case insensitivity of wildcards
        assert_equal(len(h['g*']), 1)

        h['aBc'] = 2
        assert_equal(h['abc'], 2)
        # ABC already existed so assigning to aBc should not have added any new
        # cards
        assert_equal(len(h), 3)

        del h['gEh']
        assert_equal(h.keys(), ['ABC', 'DEF'])
        assert_equal(len(h), 2)
        assert_equal(h.get('def'), 2)

        h.set('Abc', 3)
        assert_equal(h['ABC'], 3)
        h.set('gEh', 3, before='Abc')
        assert_equal(h.keys(), ['GEH', 'ABC', 'DEF'])

        assert_equal(h.pop('abC'), 3)
        assert_equal(len(h), 2)

        assert_equal(h.setdefault('def', 3), 2)
        assert_equal(len(h), 2)
        assert_equal(h.setdefault('aBc', 1), 1)
        assert_equal(len(h), 3)
        assert_equal(h.keys(), ['GEH', 'DEF', 'ABC'])

        h.update({'GeH': 1, 'iJk': 4})
        assert_equal(len(h), 4)
        assert_equal(h.keys(), ['GEH', 'DEF', 'ABC', 'IJK'])
        assert_equal(h['GEH'], 1)

        assert_equal(h.count('ijk'), 1)
        assert_equal(h.index('ijk'), 3)

        h.remove('Def')
        assert_equal(len(h), 3)
        assert_equal(h.keys(), ['GEH', 'ABC', 'IJK'])

    def test_end_in_comment(self):
        """
        Regression test for #142.  Tests a case where the comment of a card
        ends with END, and is followed by several blank cards.
        """

        data = np.arange(100).reshape((10, 10))
        hdu = pyfits.PrimaryHDU(data=data)
        hdu.header['TESTKW'] = ('Test val', 'This is the END')
        # Add a couple blanks after the END string
        hdu.header.append()
        hdu.header.append()
        hdu.writeto(self.temp('test.fits'))

        with pyfits.open(self.temp('test.fits'), memmap=False) as hdul:
            # memmap = False to avoid leaving open a mmap to the file when we
            # access the data--this causes problems on Windows when we try to
            # overwrite the file later
            assert_true('TESTKW' in hdul[0].header)
            assert_equal(hdul[0].header, hdu.header)
            assert_true((hdul[0].data == data).all())

        # Add blanks until the header is extended to two block sizes
        while len(hdu.header) < 36:
            hdu.header.append()
        with ignore_warnings():
            hdu.writeto(self.temp('test.fits'), clobber=True)

        with pyfits.open(self.temp('test.fits')) as hdul:
            assert_true('TESTKW' in hdul[0].header)
            assert_equal(hdul[0].header, hdu.header)
            assert_true((hdul[0].data == data).all())

        # Test parsing the same header when it's written to a text file
        hdu.header.totextfile(self.temp('test.hdr'))
        header2 = pyfits.Header.fromtextfile(self.temp('test.hdr'))
        assert_equal(hdu.header, header2)

    def test_assign_unicode(self):
        """
        Regression test for #134.  Assigning a unicode literal as a header
        value should not fail silently.  If the value can be converted to ASCII
        then it should just work.  Otherwise it should fail with an appropriate
        value error.

        Also tests unicode for keywords and comments.
        """

        erikku = u'\u30a8\u30ea\u30c3\u30af'

        def assign(keyword, val):
            h[keyword] = val

        h = pyfits.Header()
        h[u'FOO'] = 'BAR'
        assert_true('FOO' in h)
        assert_equal(h['FOO'], 'BAR')
        assert_equal(h[u'FOO'], 'BAR')
        assert_equal(repr(h), _pad("FOO     = 'BAR     '"))
        assert_raises(ValueError, assign, erikku, 'BAR')


        h['FOO'] = u'BAZ'
        assert_equal(h[u'FOO'], 'BAZ')
        assert_equal(h[u'FOO'], u'BAZ')
        assert_equal(repr(h), _pad("FOO     = 'BAZ     '"))
        assert_raises(ValueError, assign, 'FOO', erikku)

        h['FOO'] = ('BAR', u'BAZ')
        assert_equal(h['FOO'], 'BAR')
        assert_equal(h['FOO'], u'BAR')
        assert_equal(h.comments['FOO'], 'BAZ')
        assert_equal(h.comments['FOO'], u'BAZ')
        assert_equal(repr(h), _pad("FOO     = 'BAR     '           / BAZ"))

        h['FOO'] = (u'BAR', u'BAZ')
        assert_equal(h['FOO'], 'BAR')
        assert_equal(h['FOO'], u'BAR')
        assert_equal(h.comments['FOO'], 'BAZ')
        assert_equal(h.comments['FOO'], u'BAZ')
        assert_equal(repr(h), _pad("FOO     = 'BAR     '           / BAZ"))

        assert_raises(ValueError, assign, 'FOO', ('BAR', erikku))
        assert_raises(ValueError, assign, 'FOO', (erikku, 'BAZ'))
        assert_raises(ValueError, assign, 'FOO', (erikku, erikku))

    def test_assign_non_ascii(self):
        """
        First regression test for
        https://github.com/spacetelescope/PyFITS/issues/37

        Although test_assign_unicode ensures that Python 2 `unicode` objects
        and Python 3 `str` objects containing non-ASCII characters cannot be
        assigned to headers, there is a bug that allows Python 2 `str` objects
        of arbitrary encoding containing non-ASCII characters to be passed
        through.

        On Python 3 it should not be possible to assign bytes to a header at
        all.
        """

        h = pyfits.Header()
        if PY3:
            assert_raises(ValueError, h.set, 'TEST',
                          bytes('Hello', encoding='ascii'))
        else:
            assert_raises(ValueError, h.set, 'TEST', str('ñ'))

    def test_header_strip_whitespace(self):
        """
        Regression test for #146, and for the solution that is optional
        stripping of whitespace from the end of a header value.

        By default extra whitespace is stripped off, but if
        pyfits.STRIP_HEADER_WHITESPACE = False it should not be stripped.
        """

        h = pyfits.Header()
        h['FOO'] = 'Bar      '
        assert_equal(h['FOO'], 'Bar')
        c = pyfits.Card.fromstring("QUX     = 'Bar        '")
        h.append(c)
        assert_equal(h['QUX'], 'Bar')
        assert_equal(h.cards['FOO'].image.rstrip(),
                     "FOO     = 'Bar      '")
        assert_equal(h.cards['QUX'].image.rstrip(),
                     "QUX     = 'Bar        '")

        pyfits.STRIP_HEADER_WHITESPACE = False
        try:
            assert_equal(h['FOO'], 'Bar      ')
            assert_equal(h['QUX'], 'Bar        ')
            assert_equal(h.cards['FOO'].image.rstrip(),
                         "FOO     = 'Bar      '")
            assert_equal(h.cards['QUX'].image.rstrip(),
                         "QUX     = 'Bar        '")
        finally:
            pyfits.STRIP_HEADER_WHITESPACE = True

        assert_equal(h['FOO'], 'Bar')
        assert_equal(h['QUX'], 'Bar')
        assert_equal(h.cards['FOO'].image.rstrip(),
                     "FOO     = 'Bar      '")
        assert_equal(h.cards['QUX'].image.rstrip(),
                     "QUX     = 'Bar        '")

    def test_keep_duplicate_history_in_orig_header(self):
        """
        Regression test for #156.

        When creating a new HDU from an existing Header read from an existing
        FITS file, if the origianl header contains duplicate HISTORY values
        those duplicates should be preserved just as in the original header.

        This bug occurred due to naivete in Header.extend.
        """

        history = ['CCD parameters table ...',
                   '   reference table oref$n951041ko_ccd.fits',
                   '     INFLIGHT 12/07/2001 25/02/2002',
                   '     all bias frames'] * 3

        hdu = pyfits.PrimaryHDU()
        # Add the history entries twice
        for item in history:
            hdu.header['HISTORY'] = item

        hdu.writeto(self.temp('test.fits'))

        with pyfits.open(self.temp('test.fits')) as hdul:
            assert_equal(hdul[0].header['HISTORY'], history)

        new_hdu = pyfits.PrimaryHDU(header=hdu.header)
        assert_equal(new_hdu.header['HISTORY'], hdu.header['HISTORY'])
        new_hdu.writeto(self.temp('test2.fits'))

        with pyfits.open(self.temp('test2.fits')) as hdul:
            assert_equal(hdul[0].header['HISTORY'], history)

    def test_invalid_keyword_cards(self):
        """
        Test for #109.  Allow opening files with headers containing invalid
        keywords.
        """

        # Create a header containing a few different types of BAD headers.
        c1 = pyfits.Card.fromstring('CLFIND2D: contour = 0.30')
        c2 = pyfits.Card.fromstring('Just some random text.')
        c3 = pyfits.Card.fromstring('A' * 80)

        hdu = pyfits.PrimaryHDU()
        # This should work with some warnings
        with catch_warnings(record=True) as w:
            hdu.header.append(c1)
            hdu.header.append(c2)
            hdu.header.append(c3)
            assert_equal(len(w), 3)

        hdu.writeto(self.temp('test.fits'))

        with catch_warnings(record=True) as w:
            with pyfits.open(self.temp('test.fits')) as hdul:
                # Merely opening the file should blast some warnings about the
                # invalid keywords
                assert_equal(len(w), 3)

                header = hdul[0].header
                assert_true('CLFIND2D' in header)
                assert_true('Just som' in header)
                assert_true('AAAAAAAA' in header)

                assert_equal(header['CLFIND2D'], ': contour = 0.30')
                assert_equal(header['Just som'], 'e random text.')
                assert_equal(header['AAAAAAAA'], 'A' * 72)

                # It should not be possible to assign to the invalid keywords
                assert_raises(ValueError, header.set, 'CLFIND2D', 'foo')
                assert_raises(ValueError, header.set, 'Just som', 'foo')
                assert_raises(ValueError, header.set, 'AAAAAAAA', 'foo')

    def test_fix_hierarch_with_invalid_value(self):
        """
        Regression test for #172.  Ensures that when fixing a hierarch card it
        remains a hierarch card.
        """

        c = pyfits.Card.fromstring('HIERARCH ESO DET CHIP PXSPACE = 5e6')
        with CaptureStdio():
            c.verify('fix')
        assert_equal(str(c), _pad('HIERARCH ESO DET CHIP PXSPACE = 5E6'))

    def test_assign_inf_nan(self):
        """
        Regression test for https://github.com/spacetelescope/PyFITS/issues/11

        For the time being it should not be possible to assign the floating
        point values inf or nan to a header value, since this is not defined by
        the FITS standard.
        """

        h = pyfits.Header()

        # There is an obscure cross-platform issue that prevents things like
        # float('nan') on Windows on older versions of Python; hence it is
        # unlikely to come up in practice
        if not (sys.platform.startswith('win32') and
                sys.version_info[:2] < (2, 6)):
           assert_raises(ValueError, h.set, 'TEST', float('nan'))
           assert_raises(ValueError, h.set, 'TEST', float('inf'))

        assert_raises(ValueError, h.set, 'TEST', np.nan)
        assert_raises(ValueError, h.set, 'TEST', np.inf)

    def test_update_bool(self):
        """
        Regression test for an issue where a value of True in a header
        cannot be updated to a value of 1, and likewise for False/0.
        """

        h = pyfits.Header([('TEST', True)])
        h['TEST'] = 1
        assert h['TEST'] is not True
        assert isinstance(h['TEST'], int)
        assert h['TEST'] == 1

        h['TEST'] = np.bool_(True)
        assert h['TEST'] is True

        h['TEST'] = False
        assert h['TEST'] is False
        h['TEST'] = np.bool_(False)
        assert h['TEST'] is False

        h['TEST'] = 0
        assert h['TEST'] is not False
        assert isinstance(h['TEST'], int)
        assert h['TEST'] == 0

        h['TEST'] = np.bool_(False)
        assert h['TEST'] is False

    def test_update_numeric(self):
        """
        Regression test for https://github.com/spacetelescope/PyFITS/issues/49

        Ensure that numeric values can be upcast/downcast between int, float,
        and complex by assigning values that compare equal to the existing
        value but are a different type.
        """

        h = pyfits.Header()
        h['TEST'] = 1

        # int -> float
        h['TEST'] = 1.0
        assert isinstance(h['TEST'], float)
        assert str(h).startswith('TEST    =                  1.0')

        # float -> int
        h['TEST'] = 1
        assert isinstance(h['TEST'], int)
        assert str(h).startswith('TEST    =                    1')

        # int -> complex
        h['TEST'] = 1.0+0.0j
        assert isinstance(h['TEST'], complex)
        assert str(h).startswith('TEST    =           (1.0, 0.0)')

        # complex -> float
        h['TEST'] = 1.0
        assert isinstance(h['TEST'], float)
        assert str(h).startswith('TEST    =                  1.0')

        # float -> complex
        h['TEST'] = 1.0+0.0j
        assert isinstance(h['TEST'], complex)
        assert str(h).startswith('TEST    =           (1.0, 0.0)')

        # complex -> int
        h['TEST'] = 1
        assert isinstance(h['TEST'], int)
        assert str(h).startswith('TEST    =                    1')

        # Now the same tests but with zeros
        h['TEST'] = 0

        # int -> float
        h['TEST'] = 0.0
        assert isinstance(h['TEST'], float)
        assert str(h).startswith('TEST    =                  0.0')

        # float -> int
        h['TEST'] = 0
        assert isinstance(h['TEST'], int)
        assert str(h).startswith('TEST    =                    0')

        # int -> complex
        h['TEST'] = 0.0+0.0j
        assert isinstance(h['TEST'], complex)
        assert str(h).startswith('TEST    =           (0.0, 0.0)')

        # complex -> float
        h['TEST'] = 0.0
        assert isinstance(h['TEST'], float)
        assert str(h).startswith('TEST    =                  0.0')

        # float -> complex
        h['TEST'] = 0.0+0.0j
        assert isinstance(h['TEST'], complex)
        assert str(h).startswith('TEST    =           (0.0, 0.0)')

        # complex -> int
        h['TEST'] = 0
        assert isinstance(h['TEST'], int)
        assert str(h).startswith('TEST    =                    0')


class TestRecordValuedKeywordCards(PyfitsTestCase):
    """
    Tests for handling of record-valued keyword cards as used by the FITS WCS
    Paper IV proposal.

    These tests are derived primarily from the release notes for PyFITS 1.4 (in
    which this feature was first introduced.
    """

    def setup(self):
        super(TestRecordValuedKeywordCards, self).setup()
        self._test_header = pyfits.Header()
        self._test_header.set('DP1', 'NAXIS: 2')
        self._test_header.set('DP1', 'AXIS.1: 1')
        self._test_header.set('DP1', 'AXIS.2: 2')
        self._test_header.set('DP1', 'NAUX: 2')
        self._test_header.set('DP1', 'AUX.1.COEFF.0: 0')
        self._test_header.set('DP1', 'AUX.1.POWER.0: 1')
        self._test_header.set('DP1', 'AUX.1.COEFF.1: 0.00048828125')
        self._test_header.set('DP1', 'AUX.1.POWER.1: 1')

    def test_initialize_rvkc(self):
        """
        Test different methods for initializing a card that should be
        recognized as a RVKC
        """

        c = pyfits.Card.fromstring("DP1     = 'NAXIS: 2' / A comment")
        assert_equal(c.keyword, 'DP1.NAXIS')
        assert_equal(c.value, 2.0)
        assert_equal(c.field_specifier, 'NAXIS')
        assert_equal(c.comment, 'A comment')

        c = pyfits.Card.fromstring("DP1     = 'NAXIS: 2.1'")
        assert_equal(c.keyword, 'DP1.NAXIS')
        assert_equal(c.value, 2.1)
        assert_equal(c.field_specifier, 'NAXIS')

        c = pyfits.Card.fromstring("DP1     = 'NAXIS: a'")
        assert_equal(c.keyword, 'DP1')
        assert_equal(c.value, 'NAXIS: a')
        assert_equal(c.field_specifier, None)

        c = pyfits.Card('DP1', 'NAXIS: 2')
        assert_equal(c.keyword, 'DP1.NAXIS')
        assert_equal(c.value, 2.0)
        assert_equal(c.field_specifier, 'NAXIS')

        c = pyfits.Card('DP1', 'NAXIS: 2.0')
        assert_equal(c.keyword, 'DP1.NAXIS')
        assert_equal(c.value, 2.0)
        assert_equal(c.field_specifier, 'NAXIS')

        c = pyfits.Card('DP1', 'NAXIS: a')
        assert_equal(c.keyword, 'DP1')
        assert_equal(c.value, 'NAXIS: a')
        assert_equal(c.field_specifier, None)

        c = pyfits.Card('DP1.NAXIS', 2)
        assert_equal(c.keyword, 'DP1.NAXIS')
        assert_equal(c.value, 2.0)
        assert_equal(c.field_specifier, 'NAXIS')

        c = pyfits.Card('DP1.NAXIS', 2.0)
        assert_equal(c.keyword, 'DP1.NAXIS')
        assert_equal(c.value, 2.0)
        assert_equal(c.field_specifier, 'NAXIS')

        with ignore_warnings():
            c = pyfits.Card('DP1.NAXIS', 'a')
        assert_equal(c.keyword, 'DP1.NAXIS')
        assert_equal(c.value, 'a')
        assert_equal(c.field_specifier, None)

    def test_parse_field_specifier(self):
        """
        Tests that the field_specifier can accessed from a card read from a
        string before any other attributes are accessed.
        """

        c = pyfits.Card.fromstring("DP1     = 'NAXIS: 2' / A comment")
        assert_equal(c.field_specifier, 'NAXIS')
        assert_equal(c.keyword, 'DP1.NAXIS')
        assert_equal(c.value, 2.0)
        assert_equal(c.comment, 'A comment')

    def test_update_field_specifier(self):
        """
        Test setting the field_specifier attribute and updating the card image
        to reflect the new value.
        """

        c = pyfits.Card.fromstring("DP1     = 'NAXIS: 2' / A comment")
        assert_equal(c.field_specifier, 'NAXIS')
        c.field_specifier = 'NAXIS1'
        assert_equal(c.field_specifier, 'NAXIS1')
        assert_equal(c.keyword, 'DP1.NAXIS1')
        assert_equal(c.value, 2.0)
        assert_equal(c.comment, 'A comment')
        assert_equal(str(c).rstrip(), "DP1     = 'NAXIS1: 2' / A comment")

    def test_field_specifier_case_senstivity(self):
        """
        The keyword portion of an RVKC should still be case-insensitive, but
        the field-specifier portion should be case-sensitive.
        """

        header = pyfits.Header()
        header.set('abc.def', 1)
        header.set('abc.DEF', 2)
        assert_equal(header['abc.def'], 1)
        assert_equal(header['ABC.def'], 1)
        assert_equal(header['aBc.def'], 1)
        assert_equal(header['ABC.DEF'], 2)
        assert_false('ABC.dEf' in header)

    def test_get_rvkc_by_index(self):
        """
        Returning a RVKC from a header via index lookup should return the
        float value of the card.
        """

        assert_equal(self._test_header[0], 2.0)
        assert_true(isinstance(self._test_header[0], float))
        assert_equal(self._test_header[1], 1.0)
        assert_true(isinstance(self._test_header[1], float))

    def test_get_rvkc_by_keyword(self):
        """
        Returning a RVKC just via the keyword name should return the full value
        string of the first card with that keyword.

        This test was changed to reflect the requirement in ticket
        #184--previously it required _test_header['DP1'] to return the parsed
        float value.
        """

        assert_equal(self._test_header['DP1'], 'NAXIS: 2')

    def test_get_rvkc_by_keyword_and_field_specifier(self):
        """
        Returning a RVKC via the full keyword/field-specifier combination
        should return the floating point value associated with the RVKC.
        """

        assert_equal(self._test_header['DP1.NAXIS'], 2.0)
        assert_true(isinstance(self._test_header['DP1.NAXIS'], float))
        assert_equal(self._test_header['DP1.AUX.1.COEFF.1'], 0.00048828125)

    def test_access_nonexistent_rvkc(self):
        """
        Accessing a nonexistent RVKC should raise an IndexError for
        index-based lookup, or a KeyError for keyword lookup (like a normal
        card).
        """

        assert_raises(IndexError, lambda x: self._test_header[x], 8)
        assert_raises(KeyError, lambda k: self._test_header[k], 'DP1.AXIS.3')
        # Test the exception message
        try:
            self._test_header['DP1.AXIS.3']
        except KeyError, e:
            assert_equal(e.args[0], "Keyword 'DP1.AXIS.3' not found.")

    def test_update_rvkc(self):
        """A RVKC can be updated either via index or keyword access."""

        self._test_header[0] = 3
        assert_equal(self._test_header['DP1.NAXIS'], 3.0)
        assert_true(isinstance(self._test_header['DP1.NAXIS'], float))

        self._test_header['DP1.AXIS.1'] = 1.1
        assert_equal(self._test_header['DP1.AXIS.1'], 1.1)

    def test_update_rvkc_2(self):
        """Regression test for an issue that appeared after SVN r2412."""

        h = pyfits.Header()
        h['D2IM1.EXTVER'] = 1
        assert h['D2IM1.EXTVER'] == 1.0
        h['D2IM1.EXTVER'] = 2
        assert h['D2IM1.EXTVER'] == 2.0

    def test_raw_keyword_value(self):
        c = pyfits.Card.fromstring("DP1     = 'NAXIS: 2' / A comment")
        assert c.rawkeyword == 'DP1'
        assert c.rawvalue == 'NAXIS: 2'

        c = pyfits.Card('DP1.NAXIS', 2)
        assert c.rawkeyword == 'DP1'
        assert c.rawvalue == 'NAXIS: 2.0'

        c = pyfits.Card('DP1.NAXIS', 2.0)
        assert c.rawkeyword == 'DP1'
        assert c.rawvalue == 'NAXIS: 2.0'

    def test_rvkc_insert_after(self):
        """
        It should be possible to insert a new RVKC after an existing one
        specified by the full keyword/field-specifier combination."""

        self._test_header.set('DP1', 'AXIS.3: 1', 'a comment',
                              after='DP1.AXIS.2')
        assert_equal(self._test_header[3], 1)
        assert_equal(self._test_header['DP1.AXIS.3'], 1)

    def test_rvkc_delete(self):
        """
        Deleting a RVKC should work as with a normal card by using the full
        keyword/field-spcifier combination.
        """

        del self._test_header['DP1.AXIS.1']
        assert_equal(len(self._test_header), 7)
        assert_equal(self._test_header.keys()[0], 'DP1.NAXIS')
        assert_equal(self._test_header[0], 2)
        assert_equal(self._test_header.keys()[1], 'DP1.AXIS.2')
        assert_equal(self._test_header[1], 2)

        # Perform a subsequent delete to make sure all the index mappings were
        # updated
        del self._test_header['DP1.AXIS.2']
        assert len(self._test_header) == 6
        assert self._test_header.keys()[0] == 'DP1.NAXIS'
        assert self._test_header[0] == 2
        assert self._test_header.keys()[1] == 'DP1.NAUX'
        assert self._test_header[1] == 2

    def test_pattern_matching_keys(self):
        """Test the keyword filter strings with RVKCs."""

        cl = self._test_header['DP1.AXIS.*']
        assert_true(isinstance(cl, pyfits.Header))
        assert_equal(
            [str(c).strip() for c in cl.cards],
            ["DP1     = 'AXIS.1: 1'",
             "DP1     = 'AXIS.2: 2'"])

        cl = self._test_header['DP1.N*']
        assert_equal(
            [str(c).strip() for c in cl.cards],
            ["DP1     = 'NAXIS: 2'",
             "DP1     = 'NAUX: 2'"])

        cl = self._test_header['DP1.AUX...']
        assert_equal(
            [str(c).strip() for c in cl.cards],
            ["DP1     = 'AUX.1.COEFF.0: 0'",
             "DP1     = 'AUX.1.POWER.0: 1'",
             "DP1     = 'AUX.1.COEFF.1: 0.00048828125'",
             "DP1     = 'AUX.1.POWER.1: 1'"])

        cl = self._test_header['DP?.NAXIS']
        assert_equal(
            [str(c).strip() for c in cl.cards],
            ["DP1     = 'NAXIS: 2'"])

        cl = self._test_header['DP1.A*S.*']
        assert_equal(
            [str(c).strip() for c in cl.cards],
            ["DP1     = 'AXIS.1: 1'",
             "DP1     = 'AXIS.2: 2'"])

    def test_pattern_matching_key_deletion(self):
        """Deletion by filter strings should work."""

        del self._test_header['DP1.A*...']
        assert_equal(len(self._test_header), 2)
        assert_equal(self._test_header.keys()[0], 'DP1.NAXIS')
        assert_equal(self._test_header[0], 2)
        assert_equal(self._test_header.keys()[1], 'DP1.NAUX')
        assert_equal(self._test_header[1], 2)

    def test_successive_pattern_matching(self):
        """
        A card list returned via a filter string should be further filterable.
        """

        cl = self._test_header['DP1.A*...']
        assert_equal(
            [str(c).strip() for c in cl.cards],
            ["DP1     = 'AXIS.1: 1'",
             "DP1     = 'AXIS.2: 2'",
             "DP1     = 'AUX.1.COEFF.0: 0'",
             "DP1     = 'AUX.1.POWER.0: 1'",
             "DP1     = 'AUX.1.COEFF.1: 0.00048828125'",
             "DP1     = 'AUX.1.POWER.1: 1'"])

        cl2 = cl['*.*AUX...']
        assert_equal(
            [str(c).strip() for c in cl2.cards],
            ["DP1     = 'AUX.1.COEFF.0: 0'",
             "DP1     = 'AUX.1.POWER.0: 1'",
             "DP1     = 'AUX.1.COEFF.1: 0.00048828125'",
             "DP1     = 'AUX.1.POWER.1: 1'"])

    def test_rvkc_in_cardlist_keys(self):
        """
        The CardList.keys() method should return full keyword/field-spec values
        for RVKCs.
        """

        cl = self._test_header['DP1.AXIS.*']
        assert_equal(cl.keys(), ['DP1.AXIS.1', 'DP1.AXIS.2'])

    def test_rvkc_in_cardlist_values(self):
        """
        The CardList.values() method should return the values of all RVKCs as
        floating point values.
        """

        cl = self._test_header['DP1.AXIS.*']
        assert_equal(cl.values(), [1.0, 2.0])

    def test_rvkc_value_attribute(self):
        """
        Individual card values should be accessible by the .value attribute
        (which should return a float).
        """

        cl = self._test_header['DP1.AXIS.*']
        assert_equal(cl.cards[0].value, 1.0)
        assert_true(isinstance(cl.cards[0].value, float))

    def test_overly_permissive_parsing(self):
        """
        Regression test for #183.

        Ensures that cards with standard commentary keywords are never treated
        as RVKCs.  Also ensures that cards not stricly matching the RVKC
        pattern are not treated as such.
        """

        h = pyfits.Header()
        h['HISTORY'] = 'AXIS.1: 2'
        h['HISTORY'] = 'AXIS.2: 2'
        assert_false('HISTORY.AXIS' in h)
        assert_false('HISTORY.AXIS.1' in h)
        assert_false('HISTORY.AXIS.2' in h)
        assert_equal(h['HISTORY'], ['AXIS.1: 2', 'AXIS.2: 2'])

        # This is an example straight out of the ticket where everything after
        # the '2012' in the date value was being ignored, allowing the value to
        # successfully be parsed as a "float"
        h = pyfits.Header()
        h['HISTORY'] = 'Date: 2012-09-19T13:58:53.756061'
        assert_false('HISTORY.Date' in h)
        assert_equal(str(h.cards[0]),
                     _pad('HISTORY Date: 2012-09-19T13:58:53.756061'))

        c = pyfits.Card.fromstring(
            "        'Date: 2012-09-19T13:58:53.756061'")
        assert_equal(c.keyword, '')
        assert_equal(c.value, "'Date: 2012-09-19T13:58:53.756061'")
        assert_equal(c.field_specifier, None)

        h = pyfits.Header()
        h['FOO'] = 'Date: 2012-09-19T13:58:53.756061'
        assert_false('FOO.Date' in h)
        assert_equal(str(h.cards[0]),
                     _pad("FOO     = 'Date: 2012-09-19T13:58:53.756061'"))

    def test_overly_aggressive_rvkc_lookup(self):
        """
        Regression test for #184.

        Ensures that looking up a RVKC by keyword only (without the
        field-specifier) in a header returns the full string value of that card
        without parsing it as a RVKC.  Also ensures that a full field-specifier
        is required to match a RVKC--a partial field-specifier that doesn't
        explicitly match any record-valued keyword should result in a KeyError.
        """

        c1 = pyfits.Card.fromstring("FOO     = 'AXIS.1: 2'")
        c2 = pyfits.Card.fromstring("FOO     = 'AXIS.2: 4'")
        h = pyfits.Header([c1, c2])
        assert_equal(h['FOO'], 'AXIS.1: 2')
        assert_equal(h[('FOO', 1)], 'AXIS.2: 4')
        assert_equal(h['FOO.AXIS.1'], 2.0)
        assert_equal(h['FOO.AXIS.2'], 4.0)
        assert_false('FOO.AXIS' in h)
        assert_false('FOO.AXIS.' in h)
        assert_false('FOO.' in h)
        assert_raises(KeyError, lambda: h['FOO.AXIS'])
        assert_raises(KeyError, lambda: h['FOO.AXIS.'])
        assert_raises(KeyError, lambda: h['FOO.'])
