from difflib import Differ
from queued_object import QueuedObject

class ForgeException(Exception):
    pass

class SignatureException(ForgeException):
    pass

class InvalidKeywordArgument(SignatureException):
    pass

class FunctionCannotBeBound(SignatureException):
    pass

class ConflictingActions(ForgeException):
    pass

class MockObjectUnhashable(ForgeException):
    pass

class CannotMockException(ForgeException):
    pass
class CannotMockFunctions(CannotMockException):
    pass

UNEXPECTED_FUNCTION_CALLED_STR = "Unexpected function called!"
UNEXPECTED_SETATTR_STR = "Unexpected attribute set!"
DIFF_DESCRIPTION_STR = "(Expected: +, Got: -)"

class UnexpectedEvent(ForgeException):
    def __init__(self, expected, got):
        super(UnexpectedEvent, self).__init__()
        self.expected = expected
        self.got = got
    def __str__(self):
        returned = self._getTitle()
        returned += " %s\n" % DIFF_DESCRIPTION_STR
        debug_info = self._get_debug_info()
        if debug_info:
            returned += debug_info
        returned += self._get_diff_string()
        return returned
    def _get_debug_info(self):
        returned = ""
        if self.expected and self.expected.caller_info:
            returned += "Recorded from %s\n" % self.expected.caller_info
        if self.got and self.got.caller_info:
            returned += "Replayed from %s\n" % self.got.caller_info
        return returned
    def _get_diff_string(self):
        got_string = self._get_got_string()
        expected_string = self._get_expected_string()
        if got_string == expected_string:
            return "- %s\n+ %s" % (expected_string, got_string)
        diff = Differ().compare(
            [self._get_got_string()],
            [self._get_expected_string()],
            )
        return "\n".join(line.strip() for line in diff)
    def _get_expected_string(self):
        return self._get_description_string(self.expected)
    def _get_got_string(self):
        return self._get_description_string(self.got)
    def _get_description_string(self, x):
        if x is None:
            return "<< None >>"
        if not isinstance(x, QueuedObject):
            return str(x)
        return x.describe()
class UnexpectedCall(UnexpectedEvent):
    @classmethod
    def _getTitle(cls):
        return UNEXPECTED_FUNCTION_CALLED_STR
class UnexpectedSetattr(UnexpectedEvent):
    @classmethod
    def _getTitle(cls):
        return UNEXPECTED_SETATTR_STR

class ExpectedEventsNotFound(ForgeException):
    def __init__(self, events):
        super(ExpectedEventsNotFound, self).__init__()
        self.events = events
    def __str__(self):
        return "Expected events not found:\n%s" % "\n".join(map(str, self.events))

class InvalidEntryPoint(ForgeException):
    pass

