from pygments.lexer import RegexLexer, include
from pygments.token import *

class WaspLexer(RegexLexer):
  name = 'Wasp'
  aliases = ['wasp']
  filenames = ['*.wasp']
  mimetypes = ['text/x-wasp']

  identifier = r'[a-zA-Z~!@%^&*_+=:<.>/?]+[a-zA-Z0-9~!@%^&*_\-+=:<.>/?]*'
  hexadecimal = r'([a-f0-9]|[A-F0-9])'
  path = r'[^/\\?%*:"<> ]+'

  # Small hack to highlight commands that are imported within a code block.
  # This happens even if they are parameters to another command, because that
  # *is* permitted.
  def add_builtin (self, match):
    text = match.group(0)
    self.builtins.append(text)
    yield match.start(), Name.Builtin, text

  def add_defined (self, match):
    text = match.group(0)
    self.defined.append(text)
    yield match.start(), Name.Function, text

  def get_tokens_unprocessed (self, text):
    for index, token, value in RegexLexer.get_tokens_unprocessed(self, text):
      is_builtin = token is Name.Variable and value in self.builtins
      is_defined = token is Name.Variable and value in self.defined
      token_type = is_builtin and Name.Builtin or \
        (is_defined and Name.Function or token)
      yield index, token_type, value

  defined = [ ]

  # This is a list of *ALL* builtin commands for Wasp, including the ones that
  # an implementation is not required to provide
  builtins = [
    # comparison
    'greater-equal?', 'less-equal?', 'not-equal?', 'equal?', 'greater?',
    'less?', 'equivalent?',
    # arithmetic
    'negate', 'multiply', 'subtract', 'divide', 'modulo', 'add',
    # bitwise
    'xor', 'bitwise-and', 'bitwise-or', 'shift-right', 'shift-left',
    # logic
    'and', 'not', 'or',
    # type
    'num?', 'complex?', 'int?', 'real?', 'bool?', 'str?', 'seq?', 'kvd?',
    'data?', 'weak?', 'bytes?', 'error?'
    # control flow
    'for-each', 'return', 'yield', 'repeat', 'unless', 'when', 'let', 'if',
    'invoke', 'nothing',
    # runtime
    'copy',
  ]

  tokens = {
    'root' : [
      # comments
      (r'#\|', Comment.Multiline, 'multiline-comment'),
      (r'#.*?\n', Comment.Single),

      # whitespace
      (r'\s+', Text),

      # constants
      (r'\b(true|false|none)\b', Name.Constant),

      # literals
      (r"'" + identifier, String),
      (r'"', String.Double, 'string'),
      (r'\$([a-f0-9]{4,8}|[A-F0-9]{4,8}|.)', String.Char),
      include('number'),

      # highlight keywords
      (r'\bsyntax\b', Keyword.Reserved),
      (r'\bfrom\b', Keyword.Namespace, 'from'),
      (r'\b(define|hidden)\b', Keyword, 'command'),
      (r'\bwhere\b', Keyword, 'where-scope'),
      (r'\bdo\b', Keyword),

      # find remaining variables
      (identifier, Name.Variable),

      (r'[;{}(),\[\]]', Punctuation),
    ],

    'number' : [
      (r'(\d+\.\d+)([eE][+-]?[0-9]+)?i?,?', Number.Float),
      (r'0b[0-1]+', Number.Bin),
      (r'0o[0-7]+', Number.Oct),
      (r'0x([a-f0-9]{1,16}|[A-Z0-9]{1,16})', Number.Hex),
      (r'([1-9][0-9]*|0)', Number.Integer),
    ],

    'string' : [
      (r'"', String.Double, '#pop'),
      (r'\\(\\|t|n|u([a-f0-9]{4,8}|[A-F0-9]{4,8}))', String.Escape),
      (r'[^\\"]+', String.Double),
    ],

    'multiline-comment' : [
      (r'#\|', Comment.Multiline, '#push'),
      (r'\|#', Comment.Multiline, '#pop'),
      (r'[^|#]+', Comment.Multiline),
      (r'[|#]', Comment.Multiline),
    ],

    'from' : [
      (r'\bimport\b', Keyword.Namespace, 'import'),
      (r'\bglob\b', Keyword.Namespace, 'glob'),
      (path, Name.Namespace),
      (r'\s+', Text)
    ],

    'command' : [
      (r'\bwhere\b', Keyword, 'where'),
      (r'\bwith\b', Keyword, 'with'),
      (r'\bdo\b', Keyword, '#pop'),
      (r';', Punctuation, '#pop'),
      (identifier, add_defined),
      (r'\s+', Text)
    ],

    'where-scope' : [
      (r'\bdo\b', Keyword, '#pop'),
      (identifier, Name.Variable),
      (r'\s+', Text),
    ],

    'where' : [
      (r'\bdo\b', Keyword, '#pop:2'),
      (identifier, Name.Variable),
      (r'\s+', Text)
    ],

    'with' : [
      (identifier, Name.Variable),
      (r';', Punctuation, '#pop:2'),
      (r'\s+', Text)
    ],

    'import' : [
      (identifier, add_builtin),
      (r'\bas\b', Keyword.Namespace),
      (r';', Punctuation, '#pop:2'),
      (r'\s+', Text) # don't error when whitespace is encountered :)
    ],

    'glob' : [
      (r';', Punctuation, '#pop:2'),
      (r'([^\[\]!?\-\*]|\s)+', Text),
      (r'[\[\]!?\-\*]', Name.Builtin),
    ]
  }
