""" Data objects in group "Plant Heating and Cooling Equipment"
"""

from collections import OrderedDict
import logging
from pyidf.helper import DataObject

logger = logging.getLogger("pyidf")
logger.addHandler(logging.NullHandler())



class BoilerHotWater(DataObject):

    """ Corresponds to IDD object `Boiler:HotWater`
        This boiler model is an adaptation of the empirical model from the Building
        Loads and System Thermodynamics (BLAST) program.  Boiler performance
        curves are generated by fitting catalog data to polynomial equations.
        A constant efficiency boiler may be modeled by leaving the normalized
        boiler efficiency curve name input blank.
    """
    schema = {'min-fields': 13,
              'name': u'Boiler:HotWater',
              'pyname': u'BoilerHotWater',
              'format': None,
              'fields': OrderedDict([(u'name',
                                      {'name': u'Name',
                                       'pyname': u'name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'alpha'}),
                                     (u'fuel type',
                                      {'name': u'Fuel Type',
                                       'pyname': u'fuel_type',
                                       'required-field': True,
                                       'autosizable': False,
                                       'accepted-values': [u'Electricity',
                                                           u'NaturalGas',
                                                           u'PropaneGas',
                                                           u'FuelOil#1',
                                                           u'FuelOil#2',
                                                           u'Coal',
                                                           u'Diesel',
                                                           u'Gasoline',
                                                           u'OtherFuel1',
                                                           u'OtherFuel2'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'nominal capacity',
                                      {'name': u'Nominal Capacity',
                                       'pyname': u'nominal_capacity',
                                       'required-field': False,
                                       'autosizable': True,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'W'}),
                                     (u'nominal thermal efficiency',
                                      {'name': u'Nominal Thermal Efficiency',
                                       'pyname': u'nominal_thermal_efficiency',
                                       'minimum>': 0.0,
                                       'maximum': 1.0,
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'efficiency curve temperature evaluation variable',
                                      {'name': u'Efficiency Curve Temperature Evaluation Variable',
                                       'pyname': u'efficiency_curve_temperature_evaluation_variable',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'EnteringBoiler',
                                                           u'LeavingBoiler'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'normalized boiler efficiency curve name',
                                      {'name': u'Normalized Boiler Efficiency Curve Name',
                                       'pyname': u'normalized_boiler_efficiency_curve_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'design water outlet temperature',
                                      {'name': u'Design Water Outlet Temperature',
                                       'pyname': u'design_water_outlet_temperature',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'C'}),
                                     (u'design water flow rate',
                                      {'name': u'Design Water Flow Rate',
                                       'pyname': u'design_water_flow_rate',
                                       'required-field': False,
                                       'autosizable': True,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'm3/s'}),
                                     (u'minimum part load ratio',
                                      {'name': u'Minimum Part Load Ratio',
                                       'pyname': u'minimum_part_load_ratio',
                                       'default': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'maximum part load ratio',
                                      {'name': u'Maximum Part Load Ratio',
                                       'pyname': u'maximum_part_load_ratio',
                                       'default': 1.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'optimum part load ratio',
                                      {'name': u'Optimum Part Load Ratio',
                                       'pyname': u'optimum_part_load_ratio',
                                       'default': 1.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'boiler water inlet node name',
                                      {'name': u'Boiler Water Inlet Node Name',
                                       'pyname': u'boiler_water_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'boiler water outlet node name',
                                      {'name': u'Boiler Water Outlet Node Name',
                                       'pyname': u'boiler_water_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'water outlet upper temperature limit',
                                      {'name': u'Water Outlet Upper Temperature Limit',
                                       'pyname': u'water_outlet_upper_temperature_limit',
                                       'default': 99.9,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'C'}),
                                     (u'boiler flow mode',
                                      {'name': u'Boiler Flow Mode',
                                       'pyname': u'boiler_flow_mode',
                                       'default': u'NotModulated',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ConstantFlow',
                                                           u'LeavingSetpointModulated',
                                                           u'NotModulated'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'parasitic electric load',
                                      {'name': u'Parasitic Electric Load',
                                       'pyname': u'parasitic_electric_load',
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'W'}),
                                     (u'sizing factor',
                                      {'name': u'Sizing Factor',
                                       'pyname': u'sizing_factor',
                                       'default': 1.0,
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'})]),
              'extensible-fields': OrderedDict(),
              'unique-object': False,
              'required-object': False,
              'group': u'Plant Heating and Cooling Equipment'}

    @property
    def name(self):
        """Get name.

        Returns:
            str: the value of `name` or None if not set

        """
        return self["Name"]

    @name.setter
    def name(self, value=None):
        """Corresponds to IDD field `Name`

        Args:
            value (str): value for IDD Field `Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Name"] = value

    @property
    def fuel_type(self):
        """Get fuel_type.

        Returns:
            str: the value of `fuel_type` or None if not set

        """
        return self["Fuel Type"]

    @fuel_type.setter
    def fuel_type(self, value=None):
        """Corresponds to IDD field `Fuel Type`

        Args:
            value (str): value for IDD Field `Fuel Type`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Fuel Type"] = value

    @property
    def nominal_capacity(self):
        """Get nominal_capacity.

        Returns:
            float: the value of `nominal_capacity` or None if not set

        """
        return self["Nominal Capacity"]

    @nominal_capacity.setter
    def nominal_capacity(self, value=None):
        """Corresponds to IDD field `Nominal Capacity`

        Args:
            value (float or "Autosize"): value for IDD Field `Nominal Capacity`
                Units: W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Nominal Capacity"] = value

    @property
    def nominal_thermal_efficiency(self):
        """Get nominal_thermal_efficiency.

        Returns:
            float: the value of `nominal_thermal_efficiency` or None if not set

        """
        return self["Nominal Thermal Efficiency"]

    @nominal_thermal_efficiency.setter
    def nominal_thermal_efficiency(self, value=None):
        """Corresponds to IDD field `Nominal Thermal Efficiency` Based on the
        higher heating value of fuel.

        Args:
            value (float): value for IDD Field `Nominal Thermal Efficiency`
                value <= 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Nominal Thermal Efficiency"] = value

    @property
    def efficiency_curve_temperature_evaluation_variable(self):
        """Get efficiency_curve_temperature_evaluation_variable.

        Returns:
            str: the value of `efficiency_curve_temperature_evaluation_variable` or None if not set

        """
        return self["Efficiency Curve Temperature Evaluation Variable"]

    @efficiency_curve_temperature_evaluation_variable.setter
    def efficiency_curve_temperature_evaluation_variable(self, value=None):
        """Corresponds to IDD field `Efficiency Curve Temperature Evaluation
        Variable`

        Args:
            value (str): value for IDD Field `Efficiency Curve Temperature Evaluation Variable`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Efficiency Curve Temperature Evaluation Variable"] = value

    @property
    def normalized_boiler_efficiency_curve_name(self):
        """Get normalized_boiler_efficiency_curve_name.

        Returns:
            str: the value of `normalized_boiler_efficiency_curve_name` or None if not set

        """
        return self["Normalized Boiler Efficiency Curve Name"]

    @normalized_boiler_efficiency_curve_name.setter
    def normalized_boiler_efficiency_curve_name(self, value=None):
        """  Corresponds to IDD field `Normalized Boiler Efficiency Curve Name`
        Linear, Quadratic and Cubic efficiency curves are solely a function of PLR.
        All other efficiency curve types are a function of PLR and boiler water temperature.
        Linear = C1 + C2*PLR
        Quadratic = C1 + C2*PLR + C3*PLR^2
        Cubic = C1 + C2*PLR + C3*PLR^2 + C4*PLR^3
        Biquadratic = C1 + C2*PLR + C3*PLR^2 + C4*T + C5*T^2 + C6*PLR*T
        QuadraticLinear = C1 + C2*PLR + C3*PLR^2 + (C4 + C5*PLR + C6*PLR^2)*T
        BiCubic = C1+C2*PLR+C3*PLR^2+C4*T+C5*T^2+C6*PLR*T+C7*PLR^3+C8*T^3+C9*PLR^2*T+C10*PLR*T^2
        TriQuadratic curves are not allowed.
        Table:OneIndependentVariable Table:TwoIndependentVariables
        PLR = part-load ratio
        T = boiler water temperature (either entering or leaving).

        Args:
            value (str): value for IDD Field `Normalized Boiler Efficiency Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Normalized Boiler Efficiency Curve Name"] = value

    @property
    def design_water_outlet_temperature(self):
        """Get design_water_outlet_temperature.

        Returns:
            float: the value of `design_water_outlet_temperature` or None if not set

        """
        return self["Design Water Outlet Temperature"]

    @design_water_outlet_temperature.setter
    def design_water_outlet_temperature(self, value=None):
        """Corresponds to IDD field `Design Water Outlet Temperature`

        Args:
            value (float): value for IDD Field `Design Water Outlet Temperature`
                Units: C
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Water Outlet Temperature"] = value

    @property
    def design_water_flow_rate(self):
        """Get design_water_flow_rate.

        Returns:
            float: the value of `design_water_flow_rate` or None if not set

        """
        return self["Design Water Flow Rate"]

    @design_water_flow_rate.setter
    def design_water_flow_rate(self, value=None):
        """Corresponds to IDD field `Design Water Flow Rate`

        Args:
            value (float or "Autosize"): value for IDD Field `Design Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Water Flow Rate"] = value

    @property
    def minimum_part_load_ratio(self):
        """Get minimum_part_load_ratio.

        Returns:
            float: the value of `minimum_part_load_ratio` or None if not set

        """
        return self["Minimum Part Load Ratio"]

    @minimum_part_load_ratio.setter
    def minimum_part_load_ratio(self, value=None):
        """Corresponds to IDD field `Minimum Part Load Ratio`

        Args:
            value (float): value for IDD Field `Minimum Part Load Ratio`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Minimum Part Load Ratio"] = value

    @property
    def maximum_part_load_ratio(self):
        """Get maximum_part_load_ratio.

        Returns:
            float: the value of `maximum_part_load_ratio` or None if not set

        """
        return self["Maximum Part Load Ratio"]

    @maximum_part_load_ratio.setter
    def maximum_part_load_ratio(self, value=1.0):
        """Corresponds to IDD field `Maximum Part Load Ratio`

        Args:
            value (float): value for IDD Field `Maximum Part Load Ratio`
                Default value: 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Maximum Part Load Ratio"] = value

    @property
    def optimum_part_load_ratio(self):
        """Get optimum_part_load_ratio.

        Returns:
            float: the value of `optimum_part_load_ratio` or None if not set

        """
        return self["Optimum Part Load Ratio"]

    @optimum_part_load_ratio.setter
    def optimum_part_load_ratio(self, value=1.0):
        """Corresponds to IDD field `Optimum Part Load Ratio`

        Args:
            value (float): value for IDD Field `Optimum Part Load Ratio`
                Default value: 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Optimum Part Load Ratio"] = value

    @property
    def boiler_water_inlet_node_name(self):
        """Get boiler_water_inlet_node_name.

        Returns:
            str: the value of `boiler_water_inlet_node_name` or None if not set

        """
        return self["Boiler Water Inlet Node Name"]

    @boiler_water_inlet_node_name.setter
    def boiler_water_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Boiler Water Inlet Node Name`

        Args:
            value (str): value for IDD Field `Boiler Water Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Boiler Water Inlet Node Name"] = value

    @property
    def boiler_water_outlet_node_name(self):
        """Get boiler_water_outlet_node_name.

        Returns:
            str: the value of `boiler_water_outlet_node_name` or None if not set

        """
        return self["Boiler Water Outlet Node Name"]

    @boiler_water_outlet_node_name.setter
    def boiler_water_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Boiler Water Outlet Node Name`

        Args:
            value (str): value for IDD Field `Boiler Water Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Boiler Water Outlet Node Name"] = value

    @property
    def water_outlet_upper_temperature_limit(self):
        """Get water_outlet_upper_temperature_limit.

        Returns:
            float: the value of `water_outlet_upper_temperature_limit` or None if not set

        """
        return self["Water Outlet Upper Temperature Limit"]

    @water_outlet_upper_temperature_limit.setter
    def water_outlet_upper_temperature_limit(self, value=99.9):
        """Corresponds to IDD field `Water Outlet Upper Temperature Limit`

        Args:
            value (float): value for IDD Field `Water Outlet Upper Temperature Limit`
                Units: C
                Default value: 99.9
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Water Outlet Upper Temperature Limit"] = value

    @property
    def boiler_flow_mode(self):
        """Get boiler_flow_mode.

        Returns:
            str: the value of `boiler_flow_mode` or None if not set

        """
        return self["Boiler Flow Mode"]

    @boiler_flow_mode.setter
    def boiler_flow_mode(self, value="NotModulated"):
        """Corresponds to IDD field `Boiler Flow Mode` Select operating mode
        for fluid flow through the boiler. "NotModulated" is for either
        variable or constant pumping with flow controlled by the external plant
        system. "ConstantFlow" is for constant pumping with flow controlled by
        boiler to operate at full design flow rate.  "LeavingSetpointModulated"
        is for variable pumping with flow controlled by boiler to vary flow to
        target a leaving temperature setpoint.

        Args:
            value (str): value for IDD Field `Boiler Flow Mode`
                Default value: NotModulated
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Boiler Flow Mode"] = value

    @property
    def parasitic_electric_load(self):
        """Get parasitic_electric_load.

        Returns:
            float: the value of `parasitic_electric_load` or None if not set

        """
        return self["Parasitic Electric Load"]

    @parasitic_electric_load.setter
    def parasitic_electric_load(self, value=None):
        """Corresponds to IDD field `Parasitic Electric Load`

        Args:
            value (float): value for IDD Field `Parasitic Electric Load`
                Units: W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Parasitic Electric Load"] = value

    @property
    def sizing_factor(self):
        """Get sizing_factor.

        Returns:
            float: the value of `sizing_factor` or None if not set

        """
        return self["Sizing Factor"]

    @sizing_factor.setter
    def sizing_factor(self, value=1.0):
        """Corresponds to IDD field `Sizing Factor` Multiplies the autosized
        capacity and flow rates.

        Args:
            value (float): value for IDD Field `Sizing Factor`
                Default value: 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Sizing Factor"] = value




class BoilerSteam(DataObject):

    """ Corresponds to IDD object `Boiler:Steam`
        This boiler model is an adaptation of the empirical model from the Building
        Loads and System Thermodynamics (BLAST) program.  Boiler performance
        curves are generated by fitting catalog data to third order
        polynomial equations.  A constant efficiency boiler is modeled by setting
        the fuel use coefficients as follows: N9=1, N10=0, N11=0
    """
    schema = {'min-fields': 0,
              'name': u'Boiler:Steam',
              'pyname': u'BoilerSteam',
              'format': None,
              'fields': OrderedDict([(u'name',
                                      {'name': u'Name',
                                       'pyname': u'name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'fuel type',
                                      {'name': u'Fuel Type',
                                       'pyname': u'fuel_type',
                                       'required-field': True,
                                       'autosizable': False,
                                       'accepted-values': [u'Electricity',
                                                           u'NaturalGas',
                                                           u'PropaneGas',
                                                           u'FuelOil#1',
                                                           u'FuelOil#2',
                                                           u'Coal',
                                                           u'Diesel',
                                                           u'Gasoline',
                                                           u'OtherFuel1',
                                                           u'OtherFuel2'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'maximum operating pressure',
                                      {'name': u'Maximum Operating Pressure',
                                       'pyname': u'maximum_operating_pressure',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'Kpa'}),
                                     (u'theoretical efficiency',
                                      {'name': u'Theoretical Efficiency',
                                       'pyname': u'theoretical_efficiency',
                                       'maximum': 1.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'design outlet steam temperature',
                                      {'name': u'Design Outlet Steam Temperature',
                                       'pyname': u'design_outlet_steam_temperature',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'C'}),
                                     (u'nominal capacity',
                                      {'name': u'Nominal Capacity',
                                       'pyname': u'nominal_capacity',
                                       'required-field': False,
                                       'autosizable': True,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'W'}),
                                     (u'minimum part load ratio',
                                      {'name': u'Minimum Part Load Ratio',
                                       'pyname': u'minimum_part_load_ratio',
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'maximum part load ratio',
                                      {'name': u'Maximum Part Load Ratio',
                                       'pyname': u'maximum_part_load_ratio',
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'optimum part load ratio',
                                      {'name': u'Optimum Part Load Ratio',
                                       'pyname': u'optimum_part_load_ratio',
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 1 of fuel use function of part load ratio curve',
                                      {'name': u'Coefficient 1 of Fuel Use Function of Part Load Ratio Curve',
                                       'pyname': u'coefficient_1_of_fuel_use_function_of_part_load_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 2 of fuel use function of part load ratio curve',
                                      {'name': u'Coefficient 2 of Fuel Use Function of Part Load Ratio Curve',
                                       'pyname': u'coefficient_2_of_fuel_use_function_of_part_load_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 3 of fuel use function of part load ratio curve',
                                      {'name': u'Coefficient 3 of Fuel Use Function of Part Load Ratio Curve',
                                       'pyname': u'coefficient_3_of_fuel_use_function_of_part_load_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'water inlet node name',
                                      {'name': u'Water Inlet Node Name',
                                       'pyname': u'water_inlet_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'steam outlet node name',
                                      {'name': u'Steam Outlet Node Name',
                                       'pyname': u'steam_outlet_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'sizing factor',
                                      {'name': u'Sizing Factor',
                                       'pyname': u'sizing_factor',
                                       'default': 1.0,
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'})]),
              'extensible-fields': OrderedDict(),
              'unique-object': False,
              'required-object': False,
              'group': u'Plant Heating and Cooling Equipment'}

    @property
    def name(self):
        """Get name.

        Returns:
            str: the value of `name` or None if not set

        """
        return self["Name"]

    @name.setter
    def name(self, value=None):
        """Corresponds to IDD field `Name`

        Args:
            value (str): value for IDD Field `Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Name"] = value

    @property
    def fuel_type(self):
        """Get fuel_type.

        Returns:
            str: the value of `fuel_type` or None if not set

        """
        return self["Fuel Type"]

    @fuel_type.setter
    def fuel_type(self, value=None):
        """Corresponds to IDD field `Fuel Type`

        Args:
            value (str): value for IDD Field `Fuel Type`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Fuel Type"] = value

    @property
    def maximum_operating_pressure(self):
        """Get maximum_operating_pressure.

        Returns:
            float: the value of `maximum_operating_pressure` or None if not set

        """
        return self["Maximum Operating Pressure"]

    @maximum_operating_pressure.setter
    def maximum_operating_pressure(self, value=None):
        """Corresponds to IDD field `Maximum Operating Pressure`

        Args:
            value (float): value for IDD Field `Maximum Operating Pressure`
                Units: Kpa
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Maximum Operating Pressure"] = value

    @property
    def theoretical_efficiency(self):
        """Get theoretical_efficiency.

        Returns:
            float: the value of `theoretical_efficiency` or None if not set

        """
        return self["Theoretical Efficiency"]

    @theoretical_efficiency.setter
    def theoretical_efficiency(self, value=None):
        """Corresponds to IDD field `Theoretical Efficiency`

        Args:
            value (float): value for IDD Field `Theoretical Efficiency`
                value <= 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Theoretical Efficiency"] = value

    @property
    def design_outlet_steam_temperature(self):
        """Get design_outlet_steam_temperature.

        Returns:
            float: the value of `design_outlet_steam_temperature` or None if not set

        """
        return self["Design Outlet Steam Temperature"]

    @design_outlet_steam_temperature.setter
    def design_outlet_steam_temperature(self, value=None):
        """Corresponds to IDD field `Design Outlet Steam Temperature`

        Args:
            value (float): value for IDD Field `Design Outlet Steam Temperature`
                Units: C
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Outlet Steam Temperature"] = value

    @property
    def nominal_capacity(self):
        """Get nominal_capacity.

        Returns:
            float: the value of `nominal_capacity` or None if not set

        """
        return self["Nominal Capacity"]

    @nominal_capacity.setter
    def nominal_capacity(self, value=None):
        """Corresponds to IDD field `Nominal Capacity`

        Args:
            value (float or "Autosize"): value for IDD Field `Nominal Capacity`
                Units: W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Nominal Capacity"] = value

    @property
    def minimum_part_load_ratio(self):
        """Get minimum_part_load_ratio.

        Returns:
            float: the value of `minimum_part_load_ratio` or None if not set

        """
        return self["Minimum Part Load Ratio"]

    @minimum_part_load_ratio.setter
    def minimum_part_load_ratio(self, value=None):
        """Corresponds to IDD field `Minimum Part Load Ratio`

        Args:
            value (float): value for IDD Field `Minimum Part Load Ratio`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Minimum Part Load Ratio"] = value

    @property
    def maximum_part_load_ratio(self):
        """Get maximum_part_load_ratio.

        Returns:
            float: the value of `maximum_part_load_ratio` or None if not set

        """
        return self["Maximum Part Load Ratio"]

    @maximum_part_load_ratio.setter
    def maximum_part_load_ratio(self, value=None):
        """Corresponds to IDD field `Maximum Part Load Ratio`

        Args:
            value (float): value for IDD Field `Maximum Part Load Ratio`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Maximum Part Load Ratio"] = value

    @property
    def optimum_part_load_ratio(self):
        """Get optimum_part_load_ratio.

        Returns:
            float: the value of `optimum_part_load_ratio` or None if not set

        """
        return self["Optimum Part Load Ratio"]

    @optimum_part_load_ratio.setter
    def optimum_part_load_ratio(self, value=None):
        """Corresponds to IDD field `Optimum Part Load Ratio`

        Args:
            value (float): value for IDD Field `Optimum Part Load Ratio`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Optimum Part Load Ratio"] = value

    @property
    def coefficient_1_of_fuel_use_function_of_part_load_ratio_curve(self):
        """Get coefficient_1_of_fuel_use_function_of_part_load_ratio_curve.

        Returns:
            float: the value of `coefficient_1_of_fuel_use_function_of_part_load_ratio_curve` or None if not set

        """
        return self[
            "Coefficient 1 of Fuel Use Function of Part Load Ratio Curve"]

    @coefficient_1_of_fuel_use_function_of_part_load_ratio_curve.setter
    def coefficient_1_of_fuel_use_function_of_part_load_ratio_curve(
            self,
            value=None):
        """Corresponds to IDD field `Coefficient 1 of Fuel Use Function of Part
        Load Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 1 of Fuel Use Function of Part Load Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Coefficient 1 of Fuel Use Function of Part Load Ratio Curve"] = value

    @property
    def coefficient_2_of_fuel_use_function_of_part_load_ratio_curve(self):
        """Get coefficient_2_of_fuel_use_function_of_part_load_ratio_curve.

        Returns:
            float: the value of `coefficient_2_of_fuel_use_function_of_part_load_ratio_curve` or None if not set

        """
        return self[
            "Coefficient 2 of Fuel Use Function of Part Load Ratio Curve"]

    @coefficient_2_of_fuel_use_function_of_part_load_ratio_curve.setter
    def coefficient_2_of_fuel_use_function_of_part_load_ratio_curve(
            self,
            value=None):
        """Corresponds to IDD field `Coefficient 2 of Fuel Use Function of Part
        Load Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 2 of Fuel Use Function of Part Load Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Coefficient 2 of Fuel Use Function of Part Load Ratio Curve"] = value

    @property
    def coefficient_3_of_fuel_use_function_of_part_load_ratio_curve(self):
        """Get coefficient_3_of_fuel_use_function_of_part_load_ratio_curve.

        Returns:
            float: the value of `coefficient_3_of_fuel_use_function_of_part_load_ratio_curve` or None if not set

        """
        return self[
            "Coefficient 3 of Fuel Use Function of Part Load Ratio Curve"]

    @coefficient_3_of_fuel_use_function_of_part_load_ratio_curve.setter
    def coefficient_3_of_fuel_use_function_of_part_load_ratio_curve(
            self,
            value=None):
        """Corresponds to IDD field `Coefficient 3 of Fuel Use Function of Part
        Load Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 3 of Fuel Use Function of Part Load Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Coefficient 3 of Fuel Use Function of Part Load Ratio Curve"] = value

    @property
    def water_inlet_node_name(self):
        """Get water_inlet_node_name.

        Returns:
            str: the value of `water_inlet_node_name` or None if not set

        """
        return self["Water Inlet Node Name"]

    @water_inlet_node_name.setter
    def water_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Water Inlet Node Name`

        Args:
            value (str): value for IDD Field `Water Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Water Inlet Node Name"] = value

    @property
    def steam_outlet_node_name(self):
        """Get steam_outlet_node_name.

        Returns:
            str: the value of `steam_outlet_node_name` or None if not set

        """
        return self["Steam Outlet Node Name"]

    @steam_outlet_node_name.setter
    def steam_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Steam Outlet Node Name`

        Args:
            value (str): value for IDD Field `Steam Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Steam Outlet Node Name"] = value

    @property
    def sizing_factor(self):
        """Get sizing_factor.

        Returns:
            float: the value of `sizing_factor` or None if not set

        """
        return self["Sizing Factor"]

    @sizing_factor.setter
    def sizing_factor(self, value=1.0):
        """Corresponds to IDD field `Sizing Factor` Multiplies the autosized
        capacity and flow rates.

        Args:
            value (float): value for IDD Field `Sizing Factor`
                Default value: 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Sizing Factor"] = value




class ChillerElectricEir(DataObject):

    """ Corresponds to IDD object `Chiller:Electric:EIR`
        This chiller model is the empirical model from the DOE-2 building Energy
        simulation program. Chiller performance at off-reference conditions is modeled
        using three polynomial equations. Three curves objects are required.
    """
    schema = {'min-fields': 23,
              'name': u'Chiller:Electric:EIR',
              'pyname': u'ChillerElectricEir',
              'format': None,
              'fields': OrderedDict([(u'name',
                                      {'name': u'Name',
                                       'pyname': u'name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'alpha'}),
                                     (u'reference capacity',
                                      {'name': u'Reference Capacity',
                                       'pyname': u'reference_capacity',
                                       'minimum>': 0.0,
                                       'required-field': True,
                                       'autosizable': True,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'W'}),
                                     (u'reference cop',
                                      {'name': u'Reference COP',
                                       'pyname': u'reference_cop',
                                       'minimum>': 0.0,
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'W/W'}),
                                     (u'reference leaving chilled water temperature',
                                      {'name': u'Reference Leaving Chilled Water Temperature',
                                       'pyname': u'reference_leaving_chilled_water_temperature',
                                       'default': 6.67,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'C'}),
                                     (u'reference entering condenser fluid temperature',
                                      {'name': u'Reference Entering Condenser Fluid Temperature',
                                       'pyname': u'reference_entering_condenser_fluid_temperature',
                                       'default': 29.4,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'C'}),
                                     (u'reference chilled water flow rate',
                                      {'name': u'Reference Chilled Water Flow Rate',
                                       'pyname': u'reference_chilled_water_flow_rate',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': True,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'm3/s'}),
                                     (u'reference condenser fluid flow rate',
                                      {'name': u'Reference Condenser Fluid Flow Rate',
                                       'pyname': u'reference_condenser_fluid_flow_rate',
                                       'required-field': False,
                                       'autosizable': True,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'm3/s'}),
                                     (u'cooling capacity function of temperature curve name',
                                      {'name': u'Cooling Capacity Function of Temperature Curve Name',
                                       'pyname': u'cooling_capacity_function_of_temperature_curve_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'electric input to cooling output ratio function of temperature curve name',
                                      {'name': u'Electric Input to Cooling Output Ratio Function of Temperature Curve Name',
                                       'pyname': u'electric_input_to_cooling_output_ratio_function_of_temperature_curve_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'electric input to cooling output ratio function of part load ratio curve name',
                                      {'name': u'Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name',
                                       'pyname': u'electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'minimum part load ratio',
                                      {'name': u'Minimum Part Load Ratio',
                                       'pyname': u'minimum_part_load_ratio',
                                       'default': 0.1,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'maximum part load ratio',
                                      {'name': u'Maximum Part Load Ratio',
                                       'pyname': u'maximum_part_load_ratio',
                                       'default': 1.0,
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'optimum part load ratio',
                                      {'name': u'Optimum Part Load Ratio',
                                       'pyname': u'optimum_part_load_ratio',
                                       'default': 1.0,
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'minimum unloading ratio',
                                      {'name': u'Minimum Unloading Ratio',
                                       'pyname': u'minimum_unloading_ratio',
                                       'default': 0.2,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'chilled water inlet node name',
                                      {'name': u'Chilled Water Inlet Node Name',
                                       'pyname': u'chilled_water_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'chilled water outlet node name',
                                      {'name': u'Chilled Water Outlet Node Name',
                                       'pyname': u'chilled_water_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'condenser inlet node name',
                                      {'name': u'Condenser Inlet Node Name',
                                       'pyname': u'condenser_inlet_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'condenser outlet node name',
                                      {'name': u'Condenser Outlet Node Name',
                                       'pyname': u'condenser_outlet_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'condenser type',
                                      {'name': u'Condenser Type',
                                       'pyname': u'condenser_type',
                                       'default': u'WaterCooled',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'AirCooled',
                                                           u'WaterCooled',
                                                           u'EvaporativelyCooled'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'condenser fan power ratio',
                                      {'name': u'Condenser Fan Power Ratio',
                                       'pyname': u'condenser_fan_power_ratio',
                                       'default': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'W/W'}),
                                     (u'fraction of compressor electric consumption rejected by condenser',
                                      {'name': u'Fraction of Compressor Electric Consumption Rejected by Condenser',
                                       'pyname': u'fraction_of_compressor_electric_consumption_rejected_by_condenser',
                                       'default': 1.0,
                                       'minimum>': 0.0,
                                       'maximum': 1.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'leaving chilled water lower temperature limit',
                                      {'name': u'Leaving Chilled Water Lower Temperature Limit',
                                       'pyname': u'leaving_chilled_water_lower_temperature_limit',
                                       'default': 2.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'C'}),
                                     (u'chiller flow mode',
                                      {'name': u'Chiller Flow Mode',
                                       'pyname': u'chiller_flow_mode',
                                       'default': u'NotModulated',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ConstantFlow',
                                                           u'LeavingSetpointModulated',
                                                           u'NotModulated'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'design heat recovery water flow rate',
                                      {'name': u'Design Heat Recovery Water Flow Rate',
                                       'pyname': u'design_heat_recovery_water_flow_rate',
                                       'default': 0.0,
                                       'required-field': False,
                                       'autosizable': True,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'm3/s'}),
                                     (u'heat recovery inlet node name',
                                      {'name': u'Heat Recovery Inlet Node Name',
                                       'pyname': u'heat_recovery_inlet_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'heat recovery outlet node name',
                                      {'name': u'Heat Recovery Outlet Node Name',
                                       'pyname': u'heat_recovery_outlet_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'sizing factor',
                                      {'name': u'Sizing Factor',
                                       'pyname': u'sizing_factor',
                                       'default': 1.0,
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'basin heater capacity',
                                      {'name': u'Basin Heater Capacity',
                                       'pyname': u'basin_heater_capacity',
                                       'default': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'W/K'}),
                                     (u'basin heater setpoint temperature',
                                      {'name': u'Basin Heater Setpoint Temperature',
                                       'pyname': u'basin_heater_setpoint_temperature',
                                       'default': 2.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 2.0,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'C'}),
                                     (u'basin heater operating schedule name',
                                      {'name': u'Basin Heater Operating Schedule Name',
                                       'pyname': u'basin_heater_operating_schedule_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'condenser heat recovery relative capacity fraction',
                                      {'name': u'Condenser Heat Recovery Relative Capacity Fraction',
                                       'pyname': u'condenser_heat_recovery_relative_capacity_fraction',
                                       'maximum': 1.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'heat recovery inlet high temperature limit schedule name',
                                      {'name': u'Heat Recovery Inlet High Temperature Limit Schedule Name',
                                       'pyname': u'heat_recovery_inlet_high_temperature_limit_schedule_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'heat recovery leaving temperature setpoint node name',
                                      {'name': u'Heat Recovery Leaving Temperature Setpoint Node Name',
                                       'pyname': u'heat_recovery_leaving_temperature_setpoint_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'})]),
              'extensible-fields': OrderedDict(),
              'unique-object': False,
              'required-object': False,
              'group': u'Plant Heating and Cooling Equipment'}

    @property
    def name(self):
        """Get name.

        Returns:
            str: the value of `name` or None if not set

        """
        return self["Name"]

    @name.setter
    def name(self, value=None):
        """Corresponds to IDD field `Name`

        Args:
            value (str): value for IDD Field `Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Name"] = value

    @property
    def reference_capacity(self):
        """Get reference_capacity.

        Returns:
            float: the value of `reference_capacity` or None if not set

        """
        return self["Reference Capacity"]

    @reference_capacity.setter
    def reference_capacity(self, value=None):
        """Corresponds to IDD field `Reference Capacity`

        Args:
            value (float or "Autosize"): value for IDD Field `Reference Capacity`
                Units: W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Reference Capacity"] = value

    @property
    def reference_cop(self):
        """Get reference_cop.

        Returns:
            float: the value of `reference_cop` or None if not set

        """
        return self["Reference COP"]

    @reference_cop.setter
    def reference_cop(self, value=None):
        """Corresponds to IDD field `Reference COP` Efficiency of the chiller
        compressor (cooling output/compressor energy input). Condenser fan
        power should not be included here.

        Args:
            value (float): value for IDD Field `Reference COP`
                Units: W/W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Reference COP"] = value

    @property
    def reference_leaving_chilled_water_temperature(self):
        """Get reference_leaving_chilled_water_temperature.

        Returns:
            float: the value of `reference_leaving_chilled_water_temperature` or None if not set

        """
        return self["Reference Leaving Chilled Water Temperature"]

    @reference_leaving_chilled_water_temperature.setter
    def reference_leaving_chilled_water_temperature(self, value=6.67):
        """Corresponds to IDD field `Reference Leaving Chilled Water
        Temperature`

        Args:
            value (float): value for IDD Field `Reference Leaving Chilled Water Temperature`
                Units: C
                Default value: 6.67
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Reference Leaving Chilled Water Temperature"] = value

    @property
    def reference_entering_condenser_fluid_temperature(self):
        """Get reference_entering_condenser_fluid_temperature.

        Returns:
            float: the value of `reference_entering_condenser_fluid_temperature` or None if not set

        """
        return self["Reference Entering Condenser Fluid Temperature"]

    @reference_entering_condenser_fluid_temperature.setter
    def reference_entering_condenser_fluid_temperature(self, value=29.4):
        """Corresponds to IDD field `Reference Entering Condenser Fluid
        Temperature`

        Args:
            value (float): value for IDD Field `Reference Entering Condenser Fluid Temperature`
                Units: C
                Default value: 29.4
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Reference Entering Condenser Fluid Temperature"] = value

    @property
    def reference_chilled_water_flow_rate(self):
        """Get reference_chilled_water_flow_rate.

        Returns:
            float: the value of `reference_chilled_water_flow_rate` or None if not set

        """
        return self["Reference Chilled Water Flow Rate"]

    @reference_chilled_water_flow_rate.setter
    def reference_chilled_water_flow_rate(self, value=None):
        """Corresponds to IDD field `Reference Chilled Water Flow Rate`

        Args:
            value (float or "Autosize"): value for IDD Field `Reference Chilled Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Reference Chilled Water Flow Rate"] = value

    @property
    def reference_condenser_fluid_flow_rate(self):
        """Get reference_condenser_fluid_flow_rate.

        Returns:
            float: the value of `reference_condenser_fluid_flow_rate` or None if not set

        """
        return self["Reference Condenser Fluid Flow Rate"]

    @reference_condenser_fluid_flow_rate.setter
    def reference_condenser_fluid_flow_rate(self, value=None):
        """  Corresponds to IDD field `Reference Condenser Fluid Flow Rate`
        This field is only used for Condenser Type = AirCooled or EvaporativelyCooled
        when Heat Recovery is specified

        Args:
            value (float or "Autosize"): value for IDD Field `Reference Condenser Fluid Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Reference Condenser Fluid Flow Rate"] = value

    @property
    def cooling_capacity_function_of_temperature_curve_name(self):
        """Get cooling_capacity_function_of_temperature_curve_name.

        Returns:
            str: the value of `cooling_capacity_function_of_temperature_curve_name` or None if not set

        """
        return self["Cooling Capacity Function of Temperature Curve Name"]

    @cooling_capacity_function_of_temperature_curve_name.setter
    def cooling_capacity_function_of_temperature_curve_name(self, value=None):
        """  Corresponds to IDD field `Cooling Capacity Function of Temperature Curve Name`
        Cooling capacity as a function of CW supply temp and entering condenser temp
        Table:TwoIndependentVariables object can also be used
        curve = a + b*CWS + c*CWS**2 + d*ECT + e*ECT**2 + f*CWS*ECT
        CWS = supply (leaving) chilled water temperature(C)
        ECT = entering condenser fluid temperature(C)

        Args:
            value (str): value for IDD Field `Cooling Capacity Function of Temperature Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Cooling Capacity Function of Temperature Curve Name"] = value

    @property
    def electric_input_to_cooling_output_ratio_function_of_temperature_curve_name(
            self):
        """Get electric_input_to_cooling_output_ratio_function_of_temperature_c
        urve_name.

        Returns:
            str: the value of `electric_input_to_cooling_output_ratio_function_of_temperature_curve_name` or None if not set

        """
        return self[
            "Electric Input to Cooling Output Ratio Function of Temperature Curve Name"]

    @electric_input_to_cooling_output_ratio_function_of_temperature_curve_name.setter
    def electric_input_to_cooling_output_ratio_function_of_temperature_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Electric Input to Cooling Output Ratio Function of Temperature Curve Name`
        Electric Input Ratio (EIR) as a function of temperature
        EIR = 1/COP
        Table:TwoIndependentVariables object can also be used
        curve = a + b*CWS + c*CWS**2 + d*ECT + e*ECT**2 + f*CWS*ECT
        CWS = supply (leaving) chilled water temperature(C)
        ECT = entering condenser fluid temperature(C)

        Args:
            value (str): value for IDD Field `Electric Input to Cooling Output Ratio Function of Temperature Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Electric Input to Cooling Output Ratio Function of Temperature Curve Name"] = value

    @property
    def electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name(
            self):
        """Get electric_input_to_cooling_output_ratio_function_of_part_load_rat
        io_curve_name.

        Returns:
            str: the value of `electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name` or None if not set

        """
        return self[
            "Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name"]

    @electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name.setter
    def electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name`
        Electric Input Ratio (EIR) as a function of Part Load Ratio (PLR)
        EIR = 1/COP
        Table:OneIndependentVariable object can also be used
        quadratic curve = a + b*PLR + c*PLR**2
        PLR = part load ratio (cooling load/steady state capacity)

        Args:
            value (str): value for IDD Field `Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name"] = value

    @property
    def minimum_part_load_ratio(self):
        """Get minimum_part_load_ratio.

        Returns:
            float: the value of `minimum_part_load_ratio` or None if not set

        """
        return self["Minimum Part Load Ratio"]

    @minimum_part_load_ratio.setter
    def minimum_part_load_ratio(self, value=0.1):
        """Corresponds to IDD field `Minimum Part Load Ratio` Part load ratio
        below which the chiller starts cycling on/off to meet the load. Must be
        less than or equal to Maximum Part Load Ratio.

        Args:
            value (float): value for IDD Field `Minimum Part Load Ratio`
                Default value: 0.1
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Minimum Part Load Ratio"] = value

    @property
    def maximum_part_load_ratio(self):
        """Get maximum_part_load_ratio.

        Returns:
            float: the value of `maximum_part_load_ratio` or None if not set

        """
        return self["Maximum Part Load Ratio"]

    @maximum_part_load_ratio.setter
    def maximum_part_load_ratio(self, value=1.0):
        """Corresponds to IDD field `Maximum Part Load Ratio` Maximum allowable
        part load ratio. Must be greater than or equal to Minimum Part Load
        Ratio.

        Args:
            value (float): value for IDD Field `Maximum Part Load Ratio`
                Default value: 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Maximum Part Load Ratio"] = value

    @property
    def optimum_part_load_ratio(self):
        """Get optimum_part_load_ratio.

        Returns:
            float: the value of `optimum_part_load_ratio` or None if not set

        """
        return self["Optimum Part Load Ratio"]

    @optimum_part_load_ratio.setter
    def optimum_part_load_ratio(self, value=1.0):
        """Corresponds to IDD field `Optimum Part Load Ratio` Optimum part load
        ratio where the chiller is most efficient. Must be greater than or
        equal to the Minimum Part Load Ratio and less than or equal to the
        Maximum Part Load Ratio.

        Args:
            value (float): value for IDD Field `Optimum Part Load Ratio`
                Default value: 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Optimum Part Load Ratio"] = value

    @property
    def minimum_unloading_ratio(self):
        """Get minimum_unloading_ratio.

        Returns:
            float: the value of `minimum_unloading_ratio` or None if not set

        """
        return self["Minimum Unloading Ratio"]

    @minimum_unloading_ratio.setter
    def minimum_unloading_ratio(self, value=0.2):
        """Corresponds to IDD field `Minimum Unloading Ratio` Part load ratio
        where the chiller can no longer unload and false loading begins.
        Minimum unloading ratio must be greater than or equal to the Minimum
        Part Load Ratio and less than or equal to the Maximum Part Load Ratio.

        Args:
            value (float): value for IDD Field `Minimum Unloading Ratio`
                Default value: 0.2
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Minimum Unloading Ratio"] = value

    @property
    def chilled_water_inlet_node_name(self):
        """Get chilled_water_inlet_node_name.

        Returns:
            str: the value of `chilled_water_inlet_node_name` or None if not set

        """
        return self["Chilled Water Inlet Node Name"]

    @chilled_water_inlet_node_name.setter
    def chilled_water_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Chilled Water Inlet Node Name`

        Args:
            value (str): value for IDD Field `Chilled Water Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Inlet Node Name"] = value

    @property
    def chilled_water_outlet_node_name(self):
        """Get chilled_water_outlet_node_name.

        Returns:
            str: the value of `chilled_water_outlet_node_name` or None if not set

        """
        return self["Chilled Water Outlet Node Name"]

    @chilled_water_outlet_node_name.setter
    def chilled_water_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Chilled Water Outlet Node Name`

        Args:
            value (str): value for IDD Field `Chilled Water Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Outlet Node Name"] = value

    @property
    def condenser_inlet_node_name(self):
        """Get condenser_inlet_node_name.

        Returns:
            str: the value of `condenser_inlet_node_name` or None if not set

        """
        return self["Condenser Inlet Node Name"]

    @condenser_inlet_node_name.setter
    def condenser_inlet_node_name(self, value=None):
        """  Corresponds to IDD field `Condenser Inlet Node Name`
        Not required if air-cooled or evaporativly-cooled

        Args:
            value (str): value for IDD Field `Condenser Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Condenser Inlet Node Name"] = value

    @property
    def condenser_outlet_node_name(self):
        """Get condenser_outlet_node_name.

        Returns:
            str: the value of `condenser_outlet_node_name` or None if not set

        """
        return self["Condenser Outlet Node Name"]

    @condenser_outlet_node_name.setter
    def condenser_outlet_node_name(self, value=None):
        """  Corresponds to IDD field `Condenser Outlet Node Name`
        Not required if air-cooled or evaporatively-cooled

        Args:
            value (str): value for IDD Field `Condenser Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Condenser Outlet Node Name"] = value

    @property
    def condenser_type(self):
        """Get condenser_type.

        Returns:
            str: the value of `condenser_type` or None if not set

        """
        return self["Condenser Type"]

    @condenser_type.setter
    def condenser_type(self, value="WaterCooled"):
        """Corresponds to IDD field `Condenser Type`

        Args:
            value (str): value for IDD Field `Condenser Type`
                Default value: WaterCooled
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Type"] = value

    @property
    def condenser_fan_power_ratio(self):
        """Get condenser_fan_power_ratio.

        Returns:
            float: the value of `condenser_fan_power_ratio` or None if not set

        """
        return self["Condenser Fan Power Ratio"]

    @condenser_fan_power_ratio.setter
    def condenser_fan_power_ratio(self, value=None):
        """  Corresponds to IDD field `Condenser Fan Power Ratio`
        Use for air-cooled or evaporatively-cooled condensers.
        Ratio of condenser fan power to reference chiller capacity

        Args:
            value (float): value for IDD Field `Condenser Fan Power Ratio`
                Units: W/W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Condenser Fan Power Ratio"] = value

    @property
    def fraction_of_compressor_electric_consumption_rejected_by_condenser(
            self):
        """Get
        fraction_of_compressor_electric_consumption_rejected_by_condenser.

        Returns:
            float: the value of `fraction_of_compressor_electric_consumption_rejected_by_condenser` or None if not set

        """
        return self[
            "Fraction of Compressor Electric Consumption Rejected by Condenser"]

    @fraction_of_compressor_electric_consumption_rejected_by_condenser.setter
    def fraction_of_compressor_electric_consumption_rejected_by_condenser(
            self,
            value=1.0):
        """  Corresponds to IDD field `Fraction of Compressor Electric Consumption Rejected by Condenser`
        Fraction of compressor electrical energy that must be rejected by the condenser.
        Enter a value of 1.0 when modeling hermetic chillers.
        For open chillers, enter the compressor motor efficiency.
        This value should be greater than 0.6 for praticle applications.

        Args:
            value (float): value for IDD Field `Fraction of Compressor Electric Consumption Rejected by Condenser`
                Default value: 1.0
                value <= 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Fraction of Compressor Electric Consumption Rejected by Condenser"] = value

    @property
    def leaving_chilled_water_lower_temperature_limit(self):
        """Get leaving_chilled_water_lower_temperature_limit.

        Returns:
            float: the value of `leaving_chilled_water_lower_temperature_limit` or None if not set

        """
        return self["Leaving Chilled Water Lower Temperature Limit"]

    @leaving_chilled_water_lower_temperature_limit.setter
    def leaving_chilled_water_lower_temperature_limit(self, value=2.0):
        """Corresponds to IDD field `Leaving Chilled Water Lower Temperature
        Limit`

        Args:
            value (float): value for IDD Field `Leaving Chilled Water Lower Temperature Limit`
                Units: C
                Default value: 2.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Leaving Chilled Water Lower Temperature Limit"] = value

    @property
    def chiller_flow_mode(self):
        """Get chiller_flow_mode.

        Returns:
            str: the value of `chiller_flow_mode` or None if not set

        """
        return self["Chiller Flow Mode"]

    @chiller_flow_mode.setter
    def chiller_flow_mode(self, value="NotModulated"):
        """Corresponds to IDD field `Chiller Flow Mode` Select operating mode
        for fluid flow through the chiller. "NotModulated" is for either
        variable or constant pumping with flow controlled by the external plant
        system. "ConstantFlow" is for constant pumping with flow controlled by
        chiller to operate at full design flow rate.
        "LeavingSetpointModulated" is for variable pumping with flow controlled
        by chiller to vary flow to target a leaving temperature setpoint.

        Args:
            value (str): value for IDD Field `Chiller Flow Mode`
                Default value: NotModulated
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Flow Mode"] = value

    @property
    def design_heat_recovery_water_flow_rate(self):
        """Get design_heat_recovery_water_flow_rate.

        Returns:
            float: the value of `design_heat_recovery_water_flow_rate` or None if not set

        """
        return self["Design Heat Recovery Water Flow Rate"]

    @design_heat_recovery_water_flow_rate.setter
    def design_heat_recovery_water_flow_rate(self, value=None):
        """  Corresponds to IDD field `Design Heat Recovery Water Flow Rate`
        If non-zero, then the heat recovery inlet and outlet node names must be entered.

        Args:
            value (float or "Autosize"): value for IDD Field `Design Heat Recovery Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Design Heat Recovery Water Flow Rate"] = value

    @property
    def heat_recovery_inlet_node_name(self):
        """Get heat_recovery_inlet_node_name.

        Returns:
            str: the value of `heat_recovery_inlet_node_name` or None if not set

        """
        return self["Heat Recovery Inlet Node Name"]

    @heat_recovery_inlet_node_name.setter
    def heat_recovery_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Heat Recovery Inlet Node Name`

        Args:
            value (str): value for IDD Field `Heat Recovery Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Heat Recovery Inlet Node Name"] = value

    @property
    def heat_recovery_outlet_node_name(self):
        """Get heat_recovery_outlet_node_name.

        Returns:
            str: the value of `heat_recovery_outlet_node_name` or None if not set

        """
        return self["Heat Recovery Outlet Node Name"]

    @heat_recovery_outlet_node_name.setter
    def heat_recovery_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Heat Recovery Outlet Node Name`

        Args:
            value (str): value for IDD Field `Heat Recovery Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Heat Recovery Outlet Node Name"] = value

    @property
    def sizing_factor(self):
        """Get sizing_factor.

        Returns:
            float: the value of `sizing_factor` or None if not set

        """
        return self["Sizing Factor"]

    @sizing_factor.setter
    def sizing_factor(self, value=1.0):
        """Corresponds to IDD field `Sizing Factor` Multiplies the autosized
        capacity and flow rates.

        Args:
            value (float): value for IDD Field `Sizing Factor`
                Default value: 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Sizing Factor"] = value

    @property
    def basin_heater_capacity(self):
        """Get basin_heater_capacity.

        Returns:
            float: the value of `basin_heater_capacity` or None if not set

        """
        return self["Basin Heater Capacity"]

    @basin_heater_capacity.setter
    def basin_heater_capacity(self, value=None):
        """  Corresponds to IDD field `Basin Heater Capacity`
        This field is only used for Condenser Type = EvaporativelyCooled and for periods
        when the basin heater is available (field Basin Heater Operating Schedule Name).
        For this situation, the heater maintains the basin water temperature at the basin heater
        setpoint temperature when the outdoor air temperature falls below the setpoint temperature.
        The basin heater only operates when the chiller is not operating.

        Args:
            value (float): value for IDD Field `Basin Heater Capacity`
                Units: W/K
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Basin Heater Capacity"] = value

    @property
    def basin_heater_setpoint_temperature(self):
        """Get basin_heater_setpoint_temperature.

        Returns:
            float: the value of `basin_heater_setpoint_temperature` or None if not set

        """
        return self["Basin Heater Setpoint Temperature"]

    @basin_heater_setpoint_temperature.setter
    def basin_heater_setpoint_temperature(self, value=2.0):
        """  Corresponds to IDD field `Basin Heater Setpoint Temperature`
        This field is only used for Condenser Type = EvaporativelyCooled.
        Enter the outdoor dry-bulb temperature when the basin heater turns on.

        Args:
            value (float): value for IDD Field `Basin Heater Setpoint Temperature`
                Units: C
                Default value: 2.0
                value >= 2.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Basin Heater Setpoint Temperature"] = value

    @property
    def basin_heater_operating_schedule_name(self):
        """Get basin_heater_operating_schedule_name.

        Returns:
            str: the value of `basin_heater_operating_schedule_name` or None if not set

        """
        return self["Basin Heater Operating Schedule Name"]

    @basin_heater_operating_schedule_name.setter
    def basin_heater_operating_schedule_name(self, value=None):
        """  Corresponds to IDD field `Basin Heater Operating Schedule Name`
        This field is only used for Condenser Type = EvaporativelyCooled.
        Schedule values greater than 0 allow the basin heater to operate whenever the outdoor
        air dry-bulb temperature is below the basin heater setpoint temperature.
        If a schedule name is not entered, the basin heater is allowed to operate
        throughout the entire simulation.

        Args:
            value (str): value for IDD Field `Basin Heater Operating Schedule Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Basin Heater Operating Schedule Name"] = value

    @property
    def condenser_heat_recovery_relative_capacity_fraction(self):
        """Get condenser_heat_recovery_relative_capacity_fraction.

        Returns:
            float: the value of `condenser_heat_recovery_relative_capacity_fraction` or None if not set

        """
        return self["Condenser Heat Recovery Relative Capacity Fraction"]

    @condenser_heat_recovery_relative_capacity_fraction.setter
    def condenser_heat_recovery_relative_capacity_fraction(self, value=None):
        """Corresponds to IDD field `Condenser Heat Recovery Relative Capacity
        Fraction` This optional field is the fraction of total rejected heat
        that can be recovered at full load.

        Args:
            value (float): value for IDD Field `Condenser Heat Recovery Relative Capacity Fraction`
                value <= 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Heat Recovery Relative Capacity Fraction"] = value

    @property
    def heat_recovery_inlet_high_temperature_limit_schedule_name(self):
        """Get heat_recovery_inlet_high_temperature_limit_schedule_name.

        Returns:
            str: the value of `heat_recovery_inlet_high_temperature_limit_schedule_name` or None if not set

        """
        return self["Heat Recovery Inlet High Temperature Limit Schedule Name"]

    @heat_recovery_inlet_high_temperature_limit_schedule_name.setter
    def heat_recovery_inlet_high_temperature_limit_schedule_name(
            self,
            value=None):
        """Corresponds to IDD field `Heat Recovery Inlet High Temperature Limit
        Schedule Name` This optional schedule of temperatures will turn off
        heat recovery if inlet exceeds the value.

        Args:
            value (str): value for IDD Field `Heat Recovery Inlet High Temperature Limit Schedule Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Heat Recovery Inlet High Temperature Limit Schedule Name"] = value

    @property
    def heat_recovery_leaving_temperature_setpoint_node_name(self):
        """Get heat_recovery_leaving_temperature_setpoint_node_name.

        Returns:
            str: the value of `heat_recovery_leaving_temperature_setpoint_node_name` or None if not set

        """
        return self["Heat Recovery Leaving Temperature Setpoint Node Name"]

    @heat_recovery_leaving_temperature_setpoint_node_name.setter
    def heat_recovery_leaving_temperature_setpoint_node_name(self, value=None):
        """Corresponds to IDD field `Heat Recovery Leaving Temperature Setpoint
        Node Name` This optional field provides control over the heat recovery
        Using this triggers a model more suited to series bundle and chillers
        with higher temperature heat recovery If this field is not used, the
        bundles are modeled as being in parallel.

        Args:
            value (str): value for IDD Field `Heat Recovery Leaving Temperature Setpoint Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Heat Recovery Leaving Temperature Setpoint Node Name"] = value




class ChillerElectricReformulatedEir(DataObject):

    """ Corresponds to IDD object `Chiller:Electric:ReformulatedEIR`
        This chiller model is an empirical model, a reformulated version of Chiller:Electric:EIR
        where the performance is a function of condenser leaving fluid Temperature instead of
        condenser entering fluid Temperature. Chiller performance at off-reference conditions is
        modeled using three polynomial equations. Three curve objects are required.
    """
    schema = {'min-fields': 21,
              'name': u'Chiller:Electric:ReformulatedEIR',
              'pyname': u'ChillerElectricReformulatedEir',
              'format': None,
              'fields': OrderedDict([(u'name',
                                      {'name': u'Name',
                                       'pyname': u'name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'alpha'}),
                                     (u'reference capacity',
                                      {'name': u'Reference Capacity',
                                       'pyname': u'reference_capacity',
                                       'minimum>': 0.0,
                                       'required-field': True,
                                       'autosizable': True,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'W'}),
                                     (u'reference cop',
                                      {'name': u'Reference COP',
                                       'pyname': u'reference_cop',
                                       'minimum>': 0.0,
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'W/W'}),
                                     (u'reference leaving chilled water temperature',
                                      {'name': u'Reference Leaving Chilled Water Temperature',
                                       'pyname': u'reference_leaving_chilled_water_temperature',
                                       'default': 6.67,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'C'}),
                                     (u'reference leaving condenser water temperature',
                                      {'name': u'Reference Leaving Condenser Water Temperature',
                                       'pyname': u'reference_leaving_condenser_water_temperature',
                                       'default': 35.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'C'}),
                                     (u'reference chilled water flow rate',
                                      {'name': u'Reference Chilled Water Flow Rate',
                                       'pyname': u'reference_chilled_water_flow_rate',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': True,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'm3/s'}),
                                     (u'reference condenser water flow rate',
                                      {'name': u'Reference Condenser Water Flow Rate',
                                       'pyname': u'reference_condenser_water_flow_rate',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': True,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'm3/s'}),
                                     (u'cooling capacity function of temperature curve name',
                                      {'name': u'Cooling Capacity Function of Temperature Curve Name',
                                       'pyname': u'cooling_capacity_function_of_temperature_curve_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'electric input to cooling output ratio function of temperature curve name',
                                      {'name': u'Electric Input to Cooling Output Ratio Function of Temperature Curve Name',
                                       'pyname': u'electric_input_to_cooling_output_ratio_function_of_temperature_curve_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'electric input to cooling output ratio function of part load ratio curve name',
                                      {'name': u'Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name',
                                       'pyname': u'electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'minimum part load ratio',
                                      {'name': u'Minimum Part Load Ratio',
                                       'pyname': u'minimum_part_load_ratio',
                                       'default': 0.1,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'maximum part load ratio',
                                      {'name': u'Maximum Part Load Ratio',
                                       'pyname': u'maximum_part_load_ratio',
                                       'default': 1.0,
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'optimum part load ratio',
                                      {'name': u'Optimum Part Load Ratio',
                                       'pyname': u'optimum_part_load_ratio',
                                       'default': 1.0,
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'minimum unloading ratio',
                                      {'name': u'Minimum Unloading Ratio',
                                       'pyname': u'minimum_unloading_ratio',
                                       'default': 0.2,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'chilled water inlet node name',
                                      {'name': u'Chilled Water Inlet Node Name',
                                       'pyname': u'chilled_water_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'chilled water outlet node name',
                                      {'name': u'Chilled Water Outlet Node Name',
                                       'pyname': u'chilled_water_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'condenser inlet node name',
                                      {'name': u'Condenser Inlet Node Name',
                                       'pyname': u'condenser_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'condenser outlet node name',
                                      {'name': u'Condenser Outlet Node Name',
                                       'pyname': u'condenser_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'fraction of compressor electric consumption rejected by condenser',
                                      {'name': u'Fraction of Compressor Electric Consumption Rejected by Condenser',
                                       'pyname': u'fraction_of_compressor_electric_consumption_rejected_by_condenser',
                                       'default': 1.0,
                                       'minimum>': 0.0,
                                       'maximum': 1.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'leaving chilled water lower temperature limit',
                                      {'name': u'Leaving Chilled Water Lower Temperature Limit',
                                       'pyname': u'leaving_chilled_water_lower_temperature_limit',
                                       'default': 2.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'C'}),
                                     (u'chiller flow mode type',
                                      {'name': u'Chiller Flow Mode Type',
                                       'pyname': u'chiller_flow_mode_type',
                                       'default': u'NotModulated',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ConstantFlow',
                                                           u'LeavingSetpointModulated',
                                                           u'NotModulated'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'design heat recovery water flow rate',
                                      {'name': u'Design Heat Recovery Water Flow Rate',
                                       'pyname': u'design_heat_recovery_water_flow_rate',
                                       'default': 0.0,
                                       'required-field': False,
                                       'autosizable': True,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'm3/s'}),
                                     (u'heat recovery inlet node name',
                                      {'name': u'Heat Recovery Inlet Node Name',
                                       'pyname': u'heat_recovery_inlet_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'heat recovery outlet node name',
                                      {'name': u'Heat Recovery Outlet Node Name',
                                       'pyname': u'heat_recovery_outlet_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'sizing factor',
                                      {'name': u'Sizing Factor',
                                       'pyname': u'sizing_factor',
                                       'default': 1.0,
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'condenser heat recovery relative capacity fraction',
                                      {'name': u'Condenser Heat Recovery Relative Capacity Fraction',
                                       'pyname': u'condenser_heat_recovery_relative_capacity_fraction',
                                       'maximum': 1.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'heat recovery inlet high temperature limit schedule name',
                                      {'name': u'Heat Recovery Inlet High Temperature Limit Schedule Name',
                                       'pyname': u'heat_recovery_inlet_high_temperature_limit_schedule_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'heat recovery leaving temperature setpoint node name',
                                      {'name': u'Heat Recovery Leaving Temperature Setpoint Node Name',
                                       'pyname': u'heat_recovery_leaving_temperature_setpoint_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'})]),
              'extensible-fields': OrderedDict(),
              'unique-object': False,
              'required-object': False,
              'group': u'Plant Heating and Cooling Equipment'}

    @property
    def name(self):
        """Get name.

        Returns:
            str: the value of `name` or None if not set

        """
        return self["Name"]

    @name.setter
    def name(self, value=None):
        """Corresponds to IDD field `Name`

        Args:
            value (str): value for IDD Field `Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Name"] = value

    @property
    def reference_capacity(self):
        """Get reference_capacity.

        Returns:
            float: the value of `reference_capacity` or None if not set

        """
        return self["Reference Capacity"]

    @reference_capacity.setter
    def reference_capacity(self, value=None):
        """Corresponds to IDD field `Reference Capacity`

        Args:
            value (float or "Autosize"): value for IDD Field `Reference Capacity`
                Units: W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Reference Capacity"] = value

    @property
    def reference_cop(self):
        """Get reference_cop.

        Returns:
            float: the value of `reference_cop` or None if not set

        """
        return self["Reference COP"]

    @reference_cop.setter
    def reference_cop(self, value=None):
        """Corresponds to IDD field `Reference COP` Efficiency of the chiller
        compressor (cooling output/compressor energy input). Condenser fan
        power should not be included here.

        Args:
            value (float): value for IDD Field `Reference COP`
                Units: W/W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Reference COP"] = value

    @property
    def reference_leaving_chilled_water_temperature(self):
        """Get reference_leaving_chilled_water_temperature.

        Returns:
            float: the value of `reference_leaving_chilled_water_temperature` or None if not set

        """
        return self["Reference Leaving Chilled Water Temperature"]

    @reference_leaving_chilled_water_temperature.setter
    def reference_leaving_chilled_water_temperature(self, value=6.67):
        """Corresponds to IDD field `Reference Leaving Chilled Water
        Temperature`

        Args:
            value (float): value for IDD Field `Reference Leaving Chilled Water Temperature`
                Units: C
                Default value: 6.67
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Reference Leaving Chilled Water Temperature"] = value

    @property
    def reference_leaving_condenser_water_temperature(self):
        """Get reference_leaving_condenser_water_temperature.

        Returns:
            float: the value of `reference_leaving_condenser_water_temperature` or None if not set

        """
        return self["Reference Leaving Condenser Water Temperature"]

    @reference_leaving_condenser_water_temperature.setter
    def reference_leaving_condenser_water_temperature(self, value=35.0):
        """Corresponds to IDD field `Reference Leaving Condenser Water
        Temperature`

        Args:
            value (float): value for IDD Field `Reference Leaving Condenser Water Temperature`
                Units: C
                Default value: 35.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Reference Leaving Condenser Water Temperature"] = value

    @property
    def reference_chilled_water_flow_rate(self):
        """Get reference_chilled_water_flow_rate.

        Returns:
            float: the value of `reference_chilled_water_flow_rate` or None if not set

        """
        return self["Reference Chilled Water Flow Rate"]

    @reference_chilled_water_flow_rate.setter
    def reference_chilled_water_flow_rate(self, value=None):
        """Corresponds to IDD field `Reference Chilled Water Flow Rate`

        Args:
            value (float or "Autosize"): value for IDD Field `Reference Chilled Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Reference Chilled Water Flow Rate"] = value

    @property
    def reference_condenser_water_flow_rate(self):
        """Get reference_condenser_water_flow_rate.

        Returns:
            float: the value of `reference_condenser_water_flow_rate` or None if not set

        """
        return self["Reference Condenser Water Flow Rate"]

    @reference_condenser_water_flow_rate.setter
    def reference_condenser_water_flow_rate(self, value=None):
        """Corresponds to IDD field `Reference Condenser Water Flow Rate`

        Args:
            value (float or "Autosize"): value for IDD Field `Reference Condenser Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Reference Condenser Water Flow Rate"] = value

    @property
    def cooling_capacity_function_of_temperature_curve_name(self):
        """Get cooling_capacity_function_of_temperature_curve_name.

        Returns:
            str: the value of `cooling_capacity_function_of_temperature_curve_name` or None if not set

        """
        return self["Cooling Capacity Function of Temperature Curve Name"]

    @cooling_capacity_function_of_temperature_curve_name.setter
    def cooling_capacity_function_of_temperature_curve_name(self, value=None):
        """  Corresponds to IDD field `Cooling Capacity Function of Temperature Curve Name`
        Cooling capacity as a function of supply (leaving) chilled water temperature
        and leaving condenser fluid temperature
        Table:TwoIndependentVariables object can also be used
        curve = a + b*CWS + c*CWS**2 + d*LCT + e*LCT**2 + f*CWS*LCT
        CWS = supply (leaving) chilled water temperature(C)
        LCT = leaving condenser fluid temperature(C)

        Args:
            value (str): value for IDD Field `Cooling Capacity Function of Temperature Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Cooling Capacity Function of Temperature Curve Name"] = value

    @property
    def electric_input_to_cooling_output_ratio_function_of_temperature_curve_name(
            self):
        """Get electric_input_to_cooling_output_ratio_function_of_temperature_c
        urve_name.

        Returns:
            str: the value of `electric_input_to_cooling_output_ratio_function_of_temperature_curve_name` or None if not set

        """
        return self[
            "Electric Input to Cooling Output Ratio Function of Temperature Curve Name"]

    @electric_input_to_cooling_output_ratio_function_of_temperature_curve_name.setter
    def electric_input_to_cooling_output_ratio_function_of_temperature_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Electric Input to Cooling Output Ratio Function of Temperature Curve Name`
        Electric Input Ratio (EIR) as a function of supply (leaving) chilled water temperature
        and leaving condenser fluid temperature.   EIR = 1/COP.
        Table:TwoIndependentVariables object can also be used
        curve = a + b*CWS + c*CWS**2 + d*LCT + e*LCT**2 + f*CWS*LCT
        CWS = supply (leaving) chilled water temperature(C)
        LCT = leaving condenser fluid temperature(C)

        Args:
            value (str): value for IDD Field `Electric Input to Cooling Output Ratio Function of Temperature Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Electric Input to Cooling Output Ratio Function of Temperature Curve Name"] = value

    @property
    def electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name(
            self):
        """Get electric_input_to_cooling_output_ratio_function_of_part_load_rat
        io_curve_name.

        Returns:
            str: the value of `electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name` or None if not set

        """
        return self[
            "Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name"]

    @electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name.setter
    def electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name`
        Electric Input Ratio (EIR) as a function of Part Load Ratio (PLR)
        EIR = 1/COP
        Table:TwoIndependentVariables object can also be used
        curve = a + b*LCT + c*LCT**2 + d*PLR + e*PLR**2 + f*LCT*PLR + g*0 + h*PLR**3
        + i*0 + j*0
        PLR = part load ratio (cooling load/steady state capacity)
        LCT = leaving condenser fluid temperature(C)

        Args:
            value (str): value for IDD Field `Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name"] = value

    @property
    def minimum_part_load_ratio(self):
        """Get minimum_part_load_ratio.

        Returns:
            float: the value of `minimum_part_load_ratio` or None if not set

        """
        return self["Minimum Part Load Ratio"]

    @minimum_part_load_ratio.setter
    def minimum_part_load_ratio(self, value=0.1):
        """Corresponds to IDD field `Minimum Part Load Ratio` Part load ratio
        below which the chiller starts cycling on/off to meet the load. Must be
        less than or equal to Maximum Part Load Ratio.

        Args:
            value (float): value for IDD Field `Minimum Part Load Ratio`
                Default value: 0.1
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Minimum Part Load Ratio"] = value

    @property
    def maximum_part_load_ratio(self):
        """Get maximum_part_load_ratio.

        Returns:
            float: the value of `maximum_part_load_ratio` or None if not set

        """
        return self["Maximum Part Load Ratio"]

    @maximum_part_load_ratio.setter
    def maximum_part_load_ratio(self, value=1.0):
        """Corresponds to IDD field `Maximum Part Load Ratio` Maximum allowable
        part load ratio. Must be greater than or equal to Minimum Part Load
        Ratio.

        Args:
            value (float): value for IDD Field `Maximum Part Load Ratio`
                Default value: 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Maximum Part Load Ratio"] = value

    @property
    def optimum_part_load_ratio(self):
        """Get optimum_part_load_ratio.

        Returns:
            float: the value of `optimum_part_load_ratio` or None if not set

        """
        return self["Optimum Part Load Ratio"]

    @optimum_part_load_ratio.setter
    def optimum_part_load_ratio(self, value=1.0):
        """Corresponds to IDD field `Optimum Part Load Ratio` Optimum part load
        ratio where the chiller is most efficient. Must be greater than or
        equal to the Minimum Part Load Ratio and less than or equal to the
        Maximum Part Load Ratio.

        Args:
            value (float): value for IDD Field `Optimum Part Load Ratio`
                Default value: 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Optimum Part Load Ratio"] = value

    @property
    def minimum_unloading_ratio(self):
        """Get minimum_unloading_ratio.

        Returns:
            float: the value of `minimum_unloading_ratio` or None if not set

        """
        return self["Minimum Unloading Ratio"]

    @minimum_unloading_ratio.setter
    def minimum_unloading_ratio(self, value=0.2):
        """Corresponds to IDD field `Minimum Unloading Ratio` Part load ratio
        where the chiller can no longer unload and false loading begins.
        Minimum unloading ratio must be greater than or equal to the Minimum
        Part Load Ratio and less than or equal to the Maximum Part Load Ratio.

        Args:
            value (float): value for IDD Field `Minimum Unloading Ratio`
                Default value: 0.2
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Minimum Unloading Ratio"] = value

    @property
    def chilled_water_inlet_node_name(self):
        """Get chilled_water_inlet_node_name.

        Returns:
            str: the value of `chilled_water_inlet_node_name` or None if not set

        """
        return self["Chilled Water Inlet Node Name"]

    @chilled_water_inlet_node_name.setter
    def chilled_water_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Chilled Water Inlet Node Name`

        Args:
            value (str): value for IDD Field `Chilled Water Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Inlet Node Name"] = value

    @property
    def chilled_water_outlet_node_name(self):
        """Get chilled_water_outlet_node_name.

        Returns:
            str: the value of `chilled_water_outlet_node_name` or None if not set

        """
        return self["Chilled Water Outlet Node Name"]

    @chilled_water_outlet_node_name.setter
    def chilled_water_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Chilled Water Outlet Node Name`

        Args:
            value (str): value for IDD Field `Chilled Water Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Outlet Node Name"] = value

    @property
    def condenser_inlet_node_name(self):
        """Get condenser_inlet_node_name.

        Returns:
            str: the value of `condenser_inlet_node_name` or None if not set

        """
        return self["Condenser Inlet Node Name"]

    @condenser_inlet_node_name.setter
    def condenser_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Condenser Inlet Node Name`

        Args:
            value (str): value for IDD Field `Condenser Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Inlet Node Name"] = value

    @property
    def condenser_outlet_node_name(self):
        """Get condenser_outlet_node_name.

        Returns:
            str: the value of `condenser_outlet_node_name` or None if not set

        """
        return self["Condenser Outlet Node Name"]

    @condenser_outlet_node_name.setter
    def condenser_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Condenser Outlet Node Name`

        Args:
            value (str): value for IDD Field `Condenser Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Outlet Node Name"] = value

    @property
    def fraction_of_compressor_electric_consumption_rejected_by_condenser(
            self):
        """Get
        fraction_of_compressor_electric_consumption_rejected_by_condenser.

        Returns:
            float: the value of `fraction_of_compressor_electric_consumption_rejected_by_condenser` or None if not set

        """
        return self[
            "Fraction of Compressor Electric Consumption Rejected by Condenser"]

    @fraction_of_compressor_electric_consumption_rejected_by_condenser.setter
    def fraction_of_compressor_electric_consumption_rejected_by_condenser(
            self,
            value=1.0):
        """  Corresponds to IDD field `Fraction of Compressor Electric Consumption Rejected by Condenser`
        Fraction of compressor electrical energy that must be rejected by the condenser.
        Enter a value of 1.0 when modeling hermetic chillers.
        For open chillers, enter the compressor motor efficiency.
        This value should be greater than 0.6 for praticle applications.

        Args:
            value (float): value for IDD Field `Fraction of Compressor Electric Consumption Rejected by Condenser`
                Default value: 1.0
                value <= 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Fraction of Compressor Electric Consumption Rejected by Condenser"] = value

    @property
    def leaving_chilled_water_lower_temperature_limit(self):
        """Get leaving_chilled_water_lower_temperature_limit.

        Returns:
            float: the value of `leaving_chilled_water_lower_temperature_limit` or None if not set

        """
        return self["Leaving Chilled Water Lower Temperature Limit"]

    @leaving_chilled_water_lower_temperature_limit.setter
    def leaving_chilled_water_lower_temperature_limit(self, value=2.0):
        """Corresponds to IDD field `Leaving Chilled Water Lower Temperature
        Limit`

        Args:
            value (float): value for IDD Field `Leaving Chilled Water Lower Temperature Limit`
                Units: C
                Default value: 2.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Leaving Chilled Water Lower Temperature Limit"] = value

    @property
    def chiller_flow_mode_type(self):
        """Get chiller_flow_mode_type.

        Returns:
            str: the value of `chiller_flow_mode_type` or None if not set

        """
        return self["Chiller Flow Mode Type"]

    @chiller_flow_mode_type.setter
    def chiller_flow_mode_type(self, value="NotModulated"):
        """Corresponds to IDD field `Chiller Flow Mode Type` Select operating
        mode for fluid flow through the chiller. "NotModulated" is for either
        variable or constant pumping with flow controlled by the external plant
        system. "ConstantFlow" is for constant pumping with flow controlled by
        chiller to operate at full design flow rate.
        "LeavingSetpointModulated" is for variable pumping with flow controlled
        by chiller to vary flow to target a leaving temperature setpoint.

        Args:
            value (str): value for IDD Field `Chiller Flow Mode Type`
                Default value: NotModulated
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Flow Mode Type"] = value

    @property
    def design_heat_recovery_water_flow_rate(self):
        """Get design_heat_recovery_water_flow_rate.

        Returns:
            float: the value of `design_heat_recovery_water_flow_rate` or None if not set

        """
        return self["Design Heat Recovery Water Flow Rate"]

    @design_heat_recovery_water_flow_rate.setter
    def design_heat_recovery_water_flow_rate(self, value=None):
        """  Corresponds to IDD field `Design Heat Recovery Water Flow Rate`
        If non-zero, then the heat recovery inlet and outlet node names must be entered.

        Args:
            value (float or "Autosize"): value for IDD Field `Design Heat Recovery Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Design Heat Recovery Water Flow Rate"] = value

    @property
    def heat_recovery_inlet_node_name(self):
        """Get heat_recovery_inlet_node_name.

        Returns:
            str: the value of `heat_recovery_inlet_node_name` or None if not set

        """
        return self["Heat Recovery Inlet Node Name"]

    @heat_recovery_inlet_node_name.setter
    def heat_recovery_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Heat Recovery Inlet Node Name`

        Args:
            value (str): value for IDD Field `Heat Recovery Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Heat Recovery Inlet Node Name"] = value

    @property
    def heat_recovery_outlet_node_name(self):
        """Get heat_recovery_outlet_node_name.

        Returns:
            str: the value of `heat_recovery_outlet_node_name` or None if not set

        """
        return self["Heat Recovery Outlet Node Name"]

    @heat_recovery_outlet_node_name.setter
    def heat_recovery_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Heat Recovery Outlet Node Name`

        Args:
            value (str): value for IDD Field `Heat Recovery Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Heat Recovery Outlet Node Name"] = value

    @property
    def sizing_factor(self):
        """Get sizing_factor.

        Returns:
            float: the value of `sizing_factor` or None if not set

        """
        return self["Sizing Factor"]

    @sizing_factor.setter
    def sizing_factor(self, value=1.0):
        """Corresponds to IDD field `Sizing Factor` Multiplies the autosized
        capacity and flow rates.

        Args:
            value (float): value for IDD Field `Sizing Factor`
                Default value: 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Sizing Factor"] = value

    @property
    def condenser_heat_recovery_relative_capacity_fraction(self):
        """Get condenser_heat_recovery_relative_capacity_fraction.

        Returns:
            float: the value of `condenser_heat_recovery_relative_capacity_fraction` or None if not set

        """
        return self["Condenser Heat Recovery Relative Capacity Fraction"]

    @condenser_heat_recovery_relative_capacity_fraction.setter
    def condenser_heat_recovery_relative_capacity_fraction(self, value=None):
        """Corresponds to IDD field `Condenser Heat Recovery Relative Capacity
        Fraction` This optional field is the fraction of total rejected heat
        that can be recovered at full load.

        Args:
            value (float): value for IDD Field `Condenser Heat Recovery Relative Capacity Fraction`
                value <= 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Heat Recovery Relative Capacity Fraction"] = value

    @property
    def heat_recovery_inlet_high_temperature_limit_schedule_name(self):
        """Get heat_recovery_inlet_high_temperature_limit_schedule_name.

        Returns:
            str: the value of `heat_recovery_inlet_high_temperature_limit_schedule_name` or None if not set

        """
        return self["Heat Recovery Inlet High Temperature Limit Schedule Name"]

    @heat_recovery_inlet_high_temperature_limit_schedule_name.setter
    def heat_recovery_inlet_high_temperature_limit_schedule_name(
            self,
            value=None):
        """Corresponds to IDD field `Heat Recovery Inlet High Temperature Limit
        Schedule Name` This optional schedule of temperatures will turn off
        heat recovery if inlet exceeds the value.

        Args:
            value (str): value for IDD Field `Heat Recovery Inlet High Temperature Limit Schedule Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Heat Recovery Inlet High Temperature Limit Schedule Name"] = value

    @property
    def heat_recovery_leaving_temperature_setpoint_node_name(self):
        """Get heat_recovery_leaving_temperature_setpoint_node_name.

        Returns:
            str: the value of `heat_recovery_leaving_temperature_setpoint_node_name` or None if not set

        """
        return self["Heat Recovery Leaving Temperature Setpoint Node Name"]

    @heat_recovery_leaving_temperature_setpoint_node_name.setter
    def heat_recovery_leaving_temperature_setpoint_node_name(self, value=None):
        """Corresponds to IDD field `Heat Recovery Leaving Temperature Setpoint
        Node Name` This optional field provides control over the heat recovery
        Using this triggers a model more suited to series bundle and chillers
        with higher temperature heat recovery If this field is not used, the
        bundles are modeled as being in parallel.

        Args:
            value (str): value for IDD Field `Heat Recovery Leaving Temperature Setpoint Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Heat Recovery Leaving Temperature Setpoint Node Name"] = value




class ChillerElectric(DataObject):

    """ Corresponds to IDD object `Chiller:Electric`
        This chiller model is the empirical model from the Building Loads
        and System Thermodynamics (BLAST) program.  Chiller performance
        curves are generated by fitting catalog data to third order
        polynomial equations.  Three sets of coefficients are required.
    """
    schema = {'min-fields': 27,
              'name': u'Chiller:Electric',
              'pyname': u'ChillerElectric',
              'format': None,
              'fields': OrderedDict([(u'name',
                                      {'name': u'Name',
                                       'pyname': u'name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'condenser type',
                                      {'name': u'Condenser Type',
                                       'pyname': u'condenser_type',
                                       'default': u'AirCooled',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'AirCooled',
                                                           u'WaterCooled',
                                                           u'EvaporativelyCooled'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'nominal capacity',
                                      {'name': u'Nominal Capacity',
                                       'pyname': u'nominal_capacity',
                                       'required-field': True,
                                       'autosizable': True,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'W'}),
                                     (u'nominal cop',
                                      {'name': u'Nominal COP',
                                       'pyname': u'nominal_cop',
                                       'minimum>': 0.0,
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'W/W'}),
                                     (u'chilled water inlet node name',
                                      {'name': u'Chilled Water Inlet Node Name',
                                       'pyname': u'chilled_water_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'chilled water outlet node name',
                                      {'name': u'Chilled Water Outlet Node Name',
                                       'pyname': u'chilled_water_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'condenser inlet node name',
                                      {'name': u'Condenser Inlet Node Name',
                                       'pyname': u'condenser_inlet_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'condenser outlet node name',
                                      {'name': u'Condenser Outlet Node Name',
                                       'pyname': u'condenser_outlet_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'minimum part load ratio',
                                      {'name': u'Minimum Part Load Ratio',
                                       'pyname': u'minimum_part_load_ratio',
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'maximum part load ratio',
                                      {'name': u'Maximum Part Load Ratio',
                                       'pyname': u'maximum_part_load_ratio',
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'optimum part load ratio',
                                      {'name': u'Optimum Part Load Ratio',
                                       'pyname': u'optimum_part_load_ratio',
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'design condenser inlet temperature',
                                      {'name': u'Design Condenser Inlet Temperature',
                                       'pyname': u'design_condenser_inlet_temperature',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'C'}),
                                     (u'temperature rise coefficient',
                                      {'name': u'Temperature Rise Coefficient',
                                       'pyname': u'temperature_rise_coefficient',
                                       'minimum>': 0.0,
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'design chilled water outlet temperature',
                                      {'name': u'Design Chilled Water Outlet Temperature',
                                       'pyname': u'design_chilled_water_outlet_temperature',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'C'}),
                                     (u'design chilled water flow rate',
                                      {'name': u'Design Chilled Water Flow Rate',
                                       'pyname': u'design_chilled_water_flow_rate',
                                       'required-field': False,
                                       'autosizable': True,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'm3/s'}),
                                     (u'design condenser fluid flow rate',
                                      {'name': u'Design Condenser Fluid Flow Rate',
                                       'pyname': u'design_condenser_fluid_flow_rate',
                                       'required-field': False,
                                       'autosizable': True,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'm3/s'}),
                                     (u'coefficient 1 of capacity ratio curve',
                                      {'name': u'Coefficient 1 of Capacity Ratio Curve',
                                       'pyname': u'coefficient_1_of_capacity_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 2 of capacity ratio curve',
                                      {'name': u'Coefficient 2 of Capacity Ratio Curve',
                                       'pyname': u'coefficient_2_of_capacity_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 3 of capacity ratio curve',
                                      {'name': u'Coefficient 3 of Capacity Ratio Curve',
                                       'pyname': u'coefficient_3_of_capacity_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 1 of power ratio curve',
                                      {'name': u'Coefficient 1 of Power Ratio Curve',
                                       'pyname': u'coefficient_1_of_power_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 2 of power ratio curve',
                                      {'name': u'Coefficient 2 of Power Ratio Curve',
                                       'pyname': u'coefficient_2_of_power_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 3 of power ratio curve',
                                      {'name': u'Coefficient 3 of Power Ratio Curve',
                                       'pyname': u'coefficient_3_of_power_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 1 of full load ratio curve',
                                      {'name': u'Coefficient 1 of Full Load Ratio Curve',
                                       'pyname': u'coefficient_1_of_full_load_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 2 of full load ratio curve',
                                      {'name': u'Coefficient 2 of Full Load Ratio Curve',
                                       'pyname': u'coefficient_2_of_full_load_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 3 of full load ratio curve',
                                      {'name': u'Coefficient 3 of Full Load Ratio Curve',
                                       'pyname': u'coefficient_3_of_full_load_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'chilled water outlet temperature lower limit',
                                      {'name': u'Chilled Water Outlet Temperature Lower Limit',
                                       'pyname': u'chilled_water_outlet_temperature_lower_limit',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'C'}),
                                     (u'chiller flow mode',
                                      {'name': u'Chiller Flow Mode',
                                       'pyname': u'chiller_flow_mode',
                                       'default': u'NotModulated',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ConstantFlow',
                                                           u'LeavingSetpointModulated',
                                                           u'NotModulated'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'design heat recovery water flow rate',
                                      {'name': u'Design Heat Recovery Water Flow Rate',
                                       'pyname': u'design_heat_recovery_water_flow_rate',
                                       'default': 0.0,
                                       'required-field': False,
                                       'autosizable': True,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'm3/s'}),
                                     (u'heat recovery inlet node name',
                                      {'name': u'Heat Recovery Inlet Node Name',
                                       'pyname': u'heat_recovery_inlet_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'heat recovery outlet node name',
                                      {'name': u'Heat Recovery Outlet Node Name',
                                       'pyname': u'heat_recovery_outlet_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'sizing factor',
                                      {'name': u'Sizing Factor',
                                       'pyname': u'sizing_factor',
                                       'default': 1.0,
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'basin heater capacity',
                                      {'name': u'Basin Heater Capacity',
                                       'pyname': u'basin_heater_capacity',
                                       'default': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'W/K'}),
                                     (u'basin heater setpoint temperature',
                                      {'name': u'Basin Heater Setpoint Temperature',
                                       'pyname': u'basin_heater_setpoint_temperature',
                                       'default': 2.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 2.0,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'C'}),
                                     (u'basin heater operating schedule name',
                                      {'name': u'Basin Heater Operating Schedule Name',
                                       'pyname': u'basin_heater_operating_schedule_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'condenser heat recovery relative capacity fraction',
                                      {'name': u'Condenser Heat Recovery Relative Capacity Fraction',
                                       'pyname': u'condenser_heat_recovery_relative_capacity_fraction',
                                       'maximum': 1.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'heat recovery inlet high temperature limit schedule name',
                                      {'name': u'Heat Recovery Inlet High Temperature Limit Schedule Name',
                                       'pyname': u'heat_recovery_inlet_high_temperature_limit_schedule_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'heat recovery leaving temperature setpoint node name',
                                      {'name': u'Heat Recovery Leaving Temperature Setpoint Node Name',
                                       'pyname': u'heat_recovery_leaving_temperature_setpoint_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'})]),
              'extensible-fields': OrderedDict(),
              'unique-object': False,
              'required-object': False,
              'group': u'Plant Heating and Cooling Equipment'}

    @property
    def name(self):
        """Get name.

        Returns:
            str: the value of `name` or None if not set

        """
        return self["Name"]

    @name.setter
    def name(self, value=None):
        """Corresponds to IDD field `Name`

        Args:
            value (str): value for IDD Field `Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Name"] = value

    @property
    def condenser_type(self):
        """Get condenser_type.

        Returns:
            str: the value of `condenser_type` or None if not set

        """
        return self["Condenser Type"]

    @condenser_type.setter
    def condenser_type(self, value="AirCooled"):
        """Corresponds to IDD field `Condenser Type`

        Args:
            value (str): value for IDD Field `Condenser Type`
                Default value: AirCooled
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Type"] = value

    @property
    def nominal_capacity(self):
        """Get nominal_capacity.

        Returns:
            float: the value of `nominal_capacity` or None if not set

        """
        return self["Nominal Capacity"]

    @nominal_capacity.setter
    def nominal_capacity(self, value=None):
        """Corresponds to IDD field `Nominal Capacity`

        Args:
            value (float or "Autosize"): value for IDD Field `Nominal Capacity`
                Units: W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Nominal Capacity"] = value

    @property
    def nominal_cop(self):
        """Get nominal_cop.

        Returns:
            float: the value of `nominal_cop` or None if not set

        """
        return self["Nominal COP"]

    @nominal_cop.setter
    def nominal_cop(self, value=None):
        """Corresponds to IDD field `Nominal COP`

        Args:
            value (float): value for IDD Field `Nominal COP`
                Units: W/W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Nominal COP"] = value

    @property
    def chilled_water_inlet_node_name(self):
        """Get chilled_water_inlet_node_name.

        Returns:
            str: the value of `chilled_water_inlet_node_name` or None if not set

        """
        return self["Chilled Water Inlet Node Name"]

    @chilled_water_inlet_node_name.setter
    def chilled_water_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Chilled Water Inlet Node Name`

        Args:
            value (str): value for IDD Field `Chilled Water Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Inlet Node Name"] = value

    @property
    def chilled_water_outlet_node_name(self):
        """Get chilled_water_outlet_node_name.

        Returns:
            str: the value of `chilled_water_outlet_node_name` or None if not set

        """
        return self["Chilled Water Outlet Node Name"]

    @chilled_water_outlet_node_name.setter
    def chilled_water_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Chilled Water Outlet Node Name`

        Args:
            value (str): value for IDD Field `Chilled Water Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Outlet Node Name"] = value

    @property
    def condenser_inlet_node_name(self):
        """Get condenser_inlet_node_name.

        Returns:
            str: the value of `condenser_inlet_node_name` or None if not set

        """
        return self["Condenser Inlet Node Name"]

    @condenser_inlet_node_name.setter
    def condenser_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Condenser Inlet Node Name`

        Args:
            value (str): value for IDD Field `Condenser Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Inlet Node Name"] = value

    @property
    def condenser_outlet_node_name(self):
        """Get condenser_outlet_node_name.

        Returns:
            str: the value of `condenser_outlet_node_name` or None if not set

        """
        return self["Condenser Outlet Node Name"]

    @condenser_outlet_node_name.setter
    def condenser_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Condenser Outlet Node Name`

        Args:
            value (str): value for IDD Field `Condenser Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Outlet Node Name"] = value

    @property
    def minimum_part_load_ratio(self):
        """Get minimum_part_load_ratio.

        Returns:
            float: the value of `minimum_part_load_ratio` or None if not set

        """
        return self["Minimum Part Load Ratio"]

    @minimum_part_load_ratio.setter
    def minimum_part_load_ratio(self, value=None):
        """Corresponds to IDD field `Minimum Part Load Ratio`

        Args:
            value (float): value for IDD Field `Minimum Part Load Ratio`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Minimum Part Load Ratio"] = value

    @property
    def maximum_part_load_ratio(self):
        """Get maximum_part_load_ratio.

        Returns:
            float: the value of `maximum_part_load_ratio` or None if not set

        """
        return self["Maximum Part Load Ratio"]

    @maximum_part_load_ratio.setter
    def maximum_part_load_ratio(self, value=None):
        """Corresponds to IDD field `Maximum Part Load Ratio`

        Args:
            value (float): value for IDD Field `Maximum Part Load Ratio`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Maximum Part Load Ratio"] = value

    @property
    def optimum_part_load_ratio(self):
        """Get optimum_part_load_ratio.

        Returns:
            float: the value of `optimum_part_load_ratio` or None if not set

        """
        return self["Optimum Part Load Ratio"]

    @optimum_part_load_ratio.setter
    def optimum_part_load_ratio(self, value=None):
        """Corresponds to IDD field `Optimum Part Load Ratio`

        Args:
            value (float): value for IDD Field `Optimum Part Load Ratio`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Optimum Part Load Ratio"] = value

    @property
    def design_condenser_inlet_temperature(self):
        """Get design_condenser_inlet_temperature.

        Returns:
            float: the value of `design_condenser_inlet_temperature` or None if not set

        """
        return self["Design Condenser Inlet Temperature"]

    @design_condenser_inlet_temperature.setter
    def design_condenser_inlet_temperature(self, value=None):
        """Corresponds to IDD field `Design Condenser Inlet Temperature`

        Args:
            value (float): value for IDD Field `Design Condenser Inlet Temperature`
                Units: C
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Condenser Inlet Temperature"] = value

    @property
    def temperature_rise_coefficient(self):
        """Get temperature_rise_coefficient.

        Returns:
            float: the value of `temperature_rise_coefficient` or None if not set

        """
        return self["Temperature Rise Coefficient"]

    @temperature_rise_coefficient.setter
    def temperature_rise_coefficient(self, value=None):
        """Corresponds to IDD field `Temperature Rise Coefficient`

        Args:
            value (float): value for IDD Field `Temperature Rise Coefficient`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Temperature Rise Coefficient"] = value

    @property
    def design_chilled_water_outlet_temperature(self):
        """Get design_chilled_water_outlet_temperature.

        Returns:
            float: the value of `design_chilled_water_outlet_temperature` or None if not set

        """
        return self["Design Chilled Water Outlet Temperature"]

    @design_chilled_water_outlet_temperature.setter
    def design_chilled_water_outlet_temperature(self, value=None):
        """Corresponds to IDD field `Design Chilled Water Outlet Temperature`

        Args:
            value (float): value for IDD Field `Design Chilled Water Outlet Temperature`
                Units: C
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Chilled Water Outlet Temperature"] = value

    @property
    def design_chilled_water_flow_rate(self):
        """Get design_chilled_water_flow_rate.

        Returns:
            float: the value of `design_chilled_water_flow_rate` or None if not set

        """
        return self["Design Chilled Water Flow Rate"]

    @design_chilled_water_flow_rate.setter
    def design_chilled_water_flow_rate(self, value=None):
        """Corresponds to IDD field `Design Chilled Water Flow Rate` For
        variable volume this is the maximum flow & for constant flow this is
        the flow.

        Args:
            value (float or "Autosize"): value for IDD Field `Design Chilled Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Chilled Water Flow Rate"] = value

    @property
    def design_condenser_fluid_flow_rate(self):
        """Get design_condenser_fluid_flow_rate.

        Returns:
            float: the value of `design_condenser_fluid_flow_rate` or None if not set

        """
        return self["Design Condenser Fluid Flow Rate"]

    @design_condenser_fluid_flow_rate.setter
    def design_condenser_fluid_flow_rate(self, value=None):
        """  Corresponds to IDD field `Design Condenser Fluid Flow Rate`
        This field is only used for Condenser Type = AirCooled or EvaporativelyCooled
        when Heat Recovery is specified

        Args:
            value (float or "Autosize"): value for IDD Field `Design Condenser Fluid Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Design Condenser Fluid Flow Rate"] = value

    @property
    def coefficient_1_of_capacity_ratio_curve(self):
        """Get coefficient_1_of_capacity_ratio_curve.

        Returns:
            float: the value of `coefficient_1_of_capacity_ratio_curve` or None if not set

        """
        return self["Coefficient 1 of Capacity Ratio Curve"]

    @coefficient_1_of_capacity_ratio_curve.setter
    def coefficient_1_of_capacity_ratio_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 1 of Capacity Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 1 of Capacity Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 1 of Capacity Ratio Curve"] = value

    @property
    def coefficient_2_of_capacity_ratio_curve(self):
        """Get coefficient_2_of_capacity_ratio_curve.

        Returns:
            float: the value of `coefficient_2_of_capacity_ratio_curve` or None if not set

        """
        return self["Coefficient 2 of Capacity Ratio Curve"]

    @coefficient_2_of_capacity_ratio_curve.setter
    def coefficient_2_of_capacity_ratio_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 2 of Capacity Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 2 of Capacity Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 2 of Capacity Ratio Curve"] = value

    @property
    def coefficient_3_of_capacity_ratio_curve(self):
        """Get coefficient_3_of_capacity_ratio_curve.

        Returns:
            float: the value of `coefficient_3_of_capacity_ratio_curve` or None if not set

        """
        return self["Coefficient 3 of Capacity Ratio Curve"]

    @coefficient_3_of_capacity_ratio_curve.setter
    def coefficient_3_of_capacity_ratio_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 3 of Capacity Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 3 of Capacity Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 3 of Capacity Ratio Curve"] = value

    @property
    def coefficient_1_of_power_ratio_curve(self):
        """Get coefficient_1_of_power_ratio_curve.

        Returns:
            float: the value of `coefficient_1_of_power_ratio_curve` or None if not set

        """
        return self["Coefficient 1 of Power Ratio Curve"]

    @coefficient_1_of_power_ratio_curve.setter
    def coefficient_1_of_power_ratio_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 1 of Power Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 1 of Power Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 1 of Power Ratio Curve"] = value

    @property
    def coefficient_2_of_power_ratio_curve(self):
        """Get coefficient_2_of_power_ratio_curve.

        Returns:
            float: the value of `coefficient_2_of_power_ratio_curve` or None if not set

        """
        return self["Coefficient 2 of Power Ratio Curve"]

    @coefficient_2_of_power_ratio_curve.setter
    def coefficient_2_of_power_ratio_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 2 of Power Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 2 of Power Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 2 of Power Ratio Curve"] = value

    @property
    def coefficient_3_of_power_ratio_curve(self):
        """Get coefficient_3_of_power_ratio_curve.

        Returns:
            float: the value of `coefficient_3_of_power_ratio_curve` or None if not set

        """
        return self["Coefficient 3 of Power Ratio Curve"]

    @coefficient_3_of_power_ratio_curve.setter
    def coefficient_3_of_power_ratio_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 3 of Power Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 3 of Power Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 3 of Power Ratio Curve"] = value

    @property
    def coefficient_1_of_full_load_ratio_curve(self):
        """Get coefficient_1_of_full_load_ratio_curve.

        Returns:
            float: the value of `coefficient_1_of_full_load_ratio_curve` or None if not set

        """
        return self["Coefficient 1 of Full Load Ratio Curve"]

    @coefficient_1_of_full_load_ratio_curve.setter
    def coefficient_1_of_full_load_ratio_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 1 of Full Load Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 1 of Full Load Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 1 of Full Load Ratio Curve"] = value

    @property
    def coefficient_2_of_full_load_ratio_curve(self):
        """Get coefficient_2_of_full_load_ratio_curve.

        Returns:
            float: the value of `coefficient_2_of_full_load_ratio_curve` or None if not set

        """
        return self["Coefficient 2 of Full Load Ratio Curve"]

    @coefficient_2_of_full_load_ratio_curve.setter
    def coefficient_2_of_full_load_ratio_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 2 of Full Load Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 2 of Full Load Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 2 of Full Load Ratio Curve"] = value

    @property
    def coefficient_3_of_full_load_ratio_curve(self):
        """Get coefficient_3_of_full_load_ratio_curve.

        Returns:
            float: the value of `coefficient_3_of_full_load_ratio_curve` or None if not set

        """
        return self["Coefficient 3 of Full Load Ratio Curve"]

    @coefficient_3_of_full_load_ratio_curve.setter
    def coefficient_3_of_full_load_ratio_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 3 of Full Load Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 3 of Full Load Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 3 of Full Load Ratio Curve"] = value

    @property
    def chilled_water_outlet_temperature_lower_limit(self):
        """Get chilled_water_outlet_temperature_lower_limit.

        Returns:
            float: the value of `chilled_water_outlet_temperature_lower_limit` or None if not set

        """
        return self["Chilled Water Outlet Temperature Lower Limit"]

    @chilled_water_outlet_temperature_lower_limit.setter
    def chilled_water_outlet_temperature_lower_limit(self, value=None):
        """Corresponds to IDD field `Chilled Water Outlet Temperature Lower
        Limit`

        Args:
            value (float): value for IDD Field `Chilled Water Outlet Temperature Lower Limit`
                Units: C
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Outlet Temperature Lower Limit"] = value

    @property
    def chiller_flow_mode(self):
        """Get chiller_flow_mode.

        Returns:
            str: the value of `chiller_flow_mode` or None if not set

        """
        return self["Chiller Flow Mode"]

    @chiller_flow_mode.setter
    def chiller_flow_mode(self, value="NotModulated"):
        """Corresponds to IDD field `Chiller Flow Mode` Select operating mode
        for fluid flow through the chiller. "NotModulated" is for either
        variable or constant pumping with flow controlled by the external plant
        system. "ConstantFlow" is for constant pumping with flow controlled by
        chiller to operate at full design flow rate.
        "LeavingSetpointModulated" is for variable pumping with flow controlled
        by chiller to vary flow to target a leaving temperature setpoint.

        Args:
            value (str): value for IDD Field `Chiller Flow Mode`
                Default value: NotModulated
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Flow Mode"] = value

    @property
    def design_heat_recovery_water_flow_rate(self):
        """Get design_heat_recovery_water_flow_rate.

        Returns:
            float: the value of `design_heat_recovery_water_flow_rate` or None if not set

        """
        return self["Design Heat Recovery Water Flow Rate"]

    @design_heat_recovery_water_flow_rate.setter
    def design_heat_recovery_water_flow_rate(self, value=None):
        """  Corresponds to IDD field `Design Heat Recovery Water Flow Rate`
        If non-zero, then the heat recovery inlet and outlet node names must be entered.

        Args:
            value (float or "Autosize"): value for IDD Field `Design Heat Recovery Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Design Heat Recovery Water Flow Rate"] = value

    @property
    def heat_recovery_inlet_node_name(self):
        """Get heat_recovery_inlet_node_name.

        Returns:
            str: the value of `heat_recovery_inlet_node_name` or None if not set

        """
        return self["Heat Recovery Inlet Node Name"]

    @heat_recovery_inlet_node_name.setter
    def heat_recovery_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Heat Recovery Inlet Node Name`

        Args:
            value (str): value for IDD Field `Heat Recovery Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Heat Recovery Inlet Node Name"] = value

    @property
    def heat_recovery_outlet_node_name(self):
        """Get heat_recovery_outlet_node_name.

        Returns:
            str: the value of `heat_recovery_outlet_node_name` or None if not set

        """
        return self["Heat Recovery Outlet Node Name"]

    @heat_recovery_outlet_node_name.setter
    def heat_recovery_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Heat Recovery Outlet Node Name`

        Args:
            value (str): value for IDD Field `Heat Recovery Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Heat Recovery Outlet Node Name"] = value

    @property
    def sizing_factor(self):
        """Get sizing_factor.

        Returns:
            float: the value of `sizing_factor` or None if not set

        """
        return self["Sizing Factor"]

    @sizing_factor.setter
    def sizing_factor(self, value=1.0):
        """Corresponds to IDD field `Sizing Factor` Multiplies the autosized
        capacity and flow rates.

        Args:
            value (float): value for IDD Field `Sizing Factor`
                Default value: 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Sizing Factor"] = value

    @property
    def basin_heater_capacity(self):
        """Get basin_heater_capacity.

        Returns:
            float: the value of `basin_heater_capacity` or None if not set

        """
        return self["Basin Heater Capacity"]

    @basin_heater_capacity.setter
    def basin_heater_capacity(self, value=None):
        """  Corresponds to IDD field `Basin Heater Capacity`
        This field is only used for Condenser Type = EvaporativelyCooled and for periods
        when the basin heater is available (field Basin Heater Operating Schedule Name).
        For this situation, the heater maintains the basin water temperature at the basin heater
        setpoint temperature when the outdoor air temperature falls below the setpoint temperature.
        The basin heater only operates when the chiller is not operating.

        Args:
            value (float): value for IDD Field `Basin Heater Capacity`
                Units: W/K
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Basin Heater Capacity"] = value

    @property
    def basin_heater_setpoint_temperature(self):
        """Get basin_heater_setpoint_temperature.

        Returns:
            float: the value of `basin_heater_setpoint_temperature` or None if not set

        """
        return self["Basin Heater Setpoint Temperature"]

    @basin_heater_setpoint_temperature.setter
    def basin_heater_setpoint_temperature(self, value=2.0):
        """  Corresponds to IDD field `Basin Heater Setpoint Temperature`
        This field is only used for Condenser Type = EvaporativelyCooled.
        Enter the outdoor dry-bulb temperature when the basin heater turns on.

        Args:
            value (float): value for IDD Field `Basin Heater Setpoint Temperature`
                Units: C
                Default value: 2.0
                value >= 2.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Basin Heater Setpoint Temperature"] = value

    @property
    def basin_heater_operating_schedule_name(self):
        """Get basin_heater_operating_schedule_name.

        Returns:
            str: the value of `basin_heater_operating_schedule_name` or None if not set

        """
        return self["Basin Heater Operating Schedule Name"]

    @basin_heater_operating_schedule_name.setter
    def basin_heater_operating_schedule_name(self, value=None):
        """  Corresponds to IDD field `Basin Heater Operating Schedule Name`
        This field is only used for Condenser Type = EvaporativelyCooled.
        Schedule values greater than 0 allow the basin heater to operate whenever the outdoor
        air dry-bulb temperature is below the basin heater setpoint temperature.
        If a schedule name is not entered, the basin heater is allowed to operate
        throughout the entire simulation.

        Args:
            value (str): value for IDD Field `Basin Heater Operating Schedule Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Basin Heater Operating Schedule Name"] = value

    @property
    def condenser_heat_recovery_relative_capacity_fraction(self):
        """Get condenser_heat_recovery_relative_capacity_fraction.

        Returns:
            float: the value of `condenser_heat_recovery_relative_capacity_fraction` or None if not set

        """
        return self["Condenser Heat Recovery Relative Capacity Fraction"]

    @condenser_heat_recovery_relative_capacity_fraction.setter
    def condenser_heat_recovery_relative_capacity_fraction(self, value=None):
        """Corresponds to IDD field `Condenser Heat Recovery Relative Capacity
        Fraction` This optional field is the fraction of total rejected heat
        that can be recovered at full load.

        Args:
            value (float): value for IDD Field `Condenser Heat Recovery Relative Capacity Fraction`
                value <= 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Heat Recovery Relative Capacity Fraction"] = value

    @property
    def heat_recovery_inlet_high_temperature_limit_schedule_name(self):
        """Get heat_recovery_inlet_high_temperature_limit_schedule_name.

        Returns:
            str: the value of `heat_recovery_inlet_high_temperature_limit_schedule_name` or None if not set

        """
        return self["Heat Recovery Inlet High Temperature Limit Schedule Name"]

    @heat_recovery_inlet_high_temperature_limit_schedule_name.setter
    def heat_recovery_inlet_high_temperature_limit_schedule_name(
            self,
            value=None):
        """Corresponds to IDD field `Heat Recovery Inlet High Temperature Limit
        Schedule Name` This optional schedule of temperatures will turn off
        heat recovery if inlet exceeds the value.

        Args:
            value (str): value for IDD Field `Heat Recovery Inlet High Temperature Limit Schedule Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Heat Recovery Inlet High Temperature Limit Schedule Name"] = value

    @property
    def heat_recovery_leaving_temperature_setpoint_node_name(self):
        """Get heat_recovery_leaving_temperature_setpoint_node_name.

        Returns:
            str: the value of `heat_recovery_leaving_temperature_setpoint_node_name` or None if not set

        """
        return self["Heat Recovery Leaving Temperature Setpoint Node Name"]

    @heat_recovery_leaving_temperature_setpoint_node_name.setter
    def heat_recovery_leaving_temperature_setpoint_node_name(self, value=None):
        """Corresponds to IDD field `Heat Recovery Leaving Temperature Setpoint
        Node Name` This optional field provides control over the heat recovery
        Using this triggers a model more suited to series bundle and chillers
        with higher temperature heat recovery If this field is not used, the
        bundles are modeled as being in parallel.

        Args:
            value (str): value for IDD Field `Heat Recovery Leaving Temperature Setpoint Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Heat Recovery Leaving Temperature Setpoint Node Name"] = value




class ChillerAbsorptionIndirect(DataObject):

    """ Corresponds to IDD object `Chiller:Absorption:Indirect`
        This indirect absorption chiller model is an enhanced model from the
        Building Loads and System Thermodynamics (BLAST) program.  Chiller
        performance curves are generated by fitting catalog data to third order
        polynomial equations. The chiller capacity is a function of condenser,
        chilled water, and generator temperatures. The heat input is a function
        of part-load ratio, condenser temperature, and chilled water temperature.
    """
    schema = {'min-fields': 17,
              'name': u'Chiller:Absorption:Indirect',
              'pyname': u'ChillerAbsorptionIndirect',
              'format': None,
              'fields': OrderedDict([(u'name',
                                      {'name': u'Name',
                                       'pyname': u'name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'alpha'}),
                                     (u'nominal capacity',
                                      {'name': u'Nominal Capacity',
                                       'pyname': u'nominal_capacity',
                                       'minimum>': 0.0,
                                       'required-field': True,
                                       'autosizable': True,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'W'}),
                                     (u'nominal pumping power',
                                      {'name': u'Nominal Pumping Power',
                                       'pyname': u'nominal_pumping_power',
                                       'required-field': True,
                                       'autosizable': True,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'W'}),
                                     (u'chilled water inlet node name',
                                      {'name': u'Chilled Water Inlet Node Name',
                                       'pyname': u'chilled_water_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'chilled water outlet node name',
                                      {'name': u'Chilled Water Outlet Node Name',
                                       'pyname': u'chilled_water_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'condenser inlet node name',
                                      {'name': u'Condenser Inlet Node Name',
                                       'pyname': u'condenser_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'condenser outlet node name',
                                      {'name': u'Condenser Outlet Node Name',
                                       'pyname': u'condenser_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'minimum part load ratio',
                                      {'name': u'Minimum Part Load Ratio',
                                       'pyname': u'minimum_part_load_ratio',
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'maximum part load ratio',
                                      {'name': u'Maximum Part Load Ratio',
                                       'pyname': u'maximum_part_load_ratio',
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'optimum part load ratio',
                                      {'name': u'Optimum Part Load Ratio',
                                       'pyname': u'optimum_part_load_ratio',
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'design condenser inlet temperature',
                                      {'name': u'Design Condenser Inlet Temperature',
                                       'pyname': u'design_condenser_inlet_temperature',
                                       'default': 30.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'C'}),
                                     (u'condenser inlet temperature lower limit',
                                      {'name': u'Condenser Inlet Temperature Lower Limit',
                                       'pyname': u'condenser_inlet_temperature_lower_limit',
                                       'default': 15.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'C'}),
                                     (u'chilled water outlet temperature lower limit',
                                      {'name': u'Chilled Water Outlet Temperature Lower Limit',
                                       'pyname': u'chilled_water_outlet_temperature_lower_limit',
                                       'default': 5.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'C'}),
                                     (u'design chilled water flow rate',
                                      {'name': u'Design Chilled Water Flow Rate',
                                       'pyname': u'design_chilled_water_flow_rate',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': True,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'm3/s'}),
                                     (u'design condenser water flow rate',
                                      {'name': u'Design Condenser Water Flow Rate',
                                       'pyname': u'design_condenser_water_flow_rate',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': True,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'm3/s'}),
                                     (u'chiller flow mode',
                                      {'name': u'Chiller Flow Mode',
                                       'pyname': u'chiller_flow_mode',
                                       'default': u'NotModulated',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ConstantFlow',
                                                           u'LeavingSetpointModulated',
                                                           u'NotModulated'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'generator heat input function of part load ratio curve name',
                                      {'name': u'Generator Heat Input Function of Part Load Ratio Curve Name',
                                       'pyname': u'generator_heat_input_function_of_part_load_ratio_curve_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'pump electric input function of part load ratio curve name',
                                      {'name': u'Pump Electric Input Function of Part Load Ratio Curve Name',
                                       'pyname': u'pump_electric_input_function_of_part_load_ratio_curve_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'generator inlet node name',
                                      {'name': u'Generator Inlet Node Name',
                                       'pyname': u'generator_inlet_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'generator outlet node name',
                                      {'name': u'Generator Outlet Node Name',
                                       'pyname': u'generator_outlet_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'capacity correction function of condenser temperature curve name',
                                      {'name': u'Capacity Correction Function of Condenser Temperature Curve Name',
                                       'pyname': u'capacity_correction_function_of_condenser_temperature_curve_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'capacity correction function of chilled water temperature curve name',
                                      {'name': u'Capacity Correction Function of Chilled Water Temperature Curve Name',
                                       'pyname': u'capacity_correction_function_of_chilled_water_temperature_curve_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'capacity correction function of generator temperature curve name',
                                      {'name': u'Capacity Correction Function of Generator Temperature Curve Name',
                                       'pyname': u'capacity_correction_function_of_generator_temperature_curve_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'generator heat input correction function of condenser temperature curve name',
                                      {'name': u'Generator Heat Input Correction Function of Condenser Temperature Curve Name',
                                       'pyname': u'generator_heat_input_correction_function_of_condenser_temperature_curve_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'generator heat input correction function of chilled water temperature curve name',
                                      {'name': u'Generator Heat Input Correction Function of Chilled Water Temperature Curve Name',
                                       'pyname': u'generator_heat_input_correction_function_of_chilled_water_temperature_curve_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'generator heat source type',
                                      {'name': u'Generator Heat Source Type',
                                       'pyname': u'generator_heat_source_type',
                                       'default': u'Steam',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'HotWater',
                                                           u'Steam'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'design generator fluid flow rate',
                                      {'name': u'Design Generator Fluid Flow Rate',
                                       'pyname': u'design_generator_fluid_flow_rate',
                                       'required-field': False,
                                       'autosizable': True,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'm3/s'}),
                                     (u'temperature lower limit generator inlet',
                                      {'name': u'Temperature Lower Limit Generator Inlet',
                                       'pyname': u'temperature_lower_limit_generator_inlet',
                                       'default': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'C'}),
                                     (u'degree of subcooling in steam generator',
                                      {'name': u'Degree of Subcooling in Steam Generator',
                                       'pyname': u'degree_of_subcooling_in_steam_generator',
                                       'default': 1.0,
                                       'maximum': 20.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'C'}),
                                     (u'degree of subcooling in steam condensate loop',
                                      {'name': u'Degree of Subcooling in Steam Condensate Loop',
                                       'pyname': u'degree_of_subcooling_in_steam_condensate_loop',
                                       'default': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'C'}),
                                     (u'sizing factor',
                                      {'name': u'Sizing Factor',
                                       'pyname': u'sizing_factor',
                                       'default': 1.0,
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'})]),
              'extensible-fields': OrderedDict(),
              'unique-object': False,
              'required-object': False,
              'group': u'Plant Heating and Cooling Equipment'}

    @property
    def name(self):
        """Get name.

        Returns:
            str: the value of `name` or None if not set

        """
        return self["Name"]

    @name.setter
    def name(self, value=None):
        """Corresponds to IDD field `Name`

        Args:
            value (str): value for IDD Field `Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Name"] = value

    @property
    def nominal_capacity(self):
        """Get nominal_capacity.

        Returns:
            float: the value of `nominal_capacity` or None if not set

        """
        return self["Nominal Capacity"]

    @nominal_capacity.setter
    def nominal_capacity(self, value=None):
        """Corresponds to IDD field `Nominal Capacity`

        Args:
            value (float or "Autosize"): value for IDD Field `Nominal Capacity`
                Units: W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Nominal Capacity"] = value

    @property
    def nominal_pumping_power(self):
        """Get nominal_pumping_power.

        Returns:
            float: the value of `nominal_pumping_power` or None if not set

        """
        return self["Nominal Pumping Power"]

    @nominal_pumping_power.setter
    def nominal_pumping_power(self, value=None):
        """Corresponds to IDD field `Nominal Pumping Power`

        Args:
            value (float or "Autosize"): value for IDD Field `Nominal Pumping Power`
                Units: W
                IP-Units: W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Nominal Pumping Power"] = value

    @property
    def chilled_water_inlet_node_name(self):
        """Get chilled_water_inlet_node_name.

        Returns:
            str: the value of `chilled_water_inlet_node_name` or None if not set

        """
        return self["Chilled Water Inlet Node Name"]

    @chilled_water_inlet_node_name.setter
    def chilled_water_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Chilled Water Inlet Node Name`

        Args:
            value (str): value for IDD Field `Chilled Water Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Inlet Node Name"] = value

    @property
    def chilled_water_outlet_node_name(self):
        """Get chilled_water_outlet_node_name.

        Returns:
            str: the value of `chilled_water_outlet_node_name` or None if not set

        """
        return self["Chilled Water Outlet Node Name"]

    @chilled_water_outlet_node_name.setter
    def chilled_water_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Chilled Water Outlet Node Name`

        Args:
            value (str): value for IDD Field `Chilled Water Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Outlet Node Name"] = value

    @property
    def condenser_inlet_node_name(self):
        """Get condenser_inlet_node_name.

        Returns:
            str: the value of `condenser_inlet_node_name` or None if not set

        """
        return self["Condenser Inlet Node Name"]

    @condenser_inlet_node_name.setter
    def condenser_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Condenser Inlet Node Name`

        Args:
            value (str): value for IDD Field `Condenser Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Inlet Node Name"] = value

    @property
    def condenser_outlet_node_name(self):
        """Get condenser_outlet_node_name.

        Returns:
            str: the value of `condenser_outlet_node_name` or None if not set

        """
        return self["Condenser Outlet Node Name"]

    @condenser_outlet_node_name.setter
    def condenser_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Condenser Outlet Node Name`

        Args:
            value (str): value for IDD Field `Condenser Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Outlet Node Name"] = value

    @property
    def minimum_part_load_ratio(self):
        """Get minimum_part_load_ratio.

        Returns:
            float: the value of `minimum_part_load_ratio` or None if not set

        """
        return self["Minimum Part Load Ratio"]

    @minimum_part_load_ratio.setter
    def minimum_part_load_ratio(self, value=None):
        """Corresponds to IDD field `Minimum Part Load Ratio`

        Args:
            value (float): value for IDD Field `Minimum Part Load Ratio`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Minimum Part Load Ratio"] = value

    @property
    def maximum_part_load_ratio(self):
        """Get maximum_part_load_ratio.

        Returns:
            float: the value of `maximum_part_load_ratio` or None if not set

        """
        return self["Maximum Part Load Ratio"]

    @maximum_part_load_ratio.setter
    def maximum_part_load_ratio(self, value=None):
        """Corresponds to IDD field `Maximum Part Load Ratio`

        Args:
            value (float): value for IDD Field `Maximum Part Load Ratio`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Maximum Part Load Ratio"] = value

    @property
    def optimum_part_load_ratio(self):
        """Get optimum_part_load_ratio.

        Returns:
            float: the value of `optimum_part_load_ratio` or None if not set

        """
        return self["Optimum Part Load Ratio"]

    @optimum_part_load_ratio.setter
    def optimum_part_load_ratio(self, value=None):
        """Corresponds to IDD field `Optimum Part Load Ratio`

        Args:
            value (float): value for IDD Field `Optimum Part Load Ratio`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Optimum Part Load Ratio"] = value

    @property
    def design_condenser_inlet_temperature(self):
        """Get design_condenser_inlet_temperature.

        Returns:
            float: the value of `design_condenser_inlet_temperature` or None if not set

        """
        return self["Design Condenser Inlet Temperature"]

    @design_condenser_inlet_temperature.setter
    def design_condenser_inlet_temperature(self, value=30.0):
        """Corresponds to IDD field `Design Condenser Inlet Temperature` Used
        only when condenser flow rate is autosized.

        Args:
            value (float): value for IDD Field `Design Condenser Inlet Temperature`
                Units: C
                Default value: 30.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Condenser Inlet Temperature"] = value

    @property
    def condenser_inlet_temperature_lower_limit(self):
        """Get condenser_inlet_temperature_lower_limit.

        Returns:
            float: the value of `condenser_inlet_temperature_lower_limit` or None if not set

        """
        return self["Condenser Inlet Temperature Lower Limit"]

    @condenser_inlet_temperature_lower_limit.setter
    def condenser_inlet_temperature_lower_limit(self, value=15.0):
        """Corresponds to IDD field `Condenser Inlet Temperature Lower Limit`
        Provides warnings when entering condenser temperature is below minimum.
        Capacity is not adjusted when entering condenser temperature is below
        minimum.

        Args:
            value (float): value for IDD Field `Condenser Inlet Temperature Lower Limit`
                Units: C
                Default value: 15.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Inlet Temperature Lower Limit"] = value

    @property
    def chilled_water_outlet_temperature_lower_limit(self):
        """Get chilled_water_outlet_temperature_lower_limit.

        Returns:
            float: the value of `chilled_water_outlet_temperature_lower_limit` or None if not set

        """
        return self["Chilled Water Outlet Temperature Lower Limit"]

    @chilled_water_outlet_temperature_lower_limit.setter
    def chilled_water_outlet_temperature_lower_limit(self, value=5.0):
        """Corresponds to IDD field `Chilled Water Outlet Temperature Lower
        Limit` Capacity is adjusted when leaving chilled water temperature is
        below minimum.

        Args:
            value (float): value for IDD Field `Chilled Water Outlet Temperature Lower Limit`
                Units: C
                Default value: 5.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Outlet Temperature Lower Limit"] = value

    @property
    def design_chilled_water_flow_rate(self):
        """Get design_chilled_water_flow_rate.

        Returns:
            float: the value of `design_chilled_water_flow_rate` or None if not set

        """
        return self["Design Chilled Water Flow Rate"]

    @design_chilled_water_flow_rate.setter
    def design_chilled_water_flow_rate(self, value=None):
        """Corresponds to IDD field `Design Chilled Water Flow Rate` For
        variable flow this is the max flow & for constant flow this is the
        flow.

        Args:
            value (float or "Autosize"): value for IDD Field `Design Chilled Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Chilled Water Flow Rate"] = value

    @property
    def design_condenser_water_flow_rate(self):
        """Get design_condenser_water_flow_rate.

        Returns:
            float: the value of `design_condenser_water_flow_rate` or None if not set

        """
        return self["Design Condenser Water Flow Rate"]

    @design_condenser_water_flow_rate.setter
    def design_condenser_water_flow_rate(self, value=None):
        """Corresponds to IDD field `Design Condenser Water Flow Rate`

        Args:
            value (float or "Autosize"): value for IDD Field `Design Condenser Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Condenser Water Flow Rate"] = value

    @property
    def chiller_flow_mode(self):
        """Get chiller_flow_mode.

        Returns:
            str: the value of `chiller_flow_mode` or None if not set

        """
        return self["Chiller Flow Mode"]

    @chiller_flow_mode.setter
    def chiller_flow_mode(self, value="NotModulated"):
        """Corresponds to IDD field `Chiller Flow Mode` Select operating mode
        for fluid flow through the chiller. "NotModulated" is for either
        variable or constant pumping with flow controlled by the external plant
        system. "ConstantFlow" is for constant pumping with flow controlled by
        chiller to operate at full design flow rate.
        "LeavingSetpointModulated" is for variable pumping with flow controlled
        by chiller to vary flow to target a leaving temperature setpoint.

        Args:
            value (str): value for IDD Field `Chiller Flow Mode`
                Default value: NotModulated
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Flow Mode"] = value

    @property
    def generator_heat_input_function_of_part_load_ratio_curve_name(self):
        """Get generator_heat_input_function_of_part_load_ratio_curve_name.

        Returns:
            str: the value of `generator_heat_input_function_of_part_load_ratio_curve_name` or None if not set

        """
        return self[
            "Generator Heat Input Function of Part Load Ratio Curve Name"]

    @generator_heat_input_function_of_part_load_ratio_curve_name.setter
    def generator_heat_input_function_of_part_load_ratio_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Generator Heat Input Function of Part Load Ratio Curve Name`
        Table:OneIndependentVariable object can also be used

        Args:
            value (str): value for IDD Field `Generator Heat Input Function of Part Load Ratio Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Generator Heat Input Function of Part Load Ratio Curve Name"] = value

    @property
    def pump_electric_input_function_of_part_load_ratio_curve_name(self):
        """Get pump_electric_input_function_of_part_load_ratio_curve_name.

        Returns:
            str: the value of `pump_electric_input_function_of_part_load_ratio_curve_name` or None if not set

        """
        return self[
            "Pump Electric Input Function of Part Load Ratio Curve Name"]

    @pump_electric_input_function_of_part_load_ratio_curve_name.setter
    def pump_electric_input_function_of_part_load_ratio_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Pump Electric Input Function of Part Load Ratio Curve Name`
        Table:OneIndependentVariable object can also be used

        Args:
            value (str): value for IDD Field `Pump Electric Input Function of Part Load Ratio Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Pump Electric Input Function of Part Load Ratio Curve Name"] = value

    @property
    def generator_inlet_node_name(self):
        """Get generator_inlet_node_name.

        Returns:
            str: the value of `generator_inlet_node_name` or None if not set

        """
        return self["Generator Inlet Node Name"]

    @generator_inlet_node_name.setter
    def generator_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Generator Inlet Node Name` Enter the
        generator inlet node name which connects this chiller to a steam or hot
        water plant, otherwise leave this field blank. Generator nodes are used
        to model heat input to the chiller.

        Args:
            value (str): value for IDD Field `Generator Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Generator Inlet Node Name"] = value

    @property
    def generator_outlet_node_name(self):
        """Get generator_outlet_node_name.

        Returns:
            str: the value of `generator_outlet_node_name` or None if not set

        """
        return self["Generator Outlet Node Name"]

    @generator_outlet_node_name.setter
    def generator_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Generator Outlet Node Name` Enter the
        generator outlet node name which connects this chiller to a steam or
        hot water plant, otherwise leave this field blank. Generator nodes are
        used to model heat input to the chiller.

        Args:
            value (str): value for IDD Field `Generator Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Generator Outlet Node Name"] = value

    @property
    def capacity_correction_function_of_condenser_temperature_curve_name(self):
        """Get
        capacity_correction_function_of_condenser_temperature_curve_name.

        Returns:
            str: the value of `capacity_correction_function_of_condenser_temperature_curve_name` or None if not set

        """
        return self[
            "Capacity Correction Function of Condenser Temperature Curve Name"]

    @capacity_correction_function_of_condenser_temperature_curve_name.setter
    def capacity_correction_function_of_condenser_temperature_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Capacity Correction Function of Condenser Temperature Curve Name`
        Table:OneIndependentVariable object can also be used
        Curve which shows the change in normailized capacity to changes in condenser temperature.

        Args:
            value (str): value for IDD Field `Capacity Correction Function of Condenser Temperature Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Capacity Correction Function of Condenser Temperature Curve Name"] = value

    @property
    def capacity_correction_function_of_chilled_water_temperature_curve_name(
            self):
        """Get capacity_correction_function_of_chilled_water_temperature_curve_
        name.

        Returns:
            str: the value of `capacity_correction_function_of_chilled_water_temperature_curve_name` or None if not set

        """
        return self[
            "Capacity Correction Function of Chilled Water Temperature Curve Name"]

    @capacity_correction_function_of_chilled_water_temperature_curve_name.setter
    def capacity_correction_function_of_chilled_water_temperature_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Capacity Correction Function of Chilled Water Temperature Curve Name`
        Table:OneIndependentVariable object can also be used
        Curve which shows the change in normailized capacity to changes in leaving chilled water temperature.

        Args:
            value (str): value for IDD Field `Capacity Correction Function of Chilled Water Temperature Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Capacity Correction Function of Chilled Water Temperature Curve Name"] = value

    @property
    def capacity_correction_function_of_generator_temperature_curve_name(self):
        """Get
        capacity_correction_function_of_generator_temperature_curve_name.

        Returns:
            str: the value of `capacity_correction_function_of_generator_temperature_curve_name` or None if not set

        """
        return self[
            "Capacity Correction Function of Generator Temperature Curve Name"]

    @capacity_correction_function_of_generator_temperature_curve_name.setter
    def capacity_correction_function_of_generator_temperature_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Capacity Correction Function of Generator Temperature Curve Name`
        Table:OneIndependentVariable object can also be used
        Used when generator fluid type is hot water
        Curve which shows the change in normailized capacity to changes in generator temperature.

        Args:
            value (str): value for IDD Field `Capacity Correction Function of Generator Temperature Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Capacity Correction Function of Generator Temperature Curve Name"] = value

    @property
    def generator_heat_input_correction_function_of_condenser_temperature_curve_name(
            self):
        """Get generator_heat_input_correction_function_of_condenser_temperatur
        e_curve_name.

        Returns:
            str: the value of `generator_heat_input_correction_function_of_condenser_temperature_curve_name` or None if not set

        """
        return self[
            "Generator Heat Input Correction Function of Condenser Temperature Curve Name"]

    @generator_heat_input_correction_function_of_condenser_temperature_curve_name.setter
    def generator_heat_input_correction_function_of_condenser_temperature_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Generator Heat Input Correction Function of Condenser Temperature Curve Name`
        Table:OneIndependentVariable object can also be used
        Curve which shows the change in normailized heat input to changes in condenser temperature.

        Args:
            value (str): value for IDD Field `Generator Heat Input Correction Function of Condenser Temperature Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Generator Heat Input Correction Function of Condenser Temperature Curve Name"] = value

    @property
    def generator_heat_input_correction_function_of_chilled_water_temperature_curve_name(
            self):
        """Get generator_heat_input_correction_function_of_chilled_water_temper
        ature_curve_name.

        Returns:
            str: the value of `generator_heat_input_correction_function_of_chilled_water_temperature_curve_name` or None if not set

        """
        return self[
            "Generator Heat Input Correction Function of Chilled Water Temperature Curve Name"]

    @generator_heat_input_correction_function_of_chilled_water_temperature_curve_name.setter
    def generator_heat_input_correction_function_of_chilled_water_temperature_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Generator Heat Input Correction Function of Chilled Water Temperature Curve Name`
        Table:OneIndependentVariable object can also be used
        Curve which shows the change in normailized heat input to changes in leaving chilled water temperature.

        Args:
            value (str): value for IDD Field `Generator Heat Input Correction Function of Chilled Water Temperature Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Generator Heat Input Correction Function of Chilled Water Temperature Curve Name"] = value

    @property
    def generator_heat_source_type(self):
        """Get generator_heat_source_type.

        Returns:
            str: the value of `generator_heat_source_type` or None if not set

        """
        return self["Generator Heat Source Type"]

    @generator_heat_source_type.setter
    def generator_heat_source_type(self, value="Steam"):
        """Corresponds to IDD field `Generator Heat Source Type` The Generator
        side of the chiller can be connected to a hot water or steam plant
        where the generator inlet and outlet nodes are connected to a plant
        loop. If the generator is not connected to a plant loop, and the
        generator inlet/outlet nodes are not used, this field should be
        specified as steam or left blank. When a plant is not used, the model
        assumes steam as the heat source.

        Args:
            value (str): value for IDD Field `Generator Heat Source Type`
                Default value: Steam
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Generator Heat Source Type"] = value

    @property
    def design_generator_fluid_flow_rate(self):
        """Get design_generator_fluid_flow_rate.

        Returns:
            float: the value of `design_generator_fluid_flow_rate` or None if not set

        """
        return self["Design Generator Fluid Flow Rate"]

    @design_generator_fluid_flow_rate.setter
    def design_generator_fluid_flow_rate(self, value=None):
        """Corresponds to IDD field `Design Generator Fluid Flow Rate` For
        variable flow this is the max flow and for constant flow this is the
        flow.

        Args:
            value (float or "Autosize"): value for IDD Field `Design Generator Fluid Flow Rate`
                Units: m3/s
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Generator Fluid Flow Rate"] = value

    @property
    def temperature_lower_limit_generator_inlet(self):
        """Get temperature_lower_limit_generator_inlet.

        Returns:
            float: the value of `temperature_lower_limit_generator_inlet` or None if not set

        """
        return self["Temperature Lower Limit Generator Inlet"]

    @temperature_lower_limit_generator_inlet.setter
    def temperature_lower_limit_generator_inlet(self, value=None):
        """Corresponds to IDD field `Temperature Lower Limit Generator Inlet`
        Provides warnings when entering generator temperature is below minimum.
        Capacity is not adjusted when entering generator temperature is below
        minimum.

        Args:
            value (float): value for IDD Field `Temperature Lower Limit Generator Inlet`
                Units: C
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Temperature Lower Limit Generator Inlet"] = value

    @property
    def degree_of_subcooling_in_steam_generator(self):
        """Get degree_of_subcooling_in_steam_generator.

        Returns:
            float: the value of `degree_of_subcooling_in_steam_generator` or None if not set

        """
        return self["Degree of Subcooling in Steam Generator"]

    @degree_of_subcooling_in_steam_generator.setter
    def degree_of_subcooling_in_steam_generator(self, value=1.0):
        """Corresponds to IDD field `Degree of Subcooling in Steam Generator`
        This field is not used when the generator inlet/outlet nodes are not
        specified or the generator is connected to a hot water loop.

        Args:
            value (float): value for IDD Field `Degree of Subcooling in Steam Generator`
                Units: C
                Default value: 1.0
                value <= 20.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Degree of Subcooling in Steam Generator"] = value

    @property
    def degree_of_subcooling_in_steam_condensate_loop(self):
        """Get degree_of_subcooling_in_steam_condensate_loop.

        Returns:
            float: the value of `degree_of_subcooling_in_steam_condensate_loop` or None if not set

        """
        return self["Degree of Subcooling in Steam Condensate Loop"]

    @degree_of_subcooling_in_steam_condensate_loop.setter
    def degree_of_subcooling_in_steam_condensate_loop(self, value=None):
        """Corresponds to IDD field `Degree of Subcooling in Steam Condensate
        Loop` This field is not used when the generator inlet/outlet nodes are
        not specified or the generator is connected to a hot water loop.

        Args:
            value (float): value for IDD Field `Degree of Subcooling in Steam Condensate Loop`
                Units: C
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Degree of Subcooling in Steam Condensate Loop"] = value

    @property
    def sizing_factor(self):
        """Get sizing_factor.

        Returns:
            float: the value of `sizing_factor` or None if not set

        """
        return self["Sizing Factor"]

    @sizing_factor.setter
    def sizing_factor(self, value=1.0):
        """Corresponds to IDD field `Sizing Factor` Multiplies the autosized
        capacity and flow rates.

        Args:
            value (float): value for IDD Field `Sizing Factor`
                Default value: 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Sizing Factor"] = value




class ChillerAbsorption(DataObject):

    """ Corresponds to IDD object `Chiller:Absorption`
        This indirect absorption chiller model is the empirical model from the
        Building Loads and System Thermodynamics (BLAST) program.  Chiller
        performance curves are generated by fitting catalog data to third order
        polynomial equations.  Two sets of coefficients are required.
    """
    schema = {'min-fields': 23,
              'name': u'Chiller:Absorption',
              'pyname': u'ChillerAbsorption',
              'format': None,
              'fields': OrderedDict([(u'name',
                                      {'name': u'Name',
                                       'pyname': u'name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'nominal capacity',
                                      {'name': u'Nominal Capacity',
                                       'pyname': u'nominal_capacity',
                                       'minimum>': 0.0,
                                       'required-field': True,
                                       'autosizable': True,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'W'}),
                                     (u'nominal pumping power',
                                      {'name': u'Nominal Pumping Power',
                                       'pyname': u'nominal_pumping_power',
                                       'required-field': True,
                                       'autosizable': True,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'W'}),
                                     (u'chilled water inlet node name',
                                      {'name': u'Chilled Water Inlet Node Name',
                                       'pyname': u'chilled_water_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'chilled water outlet node name',
                                      {'name': u'Chilled Water Outlet Node Name',
                                       'pyname': u'chilled_water_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'condenser inlet node name',
                                      {'name': u'Condenser Inlet Node Name',
                                       'pyname': u'condenser_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'condenser outlet node name',
                                      {'name': u'Condenser Outlet Node Name',
                                       'pyname': u'condenser_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'minimum part load ratio',
                                      {'name': u'Minimum Part Load Ratio',
                                       'pyname': u'minimum_part_load_ratio',
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'maximum part load ratio',
                                      {'name': u'Maximum Part Load Ratio',
                                       'pyname': u'maximum_part_load_ratio',
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'optimum part load ratio',
                                      {'name': u'Optimum Part Load Ratio',
                                       'pyname': u'optimum_part_load_ratio',
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'design condenser inlet temperature',
                                      {'name': u'Design Condenser Inlet Temperature',
                                       'pyname': u'design_condenser_inlet_temperature',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'C'}),
                                     (u'design chilled water flow rate',
                                      {'name': u'Design Chilled Water Flow Rate',
                                       'pyname': u'design_chilled_water_flow_rate',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': True,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'm3/s'}),
                                     (u'design condenser water flow rate',
                                      {'name': u'Design Condenser Water Flow Rate',
                                       'pyname': u'design_condenser_water_flow_rate',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': True,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'm3/s'}),
                                     (u'coefficient 1 of the hot water or steam use part load ratio curve',
                                      {'name': u'Coefficient 1 of the Hot Water or Steam Use Part Load Ratio Curve',
                                       'pyname': u'coefficient_1_of_the_hot_water_or_steam_use_part_load_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 2 of the hot water or steam use part load ratio curve',
                                      {'name': u'Coefficient 2 of the Hot Water or Steam Use Part Load Ratio Curve',
                                       'pyname': u'coefficient_2_of_the_hot_water_or_steam_use_part_load_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 3 of the hot water or steam use part load ratio curve',
                                      {'name': u'Coefficient 3 of the Hot Water or Steam Use Part Load Ratio Curve',
                                       'pyname': u'coefficient_3_of_the_hot_water_or_steam_use_part_load_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 1 of the pump electric use part load ratio curve',
                                      {'name': u'Coefficient 1 of the Pump Electric Use Part Load Ratio Curve',
                                       'pyname': u'coefficient_1_of_the_pump_electric_use_part_load_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 2 of the pump electric use part load ratio curve',
                                      {'name': u'Coefficient 2 of the Pump Electric Use Part Load Ratio Curve',
                                       'pyname': u'coefficient_2_of_the_pump_electric_use_part_load_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 3 of the pump electric use part load ratio curve',
                                      {'name': u'Coefficient 3 of the Pump Electric Use Part Load Ratio Curve',
                                       'pyname': u'coefficient_3_of_the_pump_electric_use_part_load_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'chilled water outlet temperature lower limit',
                                      {'name': u'Chilled Water Outlet Temperature Lower Limit',
                                       'pyname': u'chilled_water_outlet_temperature_lower_limit',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'C'}),
                                     (u'generator inlet node name',
                                      {'name': u'Generator Inlet Node Name',
                                       'pyname': u'generator_inlet_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'generator outlet node name',
                                      {'name': u'Generator Outlet Node Name',
                                       'pyname': u'generator_outlet_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'chiller flow mode',
                                      {'name': u'Chiller Flow Mode',
                                       'pyname': u'chiller_flow_mode',
                                       'default': u'NotModulated',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ConstantFlow',
                                                           u'LeavingSetpointModulated',
                                                           u'NotModulated'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'generator heat source type',
                                      {'name': u'Generator Heat Source Type',
                                       'pyname': u'generator_heat_source_type',
                                       'default': u'Steam',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'Steam',
                                                           u'HotWater'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'design generator fluid flow rate',
                                      {'name': u'Design Generator Fluid Flow Rate',
                                       'pyname': u'design_generator_fluid_flow_rate',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': True,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'm3/s'}),
                                     (u'degree of subcooling in steam generator',
                                      {'name': u'Degree of Subcooling in Steam Generator',
                                       'pyname': u'degree_of_subcooling_in_steam_generator',
                                       'default': 1.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'C'}),
                                     (u'sizing factor',
                                      {'name': u'Sizing Factor',
                                       'pyname': u'sizing_factor',
                                       'default': 1.0,
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'})]),
              'extensible-fields': OrderedDict(),
              'unique-object': False,
              'required-object': False,
              'group': u'Plant Heating and Cooling Equipment'}

    @property
    def name(self):
        """Get name.

        Returns:
            str: the value of `name` or None if not set

        """
        return self["Name"]

    @name.setter
    def name(self, value=None):
        """Corresponds to IDD field `Name`

        Args:
            value (str): value for IDD Field `Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Name"] = value

    @property
    def nominal_capacity(self):
        """Get nominal_capacity.

        Returns:
            float: the value of `nominal_capacity` or None if not set

        """
        return self["Nominal Capacity"]

    @nominal_capacity.setter
    def nominal_capacity(self, value=None):
        """Corresponds to IDD field `Nominal Capacity`

        Args:
            value (float or "Autosize"): value for IDD Field `Nominal Capacity`
                Units: W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Nominal Capacity"] = value

    @property
    def nominal_pumping_power(self):
        """Get nominal_pumping_power.

        Returns:
            float: the value of `nominal_pumping_power` or None if not set

        """
        return self["Nominal Pumping Power"]

    @nominal_pumping_power.setter
    def nominal_pumping_power(self, value=None):
        """Corresponds to IDD field `Nominal Pumping Power`

        Args:
            value (float or "Autosize"): value for IDD Field `Nominal Pumping Power`
                Units: W
                IP-Units: W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Nominal Pumping Power"] = value

    @property
    def chilled_water_inlet_node_name(self):
        """Get chilled_water_inlet_node_name.

        Returns:
            str: the value of `chilled_water_inlet_node_name` or None if not set

        """
        return self["Chilled Water Inlet Node Name"]

    @chilled_water_inlet_node_name.setter
    def chilled_water_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Chilled Water Inlet Node Name`

        Args:
            value (str): value for IDD Field `Chilled Water Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Inlet Node Name"] = value

    @property
    def chilled_water_outlet_node_name(self):
        """Get chilled_water_outlet_node_name.

        Returns:
            str: the value of `chilled_water_outlet_node_name` or None if not set

        """
        return self["Chilled Water Outlet Node Name"]

    @chilled_water_outlet_node_name.setter
    def chilled_water_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Chilled Water Outlet Node Name`

        Args:
            value (str): value for IDD Field `Chilled Water Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Outlet Node Name"] = value

    @property
    def condenser_inlet_node_name(self):
        """Get condenser_inlet_node_name.

        Returns:
            str: the value of `condenser_inlet_node_name` or None if not set

        """
        return self["Condenser Inlet Node Name"]

    @condenser_inlet_node_name.setter
    def condenser_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Condenser Inlet Node Name`

        Args:
            value (str): value for IDD Field `Condenser Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Inlet Node Name"] = value

    @property
    def condenser_outlet_node_name(self):
        """Get condenser_outlet_node_name.

        Returns:
            str: the value of `condenser_outlet_node_name` or None if not set

        """
        return self["Condenser Outlet Node Name"]

    @condenser_outlet_node_name.setter
    def condenser_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Condenser Outlet Node Name`

        Args:
            value (str): value for IDD Field `Condenser Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Outlet Node Name"] = value

    @property
    def minimum_part_load_ratio(self):
        """Get minimum_part_load_ratio.

        Returns:
            float: the value of `minimum_part_load_ratio` or None if not set

        """
        return self["Minimum Part Load Ratio"]

    @minimum_part_load_ratio.setter
    def minimum_part_load_ratio(self, value=None):
        """Corresponds to IDD field `Minimum Part Load Ratio`

        Args:
            value (float): value for IDD Field `Minimum Part Load Ratio`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Minimum Part Load Ratio"] = value

    @property
    def maximum_part_load_ratio(self):
        """Get maximum_part_load_ratio.

        Returns:
            float: the value of `maximum_part_load_ratio` or None if not set

        """
        return self["Maximum Part Load Ratio"]

    @maximum_part_load_ratio.setter
    def maximum_part_load_ratio(self, value=None):
        """Corresponds to IDD field `Maximum Part Load Ratio`

        Args:
            value (float): value for IDD Field `Maximum Part Load Ratio`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Maximum Part Load Ratio"] = value

    @property
    def optimum_part_load_ratio(self):
        """Get optimum_part_load_ratio.

        Returns:
            float: the value of `optimum_part_load_ratio` or None if not set

        """
        return self["Optimum Part Load Ratio"]

    @optimum_part_load_ratio.setter
    def optimum_part_load_ratio(self, value=None):
        """Corresponds to IDD field `Optimum Part Load Ratio`

        Args:
            value (float): value for IDD Field `Optimum Part Load Ratio`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Optimum Part Load Ratio"] = value

    @property
    def design_condenser_inlet_temperature(self):
        """Get design_condenser_inlet_temperature.

        Returns:
            float: the value of `design_condenser_inlet_temperature` or None if not set

        """
        return self["Design Condenser Inlet Temperature"]

    @design_condenser_inlet_temperature.setter
    def design_condenser_inlet_temperature(self, value=None):
        """Corresponds to IDD field `Design Condenser Inlet Temperature`

        Args:
            value (float): value for IDD Field `Design Condenser Inlet Temperature`
                Units: C
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Condenser Inlet Temperature"] = value

    @property
    def design_chilled_water_flow_rate(self):
        """Get design_chilled_water_flow_rate.

        Returns:
            float: the value of `design_chilled_water_flow_rate` or None if not set

        """
        return self["Design Chilled Water Flow Rate"]

    @design_chilled_water_flow_rate.setter
    def design_chilled_water_flow_rate(self, value=None):
        """Corresponds to IDD field `Design Chilled Water Flow Rate` For
        variable volume this is the max flow & for constant flow this is the
        flow.

        Args:
            value (float or "Autosize"): value for IDD Field `Design Chilled Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Chilled Water Flow Rate"] = value

    @property
    def design_condenser_water_flow_rate(self):
        """Get design_condenser_water_flow_rate.

        Returns:
            float: the value of `design_condenser_water_flow_rate` or None if not set

        """
        return self["Design Condenser Water Flow Rate"]

    @design_condenser_water_flow_rate.setter
    def design_condenser_water_flow_rate(self, value=None):
        """Corresponds to IDD field `Design Condenser Water Flow Rate` The
        steam use coefficients below specify the steam use as a fraction of
        chiller operating capacity.

        Args:
            value (float or "Autosize"): value for IDD Field `Design Condenser Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Condenser Water Flow Rate"] = value

    @property
    def coefficient_1_of_the_hot_water_or_steam_use_part_load_ratio_curve(
            self):
        """Get
        coefficient_1_of_the_hot_water_or_steam_use_part_load_ratio_curve.

        Returns:
            float: the value of `coefficient_1_of_the_hot_water_or_steam_use_part_load_ratio_curve` or None if not set

        """
        return self[
            "Coefficient 1 of the Hot Water or Steam Use Part Load Ratio Curve"]

    @coefficient_1_of_the_hot_water_or_steam_use_part_load_ratio_curve.setter
    def coefficient_1_of_the_hot_water_or_steam_use_part_load_ratio_curve(
            self,
            value=None):
        """Corresponds to IDD field `Coefficient 1 of the Hot Water or Steam
        Use Part Load Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 1 of the Hot Water or Steam Use Part Load Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Coefficient 1 of the Hot Water or Steam Use Part Load Ratio Curve"] = value

    @property
    def coefficient_2_of_the_hot_water_or_steam_use_part_load_ratio_curve(
            self):
        """Get
        coefficient_2_of_the_hot_water_or_steam_use_part_load_ratio_curve.

        Returns:
            float: the value of `coefficient_2_of_the_hot_water_or_steam_use_part_load_ratio_curve` or None if not set

        """
        return self[
            "Coefficient 2 of the Hot Water or Steam Use Part Load Ratio Curve"]

    @coefficient_2_of_the_hot_water_or_steam_use_part_load_ratio_curve.setter
    def coefficient_2_of_the_hot_water_or_steam_use_part_load_ratio_curve(
            self,
            value=None):
        """Corresponds to IDD field `Coefficient 2 of the Hot Water or Steam
        Use Part Load Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 2 of the Hot Water or Steam Use Part Load Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Coefficient 2 of the Hot Water or Steam Use Part Load Ratio Curve"] = value

    @property
    def coefficient_3_of_the_hot_water_or_steam_use_part_load_ratio_curve(
            self):
        """Get
        coefficient_3_of_the_hot_water_or_steam_use_part_load_ratio_curve.

        Returns:
            float: the value of `coefficient_3_of_the_hot_water_or_steam_use_part_load_ratio_curve` or None if not set

        """
        return self[
            "Coefficient 3 of the Hot Water or Steam Use Part Load Ratio Curve"]

    @coefficient_3_of_the_hot_water_or_steam_use_part_load_ratio_curve.setter
    def coefficient_3_of_the_hot_water_or_steam_use_part_load_ratio_curve(
            self,
            value=None):
        """Corresponds to IDD field `Coefficient 3 of the Hot Water or Steam
        Use Part Load Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 3 of the Hot Water or Steam Use Part Load Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Coefficient 3 of the Hot Water or Steam Use Part Load Ratio Curve"] = value

    @property
    def coefficient_1_of_the_pump_electric_use_part_load_ratio_curve(self):
        """Get coefficient_1_of_the_pump_electric_use_part_load_ratio_curve.

        Returns:
            float: the value of `coefficient_1_of_the_pump_electric_use_part_load_ratio_curve` or None if not set

        """
        return self[
            "Coefficient 1 of the Pump Electric Use Part Load Ratio Curve"]

    @coefficient_1_of_the_pump_electric_use_part_load_ratio_curve.setter
    def coefficient_1_of_the_pump_electric_use_part_load_ratio_curve(
            self,
            value=None):
        """Corresponds to IDD field `Coefficient 1 of the Pump Electric Use
        Part Load Ratio Curve` The pump electric use coefficients specify the
        pumping power as a Fraction of Nominal pumping power.

        Args:
            value (float): value for IDD Field `Coefficient 1 of the Pump Electric Use Part Load Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Coefficient 1 of the Pump Electric Use Part Load Ratio Curve"] = value

    @property
    def coefficient_2_of_the_pump_electric_use_part_load_ratio_curve(self):
        """Get coefficient_2_of_the_pump_electric_use_part_load_ratio_curve.

        Returns:
            float: the value of `coefficient_2_of_the_pump_electric_use_part_load_ratio_curve` or None if not set

        """
        return self[
            "Coefficient 2 of the Pump Electric Use Part Load Ratio Curve"]

    @coefficient_2_of_the_pump_electric_use_part_load_ratio_curve.setter
    def coefficient_2_of_the_pump_electric_use_part_load_ratio_curve(
            self,
            value=None):
        """Corresponds to IDD field `Coefficient 2 of the Pump Electric Use
        Part Load Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 2 of the Pump Electric Use Part Load Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Coefficient 2 of the Pump Electric Use Part Load Ratio Curve"] = value

    @property
    def coefficient_3_of_the_pump_electric_use_part_load_ratio_curve(self):
        """Get coefficient_3_of_the_pump_electric_use_part_load_ratio_curve.

        Returns:
            float: the value of `coefficient_3_of_the_pump_electric_use_part_load_ratio_curve` or None if not set

        """
        return self[
            "Coefficient 3 of the Pump Electric Use Part Load Ratio Curve"]

    @coefficient_3_of_the_pump_electric_use_part_load_ratio_curve.setter
    def coefficient_3_of_the_pump_electric_use_part_load_ratio_curve(
            self,
            value=None):
        """Corresponds to IDD field `Coefficient 3 of the Pump Electric Use
        Part Load Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 3 of the Pump Electric Use Part Load Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Coefficient 3 of the Pump Electric Use Part Load Ratio Curve"] = value

    @property
    def chilled_water_outlet_temperature_lower_limit(self):
        """Get chilled_water_outlet_temperature_lower_limit.

        Returns:
            float: the value of `chilled_water_outlet_temperature_lower_limit` or None if not set

        """
        return self["Chilled Water Outlet Temperature Lower Limit"]

    @chilled_water_outlet_temperature_lower_limit.setter
    def chilled_water_outlet_temperature_lower_limit(self, value=None):
        """Corresponds to IDD field `Chilled Water Outlet Temperature Lower
        Limit`

        Args:
            value (float): value for IDD Field `Chilled Water Outlet Temperature Lower Limit`
                Units: C
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Outlet Temperature Lower Limit"] = value

    @property
    def generator_inlet_node_name(self):
        """Get generator_inlet_node_name.

        Returns:
            str: the value of `generator_inlet_node_name` or None if not set

        """
        return self["Generator Inlet Node Name"]

    @generator_inlet_node_name.setter
    def generator_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Generator Inlet Node Name`

        Args:
            value (str): value for IDD Field `Generator Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Generator Inlet Node Name"] = value

    @property
    def generator_outlet_node_name(self):
        """Get generator_outlet_node_name.

        Returns:
            str: the value of `generator_outlet_node_name` or None if not set

        """
        return self["Generator Outlet Node Name"]

    @generator_outlet_node_name.setter
    def generator_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Generator Outlet Node Name`

        Args:
            value (str): value for IDD Field `Generator Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Generator Outlet Node Name"] = value

    @property
    def chiller_flow_mode(self):
        """Get chiller_flow_mode.

        Returns:
            str: the value of `chiller_flow_mode` or None if not set

        """
        return self["Chiller Flow Mode"]

    @chiller_flow_mode.setter
    def chiller_flow_mode(self, value="NotModulated"):
        """Corresponds to IDD field `Chiller Flow Mode` Select operating mode
        for fluid flow through the chiller. "NotModulated" is for either
        variable or constant pumping with flow controlled by the external plant
        system. "ConstantFlow" is for constant pumping with flow controlled by
        chiller to operate at full design flow rate.
        "LeavingSetpointModulated" is for variable pumping with flow controlled
        by chiller to vary flow to target a leaving temperature setpoint.

        Args:
            value (str): value for IDD Field `Chiller Flow Mode`
                Default value: NotModulated
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Flow Mode"] = value

    @property
    def generator_heat_source_type(self):
        """Get generator_heat_source_type.

        Returns:
            str: the value of `generator_heat_source_type` or None if not set

        """
        return self["Generator Heat Source Type"]

    @generator_heat_source_type.setter
    def generator_heat_source_type(self, value="Steam"):
        """Corresponds to IDD field `Generator Heat Source Type` The Generator
        side of the chiller can be connected to a hot water or steam plant
        where the generator inlet and outlet nodes are connected to a plant
        loop. If the generator is not connected to a plant loop, and the
        generator inlet/outlet nodes are not used, this field should be
        specified as steam or left blank. When a plant is not used, the model
        assumes steam as the heat source.

        Args:
            value (str): value for IDD Field `Generator Heat Source Type`
                Default value: Steam
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Generator Heat Source Type"] = value

    @property
    def design_generator_fluid_flow_rate(self):
        """Get design_generator_fluid_flow_rate.

        Returns:
            float: the value of `design_generator_fluid_flow_rate` or None if not set

        """
        return self["Design Generator Fluid Flow Rate"]

    @design_generator_fluid_flow_rate.setter
    def design_generator_fluid_flow_rate(self, value=None):
        """Corresponds to IDD field `Design Generator Fluid Flow Rate`

        Args:
            value (float or "Autosize"): value for IDD Field `Design Generator Fluid Flow Rate`
                Units: m3/s
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Generator Fluid Flow Rate"] = value

    @property
    def degree_of_subcooling_in_steam_generator(self):
        """Get degree_of_subcooling_in_steam_generator.

        Returns:
            float: the value of `degree_of_subcooling_in_steam_generator` or None if not set

        """
        return self["Degree of Subcooling in Steam Generator"]

    @degree_of_subcooling_in_steam_generator.setter
    def degree_of_subcooling_in_steam_generator(self, value=1.0):
        """Corresponds to IDD field `Degree of Subcooling in Steam Generator`
        This field is not used when the generator inlet/outlet nodes are not
        specified or the generator is connected to a hot water loop.

        Args:
            value (float): value for IDD Field `Degree of Subcooling in Steam Generator`
                Units: C
                Default value: 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Degree of Subcooling in Steam Generator"] = value

    @property
    def sizing_factor(self):
        """Get sizing_factor.

        Returns:
            float: the value of `sizing_factor` or None if not set

        """
        return self["Sizing Factor"]

    @sizing_factor.setter
    def sizing_factor(self, value=1.0):
        """Corresponds to IDD field `Sizing Factor` Multiplies the autosized
        capacity and flow rates.

        Args:
            value (float): value for IDD Field `Sizing Factor`
                Default value: 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Sizing Factor"] = value




class ChillerConstantCop(DataObject):

    """ Corresponds to IDD object `Chiller:ConstantCOP`
        This constant COP chiller model provides a means of quickly specifying a
        Chiller where performance data is not available.
    """
    schema = {'min-fields': 12,
              'name': u'Chiller:ConstantCOP',
              'pyname': u'ChillerConstantCop',
              'format': None,
              'fields': OrderedDict([(u'name',
                                      {'name': u'Name',
                                       'pyname': u'name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'nominal capacity',
                                      {'name': u'Nominal Capacity',
                                       'pyname': u'nominal_capacity',
                                       'required-field': True,
                                       'autosizable': True,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'W'}),
                                     (u'nominal cop',
                                      {'name': u'Nominal COP',
                                       'pyname': u'nominal_cop',
                                       'minimum>': 0.0,
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'W/W'}),
                                     (u'design chilled water flow rate',
                                      {'name': u'Design Chilled Water Flow Rate',
                                       'pyname': u'design_chilled_water_flow_rate',
                                       'required-field': False,
                                       'autosizable': True,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'm3/s'}),
                                     (u'design condenser water flow rate',
                                      {'name': u'Design Condenser Water Flow Rate',
                                       'pyname': u'design_condenser_water_flow_rate',
                                       'required-field': False,
                                       'autosizable': True,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'm3/s'}),
                                     (u'chilled water inlet node name',
                                      {'name': u'Chilled Water Inlet Node Name',
                                       'pyname': u'chilled_water_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'chilled water outlet node name',
                                      {'name': u'Chilled Water Outlet Node Name',
                                       'pyname': u'chilled_water_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'condenser inlet node name',
                                      {'name': u'Condenser Inlet Node Name',
                                       'pyname': u'condenser_inlet_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'condenser outlet node name',
                                      {'name': u'Condenser Outlet Node Name',
                                       'pyname': u'condenser_outlet_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'condenser type',
                                      {'name': u'Condenser Type',
                                       'pyname': u'condenser_type',
                                       'default': u'AirCooled',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'AirCooled',
                                                           u'WaterCooled',
                                                           u'EvaporativelyCooled'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'chiller flow mode',
                                      {'name': u'Chiller Flow Mode',
                                       'pyname': u'chiller_flow_mode',
                                       'default': u'NotModulated',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ConstantFlow',
                                                           u'LeavingSetpointModulated',
                                                           u'NotModulated'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'sizing factor',
                                      {'name': u'Sizing Factor',
                                       'pyname': u'sizing_factor',
                                       'default': 1.0,
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'basin heater capacity',
                                      {'name': u'Basin Heater Capacity',
                                       'pyname': u'basin_heater_capacity',
                                       'default': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'W/K'}),
                                     (u'basin heater setpoint temperature',
                                      {'name': u'Basin Heater Setpoint Temperature',
                                       'pyname': u'basin_heater_setpoint_temperature',
                                       'default': 2.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 2.0,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'C'}),
                                     (u'basin heater operating schedule name',
                                      {'name': u'Basin Heater Operating Schedule Name',
                                       'pyname': u'basin_heater_operating_schedule_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'})]),
              'extensible-fields': OrderedDict(),
              'unique-object': False,
              'required-object': False,
              'group': u'Plant Heating and Cooling Equipment'}

    @property
    def name(self):
        """Get name.

        Returns:
            str: the value of `name` or None if not set

        """
        return self["Name"]

    @name.setter
    def name(self, value=None):
        """Corresponds to IDD field `Name`

        Args:
            value (str): value for IDD Field `Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Name"] = value

    @property
    def nominal_capacity(self):
        """Get nominal_capacity.

        Returns:
            float: the value of `nominal_capacity` or None if not set

        """
        return self["Nominal Capacity"]

    @nominal_capacity.setter
    def nominal_capacity(self, value=None):
        """Corresponds to IDD field `Nominal Capacity`

        Args:
            value (float or "Autosize"): value for IDD Field `Nominal Capacity`
                Units: W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Nominal Capacity"] = value

    @property
    def nominal_cop(self):
        """Get nominal_cop.

        Returns:
            float: the value of `nominal_cop` or None if not set

        """
        return self["Nominal COP"]

    @nominal_cop.setter
    def nominal_cop(self, value=None):
        """Corresponds to IDD field `Nominal COP`

        Args:
            value (float): value for IDD Field `Nominal COP`
                Units: W/W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Nominal COP"] = value

    @property
    def design_chilled_water_flow_rate(self):
        """Get design_chilled_water_flow_rate.

        Returns:
            float: the value of `design_chilled_water_flow_rate` or None if not set

        """
        return self["Design Chilled Water Flow Rate"]

    @design_chilled_water_flow_rate.setter
    def design_chilled_water_flow_rate(self, value=None):
        """Corresponds to IDD field `Design Chilled Water Flow Rate` For
        variable volume this is the maximum flow and for constant flow this is
        the flow.

        Args:
            value (float or "Autosize"): value for IDD Field `Design Chilled Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Chilled Water Flow Rate"] = value

    @property
    def design_condenser_water_flow_rate(self):
        """Get design_condenser_water_flow_rate.

        Returns:
            float: the value of `design_condenser_water_flow_rate` or None if not set

        """
        return self["Design Condenser Water Flow Rate"]

    @design_condenser_water_flow_rate.setter
    def design_condenser_water_flow_rate(self, value=None):
        """  Corresponds to IDD field `Design Condenser Water Flow Rate`
        This field is not used for Condenser Type = AirCooled or EvaporativelyCooled

        Args:
            value (float or "Autosize"): value for IDD Field `Design Condenser Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Design Condenser Water Flow Rate"] = value

    @property
    def chilled_water_inlet_node_name(self):
        """Get chilled_water_inlet_node_name.

        Returns:
            str: the value of `chilled_water_inlet_node_name` or None if not set

        """
        return self["Chilled Water Inlet Node Name"]

    @chilled_water_inlet_node_name.setter
    def chilled_water_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Chilled Water Inlet Node Name`

        Args:
            value (str): value for IDD Field `Chilled Water Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Inlet Node Name"] = value

    @property
    def chilled_water_outlet_node_name(self):
        """Get chilled_water_outlet_node_name.

        Returns:
            str: the value of `chilled_water_outlet_node_name` or None if not set

        """
        return self["Chilled Water Outlet Node Name"]

    @chilled_water_outlet_node_name.setter
    def chilled_water_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Chilled Water Outlet Node Name`

        Args:
            value (str): value for IDD Field `Chilled Water Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Outlet Node Name"] = value

    @property
    def condenser_inlet_node_name(self):
        """Get condenser_inlet_node_name.

        Returns:
            str: the value of `condenser_inlet_node_name` or None if not set

        """
        return self["Condenser Inlet Node Name"]

    @condenser_inlet_node_name.setter
    def condenser_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Condenser Inlet Node Name`

        Args:
            value (str): value for IDD Field `Condenser Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Inlet Node Name"] = value

    @property
    def condenser_outlet_node_name(self):
        """Get condenser_outlet_node_name.

        Returns:
            str: the value of `condenser_outlet_node_name` or None if not set

        """
        return self["Condenser Outlet Node Name"]

    @condenser_outlet_node_name.setter
    def condenser_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Condenser Outlet Node Name`

        Args:
            value (str): value for IDD Field `Condenser Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Outlet Node Name"] = value

    @property
    def condenser_type(self):
        """Get condenser_type.

        Returns:
            str: the value of `condenser_type` or None if not set

        """
        return self["Condenser Type"]

    @condenser_type.setter
    def condenser_type(self, value="AirCooled"):
        """Corresponds to IDD field `Condenser Type`

        Args:
            value (str): value for IDD Field `Condenser Type`
                Default value: AirCooled
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Type"] = value

    @property
    def chiller_flow_mode(self):
        """Get chiller_flow_mode.

        Returns:
            str: the value of `chiller_flow_mode` or None if not set

        """
        return self["Chiller Flow Mode"]

    @chiller_flow_mode.setter
    def chiller_flow_mode(self, value="NotModulated"):
        """Corresponds to IDD field `Chiller Flow Mode` Select operating mode
        for fluid flow through the chiller. "NotModulated" is for either
        variable or constant pumping with flow controlled by the external plant
        system. "ConstantFlow" is for constant pumping with flow controlled by
        chiller to operate at full design flow rate.
        "LeavingSetpointModulated" is for variable pumping with flow controlled
        by chiller to vary flow to target a leaving temperature setpoint.

        Args:
            value (str): value for IDD Field `Chiller Flow Mode`
                Default value: NotModulated
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Flow Mode"] = value

    @property
    def sizing_factor(self):
        """Get sizing_factor.

        Returns:
            float: the value of `sizing_factor` or None if not set

        """
        return self["Sizing Factor"]

    @sizing_factor.setter
    def sizing_factor(self, value=1.0):
        """Corresponds to IDD field `Sizing Factor` Multiplies the autosized
        capacity and flow rates.

        Args:
            value (float): value for IDD Field `Sizing Factor`
                Default value: 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Sizing Factor"] = value

    @property
    def basin_heater_capacity(self):
        """Get basin_heater_capacity.

        Returns:
            float: the value of `basin_heater_capacity` or None if not set

        """
        return self["Basin Heater Capacity"]

    @basin_heater_capacity.setter
    def basin_heater_capacity(self, value=None):
        """  Corresponds to IDD field `Basin Heater Capacity`
        This field is only used for Condenser Type = EvaporativelyCooled and for periods
        when the basin heater is available (field Basin Heater Operating Schedule Name).
        For this situation, the heater maintains the basin water temperature at the basin heater
        setpoint temperature when the outdoor air temperature falls below the setpoint temperature.
        The basin heater only operates when the chiller is not operating.

        Args:
            value (float): value for IDD Field `Basin Heater Capacity`
                Units: W/K
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Basin Heater Capacity"] = value

    @property
    def basin_heater_setpoint_temperature(self):
        """Get basin_heater_setpoint_temperature.

        Returns:
            float: the value of `basin_heater_setpoint_temperature` or None if not set

        """
        return self["Basin Heater Setpoint Temperature"]

    @basin_heater_setpoint_temperature.setter
    def basin_heater_setpoint_temperature(self, value=2.0):
        """  Corresponds to IDD field `Basin Heater Setpoint Temperature`
        This field is only used for Condenser Type = EvaporativelyCooled.
        Enter the outdoor dry-bulb temperature when the basin heater turns on.

        Args:
            value (float): value for IDD Field `Basin Heater Setpoint Temperature`
                Units: C
                Default value: 2.0
                value >= 2.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Basin Heater Setpoint Temperature"] = value

    @property
    def basin_heater_operating_schedule_name(self):
        """Get basin_heater_operating_schedule_name.

        Returns:
            str: the value of `basin_heater_operating_schedule_name` or None if not set

        """
        return self["Basin Heater Operating Schedule Name"]

    @basin_heater_operating_schedule_name.setter
    def basin_heater_operating_schedule_name(self, value=None):
        """  Corresponds to IDD field `Basin Heater Operating Schedule Name`
        This field is only used for Condenser Type = EvaporativelyCooled.
        Schedule values greater than 0 allow the basin heater to operate whenever the outdoor
        air dry-bulb temperature is below the basin heater setpoint temperature.
        If a schedule name is not entered, the basin heater is allowed to operate
        throughout the entire simulation.

        Args:
            value (str): value for IDD Field `Basin Heater Operating Schedule Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Basin Heater Operating Schedule Name"] = value




class ChillerEngineDriven(DataObject):

    """ Corresponds to IDD object `Chiller:EngineDriven`
        This chiller model is the empirical model from the Building Loads
        and System Thermodynamics (BLAST) program.  Chiller performance
        curves are generated by fitting catalog data to third order
        polynomial equations.  Three sets of coefficients are required.
    """
    schema = {'min-fields': 43,
              'name': u'Chiller:EngineDriven',
              'pyname': u'ChillerEngineDriven',
              'format': None,
              'fields': OrderedDict([(u'name',
                                      {'name': u'Name',
                                       'pyname': u'name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'condenser type',
                                      {'name': u'Condenser Type',
                                       'pyname': u'condenser_type',
                                       'default': u'AirCooled',
                                       'required-field': True,
                                       'autosizable': False,
                                       'accepted-values': [u'AirCooled',
                                                           u'WaterCooled',
                                                           u'EvaporativelyCooled'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'nominal capacity',
                                      {'name': u'Nominal Capacity',
                                       'pyname': u'nominal_capacity',
                                       'required-field': True,
                                       'autosizable': True,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'W'}),
                                     (u'nominal cop',
                                      {'name': u'Nominal COP',
                                       'pyname': u'nominal_cop',
                                       'minimum>': 0.0,
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'W/W'}),
                                     (u'chilled water inlet node name',
                                      {'name': u'Chilled Water Inlet Node Name',
                                       'pyname': u'chilled_water_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'chilled water outlet node name',
                                      {'name': u'Chilled Water Outlet Node Name',
                                       'pyname': u'chilled_water_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'condenser inlet node name',
                                      {'name': u'Condenser Inlet Node Name',
                                       'pyname': u'condenser_inlet_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'condenser outlet node name',
                                      {'name': u'Condenser Outlet Node Name',
                                       'pyname': u'condenser_outlet_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'minimum part load ratio',
                                      {'name': u'Minimum Part Load Ratio',
                                       'pyname': u'minimum_part_load_ratio',
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'maximum part load ratio',
                                      {'name': u'Maximum Part Load Ratio',
                                       'pyname': u'maximum_part_load_ratio',
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'optimum part load ratio',
                                      {'name': u'Optimum Part Load Ratio',
                                       'pyname': u'optimum_part_load_ratio',
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'design condenser inlet temperature',
                                      {'name': u'Design Condenser Inlet Temperature',
                                       'pyname': u'design_condenser_inlet_temperature',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'C'}),
                                     (u'temperature rise coefficient',
                                      {'name': u'Temperature Rise Coefficient',
                                       'pyname': u'temperature_rise_coefficient',
                                       'minimum>': 0.0,
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'design chilled water outlet temperature',
                                      {'name': u'Design Chilled Water Outlet Temperature',
                                       'pyname': u'design_chilled_water_outlet_temperature',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'C'}),
                                     (u'design chilled water flow rate',
                                      {'name': u'Design Chilled Water Flow Rate',
                                       'pyname': u'design_chilled_water_flow_rate',
                                       'required-field': False,
                                       'autosizable': True,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'm3/s'}),
                                     (u'design condenser water flow rate',
                                      {'name': u'Design Condenser Water Flow Rate',
                                       'pyname': u'design_condenser_water_flow_rate',
                                       'required-field': False,
                                       'autosizable': True,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'm3/s'}),
                                     (u'coefficient 1 of capacity ratio curve',
                                      {'name': u'Coefficient 1 of Capacity Ratio Curve',
                                       'pyname': u'coefficient_1_of_capacity_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 2 of capacity ratio curve',
                                      {'name': u'Coefficient 2 of Capacity Ratio Curve',
                                       'pyname': u'coefficient_2_of_capacity_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 3 of capacity ratio curve',
                                      {'name': u'Coefficient 3 of Capacity Ratio Curve',
                                       'pyname': u'coefficient_3_of_capacity_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 1 of power ratio curve',
                                      {'name': u'Coefficient 1 of Power Ratio Curve',
                                       'pyname': u'coefficient_1_of_power_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 2 of power ratio curve',
                                      {'name': u'Coefficient 2 of Power Ratio Curve',
                                       'pyname': u'coefficient_2_of_power_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 3 of power ratio curve',
                                      {'name': u'Coefficient 3 of Power Ratio Curve',
                                       'pyname': u'coefficient_3_of_power_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 1 of full load ratio curve',
                                      {'name': u'Coefficient 1 of Full Load Ratio Curve',
                                       'pyname': u'coefficient_1_of_full_load_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 2 of full load ratio curve',
                                      {'name': u'Coefficient 2 of Full Load Ratio Curve',
                                       'pyname': u'coefficient_2_of_full_load_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 3 of full load ratio curve',
                                      {'name': u'Coefficient 3 of Full Load Ratio Curve',
                                       'pyname': u'coefficient_3_of_full_load_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'chilled water outlet temperature lower limit',
                                      {'name': u'Chilled Water Outlet Temperature Lower Limit',
                                       'pyname': u'chilled_water_outlet_temperature_lower_limit',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'C'}),
                                     (u'fuel use curve name',
                                      {'name': u'Fuel Use Curve Name',
                                       'pyname': u'fuel_use_curve_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'jacket heat recovery curve name',
                                      {'name': u'Jacket Heat Recovery Curve Name',
                                       'pyname': u'jacket_heat_recovery_curve_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'lube heat recovery curve name',
                                      {'name': u'Lube Heat Recovery Curve Name',
                                       'pyname': u'lube_heat_recovery_curve_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'total exhaust energy curve name',
                                      {'name': u'Total Exhaust Energy Curve Name',
                                       'pyname': u'total_exhaust_energy_curve_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'exhaust temperature curve name',
                                      {'name': u'Exhaust Temperature Curve Name',
                                       'pyname': u'exhaust_temperature_curve_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'coefficient 1 of u-factor times area curve',
                                      {'name': u'Coefficient 1 of U-Factor Times Area Curve',
                                       'pyname': u'coefficient_1_of_ufactor_times_area_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 2 of u-factor times area curve',
                                      {'name': u'Coefficient 2 of U-Factor Times Area Curve',
                                       'pyname': u'coefficient_2_of_ufactor_times_area_curve',
                                       'maximum': 2.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'maximum exhaust flow per unit of power output',
                                      {'name': u'Maximum Exhaust Flow per Unit of Power Output',
                                       'pyname': u'maximum_exhaust_flow_per_unit_of_power_output',
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'(kg/s)/W'}),
                                     (u'design minimum exhaust temperature',
                                      {'name': u'Design Minimum Exhaust Temperature',
                                       'pyname': u'design_minimum_exhaust_temperature',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'C'}),
                                     (u'fuel type',
                                      {'name': u'Fuel Type',
                                       'pyname': u'fuel_type',
                                       'required-field': True,
                                       'autosizable': False,
                                       'accepted-values': [u'NaturalGas',
                                                           u'PropaneGas',
                                                           u'Diesel',
                                                           u'Gasoline',
                                                           u'FuelOil#1',
                                                           u'FuelOil#2',
                                                           u'OtherFuel1',
                                                           u'OtherFuel2'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'fuel higher heating value',
                                      {'name': u'Fuel Higher Heating Value',
                                       'pyname': u'fuel_higher_heating_value',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'kJ/kg'}),
                                     (u'design heat recovery water flow rate',
                                      {'name': u'Design Heat Recovery Water Flow Rate',
                                       'pyname': u'design_heat_recovery_water_flow_rate',
                                       'default': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'm3/s'}),
                                     (u'heat recovery inlet node name',
                                      {'name': u'Heat Recovery Inlet Node Name',
                                       'pyname': u'heat_recovery_inlet_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'heat recovery outlet node name',
                                      {'name': u'Heat Recovery Outlet Node Name',
                                       'pyname': u'heat_recovery_outlet_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'chiller flow mode',
                                      {'name': u'Chiller Flow Mode',
                                       'pyname': u'chiller_flow_mode',
                                       'default': u'NotModulated',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ConstantFlow',
                                                           u'LeavingSetpointModulated',
                                                           u'NotModulated'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'maximum temperature for heat recovery at heat recovery outlet node',
                                      {'name': u'Maximum Temperature for Heat Recovery at Heat Recovery Outlet Node',
                                       'pyname': u'maximum_temperature_for_heat_recovery_at_heat_recovery_outlet_node',
                                       'default': 60.0,
                                       'maximum': 100.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'C'}),
                                     (u'sizing factor',
                                      {'name': u'Sizing Factor',
                                       'pyname': u'sizing_factor',
                                       'default': 1.0,
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'basin heater capacity',
                                      {'name': u'Basin Heater Capacity',
                                       'pyname': u'basin_heater_capacity',
                                       'default': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'W/K'}),
                                     (u'basin heater setpoint temperature',
                                      {'name': u'Basin Heater Setpoint Temperature',
                                       'pyname': u'basin_heater_setpoint_temperature',
                                       'default': 2.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 2.0,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'C'}),
                                     (u'basin heater operating schedule name',
                                      {'name': u'Basin Heater Operating Schedule Name',
                                       'pyname': u'basin_heater_operating_schedule_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'})]),
              'extensible-fields': OrderedDict(),
              'unique-object': False,
              'required-object': False,
              'group': u'Plant Heating and Cooling Equipment'}

    @property
    def name(self):
        """Get name.

        Returns:
            str: the value of `name` or None if not set

        """
        return self["Name"]

    @name.setter
    def name(self, value=None):
        """Corresponds to IDD field `Name`

        Args:
            value (str): value for IDD Field `Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Name"] = value

    @property
    def condenser_type(self):
        """Get condenser_type.

        Returns:
            str: the value of `condenser_type` or None if not set

        """
        return self["Condenser Type"]

    @condenser_type.setter
    def condenser_type(self, value="AirCooled"):
        """Corresponds to IDD field `Condenser Type`

        Args:
            value (str): value for IDD Field `Condenser Type`
                Default value: AirCooled
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Type"] = value

    @property
    def nominal_capacity(self):
        """Get nominal_capacity.

        Returns:
            float: the value of `nominal_capacity` or None if not set

        """
        return self["Nominal Capacity"]

    @nominal_capacity.setter
    def nominal_capacity(self, value=None):
        """Corresponds to IDD field `Nominal Capacity`

        Args:
            value (float or "Autosize"): value for IDD Field `Nominal Capacity`
                Units: W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Nominal Capacity"] = value

    @property
    def nominal_cop(self):
        """Get nominal_cop.

        Returns:
            float: the value of `nominal_cop` or None if not set

        """
        return self["Nominal COP"]

    @nominal_cop.setter
    def nominal_cop(self, value=None):
        """Corresponds to IDD field `Nominal COP` Nominal Refrigeration Cycle
        COP.

        Args:
            value (float): value for IDD Field `Nominal COP`
                Units: W/W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Nominal COP"] = value

    @property
    def chilled_water_inlet_node_name(self):
        """Get chilled_water_inlet_node_name.

        Returns:
            str: the value of `chilled_water_inlet_node_name` or None if not set

        """
        return self["Chilled Water Inlet Node Name"]

    @chilled_water_inlet_node_name.setter
    def chilled_water_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Chilled Water Inlet Node Name`

        Args:
            value (str): value for IDD Field `Chilled Water Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Inlet Node Name"] = value

    @property
    def chilled_water_outlet_node_name(self):
        """Get chilled_water_outlet_node_name.

        Returns:
            str: the value of `chilled_water_outlet_node_name` or None if not set

        """
        return self["Chilled Water Outlet Node Name"]

    @chilled_water_outlet_node_name.setter
    def chilled_water_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Chilled Water Outlet Node Name`

        Args:
            value (str): value for IDD Field `Chilled Water Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Outlet Node Name"] = value

    @property
    def condenser_inlet_node_name(self):
        """Get condenser_inlet_node_name.

        Returns:
            str: the value of `condenser_inlet_node_name` or None if not set

        """
        return self["Condenser Inlet Node Name"]

    @condenser_inlet_node_name.setter
    def condenser_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Condenser Inlet Node Name`

        Args:
            value (str): value for IDD Field `Condenser Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Inlet Node Name"] = value

    @property
    def condenser_outlet_node_name(self):
        """Get condenser_outlet_node_name.

        Returns:
            str: the value of `condenser_outlet_node_name` or None if not set

        """
        return self["Condenser Outlet Node Name"]

    @condenser_outlet_node_name.setter
    def condenser_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Condenser Outlet Node Name`

        Args:
            value (str): value for IDD Field `Condenser Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Outlet Node Name"] = value

    @property
    def minimum_part_load_ratio(self):
        """Get minimum_part_load_ratio.

        Returns:
            float: the value of `minimum_part_load_ratio` or None if not set

        """
        return self["Minimum Part Load Ratio"]

    @minimum_part_load_ratio.setter
    def minimum_part_load_ratio(self, value=None):
        """Corresponds to IDD field `Minimum Part Load Ratio`

        Args:
            value (float): value for IDD Field `Minimum Part Load Ratio`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Minimum Part Load Ratio"] = value

    @property
    def maximum_part_load_ratio(self):
        """Get maximum_part_load_ratio.

        Returns:
            float: the value of `maximum_part_load_ratio` or None if not set

        """
        return self["Maximum Part Load Ratio"]

    @maximum_part_load_ratio.setter
    def maximum_part_load_ratio(self, value=None):
        """Corresponds to IDD field `Maximum Part Load Ratio`

        Args:
            value (float): value for IDD Field `Maximum Part Load Ratio`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Maximum Part Load Ratio"] = value

    @property
    def optimum_part_load_ratio(self):
        """Get optimum_part_load_ratio.

        Returns:
            float: the value of `optimum_part_load_ratio` or None if not set

        """
        return self["Optimum Part Load Ratio"]

    @optimum_part_load_ratio.setter
    def optimum_part_load_ratio(self, value=None):
        """Corresponds to IDD field `Optimum Part Load Ratio`

        Args:
            value (float): value for IDD Field `Optimum Part Load Ratio`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Optimum Part Load Ratio"] = value

    @property
    def design_condenser_inlet_temperature(self):
        """Get design_condenser_inlet_temperature.

        Returns:
            float: the value of `design_condenser_inlet_temperature` or None if not set

        """
        return self["Design Condenser Inlet Temperature"]

    @design_condenser_inlet_temperature.setter
    def design_condenser_inlet_temperature(self, value=None):
        """Corresponds to IDD field `Design Condenser Inlet Temperature`

        Args:
            value (float): value for IDD Field `Design Condenser Inlet Temperature`
                Units: C
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Condenser Inlet Temperature"] = value

    @property
    def temperature_rise_coefficient(self):
        """Get temperature_rise_coefficient.

        Returns:
            float: the value of `temperature_rise_coefficient` or None if not set

        """
        return self["Temperature Rise Coefficient"]

    @temperature_rise_coefficient.setter
    def temperature_rise_coefficient(self, value=None):
        """Corresponds to IDD field `Temperature Rise Coefficient`

        Args:
            value (float): value for IDD Field `Temperature Rise Coefficient`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Temperature Rise Coefficient"] = value

    @property
    def design_chilled_water_outlet_temperature(self):
        """Get design_chilled_water_outlet_temperature.

        Returns:
            float: the value of `design_chilled_water_outlet_temperature` or None if not set

        """
        return self["Design Chilled Water Outlet Temperature"]

    @design_chilled_water_outlet_temperature.setter
    def design_chilled_water_outlet_temperature(self, value=None):
        """Corresponds to IDD field `Design Chilled Water Outlet Temperature`

        Args:
            value (float): value for IDD Field `Design Chilled Water Outlet Temperature`
                Units: C
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Chilled Water Outlet Temperature"] = value

    @property
    def design_chilled_water_flow_rate(self):
        """Get design_chilled_water_flow_rate.

        Returns:
            float: the value of `design_chilled_water_flow_rate` or None if not set

        """
        return self["Design Chilled Water Flow Rate"]

    @design_chilled_water_flow_rate.setter
    def design_chilled_water_flow_rate(self, value=None):
        """Corresponds to IDD field `Design Chilled Water Flow Rate` For
        variable volume this is the maximum flow and for constant flow this is
        the flow.

        Args:
            value (float or "Autosize"): value for IDD Field `Design Chilled Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Chilled Water Flow Rate"] = value

    @property
    def design_condenser_water_flow_rate(self):
        """Get design_condenser_water_flow_rate.

        Returns:
            float: the value of `design_condenser_water_flow_rate` or None if not set

        """
        return self["Design Condenser Water Flow Rate"]

    @design_condenser_water_flow_rate.setter
    def design_condenser_water_flow_rate(self, value=None):
        """  Corresponds to IDD field `Design Condenser Water Flow Rate`
        This field is not used for Condenser Type = AirCooled or EvaporativelyCooled

        Args:
            value (float or "Autosize"): value for IDD Field `Design Condenser Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Design Condenser Water Flow Rate"] = value

    @property
    def coefficient_1_of_capacity_ratio_curve(self):
        """Get coefficient_1_of_capacity_ratio_curve.

        Returns:
            float: the value of `coefficient_1_of_capacity_ratio_curve` or None if not set

        """
        return self["Coefficient 1 of Capacity Ratio Curve"]

    @coefficient_1_of_capacity_ratio_curve.setter
    def coefficient_1_of_capacity_ratio_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 1 of Capacity Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 1 of Capacity Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 1 of Capacity Ratio Curve"] = value

    @property
    def coefficient_2_of_capacity_ratio_curve(self):
        """Get coefficient_2_of_capacity_ratio_curve.

        Returns:
            float: the value of `coefficient_2_of_capacity_ratio_curve` or None if not set

        """
        return self["Coefficient 2 of Capacity Ratio Curve"]

    @coefficient_2_of_capacity_ratio_curve.setter
    def coefficient_2_of_capacity_ratio_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 2 of Capacity Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 2 of Capacity Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 2 of Capacity Ratio Curve"] = value

    @property
    def coefficient_3_of_capacity_ratio_curve(self):
        """Get coefficient_3_of_capacity_ratio_curve.

        Returns:
            float: the value of `coefficient_3_of_capacity_ratio_curve` or None if not set

        """
        return self["Coefficient 3 of Capacity Ratio Curve"]

    @coefficient_3_of_capacity_ratio_curve.setter
    def coefficient_3_of_capacity_ratio_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 3 of Capacity Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 3 of Capacity Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 3 of Capacity Ratio Curve"] = value

    @property
    def coefficient_1_of_power_ratio_curve(self):
        """Get coefficient_1_of_power_ratio_curve.

        Returns:
            float: the value of `coefficient_1_of_power_ratio_curve` or None if not set

        """
        return self["Coefficient 1 of Power Ratio Curve"]

    @coefficient_1_of_power_ratio_curve.setter
    def coefficient_1_of_power_ratio_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 1 of Power Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 1 of Power Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 1 of Power Ratio Curve"] = value

    @property
    def coefficient_2_of_power_ratio_curve(self):
        """Get coefficient_2_of_power_ratio_curve.

        Returns:
            float: the value of `coefficient_2_of_power_ratio_curve` or None if not set

        """
        return self["Coefficient 2 of Power Ratio Curve"]

    @coefficient_2_of_power_ratio_curve.setter
    def coefficient_2_of_power_ratio_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 2 of Power Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 2 of Power Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 2 of Power Ratio Curve"] = value

    @property
    def coefficient_3_of_power_ratio_curve(self):
        """Get coefficient_3_of_power_ratio_curve.

        Returns:
            float: the value of `coefficient_3_of_power_ratio_curve` or None if not set

        """
        return self["Coefficient 3 of Power Ratio Curve"]

    @coefficient_3_of_power_ratio_curve.setter
    def coefficient_3_of_power_ratio_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 3 of Power Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 3 of Power Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 3 of Power Ratio Curve"] = value

    @property
    def coefficient_1_of_full_load_ratio_curve(self):
        """Get coefficient_1_of_full_load_ratio_curve.

        Returns:
            float: the value of `coefficient_1_of_full_load_ratio_curve` or None if not set

        """
        return self["Coefficient 1 of Full Load Ratio Curve"]

    @coefficient_1_of_full_load_ratio_curve.setter
    def coefficient_1_of_full_load_ratio_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 1 of Full Load Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 1 of Full Load Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 1 of Full Load Ratio Curve"] = value

    @property
    def coefficient_2_of_full_load_ratio_curve(self):
        """Get coefficient_2_of_full_load_ratio_curve.

        Returns:
            float: the value of `coefficient_2_of_full_load_ratio_curve` or None if not set

        """
        return self["Coefficient 2 of Full Load Ratio Curve"]

    @coefficient_2_of_full_load_ratio_curve.setter
    def coefficient_2_of_full_load_ratio_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 2 of Full Load Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 2 of Full Load Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 2 of Full Load Ratio Curve"] = value

    @property
    def coefficient_3_of_full_load_ratio_curve(self):
        """Get coefficient_3_of_full_load_ratio_curve.

        Returns:
            float: the value of `coefficient_3_of_full_load_ratio_curve` or None if not set

        """
        return self["Coefficient 3 of Full Load Ratio Curve"]

    @coefficient_3_of_full_load_ratio_curve.setter
    def coefficient_3_of_full_load_ratio_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 3 of Full Load Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 3 of Full Load Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 3 of Full Load Ratio Curve"] = value

    @property
    def chilled_water_outlet_temperature_lower_limit(self):
        """Get chilled_water_outlet_temperature_lower_limit.

        Returns:
            float: the value of `chilled_water_outlet_temperature_lower_limit` or None if not set

        """
        return self["Chilled Water Outlet Temperature Lower Limit"]

    @chilled_water_outlet_temperature_lower_limit.setter
    def chilled_water_outlet_temperature_lower_limit(self, value=None):
        """Corresponds to IDD field `Chilled Water Outlet Temperature Lower
        Limit` Special EngineDriven Chiller Parameters Below.

        Args:
            value (float): value for IDD Field `Chilled Water Outlet Temperature Lower Limit`
                Units: C
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Outlet Temperature Lower Limit"] = value

    @property
    def fuel_use_curve_name(self):
        """Get fuel_use_curve_name.

        Returns:
            str: the value of `fuel_use_curve_name` or None if not set

        """
        return self["Fuel Use Curve Name"]

    @fuel_use_curve_name.setter
    def fuel_use_curve_name(self, value=None):
        """  Corresponds to IDD field `Fuel Use Curve Name`
        Curve is a function of Part Load Ratio (PLR)
        Table:OneIndependentVariable object can also be used
        curve = a + b*PLR + c*PLR**2
        PLR = Ratio of evaporator heat transfer rate to nominal capacity

        Args:
            value (str): value for IDD Field `Fuel Use Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Fuel Use Curve Name"] = value

    @property
    def jacket_heat_recovery_curve_name(self):
        """Get jacket_heat_recovery_curve_name.

        Returns:
            str: the value of `jacket_heat_recovery_curve_name` or None if not set

        """
        return self["Jacket Heat Recovery Curve Name"]

    @jacket_heat_recovery_curve_name.setter
    def jacket_heat_recovery_curve_name(self, value=None):
        """  Corresponds to IDD field `Jacket Heat Recovery Curve Name`
        Curve is a function of Part Load Ratio (PLR)
        Table:OneIndependentVariable object can also be used
        curve = a + b*PLR + c*PLR**2
        PLR = Ratio of evaporator heat transfer rate to nominal capacity

        Args:
            value (str): value for IDD Field `Jacket Heat Recovery Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Jacket Heat Recovery Curve Name"] = value

    @property
    def lube_heat_recovery_curve_name(self):
        """Get lube_heat_recovery_curve_name.

        Returns:
            str: the value of `lube_heat_recovery_curve_name` or None if not set

        """
        return self["Lube Heat Recovery Curve Name"]

    @lube_heat_recovery_curve_name.setter
    def lube_heat_recovery_curve_name(self, value=None):
        """  Corresponds to IDD field `Lube Heat Recovery Curve Name`
        Curve is a function of Part Load Ratio (PLR)
        Table:OneIndependentVariable object can also be used
        curve = a + b*PLR + c*PLR**2
        PLR = Ratio of evaporator heat transfer rate to nominal capacity

        Args:
            value (str): value for IDD Field `Lube Heat Recovery Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Lube Heat Recovery Curve Name"] = value

    @property
    def total_exhaust_energy_curve_name(self):
        """Get total_exhaust_energy_curve_name.

        Returns:
            str: the value of `total_exhaust_energy_curve_name` or None if not set

        """
        return self["Total Exhaust Energy Curve Name"]

    @total_exhaust_energy_curve_name.setter
    def total_exhaust_energy_curve_name(self, value=None):
        """  Corresponds to IDD field `Total Exhaust Energy Curve Name`
        Curve is a function of Part Load Ratio (PLR)
        Table:OneIndependentVariable object can also be used
        curve = a + b*PLR + c*PLR**2
        PLR = Ratio of evaporator heat transfer rate to nominal capacity

        Args:
            value (str): value for IDD Field `Total Exhaust Energy Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Total Exhaust Energy Curve Name"] = value

    @property
    def exhaust_temperature_curve_name(self):
        """Get exhaust_temperature_curve_name.

        Returns:
            str: the value of `exhaust_temperature_curve_name` or None if not set

        """
        return self["Exhaust Temperature Curve Name"]

    @exhaust_temperature_curve_name.setter
    def exhaust_temperature_curve_name(self, value=None):
        """  Corresponds to IDD field `Exhaust Temperature Curve Name`
        Curve is a function of Part Load Ratio (PLR)
        curve = a + b*PLR + c*PLR**2
        Table:OneIndependentVariable object can also be used
        PLR = Ratio of evaporator heat transfer rate to nominal capacity

        Args:
            value (str): value for IDD Field `Exhaust Temperature Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Exhaust Temperature Curve Name"] = value

    @property
    def coefficient_1_of_ufactor_times_area_curve(self):
        """Get coefficient_1_of_ufactor_times_area_curve.

        Returns:
            float: the value of `coefficient_1_of_ufactor_times_area_curve` or None if not set

        """
        return self["Coefficient 1 of U-Factor Times Area Curve"]

    @coefficient_1_of_ufactor_times_area_curve.setter
    def coefficient_1_of_ufactor_times_area_curve(self, value=None):
        """  Corresponds to IDD field `Coefficient 1 of U-Factor Times Area Curve`
        curve = C1 * (nominal capacity)**C2

        Args:
            value (float): value for IDD Field `Coefficient 1 of U-Factor Times Area Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Coefficient 1 of U-Factor Times Area Curve"] = value

    @property
    def coefficient_2_of_ufactor_times_area_curve(self):
        """Get coefficient_2_of_ufactor_times_area_curve.

        Returns:
            float: the value of `coefficient_2_of_ufactor_times_area_curve` or None if not set

        """
        return self["Coefficient 2 of U-Factor Times Area Curve"]

    @coefficient_2_of_ufactor_times_area_curve.setter
    def coefficient_2_of_ufactor_times_area_curve(self, value=None):
        """  Corresponds to IDD field `Coefficient 2 of U-Factor Times Area Curve`
        curve = C1 * (nominal capacity)**C2
        typical value .9

        Args:
            value (float): value for IDD Field `Coefficient 2 of U-Factor Times Area Curve`
                value <= 2.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Coefficient 2 of U-Factor Times Area Curve"] = value

    @property
    def maximum_exhaust_flow_per_unit_of_power_output(self):
        """Get maximum_exhaust_flow_per_unit_of_power_output.

        Returns:
            float: the value of `maximum_exhaust_flow_per_unit_of_power_output` or None if not set

        """
        return self["Maximum Exhaust Flow per Unit of Power Output"]

    @maximum_exhaust_flow_per_unit_of_power_output.setter
    def maximum_exhaust_flow_per_unit_of_power_output(self, value=None):
        """Corresponds to IDD field `Maximum Exhaust Flow per Unit of Power
        Output`

        Args:
            value (float): value for IDD Field `Maximum Exhaust Flow per Unit of Power Output`
                Units: (kg/s)/W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Maximum Exhaust Flow per Unit of Power Output"] = value

    @property
    def design_minimum_exhaust_temperature(self):
        """Get design_minimum_exhaust_temperature.

        Returns:
            float: the value of `design_minimum_exhaust_temperature` or None if not set

        """
        return self["Design Minimum Exhaust Temperature"]

    @design_minimum_exhaust_temperature.setter
    def design_minimum_exhaust_temperature(self, value=None):
        """Corresponds to IDD field `Design Minimum Exhaust Temperature`

        Args:
            value (float): value for IDD Field `Design Minimum Exhaust Temperature`
                Units: C
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Minimum Exhaust Temperature"] = value

    @property
    def fuel_type(self):
        """Get fuel_type.

        Returns:
            str: the value of `fuel_type` or None if not set

        """
        return self["Fuel Type"]

    @fuel_type.setter
    def fuel_type(self, value=None):
        """Corresponds to IDD field `Fuel Type`

        Args:
            value (str): value for IDD Field `Fuel Type`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Fuel Type"] = value

    @property
    def fuel_higher_heating_value(self):
        """Get fuel_higher_heating_value.

        Returns:
            float: the value of `fuel_higher_heating_value` or None if not set

        """
        return self["Fuel Higher Heating Value"]

    @fuel_higher_heating_value.setter
    def fuel_higher_heating_value(self, value=None):
        """Corresponds to IDD field `Fuel Higher Heating Value`

        Args:
            value (float): value for IDD Field `Fuel Higher Heating Value`
                Units: kJ/kg
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Fuel Higher Heating Value"] = value

    @property
    def design_heat_recovery_water_flow_rate(self):
        """Get design_heat_recovery_water_flow_rate.

        Returns:
            float: the value of `design_heat_recovery_water_flow_rate` or None if not set

        """
        return self["Design Heat Recovery Water Flow Rate"]

    @design_heat_recovery_water_flow_rate.setter
    def design_heat_recovery_water_flow_rate(self, value=None):
        """  Corresponds to IDD field `Design Heat Recovery Water Flow Rate`
        If non-zero, then the heat recovery inlet and outlet node names must be entered.

        Args:
            value (float): value for IDD Field `Design Heat Recovery Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Design Heat Recovery Water Flow Rate"] = value

    @property
    def heat_recovery_inlet_node_name(self):
        """Get heat_recovery_inlet_node_name.

        Returns:
            str: the value of `heat_recovery_inlet_node_name` or None if not set

        """
        return self["Heat Recovery Inlet Node Name"]

    @heat_recovery_inlet_node_name.setter
    def heat_recovery_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Heat Recovery Inlet Node Name`

        Args:
            value (str): value for IDD Field `Heat Recovery Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Heat Recovery Inlet Node Name"] = value

    @property
    def heat_recovery_outlet_node_name(self):
        """Get heat_recovery_outlet_node_name.

        Returns:
            str: the value of `heat_recovery_outlet_node_name` or None if not set

        """
        return self["Heat Recovery Outlet Node Name"]

    @heat_recovery_outlet_node_name.setter
    def heat_recovery_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Heat Recovery Outlet Node Name`

        Args:
            value (str): value for IDD Field `Heat Recovery Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Heat Recovery Outlet Node Name"] = value

    @property
    def chiller_flow_mode(self):
        """Get chiller_flow_mode.

        Returns:
            str: the value of `chiller_flow_mode` or None if not set

        """
        return self["Chiller Flow Mode"]

    @chiller_flow_mode.setter
    def chiller_flow_mode(self, value="NotModulated"):
        """Corresponds to IDD field `Chiller Flow Mode` Select operating mode
        for fluid flow through the chiller. "NotModulated" is for either
        variable or constant pumping with flow controlled by the external plant
        system. "ConstantFlow" is for constant pumping with flow controlled by
        chiller to operate at full design flow rate.
        "LeavingSetpointModulated" is for variable pumping with flow controlled
        by chiller to vary flow to target a leaving temperature setpoint.

        Args:
            value (str): value for IDD Field `Chiller Flow Mode`
                Default value: NotModulated
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Flow Mode"] = value

    @property
    def maximum_temperature_for_heat_recovery_at_heat_recovery_outlet_node(
            self):
        """Get
        maximum_temperature_for_heat_recovery_at_heat_recovery_outlet_node.

        Returns:
            float: the value of `maximum_temperature_for_heat_recovery_at_heat_recovery_outlet_node` or None if not set

        """
        return self[
            "Maximum Temperature for Heat Recovery at Heat Recovery Outlet Node"]

    @maximum_temperature_for_heat_recovery_at_heat_recovery_outlet_node.setter
    def maximum_temperature_for_heat_recovery_at_heat_recovery_outlet_node(
            self,
            value=60.0):
        """Corresponds to IDD field `Maximum Temperature for Heat Recovery at
        Heat Recovery Outlet Node`

        Args:
            value (float): value for IDD Field `Maximum Temperature for Heat Recovery at Heat Recovery Outlet Node`
                Units: C
                Default value: 60.0
                value <= 100.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Maximum Temperature for Heat Recovery at Heat Recovery Outlet Node"] = value

    @property
    def sizing_factor(self):
        """Get sizing_factor.

        Returns:
            float: the value of `sizing_factor` or None if not set

        """
        return self["Sizing Factor"]

    @sizing_factor.setter
    def sizing_factor(self, value=1.0):
        """Corresponds to IDD field `Sizing Factor` Multiplies the autosized
        capacity and flow rates.

        Args:
            value (float): value for IDD Field `Sizing Factor`
                Default value: 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Sizing Factor"] = value

    @property
    def basin_heater_capacity(self):
        """Get basin_heater_capacity.

        Returns:
            float: the value of `basin_heater_capacity` or None if not set

        """
        return self["Basin Heater Capacity"]

    @basin_heater_capacity.setter
    def basin_heater_capacity(self, value=None):
        """  Corresponds to IDD field `Basin Heater Capacity`
        This field is only used for Condenser Type = EvaporativelyCooled and for periods
        when the basin heater is available (field Basin Heater Operating Schedule Name).
        For this situation, the heater maintains the basin water temperature at the basin heater
        setpoint temperature when the outdoor air temperature falls below the setpoint temperature.
        The basin heater only operates when the chiller is not operating.

        Args:
            value (float): value for IDD Field `Basin Heater Capacity`
                Units: W/K
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Basin Heater Capacity"] = value

    @property
    def basin_heater_setpoint_temperature(self):
        """Get basin_heater_setpoint_temperature.

        Returns:
            float: the value of `basin_heater_setpoint_temperature` or None if not set

        """
        return self["Basin Heater Setpoint Temperature"]

    @basin_heater_setpoint_temperature.setter
    def basin_heater_setpoint_temperature(self, value=2.0):
        """  Corresponds to IDD field `Basin Heater Setpoint Temperature`
        This field is only used for Condenser Type = EvaporativelyCooled.
        Enter the outdoor dry-bulb temperature when the basin heater turns on.

        Args:
            value (float): value for IDD Field `Basin Heater Setpoint Temperature`
                Units: C
                Default value: 2.0
                value >= 2.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Basin Heater Setpoint Temperature"] = value

    @property
    def basin_heater_operating_schedule_name(self):
        """Get basin_heater_operating_schedule_name.

        Returns:
            str: the value of `basin_heater_operating_schedule_name` or None if not set

        """
        return self["Basin Heater Operating Schedule Name"]

    @basin_heater_operating_schedule_name.setter
    def basin_heater_operating_schedule_name(self, value=None):
        """  Corresponds to IDD field `Basin Heater Operating Schedule Name`
        This field is only used for Condenser Type = EvaporativelyCooled.
        Schedule values greater than 0 allow the basin heater to operate whenever the outdoor
        air dry-bulb temperature is below the basin heater setpoint temperature.
        If a schedule name is not entered, the basin heater is allowed to operate
        throughout the entire simulation.

        Args:
            value (str): value for IDD Field `Basin Heater Operating Schedule Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Basin Heater Operating Schedule Name"] = value




class ChillerCombustionTurbine(DataObject):

    """ Corresponds to IDD object `Chiller:CombustionTurbine`
        This chiller model is the empirical model from the Building Loads
        and System Thermodynamics (BLAST) program.  Chiller performance
        curves are generated by fitting catalog data to third order
        polynomial equations.  Three sets of coefficients are required.
    """
    schema = {'min-fields': 56,
              'name': u'Chiller:CombustionTurbine',
              'pyname': u'ChillerCombustionTurbine',
              'format': None,
              'fields': OrderedDict([(u'name',
                                      {'name': u'Name',
                                       'pyname': u'name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'condenser type',
                                      {'name': u'Condenser Type',
                                       'pyname': u'condenser_type',
                                       'default': u'AirCooled',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'AirCooled',
                                                           u'WaterCooled',
                                                           u'EvaporativelyCooled'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'nominal capacity',
                                      {'name': u'Nominal Capacity',
                                       'pyname': u'nominal_capacity',
                                       'required-field': True,
                                       'autosizable': True,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'W'}),
                                     (u'nominal cop',
                                      {'name': u'Nominal COP',
                                       'pyname': u'nominal_cop',
                                       'minimum>': 0.0,
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'W/W'}),
                                     (u'chilled water inlet node name',
                                      {'name': u'Chilled Water Inlet Node Name',
                                       'pyname': u'chilled_water_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'chilled water outlet node name',
                                      {'name': u'Chilled Water Outlet Node Name',
                                       'pyname': u'chilled_water_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'condenser inlet node name',
                                      {'name': u'Condenser Inlet Node Name',
                                       'pyname': u'condenser_inlet_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'condenser outlet node name',
                                      {'name': u'Condenser Outlet Node Name',
                                       'pyname': u'condenser_outlet_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'minimum part load ratio',
                                      {'name': u'Minimum Part Load Ratio',
                                       'pyname': u'minimum_part_load_ratio',
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'maximum part load ratio',
                                      {'name': u'Maximum Part Load Ratio',
                                       'pyname': u'maximum_part_load_ratio',
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'optimum part load ratio',
                                      {'name': u'Optimum Part Load Ratio',
                                       'pyname': u'optimum_part_load_ratio',
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'design condenser inlet temperature',
                                      {'name': u'Design Condenser Inlet Temperature',
                                       'pyname': u'design_condenser_inlet_temperature',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'C'}),
                                     (u'temperature rise coefficient',
                                      {'name': u'Temperature Rise Coefficient',
                                       'pyname': u'temperature_rise_coefficient',
                                       'minimum>': 0.0,
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'design chilled water outlet temperature',
                                      {'name': u'Design Chilled Water Outlet Temperature',
                                       'pyname': u'design_chilled_water_outlet_temperature',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'C'}),
                                     (u'design chilled water flow rate',
                                      {'name': u'Design Chilled Water Flow Rate',
                                       'pyname': u'design_chilled_water_flow_rate',
                                       'required-field': False,
                                       'autosizable': True,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'm3/s'}),
                                     (u'design condenser water flow rate',
                                      {'name': u'Design Condenser Water Flow Rate',
                                       'pyname': u'design_condenser_water_flow_rate',
                                       'required-field': False,
                                       'autosizable': True,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'm3/s'}),
                                     (u'coefficient 1 of capacity ratio curve',
                                      {'name': u'Coefficient 1 of Capacity Ratio Curve',
                                       'pyname': u'coefficient_1_of_capacity_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 2 of capacity ratio curve',
                                      {'name': u'Coefficient 2 of Capacity Ratio Curve',
                                       'pyname': u'coefficient_2_of_capacity_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 3 of capacity ratio curve',
                                      {'name': u'Coefficient 3 of Capacity Ratio Curve',
                                       'pyname': u'coefficient_3_of_capacity_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 1 of power ratio curve',
                                      {'name': u'Coefficient 1 of Power Ratio Curve',
                                       'pyname': u'coefficient_1_of_power_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 2 of power ratio curve',
                                      {'name': u'Coefficient 2 of Power Ratio Curve',
                                       'pyname': u'coefficient_2_of_power_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 3 of power ratio curve',
                                      {'name': u'Coefficient 3 of Power Ratio Curve',
                                       'pyname': u'coefficient_3_of_power_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 1 of full load ratio curve',
                                      {'name': u'Coefficient 1 of Full Load Ratio Curve',
                                       'pyname': u'coefficient_1_of_full_load_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 2 of full load ratio curve',
                                      {'name': u'Coefficient 2 of Full Load Ratio Curve',
                                       'pyname': u'coefficient_2_of_full_load_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 3 of full load ratio curve',
                                      {'name': u'Coefficient 3 of Full Load Ratio Curve',
                                       'pyname': u'coefficient_3_of_full_load_ratio_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'chilled water outlet temperature lower limit',
                                      {'name': u'Chilled Water Outlet Temperature Lower Limit',
                                       'pyname': u'chilled_water_outlet_temperature_lower_limit',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'C'}),
                                     (u'coefficient 1 of fuel input curve',
                                      {'name': u'Coefficient 1 of Fuel Input Curve',
                                       'pyname': u'coefficient_1_of_fuel_input_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 2 of fuel input curve',
                                      {'name': u'Coefficient 2 of Fuel Input Curve',
                                       'pyname': u'coefficient_2_of_fuel_input_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 3 of fuel input curve',
                                      {'name': u'Coefficient 3 of Fuel Input Curve',
                                       'pyname': u'coefficient_3_of_fuel_input_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 1 of temperature based fuel input curve',
                                      {'name': u'Coefficient 1 of Temperature Based Fuel Input Curve',
                                       'pyname': u'coefficient_1_of_temperature_based_fuel_input_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 2 of temperature based fuel input curve',
                                      {'name': u'Coefficient 2 of Temperature Based Fuel Input Curve',
                                       'pyname': u'coefficient_2_of_temperature_based_fuel_input_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 3 of temperature based fuel input curve',
                                      {'name': u'Coefficient 3 of Temperature Based Fuel Input Curve',
                                       'pyname': u'coefficient_3_of_temperature_based_fuel_input_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 1 of exhaust flow curve',
                                      {'name': u'Coefficient 1 of Exhaust Flow Curve',
                                       'pyname': u'coefficient_1_of_exhaust_flow_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 2 of exhaust flow curve',
                                      {'name': u'Coefficient 2 of Exhaust Flow Curve',
                                       'pyname': u'coefficient_2_of_exhaust_flow_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 3 of exhaust flow curve',
                                      {'name': u'Coefficient 3 of Exhaust Flow Curve',
                                       'pyname': u'coefficient_3_of_exhaust_flow_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 1 of exhaust gas temperature curve',
                                      {'name': u'Coefficient 1 of Exhaust Gas Temperature Curve',
                                       'pyname': u'coefficient_1_of_exhaust_gas_temperature_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 2 of exhaust gas temperature curve',
                                      {'name': u'Coefficient 2 of Exhaust Gas Temperature Curve',
                                       'pyname': u'coefficient_2_of_exhaust_gas_temperature_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 3 of exhaust gas temperature curve',
                                      {'name': u'Coefficient 3 of Exhaust Gas Temperature Curve',
                                       'pyname': u'coefficient_3_of_exhaust_gas_temperature_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 1 of temperature based exhaust gas temperature curve',
                                      {'name': u'Coefficient 1 of Temperature Based Exhaust Gas Temperature Curve',
                                       'pyname': u'coefficient_1_of_temperature_based_exhaust_gas_temperature_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 2 of temperature based exhaust gas temperature curve',
                                      {'name': u'Coefficient 2 of Temperature Based Exhaust Gas Temperature Curve',
                                       'pyname': u'coefficient_2_of_temperature_based_exhaust_gas_temperature_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 3 of temperature based exhaust gas temperature curve',
                                      {'name': u'Coefficient 3 of Temperature Based Exhaust Gas Temperature Curve',
                                       'pyname': u'coefficient_3_of_temperature_based_exhaust_gas_temperature_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 1 of recovery lube heat curve',
                                      {'name': u'Coefficient 1 of Recovery Lube Heat Curve',
                                       'pyname': u'coefficient_1_of_recovery_lube_heat_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 2 of recovery lube heat curve',
                                      {'name': u'Coefficient 2 of Recovery Lube Heat Curve',
                                       'pyname': u'coefficient_2_of_recovery_lube_heat_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 3 of recovery lube heat curve',
                                      {'name': u'Coefficient 3 of Recovery Lube Heat Curve',
                                       'pyname': u'coefficient_3_of_recovery_lube_heat_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 1 of u-factor times area curve',
                                      {'name': u'Coefficient 1 of U-Factor Times Area Curve',
                                       'pyname': u'coefficient_1_of_ufactor_times_area_curve',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'coefficient 2 of u-factor times area curve',
                                      {'name': u'Coefficient 2 of U-Factor Times Area Curve',
                                       'pyname': u'coefficient_2_of_ufactor_times_area_curve',
                                       'maximum': 2.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'gas turbine engine capacity',
                                      {'name': u'Gas Turbine Engine Capacity',
                                       'pyname': u'gas_turbine_engine_capacity',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': True,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'W'}),
                                     (u'maximum exhaust flow per unit of power output',
                                      {'name': u'Maximum Exhaust Flow per Unit of Power Output',
                                       'pyname': u'maximum_exhaust_flow_per_unit_of_power_output',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'(kg/s)/W'}),
                                     (u'design steam saturation temperature',
                                      {'name': u'Design Steam Saturation Temperature',
                                       'pyname': u'design_steam_saturation_temperature',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'C'}),
                                     (u'fuel higher heating value',
                                      {'name': u'Fuel Higher Heating Value',
                                       'pyname': u'fuel_higher_heating_value',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'kJ/kg'}),
                                     (u'design heat recovery water flow rate',
                                      {'name': u'Design Heat Recovery Water Flow Rate',
                                       'pyname': u'design_heat_recovery_water_flow_rate',
                                       'default': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'm3/s'}),
                                     (u'heat recovery inlet node name',
                                      {'name': u'Heat Recovery Inlet Node Name',
                                       'pyname': u'heat_recovery_inlet_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'heat recovery outlet node name',
                                      {'name': u'Heat Recovery Outlet Node Name',
                                       'pyname': u'heat_recovery_outlet_node_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'chiller flow mode',
                                      {'name': u'Chiller Flow Mode',
                                       'pyname': u'chiller_flow_mode',
                                       'default': u'NotModulated',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ConstantFlow',
                                                           u'LeavingSetpointModulated',
                                                           u'NotModulated'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'fuel type',
                                      {'name': u'Fuel Type',
                                       'pyname': u'fuel_type',
                                       'default': u'NaturalGas',
                                       'required-field': True,
                                       'autosizable': False,
                                       'accepted-values': [u'NaturalGas',
                                                           u'PropaneGas',
                                                           u'Diesel',
                                                           u'Gasoline',
                                                           u'FuelOil#1',
                                                           u'FuelOil#2',
                                                           u'OtherFuel1',
                                                           u'OtherFuel2'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'heat recovery maximum temperature',
                                      {'name': u'Heat Recovery Maximum Temperature',
                                       'pyname': u'heat_recovery_maximum_temperature',
                                       'default': 80.0,
                                       'maximum': 100.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'C'}),
                                     (u'sizing factor',
                                      {'name': u'Sizing Factor',
                                       'pyname': u'sizing_factor',
                                       'default': 1.0,
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'basin heater capacity',
                                      {'name': u'Basin Heater Capacity',
                                       'pyname': u'basin_heater_capacity',
                                       'default': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'W/K'}),
                                     (u'basin heater setpoint temperature',
                                      {'name': u'Basin Heater Setpoint Temperature',
                                       'pyname': u'basin_heater_setpoint_temperature',
                                       'default': 2.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 2.0,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'C'}),
                                     (u'basin heater operating schedule name',
                                      {'name': u'Basin Heater Operating Schedule Name',
                                       'pyname': u'basin_heater_operating_schedule_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'})]),
              'extensible-fields': OrderedDict(),
              'unique-object': False,
              'required-object': False,
              'group': u'Plant Heating and Cooling Equipment'}

    @property
    def name(self):
        """Get name.

        Returns:
            str: the value of `name` or None if not set

        """
        return self["Name"]

    @name.setter
    def name(self, value=None):
        """Corresponds to IDD field `Name`

        Args:
            value (str): value for IDD Field `Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Name"] = value

    @property
    def condenser_type(self):
        """Get condenser_type.

        Returns:
            str: the value of `condenser_type` or None if not set

        """
        return self["Condenser Type"]

    @condenser_type.setter
    def condenser_type(self, value="AirCooled"):
        """Corresponds to IDD field `Condenser Type`

        Args:
            value (str): value for IDD Field `Condenser Type`
                Default value: AirCooled
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Type"] = value

    @property
    def nominal_capacity(self):
        """Get nominal_capacity.

        Returns:
            float: the value of `nominal_capacity` or None if not set

        """
        return self["Nominal Capacity"]

    @nominal_capacity.setter
    def nominal_capacity(self, value=None):
        """Corresponds to IDD field `Nominal Capacity`

        Args:
            value (float or "Autosize"): value for IDD Field `Nominal Capacity`
                Units: W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Nominal Capacity"] = value

    @property
    def nominal_cop(self):
        """Get nominal_cop.

        Returns:
            float: the value of `nominal_cop` or None if not set

        """
        return self["Nominal COP"]

    @nominal_cop.setter
    def nominal_cop(self, value=None):
        """Corresponds to IDD field `Nominal COP`

        Args:
            value (float): value for IDD Field `Nominal COP`
                Units: W/W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Nominal COP"] = value

    @property
    def chilled_water_inlet_node_name(self):
        """Get chilled_water_inlet_node_name.

        Returns:
            str: the value of `chilled_water_inlet_node_name` or None if not set

        """
        return self["Chilled Water Inlet Node Name"]

    @chilled_water_inlet_node_name.setter
    def chilled_water_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Chilled Water Inlet Node Name`

        Args:
            value (str): value for IDD Field `Chilled Water Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Inlet Node Name"] = value

    @property
    def chilled_water_outlet_node_name(self):
        """Get chilled_water_outlet_node_name.

        Returns:
            str: the value of `chilled_water_outlet_node_name` or None if not set

        """
        return self["Chilled Water Outlet Node Name"]

    @chilled_water_outlet_node_name.setter
    def chilled_water_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Chilled Water Outlet Node Name`

        Args:
            value (str): value for IDD Field `Chilled Water Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Outlet Node Name"] = value

    @property
    def condenser_inlet_node_name(self):
        """Get condenser_inlet_node_name.

        Returns:
            str: the value of `condenser_inlet_node_name` or None if not set

        """
        return self["Condenser Inlet Node Name"]

    @condenser_inlet_node_name.setter
    def condenser_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Condenser Inlet Node Name`

        Args:
            value (str): value for IDD Field `Condenser Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Inlet Node Name"] = value

    @property
    def condenser_outlet_node_name(self):
        """Get condenser_outlet_node_name.

        Returns:
            str: the value of `condenser_outlet_node_name` or None if not set

        """
        return self["Condenser Outlet Node Name"]

    @condenser_outlet_node_name.setter
    def condenser_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Condenser Outlet Node Name`

        Args:
            value (str): value for IDD Field `Condenser Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Outlet Node Name"] = value

    @property
    def minimum_part_load_ratio(self):
        """Get minimum_part_load_ratio.

        Returns:
            float: the value of `minimum_part_load_ratio` or None if not set

        """
        return self["Minimum Part Load Ratio"]

    @minimum_part_load_ratio.setter
    def minimum_part_load_ratio(self, value=None):
        """Corresponds to IDD field `Minimum Part Load Ratio`

        Args:
            value (float): value for IDD Field `Minimum Part Load Ratio`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Minimum Part Load Ratio"] = value

    @property
    def maximum_part_load_ratio(self):
        """Get maximum_part_load_ratio.

        Returns:
            float: the value of `maximum_part_load_ratio` or None if not set

        """
        return self["Maximum Part Load Ratio"]

    @maximum_part_load_ratio.setter
    def maximum_part_load_ratio(self, value=None):
        """Corresponds to IDD field `Maximum Part Load Ratio`

        Args:
            value (float): value for IDD Field `Maximum Part Load Ratio`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Maximum Part Load Ratio"] = value

    @property
    def optimum_part_load_ratio(self):
        """Get optimum_part_load_ratio.

        Returns:
            float: the value of `optimum_part_load_ratio` or None if not set

        """
        return self["Optimum Part Load Ratio"]

    @optimum_part_load_ratio.setter
    def optimum_part_load_ratio(self, value=None):
        """Corresponds to IDD field `Optimum Part Load Ratio`

        Args:
            value (float): value for IDD Field `Optimum Part Load Ratio`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Optimum Part Load Ratio"] = value

    @property
    def design_condenser_inlet_temperature(self):
        """Get design_condenser_inlet_temperature.

        Returns:
            float: the value of `design_condenser_inlet_temperature` or None if not set

        """
        return self["Design Condenser Inlet Temperature"]

    @design_condenser_inlet_temperature.setter
    def design_condenser_inlet_temperature(self, value=None):
        """Corresponds to IDD field `Design Condenser Inlet Temperature`

        Args:
            value (float): value for IDD Field `Design Condenser Inlet Temperature`
                Units: C
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Condenser Inlet Temperature"] = value

    @property
    def temperature_rise_coefficient(self):
        """Get temperature_rise_coefficient.

        Returns:
            float: the value of `temperature_rise_coefficient` or None if not set

        """
        return self["Temperature Rise Coefficient"]

    @temperature_rise_coefficient.setter
    def temperature_rise_coefficient(self, value=None):
        """Corresponds to IDD field `Temperature Rise Coefficient`

        Args:
            value (float): value for IDD Field `Temperature Rise Coefficient`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Temperature Rise Coefficient"] = value

    @property
    def design_chilled_water_outlet_temperature(self):
        """Get design_chilled_water_outlet_temperature.

        Returns:
            float: the value of `design_chilled_water_outlet_temperature` or None if not set

        """
        return self["Design Chilled Water Outlet Temperature"]

    @design_chilled_water_outlet_temperature.setter
    def design_chilled_water_outlet_temperature(self, value=None):
        """Corresponds to IDD field `Design Chilled Water Outlet Temperature`

        Args:
            value (float): value for IDD Field `Design Chilled Water Outlet Temperature`
                Units: C
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Chilled Water Outlet Temperature"] = value

    @property
    def design_chilled_water_flow_rate(self):
        """Get design_chilled_water_flow_rate.

        Returns:
            float: the value of `design_chilled_water_flow_rate` or None if not set

        """
        return self["Design Chilled Water Flow Rate"]

    @design_chilled_water_flow_rate.setter
    def design_chilled_water_flow_rate(self, value=None):
        """Corresponds to IDD field `Design Chilled Water Flow Rate` For
        variable volume this is the max flow & for constant flow this is the
        flow.

        Args:
            value (float or "Autosize"): value for IDD Field `Design Chilled Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Chilled Water Flow Rate"] = value

    @property
    def design_condenser_water_flow_rate(self):
        """Get design_condenser_water_flow_rate.

        Returns:
            float: the value of `design_condenser_water_flow_rate` or None if not set

        """
        return self["Design Condenser Water Flow Rate"]

    @design_condenser_water_flow_rate.setter
    def design_condenser_water_flow_rate(self, value=None):
        """  Corresponds to IDD field `Design Condenser Water Flow Rate`
        This field is not used for Condenser Type = AirCooled or EvaporativelyCooled

        Args:
            value (float or "Autosize"): value for IDD Field `Design Condenser Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Design Condenser Water Flow Rate"] = value

    @property
    def coefficient_1_of_capacity_ratio_curve(self):
        """Get coefficient_1_of_capacity_ratio_curve.

        Returns:
            float: the value of `coefficient_1_of_capacity_ratio_curve` or None if not set

        """
        return self["Coefficient 1 of Capacity Ratio Curve"]

    @coefficient_1_of_capacity_ratio_curve.setter
    def coefficient_1_of_capacity_ratio_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 1 of Capacity Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 1 of Capacity Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 1 of Capacity Ratio Curve"] = value

    @property
    def coefficient_2_of_capacity_ratio_curve(self):
        """Get coefficient_2_of_capacity_ratio_curve.

        Returns:
            float: the value of `coefficient_2_of_capacity_ratio_curve` or None if not set

        """
        return self["Coefficient 2 of Capacity Ratio Curve"]

    @coefficient_2_of_capacity_ratio_curve.setter
    def coefficient_2_of_capacity_ratio_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 2 of Capacity Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 2 of Capacity Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 2 of Capacity Ratio Curve"] = value

    @property
    def coefficient_3_of_capacity_ratio_curve(self):
        """Get coefficient_3_of_capacity_ratio_curve.

        Returns:
            float: the value of `coefficient_3_of_capacity_ratio_curve` or None if not set

        """
        return self["Coefficient 3 of Capacity Ratio Curve"]

    @coefficient_3_of_capacity_ratio_curve.setter
    def coefficient_3_of_capacity_ratio_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 3 of Capacity Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 3 of Capacity Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 3 of Capacity Ratio Curve"] = value

    @property
    def coefficient_1_of_power_ratio_curve(self):
        """Get coefficient_1_of_power_ratio_curve.

        Returns:
            float: the value of `coefficient_1_of_power_ratio_curve` or None if not set

        """
        return self["Coefficient 1 of Power Ratio Curve"]

    @coefficient_1_of_power_ratio_curve.setter
    def coefficient_1_of_power_ratio_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 1 of Power Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 1 of Power Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 1 of Power Ratio Curve"] = value

    @property
    def coefficient_2_of_power_ratio_curve(self):
        """Get coefficient_2_of_power_ratio_curve.

        Returns:
            float: the value of `coefficient_2_of_power_ratio_curve` or None if not set

        """
        return self["Coefficient 2 of Power Ratio Curve"]

    @coefficient_2_of_power_ratio_curve.setter
    def coefficient_2_of_power_ratio_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 2 of Power Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 2 of Power Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 2 of Power Ratio Curve"] = value

    @property
    def coefficient_3_of_power_ratio_curve(self):
        """Get coefficient_3_of_power_ratio_curve.

        Returns:
            float: the value of `coefficient_3_of_power_ratio_curve` or None if not set

        """
        return self["Coefficient 3 of Power Ratio Curve"]

    @coefficient_3_of_power_ratio_curve.setter
    def coefficient_3_of_power_ratio_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 3 of Power Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 3 of Power Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 3 of Power Ratio Curve"] = value

    @property
    def coefficient_1_of_full_load_ratio_curve(self):
        """Get coefficient_1_of_full_load_ratio_curve.

        Returns:
            float: the value of `coefficient_1_of_full_load_ratio_curve` or None if not set

        """
        return self["Coefficient 1 of Full Load Ratio Curve"]

    @coefficient_1_of_full_load_ratio_curve.setter
    def coefficient_1_of_full_load_ratio_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 1 of Full Load Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 1 of Full Load Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 1 of Full Load Ratio Curve"] = value

    @property
    def coefficient_2_of_full_load_ratio_curve(self):
        """Get coefficient_2_of_full_load_ratio_curve.

        Returns:
            float: the value of `coefficient_2_of_full_load_ratio_curve` or None if not set

        """
        return self["Coefficient 2 of Full Load Ratio Curve"]

    @coefficient_2_of_full_load_ratio_curve.setter
    def coefficient_2_of_full_load_ratio_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 2 of Full Load Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 2 of Full Load Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 2 of Full Load Ratio Curve"] = value

    @property
    def coefficient_3_of_full_load_ratio_curve(self):
        """Get coefficient_3_of_full_load_ratio_curve.

        Returns:
            float: the value of `coefficient_3_of_full_load_ratio_curve` or None if not set

        """
        return self["Coefficient 3 of Full Load Ratio Curve"]

    @coefficient_3_of_full_load_ratio_curve.setter
    def coefficient_3_of_full_load_ratio_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 3 of Full Load Ratio Curve`

        Args:
            value (float): value for IDD Field `Coefficient 3 of Full Load Ratio Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 3 of Full Load Ratio Curve"] = value

    @property
    def chilled_water_outlet_temperature_lower_limit(self):
        """Get chilled_water_outlet_temperature_lower_limit.

        Returns:
            float: the value of `chilled_water_outlet_temperature_lower_limit` or None if not set

        """
        return self["Chilled Water Outlet Temperature Lower Limit"]

    @chilled_water_outlet_temperature_lower_limit.setter
    def chilled_water_outlet_temperature_lower_limit(self, value=None):
        """Corresponds to IDD field `Chilled Water Outlet Temperature Lower
        Limit` Special Gas Turbine Chiller Parameters Below.

        Args:
            value (float): value for IDD Field `Chilled Water Outlet Temperature Lower Limit`
                Units: C
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Outlet Temperature Lower Limit"] = value

    @property
    def coefficient_1_of_fuel_input_curve(self):
        """Get coefficient_1_of_fuel_input_curve.

        Returns:
            float: the value of `coefficient_1_of_fuel_input_curve` or None if not set

        """
        return self["Coefficient 1 of Fuel Input Curve"]

    @coefficient_1_of_fuel_input_curve.setter
    def coefficient_1_of_fuel_input_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 1 of Fuel Input Curve`

        Args:
            value (float): value for IDD Field `Coefficient 1 of Fuel Input Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 1 of Fuel Input Curve"] = value

    @property
    def coefficient_2_of_fuel_input_curve(self):
        """Get coefficient_2_of_fuel_input_curve.

        Returns:
            float: the value of `coefficient_2_of_fuel_input_curve` or None if not set

        """
        return self["Coefficient 2 of Fuel Input Curve"]

    @coefficient_2_of_fuel_input_curve.setter
    def coefficient_2_of_fuel_input_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 2 of Fuel Input Curve`

        Args:
            value (float): value for IDD Field `Coefficient 2 of Fuel Input Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 2 of Fuel Input Curve"] = value

    @property
    def coefficient_3_of_fuel_input_curve(self):
        """Get coefficient_3_of_fuel_input_curve.

        Returns:
            float: the value of `coefficient_3_of_fuel_input_curve` or None if not set

        """
        return self["Coefficient 3 of Fuel Input Curve"]

    @coefficient_3_of_fuel_input_curve.setter
    def coefficient_3_of_fuel_input_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 3 of Fuel Input Curve`

        Args:
            value (float): value for IDD Field `Coefficient 3 of Fuel Input Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 3 of Fuel Input Curve"] = value

    @property
    def coefficient_1_of_temperature_based_fuel_input_curve(self):
        """Get coefficient_1_of_temperature_based_fuel_input_curve.

        Returns:
            float: the value of `coefficient_1_of_temperature_based_fuel_input_curve` or None if not set

        """
        return self["Coefficient 1 of Temperature Based Fuel Input Curve"]

    @coefficient_1_of_temperature_based_fuel_input_curve.setter
    def coefficient_1_of_temperature_based_fuel_input_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 1 of Temperature Based Fuel
        Input Curve`

        Args:
            value (float): value for IDD Field `Coefficient 1 of Temperature Based Fuel Input Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 1 of Temperature Based Fuel Input Curve"] = value

    @property
    def coefficient_2_of_temperature_based_fuel_input_curve(self):
        """Get coefficient_2_of_temperature_based_fuel_input_curve.

        Returns:
            float: the value of `coefficient_2_of_temperature_based_fuel_input_curve` or None if not set

        """
        return self["Coefficient 2 of Temperature Based Fuel Input Curve"]

    @coefficient_2_of_temperature_based_fuel_input_curve.setter
    def coefficient_2_of_temperature_based_fuel_input_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 2 of Temperature Based Fuel
        Input Curve`

        Args:
            value (float): value for IDD Field `Coefficient 2 of Temperature Based Fuel Input Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 2 of Temperature Based Fuel Input Curve"] = value

    @property
    def coefficient_3_of_temperature_based_fuel_input_curve(self):
        """Get coefficient_3_of_temperature_based_fuel_input_curve.

        Returns:
            float: the value of `coefficient_3_of_temperature_based_fuel_input_curve` or None if not set

        """
        return self["Coefficient 3 of Temperature Based Fuel Input Curve"]

    @coefficient_3_of_temperature_based_fuel_input_curve.setter
    def coefficient_3_of_temperature_based_fuel_input_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 3 of Temperature Based Fuel
        Input Curve`

        Args:
            value (float): value for IDD Field `Coefficient 3 of Temperature Based Fuel Input Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 3 of Temperature Based Fuel Input Curve"] = value

    @property
    def coefficient_1_of_exhaust_flow_curve(self):
        """Get coefficient_1_of_exhaust_flow_curve.

        Returns:
            float: the value of `coefficient_1_of_exhaust_flow_curve` or None if not set

        """
        return self["Coefficient 1 of Exhaust Flow Curve"]

    @coefficient_1_of_exhaust_flow_curve.setter
    def coefficient_1_of_exhaust_flow_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 1 of Exhaust Flow Curve`

        Args:
            value (float): value for IDD Field `Coefficient 1 of Exhaust Flow Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 1 of Exhaust Flow Curve"] = value

    @property
    def coefficient_2_of_exhaust_flow_curve(self):
        """Get coefficient_2_of_exhaust_flow_curve.

        Returns:
            float: the value of `coefficient_2_of_exhaust_flow_curve` or None if not set

        """
        return self["Coefficient 2 of Exhaust Flow Curve"]

    @coefficient_2_of_exhaust_flow_curve.setter
    def coefficient_2_of_exhaust_flow_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 2 of Exhaust Flow Curve`

        Args:
            value (float): value for IDD Field `Coefficient 2 of Exhaust Flow Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 2 of Exhaust Flow Curve"] = value

    @property
    def coefficient_3_of_exhaust_flow_curve(self):
        """Get coefficient_3_of_exhaust_flow_curve.

        Returns:
            float: the value of `coefficient_3_of_exhaust_flow_curve` or None if not set

        """
        return self["Coefficient 3 of Exhaust Flow Curve"]

    @coefficient_3_of_exhaust_flow_curve.setter
    def coefficient_3_of_exhaust_flow_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 3 of Exhaust Flow Curve`

        Args:
            value (float): value for IDD Field `Coefficient 3 of Exhaust Flow Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 3 of Exhaust Flow Curve"] = value

    @property
    def coefficient_1_of_exhaust_gas_temperature_curve(self):
        """Get coefficient_1_of_exhaust_gas_temperature_curve.

        Returns:
            float: the value of `coefficient_1_of_exhaust_gas_temperature_curve` or None if not set

        """
        return self["Coefficient 1 of Exhaust Gas Temperature Curve"]

    @coefficient_1_of_exhaust_gas_temperature_curve.setter
    def coefficient_1_of_exhaust_gas_temperature_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 1 of Exhaust Gas Temperature
        Curve`

        Args:
            value (float): value for IDD Field `Coefficient 1 of Exhaust Gas Temperature Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 1 of Exhaust Gas Temperature Curve"] = value

    @property
    def coefficient_2_of_exhaust_gas_temperature_curve(self):
        """Get coefficient_2_of_exhaust_gas_temperature_curve.

        Returns:
            float: the value of `coefficient_2_of_exhaust_gas_temperature_curve` or None if not set

        """
        return self["Coefficient 2 of Exhaust Gas Temperature Curve"]

    @coefficient_2_of_exhaust_gas_temperature_curve.setter
    def coefficient_2_of_exhaust_gas_temperature_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 2 of Exhaust Gas Temperature
        Curve`

        Args:
            value (float): value for IDD Field `Coefficient 2 of Exhaust Gas Temperature Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 2 of Exhaust Gas Temperature Curve"] = value

    @property
    def coefficient_3_of_exhaust_gas_temperature_curve(self):
        """Get coefficient_3_of_exhaust_gas_temperature_curve.

        Returns:
            float: the value of `coefficient_3_of_exhaust_gas_temperature_curve` or None if not set

        """
        return self["Coefficient 3 of Exhaust Gas Temperature Curve"]

    @coefficient_3_of_exhaust_gas_temperature_curve.setter
    def coefficient_3_of_exhaust_gas_temperature_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 3 of Exhaust Gas Temperature
        Curve`

        Args:
            value (float): value for IDD Field `Coefficient 3 of Exhaust Gas Temperature Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 3 of Exhaust Gas Temperature Curve"] = value

    @property
    def coefficient_1_of_temperature_based_exhaust_gas_temperature_curve(self):
        """Get
        coefficient_1_of_temperature_based_exhaust_gas_temperature_curve.

        Returns:
            float: the value of `coefficient_1_of_temperature_based_exhaust_gas_temperature_curve` or None if not set

        """
        return self[
            "Coefficient 1 of Temperature Based Exhaust Gas Temperature Curve"]

    @coefficient_1_of_temperature_based_exhaust_gas_temperature_curve.setter
    def coefficient_1_of_temperature_based_exhaust_gas_temperature_curve(
            self,
            value=None):
        """Corresponds to IDD field `Coefficient 1 of Temperature Based Exhaust
        Gas Temperature Curve`

        Args:
            value (float): value for IDD Field `Coefficient 1 of Temperature Based Exhaust Gas Temperature Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Coefficient 1 of Temperature Based Exhaust Gas Temperature Curve"] = value

    @property
    def coefficient_2_of_temperature_based_exhaust_gas_temperature_curve(self):
        """Get
        coefficient_2_of_temperature_based_exhaust_gas_temperature_curve.

        Returns:
            float: the value of `coefficient_2_of_temperature_based_exhaust_gas_temperature_curve` or None if not set

        """
        return self[
            "Coefficient 2 of Temperature Based Exhaust Gas Temperature Curve"]

    @coefficient_2_of_temperature_based_exhaust_gas_temperature_curve.setter
    def coefficient_2_of_temperature_based_exhaust_gas_temperature_curve(
            self,
            value=None):
        """Corresponds to IDD field `Coefficient 2 of Temperature Based Exhaust
        Gas Temperature Curve`

        Args:
            value (float): value for IDD Field `Coefficient 2 of Temperature Based Exhaust Gas Temperature Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Coefficient 2 of Temperature Based Exhaust Gas Temperature Curve"] = value

    @property
    def coefficient_3_of_temperature_based_exhaust_gas_temperature_curve(self):
        """Get
        coefficient_3_of_temperature_based_exhaust_gas_temperature_curve.

        Returns:
            float: the value of `coefficient_3_of_temperature_based_exhaust_gas_temperature_curve` or None if not set

        """
        return self[
            "Coefficient 3 of Temperature Based Exhaust Gas Temperature Curve"]

    @coefficient_3_of_temperature_based_exhaust_gas_temperature_curve.setter
    def coefficient_3_of_temperature_based_exhaust_gas_temperature_curve(
            self,
            value=None):
        """Corresponds to IDD field `Coefficient 3 of Temperature Based Exhaust
        Gas Temperature Curve`

        Args:
            value (float): value for IDD Field `Coefficient 3 of Temperature Based Exhaust Gas Temperature Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Coefficient 3 of Temperature Based Exhaust Gas Temperature Curve"] = value

    @property
    def coefficient_1_of_recovery_lube_heat_curve(self):
        """Get coefficient_1_of_recovery_lube_heat_curve.

        Returns:
            float: the value of `coefficient_1_of_recovery_lube_heat_curve` or None if not set

        """
        return self["Coefficient 1 of Recovery Lube Heat Curve"]

    @coefficient_1_of_recovery_lube_heat_curve.setter
    def coefficient_1_of_recovery_lube_heat_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 1 of Recovery Lube Heat Curve`

        Args:
            value (float): value for IDD Field `Coefficient 1 of Recovery Lube Heat Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 1 of Recovery Lube Heat Curve"] = value

    @property
    def coefficient_2_of_recovery_lube_heat_curve(self):
        """Get coefficient_2_of_recovery_lube_heat_curve.

        Returns:
            float: the value of `coefficient_2_of_recovery_lube_heat_curve` or None if not set

        """
        return self["Coefficient 2 of Recovery Lube Heat Curve"]

    @coefficient_2_of_recovery_lube_heat_curve.setter
    def coefficient_2_of_recovery_lube_heat_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 2 of Recovery Lube Heat Curve`

        Args:
            value (float): value for IDD Field `Coefficient 2 of Recovery Lube Heat Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 2 of Recovery Lube Heat Curve"] = value

    @property
    def coefficient_3_of_recovery_lube_heat_curve(self):
        """Get coefficient_3_of_recovery_lube_heat_curve.

        Returns:
            float: the value of `coefficient_3_of_recovery_lube_heat_curve` or None if not set

        """
        return self["Coefficient 3 of Recovery Lube Heat Curve"]

    @coefficient_3_of_recovery_lube_heat_curve.setter
    def coefficient_3_of_recovery_lube_heat_curve(self, value=None):
        """Corresponds to IDD field `Coefficient 3 of Recovery Lube Heat Curve`

        Args:
            value (float): value for IDD Field `Coefficient 3 of Recovery Lube Heat Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Coefficient 3 of Recovery Lube Heat Curve"] = value

    @property
    def coefficient_1_of_ufactor_times_area_curve(self):
        """Get coefficient_1_of_ufactor_times_area_curve.

        Returns:
            float: the value of `coefficient_1_of_ufactor_times_area_curve` or None if not set

        """
        return self["Coefficient 1 of U-Factor Times Area Curve"]

    @coefficient_1_of_ufactor_times_area_curve.setter
    def coefficient_1_of_ufactor_times_area_curve(self, value=None):
        """  Corresponds to IDD field `Coefficient 1 of U-Factor Times Area Curve`

        Args:
            value (float): value for IDD Field `Coefficient 1 of U-Factor Times Area Curve`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Coefficient 1 of U-Factor Times Area Curve"] = value

    @property
    def coefficient_2_of_ufactor_times_area_curve(self):
        """Get coefficient_2_of_ufactor_times_area_curve.

        Returns:
            float: the value of `coefficient_2_of_ufactor_times_area_curve` or None if not set

        """
        return self["Coefficient 2 of U-Factor Times Area Curve"]

    @coefficient_2_of_ufactor_times_area_curve.setter
    def coefficient_2_of_ufactor_times_area_curve(self, value=None):
        """  Corresponds to IDD field `Coefficient 2 of U-Factor Times Area Curve`
        typical value .9

        Args:
            value (float): value for IDD Field `Coefficient 2 of U-Factor Times Area Curve`
                value <= 2.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Coefficient 2 of U-Factor Times Area Curve"] = value

    @property
    def gas_turbine_engine_capacity(self):
        """Get gas_turbine_engine_capacity.

        Returns:
            float: the value of `gas_turbine_engine_capacity` or None if not set

        """
        return self["Gas Turbine Engine Capacity"]

    @gas_turbine_engine_capacity.setter
    def gas_turbine_engine_capacity(self, value=None):
        """Corresponds to IDD field `Gas Turbine Engine Capacity`

        Args:
            value (float or "Autosize"): value for IDD Field `Gas Turbine Engine Capacity`
                Units: W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Gas Turbine Engine Capacity"] = value

    @property
    def maximum_exhaust_flow_per_unit_of_power_output(self):
        """Get maximum_exhaust_flow_per_unit_of_power_output.

        Returns:
            float: the value of `maximum_exhaust_flow_per_unit_of_power_output` or None if not set

        """
        return self["Maximum Exhaust Flow per Unit of Power Output"]

    @maximum_exhaust_flow_per_unit_of_power_output.setter
    def maximum_exhaust_flow_per_unit_of_power_output(self, value=None):
        """Corresponds to IDD field `Maximum Exhaust Flow per Unit of Power
        Output`

        Args:
            value (float): value for IDD Field `Maximum Exhaust Flow per Unit of Power Output`
                Units: (kg/s)/W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Maximum Exhaust Flow per Unit of Power Output"] = value

    @property
    def design_steam_saturation_temperature(self):
        """Get design_steam_saturation_temperature.

        Returns:
            float: the value of `design_steam_saturation_temperature` or None if not set

        """
        return self["Design Steam Saturation Temperature"]

    @design_steam_saturation_temperature.setter
    def design_steam_saturation_temperature(self, value=None):
        """Corresponds to IDD field `Design Steam Saturation Temperature`

        Args:
            value (float): value for IDD Field `Design Steam Saturation Temperature`
                Units: C
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Steam Saturation Temperature"] = value

    @property
    def fuel_higher_heating_value(self):
        """Get fuel_higher_heating_value.

        Returns:
            float: the value of `fuel_higher_heating_value` or None if not set

        """
        return self["Fuel Higher Heating Value"]

    @fuel_higher_heating_value.setter
    def fuel_higher_heating_value(self, value=None):
        """Corresponds to IDD field `Fuel Higher Heating Value`

        Args:
            value (float): value for IDD Field `Fuel Higher Heating Value`
                Units: kJ/kg
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Fuel Higher Heating Value"] = value

    @property
    def design_heat_recovery_water_flow_rate(self):
        """Get design_heat_recovery_water_flow_rate.

        Returns:
            float: the value of `design_heat_recovery_water_flow_rate` or None if not set

        """
        return self["Design Heat Recovery Water Flow Rate"]

    @design_heat_recovery_water_flow_rate.setter
    def design_heat_recovery_water_flow_rate(self, value=None):
        """  Corresponds to IDD field `Design Heat Recovery Water Flow Rate`
        If non-zero, then the heat recovery inlet and outlet node names must be entered.

        Args:
            value (float): value for IDD Field `Design Heat Recovery Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Design Heat Recovery Water Flow Rate"] = value

    @property
    def heat_recovery_inlet_node_name(self):
        """Get heat_recovery_inlet_node_name.

        Returns:
            str: the value of `heat_recovery_inlet_node_name` or None if not set

        """
        return self["Heat Recovery Inlet Node Name"]

    @heat_recovery_inlet_node_name.setter
    def heat_recovery_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Heat Recovery Inlet Node Name`

        Args:
            value (str): value for IDD Field `Heat Recovery Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Heat Recovery Inlet Node Name"] = value

    @property
    def heat_recovery_outlet_node_name(self):
        """Get heat_recovery_outlet_node_name.

        Returns:
            str: the value of `heat_recovery_outlet_node_name` or None if not set

        """
        return self["Heat Recovery Outlet Node Name"]

    @heat_recovery_outlet_node_name.setter
    def heat_recovery_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Heat Recovery Outlet Node Name`

        Args:
            value (str): value for IDD Field `Heat Recovery Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Heat Recovery Outlet Node Name"] = value

    @property
    def chiller_flow_mode(self):
        """Get chiller_flow_mode.

        Returns:
            str: the value of `chiller_flow_mode` or None if not set

        """
        return self["Chiller Flow Mode"]

    @chiller_flow_mode.setter
    def chiller_flow_mode(self, value="NotModulated"):
        """Corresponds to IDD field `Chiller Flow Mode` Select operating mode
        for fluid flow through the chiller. "NotModulated" is for either
        variable or constant pumping with flow controlled by the external plant
        system. "ConstantFlow" is for constant pumping with flow controlled by
        chiller to operate at full design flow rate.
        "LeavingSetpointModulated" is for variable pumping with flow controlled
        by chiller to vary flow to target a leaving temperature setpoint.

        Args:
            value (str): value for IDD Field `Chiller Flow Mode`
                Default value: NotModulated
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Flow Mode"] = value

    @property
    def fuel_type(self):
        """Get fuel_type.

        Returns:
            str: the value of `fuel_type` or None if not set

        """
        return self["Fuel Type"]

    @fuel_type.setter
    def fuel_type(self, value="NaturalGas"):
        """Corresponds to IDD field `Fuel Type`

        Args:
            value (str): value for IDD Field `Fuel Type`
                Default value: NaturalGas
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Fuel Type"] = value

    @property
    def heat_recovery_maximum_temperature(self):
        """Get heat_recovery_maximum_temperature.

        Returns:
            float: the value of `heat_recovery_maximum_temperature` or None if not set

        """
        return self["Heat Recovery Maximum Temperature"]

    @heat_recovery_maximum_temperature.setter
    def heat_recovery_maximum_temperature(self, value=80.0):
        """Corresponds to IDD field `Heat Recovery Maximum Temperature`

        Args:
            value (float): value for IDD Field `Heat Recovery Maximum Temperature`
                Units: C
                Default value: 80.0
                value <= 100.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Heat Recovery Maximum Temperature"] = value

    @property
    def sizing_factor(self):
        """Get sizing_factor.

        Returns:
            float: the value of `sizing_factor` or None if not set

        """
        return self["Sizing Factor"]

    @sizing_factor.setter
    def sizing_factor(self, value=1.0):
        """Corresponds to IDD field `Sizing Factor` Multiplies the autosized
        capacity and flow rates.

        Args:
            value (float): value for IDD Field `Sizing Factor`
                Default value: 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Sizing Factor"] = value

    @property
    def basin_heater_capacity(self):
        """Get basin_heater_capacity.

        Returns:
            float: the value of `basin_heater_capacity` or None if not set

        """
        return self["Basin Heater Capacity"]

    @basin_heater_capacity.setter
    def basin_heater_capacity(self, value=None):
        """  Corresponds to IDD field `Basin Heater Capacity`
        This field is only used for Condenser Type = EvaporativelyCooled and for periods
        when the basin heater is available (field Basin Heater Operating Schedule Name).
        For this situation, The heater maintains the basin water temperature at the basin heater
        setpoint temperature when the outdoor air temperature falls below the setpoint temperature.
        The basin heater only operates when the chiller is not operating.

        Args:
            value (float): value for IDD Field `Basin Heater Capacity`
                Units: W/K
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Basin Heater Capacity"] = value

    @property
    def basin_heater_setpoint_temperature(self):
        """Get basin_heater_setpoint_temperature.

        Returns:
            float: the value of `basin_heater_setpoint_temperature` or None if not set

        """
        return self["Basin Heater Setpoint Temperature"]

    @basin_heater_setpoint_temperature.setter
    def basin_heater_setpoint_temperature(self, value=2.0):
        """  Corresponds to IDD field `Basin Heater Setpoint Temperature`
        This field is only used for Condenser Type = EvaporativelyCooled.
        Enter the outdoor dry-bulb temperature when the basin heater turns on.

        Args:
            value (float): value for IDD Field `Basin Heater Setpoint Temperature`
                Units: C
                Default value: 2.0
                value >= 2.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Basin Heater Setpoint Temperature"] = value

    @property
    def basin_heater_operating_schedule_name(self):
        """Get basin_heater_operating_schedule_name.

        Returns:
            str: the value of `basin_heater_operating_schedule_name` or None if not set

        """
        return self["Basin Heater Operating Schedule Name"]

    @basin_heater_operating_schedule_name.setter
    def basin_heater_operating_schedule_name(self, value=None):
        """  Corresponds to IDD field `Basin Heater Operating Schedule Name`
        This field is only used for Condenser Type = EvaporativelyCooled.
        Schedule values greater than 0 allow the basin heater to operate whenever the outdoor
        air dry-bulb temperature is below the basin heater setpoint temperature.
        If a schedule name is not entered, the basin heater is allowed to operate
        throughout the entire simulation.

        Args:
            value (str): value for IDD Field `Basin Heater Operating Schedule Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Basin Heater Operating Schedule Name"] = value




class ChillerHeaterAbsorptionDirectFired(DataObject):

    """ Corresponds to IDD object `ChillerHeater:Absorption:DirectFired`
        Direct fired gas absorption chiller-heater using performance curves similar to DOE-2
    """
    schema = {'min-fields': 35,
              'name': u'ChillerHeater:Absorption:DirectFired',
              'pyname': u'ChillerHeaterAbsorptionDirectFired',
              'format': None,
              'fields': OrderedDict([(u'name',
                                      {'name': u'Name',
                                       'pyname': u'name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'nominal cooling capacity',
                                      {'name': u'Nominal Cooling Capacity',
                                       'pyname': u'nominal_cooling_capacity',
                                       'default': 'autosize',
                                       'required-field': True,
                                       'autosizable': True,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'W'}),
                                     (u'heating to cooling capacity ratio',
                                      {'name': u'Heating to Cooling Capacity Ratio',
                                       'pyname': u'heating_to_cooling_capacity_ratio',
                                       'default': 0.8,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'fuel input to cooling output ratio',
                                      {'name': u'Fuel Input to Cooling Output Ratio',
                                       'pyname': u'fuel_input_to_cooling_output_ratio',
                                       'default': 0.97,
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'fuel input to heating output ratio',
                                      {'name': u'Fuel Input to Heating Output Ratio',
                                       'pyname': u'fuel_input_to_heating_output_ratio',
                                       'default': 1.25,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'electric input to cooling output ratio',
                                      {'name': u'Electric Input to Cooling Output Ratio',
                                       'pyname': u'electric_input_to_cooling_output_ratio',
                                       'default': 0.01,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'electric input to heating output ratio',
                                      {'name': u'Electric Input to Heating Output Ratio',
                                       'pyname': u'electric_input_to_heating_output_ratio',
                                       'default': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'chilled water inlet node name',
                                      {'name': u'Chilled Water Inlet Node Name',
                                       'pyname': u'chilled_water_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'chilled water outlet node name',
                                      {'name': u'Chilled Water Outlet Node Name',
                                       'pyname': u'chilled_water_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'condenser inlet node name',
                                      {'name': u'Condenser Inlet Node Name',
                                       'pyname': u'condenser_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'condenser outlet node name',
                                      {'name': u'Condenser Outlet Node Name',
                                       'pyname': u'condenser_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'hot water inlet node name',
                                      {'name': u'Hot Water Inlet Node Name',
                                       'pyname': u'hot_water_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'hot water outlet node name',
                                      {'name': u'Hot Water Outlet Node Name',
                                       'pyname': u'hot_water_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'minimum part load ratio',
                                      {'name': u'Minimum Part Load Ratio',
                                       'pyname': u'minimum_part_load_ratio',
                                       'default': 0.1,
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'maximum part load ratio',
                                      {'name': u'Maximum Part Load Ratio',
                                       'pyname': u'maximum_part_load_ratio',
                                       'default': 1.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.5,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'optimum part load ratio',
                                      {'name': u'Optimum Part Load Ratio',
                                       'pyname': u'optimum_part_load_ratio',
                                       'default': 1.0,
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'design entering condenser water temperature',
                                      {'name': u'Design Entering Condenser Water Temperature',
                                       'pyname': u'design_entering_condenser_water_temperature',
                                       'default': 29.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'C'}),
                                     (u'design leaving chilled water temperature',
                                      {'name': u'Design Leaving Chilled Water Temperature',
                                       'pyname': u'design_leaving_chilled_water_temperature',
                                       'default': 7.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'C'}),
                                     (u'design chilled water flow rate',
                                      {'name': u'Design Chilled Water Flow Rate',
                                       'pyname': u'design_chilled_water_flow_rate',
                                       'default': 'autosize',
                                       'required-field': False,
                                       'autosizable': True,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'm3/s'}),
                                     (u'design condenser water flow rate',
                                      {'name': u'Design Condenser Water Flow Rate',
                                       'pyname': u'design_condenser_water_flow_rate',
                                       'default': 'autosize',
                                       'required-field': True,
                                       'autosizable': True,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'm3/s'}),
                                     (u'design hot water flow rate',
                                      {'name': u'Design Hot Water Flow Rate',
                                       'pyname': u'design_hot_water_flow_rate',
                                       'default': 'autosize',
                                       'required-field': False,
                                       'autosizable': True,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'm3/s'}),
                                     (u'cooling capacity function of temperature curve name',
                                      {'name': u'Cooling Capacity Function of Temperature Curve Name',
                                       'pyname': u'cooling_capacity_function_of_temperature_curve_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'fuel input to cooling output ratio function of temperature curve name',
                                      {'name': u'Fuel Input to Cooling Output Ratio Function of Temperature Curve Name',
                                       'pyname': u'fuel_input_to_cooling_output_ratio_function_of_temperature_curve_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'fuel input to cooling output ratio function of part load ratio curve name',
                                      {'name': u'Fuel Input to Cooling Output Ratio Function of Part Load Ratio Curve Name',
                                       'pyname': u'fuel_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'electric input to cooling output ratio function of temperature curve name',
                                      {'name': u'Electric Input to Cooling Output Ratio Function of Temperature Curve Name',
                                       'pyname': u'electric_input_to_cooling_output_ratio_function_of_temperature_curve_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'electric input to cooling output ratio function of part load ratio curve name',
                                      {'name': u'Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name',
                                       'pyname': u'electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'heating capacity function of cooling capacity curve name',
                                      {'name': u'Heating Capacity Function of Cooling Capacity Curve Name',
                                       'pyname': u'heating_capacity_function_of_cooling_capacity_curve_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'fuel input to heat output ratio during heating only operation curve name',
                                      {'name': u'Fuel Input to Heat Output Ratio During Heating Only Operation Curve Name',
                                       'pyname': u'fuel_input_to_heat_output_ratio_during_heating_only_operation_curve_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'temperature curve input variable',
                                      {'name': u'Temperature Curve Input Variable',
                                       'pyname': u'temperature_curve_input_variable',
                                       'default': u'EnteringCondenser',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'LeavingCondenser',
                                                           u'EnteringCondenser'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'condenser type',
                                      {'name': u'Condenser Type',
                                       'pyname': u'condenser_type',
                                       'default': u'WaterCooled',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'AirCooled',
                                                           u'WaterCooled'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'chilled water temperature lower limit',
                                      {'name': u'Chilled Water Temperature Lower Limit',
                                       'pyname': u'chilled_water_temperature_lower_limit',
                                       'default': 2.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'C'}),
                                     (u'fuel higher heating value',
                                      {'name': u'Fuel Higher Heating Value',
                                       'pyname': u'fuel_higher_heating_value',
                                       'default': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'kJ/kg'}),
                                     (u'chiller flow mode',
                                      {'name': u'Chiller Flow Mode',
                                       'pyname': u'chiller_flow_mode',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'fuel type',
                                      {'name': u'Fuel Type',
                                       'pyname': u'fuel_type',
                                       'default': u'NaturalGas',
                                       'required-field': True,
                                       'autosizable': False,
                                       'accepted-values': [u'NaturalGas',
                                                           u'PropaneGas',
                                                           u'Diesel',
                                                           u'Gasoline',
                                                           u'FuelOil#1',
                                                           u'FuelOil#2',
                                                           u'OtherFuel1',
                                                           u'OtherFuel2'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'sizing factor',
                                      {'name': u'Sizing Factor',
                                       'pyname': u'sizing_factor',
                                       'default': 1.0,
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'})]),
              'extensible-fields': OrderedDict(),
              'unique-object': False,
              'required-object': False,
              'group': u'Plant Heating and Cooling Equipment'}

    @property
    def name(self):
        """Get name.

        Returns:
            str: the value of `name` or None if not set

        """
        return self["Name"]

    @name.setter
    def name(self, value=None):
        """Corresponds to IDD field `Name`

        Args:
            value (str): value for IDD Field `Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Name"] = value

    @property
    def nominal_cooling_capacity(self):
        """Get nominal_cooling_capacity.

        Returns:
            float: the value of `nominal_cooling_capacity` or None if not set

        """
        return self["Nominal Cooling Capacity"]

    @nominal_cooling_capacity.setter
    def nominal_cooling_capacity(self, value="autosize"):
        """Corresponds to IDD field `Nominal Cooling Capacity`

        Args:
            value (float or "Autosize"): value for IDD Field `Nominal Cooling Capacity`
                Units: W
                Default value: "autosize"
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Nominal Cooling Capacity"] = value

    @property
    def heating_to_cooling_capacity_ratio(self):
        """Get heating_to_cooling_capacity_ratio.

        Returns:
            float: the value of `heating_to_cooling_capacity_ratio` or None if not set

        """
        return self["Heating to Cooling Capacity Ratio"]

    @heating_to_cooling_capacity_ratio.setter
    def heating_to_cooling_capacity_ratio(self, value=0.8):
        """Corresponds to IDD field `Heating to Cooling Capacity Ratio` A
        positive fraction that represents the ratio of the heating capacity
        divided by the cooling capacity at rated conditions.

        Args:
            value (float): value for IDD Field `Heating to Cooling Capacity Ratio`
                Default value: 0.8
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Heating to Cooling Capacity Ratio"] = value

    @property
    def fuel_input_to_cooling_output_ratio(self):
        """Get fuel_input_to_cooling_output_ratio.

        Returns:
            float: the value of `fuel_input_to_cooling_output_ratio` or None if not set

        """
        return self["Fuel Input to Cooling Output Ratio"]

    @fuel_input_to_cooling_output_ratio.setter
    def fuel_input_to_cooling_output_ratio(self, value=0.97):
        """Corresponds to IDD field `Fuel Input to Cooling Output Ratio` The
        positive fraction that represents the ratio of the instantaneous fuel
        used divided by the cooling capacity at rated conditions.

        Args:
            value (float): value for IDD Field `Fuel Input to Cooling Output Ratio`
                Default value: 0.97
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Fuel Input to Cooling Output Ratio"] = value

    @property
    def fuel_input_to_heating_output_ratio(self):
        """Get fuel_input_to_heating_output_ratio.

        Returns:
            float: the value of `fuel_input_to_heating_output_ratio` or None if not set

        """
        return self["Fuel Input to Heating Output Ratio"]

    @fuel_input_to_heating_output_ratio.setter
    def fuel_input_to_heating_output_ratio(self, value=1.25):
        """Corresponds to IDD field `Fuel Input to Heating Output Ratio` The
        positive fraction that represents the ratio of the instantaneous fuel
        used divided by the nominal heating capacity.

        Args:
            value (float): value for IDD Field `Fuel Input to Heating Output Ratio`
                Default value: 1.25
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Fuel Input to Heating Output Ratio"] = value

    @property
    def electric_input_to_cooling_output_ratio(self):
        """Get electric_input_to_cooling_output_ratio.

        Returns:
            float: the value of `electric_input_to_cooling_output_ratio` or None if not set

        """
        return self["Electric Input to Cooling Output Ratio"]

    @electric_input_to_cooling_output_ratio.setter
    def electric_input_to_cooling_output_ratio(self, value=0.01):
        """Corresponds to IDD field `Electric Input to Cooling Output Ratio`
        The positive fraction that represents the ratio of the instantaneous
        electricity used divided by the cooling capacity at rated conditions.
        If the chiller is both heating and cooling only the cooling electricity
        is used.

        Args:
            value (float): value for IDD Field `Electric Input to Cooling Output Ratio`
                Default value: 0.01
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Electric Input to Cooling Output Ratio"] = value

    @property
    def electric_input_to_heating_output_ratio(self):
        """Get electric_input_to_heating_output_ratio.

        Returns:
            float: the value of `electric_input_to_heating_output_ratio` or None if not set

        """
        return self["Electric Input to Heating Output Ratio"]

    @electric_input_to_heating_output_ratio.setter
    def electric_input_to_heating_output_ratio(self, value=None):
        """Corresponds to IDD field `Electric Input to Heating Output Ratio`
        The positive fraction that represents the ratio of the instantaneous
        electricity used divided by the nominal heating capacity. If the
        chiller is both heating and cooling only the cooling electricity is
        used.

        Args:
            value (float): value for IDD Field `Electric Input to Heating Output Ratio`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Electric Input to Heating Output Ratio"] = value

    @property
    def chilled_water_inlet_node_name(self):
        """Get chilled_water_inlet_node_name.

        Returns:
            str: the value of `chilled_water_inlet_node_name` or None if not set

        """
        return self["Chilled Water Inlet Node Name"]

    @chilled_water_inlet_node_name.setter
    def chilled_water_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Chilled Water Inlet Node Name`

        Args:
            value (str): value for IDD Field `Chilled Water Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Inlet Node Name"] = value

    @property
    def chilled_water_outlet_node_name(self):
        """Get chilled_water_outlet_node_name.

        Returns:
            str: the value of `chilled_water_outlet_node_name` or None if not set

        """
        return self["Chilled Water Outlet Node Name"]

    @chilled_water_outlet_node_name.setter
    def chilled_water_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Chilled Water Outlet Node Name`

        Args:
            value (str): value for IDD Field `Chilled Water Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Outlet Node Name"] = value

    @property
    def condenser_inlet_node_name(self):
        """Get condenser_inlet_node_name.

        Returns:
            str: the value of `condenser_inlet_node_name` or None if not set

        """
        return self["Condenser Inlet Node Name"]

    @condenser_inlet_node_name.setter
    def condenser_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Condenser Inlet Node Name`

        Args:
            value (str): value for IDD Field `Condenser Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Inlet Node Name"] = value

    @property
    def condenser_outlet_node_name(self):
        """Get condenser_outlet_node_name.

        Returns:
            str: the value of `condenser_outlet_node_name` or None if not set

        """
        return self["Condenser Outlet Node Name"]

    @condenser_outlet_node_name.setter
    def condenser_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Condenser Outlet Node Name`

        Args:
            value (str): value for IDD Field `Condenser Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Outlet Node Name"] = value

    @property
    def hot_water_inlet_node_name(self):
        """Get hot_water_inlet_node_name.

        Returns:
            str: the value of `hot_water_inlet_node_name` or None if not set

        """
        return self["Hot Water Inlet Node Name"]

    @hot_water_inlet_node_name.setter
    def hot_water_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Hot Water Inlet Node Name`

        Args:
            value (str): value for IDD Field `Hot Water Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Hot Water Inlet Node Name"] = value

    @property
    def hot_water_outlet_node_name(self):
        """Get hot_water_outlet_node_name.

        Returns:
            str: the value of `hot_water_outlet_node_name` or None if not set

        """
        return self["Hot Water Outlet Node Name"]

    @hot_water_outlet_node_name.setter
    def hot_water_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Hot Water Outlet Node Name`

        Args:
            value (str): value for IDD Field `Hot Water Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Hot Water Outlet Node Name"] = value

    @property
    def minimum_part_load_ratio(self):
        """Get minimum_part_load_ratio.

        Returns:
            float: the value of `minimum_part_load_ratio` or None if not set

        """
        return self["Minimum Part Load Ratio"]

    @minimum_part_load_ratio.setter
    def minimum_part_load_ratio(self, value=0.1):
        """Corresponds to IDD field `Minimum Part Load Ratio` The positive
        fraction that represents the minimum cooling output possible when
        operated continually at rated temperature conditions divided by the
        nominal cooling capacity at those same conditions.  If the load on the
        chiller is below this fraction the chiller will cycle.

        Args:
            value (float): value for IDD Field `Minimum Part Load Ratio`
                Default value: 0.1
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Minimum Part Load Ratio"] = value

    @property
    def maximum_part_load_ratio(self):
        """Get maximum_part_load_ratio.

        Returns:
            float: the value of `maximum_part_load_ratio` or None if not set

        """
        return self["Maximum Part Load Ratio"]

    @maximum_part_load_ratio.setter
    def maximum_part_load_ratio(self, value=1.0):
        """  Corresponds to IDD field `Maximum Part Load Ratio`
        The positive fraction that represents the maximum cooling output possible at
        rated temperature conditions divided by the nominal cooling capacity at those
        same conditions.  If greater than 1.0, the chiller is typically thought of as
        capable of being overloaded.

        Args:
            value (float): value for IDD Field `Maximum Part Load Ratio`
                Default value: 1.0
                value >= 0.5
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Maximum Part Load Ratio"] = value

    @property
    def optimum_part_load_ratio(self):
        """Get optimum_part_load_ratio.

        Returns:
            float: the value of `optimum_part_load_ratio` or None if not set

        """
        return self["Optimum Part Load Ratio"]

    @optimum_part_load_ratio.setter
    def optimum_part_load_ratio(self, value=1.0):
        """Corresponds to IDD field `Optimum Part Load Ratio` The positive
        fraction that represents the optimal cooling output at rated
        temperature conditions divided by the nominal cooling capacity at those
        same conditions.  It represents the most desirable operating point for
        the chiller.

        Args:
            value (float): value for IDD Field `Optimum Part Load Ratio`
                Default value: 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Optimum Part Load Ratio"] = value

    @property
    def design_entering_condenser_water_temperature(self):
        """Get design_entering_condenser_water_temperature.

        Returns:
            float: the value of `design_entering_condenser_water_temperature` or None if not set

        """
        return self["Design Entering Condenser Water Temperature"]

    @design_entering_condenser_water_temperature.setter
    def design_entering_condenser_water_temperature(self, value=29.0):
        """Corresponds to IDD field `Design Entering Condenser Water
        Temperature` The temperature of the water entering the condenser of the
        chiller when operating at design conditions.  This is usually based on
        the temperature delivered by the cooling tower in a water cooled
        application.

        Args:
            value (float): value for IDD Field `Design Entering Condenser Water Temperature`
                Units: C
                Default value: 29.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Entering Condenser Water Temperature"] = value

    @property
    def design_leaving_chilled_water_temperature(self):
        """Get design_leaving_chilled_water_temperature.

        Returns:
            float: the value of `design_leaving_chilled_water_temperature` or None if not set

        """
        return self["Design Leaving Chilled Water Temperature"]

    @design_leaving_chilled_water_temperature.setter
    def design_leaving_chilled_water_temperature(self, value=7.0):
        """Corresponds to IDD field `Design Leaving Chilled Water Temperature`
        The temperature of the water leaving the evaporator of the chiller when
        operating at design conditions also called the chilled water supply
        temperature or leaving chilled water temperature.

        Args:
            value (float): value for IDD Field `Design Leaving Chilled Water Temperature`
                Units: C
                Default value: 7.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Leaving Chilled Water Temperature"] = value

    @property
    def design_chilled_water_flow_rate(self):
        """Get design_chilled_water_flow_rate.

        Returns:
            float: the value of `design_chilled_water_flow_rate` or None if not set

        """
        return self["Design Chilled Water Flow Rate"]

    @design_chilled_water_flow_rate.setter
    def design_chilled_water_flow_rate(self, value="autosize"):
        """Corresponds to IDD field `Design Chilled Water Flow Rate` For
        variable volume this is the max flow & for constant flow this is the
        flow.

        Args:
            value (float or "Autosize"): value for IDD Field `Design Chilled Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                Default value: "autosize"
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Chilled Water Flow Rate"] = value

    @property
    def design_condenser_water_flow_rate(self):
        """Get design_condenser_water_flow_rate.

        Returns:
            float: the value of `design_condenser_water_flow_rate` or None if not set

        """
        return self["Design Condenser Water Flow Rate"]

    @design_condenser_water_flow_rate.setter
    def design_condenser_water_flow_rate(self, value="autosize"):
        """  Corresponds to IDD field `Design Condenser Water Flow Rate`
        The water flow rate at design conditions through the condenser.
        This field is not used for Condenser Type = AirCooled

        Args:
            value (float or "Autosize"): value for IDD Field `Design Condenser Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                Default value: "autosize"
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Design Condenser Water Flow Rate"] = value

    @property
    def design_hot_water_flow_rate(self):
        """Get design_hot_water_flow_rate.

        Returns:
            float: the value of `design_hot_water_flow_rate` or None if not set

        """
        return self["Design Hot Water Flow Rate"]

    @design_hot_water_flow_rate.setter
    def design_hot_water_flow_rate(self, value="autosize"):
        """Corresponds to IDD field `Design Hot Water Flow Rate` The water flow
        rate at design conditions through the heater side.

        Args:
            value (float or "Autosize"): value for IDD Field `Design Hot Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                Default value: "autosize"
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Hot Water Flow Rate"] = value

    @property
    def cooling_capacity_function_of_temperature_curve_name(self):
        """Get cooling_capacity_function_of_temperature_curve_name.

        Returns:
            str: the value of `cooling_capacity_function_of_temperature_curve_name` or None if not set

        """
        return self["Cooling Capacity Function of Temperature Curve Name"]

    @cooling_capacity_function_of_temperature_curve_name.setter
    def cooling_capacity_function_of_temperature_curve_name(self, value=None):
        """  Corresponds to IDD field `Cooling Capacity Function of Temperature Curve Name`
        The CoolCapFT curve represents the fraction of the cooling capacity of the chiller as it
        varies by temperature.  The curve is normalized so that at design conditions the
        value of the curve should be 1.0.  This is a biquadratic curve with the
        input variables being the leaving chilled water temperature and either
        the entering or leaving condenser water temperature.
        Table:OneIndependentVariable object can also be used

        Args:
            value (str): value for IDD Field `Cooling Capacity Function of Temperature Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Cooling Capacity Function of Temperature Curve Name"] = value

    @property
    def fuel_input_to_cooling_output_ratio_function_of_temperature_curve_name(
            self):
        """Get fuel_input_to_cooling_output_ratio_function_of_temperature_curve
        _name.

        Returns:
            str: the value of `fuel_input_to_cooling_output_ratio_function_of_temperature_curve_name` or None if not set

        """
        return self[
            "Fuel Input to Cooling Output Ratio Function of Temperature Curve Name"]

    @fuel_input_to_cooling_output_ratio_function_of_temperature_curve_name.setter
    def fuel_input_to_cooling_output_ratio_function_of_temperature_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Fuel Input to Cooling Output Ratio Function of Temperature Curve Name`
        The curve represents the fraction of the fuel input to the chiller at full load as
        it varies by temperature.  The curve is normalized so that at design conditions the
        value of the curve should be 1.0.  This is a biquadratic curve with the
        input variables being the leaving chilled water temperature and either
        the entering or leaving condenser water temperature.
        Table:OneIndependentVariable object can also be used

        Args:
            value (str): value for IDD Field `Fuel Input to Cooling Output Ratio Function of Temperature Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Fuel Input to Cooling Output Ratio Function of Temperature Curve Name"] = value

    @property
    def fuel_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name(
            self):
        """Get fuel_input_to_cooling_output_ratio_function_of_part_load_ratio_c
        urve_name.

        Returns:
            str: the value of `fuel_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name` or None if not set

        """
        return self[
            "Fuel Input to Cooling Output Ratio Function of Part Load Ratio Curve Name"]

    @fuel_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name.setter
    def fuel_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Fuel Input to Cooling Output Ratio Function of Part Load Ratio Curve Name`
        The curve represents the fraction of the fuel input to the chiller as the load on
        the chiller varies but the operating temperatures remain at the design values.
        The curve is normalized so that at full load the value of the curve should be 1.0.
        The curve is usually linear or quadratic.
        Table:OneIndependentVariable object can also be used

        Args:
            value (str): value for IDD Field `Fuel Input to Cooling Output Ratio Function of Part Load Ratio Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Fuel Input to Cooling Output Ratio Function of Part Load Ratio Curve Name"] = value

    @property
    def electric_input_to_cooling_output_ratio_function_of_temperature_curve_name(
            self):
        """Get electric_input_to_cooling_output_ratio_function_of_temperature_c
        urve_name.

        Returns:
            str: the value of `electric_input_to_cooling_output_ratio_function_of_temperature_curve_name` or None if not set

        """
        return self[
            "Electric Input to Cooling Output Ratio Function of Temperature Curve Name"]

    @electric_input_to_cooling_output_ratio_function_of_temperature_curve_name.setter
    def electric_input_to_cooling_output_ratio_function_of_temperature_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Electric Input to Cooling Output Ratio Function of Temperature Curve Name`
        The curve represents the fraction of the electricity to the chiller at full load as
        it varies by temperature.  The curve is normalized so that at design conditions the
        value of the curve should be 1.0.  This is a biquadratic curve with the
        input variables being the leaving chilled water temperature and either
        the entering or leaving condenser water temperature.
        Table:OneIndependentVariable object can also be used

        Args:
            value (str): value for IDD Field `Electric Input to Cooling Output Ratio Function of Temperature Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Electric Input to Cooling Output Ratio Function of Temperature Curve Name"] = value

    @property
    def electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name(
            self):
        """Get electric_input_to_cooling_output_ratio_function_of_part_load_rat
        io_curve_name.

        Returns:
            str: the value of `electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name` or None if not set

        """
        return self[
            "Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name"]

    @electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name.setter
    def electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name`
        The curve represents the fraction of the electricity to the chiller as the load on
        the chiller varies but the operating temperatures remain at the design values.
        The curve is normalized so that at full load the value of the curve should be 1.0.
        The curve is usually linear or quadratic.
        Table:OneIndependentVariable object can also be used

        Args:
            value (str): value for IDD Field `Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name"] = value

    @property
    def heating_capacity_function_of_cooling_capacity_curve_name(self):
        """Get heating_capacity_function_of_cooling_capacity_curve_name.

        Returns:
            str: the value of `heating_capacity_function_of_cooling_capacity_curve_name` or None if not set

        """
        return self["Heating Capacity Function of Cooling Capacity Curve Name"]

    @heating_capacity_function_of_cooling_capacity_curve_name.setter
    def heating_capacity_function_of_cooling_capacity_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Heating Capacity Function of Cooling Capacity Curve Name`
        The curve represents how the heating capacity of the chiller varies with cooling
        capacity when the chiller is simultaeous heating and cooling.  The curve is normalized
        so an input of 1.0 represents the nominal cooling capacity and an output of 1.0
        represents the full heating capacity (see the Heating to cooling capacity ratio input)
        The curve is usually linear or quadratic.
        Table:OneIndependentVariable object can also be used

        Args:
            value (str): value for IDD Field `Heating Capacity Function of Cooling Capacity Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Heating Capacity Function of Cooling Capacity Curve Name"] = value

    @property
    def fuel_input_to_heat_output_ratio_during_heating_only_operation_curve_name(
            self):
        """Get fuel_input_to_heat_output_ratio_during_heating_only_operation_cu
        rve_name.

        Returns:
            str: the value of `fuel_input_to_heat_output_ratio_during_heating_only_operation_curve_name` or None if not set

        """
        return self[
            "Fuel Input to Heat Output Ratio During Heating Only Operation Curve Name"]

    @fuel_input_to_heat_output_ratio_during_heating_only_operation_curve_name.setter
    def fuel_input_to_heat_output_ratio_during_heating_only_operation_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Fuel Input to Heat Output Ratio During Heating Only Operation Curve Name`
        When the chiller is operating as only a heater, this curve is used to represent the
        fraction of fuel used as the heating load varies.  It is normalized so that a value
        of 1.0 is the full heating capacity.  The curve is usually linear or quadratic and
        will probably be similar to a boiler curve for most chillers.
        Table:OneIndependentVariable object can also be used

        Args:
            value (str): value for IDD Field `Fuel Input to Heat Output Ratio During Heating Only Operation Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Fuel Input to Heat Output Ratio During Heating Only Operation Curve Name"] = value

    @property
    def temperature_curve_input_variable(self):
        """Get temperature_curve_input_variable.

        Returns:
            str: the value of `temperature_curve_input_variable` or None if not set

        """
        return self["Temperature Curve Input Variable"]

    @temperature_curve_input_variable.setter
    def temperature_curve_input_variable(self, value="EnteringCondenser"):
        """Corresponds to IDD field `Temperature Curve Input Variable` Sets the
        second independent variable in the three temperature dependent
        performance curves to either the leaving or entering condenser water
        temperature.  Manufacturers express the performance of their chillers
        using either the leaving condenser water temperature (to the tower) or
        the entering condenser water temperature (from the tower).

        Args:
            value (str): value for IDD Field `Temperature Curve Input Variable`
                Default value: EnteringCondenser
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Temperature Curve Input Variable"] = value

    @property
    def condenser_type(self):
        """Get condenser_type.

        Returns:
            str: the value of `condenser_type` or None if not set

        """
        return self["Condenser Type"]

    @condenser_type.setter
    def condenser_type(self, value="WaterCooled"):
        """Corresponds to IDD field `Condenser Type` The condenser can either
        be air cooled or connected to a cooling tower.

        Args:
            value (str): value for IDD Field `Condenser Type`
                Default value: WaterCooled
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Type"] = value

    @property
    def chilled_water_temperature_lower_limit(self):
        """Get chilled_water_temperature_lower_limit.

        Returns:
            float: the value of `chilled_water_temperature_lower_limit` or None if not set

        """
        return self["Chilled Water Temperature Lower Limit"]

    @chilled_water_temperature_lower_limit.setter
    def chilled_water_temperature_lower_limit(self, value=2.0):
        """Corresponds to IDD field `Chilled Water Temperature Lower Limit` The
        chilled water supply temperature below which the chiller will shut off.

        Args:
            value (float): value for IDD Field `Chilled Water Temperature Lower Limit`
                Units: C
                Default value: 2.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Temperature Lower Limit"] = value

    @property
    def fuel_higher_heating_value(self):
        """Get fuel_higher_heating_value.

        Returns:
            float: the value of `fuel_higher_heating_value` or None if not set

        """
        return self["Fuel Higher Heating Value"]

    @fuel_higher_heating_value.setter
    def fuel_higher_heating_value(self, value=None):
        """Corresponds to IDD field `Fuel Higher Heating Value` Not currently
        used.

        Args:
            value (float): value for IDD Field `Fuel Higher Heating Value`
                Units: kJ/kg
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Fuel Higher Heating Value"] = value

    @property
    def chiller_flow_mode(self):
        """Get chiller_flow_mode.

        Returns:
            str: the value of `chiller_flow_mode` or None if not set

        """
        return self["Chiller Flow Mode"]

    @chiller_flow_mode.setter
    def chiller_flow_mode(self, value=None):
        """Corresponds to IDD field `Chiller Flow Mode` This field is not
        really used and will be deleted from the object. The required
        information is gotten internally or not needed by the program.

        Args:
            value (str): value for IDD Field `Chiller Flow Mode`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Flow Mode"] = value

    @property
    def fuel_type(self):
        """Get fuel_type.

        Returns:
            str: the value of `fuel_type` or None if not set

        """
        return self["Fuel Type"]

    @fuel_type.setter
    def fuel_type(self, value="NaturalGas"):
        """Corresponds to IDD field `Fuel Type`

        Args:
            value (str): value for IDD Field `Fuel Type`
                Default value: NaturalGas
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Fuel Type"] = value

    @property
    def sizing_factor(self):
        """Get sizing_factor.

        Returns:
            float: the value of `sizing_factor` or None if not set

        """
        return self["Sizing Factor"]

    @sizing_factor.setter
    def sizing_factor(self, value=1.0):
        """Corresponds to IDD field `Sizing Factor` Multiplies the autosized
        capacity and flow rates.

        Args:
            value (float): value for IDD Field `Sizing Factor`
                Default value: 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Sizing Factor"] = value




class ChillerHeaterAbsorptionDoubleEffect(DataObject):

    """ Corresponds to IDD object `ChillerHeater:Absorption:DoubleEffect`
        Exhaust fired absorption chiller-heater using performance curves similar to DOE-2
    """
    schema = {'min-fields': 34,
              'name': u'ChillerHeater:Absorption:DoubleEffect',
              'pyname': u'ChillerHeaterAbsorptionDoubleEffect',
              'format': None,
              'fields': OrderedDict([(u'name',
                                      {'name': u'Name',
                                       'pyname': u'name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'alpha'}),
                                     (u'nominal cooling capacity',
                                      {'name': u'Nominal Cooling Capacity',
                                       'pyname': u'nominal_cooling_capacity',
                                       'default': 'autosize',
                                       'required-field': True,
                                       'autosizable': True,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'W'}),
                                     (u'heating to cooling capacity ratio',
                                      {'name': u'Heating to Cooling Capacity Ratio',
                                       'pyname': u'heating_to_cooling_capacity_ratio',
                                       'default': 0.8,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'thermal energy input to cooling output ratio',
                                      {'name': u'Thermal Energy Input to Cooling Output Ratio',
                                       'pyname': u'thermal_energy_input_to_cooling_output_ratio',
                                       'default': 0.97,
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'thermal energy input to heating output ratio',
                                      {'name': u'Thermal Energy Input to Heating Output Ratio',
                                       'pyname': u'thermal_energy_input_to_heating_output_ratio',
                                       'default': 1.25,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'electric input to cooling output ratio',
                                      {'name': u'Electric Input to Cooling Output Ratio',
                                       'pyname': u'electric_input_to_cooling_output_ratio',
                                       'default': 0.01,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'electric input to heating output ratio',
                                      {'name': u'Electric Input to Heating Output Ratio',
                                       'pyname': u'electric_input_to_heating_output_ratio',
                                       'default': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'chilled water inlet node name',
                                      {'name': u'Chilled Water Inlet Node Name',
                                       'pyname': u'chilled_water_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'chilled water outlet node name',
                                      {'name': u'Chilled Water Outlet Node Name',
                                       'pyname': u'chilled_water_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'condenser inlet node name',
                                      {'name': u'Condenser Inlet Node Name',
                                       'pyname': u'condenser_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'condenser outlet node name',
                                      {'name': u'Condenser Outlet Node Name',
                                       'pyname': u'condenser_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'hot water inlet node name',
                                      {'name': u'Hot Water Inlet Node Name',
                                       'pyname': u'hot_water_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'hot water outlet node name',
                                      {'name': u'Hot Water Outlet Node Name',
                                       'pyname': u'hot_water_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'minimum part load ratio',
                                      {'name': u'Minimum Part Load Ratio',
                                       'pyname': u'minimum_part_load_ratio',
                                       'default': 0.1,
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'maximum part load ratio',
                                      {'name': u'Maximum Part Load Ratio',
                                       'pyname': u'maximum_part_load_ratio',
                                       'default': 1.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.5,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'optimum part load ratio',
                                      {'name': u'Optimum Part Load Ratio',
                                       'pyname': u'optimum_part_load_ratio',
                                       'default': 1.0,
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'design entering condenser water temperature',
                                      {'name': u'Design Entering Condenser Water Temperature',
                                       'pyname': u'design_entering_condenser_water_temperature',
                                       'default': 29.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'C'}),
                                     (u'design leaving chilled water temperature',
                                      {'name': u'Design Leaving Chilled Water Temperature',
                                       'pyname': u'design_leaving_chilled_water_temperature',
                                       'default': 7.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'C'}),
                                     (u'design chilled water flow rate',
                                      {'name': u'Design Chilled Water Flow Rate',
                                       'pyname': u'design_chilled_water_flow_rate',
                                       'default': 'autosize',
                                       'required-field': False,
                                       'autosizable': True,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'm3/s'}),
                                     (u'design condenser water flow rate',
                                      {'name': u'Design Condenser Water Flow Rate',
                                       'pyname': u'design_condenser_water_flow_rate',
                                       'default': 'autosize',
                                       'required-field': True,
                                       'autosizable': True,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'm3/s'}),
                                     (u'design hot water flow rate',
                                      {'name': u'Design Hot Water Flow Rate',
                                       'pyname': u'design_hot_water_flow_rate',
                                       'default': 'autosize',
                                       'required-field': False,
                                       'autosizable': True,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'm3/s'}),
                                     (u'cooling capacity function of temperature curve name',
                                      {'name': u'Cooling Capacity Function of Temperature Curve Name',
                                       'pyname': u'cooling_capacity_function_of_temperature_curve_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'fuel input to cooling output ratio function of temperature curve name',
                                      {'name': u'Fuel Input to Cooling Output Ratio Function of Temperature Curve Name',
                                       'pyname': u'fuel_input_to_cooling_output_ratio_function_of_temperature_curve_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'fuel input to cooling output ratio function of part load ratio curve name',
                                      {'name': u'Fuel Input to Cooling Output Ratio Function of Part Load Ratio Curve Name',
                                       'pyname': u'fuel_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'electric input to cooling output ratio function of temperature curve name',
                                      {'name': u'Electric Input to Cooling Output Ratio Function of Temperature Curve Name',
                                       'pyname': u'electric_input_to_cooling_output_ratio_function_of_temperature_curve_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'electric input to cooling output ratio function of part load ratio curve name',
                                      {'name': u'Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name',
                                       'pyname': u'electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'heating capacity function of cooling capacity curve name',
                                      {'name': u'Heating Capacity Function of Cooling Capacity Curve Name',
                                       'pyname': u'heating_capacity_function_of_cooling_capacity_curve_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'fuel input to heat output ratio during heating only operation curve name',
                                      {'name': u'Fuel Input to Heat Output Ratio During Heating Only Operation Curve Name',
                                       'pyname': u'fuel_input_to_heat_output_ratio_during_heating_only_operation_curve_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'temperature curve input variable',
                                      {'name': u'Temperature Curve Input Variable',
                                       'pyname': u'temperature_curve_input_variable',
                                       'default': u'EnteringCondenser',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'LeavingCondenser',
                                                           u'EnteringCondenser'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'condenser type',
                                      {'name': u'Condenser Type',
                                       'pyname': u'condenser_type',
                                       'default': u'WaterCooled',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'AirCooled',
                                                           u'WaterCooled'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'chilled water temperature lower limit',
                                      {'name': u'Chilled Water Temperature Lower Limit',
                                       'pyname': u'chilled_water_temperature_lower_limit',
                                       'default': 2.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'C'}),
                                     (u'exhaust source object type',
                                      {'name': u'Exhaust Source Object Type',
                                       'pyname': u'exhaust_source_object_type',
                                       'required-field': True,
                                       'autosizable': False,
                                       'accepted-values': [u'Generator:MicroTurbine'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'exhaust source object name',
                                      {'name': u'Exhaust Source Object Name',
                                       'pyname': u'exhaust_source_object_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'sizing factor',
                                      {'name': u'Sizing Factor',
                                       'pyname': u'sizing_factor',
                                       'default': 1.0,
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'})]),
              'extensible-fields': OrderedDict(),
              'unique-object': False,
              'required-object': False,
              'group': u'Plant Heating and Cooling Equipment'}

    @property
    def name(self):
        """Get name.

        Returns:
            str: the value of `name` or None if not set

        """
        return self["Name"]

    @name.setter
    def name(self, value=None):
        """Corresponds to IDD field `Name`

        Args:
            value (str): value for IDD Field `Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Name"] = value

    @property
    def nominal_cooling_capacity(self):
        """Get nominal_cooling_capacity.

        Returns:
            float: the value of `nominal_cooling_capacity` or None if not set

        """
        return self["Nominal Cooling Capacity"]

    @nominal_cooling_capacity.setter
    def nominal_cooling_capacity(self, value="autosize"):
        """Corresponds to IDD field `Nominal Cooling Capacity`

        Args:
            value (float or "Autosize"): value for IDD Field `Nominal Cooling Capacity`
                Units: W
                Default value: "autosize"
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Nominal Cooling Capacity"] = value

    @property
    def heating_to_cooling_capacity_ratio(self):
        """Get heating_to_cooling_capacity_ratio.

        Returns:
            float: the value of `heating_to_cooling_capacity_ratio` or None if not set

        """
        return self["Heating to Cooling Capacity Ratio"]

    @heating_to_cooling_capacity_ratio.setter
    def heating_to_cooling_capacity_ratio(self, value=0.8):
        """Corresponds to IDD field `Heating to Cooling Capacity Ratio` A
        positive fraction that represents the ratio of the heating capacity
        divided by the cooling capacity at rated conditions.

        Args:
            value (float): value for IDD Field `Heating to Cooling Capacity Ratio`
                Default value: 0.8
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Heating to Cooling Capacity Ratio"] = value

    @property
    def thermal_energy_input_to_cooling_output_ratio(self):
        """Get thermal_energy_input_to_cooling_output_ratio.

        Returns:
            float: the value of `thermal_energy_input_to_cooling_output_ratio` or None if not set

        """
        return self["Thermal Energy Input to Cooling Output Ratio"]

    @thermal_energy_input_to_cooling_output_ratio.setter
    def thermal_energy_input_to_cooling_output_ratio(self, value=0.97):
        """Corresponds to IDD field `Thermal Energy Input to Cooling Output
        Ratio` The positive fraction that represents the ratio of the
        instantaneous fuel used divided by the cooling capacity at rated
        conditions.

        Args:
            value (float): value for IDD Field `Thermal Energy Input to Cooling Output Ratio`
                Default value: 0.97
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Thermal Energy Input to Cooling Output Ratio"] = value

    @property
    def thermal_energy_input_to_heating_output_ratio(self):
        """Get thermal_energy_input_to_heating_output_ratio.

        Returns:
            float: the value of `thermal_energy_input_to_heating_output_ratio` or None if not set

        """
        return self["Thermal Energy Input to Heating Output Ratio"]

    @thermal_energy_input_to_heating_output_ratio.setter
    def thermal_energy_input_to_heating_output_ratio(self, value=1.25):
        """Corresponds to IDD field `Thermal Energy Input to Heating Output
        Ratio` The positive fraction that represents the ratio of the
        instantaneous fuel used divided by the nominal heating capacity.

        Args:
            value (float): value for IDD Field `Thermal Energy Input to Heating Output Ratio`
                Default value: 1.25
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Thermal Energy Input to Heating Output Ratio"] = value

    @property
    def electric_input_to_cooling_output_ratio(self):
        """Get electric_input_to_cooling_output_ratio.

        Returns:
            float: the value of `electric_input_to_cooling_output_ratio` or None if not set

        """
        return self["Electric Input to Cooling Output Ratio"]

    @electric_input_to_cooling_output_ratio.setter
    def electric_input_to_cooling_output_ratio(self, value=0.01):
        """Corresponds to IDD field `Electric Input to Cooling Output Ratio`
        The positive fraction that represents the ratio of the instantaneous
        electricity used divided by the cooling capacity at rated conditions.
        If the chiller is both heating and cooling only the cooling electricity
        is used.

        Args:
            value (float): value for IDD Field `Electric Input to Cooling Output Ratio`
                Default value: 0.01
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Electric Input to Cooling Output Ratio"] = value

    @property
    def electric_input_to_heating_output_ratio(self):
        """Get electric_input_to_heating_output_ratio.

        Returns:
            float: the value of `electric_input_to_heating_output_ratio` or None if not set

        """
        return self["Electric Input to Heating Output Ratio"]

    @electric_input_to_heating_output_ratio.setter
    def electric_input_to_heating_output_ratio(self, value=None):
        """Corresponds to IDD field `Electric Input to Heating Output Ratio`
        The positive fraction that represents the ratio of the instantaneous
        electricity used divided by the nominal heating capacity. If the
        chiller is both heating and cooling only the cooling electricity is
        used.

        Args:
            value (float): value for IDD Field `Electric Input to Heating Output Ratio`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Electric Input to Heating Output Ratio"] = value

    @property
    def chilled_water_inlet_node_name(self):
        """Get chilled_water_inlet_node_name.

        Returns:
            str: the value of `chilled_water_inlet_node_name` or None if not set

        """
        return self["Chilled Water Inlet Node Name"]

    @chilled_water_inlet_node_name.setter
    def chilled_water_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Chilled Water Inlet Node Name`

        Args:
            value (str): value for IDD Field `Chilled Water Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Inlet Node Name"] = value

    @property
    def chilled_water_outlet_node_name(self):
        """Get chilled_water_outlet_node_name.

        Returns:
            str: the value of `chilled_water_outlet_node_name` or None if not set

        """
        return self["Chilled Water Outlet Node Name"]

    @chilled_water_outlet_node_name.setter
    def chilled_water_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Chilled Water Outlet Node Name`

        Args:
            value (str): value for IDD Field `Chilled Water Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Outlet Node Name"] = value

    @property
    def condenser_inlet_node_name(self):
        """Get condenser_inlet_node_name.

        Returns:
            str: the value of `condenser_inlet_node_name` or None if not set

        """
        return self["Condenser Inlet Node Name"]

    @condenser_inlet_node_name.setter
    def condenser_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Condenser Inlet Node Name`

        Args:
            value (str): value for IDD Field `Condenser Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Inlet Node Name"] = value

    @property
    def condenser_outlet_node_name(self):
        """Get condenser_outlet_node_name.

        Returns:
            str: the value of `condenser_outlet_node_name` or None if not set

        """
        return self["Condenser Outlet Node Name"]

    @condenser_outlet_node_name.setter
    def condenser_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Condenser Outlet Node Name`

        Args:
            value (str): value for IDD Field `Condenser Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Outlet Node Name"] = value

    @property
    def hot_water_inlet_node_name(self):
        """Get hot_water_inlet_node_name.

        Returns:
            str: the value of `hot_water_inlet_node_name` or None if not set

        """
        return self["Hot Water Inlet Node Name"]

    @hot_water_inlet_node_name.setter
    def hot_water_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Hot Water Inlet Node Name`

        Args:
            value (str): value for IDD Field `Hot Water Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Hot Water Inlet Node Name"] = value

    @property
    def hot_water_outlet_node_name(self):
        """Get hot_water_outlet_node_name.

        Returns:
            str: the value of `hot_water_outlet_node_name` or None if not set

        """
        return self["Hot Water Outlet Node Name"]

    @hot_water_outlet_node_name.setter
    def hot_water_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Hot Water Outlet Node Name`

        Args:
            value (str): value for IDD Field `Hot Water Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Hot Water Outlet Node Name"] = value

    @property
    def minimum_part_load_ratio(self):
        """Get minimum_part_load_ratio.

        Returns:
            float: the value of `minimum_part_load_ratio` or None if not set

        """
        return self["Minimum Part Load Ratio"]

    @minimum_part_load_ratio.setter
    def minimum_part_load_ratio(self, value=0.1):
        """Corresponds to IDD field `Minimum Part Load Ratio` The positive
        fraction that represents the minimum cooling output possible when
        operated continually at rated temperature conditions divided by the
        nominal cooling capacity at those same conditions.  If the load on the
        chiller is below this fraction the chiller will cycle.

        Args:
            value (float): value for IDD Field `Minimum Part Load Ratio`
                Default value: 0.1
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Minimum Part Load Ratio"] = value

    @property
    def maximum_part_load_ratio(self):
        """Get maximum_part_load_ratio.

        Returns:
            float: the value of `maximum_part_load_ratio` or None if not set

        """
        return self["Maximum Part Load Ratio"]

    @maximum_part_load_ratio.setter
    def maximum_part_load_ratio(self, value=1.0):
        """  Corresponds to IDD field `Maximum Part Load Ratio`
        The positive fraction that represents the maximum cooling output possible at
        rated temperature conditions divided by the nominal cooling capacity at those
        same conditions.  If greater than 1.0, the chiller is typically thought of as
        capable of being overloaded.

        Args:
            value (float): value for IDD Field `Maximum Part Load Ratio`
                Default value: 1.0
                value >= 0.5
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Maximum Part Load Ratio"] = value

    @property
    def optimum_part_load_ratio(self):
        """Get optimum_part_load_ratio.

        Returns:
            float: the value of `optimum_part_load_ratio` or None if not set

        """
        return self["Optimum Part Load Ratio"]

    @optimum_part_load_ratio.setter
    def optimum_part_load_ratio(self, value=1.0):
        """Corresponds to IDD field `Optimum Part Load Ratio` The positive
        fraction that represents the optimal cooling output at rated
        temperature conditions divided by the nominal cooling capacity at those
        same conditions.  It represents the most desirable operating point for
        the chiller.

        Args:
            value (float): value for IDD Field `Optimum Part Load Ratio`
                Default value: 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Optimum Part Load Ratio"] = value

    @property
    def design_entering_condenser_water_temperature(self):
        """Get design_entering_condenser_water_temperature.

        Returns:
            float: the value of `design_entering_condenser_water_temperature` or None if not set

        """
        return self["Design Entering Condenser Water Temperature"]

    @design_entering_condenser_water_temperature.setter
    def design_entering_condenser_water_temperature(self, value=29.0):
        """Corresponds to IDD field `Design Entering Condenser Water
        Temperature` The temperature of the water entering the condenser of the
        chiller when operating at design conditions.  This is usually based on
        the temperature delivered by the cooling tower in a water cooled
        application.

        Args:
            value (float): value for IDD Field `Design Entering Condenser Water Temperature`
                Units: C
                Default value: 29.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Entering Condenser Water Temperature"] = value

    @property
    def design_leaving_chilled_water_temperature(self):
        """Get design_leaving_chilled_water_temperature.

        Returns:
            float: the value of `design_leaving_chilled_water_temperature` or None if not set

        """
        return self["Design Leaving Chilled Water Temperature"]

    @design_leaving_chilled_water_temperature.setter
    def design_leaving_chilled_water_temperature(self, value=7.0):
        """Corresponds to IDD field `Design Leaving Chilled Water Temperature`
        The temperature of the water leaving the evaporator of the chiller when
        operating at design conditions also called the chilled water supply
        temperature or leaving chilled water temperature.

        Args:
            value (float): value for IDD Field `Design Leaving Chilled Water Temperature`
                Units: C
                Default value: 7.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Leaving Chilled Water Temperature"] = value

    @property
    def design_chilled_water_flow_rate(self):
        """Get design_chilled_water_flow_rate.

        Returns:
            float: the value of `design_chilled_water_flow_rate` or None if not set

        """
        return self["Design Chilled Water Flow Rate"]

    @design_chilled_water_flow_rate.setter
    def design_chilled_water_flow_rate(self, value="autosize"):
        """Corresponds to IDD field `Design Chilled Water Flow Rate` For
        variable volume this is the max flow & for constant flow this is the
        flow.

        Args:
            value (float or "Autosize"): value for IDD Field `Design Chilled Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                Default value: "autosize"
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Chilled Water Flow Rate"] = value

    @property
    def design_condenser_water_flow_rate(self):
        """Get design_condenser_water_flow_rate.

        Returns:
            float: the value of `design_condenser_water_flow_rate` or None if not set

        """
        return self["Design Condenser Water Flow Rate"]

    @design_condenser_water_flow_rate.setter
    def design_condenser_water_flow_rate(self, value="autosize"):
        """  Corresponds to IDD field `Design Condenser Water Flow Rate`
        The water flow rate at design conditions through the condenser.
        This field is not used for Condenser Type = AirCooled

        Args:
            value (float or "Autosize"): value for IDD Field `Design Condenser Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                Default value: "autosize"
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Design Condenser Water Flow Rate"] = value

    @property
    def design_hot_water_flow_rate(self):
        """Get design_hot_water_flow_rate.

        Returns:
            float: the value of `design_hot_water_flow_rate` or None if not set

        """
        return self["Design Hot Water Flow Rate"]

    @design_hot_water_flow_rate.setter
    def design_hot_water_flow_rate(self, value="autosize"):
        """Corresponds to IDD field `Design Hot Water Flow Rate` The water flow
        rate at design conditions through the heater side.

        Args:
            value (float or "Autosize"): value for IDD Field `Design Hot Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                Default value: "autosize"
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Hot Water Flow Rate"] = value

    @property
    def cooling_capacity_function_of_temperature_curve_name(self):
        """Get cooling_capacity_function_of_temperature_curve_name.

        Returns:
            str: the value of `cooling_capacity_function_of_temperature_curve_name` or None if not set

        """
        return self["Cooling Capacity Function of Temperature Curve Name"]

    @cooling_capacity_function_of_temperature_curve_name.setter
    def cooling_capacity_function_of_temperature_curve_name(self, value=None):
        """  Corresponds to IDD field `Cooling Capacity Function of Temperature Curve Name`
        The CoolCapFT curve represents the fraction of the cooling capacity of the chiller as it
        varies by temperature.  The curve is normalized so that at design conditions the
        value of the curve should be 1.0.  This is a biquadratic curve with the
        input variables being the leaving chilled water temperature and either
        the entering or leaving condenser water temperature.
        Table:OneIndependentVariable object can also be used

        Args:
            value (str): value for IDD Field `Cooling Capacity Function of Temperature Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Cooling Capacity Function of Temperature Curve Name"] = value

    @property
    def fuel_input_to_cooling_output_ratio_function_of_temperature_curve_name(
            self):
        """Get fuel_input_to_cooling_output_ratio_function_of_temperature_curve
        _name.

        Returns:
            str: the value of `fuel_input_to_cooling_output_ratio_function_of_temperature_curve_name` or None if not set

        """
        return self[
            "Fuel Input to Cooling Output Ratio Function of Temperature Curve Name"]

    @fuel_input_to_cooling_output_ratio_function_of_temperature_curve_name.setter
    def fuel_input_to_cooling_output_ratio_function_of_temperature_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Fuel Input to Cooling Output Ratio Function of Temperature Curve Name`
        The curve represents the fraction of the fuel input to the chiller at full load as
        it varies by temperature.  The curve is normalized so that at design conditions the
        value of the curve should be 1.0.  This is a biquadratic curve with the
        input variables being the leaving chilled water temperature and either
        the entering or leaving condenser water temperature.
        Table:OneIndependentVariable object can also be used

        Args:
            value (str): value for IDD Field `Fuel Input to Cooling Output Ratio Function of Temperature Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Fuel Input to Cooling Output Ratio Function of Temperature Curve Name"] = value

    @property
    def fuel_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name(
            self):
        """Get fuel_input_to_cooling_output_ratio_function_of_part_load_ratio_c
        urve_name.

        Returns:
            str: the value of `fuel_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name` or None if not set

        """
        return self[
            "Fuel Input to Cooling Output Ratio Function of Part Load Ratio Curve Name"]

    @fuel_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name.setter
    def fuel_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Fuel Input to Cooling Output Ratio Function of Part Load Ratio Curve Name`
        The curve represents the fraction of the fuel input to the chiller as the load on
        the chiller varies but the operating temperatures remain at the design values.
        The curve is normalized so that at full load the value of the curve should be 1.0.
        The curve is usually linear or quadratic.
        Table:OneIndependentVariable object can also be used

        Args:
            value (str): value for IDD Field `Fuel Input to Cooling Output Ratio Function of Part Load Ratio Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Fuel Input to Cooling Output Ratio Function of Part Load Ratio Curve Name"] = value

    @property
    def electric_input_to_cooling_output_ratio_function_of_temperature_curve_name(
            self):
        """Get electric_input_to_cooling_output_ratio_function_of_temperature_c
        urve_name.

        Returns:
            str: the value of `electric_input_to_cooling_output_ratio_function_of_temperature_curve_name` or None if not set

        """
        return self[
            "Electric Input to Cooling Output Ratio Function of Temperature Curve Name"]

    @electric_input_to_cooling_output_ratio_function_of_temperature_curve_name.setter
    def electric_input_to_cooling_output_ratio_function_of_temperature_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Electric Input to Cooling Output Ratio Function of Temperature Curve Name`
        The curve represents the fraction of the electricity to the chiller at full load as
        it varies by temperature.  The curve is normalized so that at design conditions the
        value of the curve should be 1.0.  This is a biquadratic curve with the
        input variables being the leaving chilled water temperature and either
        the entering or leaving condenser water temperature.
        Table:OneIndependentVariable object can also be used

        Args:
            value (str): value for IDD Field `Electric Input to Cooling Output Ratio Function of Temperature Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Electric Input to Cooling Output Ratio Function of Temperature Curve Name"] = value

    @property
    def electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name(
            self):
        """Get electric_input_to_cooling_output_ratio_function_of_part_load_rat
        io_curve_name.

        Returns:
            str: the value of `electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name` or None if not set

        """
        return self[
            "Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name"]

    @electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name.setter
    def electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name`
        The curve represents the fraction of the electricity to the chiller as the load on
        the chiller varies but the operating temperatures remain at the design values.
        The curve is normalized so that at full load the value of the curve should be 1.0.
        The curve is usually linear or quadratic.
        Table:OneIndependentVariable object can also be used

        Args:
            value (str): value for IDD Field `Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name"] = value

    @property
    def heating_capacity_function_of_cooling_capacity_curve_name(self):
        """Get heating_capacity_function_of_cooling_capacity_curve_name.

        Returns:
            str: the value of `heating_capacity_function_of_cooling_capacity_curve_name` or None if not set

        """
        return self["Heating Capacity Function of Cooling Capacity Curve Name"]

    @heating_capacity_function_of_cooling_capacity_curve_name.setter
    def heating_capacity_function_of_cooling_capacity_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Heating Capacity Function of Cooling Capacity Curve Name`
        The curve represents how the heating capacity of the chiller varies with cooling
        capacity when the chiller is simultaeous heating and cooling.  The curve is normalized
        so an input of 1.0 represents the nominal cooling capacity and an output of 1.0
        represents the full heating capacity (see the Heating to cooling capacity ratio input)
        The curve is usually linear or quadratic.
        Table:OneIndependentVariable object can also be used

        Args:
            value (str): value for IDD Field `Heating Capacity Function of Cooling Capacity Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Heating Capacity Function of Cooling Capacity Curve Name"] = value

    @property
    def fuel_input_to_heat_output_ratio_during_heating_only_operation_curve_name(
            self):
        """Get fuel_input_to_heat_output_ratio_during_heating_only_operation_cu
        rve_name.

        Returns:
            str: the value of `fuel_input_to_heat_output_ratio_during_heating_only_operation_curve_name` or None if not set

        """
        return self[
            "Fuel Input to Heat Output Ratio During Heating Only Operation Curve Name"]

    @fuel_input_to_heat_output_ratio_during_heating_only_operation_curve_name.setter
    def fuel_input_to_heat_output_ratio_during_heating_only_operation_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Fuel Input to Heat Output Ratio During Heating Only Operation Curve Name`
        When the chiller is operating as only a heater, this curve is used to represent the
        fraction of fuel used as the heating load varies.  It is normalized so that a value
        of 1.0 is the full heating capacity.  The curve is usually linear or quadratic and
        will probably be similar to a boiler curve for most chillers.
        Table:OneIndependentVariable object can also be used

        Args:
            value (str): value for IDD Field `Fuel Input to Heat Output Ratio During Heating Only Operation Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Fuel Input to Heat Output Ratio During Heating Only Operation Curve Name"] = value

    @property
    def temperature_curve_input_variable(self):
        """Get temperature_curve_input_variable.

        Returns:
            str: the value of `temperature_curve_input_variable` or None if not set

        """
        return self["Temperature Curve Input Variable"]

    @temperature_curve_input_variable.setter
    def temperature_curve_input_variable(self, value="EnteringCondenser"):
        """Corresponds to IDD field `Temperature Curve Input Variable` Sets the
        second independent variable in the three temperature dependent
        performance curves to either the leaving or entering condenser water
        temperature.  Manufacturers express the performance of their chillers
        using either the leaving condenser water temperature (to the tower) or
        the entering condenser water temperature (from the tower).

        Args:
            value (str): value for IDD Field `Temperature Curve Input Variable`
                Default value: EnteringCondenser
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Temperature Curve Input Variable"] = value

    @property
    def condenser_type(self):
        """Get condenser_type.

        Returns:
            str: the value of `condenser_type` or None if not set

        """
        return self["Condenser Type"]

    @condenser_type.setter
    def condenser_type(self, value="WaterCooled"):
        """Corresponds to IDD field `Condenser Type` The condenser can either
        be air cooled or connected to a cooling tower.

        Args:
            value (str): value for IDD Field `Condenser Type`
                Default value: WaterCooled
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Type"] = value

    @property
    def chilled_water_temperature_lower_limit(self):
        """Get chilled_water_temperature_lower_limit.

        Returns:
            float: the value of `chilled_water_temperature_lower_limit` or None if not set

        """
        return self["Chilled Water Temperature Lower Limit"]

    @chilled_water_temperature_lower_limit.setter
    def chilled_water_temperature_lower_limit(self, value=2.0):
        """Corresponds to IDD field `Chilled Water Temperature Lower Limit` The
        chilled water supply temperature below which the chiller will shut off.

        Args:
            value (float): value for IDD Field `Chilled Water Temperature Lower Limit`
                Units: C
                Default value: 2.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Temperature Lower Limit"] = value

    @property
    def exhaust_source_object_type(self):
        """Get exhaust_source_object_type.

        Returns:
            str: the value of `exhaust_source_object_type` or None if not set

        """
        return self["Exhaust Source Object Type"]

    @exhaust_source_object_type.setter
    def exhaust_source_object_type(self, value=None):
        """Corresponds to IDD field `Exhaust Source Object Type`

        Args:
            value (str): value for IDD Field `Exhaust Source Object Type`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Exhaust Source Object Type"] = value

    @property
    def exhaust_source_object_name(self):
        """Get exhaust_source_object_name.

        Returns:
            str: the value of `exhaust_source_object_name` or None if not set

        """
        return self["Exhaust Source Object Name"]

    @exhaust_source_object_name.setter
    def exhaust_source_object_name(self, value=None):
        """Corresponds to IDD field `Exhaust Source Object Name`

        Args:
            value (str): value for IDD Field `Exhaust Source Object Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Exhaust Source Object Name"] = value

    @property
    def sizing_factor(self):
        """Get sizing_factor.

        Returns:
            float: the value of `sizing_factor` or None if not set

        """
        return self["Sizing Factor"]

    @sizing_factor.setter
    def sizing_factor(self, value=1.0):
        """Corresponds to IDD field `Sizing Factor` Multiplies the autosized
        capacity and flow rates.

        Args:
            value (float): value for IDD Field `Sizing Factor`
                Default value: 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Sizing Factor"] = value




class HeatPumpWaterToWaterEquationFitHeating(DataObject):

    """ Corresponds to IDD object `HeatPump:WaterToWater:EquationFit:Heating`
        simple water-water hp curve-fit model
    """
    schema = {'min-fields': 19,
              'name': u'HeatPump:WaterToWater:EquationFit:Heating',
              'pyname': u'HeatPumpWaterToWaterEquationFitHeating',
              'format': None,
              'fields': OrderedDict([(u'name',
                                      {'name': u'Name',
                                       'pyname': u'name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'source side inlet node name',
                                      {'name': u'Source Side Inlet Node Name',
                                       'pyname': u'source_side_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'source side outlet node name',
                                      {'name': u'Source Side Outlet Node Name',
                                       'pyname': u'source_side_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'load side inlet node name',
                                      {'name': u'Load Side Inlet Node Name',
                                       'pyname': u'load_side_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'load side outlet node name',
                                      {'name': u'Load Side Outlet Node Name',
                                       'pyname': u'load_side_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'rated load side flow rate',
                                      {'name': u'Rated Load Side Flow Rate',
                                       'pyname': u'rated_load_side_flow_rate',
                                       'minimum>': 0.0,
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'm3/s'}),
                                     (u'rated source side flow rate',
                                      {'name': u'Rated Source Side Flow Rate',
                                       'pyname': u'rated_source_side_flow_rate',
                                       'minimum>': 0.0,
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'm3/s'}),
                                     (u'rated heating capacity',
                                      {'name': u'Rated Heating Capacity',
                                       'pyname': u'rated_heating_capacity',
                                       'minimum>': 0.0,
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'W'}),
                                     (u'rated heating power consumption',
                                      {'name': u'Rated Heating Power Consumption',
                                       'pyname': u'rated_heating_power_consumption',
                                       'minimum>': 0.0,
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'W'}),
                                     (u'heating capacity coefficient 1',
                                      {'name': u'Heating Capacity Coefficient 1',
                                       'pyname': u'heating_capacity_coefficient_1',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'heating capacity coefficient 2',
                                      {'name': u'Heating Capacity Coefficient 2',
                                       'pyname': u'heating_capacity_coefficient_2',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'heating capacity coefficient 3',
                                      {'name': u'Heating Capacity Coefficient 3',
                                       'pyname': u'heating_capacity_coefficient_3',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'heating capacity coefficient 4',
                                      {'name': u'Heating Capacity Coefficient 4',
                                       'pyname': u'heating_capacity_coefficient_4',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'heating capacity coefficient 5',
                                      {'name': u'Heating Capacity Coefficient 5',
                                       'pyname': u'heating_capacity_coefficient_5',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'heating compressor power coefficient 1',
                                      {'name': u'Heating Compressor Power Coefficient 1',
                                       'pyname': u'heating_compressor_power_coefficient_1',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'heating compressor power coefficient 2',
                                      {'name': u'Heating Compressor Power Coefficient 2',
                                       'pyname': u'heating_compressor_power_coefficient_2',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'heating compressor power coefficient 3',
                                      {'name': u'Heating Compressor Power Coefficient 3',
                                       'pyname': u'heating_compressor_power_coefficient_3',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'heating compressor power coefficient 4',
                                      {'name': u'Heating Compressor Power Coefficient 4',
                                       'pyname': u'heating_compressor_power_coefficient_4',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'heating compressor power coefficient 5',
                                      {'name': u'Heating Compressor Power Coefficient 5',
                                       'pyname': u'heating_compressor_power_coefficient_5',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'})]),
              'extensible-fields': OrderedDict(),
              'unique-object': False,
              'required-object': False,
              'group': u'Plant Heating and Cooling Equipment'}

    @property
    def name(self):
        """Get name.

        Returns:
            str: the value of `name` or None if not set

        """
        return self["Name"]

    @name.setter
    def name(self, value=None):
        """Corresponds to IDD field `Name`

        Args:
            value (str): value for IDD Field `Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Name"] = value

    @property
    def source_side_inlet_node_name(self):
        """Get source_side_inlet_node_name.

        Returns:
            str: the value of `source_side_inlet_node_name` or None if not set

        """
        return self["Source Side Inlet Node Name"]

    @source_side_inlet_node_name.setter
    def source_side_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Source Side Inlet Node Name`

        Args:
            value (str): value for IDD Field `Source Side Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Source Side Inlet Node Name"] = value

    @property
    def source_side_outlet_node_name(self):
        """Get source_side_outlet_node_name.

        Returns:
            str: the value of `source_side_outlet_node_name` or None if not set

        """
        return self["Source Side Outlet Node Name"]

    @source_side_outlet_node_name.setter
    def source_side_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Source Side Outlet Node Name`

        Args:
            value (str): value for IDD Field `Source Side Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Source Side Outlet Node Name"] = value

    @property
    def load_side_inlet_node_name(self):
        """Get load_side_inlet_node_name.

        Returns:
            str: the value of `load_side_inlet_node_name` or None if not set

        """
        return self["Load Side Inlet Node Name"]

    @load_side_inlet_node_name.setter
    def load_side_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Load Side Inlet Node Name`

        Args:
            value (str): value for IDD Field `Load Side Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Load Side Inlet Node Name"] = value

    @property
    def load_side_outlet_node_name(self):
        """Get load_side_outlet_node_name.

        Returns:
            str: the value of `load_side_outlet_node_name` or None if not set

        """
        return self["Load Side Outlet Node Name"]

    @load_side_outlet_node_name.setter
    def load_side_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Load Side Outlet Node Name`

        Args:
            value (str): value for IDD Field `Load Side Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Load Side Outlet Node Name"] = value

    @property
    def rated_load_side_flow_rate(self):
        """Get rated_load_side_flow_rate.

        Returns:
            float: the value of `rated_load_side_flow_rate` or None if not set

        """
        return self["Rated Load Side Flow Rate"]

    @rated_load_side_flow_rate.setter
    def rated_load_side_flow_rate(self, value=None):
        """Corresponds to IDD field `Rated Load Side Flow Rate`

        Args:
            value (float): value for IDD Field `Rated Load Side Flow Rate`
                Units: m3/s
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Rated Load Side Flow Rate"] = value

    @property
    def rated_source_side_flow_rate(self):
        """Get rated_source_side_flow_rate.

        Returns:
            float: the value of `rated_source_side_flow_rate` or None if not set

        """
        return self["Rated Source Side Flow Rate"]

    @rated_source_side_flow_rate.setter
    def rated_source_side_flow_rate(self, value=None):
        """Corresponds to IDD field `Rated Source Side Flow Rate`

        Args:
            value (float): value for IDD Field `Rated Source Side Flow Rate`
                Units: m3/s
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Rated Source Side Flow Rate"] = value

    @property
    def rated_heating_capacity(self):
        """Get rated_heating_capacity.

        Returns:
            float: the value of `rated_heating_capacity` or None if not set

        """
        return self["Rated Heating Capacity"]

    @rated_heating_capacity.setter
    def rated_heating_capacity(self, value=None):
        """Corresponds to IDD field `Rated Heating Capacity`

        Args:
            value (float): value for IDD Field `Rated Heating Capacity`
                Units: W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Rated Heating Capacity"] = value

    @property
    def rated_heating_power_consumption(self):
        """Get rated_heating_power_consumption.

        Returns:
            float: the value of `rated_heating_power_consumption` or None if not set

        """
        return self["Rated Heating Power Consumption"]

    @rated_heating_power_consumption.setter
    def rated_heating_power_consumption(self, value=None):
        """Corresponds to IDD field `Rated Heating Power Consumption`

        Args:
            value (float): value for IDD Field `Rated Heating Power Consumption`
                Units: W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Rated Heating Power Consumption"] = value

    @property
    def heating_capacity_coefficient_1(self):
        """Get heating_capacity_coefficient_1.

        Returns:
            float: the value of `heating_capacity_coefficient_1` or None if not set

        """
        return self["Heating Capacity Coefficient 1"]

    @heating_capacity_coefficient_1.setter
    def heating_capacity_coefficient_1(self, value=None):
        """Corresponds to IDD field `Heating Capacity Coefficient 1`

        Args:
            value (float): value for IDD Field `Heating Capacity Coefficient 1`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Heating Capacity Coefficient 1"] = value

    @property
    def heating_capacity_coefficient_2(self):
        """Get heating_capacity_coefficient_2.

        Returns:
            float: the value of `heating_capacity_coefficient_2` or None if not set

        """
        return self["Heating Capacity Coefficient 2"]

    @heating_capacity_coefficient_2.setter
    def heating_capacity_coefficient_2(self, value=None):
        """Corresponds to IDD field `Heating Capacity Coefficient 2`

        Args:
            value (float): value for IDD Field `Heating Capacity Coefficient 2`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Heating Capacity Coefficient 2"] = value

    @property
    def heating_capacity_coefficient_3(self):
        """Get heating_capacity_coefficient_3.

        Returns:
            float: the value of `heating_capacity_coefficient_3` or None if not set

        """
        return self["Heating Capacity Coefficient 3"]

    @heating_capacity_coefficient_3.setter
    def heating_capacity_coefficient_3(self, value=None):
        """Corresponds to IDD field `Heating Capacity Coefficient 3`

        Args:
            value (float): value for IDD Field `Heating Capacity Coefficient 3`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Heating Capacity Coefficient 3"] = value

    @property
    def heating_capacity_coefficient_4(self):
        """Get heating_capacity_coefficient_4.

        Returns:
            float: the value of `heating_capacity_coefficient_4` or None if not set

        """
        return self["Heating Capacity Coefficient 4"]

    @heating_capacity_coefficient_4.setter
    def heating_capacity_coefficient_4(self, value=None):
        """Corresponds to IDD field `Heating Capacity Coefficient 4`

        Args:
            value (float): value for IDD Field `Heating Capacity Coefficient 4`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Heating Capacity Coefficient 4"] = value

    @property
    def heating_capacity_coefficient_5(self):
        """Get heating_capacity_coefficient_5.

        Returns:
            float: the value of `heating_capacity_coefficient_5` or None if not set

        """
        return self["Heating Capacity Coefficient 5"]

    @heating_capacity_coefficient_5.setter
    def heating_capacity_coefficient_5(self, value=None):
        """Corresponds to IDD field `Heating Capacity Coefficient 5`

        Args:
            value (float): value for IDD Field `Heating Capacity Coefficient 5`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Heating Capacity Coefficient 5"] = value

    @property
    def heating_compressor_power_coefficient_1(self):
        """Get heating_compressor_power_coefficient_1.

        Returns:
            float: the value of `heating_compressor_power_coefficient_1` or None if not set

        """
        return self["Heating Compressor Power Coefficient 1"]

    @heating_compressor_power_coefficient_1.setter
    def heating_compressor_power_coefficient_1(self, value=None):
        """Corresponds to IDD field `Heating Compressor Power Coefficient 1`

        Args:
            value (float): value for IDD Field `Heating Compressor Power Coefficient 1`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Heating Compressor Power Coefficient 1"] = value

    @property
    def heating_compressor_power_coefficient_2(self):
        """Get heating_compressor_power_coefficient_2.

        Returns:
            float: the value of `heating_compressor_power_coefficient_2` or None if not set

        """
        return self["Heating Compressor Power Coefficient 2"]

    @heating_compressor_power_coefficient_2.setter
    def heating_compressor_power_coefficient_2(self, value=None):
        """Corresponds to IDD field `Heating Compressor Power Coefficient 2`

        Args:
            value (float): value for IDD Field `Heating Compressor Power Coefficient 2`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Heating Compressor Power Coefficient 2"] = value

    @property
    def heating_compressor_power_coefficient_3(self):
        """Get heating_compressor_power_coefficient_3.

        Returns:
            float: the value of `heating_compressor_power_coefficient_3` or None if not set

        """
        return self["Heating Compressor Power Coefficient 3"]

    @heating_compressor_power_coefficient_3.setter
    def heating_compressor_power_coefficient_3(self, value=None):
        """Corresponds to IDD field `Heating Compressor Power Coefficient 3`

        Args:
            value (float): value for IDD Field `Heating Compressor Power Coefficient 3`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Heating Compressor Power Coefficient 3"] = value

    @property
    def heating_compressor_power_coefficient_4(self):
        """Get heating_compressor_power_coefficient_4.

        Returns:
            float: the value of `heating_compressor_power_coefficient_4` or None if not set

        """
        return self["Heating Compressor Power Coefficient 4"]

    @heating_compressor_power_coefficient_4.setter
    def heating_compressor_power_coefficient_4(self, value=None):
        """Corresponds to IDD field `Heating Compressor Power Coefficient 4`

        Args:
            value (float): value for IDD Field `Heating Compressor Power Coefficient 4`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Heating Compressor Power Coefficient 4"] = value

    @property
    def heating_compressor_power_coefficient_5(self):
        """Get heating_compressor_power_coefficient_5.

        Returns:
            float: the value of `heating_compressor_power_coefficient_5` or None if not set

        """
        return self["Heating Compressor Power Coefficient 5"]

    @heating_compressor_power_coefficient_5.setter
    def heating_compressor_power_coefficient_5(self, value=None):
        """Corresponds to IDD field `Heating Compressor Power Coefficient 5`

        Args:
            value (float): value for IDD Field `Heating Compressor Power Coefficient 5`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Heating Compressor Power Coefficient 5"] = value




class HeatPumpWaterToWaterEquationFitCooling(DataObject):

    """ Corresponds to IDD object `HeatPump:WaterToWater:EquationFit:Cooling`
        simple water-water heatpump curve-fit model
    """
    schema = {'min-fields': 19,
              'name': u'HeatPump:WaterToWater:EquationFit:Cooling',
              'pyname': u'HeatPumpWaterToWaterEquationFitCooling',
              'format': None,
              'fields': OrderedDict([(u'name',
                                      {'name': u'Name',
                                       'pyname': u'name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'source side inlet node name',
                                      {'name': u'Source Side Inlet Node Name',
                                       'pyname': u'source_side_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'source side outlet node name',
                                      {'name': u'Source Side Outlet Node Name',
                                       'pyname': u'source_side_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'load side inlet node name',
                                      {'name': u'Load Side Inlet Node Name',
                                       'pyname': u'load_side_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'load side outlet node name',
                                      {'name': u'Load Side Outlet Node Name',
                                       'pyname': u'load_side_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'rated load side flow rate',
                                      {'name': u'Rated Load Side Flow Rate',
                                       'pyname': u'rated_load_side_flow_rate',
                                       'minimum>': 0.0,
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'm3/s'}),
                                     (u'rated source side flow rate',
                                      {'name': u'Rated Source Side Flow Rate',
                                       'pyname': u'rated_source_side_flow_rate',
                                       'minimum>': 0.0,
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'm3/s'}),
                                     (u'rated cooling capacity',
                                      {'name': u'Rated Cooling Capacity',
                                       'pyname': u'rated_cooling_capacity',
                                       'minimum>': 0.0,
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'W'}),
                                     (u'rated cooling power consumption',
                                      {'name': u'Rated Cooling Power Consumption',
                                       'pyname': u'rated_cooling_power_consumption',
                                       'minimum>': 0.0,
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'W'}),
                                     (u'cooling capacity coefficient 1',
                                      {'name': u'Cooling Capacity Coefficient 1',
                                       'pyname': u'cooling_capacity_coefficient_1',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'cooling capacity coefficient 2',
                                      {'name': u'Cooling Capacity Coefficient 2',
                                       'pyname': u'cooling_capacity_coefficient_2',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'cooling capacity coefficient 3',
                                      {'name': u'Cooling Capacity Coefficient 3',
                                       'pyname': u'cooling_capacity_coefficient_3',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'cooling capacity coefficient 4',
                                      {'name': u'Cooling Capacity Coefficient 4',
                                       'pyname': u'cooling_capacity_coefficient_4',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'cooling capacity coefficient 5',
                                      {'name': u'Cooling Capacity Coefficient 5',
                                       'pyname': u'cooling_capacity_coefficient_5',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'cooling compressor power coefficient 1',
                                      {'name': u'Cooling Compressor Power Coefficient 1',
                                       'pyname': u'cooling_compressor_power_coefficient_1',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'cooling compressor power coefficient 2',
                                      {'name': u'Cooling Compressor Power Coefficient 2',
                                       'pyname': u'cooling_compressor_power_coefficient_2',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'cooling compressor power coefficient 3',
                                      {'name': u'Cooling Compressor Power Coefficient 3',
                                       'pyname': u'cooling_compressor_power_coefficient_3',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'cooling compressor power coefficient 4',
                                      {'name': u'Cooling Compressor Power Coefficient 4',
                                       'pyname': u'cooling_compressor_power_coefficient_4',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'cooling compressor power coefficient 5',
                                      {'name': u'Cooling Compressor Power Coefficient 5',
                                       'pyname': u'cooling_compressor_power_coefficient_5',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'})]),
              'extensible-fields': OrderedDict(),
              'unique-object': False,
              'required-object': False,
              'group': u'Plant Heating and Cooling Equipment'}

    @property
    def name(self):
        """Get name.

        Returns:
            str: the value of `name` or None if not set

        """
        return self["Name"]

    @name.setter
    def name(self, value=None):
        """Corresponds to IDD field `Name`

        Args:
            value (str): value for IDD Field `Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Name"] = value

    @property
    def source_side_inlet_node_name(self):
        """Get source_side_inlet_node_name.

        Returns:
            str: the value of `source_side_inlet_node_name` or None if not set

        """
        return self["Source Side Inlet Node Name"]

    @source_side_inlet_node_name.setter
    def source_side_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Source Side Inlet Node Name`

        Args:
            value (str): value for IDD Field `Source Side Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Source Side Inlet Node Name"] = value

    @property
    def source_side_outlet_node_name(self):
        """Get source_side_outlet_node_name.

        Returns:
            str: the value of `source_side_outlet_node_name` or None if not set

        """
        return self["Source Side Outlet Node Name"]

    @source_side_outlet_node_name.setter
    def source_side_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Source Side Outlet Node Name`

        Args:
            value (str): value for IDD Field `Source Side Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Source Side Outlet Node Name"] = value

    @property
    def load_side_inlet_node_name(self):
        """Get load_side_inlet_node_name.

        Returns:
            str: the value of `load_side_inlet_node_name` or None if not set

        """
        return self["Load Side Inlet Node Name"]

    @load_side_inlet_node_name.setter
    def load_side_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Load Side Inlet Node Name`

        Args:
            value (str): value for IDD Field `Load Side Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Load Side Inlet Node Name"] = value

    @property
    def load_side_outlet_node_name(self):
        """Get load_side_outlet_node_name.

        Returns:
            str: the value of `load_side_outlet_node_name` or None if not set

        """
        return self["Load Side Outlet Node Name"]

    @load_side_outlet_node_name.setter
    def load_side_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Load Side Outlet Node Name`

        Args:
            value (str): value for IDD Field `Load Side Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Load Side Outlet Node Name"] = value

    @property
    def rated_load_side_flow_rate(self):
        """Get rated_load_side_flow_rate.

        Returns:
            float: the value of `rated_load_side_flow_rate` or None if not set

        """
        return self["Rated Load Side Flow Rate"]

    @rated_load_side_flow_rate.setter
    def rated_load_side_flow_rate(self, value=None):
        """Corresponds to IDD field `Rated Load Side Flow Rate`

        Args:
            value (float): value for IDD Field `Rated Load Side Flow Rate`
                Units: m3/s
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Rated Load Side Flow Rate"] = value

    @property
    def rated_source_side_flow_rate(self):
        """Get rated_source_side_flow_rate.

        Returns:
            float: the value of `rated_source_side_flow_rate` or None if not set

        """
        return self["Rated Source Side Flow Rate"]

    @rated_source_side_flow_rate.setter
    def rated_source_side_flow_rate(self, value=None):
        """Corresponds to IDD field `Rated Source Side Flow Rate`

        Args:
            value (float): value for IDD Field `Rated Source Side Flow Rate`
                Units: m3/s
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Rated Source Side Flow Rate"] = value

    @property
    def rated_cooling_capacity(self):
        """Get rated_cooling_capacity.

        Returns:
            float: the value of `rated_cooling_capacity` or None if not set

        """
        return self["Rated Cooling Capacity"]

    @rated_cooling_capacity.setter
    def rated_cooling_capacity(self, value=None):
        """Corresponds to IDD field `Rated Cooling Capacity`

        Args:
            value (float): value for IDD Field `Rated Cooling Capacity`
                Units: W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Rated Cooling Capacity"] = value

    @property
    def rated_cooling_power_consumption(self):
        """Get rated_cooling_power_consumption.

        Returns:
            float: the value of `rated_cooling_power_consumption` or None if not set

        """
        return self["Rated Cooling Power Consumption"]

    @rated_cooling_power_consumption.setter
    def rated_cooling_power_consumption(self, value=None):
        """Corresponds to IDD field `Rated Cooling Power Consumption`

        Args:
            value (float): value for IDD Field `Rated Cooling Power Consumption`
                Units: W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Rated Cooling Power Consumption"] = value

    @property
    def cooling_capacity_coefficient_1(self):
        """Get cooling_capacity_coefficient_1.

        Returns:
            float: the value of `cooling_capacity_coefficient_1` or None if not set

        """
        return self["Cooling Capacity Coefficient 1"]

    @cooling_capacity_coefficient_1.setter
    def cooling_capacity_coefficient_1(self, value=None):
        """Corresponds to IDD field `Cooling Capacity Coefficient 1`

        Args:
            value (float): value for IDD Field `Cooling Capacity Coefficient 1`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Cooling Capacity Coefficient 1"] = value

    @property
    def cooling_capacity_coefficient_2(self):
        """Get cooling_capacity_coefficient_2.

        Returns:
            float: the value of `cooling_capacity_coefficient_2` or None if not set

        """
        return self["Cooling Capacity Coefficient 2"]

    @cooling_capacity_coefficient_2.setter
    def cooling_capacity_coefficient_2(self, value=None):
        """Corresponds to IDD field `Cooling Capacity Coefficient 2`

        Args:
            value (float): value for IDD Field `Cooling Capacity Coefficient 2`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Cooling Capacity Coefficient 2"] = value

    @property
    def cooling_capacity_coefficient_3(self):
        """Get cooling_capacity_coefficient_3.

        Returns:
            float: the value of `cooling_capacity_coefficient_3` or None if not set

        """
        return self["Cooling Capacity Coefficient 3"]

    @cooling_capacity_coefficient_3.setter
    def cooling_capacity_coefficient_3(self, value=None):
        """Corresponds to IDD field `Cooling Capacity Coefficient 3`

        Args:
            value (float): value for IDD Field `Cooling Capacity Coefficient 3`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Cooling Capacity Coefficient 3"] = value

    @property
    def cooling_capacity_coefficient_4(self):
        """Get cooling_capacity_coefficient_4.

        Returns:
            float: the value of `cooling_capacity_coefficient_4` or None if not set

        """
        return self["Cooling Capacity Coefficient 4"]

    @cooling_capacity_coefficient_4.setter
    def cooling_capacity_coefficient_4(self, value=None):
        """Corresponds to IDD field `Cooling Capacity Coefficient 4`

        Args:
            value (float): value for IDD Field `Cooling Capacity Coefficient 4`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Cooling Capacity Coefficient 4"] = value

    @property
    def cooling_capacity_coefficient_5(self):
        """Get cooling_capacity_coefficient_5.

        Returns:
            float: the value of `cooling_capacity_coefficient_5` or None if not set

        """
        return self["Cooling Capacity Coefficient 5"]

    @cooling_capacity_coefficient_5.setter
    def cooling_capacity_coefficient_5(self, value=None):
        """Corresponds to IDD field `Cooling Capacity Coefficient 5`

        Args:
            value (float): value for IDD Field `Cooling Capacity Coefficient 5`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Cooling Capacity Coefficient 5"] = value

    @property
    def cooling_compressor_power_coefficient_1(self):
        """Get cooling_compressor_power_coefficient_1.

        Returns:
            float: the value of `cooling_compressor_power_coefficient_1` or None if not set

        """
        return self["Cooling Compressor Power Coefficient 1"]

    @cooling_compressor_power_coefficient_1.setter
    def cooling_compressor_power_coefficient_1(self, value=None):
        """Corresponds to IDD field `Cooling Compressor Power Coefficient 1`

        Args:
            value (float): value for IDD Field `Cooling Compressor Power Coefficient 1`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Cooling Compressor Power Coefficient 1"] = value

    @property
    def cooling_compressor_power_coefficient_2(self):
        """Get cooling_compressor_power_coefficient_2.

        Returns:
            float: the value of `cooling_compressor_power_coefficient_2` or None if not set

        """
        return self["Cooling Compressor Power Coefficient 2"]

    @cooling_compressor_power_coefficient_2.setter
    def cooling_compressor_power_coefficient_2(self, value=None):
        """Corresponds to IDD field `Cooling Compressor Power Coefficient 2`

        Args:
            value (float): value for IDD Field `Cooling Compressor Power Coefficient 2`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Cooling Compressor Power Coefficient 2"] = value

    @property
    def cooling_compressor_power_coefficient_3(self):
        """Get cooling_compressor_power_coefficient_3.

        Returns:
            float: the value of `cooling_compressor_power_coefficient_3` or None if not set

        """
        return self["Cooling Compressor Power Coefficient 3"]

    @cooling_compressor_power_coefficient_3.setter
    def cooling_compressor_power_coefficient_3(self, value=None):
        """Corresponds to IDD field `Cooling Compressor Power Coefficient 3`

        Args:
            value (float): value for IDD Field `Cooling Compressor Power Coefficient 3`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Cooling Compressor Power Coefficient 3"] = value

    @property
    def cooling_compressor_power_coefficient_4(self):
        """Get cooling_compressor_power_coefficient_4.

        Returns:
            float: the value of `cooling_compressor_power_coefficient_4` or None if not set

        """
        return self["Cooling Compressor Power Coefficient 4"]

    @cooling_compressor_power_coefficient_4.setter
    def cooling_compressor_power_coefficient_4(self, value=None):
        """Corresponds to IDD field `Cooling Compressor Power Coefficient 4`

        Args:
            value (float): value for IDD Field `Cooling Compressor Power Coefficient 4`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Cooling Compressor Power Coefficient 4"] = value

    @property
    def cooling_compressor_power_coefficient_5(self):
        """Get cooling_compressor_power_coefficient_5.

        Returns:
            float: the value of `cooling_compressor_power_coefficient_5` or None if not set

        """
        return self["Cooling Compressor Power Coefficient 5"]

    @cooling_compressor_power_coefficient_5.setter
    def cooling_compressor_power_coefficient_5(self, value=None):
        """Corresponds to IDD field `Cooling Compressor Power Coefficient 5`

        Args:
            value (float): value for IDD Field `Cooling Compressor Power Coefficient 5`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Cooling Compressor Power Coefficient 5"] = value




class HeatPumpWaterToWaterParameterEstimationCooling(DataObject):

    """ Corresponds to IDD object `HeatPump:WaterToWater:ParameterEstimation:Cooling`
        OSU parameter estimation model
    """
    schema = {'min-fields': 20,
              'name': u'HeatPump:WaterToWater:ParameterEstimation:Cooling',
              'pyname': u'HeatPumpWaterToWaterParameterEstimationCooling',
              'format': None,
              'fields': OrderedDict([(u'name',
                                      {'name': u'Name',
                                       'pyname': u'name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'source side inlet node name',
                                      {'name': u'Source Side Inlet Node Name',
                                       'pyname': u'source_side_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'source side outlet node name',
                                      {'name': u'Source Side Outlet Node Name',
                                       'pyname': u'source_side_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'load side inlet node name',
                                      {'name': u'Load Side Inlet Node Name',
                                       'pyname': u'load_side_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'load side outlet node name',
                                      {'name': u'Load Side Outlet Node Name',
                                       'pyname': u'load_side_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'nominal cop',
                                      {'name': u'Nominal COP',
                                       'pyname': u'nominal_cop',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'W/W'}),
                                     (u'nominal capacity',
                                      {'name': u'Nominal Capacity',
                                       'pyname': u'nominal_capacity',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'W'}),
                                     (u'minimum part load ratio',
                                      {'name': u'Minimum Part Load Ratio',
                                       'pyname': u'minimum_part_load_ratio',
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'maximum part load ratio',
                                      {'name': u'Maximum Part Load Ratio',
                                       'pyname': u'maximum_part_load_ratio',
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'optimum part load ratio',
                                      {'name': u'Optimum Part Load Ratio',
                                       'pyname': u'optimum_part_load_ratio',
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'load side flow rate',
                                      {'name': u'Load Side Flow Rate',
                                       'pyname': u'load_side_flow_rate',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'm3/s'}),
                                     (u'source side flow rate',
                                      {'name': u'Source Side Flow Rate',
                                       'pyname': u'source_side_flow_rate',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'm3/s'}),
                                     (u'load side heat transfer coefficient',
                                      {'name': u'Load Side Heat Transfer Coefficient',
                                       'pyname': u'load_side_heat_transfer_coefficient',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'W/K'}),
                                     (u'source side heat transfer coefficient',
                                      {'name': u'Source Side Heat Transfer Coefficient',
                                       'pyname': u'source_side_heat_transfer_coefficient',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'W/K'}),
                                     (u'piston displacement',
                                      {'name': u'Piston Displacement',
                                       'pyname': u'piston_displacement',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'm3/s'}),
                                     (u'compressor clearance factor',
                                      {'name': u'Compressor Clearance Factor',
                                       'pyname': u'compressor_clearance_factor',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'compressor suction and discharge pressure drop',
                                      {'name': u'Compressor Suction and Discharge Pressure Drop',
                                       'pyname': u'compressor_suction_and_discharge_pressure_drop',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'Pa'}),
                                     (u'superheating',
                                      {'name': u'Superheating',
                                       'pyname': u'superheating',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'C'}),
                                     (u'constant part of electromechanical power losses',
                                      {'name': u'Constant Part of Electromechanical Power Losses',
                                       'pyname': u'constant_part_of_electromechanical_power_losses',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'W'}),
                                     (u'loss factor',
                                      {'name': u'Loss Factor',
                                       'pyname': u'loss_factor',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'high pressure cut off',
                                      {'name': u'High Pressure Cut Off',
                                       'pyname': u'high_pressure_cut_off',
                                       'default': 500000000.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'Pa'}),
                                     (u'low pressure cut off',
                                      {'name': u'Low Pressure Cut Off',
                                       'pyname': u'low_pressure_cut_off',
                                       'default': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'Pa'})]),
              'extensible-fields': OrderedDict(),
              'unique-object': False,
              'required-object': False,
              'group': u'Plant Heating and Cooling Equipment'}

    @property
    def name(self):
        """Get name.

        Returns:
            str: the value of `name` or None if not set

        """
        return self["Name"]

    @name.setter
    def name(self, value=None):
        """Corresponds to IDD field `Name`

        Args:
            value (str): value for IDD Field `Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Name"] = value

    @property
    def source_side_inlet_node_name(self):
        """Get source_side_inlet_node_name.

        Returns:
            str: the value of `source_side_inlet_node_name` or None if not set

        """
        return self["Source Side Inlet Node Name"]

    @source_side_inlet_node_name.setter
    def source_side_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Source Side Inlet Node Name`

        Args:
            value (str): value for IDD Field `Source Side Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Source Side Inlet Node Name"] = value

    @property
    def source_side_outlet_node_name(self):
        """Get source_side_outlet_node_name.

        Returns:
            str: the value of `source_side_outlet_node_name` or None if not set

        """
        return self["Source Side Outlet Node Name"]

    @source_side_outlet_node_name.setter
    def source_side_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Source Side Outlet Node Name`

        Args:
            value (str): value for IDD Field `Source Side Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Source Side Outlet Node Name"] = value

    @property
    def load_side_inlet_node_name(self):
        """Get load_side_inlet_node_name.

        Returns:
            str: the value of `load_side_inlet_node_name` or None if not set

        """
        return self["Load Side Inlet Node Name"]

    @load_side_inlet_node_name.setter
    def load_side_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Load Side Inlet Node Name`

        Args:
            value (str): value for IDD Field `Load Side Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Load Side Inlet Node Name"] = value

    @property
    def load_side_outlet_node_name(self):
        """Get load_side_outlet_node_name.

        Returns:
            str: the value of `load_side_outlet_node_name` or None if not set

        """
        return self["Load Side Outlet Node Name"]

    @load_side_outlet_node_name.setter
    def load_side_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Load Side Outlet Node Name`

        Args:
            value (str): value for IDD Field `Load Side Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Load Side Outlet Node Name"] = value

    @property
    def nominal_cop(self):
        """Get nominal_cop.

        Returns:
            float: the value of `nominal_cop` or None if not set

        """
        return self["Nominal COP"]

    @nominal_cop.setter
    def nominal_cop(self, value=None):
        """Corresponds to IDD field `Nominal COP`

        Args:
            value (float): value for IDD Field `Nominal COP`
                Units: W/W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Nominal COP"] = value

    @property
    def nominal_capacity(self):
        """Get nominal_capacity.

        Returns:
            float: the value of `nominal_capacity` or None if not set

        """
        return self["Nominal Capacity"]

    @nominal_capacity.setter
    def nominal_capacity(self, value=None):
        """Corresponds to IDD field `Nominal Capacity`

        Args:
            value (float): value for IDD Field `Nominal Capacity`
                Units: W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Nominal Capacity"] = value

    @property
    def minimum_part_load_ratio(self):
        """Get minimum_part_load_ratio.

        Returns:
            float: the value of `minimum_part_load_ratio` or None if not set

        """
        return self["Minimum Part Load Ratio"]

    @minimum_part_load_ratio.setter
    def minimum_part_load_ratio(self, value=None):
        """Corresponds to IDD field `Minimum Part Load Ratio`

        Args:
            value (float): value for IDD Field `Minimum Part Load Ratio`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Minimum Part Load Ratio"] = value

    @property
    def maximum_part_load_ratio(self):
        """Get maximum_part_load_ratio.

        Returns:
            float: the value of `maximum_part_load_ratio` or None if not set

        """
        return self["Maximum Part Load Ratio"]

    @maximum_part_load_ratio.setter
    def maximum_part_load_ratio(self, value=None):
        """Corresponds to IDD field `Maximum Part Load Ratio`

        Args:
            value (float): value for IDD Field `Maximum Part Load Ratio`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Maximum Part Load Ratio"] = value

    @property
    def optimum_part_load_ratio(self):
        """Get optimum_part_load_ratio.

        Returns:
            float: the value of `optimum_part_load_ratio` or None if not set

        """
        return self["Optimum Part Load Ratio"]

    @optimum_part_load_ratio.setter
    def optimum_part_load_ratio(self, value=None):
        """Corresponds to IDD field `Optimum Part Load Ratio`

        Args:
            value (float): value for IDD Field `Optimum Part Load Ratio`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Optimum Part Load Ratio"] = value

    @property
    def load_side_flow_rate(self):
        """Get load_side_flow_rate.

        Returns:
            float: the value of `load_side_flow_rate` or None if not set

        """
        return self["Load Side Flow Rate"]

    @load_side_flow_rate.setter
    def load_side_flow_rate(self, value=None):
        """Corresponds to IDD field `Load Side Flow Rate`

        Args:
            value (float): value for IDD Field `Load Side Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Load Side Flow Rate"] = value

    @property
    def source_side_flow_rate(self):
        """Get source_side_flow_rate.

        Returns:
            float: the value of `source_side_flow_rate` or None if not set

        """
        return self["Source Side Flow Rate"]

    @source_side_flow_rate.setter
    def source_side_flow_rate(self, value=None):
        """Corresponds to IDD field `Source Side Flow Rate`

        Args:
            value (float): value for IDD Field `Source Side Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Source Side Flow Rate"] = value

    @property
    def load_side_heat_transfer_coefficient(self):
        """Get load_side_heat_transfer_coefficient.

        Returns:
            float: the value of `load_side_heat_transfer_coefficient` or None if not set

        """
        return self["Load Side Heat Transfer Coefficient"]

    @load_side_heat_transfer_coefficient.setter
    def load_side_heat_transfer_coefficient(self, value=None):
        """Corresponds to IDD field `Load Side Heat Transfer Coefficient`

        Args:
            value (float): value for IDD Field `Load Side Heat Transfer Coefficient`
                Units: W/K
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Load Side Heat Transfer Coefficient"] = value

    @property
    def source_side_heat_transfer_coefficient(self):
        """Get source_side_heat_transfer_coefficient.

        Returns:
            float: the value of `source_side_heat_transfer_coefficient` or None if not set

        """
        return self["Source Side Heat Transfer Coefficient"]

    @source_side_heat_transfer_coefficient.setter
    def source_side_heat_transfer_coefficient(self, value=None):
        """Corresponds to IDD field `Source Side Heat Transfer Coefficient`

        Args:
            value (float): value for IDD Field `Source Side Heat Transfer Coefficient`
                Units: W/K
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Source Side Heat Transfer Coefficient"] = value

    @property
    def piston_displacement(self):
        """Get piston_displacement.

        Returns:
            float: the value of `piston_displacement` or None if not set

        """
        return self["Piston Displacement"]

    @piston_displacement.setter
    def piston_displacement(self, value=None):
        """Corresponds to IDD field `Piston Displacement`

        Args:
            value (float): value for IDD Field `Piston Displacement`
                Units: m3/s
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Piston Displacement"] = value

    @property
    def compressor_clearance_factor(self):
        """Get compressor_clearance_factor.

        Returns:
            float: the value of `compressor_clearance_factor` or None if not set

        """
        return self["Compressor Clearance Factor"]

    @compressor_clearance_factor.setter
    def compressor_clearance_factor(self, value=None):
        """Corresponds to IDD field `Compressor Clearance Factor`

        Args:
            value (float): value for IDD Field `Compressor Clearance Factor`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Compressor Clearance Factor"] = value

    @property
    def compressor_suction_and_discharge_pressure_drop(self):
        """Get compressor_suction_and_discharge_pressure_drop.

        Returns:
            float: the value of `compressor_suction_and_discharge_pressure_drop` or None if not set

        """
        return self["Compressor Suction and Discharge Pressure Drop"]

    @compressor_suction_and_discharge_pressure_drop.setter
    def compressor_suction_and_discharge_pressure_drop(self, value=None):
        """Corresponds to IDD field `Compressor Suction and Discharge Pressure
        Drop`

        Args:
            value (float): value for IDD Field `Compressor Suction and Discharge Pressure Drop`
                Units: Pa
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Compressor Suction and Discharge Pressure Drop"] = value

    @property
    def superheating(self):
        """Get superheating.

        Returns:
            float: the value of `superheating` or None if not set

        """
        return self["Superheating"]

    @superheating.setter
    def superheating(self, value=None):
        """Corresponds to IDD field `Superheating`

        Args:
            value (float): value for IDD Field `Superheating`
                Units: C
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Superheating"] = value

    @property
    def constant_part_of_electromechanical_power_losses(self):
        """Get constant_part_of_electromechanical_power_losses.

        Returns:
            float: the value of `constant_part_of_electromechanical_power_losses` or None if not set

        """
        return self["Constant Part of Electromechanical Power Losses"]

    @constant_part_of_electromechanical_power_losses.setter
    def constant_part_of_electromechanical_power_losses(self, value=None):
        """Corresponds to IDD field `Constant Part of Electromechanical Power
        Losses`

        Args:
            value (float): value for IDD Field `Constant Part of Electromechanical Power Losses`
                Units: W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Constant Part of Electromechanical Power Losses"] = value

    @property
    def loss_factor(self):
        """Get loss_factor.

        Returns:
            float: the value of `loss_factor` or None if not set

        """
        return self["Loss Factor"]

    @loss_factor.setter
    def loss_factor(self, value=None):
        """Corresponds to IDD field `Loss Factor` Used to define
        electromechanical loss that is proportional to the theoretical power %

        Args:
            value (float): value for IDD Field `Loss Factor`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Loss Factor"] = value

    @property
    def high_pressure_cut_off(self):
        """Get high_pressure_cut_off.

        Returns:
            float: the value of `high_pressure_cut_off` or None if not set

        """
        return self["High Pressure Cut Off"]

    @high_pressure_cut_off.setter
    def high_pressure_cut_off(self, value=500000000.0):
        """Corresponds to IDD field `High Pressure Cut Off`

        Args:
            value (float): value for IDD Field `High Pressure Cut Off`
                Units: Pa
                Default value: 500000000.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["High Pressure Cut Off"] = value

    @property
    def low_pressure_cut_off(self):
        """Get low_pressure_cut_off.

        Returns:
            float: the value of `low_pressure_cut_off` or None if not set

        """
        return self["Low Pressure Cut Off"]

    @low_pressure_cut_off.setter
    def low_pressure_cut_off(self, value=None):
        """Corresponds to IDD field `Low Pressure Cut Off`

        Args:
            value (float): value for IDD Field `Low Pressure Cut Off`
                Units: Pa
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Low Pressure Cut Off"] = value




class HeatPumpWaterToWaterParameterEstimationHeating(DataObject):

    """ Corresponds to IDD object `HeatPump:WaterToWater:ParameterEstimation:Heating`
        OSU parameter estimation model
    """
    schema = {'min-fields': 20,
              'name': u'HeatPump:WaterToWater:ParameterEstimation:Heating',
              'pyname': u'HeatPumpWaterToWaterParameterEstimationHeating',
              'format': None,
              'fields': OrderedDict([(u'name',
                                      {'name': u'Name',
                                       'pyname': u'name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'source side inlet node name',
                                      {'name': u'Source Side Inlet Node Name',
                                       'pyname': u'source_side_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'source side outlet node name',
                                      {'name': u'Source Side Outlet Node Name',
                                       'pyname': u'source_side_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'load side inlet node name',
                                      {'name': u'Load Side Inlet Node Name',
                                       'pyname': u'load_side_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'load side outlet node name',
                                      {'name': u'Load Side Outlet Node Name',
                                       'pyname': u'load_side_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'nominal cop',
                                      {'name': u'Nominal COP',
                                       'pyname': u'nominal_cop',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'W/W'}),
                                     (u'nominal capacity',
                                      {'name': u'Nominal Capacity',
                                       'pyname': u'nominal_capacity',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'W'}),
                                     (u'minimum part load ratio',
                                      {'name': u'Minimum Part Load Ratio',
                                       'pyname': u'minimum_part_load_ratio',
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'maximum part load ratio',
                                      {'name': u'Maximum Part Load Ratio',
                                       'pyname': u'maximum_part_load_ratio',
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'optimum part load ratio',
                                      {'name': u'Optimum Part Load Ratio',
                                       'pyname': u'optimum_part_load_ratio',
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'load side flow rate',
                                      {'name': u'Load Side Flow Rate',
                                       'pyname': u'load_side_flow_rate',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'm3/s'}),
                                     (u'source side flow rate',
                                      {'name': u'Source Side Flow Rate',
                                       'pyname': u'source_side_flow_rate',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'm3/s'}),
                                     (u'load side heat transfer coefficient',
                                      {'name': u'Load Side Heat Transfer Coefficient',
                                       'pyname': u'load_side_heat_transfer_coefficient',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'W/K'}),
                                     (u'source side heat transfer coefficient',
                                      {'name': u'Source Side Heat Transfer Coefficient',
                                       'pyname': u'source_side_heat_transfer_coefficient',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'W/K'}),
                                     (u'piston displacement',
                                      {'name': u'Piston Displacement',
                                       'pyname': u'piston_displacement',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'm3/s'}),
                                     (u'compressor clearance factor',
                                      {'name': u'Compressor Clearance Factor',
                                       'pyname': u'compressor_clearance_factor',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'compressor suction and discharge pressure drop',
                                      {'name': u'Compressor Suction and Discharge Pressure Drop',
                                       'pyname': u'compressor_suction_and_discharge_pressure_drop',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'Pa'}),
                                     (u'superheating',
                                      {'name': u'Superheating',
                                       'pyname': u'superheating',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'C'}),
                                     (u'constant part of electromechanical power losses',
                                      {'name': u'Constant Part of Electromechanical Power Losses',
                                       'pyname': u'constant_part_of_electromechanical_power_losses',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'W'}),
                                     (u'loss factor',
                                      {'name': u'Loss Factor',
                                       'pyname': u'loss_factor',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'real'}),
                                     (u'high pressure cut off',
                                      {'name': u'High Pressure Cut Off',
                                       'pyname': u'high_pressure_cut_off',
                                       'default': 500000000.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'Pa'}),
                                     (u'low pressure cut off',
                                      {'name': u'Low Pressure Cut Off',
                                       'pyname': u'low_pressure_cut_off',
                                       'default': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'Pa'})]),
              'extensible-fields': OrderedDict(),
              'unique-object': False,
              'required-object': False,
              'group': u'Plant Heating and Cooling Equipment'}

    @property
    def name(self):
        """Get name.

        Returns:
            str: the value of `name` or None if not set

        """
        return self["Name"]

    @name.setter
    def name(self, value=None):
        """Corresponds to IDD field `Name`

        Args:
            value (str): value for IDD Field `Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Name"] = value

    @property
    def source_side_inlet_node_name(self):
        """Get source_side_inlet_node_name.

        Returns:
            str: the value of `source_side_inlet_node_name` or None if not set

        """
        return self["Source Side Inlet Node Name"]

    @source_side_inlet_node_name.setter
    def source_side_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Source Side Inlet Node Name`

        Args:
            value (str): value for IDD Field `Source Side Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Source Side Inlet Node Name"] = value

    @property
    def source_side_outlet_node_name(self):
        """Get source_side_outlet_node_name.

        Returns:
            str: the value of `source_side_outlet_node_name` or None if not set

        """
        return self["Source Side Outlet Node Name"]

    @source_side_outlet_node_name.setter
    def source_side_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Source Side Outlet Node Name`

        Args:
            value (str): value for IDD Field `Source Side Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Source Side Outlet Node Name"] = value

    @property
    def load_side_inlet_node_name(self):
        """Get load_side_inlet_node_name.

        Returns:
            str: the value of `load_side_inlet_node_name` or None if not set

        """
        return self["Load Side Inlet Node Name"]

    @load_side_inlet_node_name.setter
    def load_side_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Load Side Inlet Node Name`

        Args:
            value (str): value for IDD Field `Load Side Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Load Side Inlet Node Name"] = value

    @property
    def load_side_outlet_node_name(self):
        """Get load_side_outlet_node_name.

        Returns:
            str: the value of `load_side_outlet_node_name` or None if not set

        """
        return self["Load Side Outlet Node Name"]

    @load_side_outlet_node_name.setter
    def load_side_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Load Side Outlet Node Name`

        Args:
            value (str): value for IDD Field `Load Side Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Load Side Outlet Node Name"] = value

    @property
    def nominal_cop(self):
        """Get nominal_cop.

        Returns:
            float: the value of `nominal_cop` or None if not set

        """
        return self["Nominal COP"]

    @nominal_cop.setter
    def nominal_cop(self, value=None):
        """Corresponds to IDD field `Nominal COP`

        Args:
            value (float): value for IDD Field `Nominal COP`
                Units: W/W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Nominal COP"] = value

    @property
    def nominal_capacity(self):
        """Get nominal_capacity.

        Returns:
            float: the value of `nominal_capacity` or None if not set

        """
        return self["Nominal Capacity"]

    @nominal_capacity.setter
    def nominal_capacity(self, value=None):
        """Corresponds to IDD field `Nominal Capacity`

        Args:
            value (float): value for IDD Field `Nominal Capacity`
                Units: W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Nominal Capacity"] = value

    @property
    def minimum_part_load_ratio(self):
        """Get minimum_part_load_ratio.

        Returns:
            float: the value of `minimum_part_load_ratio` or None if not set

        """
        return self["Minimum Part Load Ratio"]

    @minimum_part_load_ratio.setter
    def minimum_part_load_ratio(self, value=None):
        """Corresponds to IDD field `Minimum Part Load Ratio`

        Args:
            value (float): value for IDD Field `Minimum Part Load Ratio`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Minimum Part Load Ratio"] = value

    @property
    def maximum_part_load_ratio(self):
        """Get maximum_part_load_ratio.

        Returns:
            float: the value of `maximum_part_load_ratio` or None if not set

        """
        return self["Maximum Part Load Ratio"]

    @maximum_part_load_ratio.setter
    def maximum_part_load_ratio(self, value=None):
        """Corresponds to IDD field `Maximum Part Load Ratio`

        Args:
            value (float): value for IDD Field `Maximum Part Load Ratio`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Maximum Part Load Ratio"] = value

    @property
    def optimum_part_load_ratio(self):
        """Get optimum_part_load_ratio.

        Returns:
            float: the value of `optimum_part_load_ratio` or None if not set

        """
        return self["Optimum Part Load Ratio"]

    @optimum_part_load_ratio.setter
    def optimum_part_load_ratio(self, value=None):
        """Corresponds to IDD field `Optimum Part Load Ratio`

        Args:
            value (float): value for IDD Field `Optimum Part Load Ratio`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Optimum Part Load Ratio"] = value

    @property
    def load_side_flow_rate(self):
        """Get load_side_flow_rate.

        Returns:
            float: the value of `load_side_flow_rate` or None if not set

        """
        return self["Load Side Flow Rate"]

    @load_side_flow_rate.setter
    def load_side_flow_rate(self, value=None):
        """Corresponds to IDD field `Load Side Flow Rate`

        Args:
            value (float): value for IDD Field `Load Side Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Load Side Flow Rate"] = value

    @property
    def source_side_flow_rate(self):
        """Get source_side_flow_rate.

        Returns:
            float: the value of `source_side_flow_rate` or None if not set

        """
        return self["Source Side Flow Rate"]

    @source_side_flow_rate.setter
    def source_side_flow_rate(self, value=None):
        """Corresponds to IDD field `Source Side Flow Rate`

        Args:
            value (float): value for IDD Field `Source Side Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Source Side Flow Rate"] = value

    @property
    def load_side_heat_transfer_coefficient(self):
        """Get load_side_heat_transfer_coefficient.

        Returns:
            float: the value of `load_side_heat_transfer_coefficient` or None if not set

        """
        return self["Load Side Heat Transfer Coefficient"]

    @load_side_heat_transfer_coefficient.setter
    def load_side_heat_transfer_coefficient(self, value=None):
        """Corresponds to IDD field `Load Side Heat Transfer Coefficient`

        Args:
            value (float): value for IDD Field `Load Side Heat Transfer Coefficient`
                Units: W/K
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Load Side Heat Transfer Coefficient"] = value

    @property
    def source_side_heat_transfer_coefficient(self):
        """Get source_side_heat_transfer_coefficient.

        Returns:
            float: the value of `source_side_heat_transfer_coefficient` or None if not set

        """
        return self["Source Side Heat Transfer Coefficient"]

    @source_side_heat_transfer_coefficient.setter
    def source_side_heat_transfer_coefficient(self, value=None):
        """Corresponds to IDD field `Source Side Heat Transfer Coefficient`

        Args:
            value (float): value for IDD Field `Source Side Heat Transfer Coefficient`
                Units: W/K
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Source Side Heat Transfer Coefficient"] = value

    @property
    def piston_displacement(self):
        """Get piston_displacement.

        Returns:
            float: the value of `piston_displacement` or None if not set

        """
        return self["Piston Displacement"]

    @piston_displacement.setter
    def piston_displacement(self, value=None):
        """Corresponds to IDD field `Piston Displacement`

        Args:
            value (float): value for IDD Field `Piston Displacement`
                Units: m3/s
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Piston Displacement"] = value

    @property
    def compressor_clearance_factor(self):
        """Get compressor_clearance_factor.

        Returns:
            float: the value of `compressor_clearance_factor` or None if not set

        """
        return self["Compressor Clearance Factor"]

    @compressor_clearance_factor.setter
    def compressor_clearance_factor(self, value=None):
        """Corresponds to IDD field `Compressor Clearance Factor`

        Args:
            value (float): value for IDD Field `Compressor Clearance Factor`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Compressor Clearance Factor"] = value

    @property
    def compressor_suction_and_discharge_pressure_drop(self):
        """Get compressor_suction_and_discharge_pressure_drop.

        Returns:
            float: the value of `compressor_suction_and_discharge_pressure_drop` or None if not set

        """
        return self["Compressor Suction and Discharge Pressure Drop"]

    @compressor_suction_and_discharge_pressure_drop.setter
    def compressor_suction_and_discharge_pressure_drop(self, value=None):
        """Corresponds to IDD field `Compressor Suction and Discharge Pressure
        Drop`

        Args:
            value (float): value for IDD Field `Compressor Suction and Discharge Pressure Drop`
                Units: Pa
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Compressor Suction and Discharge Pressure Drop"] = value

    @property
    def superheating(self):
        """Get superheating.

        Returns:
            float: the value of `superheating` or None if not set

        """
        return self["Superheating"]

    @superheating.setter
    def superheating(self, value=None):
        """Corresponds to IDD field `Superheating`

        Args:
            value (float): value for IDD Field `Superheating`
                Units: C
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Superheating"] = value

    @property
    def constant_part_of_electromechanical_power_losses(self):
        """Get constant_part_of_electromechanical_power_losses.

        Returns:
            float: the value of `constant_part_of_electromechanical_power_losses` or None if not set

        """
        return self["Constant Part of Electromechanical Power Losses"]

    @constant_part_of_electromechanical_power_losses.setter
    def constant_part_of_electromechanical_power_losses(self, value=None):
        """Corresponds to IDD field `Constant Part of Electromechanical Power
        Losses`

        Args:
            value (float): value for IDD Field `Constant Part of Electromechanical Power Losses`
                Units: W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Constant Part of Electromechanical Power Losses"] = value

    @property
    def loss_factor(self):
        """Get loss_factor.

        Returns:
            float: the value of `loss_factor` or None if not set

        """
        return self["Loss Factor"]

    @loss_factor.setter
    def loss_factor(self, value=None):
        """Corresponds to IDD field `Loss Factor` Used to define
        electromechanical loss that is proportional to the theoretical power %

        Args:
            value (float): value for IDD Field `Loss Factor`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Loss Factor"] = value

    @property
    def high_pressure_cut_off(self):
        """Get high_pressure_cut_off.

        Returns:
            float: the value of `high_pressure_cut_off` or None if not set

        """
        return self["High Pressure Cut Off"]

    @high_pressure_cut_off.setter
    def high_pressure_cut_off(self, value=500000000.0):
        """Corresponds to IDD field `High Pressure Cut Off`

        Args:
            value (float): value for IDD Field `High Pressure Cut Off`
                Units: Pa
                Default value: 500000000.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["High Pressure Cut Off"] = value

    @property
    def low_pressure_cut_off(self):
        """Get low_pressure_cut_off.

        Returns:
            float: the value of `low_pressure_cut_off` or None if not set

        """
        return self["Low Pressure Cut Off"]

    @low_pressure_cut_off.setter
    def low_pressure_cut_off(self, value=None):
        """Corresponds to IDD field `Low Pressure Cut Off`

        Args:
            value (float): value for IDD Field `Low Pressure Cut Off`
                Units: Pa
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Low Pressure Cut Off"] = value




class DistrictCooling(DataObject):

    """Corresponds to IDD object `DistrictCooling` Centralized source of
    chilled water, such as a district cooling system."""
    schema = {'min-fields': 0,
              'name': u'DistrictCooling',
              'pyname': u'DistrictCooling',
              'format': None,
              'fields': OrderedDict([(u'name',
                                      {'name': u'Name',
                                       'pyname': u'name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'chilled water inlet node name',
                                      {'name': u'Chilled Water Inlet Node Name',
                                       'pyname': u'chilled_water_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'chilled water outlet node name',
                                      {'name': u'Chilled Water Outlet Node Name',
                                       'pyname': u'chilled_water_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'nominal capacity',
                                      {'name': u'Nominal Capacity',
                                       'pyname': u'nominal_capacity',
                                       'required-field': True,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'W'}),
                                     (u'capacity fraction schedule name',
                                      {'name': u'Capacity Fraction Schedule Name',
                                       'pyname': u'capacity_fraction_schedule_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'})]),
              'extensible-fields': OrderedDict(),
              'unique-object': False,
              'required-object': False,
              'group': u'Plant Heating and Cooling Equipment'}

    @property
    def name(self):
        """Get name.

        Returns:
            str: the value of `name` or None if not set

        """
        return self["Name"]

    @name.setter
    def name(self, value=None):
        """Corresponds to IDD field `Name`

        Args:
            value (str): value for IDD Field `Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Name"] = value

    @property
    def chilled_water_inlet_node_name(self):
        """Get chilled_water_inlet_node_name.

        Returns:
            str: the value of `chilled_water_inlet_node_name` or None if not set

        """
        return self["Chilled Water Inlet Node Name"]

    @chilled_water_inlet_node_name.setter
    def chilled_water_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Chilled Water Inlet Node Name`

        Args:
            value (str): value for IDD Field `Chilled Water Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Inlet Node Name"] = value

    @property
    def chilled_water_outlet_node_name(self):
        """Get chilled_water_outlet_node_name.

        Returns:
            str: the value of `chilled_water_outlet_node_name` or None if not set

        """
        return self["Chilled Water Outlet Node Name"]

    @chilled_water_outlet_node_name.setter
    def chilled_water_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Chilled Water Outlet Node Name`

        Args:
            value (str): value for IDD Field `Chilled Water Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Outlet Node Name"] = value

    @property
    def nominal_capacity(self):
        """Get nominal_capacity.

        Returns:
            float: the value of `nominal_capacity` or None if not set

        """
        return self["Nominal Capacity"]

    @nominal_capacity.setter
    def nominal_capacity(self, value=None):
        """Corresponds to IDD field `Nominal Capacity`

        Args:
            value (float): value for IDD Field `Nominal Capacity`
                Units: W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Nominal Capacity"] = value

    @property
    def capacity_fraction_schedule_name(self):
        """Get capacity_fraction_schedule_name.

        Returns:
            str: the value of `capacity_fraction_schedule_name` or None if not set

        """
        return self["Capacity Fraction Schedule Name"]

    @capacity_fraction_schedule_name.setter
    def capacity_fraction_schedule_name(self, value=None):
        """Corresponds to IDD field `Capacity Fraction Schedule Name` Schedule
        values are multiplied by Nominal Capacity for current capacity.

        Args:
            value (str): value for IDD Field `Capacity Fraction Schedule Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Capacity Fraction Schedule Name"] = value




class DistrictHeating(DataObject):

    """Corresponds to IDD object `DistrictHeating` Centralized source of hot
    water, such as a district heating system."""
    schema = {'min-fields': 0,
              'name': u'DistrictHeating',
              'pyname': u'DistrictHeating',
              'format': None,
              'fields': OrderedDict([(u'name',
                                      {'name': u'Name',
                                       'pyname': u'name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'hot water inlet node name',
                                      {'name': u'Hot Water Inlet Node Name',
                                       'pyname': u'hot_water_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'hot water outlet node name',
                                      {'name': u'Hot Water Outlet Node Name',
                                       'pyname': u'hot_water_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'nominal capacity',
                                      {'name': u'Nominal Capacity',
                                       'pyname': u'nominal_capacity',
                                       'required-field': True,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': 'real',
                                       'unit': u'W'}),
                                     (u'capacity fraction schedule name',
                                      {'name': u'Capacity Fraction Schedule Name',
                                       'pyname': u'capacity_fraction_schedule_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'})]),
              'extensible-fields': OrderedDict(),
              'unique-object': False,
              'required-object': False,
              'group': u'Plant Heating and Cooling Equipment'}

    @property
    def name(self):
        """Get name.

        Returns:
            str: the value of `name` or None if not set

        """
        return self["Name"]

    @name.setter
    def name(self, value=None):
        """Corresponds to IDD field `Name`

        Args:
            value (str): value for IDD Field `Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Name"] = value

    @property
    def hot_water_inlet_node_name(self):
        """Get hot_water_inlet_node_name.

        Returns:
            str: the value of `hot_water_inlet_node_name` or None if not set

        """
        return self["Hot Water Inlet Node Name"]

    @hot_water_inlet_node_name.setter
    def hot_water_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Hot Water Inlet Node Name`

        Args:
            value (str): value for IDD Field `Hot Water Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Hot Water Inlet Node Name"] = value

    @property
    def hot_water_outlet_node_name(self):
        """Get hot_water_outlet_node_name.

        Returns:
            str: the value of `hot_water_outlet_node_name` or None if not set

        """
        return self["Hot Water Outlet Node Name"]

    @hot_water_outlet_node_name.setter
    def hot_water_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Hot Water Outlet Node Name`

        Args:
            value (str): value for IDD Field `Hot Water Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Hot Water Outlet Node Name"] = value

    @property
    def nominal_capacity(self):
        """Get nominal_capacity.

        Returns:
            float: the value of `nominal_capacity` or None if not set

        """
        return self["Nominal Capacity"]

    @nominal_capacity.setter
    def nominal_capacity(self, value=None):
        """Corresponds to IDD field `Nominal Capacity`

        Args:
            value (float): value for IDD Field `Nominal Capacity`
                Units: W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Nominal Capacity"] = value

    @property
    def capacity_fraction_schedule_name(self):
        """Get capacity_fraction_schedule_name.

        Returns:
            str: the value of `capacity_fraction_schedule_name` or None if not set

        """
        return self["Capacity Fraction Schedule Name"]

    @capacity_fraction_schedule_name.setter
    def capacity_fraction_schedule_name(self, value=None):
        """Corresponds to IDD field `Capacity Fraction Schedule Name` Schedule
        values are multiplied by Nominal Capacity for current capacity.

        Args:
            value (str): value for IDD Field `Capacity Fraction Schedule Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Capacity Fraction Schedule Name"] = value




class PlantComponentTemperatureSource(DataObject):

    """ Corresponds to IDD object `PlantComponent:TemperatureSource`
        Simulates an object of pre-determined (constant or scheduled) source temperature
        The object introduces fluid into the plant loop at the specified temperature and
        at the same flow rate as the fluid enters the component
        Fluid entering the component vanishes equivalent to the relief air in an air system
    """
    schema = {'min-fields': 0,
              'name': u'PlantComponent:TemperatureSource',
              'pyname': u'PlantComponentTemperatureSource',
              'format': None,
              'fields': OrderedDict([(u'name',
                                      {'name': u'Name',
                                       'pyname': u'name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'alpha'}),
                                     (u'inlet node',
                                      {'name': u'Inlet Node',
                                       'pyname': u'inlet_node',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'outlet node',
                                      {'name': u'Outlet Node',
                                       'pyname': u'outlet_node',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'node'}),
                                     (u'design volume flow rate',
                                      {'name': u'Design Volume Flow Rate',
                                       'pyname': u'design_volume_flow_rate',
                                       'minimum>': 0.0,
                                       'required-field': True,
                                       'autosizable': True,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'm3/s'}),
                                     (u'temperature specification type',
                                      {'name': u'Temperature Specification Type',
                                       'pyname': u'temperature_specification_type',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'Constant',
                                                           u'Scheduled'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'source temperature',
                                      {'name': u'Source Temperature',
                                       'pyname': u'source_temperature',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'C'}),
                                     (u'source temperature schedule name',
                                      {'name': u'Source Temperature Schedule Name',
                                       'pyname': u'source_temperature_schedule_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list',
                                       'unit': u'C'})]),
              'extensible-fields': OrderedDict(),
              'unique-object': False,
              'required-object': False,
              'group': u'Plant Heating and Cooling Equipment'}

    @property
    def name(self):
        """Get name.

        Returns:
            str: the value of `name` or None if not set

        """
        return self["Name"]

    @name.setter
    def name(self, value=None):
        """Corresponds to IDD field `Name` Component Name.

        Args:
            value (str): value for IDD Field `Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Name"] = value

    @property
    def inlet_node(self):
        """Get inlet_node.

        Returns:
            str: the value of `inlet_node` or None if not set

        """
        return self["Inlet Node"]

    @inlet_node.setter
    def inlet_node(self, value=None):
        """Corresponds to IDD field `Inlet Node` Name of the source inlet node.

        Args:
            value (str): value for IDD Field `Inlet Node`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Inlet Node"] = value

    @property
    def outlet_node(self):
        """Get outlet_node.

        Returns:
            str: the value of `outlet_node` or None if not set

        """
        return self["Outlet Node"]

    @outlet_node.setter
    def outlet_node(self, value=None):
        """Corresponds to IDD field `Outlet Node` Name of the source outlet
        node.

        Args:
            value (str): value for IDD Field `Outlet Node`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Outlet Node"] = value

    @property
    def design_volume_flow_rate(self):
        """Get design_volume_flow_rate.

        Returns:
            float: the value of `design_volume_flow_rate` or None if not set

        """
        return self["Design Volume Flow Rate"]

    @design_volume_flow_rate.setter
    def design_volume_flow_rate(self, value=None):
        """Corresponds to IDD field `Design Volume Flow Rate` The design
        volumetric flow rate for this source.

        Args:
            value (float or "Autosize"): value for IDD Field `Design Volume Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Volume Flow Rate"] = value

    @property
    def temperature_specification_type(self):
        """Get temperature_specification_type.

        Returns:
            str: the value of `temperature_specification_type` or None if not set

        """
        return self["Temperature Specification Type"]

    @temperature_specification_type.setter
    def temperature_specification_type(self, value=None):
        """Corresponds to IDD field `Temperature Specification Type`

        Args:
            value (str): value for IDD Field `Temperature Specification Type`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Temperature Specification Type"] = value

    @property
    def source_temperature(self):
        """Get source_temperature.

        Returns:
            float: the value of `source_temperature` or None if not set

        """
        return self["Source Temperature"]

    @source_temperature.setter
    def source_temperature(self, value=None):
        """  Corresponds to IDD field `Source Temperature`
        Used if Temperature Specification Type = Constant

        Args:
            value (float): value for IDD Field `Source Temperature`
                Units: C
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Source Temperature"] = value

    @property
    def source_temperature_schedule_name(self):
        """Get source_temperature_schedule_name.

        Returns:
            str: the value of `source_temperature_schedule_name` or None if not set

        """
        return self["Source Temperature Schedule Name"]

    @source_temperature_schedule_name.setter
    def source_temperature_schedule_name(self, value=None):
        """  Corresponds to IDD field `Source Temperature Schedule Name`
        Used if Temperature Specification Type = Scheduled

        Args:
            value (str): value for IDD Field `Source Temperature Schedule Name`
                Units: C
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Source Temperature Schedule Name"] = value




class CentralHeatPumpSystem(DataObject):

    """Corresponds to IDD object `CentralHeatPumpSystem` This chiller bank can
    contain multiple chiller heaters and heat pump performance objects.

    Its function is to encapsulate the extra controls needed to turn individual modules on/off
    and whether they are to operate in cooling-only, heating-only or simultaneous cooling/heating
    mode and whether to connect the source water to the evaporator or condenser side.

    """
    schema = {'min-fields': 14,
              'name': u'CentralHeatPumpSystem',
              'pyname': u'CentralHeatPumpSystem',
              'format': None,
              'fields': OrderedDict([(u'name',
                                      {'name': u'Name',
                                       'pyname': u'name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'alpha'}),
                                     (u'control method',
                                      {'name': u'Control Method',
                                       'pyname': u'control_method',
                                       'default': u'SmartMixing',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'SmartMixing'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'cooling loop inlet node name',
                                      {'name': u'Cooling Loop Inlet Node Name',
                                       'pyname': u'cooling_loop_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'alpha'}),
                                     (u'cooling loop outlet node name',
                                      {'name': u'Cooling Loop Outlet Node Name',
                                       'pyname': u'cooling_loop_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'alpha'}),
                                     (u'source loop inlet node name',
                                      {'name': u'Source Loop Inlet Node Name',
                                       'pyname': u'source_loop_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'alpha'}),
                                     (u'source loop outlet node name',
                                      {'name': u'Source Loop Outlet Node Name',
                                       'pyname': u'source_loop_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'alpha'}),
                                     (u'heating loop inlet node name',
                                      {'name': u'Heating Loop Inlet Node Name',
                                       'pyname': u'heating_loop_inlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'alpha'}),
                                     (u'heating loop outlet node name',
                                      {'name': u'Heating Loop Outlet Node Name',
                                       'pyname': u'heating_loop_outlet_node_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'alpha'}),
                                     (u'ancillary power',
                                      {'name': u'Ancillary Power',
                                       'pyname': u'ancillary_power',
                                       'default': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'W'}),
                                     (u'ancillary operation schedule name',
                                      {'name': u'Ancillary Operation Schedule Name',
                                       'pyname': u'ancillary_operation_schedule_name',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'chiller heater modules performance component object type 1',
                                      {'name': u'Chiller Heater Modules Performance Component Object Type 1',
                                       'pyname': u'chiller_heater_modules_performance_component_object_type_1',
                                       'required-field': True,
                                       'autosizable': False,
                                       'accepted-values': [u'ChillerHeaterPerformance:Electric:EIR'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'chiller heater modules performance component name 1',
                                      {'name': u'Chiller Heater Modules Performance Component Name 1',
                                       'pyname': u'chiller_heater_modules_performance_component_name_1',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'chiller heater modules control schedule name 1',
                                      {'name': u'Chiller Heater Modules Control Schedule Name 1',
                                       'pyname': u'chiller_heater_modules_control_schedule_name_1',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'number of chiller heater modules 1',
                                      {'name': u'Number of Chiller Heater Modules 1',
                                       'pyname': u'number_of_chiller_heater_modules_1',
                                       'default': 1,
                                       'required-field': True,
                                       'autosizable': False,
                                       'minimum': 1,
                                       'autocalculatable': False,
                                       'type': u'integer'}),
                                     (u'chiller heater modules performance component object type 2',
                                      {'name': u'Chiller Heater Modules Performance Component Object Type 2',
                                       'pyname': u'chiller_heater_modules_performance_component_object_type_2',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ChillerHeaterPerformance:Electric:EIR'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'chiller heater modules performance component name 2',
                                      {'name': u'Chiller Heater Modules Performance Component Name 2',
                                       'pyname': u'chiller_heater_modules_performance_component_name_2',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'chiller heater modules control schedule name 2',
                                      {'name': u'Chiller Heater Modules Control Schedule Name 2',
                                       'pyname': u'chiller_heater_modules_control_schedule_name_2',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'number of chiller heater modules 2',
                                      {'name': u'Number of Chiller Heater Modules 2',
                                       'pyname': u'number_of_chiller_heater_modules_2',
                                       'default': 1,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 1,
                                       'autocalculatable': False,
                                       'type': u'integer'}),
                                     (u'chiller heater performance component object type 3',
                                      {'name': u'Chiller Heater Performance Component Object Type 3',
                                       'pyname': u'chiller_heater_performance_component_object_type_3',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ChillerHeaterPerformance:Electric:EIR'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'chiller heater performance component name 3',
                                      {'name': u'Chiller Heater Performance Component Name 3',
                                       'pyname': u'chiller_heater_performance_component_name_3',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'chiller heater modules control schedule name 3',
                                      {'name': u'Chiller Heater Modules Control Schedule Name 3',
                                       'pyname': u'chiller_heater_modules_control_schedule_name_3',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'number of chiller heater modules 3',
                                      {'name': u'Number of Chiller Heater Modules 3',
                                       'pyname': u'number_of_chiller_heater_modules_3',
                                       'default': 1,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 1,
                                       'autocalculatable': False,
                                       'type': u'integer'}),
                                     (u'chiller heater modules performance component object type 4',
                                      {'name': u'Chiller Heater Modules Performance Component Object Type 4',
                                       'pyname': u'chiller_heater_modules_performance_component_object_type_4',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ChillerHeaterPerformance:Electric:EIR'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'chiller heater modules performance component name 4',
                                      {'name': u'Chiller Heater Modules Performance Component Name 4',
                                       'pyname': u'chiller_heater_modules_performance_component_name_4',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'chiller heater modules control schedule name 4',
                                      {'name': u'Chiller Heater Modules Control Schedule Name 4',
                                       'pyname': u'chiller_heater_modules_control_schedule_name_4',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'number of chiller heater modules 4',
                                      {'name': u'Number of Chiller Heater Modules 4',
                                       'pyname': u'number_of_chiller_heater_modules_4',
                                       'default': 1,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 1,
                                       'autocalculatable': False,
                                       'type': u'integer'}),
                                     (u'chiller heater modules performance component object type 5',
                                      {'name': u'Chiller Heater Modules Performance Component Object Type 5',
                                       'pyname': u'chiller_heater_modules_performance_component_object_type_5',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ChillerHeaterPerformance:Electric:EIR'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'chiller heater models performance component name 5',
                                      {'name': u'Chiller Heater Models Performance Component Name 5',
                                       'pyname': u'chiller_heater_models_performance_component_name_5',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'chiller heater modules control schedule name 5',
                                      {'name': u'Chiller Heater Modules Control Schedule Name 5',
                                       'pyname': u'chiller_heater_modules_control_schedule_name_5',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'number of chiller heater modules 5',
                                      {'name': u'Number of Chiller Heater Modules 5',
                                       'pyname': u'number_of_chiller_heater_modules_5',
                                       'default': 1,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 1,
                                       'autocalculatable': False,
                                       'type': u'integer'}),
                                     (u'chiller heater modules performance component object type 6',
                                      {'name': u'Chiller Heater Modules Performance Component Object Type 6',
                                       'pyname': u'chiller_heater_modules_performance_component_object_type_6',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ChillerHeaterPerformance:Electric:EIR'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'chiller heater modules performance component name 6',
                                      {'name': u'Chiller Heater Modules Performance Component Name 6',
                                       'pyname': u'chiller_heater_modules_performance_component_name_6',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'chiller heater modules control schedule name 6',
                                      {'name': u'Chiller Heater Modules Control Schedule Name 6',
                                       'pyname': u'chiller_heater_modules_control_schedule_name_6',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'number of chiller heater modules 6',
                                      {'name': u'Number of Chiller Heater Modules 6',
                                       'pyname': u'number_of_chiller_heater_modules_6',
                                       'default': 1,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 1,
                                       'autocalculatable': False,
                                       'type': u'integer'}),
                                     (u'chiller heater modules performance component object type 7',
                                      {'name': u'Chiller Heater Modules Performance Component Object Type 7',
                                       'pyname': u'chiller_heater_modules_performance_component_object_type_7',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ChillerHeaterPerformance:Electric:EIR'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'chiller heater modules performance component name 7',
                                      {'name': u'Chiller Heater Modules Performance Component Name 7',
                                       'pyname': u'chiller_heater_modules_performance_component_name_7',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'chiller heater modules control schedule name 7',
                                      {'name': u'Chiller Heater Modules Control Schedule Name 7',
                                       'pyname': u'chiller_heater_modules_control_schedule_name_7',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'number of chiller heater modules 7',
                                      {'name': u'Number of Chiller Heater Modules 7',
                                       'pyname': u'number_of_chiller_heater_modules_7',
                                       'default': 1,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 1,
                                       'autocalculatable': False,
                                       'type': u'integer'}),
                                     (u'chiller heater modules performance component object type 8',
                                      {'name': u'Chiller Heater Modules Performance Component Object Type 8',
                                       'pyname': u'chiller_heater_modules_performance_component_object_type_8',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ChillerHeaterPerformance:Electric:EIR'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'chiller heater modules performance component name 8',
                                      {'name': u'Chiller Heater Modules Performance Component Name 8',
                                       'pyname': u'chiller_heater_modules_performance_component_name_8',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'chiller heater modules control schedule name 8',
                                      {'name': u'Chiller Heater Modules Control Schedule Name 8',
                                       'pyname': u'chiller_heater_modules_control_schedule_name_8',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'number of chiller heater modules 8',
                                      {'name': u'Number of Chiller Heater Modules 8',
                                       'pyname': u'number_of_chiller_heater_modules_8',
                                       'default': 1,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 1,
                                       'autocalculatable': False,
                                       'type': u'integer'}),
                                     (u'chiller heater modules performance component object type 9',
                                      {'name': u'Chiller Heater Modules Performance Component Object Type 9',
                                       'pyname': u'chiller_heater_modules_performance_component_object_type_9',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ChillerHeaterPerformance:Electric:EIR'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'chiller heater modules performance component name 9',
                                      {'name': u'Chiller Heater Modules Performance Component Name 9',
                                       'pyname': u'chiller_heater_modules_performance_component_name_9',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'chiller heater modules control schedule name 9',
                                      {'name': u'Chiller Heater Modules Control Schedule Name 9',
                                       'pyname': u'chiller_heater_modules_control_schedule_name_9',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'number of chiller heater modules 9',
                                      {'name': u'Number of Chiller Heater Modules 9',
                                       'pyname': u'number_of_chiller_heater_modules_9',
                                       'default': 1,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 1,
                                       'autocalculatable': False,
                                       'type': u'integer'}),
                                     (u'chiller heater modules performance component object type 10',
                                      {'name': u'Chiller Heater Modules Performance Component Object Type 10',
                                       'pyname': u'chiller_heater_modules_performance_component_object_type_10',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ChillerHeaterPerformance:Electric:EIR'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'chiller heater modules performance component name 10',
                                      {'name': u'Chiller Heater Modules Performance Component Name 10',
                                       'pyname': u'chiller_heater_modules_performance_component_name_10',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'chiller heater modules control schedule name 10',
                                      {'name': u'Chiller Heater Modules Control Schedule Name 10',
                                       'pyname': u'chiller_heater_modules_control_schedule_name_10',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'number of chiller heater modules 10',
                                      {'name': u'Number of Chiller Heater Modules 10',
                                       'pyname': u'number_of_chiller_heater_modules_10',
                                       'default': 1,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 1,
                                       'autocalculatable': False,
                                       'type': u'integer'}),
                                     (u'chiller heater modules performance component object type 11',
                                      {'name': u'Chiller Heater Modules Performance Component Object Type 11',
                                       'pyname': u'chiller_heater_modules_performance_component_object_type_11',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ChillerHeaterPerformance:Electric:EIR'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'chiller heater modules performance component name 11',
                                      {'name': u'Chiller Heater Modules Performance Component Name 11',
                                       'pyname': u'chiller_heater_modules_performance_component_name_11',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'chiller heater module control schedule name 11',
                                      {'name': u'Chiller Heater Module Control Schedule Name 11',
                                       'pyname': u'chiller_heater_module_control_schedule_name_11',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'number of chiller heater modules 11',
                                      {'name': u'Number of Chiller Heater Modules 11',
                                       'pyname': u'number_of_chiller_heater_modules_11',
                                       'default': 1,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 1,
                                       'autocalculatable': False,
                                       'type': u'integer'}),
                                     (u'chiller heater modules performance component object type 12',
                                      {'name': u'Chiller Heater Modules Performance Component Object Type 12',
                                       'pyname': u'chiller_heater_modules_performance_component_object_type_12',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ChillerHeaterPerformance:Electric:EIR'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'chiller heater modules performance component name 12',
                                      {'name': u'Chiller Heater Modules Performance Component Name 12',
                                       'pyname': u'chiller_heater_modules_performance_component_name_12',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'chiller heater modules control schedule name 12',
                                      {'name': u'Chiller Heater Modules Control Schedule Name 12',
                                       'pyname': u'chiller_heater_modules_control_schedule_name_12',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'number of chiller heater modules 12',
                                      {'name': u'Number of Chiller Heater Modules 12',
                                       'pyname': u'number_of_chiller_heater_modules_12',
                                       'default': 1,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 1,
                                       'autocalculatable': False,
                                       'type': u'integer'}),
                                     (u'chiller heater modules performance component object type 13',
                                      {'name': u'Chiller Heater Modules Performance Component Object Type 13',
                                       'pyname': u'chiller_heater_modules_performance_component_object_type_13',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ChillerHeaterPerformance:Electric:EIR'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'chiller heater modules performance component name 13',
                                      {'name': u'Chiller Heater Modules Performance Component Name 13',
                                       'pyname': u'chiller_heater_modules_performance_component_name_13',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'chiller heater modules control schedule name 13',
                                      {'name': u'Chiller Heater Modules Control Schedule Name 13',
                                       'pyname': u'chiller_heater_modules_control_schedule_name_13',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'number of chiller heater modules 13',
                                      {'name': u'Number of Chiller Heater Modules 13',
                                       'pyname': u'number_of_chiller_heater_modules_13',
                                       'default': 1,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 1,
                                       'autocalculatable': False,
                                       'type': u'integer'}),
                                     (u'chiller heater modules performance component object type 14',
                                      {'name': u'Chiller Heater Modules Performance Component Object Type 14',
                                       'pyname': u'chiller_heater_modules_performance_component_object_type_14',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ChillerHeaterPerformance:Electric:EIR'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'chiller heater modules performance component name 14',
                                      {'name': u'Chiller Heater Modules Performance Component Name 14',
                                       'pyname': u'chiller_heater_modules_performance_component_name_14',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'chiller heater modules control schedule name 14',
                                      {'name': u'Chiller Heater Modules Control Schedule Name 14',
                                       'pyname': u'chiller_heater_modules_control_schedule_name_14',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'number of chiller heater modules 14',
                                      {'name': u'Number of Chiller Heater Modules 14',
                                       'pyname': u'number_of_chiller_heater_modules_14',
                                       'default': 1,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 1,
                                       'autocalculatable': False,
                                       'type': u'integer'}),
                                     (u'chiller heater modules performance component object type 15',
                                      {'name': u'Chiller Heater Modules Performance Component Object Type 15',
                                       'pyname': u'chiller_heater_modules_performance_component_object_type_15',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ChillerHeaterPerformance:Electric:EIR'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'chiller heater modules performance component name 15',
                                      {'name': u'Chiller Heater Modules Performance Component Name 15',
                                       'pyname': u'chiller_heater_modules_performance_component_name_15',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'chiller heater modules control schedule name 15',
                                      {'name': u'Chiller Heater Modules Control Schedule Name 15',
                                       'pyname': u'chiller_heater_modules_control_schedule_name_15',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'number of chiller heater modules 15',
                                      {'name': u'Number of Chiller Heater Modules 15',
                                       'pyname': u'number_of_chiller_heater_modules_15',
                                       'default': 1,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 1,
                                       'autocalculatable': False,
                                       'type': u'integer'}),
                                     (u'chiller heater modules performance component object type 16',
                                      {'name': u'Chiller Heater Modules Performance Component Object Type 16',
                                       'pyname': u'chiller_heater_modules_performance_component_object_type_16',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ChillerHeaterPerformance:Electric:EIR'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'chiller heater modules performance component name 16',
                                      {'name': u'Chiller Heater Modules Performance Component Name 16',
                                       'pyname': u'chiller_heater_modules_performance_component_name_16',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'chiller heater modules control schedule name 16',
                                      {'name': u'Chiller Heater Modules Control Schedule Name 16',
                                       'pyname': u'chiller_heater_modules_control_schedule_name_16',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'number of chiller heater modules 16',
                                      {'name': u'Number of Chiller Heater Modules 16',
                                       'pyname': u'number_of_chiller_heater_modules_16',
                                       'default': 1,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 1,
                                       'autocalculatable': False,
                                       'type': u'integer'}),
                                     (u'chiller heater modules performance component object type 17',
                                      {'name': u'Chiller Heater Modules Performance Component Object Type 17',
                                       'pyname': u'chiller_heater_modules_performance_component_object_type_17',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ChillerHeaterPerformance:Electric:EIR'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'chiller heater modules performance component name 17',
                                      {'name': u'Chiller Heater Modules Performance Component Name 17',
                                       'pyname': u'chiller_heater_modules_performance_component_name_17',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'chiller heater modules control schedule name 17',
                                      {'name': u'Chiller Heater Modules Control Schedule Name 17',
                                       'pyname': u'chiller_heater_modules_control_schedule_name_17',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'number of chiller heater modules 17',
                                      {'name': u'Number of Chiller Heater Modules 17',
                                       'pyname': u'number_of_chiller_heater_modules_17',
                                       'default': 1,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 1,
                                       'autocalculatable': False,
                                       'type': u'integer'}),
                                     (u'chiller heater modules performance component object type 18',
                                      {'name': u'Chiller Heater Modules Performance Component Object Type 18',
                                       'pyname': u'chiller_heater_modules_performance_component_object_type_18',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ChillerHeaterPerformance:Electric:EIR'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'chiller heater modules performance component name 18',
                                      {'name': u'Chiller Heater Modules Performance Component Name 18',
                                       'pyname': u'chiller_heater_modules_performance_component_name_18',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'chiller heater modules control control schedule name 18',
                                      {'name': u'Chiller Heater Modules Control Control Schedule Name 18',
                                       'pyname': u'chiller_heater_modules_control_control_schedule_name_18',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'number of chiller heater modules 18',
                                      {'name': u'Number of Chiller Heater Modules 18',
                                       'pyname': u'number_of_chiller_heater_modules_18',
                                       'default': 1,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 1,
                                       'autocalculatable': False,
                                       'type': u'integer'}),
                                     (u'chiller heater modules performance component object type 19',
                                      {'name': u'Chiller Heater Modules Performance Component Object Type 19',
                                       'pyname': u'chiller_heater_modules_performance_component_object_type_19',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ChillerHeaterPerformance:Electric:EIR'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'chiller heater modules performance component name 19',
                                      {'name': u'Chiller Heater Modules Performance Component Name 19',
                                       'pyname': u'chiller_heater_modules_performance_component_name_19',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'chiller heater modules control schedule name 19',
                                      {'name': u'Chiller Heater Modules Control Schedule Name 19',
                                       'pyname': u'chiller_heater_modules_control_schedule_name_19',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'number of chiller heater modules 19',
                                      {'name': u'Number of Chiller Heater Modules 19',
                                       'pyname': u'number_of_chiller_heater_modules_19',
                                       'default': 1,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 1,
                                       'autocalculatable': False,
                                       'type': u'integer'}),
                                     (u'chiller heater modules performance component object type 20',
                                      {'name': u'Chiller Heater Modules Performance Component Object Type 20',
                                       'pyname': u'chiller_heater_modules_performance_component_object_type_20',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ChillerHeaterPerformance:Electric:EIR'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'chiller heater modules performance component name 20',
                                      {'name': u'Chiller Heater Modules Performance Component Name 20',
                                       'pyname': u'chiller_heater_modules_performance_component_name_20',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'chiller heater modules control schedule name 20',
                                      {'name': u'Chiller Heater Modules Control Schedule Name 20',
                                       'pyname': u'chiller_heater_modules_control_schedule_name_20',
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'number of chiller heater modules 20',
                                      {'name': u'Number of Chiller Heater Modules 20',
                                       'pyname': u'number_of_chiller_heater_modules_20',
                                       'default': 1,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 1,
                                       'autocalculatable': False,
                                       'type': u'integer'})]),
              'extensible-fields': OrderedDict(),
              'unique-object': False,
              'required-object': False,
              'group': u'Plant Heating and Cooling Equipment'}

    @property
    def name(self):
        """Get name.

        Returns:
            str: the value of `name` or None if not set

        """
        return self["Name"]

    @name.setter
    def name(self, value=None):
        """Corresponds to IDD field `Name`

        Args:
            value (str): value for IDD Field `Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Name"] = value

    @property
    def control_method(self):
        """Get control_method.

        Returns:
            str: the value of `control_method` or None if not set

        """
        return self["Control Method"]

    @control_method.setter
    def control_method(self, value="SmartMixing"):
        """Corresponds to IDD field `Control Method`

        Args:
            value (str): value for IDD Field `Control Method`
                Default value: SmartMixing
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Control Method"] = value

    @property
    def cooling_loop_inlet_node_name(self):
        """Get cooling_loop_inlet_node_name.

        Returns:
            str: the value of `cooling_loop_inlet_node_name` or None if not set

        """
        return self["Cooling Loop Inlet Node Name"]

    @cooling_loop_inlet_node_name.setter
    def cooling_loop_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Cooling Loop Inlet Node Name`

        Args:
            value (str): value for IDD Field `Cooling Loop Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Cooling Loop Inlet Node Name"] = value

    @property
    def cooling_loop_outlet_node_name(self):
        """Get cooling_loop_outlet_node_name.

        Returns:
            str: the value of `cooling_loop_outlet_node_name` or None if not set

        """
        return self["Cooling Loop Outlet Node Name"]

    @cooling_loop_outlet_node_name.setter
    def cooling_loop_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Cooling Loop Outlet Node Name`

        Args:
            value (str): value for IDD Field `Cooling Loop Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Cooling Loop Outlet Node Name"] = value

    @property
    def source_loop_inlet_node_name(self):
        """Get source_loop_inlet_node_name.

        Returns:
            str: the value of `source_loop_inlet_node_name` or None if not set

        """
        return self["Source Loop Inlet Node Name"]

    @source_loop_inlet_node_name.setter
    def source_loop_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Source Loop Inlet Node Name`

        Args:
            value (str): value for IDD Field `Source Loop Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Source Loop Inlet Node Name"] = value

    @property
    def source_loop_outlet_node_name(self):
        """Get source_loop_outlet_node_name.

        Returns:
            str: the value of `source_loop_outlet_node_name` or None if not set

        """
        return self["Source Loop Outlet Node Name"]

    @source_loop_outlet_node_name.setter
    def source_loop_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Source Loop Outlet Node Name`

        Args:
            value (str): value for IDD Field `Source Loop Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Source Loop Outlet Node Name"] = value

    @property
    def heating_loop_inlet_node_name(self):
        """Get heating_loop_inlet_node_name.

        Returns:
            str: the value of `heating_loop_inlet_node_name` or None if not set

        """
        return self["Heating Loop Inlet Node Name"]

    @heating_loop_inlet_node_name.setter
    def heating_loop_inlet_node_name(self, value=None):
        """Corresponds to IDD field `Heating Loop Inlet Node Name`

        Args:
            value (str): value for IDD Field `Heating Loop Inlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Heating Loop Inlet Node Name"] = value

    @property
    def heating_loop_outlet_node_name(self):
        """Get heating_loop_outlet_node_name.

        Returns:
            str: the value of `heating_loop_outlet_node_name` or None if not set

        """
        return self["Heating Loop Outlet Node Name"]

    @heating_loop_outlet_node_name.setter
    def heating_loop_outlet_node_name(self, value=None):
        """Corresponds to IDD field `Heating Loop Outlet Node Name`

        Args:
            value (str): value for IDD Field `Heating Loop Outlet Node Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Heating Loop Outlet Node Name"] = value

    @property
    def ancillary_power(self):
        """Get ancillary_power.

        Returns:
            float: the value of `ancillary_power` or None if not set

        """
        return self["Ancillary Power"]

    @ancillary_power.setter
    def ancillary_power(self, value=None):
        """Corresponds to IDD field `Ancillary Power` Power as demanded from
        any auxiliary controls.

        Args:
            value (float): value for IDD Field `Ancillary Power`
                Units: W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Ancillary Power"] = value

    @property
    def ancillary_operation_schedule_name(self):
        """Get ancillary_operation_schedule_name.

        Returns:
            str: the value of `ancillary_operation_schedule_name` or None if not set

        """
        return self["Ancillary Operation Schedule Name"]

    @ancillary_operation_schedule_name.setter
    def ancillary_operation_schedule_name(self, value=None):
        """Corresponds to IDD field `Ancillary Operation Schedule Name` This
        value from this schedule is multiplied times the Ancilliary Power.

        Args:
            value (str): value for IDD Field `Ancillary Operation Schedule Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Ancillary Operation Schedule Name"] = value

    @property
    def chiller_heater_modules_performance_component_object_type_1(self):
        """Get chiller_heater_modules_performance_component_object_type_1.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_object_type_1` or None if not set

        """
        return self[
            "Chiller Heater Modules Performance Component Object Type 1"]

    @chiller_heater_modules_performance_component_object_type_1.setter
    def chiller_heater_modules_performance_component_object_type_1(
            self,
            value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Object Type 1`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Object Type 1`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Chiller Heater Modules Performance Component Object Type 1"] = value

    @property
    def chiller_heater_modules_performance_component_name_1(self):
        """Get chiller_heater_modules_performance_component_name_1.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_name_1` or None if not set

        """
        return self["Chiller Heater Modules Performance Component Name 1"]

    @chiller_heater_modules_performance_component_name_1.setter
    def chiller_heater_modules_performance_component_name_1(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Name 1`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Name 1`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Performance Component Name 1"] = value

    @property
    def chiller_heater_modules_control_schedule_name_1(self):
        """Get chiller_heater_modules_control_schedule_name_1.

        Returns:
            str: the value of `chiller_heater_modules_control_schedule_name_1` or None if not set

        """
        return self["Chiller Heater Modules Control Schedule Name 1"]

    @chiller_heater_modules_control_schedule_name_1.setter
    def chiller_heater_modules_control_schedule_name_1(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Control Schedule
        Name 1`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Control Schedule Name 1`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Control Schedule Name 1"] = value

    @property
    def number_of_chiller_heater_modules_1(self):
        """Get number_of_chiller_heater_modules_1.

        Returns:
            int: the value of `number_of_chiller_heater_modules_1` or None if not set

        """
        return self["Number of Chiller Heater Modules 1"]

    @number_of_chiller_heater_modules_1.setter
    def number_of_chiller_heater_modules_1(self, value=1):
        """Corresponds to IDD field `Number of Chiller Heater Modules 1`

        Args:
            value (int): value for IDD Field `Number of Chiller Heater Modules 1`
                Default value: 1
                value >= 1
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Number of Chiller Heater Modules 1"] = value

    @property
    def chiller_heater_modules_performance_component_object_type_2(self):
        """Get chiller_heater_modules_performance_component_object_type_2.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_object_type_2` or None if not set

        """
        return self[
            "Chiller Heater Modules Performance Component Object Type 2"]

    @chiller_heater_modules_performance_component_object_type_2.setter
    def chiller_heater_modules_performance_component_object_type_2(
            self,
            value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Object Type 2`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Object Type 2`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Chiller Heater Modules Performance Component Object Type 2"] = value

    @property
    def chiller_heater_modules_performance_component_name_2(self):
        """Get chiller_heater_modules_performance_component_name_2.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_name_2` or None if not set

        """
        return self["Chiller Heater Modules Performance Component Name 2"]

    @chiller_heater_modules_performance_component_name_2.setter
    def chiller_heater_modules_performance_component_name_2(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Name 2`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Name 2`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Performance Component Name 2"] = value

    @property
    def chiller_heater_modules_control_schedule_name_2(self):
        """Get chiller_heater_modules_control_schedule_name_2.

        Returns:
            str: the value of `chiller_heater_modules_control_schedule_name_2` or None if not set

        """
        return self["Chiller Heater Modules Control Schedule Name 2"]

    @chiller_heater_modules_control_schedule_name_2.setter
    def chiller_heater_modules_control_schedule_name_2(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Control Schedule
        Name 2`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Control Schedule Name 2`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Control Schedule Name 2"] = value

    @property
    def number_of_chiller_heater_modules_2(self):
        """Get number_of_chiller_heater_modules_2.

        Returns:
            int: the value of `number_of_chiller_heater_modules_2` or None if not set

        """
        return self["Number of Chiller Heater Modules 2"]

    @number_of_chiller_heater_modules_2.setter
    def number_of_chiller_heater_modules_2(self, value=1):
        """Corresponds to IDD field `Number of Chiller Heater Modules 2`

        Args:
            value (int): value for IDD Field `Number of Chiller Heater Modules 2`
                Default value: 1
                value >= 1
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Number of Chiller Heater Modules 2"] = value

    @property
    def chiller_heater_performance_component_object_type_3(self):
        """Get chiller_heater_performance_component_object_type_3.

        Returns:
            str: the value of `chiller_heater_performance_component_object_type_3` or None if not set

        """
        return self["Chiller Heater Performance Component Object Type 3"]

    @chiller_heater_performance_component_object_type_3.setter
    def chiller_heater_performance_component_object_type_3(self, value=None):
        """Corresponds to IDD field `Chiller Heater Performance Component
        Object Type 3`

        Args:
            value (str): value for IDD Field `Chiller Heater Performance Component Object Type 3`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Performance Component Object Type 3"] = value

    @property
    def chiller_heater_performance_component_name_3(self):
        """Get chiller_heater_performance_component_name_3.

        Returns:
            str: the value of `chiller_heater_performance_component_name_3` or None if not set

        """
        return self["Chiller Heater Performance Component Name 3"]

    @chiller_heater_performance_component_name_3.setter
    def chiller_heater_performance_component_name_3(self, value=None):
        """Corresponds to IDD field `Chiller Heater Performance Component Name
        3`

        Args:
            value (str): value for IDD Field `Chiller Heater Performance Component Name 3`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Performance Component Name 3"] = value

    @property
    def chiller_heater_modules_control_schedule_name_3(self):
        """Get chiller_heater_modules_control_schedule_name_3.

        Returns:
            str: the value of `chiller_heater_modules_control_schedule_name_3` or None if not set

        """
        return self["Chiller Heater Modules Control Schedule Name 3"]

    @chiller_heater_modules_control_schedule_name_3.setter
    def chiller_heater_modules_control_schedule_name_3(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Control Schedule
        Name 3`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Control Schedule Name 3`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Control Schedule Name 3"] = value

    @property
    def number_of_chiller_heater_modules_3(self):
        """Get number_of_chiller_heater_modules_3.

        Returns:
            int: the value of `number_of_chiller_heater_modules_3` or None if not set

        """
        return self["Number of Chiller Heater Modules 3"]

    @number_of_chiller_heater_modules_3.setter
    def number_of_chiller_heater_modules_3(self, value=1):
        """Corresponds to IDD field `Number of Chiller Heater Modules 3`

        Args:
            value (int): value for IDD Field `Number of Chiller Heater Modules 3`
                Default value: 1
                value >= 1
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Number of Chiller Heater Modules 3"] = value

    @property
    def chiller_heater_modules_performance_component_object_type_4(self):
        """Get chiller_heater_modules_performance_component_object_type_4.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_object_type_4` or None if not set

        """
        return self[
            "Chiller Heater Modules Performance Component Object Type 4"]

    @chiller_heater_modules_performance_component_object_type_4.setter
    def chiller_heater_modules_performance_component_object_type_4(
            self,
            value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Object Type 4`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Object Type 4`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Chiller Heater Modules Performance Component Object Type 4"] = value

    @property
    def chiller_heater_modules_performance_component_name_4(self):
        """Get chiller_heater_modules_performance_component_name_4.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_name_4` or None if not set

        """
        return self["Chiller Heater Modules Performance Component Name 4"]

    @chiller_heater_modules_performance_component_name_4.setter
    def chiller_heater_modules_performance_component_name_4(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Name 4`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Name 4`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Performance Component Name 4"] = value

    @property
    def chiller_heater_modules_control_schedule_name_4(self):
        """Get chiller_heater_modules_control_schedule_name_4.

        Returns:
            str: the value of `chiller_heater_modules_control_schedule_name_4` or None if not set

        """
        return self["Chiller Heater Modules Control Schedule Name 4"]

    @chiller_heater_modules_control_schedule_name_4.setter
    def chiller_heater_modules_control_schedule_name_4(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Control Schedule
        Name 4`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Control Schedule Name 4`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Control Schedule Name 4"] = value

    @property
    def number_of_chiller_heater_modules_4(self):
        """Get number_of_chiller_heater_modules_4.

        Returns:
            int: the value of `number_of_chiller_heater_modules_4` or None if not set

        """
        return self["Number of Chiller Heater Modules 4"]

    @number_of_chiller_heater_modules_4.setter
    def number_of_chiller_heater_modules_4(self, value=1):
        """Corresponds to IDD field `Number of Chiller Heater Modules 4`

        Args:
            value (int): value for IDD Field `Number of Chiller Heater Modules 4`
                Default value: 1
                value >= 1
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Number of Chiller Heater Modules 4"] = value

    @property
    def chiller_heater_modules_performance_component_object_type_5(self):
        """Get chiller_heater_modules_performance_component_object_type_5.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_object_type_5` or None if not set

        """
        return self[
            "Chiller Heater Modules Performance Component Object Type 5"]

    @chiller_heater_modules_performance_component_object_type_5.setter
    def chiller_heater_modules_performance_component_object_type_5(
            self,
            value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Object Type 5`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Object Type 5`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Chiller Heater Modules Performance Component Object Type 5"] = value

    @property
    def chiller_heater_models_performance_component_name_5(self):
        """Get chiller_heater_models_performance_component_name_5.

        Returns:
            str: the value of `chiller_heater_models_performance_component_name_5` or None if not set

        """
        return self["Chiller Heater Models Performance Component Name 5"]

    @chiller_heater_models_performance_component_name_5.setter
    def chiller_heater_models_performance_component_name_5(self, value=None):
        """Corresponds to IDD field `Chiller Heater Models Performance
        Component Name 5`

        Args:
            value (str): value for IDD Field `Chiller Heater Models Performance Component Name 5`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Models Performance Component Name 5"] = value

    @property
    def chiller_heater_modules_control_schedule_name_5(self):
        """Get chiller_heater_modules_control_schedule_name_5.

        Returns:
            str: the value of `chiller_heater_modules_control_schedule_name_5` or None if not set

        """
        return self["Chiller Heater Modules Control Schedule Name 5"]

    @chiller_heater_modules_control_schedule_name_5.setter
    def chiller_heater_modules_control_schedule_name_5(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Control Schedule
        Name 5`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Control Schedule Name 5`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Control Schedule Name 5"] = value

    @property
    def number_of_chiller_heater_modules_5(self):
        """Get number_of_chiller_heater_modules_5.

        Returns:
            int: the value of `number_of_chiller_heater_modules_5` or None if not set

        """
        return self["Number of Chiller Heater Modules 5"]

    @number_of_chiller_heater_modules_5.setter
    def number_of_chiller_heater_modules_5(self, value=1):
        """Corresponds to IDD field `Number of Chiller Heater Modules 5`

        Args:
            value (int): value for IDD Field `Number of Chiller Heater Modules 5`
                Default value: 1
                value >= 1
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Number of Chiller Heater Modules 5"] = value

    @property
    def chiller_heater_modules_performance_component_object_type_6(self):
        """Get chiller_heater_modules_performance_component_object_type_6.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_object_type_6` or None if not set

        """
        return self[
            "Chiller Heater Modules Performance Component Object Type 6"]

    @chiller_heater_modules_performance_component_object_type_6.setter
    def chiller_heater_modules_performance_component_object_type_6(
            self,
            value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Object Type 6`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Object Type 6`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Chiller Heater Modules Performance Component Object Type 6"] = value

    @property
    def chiller_heater_modules_performance_component_name_6(self):
        """Get chiller_heater_modules_performance_component_name_6.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_name_6` or None if not set

        """
        return self["Chiller Heater Modules Performance Component Name 6"]

    @chiller_heater_modules_performance_component_name_6.setter
    def chiller_heater_modules_performance_component_name_6(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Name 6`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Name 6`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Performance Component Name 6"] = value

    @property
    def chiller_heater_modules_control_schedule_name_6(self):
        """Get chiller_heater_modules_control_schedule_name_6.

        Returns:
            str: the value of `chiller_heater_modules_control_schedule_name_6` or None if not set

        """
        return self["Chiller Heater Modules Control Schedule Name 6"]

    @chiller_heater_modules_control_schedule_name_6.setter
    def chiller_heater_modules_control_schedule_name_6(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Control Schedule
        Name 6`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Control Schedule Name 6`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Control Schedule Name 6"] = value

    @property
    def number_of_chiller_heater_modules_6(self):
        """Get number_of_chiller_heater_modules_6.

        Returns:
            int: the value of `number_of_chiller_heater_modules_6` or None if not set

        """
        return self["Number of Chiller Heater Modules 6"]

    @number_of_chiller_heater_modules_6.setter
    def number_of_chiller_heater_modules_6(self, value=1):
        """Corresponds to IDD field `Number of Chiller Heater Modules 6`

        Args:
            value (int): value for IDD Field `Number of Chiller Heater Modules 6`
                Default value: 1
                value >= 1
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Number of Chiller Heater Modules 6"] = value

    @property
    def chiller_heater_modules_performance_component_object_type_7(self):
        """Get chiller_heater_modules_performance_component_object_type_7.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_object_type_7` or None if not set

        """
        return self[
            "Chiller Heater Modules Performance Component Object Type 7"]

    @chiller_heater_modules_performance_component_object_type_7.setter
    def chiller_heater_modules_performance_component_object_type_7(
            self,
            value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Object Type 7`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Object Type 7`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Chiller Heater Modules Performance Component Object Type 7"] = value

    @property
    def chiller_heater_modules_performance_component_name_7(self):
        """Get chiller_heater_modules_performance_component_name_7.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_name_7` or None if not set

        """
        return self["Chiller Heater Modules Performance Component Name 7"]

    @chiller_heater_modules_performance_component_name_7.setter
    def chiller_heater_modules_performance_component_name_7(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Name 7`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Name 7`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Performance Component Name 7"] = value

    @property
    def chiller_heater_modules_control_schedule_name_7(self):
        """Get chiller_heater_modules_control_schedule_name_7.

        Returns:
            str: the value of `chiller_heater_modules_control_schedule_name_7` or None if not set

        """
        return self["Chiller Heater Modules Control Schedule Name 7"]

    @chiller_heater_modules_control_schedule_name_7.setter
    def chiller_heater_modules_control_schedule_name_7(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Control Schedule
        Name 7`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Control Schedule Name 7`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Control Schedule Name 7"] = value

    @property
    def number_of_chiller_heater_modules_7(self):
        """Get number_of_chiller_heater_modules_7.

        Returns:
            int: the value of `number_of_chiller_heater_modules_7` or None if not set

        """
        return self["Number of Chiller Heater Modules 7"]

    @number_of_chiller_heater_modules_7.setter
    def number_of_chiller_heater_modules_7(self, value=1):
        """Corresponds to IDD field `Number of Chiller Heater Modules 7`

        Args:
            value (int): value for IDD Field `Number of Chiller Heater Modules 7`
                Default value: 1
                value >= 1
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Number of Chiller Heater Modules 7"] = value

    @property
    def chiller_heater_modules_performance_component_object_type_8(self):
        """Get chiller_heater_modules_performance_component_object_type_8.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_object_type_8` or None if not set

        """
        return self[
            "Chiller Heater Modules Performance Component Object Type 8"]

    @chiller_heater_modules_performance_component_object_type_8.setter
    def chiller_heater_modules_performance_component_object_type_8(
            self,
            value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Object Type 8`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Object Type 8`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Chiller Heater Modules Performance Component Object Type 8"] = value

    @property
    def chiller_heater_modules_performance_component_name_8(self):
        """Get chiller_heater_modules_performance_component_name_8.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_name_8` or None if not set

        """
        return self["Chiller Heater Modules Performance Component Name 8"]

    @chiller_heater_modules_performance_component_name_8.setter
    def chiller_heater_modules_performance_component_name_8(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Name 8`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Name 8`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Performance Component Name 8"] = value

    @property
    def chiller_heater_modules_control_schedule_name_8(self):
        """Get chiller_heater_modules_control_schedule_name_8.

        Returns:
            str: the value of `chiller_heater_modules_control_schedule_name_8` or None if not set

        """
        return self["Chiller Heater Modules Control Schedule Name 8"]

    @chiller_heater_modules_control_schedule_name_8.setter
    def chiller_heater_modules_control_schedule_name_8(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Control Schedule
        Name 8`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Control Schedule Name 8`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Control Schedule Name 8"] = value

    @property
    def number_of_chiller_heater_modules_8(self):
        """Get number_of_chiller_heater_modules_8.

        Returns:
            int: the value of `number_of_chiller_heater_modules_8` or None if not set

        """
        return self["Number of Chiller Heater Modules 8"]

    @number_of_chiller_heater_modules_8.setter
    def number_of_chiller_heater_modules_8(self, value=1):
        """Corresponds to IDD field `Number of Chiller Heater Modules 8`

        Args:
            value (int): value for IDD Field `Number of Chiller Heater Modules 8`
                Default value: 1
                value >= 1
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Number of Chiller Heater Modules 8"] = value

    @property
    def chiller_heater_modules_performance_component_object_type_9(self):
        """Get chiller_heater_modules_performance_component_object_type_9.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_object_type_9` or None if not set

        """
        return self[
            "Chiller Heater Modules Performance Component Object Type 9"]

    @chiller_heater_modules_performance_component_object_type_9.setter
    def chiller_heater_modules_performance_component_object_type_9(
            self,
            value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Object Type 9`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Object Type 9`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Chiller Heater Modules Performance Component Object Type 9"] = value

    @property
    def chiller_heater_modules_performance_component_name_9(self):
        """Get chiller_heater_modules_performance_component_name_9.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_name_9` or None if not set

        """
        return self["Chiller Heater Modules Performance Component Name 9"]

    @chiller_heater_modules_performance_component_name_9.setter
    def chiller_heater_modules_performance_component_name_9(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Name 9`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Name 9`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Performance Component Name 9"] = value

    @property
    def chiller_heater_modules_control_schedule_name_9(self):
        """Get chiller_heater_modules_control_schedule_name_9.

        Returns:
            str: the value of `chiller_heater_modules_control_schedule_name_9` or None if not set

        """
        return self["Chiller Heater Modules Control Schedule Name 9"]

    @chiller_heater_modules_control_schedule_name_9.setter
    def chiller_heater_modules_control_schedule_name_9(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Control Schedule
        Name 9`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Control Schedule Name 9`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Control Schedule Name 9"] = value

    @property
    def number_of_chiller_heater_modules_9(self):
        """Get number_of_chiller_heater_modules_9.

        Returns:
            int: the value of `number_of_chiller_heater_modules_9` or None if not set

        """
        return self["Number of Chiller Heater Modules 9"]

    @number_of_chiller_heater_modules_9.setter
    def number_of_chiller_heater_modules_9(self, value=1):
        """Corresponds to IDD field `Number of Chiller Heater Modules 9`

        Args:
            value (int): value for IDD Field `Number of Chiller Heater Modules 9`
                Default value: 1
                value >= 1
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Number of Chiller Heater Modules 9"] = value

    @property
    def chiller_heater_modules_performance_component_object_type_10(self):
        """Get chiller_heater_modules_performance_component_object_type_10.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_object_type_10` or None if not set

        """
        return self[
            "Chiller Heater Modules Performance Component Object Type 10"]

    @chiller_heater_modules_performance_component_object_type_10.setter
    def chiller_heater_modules_performance_component_object_type_10(
            self,
            value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Object Type 10`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Object Type 10`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Chiller Heater Modules Performance Component Object Type 10"] = value

    @property
    def chiller_heater_modules_performance_component_name_10(self):
        """Get chiller_heater_modules_performance_component_name_10.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_name_10` or None if not set

        """
        return self["Chiller Heater Modules Performance Component Name 10"]

    @chiller_heater_modules_performance_component_name_10.setter
    def chiller_heater_modules_performance_component_name_10(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Name 10`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Name 10`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Performance Component Name 10"] = value

    @property
    def chiller_heater_modules_control_schedule_name_10(self):
        """Get chiller_heater_modules_control_schedule_name_10.

        Returns:
            str: the value of `chiller_heater_modules_control_schedule_name_10` or None if not set

        """
        return self["Chiller Heater Modules Control Schedule Name 10"]

    @chiller_heater_modules_control_schedule_name_10.setter
    def chiller_heater_modules_control_schedule_name_10(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Control Schedule
        Name 10`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Control Schedule Name 10`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Control Schedule Name 10"] = value

    @property
    def number_of_chiller_heater_modules_10(self):
        """Get number_of_chiller_heater_modules_10.

        Returns:
            int: the value of `number_of_chiller_heater_modules_10` or None if not set

        """
        return self["Number of Chiller Heater Modules 10"]

    @number_of_chiller_heater_modules_10.setter
    def number_of_chiller_heater_modules_10(self, value=1):
        """Corresponds to IDD field `Number of Chiller Heater Modules 10`

        Args:
            value (int): value for IDD Field `Number of Chiller Heater Modules 10`
                Default value: 1
                value >= 1
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Number of Chiller Heater Modules 10"] = value

    @property
    def chiller_heater_modules_performance_component_object_type_11(self):
        """Get chiller_heater_modules_performance_component_object_type_11.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_object_type_11` or None if not set

        """
        return self[
            "Chiller Heater Modules Performance Component Object Type 11"]

    @chiller_heater_modules_performance_component_object_type_11.setter
    def chiller_heater_modules_performance_component_object_type_11(
            self,
            value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Object Type 11`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Object Type 11`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Chiller Heater Modules Performance Component Object Type 11"] = value

    @property
    def chiller_heater_modules_performance_component_name_11(self):
        """Get chiller_heater_modules_performance_component_name_11.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_name_11` or None if not set

        """
        return self["Chiller Heater Modules Performance Component Name 11"]

    @chiller_heater_modules_performance_component_name_11.setter
    def chiller_heater_modules_performance_component_name_11(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Name 11`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Name 11`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Performance Component Name 11"] = value

    @property
    def chiller_heater_module_control_schedule_name_11(self):
        """Get chiller_heater_module_control_schedule_name_11.

        Returns:
            str: the value of `chiller_heater_module_control_schedule_name_11` or None if not set

        """
        return self["Chiller Heater Module Control Schedule Name 11"]

    @chiller_heater_module_control_schedule_name_11.setter
    def chiller_heater_module_control_schedule_name_11(self, value=None):
        """Corresponds to IDD field `Chiller Heater Module Control Schedule
        Name 11`

        Args:
            value (str): value for IDD Field `Chiller Heater Module Control Schedule Name 11`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Module Control Schedule Name 11"] = value

    @property
    def number_of_chiller_heater_modules_11(self):
        """Get number_of_chiller_heater_modules_11.

        Returns:
            int: the value of `number_of_chiller_heater_modules_11` or None if not set

        """
        return self["Number of Chiller Heater Modules 11"]

    @number_of_chiller_heater_modules_11.setter
    def number_of_chiller_heater_modules_11(self, value=1):
        """Corresponds to IDD field `Number of Chiller Heater Modules 11`

        Args:
            value (int): value for IDD Field `Number of Chiller Heater Modules 11`
                Default value: 1
                value >= 1
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Number of Chiller Heater Modules 11"] = value

    @property
    def chiller_heater_modules_performance_component_object_type_12(self):
        """Get chiller_heater_modules_performance_component_object_type_12.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_object_type_12` or None if not set

        """
        return self[
            "Chiller Heater Modules Performance Component Object Type 12"]

    @chiller_heater_modules_performance_component_object_type_12.setter
    def chiller_heater_modules_performance_component_object_type_12(
            self,
            value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Object Type 12`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Object Type 12`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Chiller Heater Modules Performance Component Object Type 12"] = value

    @property
    def chiller_heater_modules_performance_component_name_12(self):
        """Get chiller_heater_modules_performance_component_name_12.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_name_12` or None if not set

        """
        return self["Chiller Heater Modules Performance Component Name 12"]

    @chiller_heater_modules_performance_component_name_12.setter
    def chiller_heater_modules_performance_component_name_12(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Name 12`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Name 12`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Performance Component Name 12"] = value

    @property
    def chiller_heater_modules_control_schedule_name_12(self):
        """Get chiller_heater_modules_control_schedule_name_12.

        Returns:
            str: the value of `chiller_heater_modules_control_schedule_name_12` or None if not set

        """
        return self["Chiller Heater Modules Control Schedule Name 12"]

    @chiller_heater_modules_control_schedule_name_12.setter
    def chiller_heater_modules_control_schedule_name_12(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Control Schedule
        Name 12`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Control Schedule Name 12`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Control Schedule Name 12"] = value

    @property
    def number_of_chiller_heater_modules_12(self):
        """Get number_of_chiller_heater_modules_12.

        Returns:
            int: the value of `number_of_chiller_heater_modules_12` or None if not set

        """
        return self["Number of Chiller Heater Modules 12"]

    @number_of_chiller_heater_modules_12.setter
    def number_of_chiller_heater_modules_12(self, value=1):
        """Corresponds to IDD field `Number of Chiller Heater Modules 12`

        Args:
            value (int): value for IDD Field `Number of Chiller Heater Modules 12`
                Default value: 1
                value >= 1
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Number of Chiller Heater Modules 12"] = value

    @property
    def chiller_heater_modules_performance_component_object_type_13(self):
        """Get chiller_heater_modules_performance_component_object_type_13.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_object_type_13` or None if not set

        """
        return self[
            "Chiller Heater Modules Performance Component Object Type 13"]

    @chiller_heater_modules_performance_component_object_type_13.setter
    def chiller_heater_modules_performance_component_object_type_13(
            self,
            value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Object Type 13`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Object Type 13`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Chiller Heater Modules Performance Component Object Type 13"] = value

    @property
    def chiller_heater_modules_performance_component_name_13(self):
        """Get chiller_heater_modules_performance_component_name_13.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_name_13` or None if not set

        """
        return self["Chiller Heater Modules Performance Component Name 13"]

    @chiller_heater_modules_performance_component_name_13.setter
    def chiller_heater_modules_performance_component_name_13(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Name 13`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Name 13`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Performance Component Name 13"] = value

    @property
    def chiller_heater_modules_control_schedule_name_13(self):
        """Get chiller_heater_modules_control_schedule_name_13.

        Returns:
            str: the value of `chiller_heater_modules_control_schedule_name_13` or None if not set

        """
        return self["Chiller Heater Modules Control Schedule Name 13"]

    @chiller_heater_modules_control_schedule_name_13.setter
    def chiller_heater_modules_control_schedule_name_13(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Control Schedule
        Name 13`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Control Schedule Name 13`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Control Schedule Name 13"] = value

    @property
    def number_of_chiller_heater_modules_13(self):
        """Get number_of_chiller_heater_modules_13.

        Returns:
            int: the value of `number_of_chiller_heater_modules_13` or None if not set

        """
        return self["Number of Chiller Heater Modules 13"]

    @number_of_chiller_heater_modules_13.setter
    def number_of_chiller_heater_modules_13(self, value=1):
        """Corresponds to IDD field `Number of Chiller Heater Modules 13`

        Args:
            value (int): value for IDD Field `Number of Chiller Heater Modules 13`
                Default value: 1
                value >= 1
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Number of Chiller Heater Modules 13"] = value

    @property
    def chiller_heater_modules_performance_component_object_type_14(self):
        """Get chiller_heater_modules_performance_component_object_type_14.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_object_type_14` or None if not set

        """
        return self[
            "Chiller Heater Modules Performance Component Object Type 14"]

    @chiller_heater_modules_performance_component_object_type_14.setter
    def chiller_heater_modules_performance_component_object_type_14(
            self,
            value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Object Type 14`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Object Type 14`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Chiller Heater Modules Performance Component Object Type 14"] = value

    @property
    def chiller_heater_modules_performance_component_name_14(self):
        """Get chiller_heater_modules_performance_component_name_14.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_name_14` or None if not set

        """
        return self["Chiller Heater Modules Performance Component Name 14"]

    @chiller_heater_modules_performance_component_name_14.setter
    def chiller_heater_modules_performance_component_name_14(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Name 14`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Name 14`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Performance Component Name 14"] = value

    @property
    def chiller_heater_modules_control_schedule_name_14(self):
        """Get chiller_heater_modules_control_schedule_name_14.

        Returns:
            str: the value of `chiller_heater_modules_control_schedule_name_14` or None if not set

        """
        return self["Chiller Heater Modules Control Schedule Name 14"]

    @chiller_heater_modules_control_schedule_name_14.setter
    def chiller_heater_modules_control_schedule_name_14(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Control Schedule
        Name 14`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Control Schedule Name 14`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Control Schedule Name 14"] = value

    @property
    def number_of_chiller_heater_modules_14(self):
        """Get number_of_chiller_heater_modules_14.

        Returns:
            int: the value of `number_of_chiller_heater_modules_14` or None if not set

        """
        return self["Number of Chiller Heater Modules 14"]

    @number_of_chiller_heater_modules_14.setter
    def number_of_chiller_heater_modules_14(self, value=1):
        """Corresponds to IDD field `Number of Chiller Heater Modules 14`

        Args:
            value (int): value for IDD Field `Number of Chiller Heater Modules 14`
                Default value: 1
                value >= 1
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Number of Chiller Heater Modules 14"] = value

    @property
    def chiller_heater_modules_performance_component_object_type_15(self):
        """Get chiller_heater_modules_performance_component_object_type_15.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_object_type_15` or None if not set

        """
        return self[
            "Chiller Heater Modules Performance Component Object Type 15"]

    @chiller_heater_modules_performance_component_object_type_15.setter
    def chiller_heater_modules_performance_component_object_type_15(
            self,
            value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Object Type 15`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Object Type 15`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Chiller Heater Modules Performance Component Object Type 15"] = value

    @property
    def chiller_heater_modules_performance_component_name_15(self):
        """Get chiller_heater_modules_performance_component_name_15.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_name_15` or None if not set

        """
        return self["Chiller Heater Modules Performance Component Name 15"]

    @chiller_heater_modules_performance_component_name_15.setter
    def chiller_heater_modules_performance_component_name_15(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Name 15`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Name 15`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Performance Component Name 15"] = value

    @property
    def chiller_heater_modules_control_schedule_name_15(self):
        """Get chiller_heater_modules_control_schedule_name_15.

        Returns:
            str: the value of `chiller_heater_modules_control_schedule_name_15` or None if not set

        """
        return self["Chiller Heater Modules Control Schedule Name 15"]

    @chiller_heater_modules_control_schedule_name_15.setter
    def chiller_heater_modules_control_schedule_name_15(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Control Schedule
        Name 15`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Control Schedule Name 15`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Control Schedule Name 15"] = value

    @property
    def number_of_chiller_heater_modules_15(self):
        """Get number_of_chiller_heater_modules_15.

        Returns:
            int: the value of `number_of_chiller_heater_modules_15` or None if not set

        """
        return self["Number of Chiller Heater Modules 15"]

    @number_of_chiller_heater_modules_15.setter
    def number_of_chiller_heater_modules_15(self, value=1):
        """Corresponds to IDD field `Number of Chiller Heater Modules 15`

        Args:
            value (int): value for IDD Field `Number of Chiller Heater Modules 15`
                Default value: 1
                value >= 1
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Number of Chiller Heater Modules 15"] = value

    @property
    def chiller_heater_modules_performance_component_object_type_16(self):
        """Get chiller_heater_modules_performance_component_object_type_16.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_object_type_16` or None if not set

        """
        return self[
            "Chiller Heater Modules Performance Component Object Type 16"]

    @chiller_heater_modules_performance_component_object_type_16.setter
    def chiller_heater_modules_performance_component_object_type_16(
            self,
            value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Object Type 16`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Object Type 16`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Chiller Heater Modules Performance Component Object Type 16"] = value

    @property
    def chiller_heater_modules_performance_component_name_16(self):
        """Get chiller_heater_modules_performance_component_name_16.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_name_16` or None if not set

        """
        return self["Chiller Heater Modules Performance Component Name 16"]

    @chiller_heater_modules_performance_component_name_16.setter
    def chiller_heater_modules_performance_component_name_16(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Name 16`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Name 16`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Performance Component Name 16"] = value

    @property
    def chiller_heater_modules_control_schedule_name_16(self):
        """Get chiller_heater_modules_control_schedule_name_16.

        Returns:
            str: the value of `chiller_heater_modules_control_schedule_name_16` or None if not set

        """
        return self["Chiller Heater Modules Control Schedule Name 16"]

    @chiller_heater_modules_control_schedule_name_16.setter
    def chiller_heater_modules_control_schedule_name_16(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Control Schedule
        Name 16`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Control Schedule Name 16`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Control Schedule Name 16"] = value

    @property
    def number_of_chiller_heater_modules_16(self):
        """Get number_of_chiller_heater_modules_16.

        Returns:
            int: the value of `number_of_chiller_heater_modules_16` or None if not set

        """
        return self["Number of Chiller Heater Modules 16"]

    @number_of_chiller_heater_modules_16.setter
    def number_of_chiller_heater_modules_16(self, value=1):
        """Corresponds to IDD field `Number of Chiller Heater Modules 16`

        Args:
            value (int): value for IDD Field `Number of Chiller Heater Modules 16`
                Default value: 1
                value >= 1
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Number of Chiller Heater Modules 16"] = value

    @property
    def chiller_heater_modules_performance_component_object_type_17(self):
        """Get chiller_heater_modules_performance_component_object_type_17.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_object_type_17` or None if not set

        """
        return self[
            "Chiller Heater Modules Performance Component Object Type 17"]

    @chiller_heater_modules_performance_component_object_type_17.setter
    def chiller_heater_modules_performance_component_object_type_17(
            self,
            value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Object Type 17`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Object Type 17`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Chiller Heater Modules Performance Component Object Type 17"] = value

    @property
    def chiller_heater_modules_performance_component_name_17(self):
        """Get chiller_heater_modules_performance_component_name_17.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_name_17` or None if not set

        """
        return self["Chiller Heater Modules Performance Component Name 17"]

    @chiller_heater_modules_performance_component_name_17.setter
    def chiller_heater_modules_performance_component_name_17(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Name 17`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Name 17`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Performance Component Name 17"] = value

    @property
    def chiller_heater_modules_control_schedule_name_17(self):
        """Get chiller_heater_modules_control_schedule_name_17.

        Returns:
            str: the value of `chiller_heater_modules_control_schedule_name_17` or None if not set

        """
        return self["Chiller Heater Modules Control Schedule Name 17"]

    @chiller_heater_modules_control_schedule_name_17.setter
    def chiller_heater_modules_control_schedule_name_17(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Control Schedule
        Name 17`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Control Schedule Name 17`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Control Schedule Name 17"] = value

    @property
    def number_of_chiller_heater_modules_17(self):
        """Get number_of_chiller_heater_modules_17.

        Returns:
            int: the value of `number_of_chiller_heater_modules_17` or None if not set

        """
        return self["Number of Chiller Heater Modules 17"]

    @number_of_chiller_heater_modules_17.setter
    def number_of_chiller_heater_modules_17(self, value=1):
        """Corresponds to IDD field `Number of Chiller Heater Modules 17`

        Args:
            value (int): value for IDD Field `Number of Chiller Heater Modules 17`
                Default value: 1
                value >= 1
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Number of Chiller Heater Modules 17"] = value

    @property
    def chiller_heater_modules_performance_component_object_type_18(self):
        """Get chiller_heater_modules_performance_component_object_type_18.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_object_type_18` or None if not set

        """
        return self[
            "Chiller Heater Modules Performance Component Object Type 18"]

    @chiller_heater_modules_performance_component_object_type_18.setter
    def chiller_heater_modules_performance_component_object_type_18(
            self,
            value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Object Type 18`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Object Type 18`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Chiller Heater Modules Performance Component Object Type 18"] = value

    @property
    def chiller_heater_modules_performance_component_name_18(self):
        """Get chiller_heater_modules_performance_component_name_18.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_name_18` or None if not set

        """
        return self["Chiller Heater Modules Performance Component Name 18"]

    @chiller_heater_modules_performance_component_name_18.setter
    def chiller_heater_modules_performance_component_name_18(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Name 18`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Name 18`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Performance Component Name 18"] = value

    @property
    def chiller_heater_modules_control_control_schedule_name_18(self):
        """Get chiller_heater_modules_control_control_schedule_name_18.

        Returns:
            str: the value of `chiller_heater_modules_control_control_schedule_name_18` or None if not set

        """
        return self["Chiller Heater Modules Control Control Schedule Name 18"]

    @chiller_heater_modules_control_control_schedule_name_18.setter
    def chiller_heater_modules_control_control_schedule_name_18(
            self,
            value=None):
        """Corresponds to IDD field `Chiller Heater Modules Control Control
        Schedule Name 18`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Control Control Schedule Name 18`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Control Control Schedule Name 18"] = value

    @property
    def number_of_chiller_heater_modules_18(self):
        """Get number_of_chiller_heater_modules_18.

        Returns:
            int: the value of `number_of_chiller_heater_modules_18` or None if not set

        """
        return self["Number of Chiller Heater Modules 18"]

    @number_of_chiller_heater_modules_18.setter
    def number_of_chiller_heater_modules_18(self, value=1):
        """Corresponds to IDD field `Number of Chiller Heater Modules 18`

        Args:
            value (int): value for IDD Field `Number of Chiller Heater Modules 18`
                Default value: 1
                value >= 1
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Number of Chiller Heater Modules 18"] = value

    @property
    def chiller_heater_modules_performance_component_object_type_19(self):
        """Get chiller_heater_modules_performance_component_object_type_19.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_object_type_19` or None if not set

        """
        return self[
            "Chiller Heater Modules Performance Component Object Type 19"]

    @chiller_heater_modules_performance_component_object_type_19.setter
    def chiller_heater_modules_performance_component_object_type_19(
            self,
            value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Object Type 19`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Object Type 19`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Chiller Heater Modules Performance Component Object Type 19"] = value

    @property
    def chiller_heater_modules_performance_component_name_19(self):
        """Get chiller_heater_modules_performance_component_name_19.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_name_19` or None if not set

        """
        return self["Chiller Heater Modules Performance Component Name 19"]

    @chiller_heater_modules_performance_component_name_19.setter
    def chiller_heater_modules_performance_component_name_19(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Name 19`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Name 19`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Performance Component Name 19"] = value

    @property
    def chiller_heater_modules_control_schedule_name_19(self):
        """Get chiller_heater_modules_control_schedule_name_19.

        Returns:
            str: the value of `chiller_heater_modules_control_schedule_name_19` or None if not set

        """
        return self["Chiller Heater Modules Control Schedule Name 19"]

    @chiller_heater_modules_control_schedule_name_19.setter
    def chiller_heater_modules_control_schedule_name_19(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Control Schedule
        Name 19`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Control Schedule Name 19`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Control Schedule Name 19"] = value

    @property
    def number_of_chiller_heater_modules_19(self):
        """Get number_of_chiller_heater_modules_19.

        Returns:
            int: the value of `number_of_chiller_heater_modules_19` or None if not set

        """
        return self["Number of Chiller Heater Modules 19"]

    @number_of_chiller_heater_modules_19.setter
    def number_of_chiller_heater_modules_19(self, value=1):
        """Corresponds to IDD field `Number of Chiller Heater Modules 19`

        Args:
            value (int): value for IDD Field `Number of Chiller Heater Modules 19`
                Default value: 1
                value >= 1
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Number of Chiller Heater Modules 19"] = value

    @property
    def chiller_heater_modules_performance_component_object_type_20(self):
        """Get chiller_heater_modules_performance_component_object_type_20.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_object_type_20` or None if not set

        """
        return self[
            "Chiller Heater Modules Performance Component Object Type 20"]

    @chiller_heater_modules_performance_component_object_type_20.setter
    def chiller_heater_modules_performance_component_object_type_20(
            self,
            value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Object Type 20`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Object Type 20`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Chiller Heater Modules Performance Component Object Type 20"] = value

    @property
    def chiller_heater_modules_performance_component_name_20(self):
        """Get chiller_heater_modules_performance_component_name_20.

        Returns:
            str: the value of `chiller_heater_modules_performance_component_name_20` or None if not set

        """
        return self["Chiller Heater Modules Performance Component Name 20"]

    @chiller_heater_modules_performance_component_name_20.setter
    def chiller_heater_modules_performance_component_name_20(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Performance
        Component Name 20`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Performance Component Name 20`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Performance Component Name 20"] = value

    @property
    def chiller_heater_modules_control_schedule_name_20(self):
        """Get chiller_heater_modules_control_schedule_name_20.

        Returns:
            str: the value of `chiller_heater_modules_control_schedule_name_20` or None if not set

        """
        return self["Chiller Heater Modules Control Schedule Name 20"]

    @chiller_heater_modules_control_schedule_name_20.setter
    def chiller_heater_modules_control_schedule_name_20(self, value=None):
        """Corresponds to IDD field `Chiller Heater Modules Control Schedule
        Name 20`

        Args:
            value (str): value for IDD Field `Chiller Heater Modules Control Schedule Name 20`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chiller Heater Modules Control Schedule Name 20"] = value

    @property
    def number_of_chiller_heater_modules_20(self):
        """Get number_of_chiller_heater_modules_20.

        Returns:
            int: the value of `number_of_chiller_heater_modules_20` or None if not set

        """
        return self["Number of Chiller Heater Modules 20"]

    @number_of_chiller_heater_modules_20.setter
    def number_of_chiller_heater_modules_20(self, value=1):
        """Corresponds to IDD field `Number of Chiller Heater Modules 20`

        Args:
            value (int): value for IDD Field `Number of Chiller Heater Modules 20`
                Default value: 1
                value >= 1
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Number of Chiller Heater Modules 20"] = value




class ChillerHeaterPerformanceElectricEir(DataObject):

    """ Corresponds to IDD object `ChillerHeaterPerformance:Electric:EIR`
        This chiller model is a generic chiller-heater where the cooling mode performance is a
        function of condenser entering or leaving fluid temperature and the heating mode performance
        is typically a function of condenser leaving fluid temperature. Performance at off-reference
        conditions is modeled using three polynomial equations per mode. Six curve objects are required.
    """
    schema = {'min-fields': 29,
              'name': u'ChillerHeaterPerformance:Electric:EIR',
              'pyname': u'ChillerHeaterPerformanceElectricEir',
              'format': None,
              'fields': OrderedDict([(u'name',
                                      {'name': u'Name',
                                       'pyname': u'name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'alpha'}),
                                     (u'reference cooling mode evaporator capacity',
                                      {'name': u'Reference Cooling Mode Evaporator Capacity',
                                       'pyname': u'reference_cooling_mode_evaporator_capacity',
                                       'minimum>': 0.0,
                                       'required-field': True,
                                       'autosizable': True,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'W'}),
                                     (u'reference cooling mode cop',
                                      {'name': u'Reference Cooling Mode COP',
                                       'pyname': u'reference_cooling_mode_cop',
                                       'minimum>': 0.0,
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'W/W'}),
                                     (u'reference cooling mode leaving chilled water temperature',
                                      {'name': u'Reference Cooling Mode Leaving Chilled Water Temperature',
                                       'pyname': u'reference_cooling_mode_leaving_chilled_water_temperature',
                                       'default': 6.67,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'C'}),
                                     (u'reference cooling mode entering condenser fluid temperature',
                                      {'name': u'Reference Cooling Mode Entering Condenser Fluid Temperature',
                                       'pyname': u'reference_cooling_mode_entering_condenser_fluid_temperature',
                                       'default': 29.44,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'C'}),
                                     (u'reference cooling mode leaving condenser water temperature',
                                      {'name': u'Reference Cooling Mode Leaving Condenser Water Temperature',
                                       'pyname': u'reference_cooling_mode_leaving_condenser_water_temperature',
                                       'default': 35.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'C'}),
                                     (u'reference heating mode cooling capacity ratio',
                                      {'name': u'Reference Heating Mode Cooling Capacity Ratio',
                                       'pyname': u'reference_heating_mode_cooling_capacity_ratio',
                                       'default': 0.75,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'reference heating mode cooling power input ratio',
                                      {'name': u'Reference Heating Mode Cooling Power Input Ratio',
                                       'pyname': u'reference_heating_mode_cooling_power_input_ratio',
                                       'default': 1.38,
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'reference heating mode leaving chilled water temperature',
                                      {'name': u'Reference Heating Mode Leaving Chilled Water Temperature',
                                       'pyname': u'reference_heating_mode_leaving_chilled_water_temperature',
                                       'default': 6.67,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'C'}),
                                     (u'reference heating mode leaving condenser water temperature',
                                      {'name': u'Reference Heating Mode Leaving Condenser Water Temperature',
                                       'pyname': u'reference_heating_mode_leaving_condenser_water_temperature',
                                       'default': 49.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'C'}),
                                     (u'reference heating mode entering condenser fluid temperature',
                                      {'name': u'Reference Heating Mode Entering Condenser Fluid Temperature',
                                       'pyname': u'reference_heating_mode_entering_condenser_fluid_temperature',
                                       'default': 29.44,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'C'}),
                                     (u'heating mode entering chilled water temperature low limit',
                                      {'name': u'Heating Mode Entering Chilled Water Temperature Low Limit',
                                       'pyname': u'heating_mode_entering_chilled_water_temperature_low_limit',
                                       'default': 12.22,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'C'}),
                                     (u'chilled water flow mode type',
                                      {'name': u'Chilled Water Flow Mode Type',
                                       'pyname': u'chilled_water_flow_mode_type',
                                       'default': u'ConstantFlow',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'ConstantFlow',
                                                           u'VariableFlow'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'design chilled water flow rate',
                                      {'name': u'Design Chilled Water Flow Rate',
                                       'pyname': u'design_chilled_water_flow_rate',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': True,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'm3/s'}),
                                     (u'design condenser water flow rate',
                                      {'name': u'Design Condenser Water Flow Rate',
                                       'pyname': u'design_condenser_water_flow_rate',
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': True,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'm3/s'}),
                                     (u'design hot water flow rate',
                                      {'name': u'Design Hot Water Flow Rate',
                                       'pyname': u'design_hot_water_flow_rate',
                                       'default': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real',
                                       'unit': u'm3/s'}),
                                     (u'compressor motor efficiency',
                                      {'name': u'Compressor Motor Efficiency',
                                       'pyname': u'compressor_motor_efficiency',
                                       'default': 1.0,
                                       'maximum': 1.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'minimum': 0.0,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'condenser type',
                                      {'name': u'Condenser Type',
                                       'pyname': u'condenser_type',
                                       'default': u'WaterCooled',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'AirCooled',
                                                           u'WaterCooled'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'cooling mode temperature curve condenser water independent variable',
                                      {'name': u'Cooling Mode Temperature Curve Condenser Water Independent Variable',
                                       'pyname': u'cooling_mode_temperature_curve_condenser_water_independent_variable',
                                       'default': u'EnteringCondenser',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'EnteringCondenser',
                                                           u'LeavingCondenser'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'cooling mode cooling capacity function of temperature curve name',
                                      {'name': u'Cooling Mode Cooling Capacity Function of Temperature Curve Name',
                                       'pyname': u'cooling_mode_cooling_capacity_function_of_temperature_curve_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'cooling mode electric input to cooling output ratio function of temperature curve name',
                                      {'name': u'Cooling Mode Electric Input to Cooling Output Ratio Function of Temperature Curve Name',
                                       'pyname': u'cooling_mode_electric_input_to_cooling_output_ratio_function_of_temperature_curve_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'cooling mode electric input to cooling output ratio function of part load ratio curve name',
                                      {'name': u'Cooling Mode Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name',
                                       'pyname': u'cooling_mode_electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'cooling mode cooling capacity optimum part load ratio',
                                      {'name': u'Cooling Mode Cooling Capacity Optimum Part Load Ratio',
                                       'pyname': u'cooling_mode_cooling_capacity_optimum_part_load_ratio',
                                       'default': 1.0,
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'heating mode temperature curve condenser water independent variable',
                                      {'name': u'Heating Mode Temperature Curve Condenser Water Independent Variable',
                                       'pyname': u'heating_mode_temperature_curve_condenser_water_independent_variable',
                                       'default': u'LeavingCondenser',
                                       'required-field': False,
                                       'autosizable': False,
                                       'accepted-values': [u'EnteringCondenser',
                                                           u'LeavingCondenser'],
                                       'autocalculatable': False,
                                       'type': 'alpha'}),
                                     (u'heating mode cooling capacity function of temperature curve name',
                                      {'name': u'Heating Mode Cooling Capacity Function of Temperature Curve Name',
                                       'pyname': u'heating_mode_cooling_capacity_function_of_temperature_curve_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'heating mode electric input to cooling output ratio function of temperature curve name',
                                      {'name': u'Heating Mode Electric Input to Cooling Output Ratio Function of Temperature Curve Name',
                                       'pyname': u'heating_mode_electric_input_to_cooling_output_ratio_function_of_temperature_curve_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'heating mode electric input to cooling output ratio function of part load ratio curve name',
                                      {'name': u'Heating Mode Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name',
                                       'pyname': u'heating_mode_electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name',
                                       'required-field': True,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'object-list'}),
                                     (u'heating mode cooling capacity optimum part load ratio',
                                      {'name': u'Heating Mode Cooling Capacity Optimum Part Load Ratio',
                                       'pyname': u'heating_mode_cooling_capacity_optimum_part_load_ratio',
                                       'default': 1.0,
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'}),
                                     (u'sizing factor',
                                      {'name': u'Sizing Factor',
                                       'pyname': u'sizing_factor',
                                       'default': 1.0,
                                       'minimum>': 0.0,
                                       'required-field': False,
                                       'autosizable': False,
                                       'autocalculatable': False,
                                       'type': u'real'})]),
              'extensible-fields': OrderedDict(),
              'unique-object': False,
              'required-object': False,
              'group': u'Plant Heating and Cooling Equipment'}

    @property
    def name(self):
        """Get name.

        Returns:
            str: the value of `name` or None if not set

        """
        return self["Name"]

    @name.setter
    def name(self, value=None):
        """Corresponds to IDD field `Name`

        Args:
            value (str): value for IDD Field `Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Name"] = value

    @property
    def reference_cooling_mode_evaporator_capacity(self):
        """Get reference_cooling_mode_evaporator_capacity.

        Returns:
            float: the value of `reference_cooling_mode_evaporator_capacity` or None if not set

        """
        return self["Reference Cooling Mode Evaporator Capacity"]

    @reference_cooling_mode_evaporator_capacity.setter
    def reference_cooling_mode_evaporator_capacity(self, value=None):
        """Corresponds to IDD field `Reference Cooling Mode Evaporator
        Capacity`

        Args:
            value (float or "Autosize"): value for IDD Field `Reference Cooling Mode Evaporator Capacity`
                Units: W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Reference Cooling Mode Evaporator Capacity"] = value

    @property
    def reference_cooling_mode_cop(self):
        """Get reference_cooling_mode_cop.

        Returns:
            float: the value of `reference_cooling_mode_cop` or None if not set

        """
        return self["Reference Cooling Mode COP"]

    @reference_cooling_mode_cop.setter
    def reference_cooling_mode_cop(self, value=None):
        """Corresponds to IDD field `Reference Cooling Mode COP` Efficiency of
        the chiller compressor (cooling output/compressor energy input).

        Args:
            value (float): value for IDD Field `Reference Cooling Mode COP`
                Units: W/W
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Reference Cooling Mode COP"] = value

    @property
    def reference_cooling_mode_leaving_chilled_water_temperature(self):
        """Get reference_cooling_mode_leaving_chilled_water_temperature.

        Returns:
            float: the value of `reference_cooling_mode_leaving_chilled_water_temperature` or None if not set

        """
        return self["Reference Cooling Mode Leaving Chilled Water Temperature"]

    @reference_cooling_mode_leaving_chilled_water_temperature.setter
    def reference_cooling_mode_leaving_chilled_water_temperature(
            self,
            value=6.67):
        """Corresponds to IDD field `Reference Cooling Mode Leaving Chilled
        Water Temperature`

        Args:
            value (float): value for IDD Field `Reference Cooling Mode Leaving Chilled Water Temperature`
                Units: C
                Default value: 6.67
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Reference Cooling Mode Leaving Chilled Water Temperature"] = value

    @property
    def reference_cooling_mode_entering_condenser_fluid_temperature(self):
        """Get reference_cooling_mode_entering_condenser_fluid_temperature.

        Returns:
            float: the value of `reference_cooling_mode_entering_condenser_fluid_temperature` or None if not set

        """
        return self[
            "Reference Cooling Mode Entering Condenser Fluid Temperature"]

    @reference_cooling_mode_entering_condenser_fluid_temperature.setter
    def reference_cooling_mode_entering_condenser_fluid_temperature(
            self,
            value=29.44):
        """Corresponds to IDD field `Reference Cooling Mode Entering Condenser
        Fluid Temperature`

        Args:
            value (float): value for IDD Field `Reference Cooling Mode Entering Condenser Fluid Temperature`
                Units: C
                Default value: 29.44
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Reference Cooling Mode Entering Condenser Fluid Temperature"] = value

    @property
    def reference_cooling_mode_leaving_condenser_water_temperature(self):
        """Get reference_cooling_mode_leaving_condenser_water_temperature.

        Returns:
            float: the value of `reference_cooling_mode_leaving_condenser_water_temperature` or None if not set

        """
        return self[
            "Reference Cooling Mode Leaving Condenser Water Temperature"]

    @reference_cooling_mode_leaving_condenser_water_temperature.setter
    def reference_cooling_mode_leaving_condenser_water_temperature(
            self,
            value=35.0):
        """Corresponds to IDD field `Reference Cooling Mode Leaving Condenser
        Water Temperature`

        Args:
            value (float): value for IDD Field `Reference Cooling Mode Leaving Condenser Water Temperature`
                Units: C
                Default value: 35.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Reference Cooling Mode Leaving Condenser Water Temperature"] = value

    @property
    def reference_heating_mode_cooling_capacity_ratio(self):
        """Get reference_heating_mode_cooling_capacity_ratio.

        Returns:
            float: the value of `reference_heating_mode_cooling_capacity_ratio` or None if not set

        """
        return self["Reference Heating Mode Cooling Capacity Ratio"]

    @reference_heating_mode_cooling_capacity_ratio.setter
    def reference_heating_mode_cooling_capacity_ratio(self, value=0.75):
        """  Corresponds to IDD field `Reference Heating Mode Cooling Capacity Ratio`
        During simultaneous cooling-heating mode, this ratio is relative to the Reference Cooling Mode Cooling Capacity
        (Evaporator capacity at simul clg-htg mode ref condition)/ (Evaporator capacity at cooling mode ref condition)

        Args:
            value (float): value for IDD Field `Reference Heating Mode Cooling Capacity Ratio`
                Default value: 0.75
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Reference Heating Mode Cooling Capacity Ratio"] = value

    @property
    def reference_heating_mode_cooling_power_input_ratio(self):
        """Get reference_heating_mode_cooling_power_input_ratio.

        Returns:
            float: the value of `reference_heating_mode_cooling_power_input_ratio` or None if not set

        """
        return self["Reference Heating Mode Cooling Power Input Ratio"]

    @reference_heating_mode_cooling_power_input_ratio.setter
    def reference_heating_mode_cooling_power_input_ratio(self, value=1.38):
        """  Corresponds to IDD field `Reference Heating Mode Cooling Power Input Ratio`
        During simultaneous cooling-heating mode, this ratio is relative to the Reference Cooling Mode COP
        (Power at simultaneous clg-htg mode reference condition)/ (Power at cooling mode reference condition)

        Args:
            value (float): value for IDD Field `Reference Heating Mode Cooling Power Input Ratio`
                Default value: 1.38
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Reference Heating Mode Cooling Power Input Ratio"] = value

    @property
    def reference_heating_mode_leaving_chilled_water_temperature(self):
        """Get reference_heating_mode_leaving_chilled_water_temperature.

        Returns:
            float: the value of `reference_heating_mode_leaving_chilled_water_temperature` or None if not set

        """
        return self["Reference Heating Mode Leaving Chilled Water Temperature"]

    @reference_heating_mode_leaving_chilled_water_temperature.setter
    def reference_heating_mode_leaving_chilled_water_temperature(
            self,
            value=6.67):
        """  Corresponds to IDD field `Reference Heating Mode Leaving Chilled Water Temperature`
        During simultaneous cooling-heating mode

        Args:
            value (float): value for IDD Field `Reference Heating Mode Leaving Chilled Water Temperature`
                Units: C
                Default value: 6.67
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Reference Heating Mode Leaving Chilled Water Temperature"] = value

    @property
    def reference_heating_mode_leaving_condenser_water_temperature(self):
        """Get reference_heating_mode_leaving_condenser_water_temperature.

        Returns:
            float: the value of `reference_heating_mode_leaving_condenser_water_temperature` or None if not set

        """
        return self[
            "Reference Heating Mode Leaving Condenser Water Temperature"]

    @reference_heating_mode_leaving_condenser_water_temperature.setter
    def reference_heating_mode_leaving_condenser_water_temperature(
            self,
            value=49.0):
        """  Corresponds to IDD field `Reference Heating Mode Leaving Condenser Water Temperature`
        During simultaneous cooling-heating mode

        Args:
            value (float): value for IDD Field `Reference Heating Mode Leaving Condenser Water Temperature`
                Units: C
                Default value: 49.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Reference Heating Mode Leaving Condenser Water Temperature"] = value

    @property
    def reference_heating_mode_entering_condenser_fluid_temperature(self):
        """Get reference_heating_mode_entering_condenser_fluid_temperature.

        Returns:
            float: the value of `reference_heating_mode_entering_condenser_fluid_temperature` or None if not set

        """
        return self[
            "Reference Heating Mode Entering Condenser Fluid Temperature"]

    @reference_heating_mode_entering_condenser_fluid_temperature.setter
    def reference_heating_mode_entering_condenser_fluid_temperature(
            self,
            value=29.44):
        """Corresponds to IDD field `Reference Heating Mode Entering Condenser
        Fluid Temperature`

        Args:
            value (float): value for IDD Field `Reference Heating Mode Entering Condenser Fluid Temperature`
                Units: C
                Default value: 29.44
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Reference Heating Mode Entering Condenser Fluid Temperature"] = value

    @property
    def heating_mode_entering_chilled_water_temperature_low_limit(self):
        """Get heating_mode_entering_chilled_water_temperature_low_limit.

        Returns:
            float: the value of `heating_mode_entering_chilled_water_temperature_low_limit` or None if not set

        """
        return self[
            "Heating Mode Entering Chilled Water Temperature Low Limit"]

    @heating_mode_entering_chilled_water_temperature_low_limit.setter
    def heating_mode_entering_chilled_water_temperature_low_limit(
            self,
            value=12.22):
        """  Corresponds to IDD field `Heating Mode Entering Chilled Water Temperature Low Limit`
        During simultaneous cooling-heating mode

        Args:
            value (float): value for IDD Field `Heating Mode Entering Chilled Water Temperature Low Limit`
                Units: C
                Default value: 12.22
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Heating Mode Entering Chilled Water Temperature Low Limit"] = value

    @property
    def chilled_water_flow_mode_type(self):
        """Get chilled_water_flow_mode_type.

        Returns:
            str: the value of `chilled_water_flow_mode_type` or None if not set

        """
        return self["Chilled Water Flow Mode Type"]

    @chilled_water_flow_mode_type.setter
    def chilled_water_flow_mode_type(self, value="ConstantFlow"):
        """Corresponds to IDD field `Chilled Water Flow Mode Type` Sets chilled
        water flow rate to either constant or variable.

        Args:
            value (str): value for IDD Field `Chilled Water Flow Mode Type`
                Default value: ConstantFlow
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Chilled Water Flow Mode Type"] = value

    @property
    def design_chilled_water_flow_rate(self):
        """Get design_chilled_water_flow_rate.

        Returns:
            float: the value of `design_chilled_water_flow_rate` or None if not set

        """
        return self["Design Chilled Water Flow Rate"]

    @design_chilled_water_flow_rate.setter
    def design_chilled_water_flow_rate(self, value=None):
        """Corresponds to IDD field `Design Chilled Water Flow Rate`

        Args:
            value (float or "Autosize"): value for IDD Field `Design Chilled Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Chilled Water Flow Rate"] = value

    @property
    def design_condenser_water_flow_rate(self):
        """Get design_condenser_water_flow_rate.

        Returns:
            float: the value of `design_condenser_water_flow_rate` or None if not set

        """
        return self["Design Condenser Water Flow Rate"]

    @design_condenser_water_flow_rate.setter
    def design_condenser_water_flow_rate(self, value=None):
        """Corresponds to IDD field `Design Condenser Water Flow Rate`

        Args:
            value (float or "Autosize"): value for IDD Field `Design Condenser Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Condenser Water Flow Rate"] = value

    @property
    def design_hot_water_flow_rate(self):
        """Get design_hot_water_flow_rate.

        Returns:
            float: the value of `design_hot_water_flow_rate` or None if not set

        """
        return self["Design Hot Water Flow Rate"]

    @design_hot_water_flow_rate.setter
    def design_hot_water_flow_rate(self, value=None):
        """Corresponds to IDD field `Design Hot Water Flow Rate`

        Args:
            value (float): value for IDD Field `Design Hot Water Flow Rate`
                Units: m3/s
                IP-Units: gal/min
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Design Hot Water Flow Rate"] = value

    @property
    def compressor_motor_efficiency(self):
        """Get compressor_motor_efficiency.

        Returns:
            float: the value of `compressor_motor_efficiency` or None if not set

        """
        return self["Compressor Motor Efficiency"]

    @compressor_motor_efficiency.setter
    def compressor_motor_efficiency(self, value=1.0):
        """  Corresponds to IDD field `Compressor Motor Efficiency`
        Fraction of compressor electrical energy that must be rejected by the condenser.
        Enter 1.0 or leave this field blank for a hermetic compressor.

        Args:
            value (float): value for IDD Field `Compressor Motor Efficiency`
                Default value: 1.0
                value <= 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Compressor Motor Efficiency"] = value

    @property
    def condenser_type(self):
        """Get condenser_type.

        Returns:
            str: the value of `condenser_type` or None if not set

        """
        return self["Condenser Type"]

    @condenser_type.setter
    def condenser_type(self, value="WaterCooled"):
        """Corresponds to IDD field `Condenser Type`

        Args:
            value (str): value for IDD Field `Condenser Type`
                Default value: WaterCooled
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Condenser Type"] = value

    @property
    def cooling_mode_temperature_curve_condenser_water_independent_variable(
            self):
        """Get
        cooling_mode_temperature_curve_condenser_water_independent_variable.

        Returns:
            str: the value of `cooling_mode_temperature_curve_condenser_water_independent_variable` or None if not set

        """
        return self[
            "Cooling Mode Temperature Curve Condenser Water Independent Variable"]

    @cooling_mode_temperature_curve_condenser_water_independent_variable.setter
    def cooling_mode_temperature_curve_condenser_water_independent_variable(
            self,
            value="EnteringCondenser"):
        """Corresponds to IDD field `Cooling Mode Temperature Curve Condenser
        Water Independent Variable` Sets the second independent variable in the
        three temperature dependent performance curves to either the leaving or
        entering condenser water temperature. Manufacturers express the
        performance of their chillers using either the leaving condenser water
        temperature (to the tower) or the entering condenser water temperature
        (from the tower). Cooling mode is generally a stronger function of
        Entering Condenser Fluid Temperature.

        Args:
            value (str): value for IDD Field `Cooling Mode Temperature Curve Condenser Water Independent Variable`
                Default value: EnteringCondenser
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Cooling Mode Temperature Curve Condenser Water Independent Variable"] = value

    @property
    def cooling_mode_cooling_capacity_function_of_temperature_curve_name(self):
        """Get
        cooling_mode_cooling_capacity_function_of_temperature_curve_name.

        Returns:
            str: the value of `cooling_mode_cooling_capacity_function_of_temperature_curve_name` or None if not set

        """
        return self[
            "Cooling Mode Cooling Capacity Function of Temperature Curve Name"]

    @cooling_mode_cooling_capacity_function_of_temperature_curve_name.setter
    def cooling_mode_cooling_capacity_function_of_temperature_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Cooling Mode Cooling Capacity Function of Temperature Curve Name`
        Cooling capacity as a function of leaving chilled water temperature
        and either entering or leaving condenser fluid temperature
        Table:TwoIndependentVariables object can also be used
        curve = a + b*CWS + c*CWS**2 + d*TCond + e*TCond**2 + f*CWS*TCond
        If ClgModeCondWaterCurveInputVariable = EnteringCondenser, TCond = ECT
        If ClgModeCondWaterCurveInputVariable = LeavingCondenser, TCond = LCT
        CWS = supply (leaving) chilled water temperature(C)
        LCT = leaving condenser fluid temperature(C)
        ECT = entering condenser fluid temperature(C)

        Args:
            value (str): value for IDD Field `Cooling Mode Cooling Capacity Function of Temperature Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Cooling Mode Cooling Capacity Function of Temperature Curve Name"] = value

    @property
    def cooling_mode_electric_input_to_cooling_output_ratio_function_of_temperature_curve_name(
            self):
        """Get cooling_mode_electric_input_to_cooling_output_ratio_function_of_
        temperature_curve_name.

        Returns:
            str: the value of `cooling_mode_electric_input_to_cooling_output_ratio_function_of_temperature_curve_name` or None if not set

        """
        return self[
            "Cooling Mode Electric Input to Cooling Output Ratio Function of Temperature Curve Name"]

    @cooling_mode_electric_input_to_cooling_output_ratio_function_of_temperature_curve_name.setter
    def cooling_mode_electric_input_to_cooling_output_ratio_function_of_temperature_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Cooling Mode Electric Input to Cooling Output Ratio Function of Temperature Curve Name`
        Electric Input Ratio (EIR) as a function of supply (leaving) chilled water temperature
        and leaving condenser fluid temperature.   EIR = 1/COP.
        Table:TwoIndependentVariables object can also be used
        curve = a + b*CWS + c*CWS**2 + d*TCond + e*TCond**2 + f*CWS*TCond
        If ClgModeCondWaterCurveInputVariable = EnteringCondenser, TCond = ECT
        If ClgModeCondWaterCurveInputVariable = LeavingCondenser, TCond = LCT
        CWS = supply (leaving) chilled water temperature(C)
        LCT = leaving condenser fluid temperature(C)
        ECT = entering condenser fluid temperature(C)

        Args:
            value (str): value for IDD Field `Cooling Mode Electric Input to Cooling Output Ratio Function of Temperature Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Cooling Mode Electric Input to Cooling Output Ratio Function of Temperature Curve Name"] = value

    @property
    def cooling_mode_electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name(
            self):
        """Get cooling_mode_electric_input_to_cooling_output_ratio_function_of_
        part_load_ratio_curve_name.

        Returns:
            str: the value of `cooling_mode_electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name` or None if not set

        """
        return self[
            "Cooling Mode Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name"]

    @cooling_mode_electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name.setter
    def cooling_mode_electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Cooling Mode Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name`
        Electric Input Ratio (EIR) as a function of Part Load Ratio (PLR)
        EIR = 1/COP
        Table:OneIndependentVariable object can also be used
        Table:TwoIndependentVariables object can also be used
        Bicubic = a + b*TCond + c*TCond**2 + d*PLR + e*PLR**2 + f*TCond*PLR +g*0 + h*PLR**3+i*0+j*0
        If ClgModeCondWaterCurveInputVariable = EnteringCondenser, TCond = ECT
        If ClgModeCondWaterCurveInputVariable = LeavingCondenser, TCond = LCT
        Normally, a bicubic curve here should be in terms of LCT rather than ECT
        Also, a bicubic curve is more applicable for variable-speed compressor motor drives
        or
        Quadratic = a + b*PLR + c*PLR**2
        PLR = part load ratio (cooling load/steady-state capacity)
        LCT = leaving condenser fluid temperature(C)
        ECT = entering condenser fluid temperature(C)

        Args:
            value (str): value for IDD Field `Cooling Mode Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Cooling Mode Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name"] = value

    @property
    def cooling_mode_cooling_capacity_optimum_part_load_ratio(self):
        """Get cooling_mode_cooling_capacity_optimum_part_load_ratio.

        Returns:
            float: the value of `cooling_mode_cooling_capacity_optimum_part_load_ratio` or None if not set

        """
        return self["Cooling Mode Cooling Capacity Optimum Part Load Ratio"]

    @cooling_mode_cooling_capacity_optimum_part_load_ratio.setter
    def cooling_mode_cooling_capacity_optimum_part_load_ratio(self, value=1.0):
        """  Corresponds to IDD field `Cooling Mode Cooling Capacity Optimum Part Load Ratio`
        Optimum part load ratio where the chiller is most efficient.
        Must be greater than or equal to the Minimum Part Load Ratio
        and less than or equal to the Maximum Part Load Ratio.
        The Min/Max PLR are taken from their assoicated EIR-FPLR curve references.

        Args:
            value (float): value for IDD Field `Cooling Mode Cooling Capacity Optimum Part Load Ratio`
                Default value: 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Cooling Mode Cooling Capacity Optimum Part Load Ratio"] = value

    @property
    def heating_mode_temperature_curve_condenser_water_independent_variable(
            self):
        """Get
        heating_mode_temperature_curve_condenser_water_independent_variable.

        Returns:
            str: the value of `heating_mode_temperature_curve_condenser_water_independent_variable` or None if not set

        """
        return self[
            "Heating Mode Temperature Curve Condenser Water Independent Variable"]

    @heating_mode_temperature_curve_condenser_water_independent_variable.setter
    def heating_mode_temperature_curve_condenser_water_independent_variable(
            self,
            value="LeavingCondenser"):
        """Corresponds to IDD field `Heating Mode Temperature Curve Condenser
        Water Independent Variable` Sets the second independent variable in the
        three temperature dependent performance curves to either the leaving or
        entering condenser water temperature. Manufacturers express the
        performance of their chillers using either the leaving condenser water
        temperature (to the tower) or the entering condenser water temperature
        (from the tower). Heating mode (or Simul Clg/Htg Load) should be a
        function of Leaving Condenser Fluid Temperature Only use
        EnteringCondenser as a last resort in case no performance data exists
        for LeavingCondenser.

        Args:
            value (str): value for IDD Field `Heating Mode Temperature Curve Condenser Water Independent Variable`
                Default value: LeavingCondenser
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self[
            "Heating Mode Temperature Curve Condenser Water Independent Variable"] = value

    @property
    def heating_mode_cooling_capacity_function_of_temperature_curve_name(self):
        """Get
        heating_mode_cooling_capacity_function_of_temperature_curve_name.

        Returns:
            str: the value of `heating_mode_cooling_capacity_function_of_temperature_curve_name` or None if not set

        """
        return self[
            "Heating Mode Cooling Capacity Function of Temperature Curve Name"]

    @heating_mode_cooling_capacity_function_of_temperature_curve_name.setter
    def heating_mode_cooling_capacity_function_of_temperature_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Heating Mode Cooling Capacity Function of Temperature Curve Name`
        Evaporator (cooling) capacity as a function of leaving chilled water temperature
        and leaving condenser fluid temperature when in heating or simultaneous cool/heat mode
        Table:TwoIndependentVariables object can also be used
        curve = a + b*CWS + c*CWS**2 + d*TCond + e*TCond**2 + f*CWS*TCond
        If independent variable = EnteringCondenser, TCond = ECT
        If independent variable = LeavingCondenser, TCond = LCT
        CWS = supply (leaving) chilled water temperature(C)
        LCT = leaving condenser fluid temperature(C)
        ECT = entering condenser fluid temperature(C)

        Args:
            value (str): value for IDD Field `Heating Mode Cooling Capacity Function of Temperature Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Heating Mode Cooling Capacity Function of Temperature Curve Name"] = value

    @property
    def heating_mode_electric_input_to_cooling_output_ratio_function_of_temperature_curve_name(
            self):
        """Get heating_mode_electric_input_to_cooling_output_ratio_function_of_
        temperature_curve_name.

        Returns:
            str: the value of `heating_mode_electric_input_to_cooling_output_ratio_function_of_temperature_curve_name` or None if not set

        """
        return self[
            "Heating Mode Electric Input to Cooling Output Ratio Function of Temperature Curve Name"]

    @heating_mode_electric_input_to_cooling_output_ratio_function_of_temperature_curve_name.setter
    def heating_mode_electric_input_to_cooling_output_ratio_function_of_temperature_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Heating Mode Electric Input to Cooling Output Ratio Function of Temperature Curve Name`
        Electric Input Ratio (EIR) as a function of leaving chilled water temperature when in heating or simultaneous cool/heat mode
        and leaving condenser fluid temperature.   EIR = 1/COP.
        Table:TwoIndependentVariables object can also be used
        curve = a + b*CWS + c*CWS**2 + d*TCond + e*TCond**2 + f*CWS*TCond
        If independent variable = EnteringCondenser, TCond = ECT
        If independent variable = LeavingCondenser, TCond = LCT
        CWS = leaving chilled water temperature(C)
        LCT = leaving condenser fluid temperature(C)
        ECT = entering condenser fluid temperature(C)

        Args:
            value (str): value for IDD Field `Heating Mode Electric Input to Cooling Output Ratio Function of Temperature Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Heating Mode Electric Input to Cooling Output Ratio Function of Temperature Curve Name"] = value

    @property
    def heating_mode_electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name(
            self):
        """Get heating_mode_electric_input_to_cooling_output_ratio_function_of_
        part_load_ratio_curve_name.

        Returns:
            str: the value of `heating_mode_electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name` or None if not set

        """
        return self[
            "Heating Mode Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name"]

    @heating_mode_electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name.setter
    def heating_mode_electric_input_to_cooling_output_ratio_function_of_part_load_ratio_curve_name(
            self,
            value=None):
        """  Corresponds to IDD field `Heating Mode Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name`
        Electric Input Ratio (EIR) as a function of Part Load Ratio (PLR) when in heating or simultaneous cool/heat mode
        EIR = 1/COP
        Table:OneIndependentVariable object can also be used
        Table:TwoIndependentVariables object can also be used
        Bicubic = a + b*LCT + c*LCT**2 + d*PLR + e*PLR**2 + f*LCT*PLR + g*0 + h*PLR**3 + i*0 + j*0
        Normally, a bicubic curve here should be in terms of LCT rather than ECT
        Also, a bicubic curve is more applicable for variable-speed compressor motor drives
        or
        Quadratic = a + b*PLR + c*PLR**2
        PLR = part load ratio (cooling load/steady-state capacity)
        LCT = leaving condenser fluid temperature(C)
        ECT = entering condenser fluid temperature(C)

        Args:
            value (str): value for IDD Field `Heating Mode Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name`
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self[
            "Heating Mode Electric Input to Cooling Output Ratio Function of Part Load Ratio Curve Name"] = value

    @property
    def heating_mode_cooling_capacity_optimum_part_load_ratio(self):
        """Get heating_mode_cooling_capacity_optimum_part_load_ratio.

        Returns:
            float: the value of `heating_mode_cooling_capacity_optimum_part_load_ratio` or None if not set

        """
        return self["Heating Mode Cooling Capacity Optimum Part Load Ratio"]

    @heating_mode_cooling_capacity_optimum_part_load_ratio.setter
    def heating_mode_cooling_capacity_optimum_part_load_ratio(self, value=1.0):
        """  Corresponds to IDD field `Heating Mode Cooling Capacity Optimum Part Load Ratio`
        Optimum part load ratio where the chiller is most efficient when in heating or simultaneous cool/heat mode.
        Must be greater than or equal to the Minimum Part Load Ratio
        and less than or equal to the Maximum Part Load Ratio.
        The Min/Max PLR are taken from their assoicated EIR-FPLR curve references.

        Args:
            value (float): value for IDD Field `Heating Mode Cooling Capacity Optimum Part Load Ratio`
                Default value: 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value
        """
        self["Heating Mode Cooling Capacity Optimum Part Load Ratio"] = value

    @property
    def sizing_factor(self):
        """Get sizing_factor.

        Returns:
            float: the value of `sizing_factor` or None if not set

        """
        return self["Sizing Factor"]

    @sizing_factor.setter
    def sizing_factor(self, value=1.0):
        """Corresponds to IDD field `Sizing Factor` Multiplies the autosized
        capacity and flow rates.

        Args:
            value (float): value for IDD Field `Sizing Factor`
                Default value: 1.0
                if `value` is None it will not be checked against the
                specification and is assumed to be a missing value

        Raises:
            ValueError: if `value` is not a valid value

        """
        self["Sizing Factor"] = value


