#!/usr/bin/env python
# -*- coding: utf-8 -*-
# ==============================================================================
"""
author
======

Novimir Antoniuk Pablant
- npablant@pppl.gov
- novimir.pablant@amicitas.com


purpose
=======
Build the pyidlrpc module.


description
===========

This is a standard python setup script that can be used to build and
install the pyidlrpc module.  This suports all of the standard distuils 
commands. If setuputils is installed, the extended commands will work as 
well.

To install pyidlrpc use the following command:
python setup.py install

To build the pyidlrpc module as a library (*.so) that can be directly
imported from python use:
python setup.py build_ext --inplace


This setup script also includes a custom clean command which will remove
any build files from the source directory.  The optional keword all will
also remove the built library.  For example:
python setup.py clean --all


requirements
============

Python 2.7, 3.0 or greater
Cython

Python Packages:
  numpy
  psutil

All of these dependencies are available on all platforms using your
favorite package manager (macports, eazy-install, apt-get, etc.)
   

"""

import os
import platform
import numpy

version = '0.1.1'

try:
    from setuptools.core import setup
except:
    from distutils.core import setup

from distutils.cmd import Command
from distutils.extension import Extension

# Since pyidlrpc is still in the development stage I am going
# to always use cython if it is available.
use_cython = True
try:
    from Cython.Build import cythonize
except ImportError:
    use_cython = False


# Choose which libraries to use.  I was trying to build a version
# of idlrpc that could handle structures.  For the time being I am
# giving up on this though and just using the version shiped with
# IDL.
use_modified_idlrpc = False


if platform.architecture()[0] == '32bit':
    machine = 'i386'
elif platform.architecture()[0] == '64bit':
    machine = 'x86_64'
else:
    raise Exception("Platform type could not be determined.")



# Find the idl installation directory.
# This can almost always be found using the 'IDL_DIR' environment variable.
# The problem though is that installation will sometimes require the use
# of sudo, which on some systems does not preserve environment variables.
if platform.system() == 'Darwin':
    if 'IDL_DIR' in os.environ:
        idl_dir = os.environ['IDL_DIR']
    elif os.path.exists('/Applications/itt/idl/idl/bin/'):
        idl_dir = '/Applications/itt/idl/idl/'
    elif os.path.exists('/Applications/exelis/idl/bin/'):
        idl_dir = '/Applications/exelis/idl/'
    else:
        raise Exception("Could not find idl installation directory.  Use the --idl-dir option.")
                        
    idl_bin_dir = idl_dir+'/bin/bin.darwin.'+machine
    
elif platform.system() == 'Linux':
    if 'IDL_DIR' in os.environ:
        idl_dir = os.environ['IDL_DIR']
    elif os.path.exists('/usr/local/itt/idl/idl/bin/'):
        idl_dir = '/usr/local/itt/idl/idl/'
    elif os.path.exists('/usr/local/exelis/idl/bin/'):
        idl_dir = '/usr/local/exelis/idl/'
    else:
        raise Exception("Could not find idl installation directory.  Use the --idl-dir option.")
    
    idl_bin_dir = idl_dir+'/bin/bin.linux.'+machine
else:
    raise Exception("Location of IDL bin directory unknown for platform: {}.".format(platform.system()))




if use_modified_idlrpc:
    include_dirs = ['/u/npablant/code/utilities/python/mir_idlrpc/rpc'
                    ,numpy.get_include()]
    library_dirs = ['/u/npablant/code/utilities/python/mir_idlrpc/rpc'
                    ,idl_bin_dir]
    libraries = ['idl_rpc', 'idl']
else:
    include_dirs = [idl_dir+'/external/include'
                    ,numpy.get_include()]
    library_dirs = [idl_bin_dir]
    libraries = ['idl_rpc']


# Define the extension.
#
# There is a bug in cython that does not allow me to create
# certain numpy objects if language="c++".  With c I get
# warnings instead of errors. 
# (I don't know if this is still true 2014-01-06)

if use_cython:
    ext = '.pyx'
else:
    ext = '.c'
    
extensions = [
    Extension('pyidlrpc'
              ,['pyidlrpc'+ext]
              ,include_dirs=include_dirs
              ,libraries=libraries
              ,library_dirs=library_dirs
    #,extra_compile_args=['-Wl,-rpath='+idl_dir+'/external/include', '-Wl,-rpath='+idl_bin_dir]
              )
    ]

# Cythonize.
if use_cython:
    extensions = cythonize(extensions)


class CleanCommand(Command):
    description = "custom clean command that forcefully removes dist/build directories"
    user_options = [("all", None, "Clean all files.")]
    def initialize_options(self):
        self.cwd = None
        self.all = None
    def finalize_options(self):
        self.cwd = os.getcwd()
    def run(self):
        assert os.getcwd() == self.cwd, 'Must be in package root: %s' % self.cwd
        os.system('rm -rf ./build ./pyidlrpc.c ./pyidlrpc.cpp ./setup.cfg')
        if self.all:
            os.system('rm -rf ./pyidlrpc.so ./pyidlrpc.egg-info ./dist')


# Read in the README file to use as a long description.
with open('README') as file:
    long_description = file.read()

# Options for PyPI
classifiers=[
    "Development Status :: 4 - Beta"
    ,"License :: OSI Approved :: MIT License"
    ,"Topic :: Software Development :: Interpreters"
    ,"Programming Language :: Cython"
    ,"Programming Language :: Python"
    ,"Programming Language :: Other"
    ]

# Setup options for setup.
params = {'name':'pyidlrpc'
          ,'version':version
          ,'description':'A library to call IDL (Interactive Data Language) from python.'
          ,'long_description':long_description
          ,'author':'Novimir Antoniuk Pablant'
          ,'author_email':'novimir.pablant@amicitas.com'
          ,'url':'http://amicitas.bitbucket.org/pyidlrpc/'
          ,'license':'MIT'
          ,'ext_modules':extensions
          ,'packages':None
          ,'classifiers':classifiers
          ,'install_requires':['numpy', 'psutil']
          }

# Override the C-extension building so that it knows about '.pyx'
# Cython files
params['cmdclass'] = dict(clean=CleanCommand)

# Call the actual building/packaging function (see distutils docs)
setup(**params)
