# -*- coding: utf-8 -*-

import json
from types import StringTypes

# Base response object
class BaseJSONRPCResponse(object):
    base_response = { 'jsonrpc':'2.0', }
    def dumps(self, **response):
        response.update(self.base_response)
        return json.dumps(response)

# Exception Hierarchy

class RPCError(Exception, BaseJSONRPCResponse):
    def response(self, req_id=None):
        error = {'code':self.code, 'message':self.message}
        return self.dumps(error=error, id=req_id)
class RPCServerError(RPCError):
    code = -32000
    message = 'Server error'
class RPCInvalidRequest(RPCError):
    code = -32600
    message = 'Invalid request'
class RPCMethodNotFound(RPCError):
    code = -32601
    message = 'Method not found'
class RPCInvalidParameters(RPCError):
    code = -32602
    message = 'Invalid parameters'
class RPCInternalError(RPCError):
    code = -32603
    message = 'Internal error'
class RPCParseError(RPCError):
    code = -32700
    message = 'Parse error'

# Core objects

class JSONRPCResponse(BaseJSONRPCResponse):
    def response(self, result, req_id):
        return self.dumps(result=result, id=req_id)

class JSONRPC(object):
    def __init__(self, methods):
        """ initialize the RPC handler

        @param methods : method map
        """
        self.methods = methods

    def __call__(self, *args, **kwargs):
        return self.main_request_handler(*args, **kwargs)

    def main_request_handler(self, data):
        """ load json string and delegate
        to batch or single request handler

        invoke error_response with :
         - RPCParseError if data can't be loaded
         - RPCInvalidRequest if loaded data is empty

        @param data : a json string request
        @return : a json string response or None
        """
        try:
            data = json.loads(data)
        except Exception as error:
            return self.error_response(RPCParseError(error))
        if not data:
            return self.error_response(RPCInvalidRequest())

        if isinstance(data, list):
            response = self.batch_request_handler(data)
        else:
            response = self.single_request_handler(data)
        return response

    def batch_request_handler(self, data):
        """ handle a list of request objects

        call single request handler multiple times
        on items in data, filter out empty results,
        join responses and format as list or return None
        if there is no non-empty responses (notifications)

        @param data : a list of request objects
        @result : a response string or None
        """
        results = list(map(self.single_request_handler, data))
        results = list(filter(bool, results))
        if results :
            return "[%s]" % ",".join(results)

    def single_request_handler(self, data):
        """ handle a request object

        check if the request is valid and get its useful params
        invoke error response with any exception triggered by sanity check

        dispatch then return result response if not quiet
        invoke error response with any exception triggered if not quiet

        @param data : a request object
        @result : a response string or None
        """
        try:
            method, params, req_id, quiet = self.validate_request(data)
        except Exception as error:
            return self.error_response(error)

        try:
            result = self.dispatch_request(method, params)
            if not quiet:
                return self.result_response(result, req_id)
        except Exception as error:
            if not quiet:
                return self.error_response(error, req_id)

    def validate_request(self, request):
        """ return request parameters if valid

        a request object is valid if :
         - request is a dict
         - request['jsonrpc'] == '2.0'
         - request['method'] is a non empty string

        raise RPCInvalidRequest if request is not valid

        a request is a notification (quiet) if
        request['id'] is null or does not exist

        @param request: a request object
        @result : method, params, req_id, quiet
        """
        if not isinstance(request, dict):
            raise RPCInvalidRequest()

        req_id = request.get('id', None)
        quiet = req_id is None

        if request['jsonrpc'] != '2.0':
            raise RPCInvalidRequest()

        method = request.get('method', None)
        params = request.get('params', [])

        if (    not method or
                not isinstance(method, StringTypes)):
            raise RPCInvalidRequest()

        return method, params, req_id, quiet

    def dispatch_request(self, method, params):
        """ call appropriate method in the method map

        @param method : the method name
        @param params : a list or dict of arguments
        """
        try:
            func = self.methods[method]
        except KeyError:
            raise RPCMethodNotFound()
        except Exception as error:
            raise RPCInternalError(error)

        try:
            result = func(*params) if isinstance(params, list) else func(**params)
        except TypeError as error:
            raise RPCInvalidParameters(error)
        except Exception as error:
            raise RPCServerError(error)
        return result

    def result_response(self, result, req_id=None):
        """ bake a JSON RPC result response
        """
        try:
            return JSONRPCResponse().response(result, req_id)
        except Exception as error:
            return self.error_response(
                    RPCInternalError(error),
                    req_id
                    )

    def error_response(self, error, req_id=None):
        """ bake a JSON RPC error response
        """
        if not isinstance(error, RPCError):
            error = RPCServerError(error)
        return error.response(req_id)

