""" Control Widgets.  Presently comprises a Vertical Slider and derivatives.

    HorizontalSlider and HorizontalSlider2 added by Bill Winder
    AreaSlider and AreaSlider2 added by Bill Winder

    Copyright (C) 2008, 2009, 2010 Luke Kenneth Casson Leighton <lkcl@lkcl.net>
    Copyright (C) 2010 - Cedric Gestes <gestes@aldebaran-robotics.com>
    Copyright (C) 2009, 2010 - Bill Winder <wgwinder@gmail.com>


    TODO: All controls with draggable=True do not fire the OnFocus method
    on single click.  the control does not activate the OnFocus method.
    Clicking the handle does fire OnFocus, however.

"""

from pyjamas import Factory
from pyjamas import DOM
from pyjamas.ui import Focus
from pyjamas.ui.Control import Control


class VerticalSlider(Control):

    def __init__(self, min_value, max_value, start_value=None, step=None,
                       **ka):

        ka["StyleName"] = ka.get('StyleName', "gwt-VerticalSlider")

        # XXX FIXME: Focus.createFocusable is here for a reason...
        element = ka.pop('Element', None) or Focus.createFocusable()
        DOM.setStyleAttribute(element, "position", "relative")
        DOM.setStyleAttribute(element, "overflow", "hidden")

        self.handle = DOM.createDiv()
        DOM.appendChild(element, self.handle)

        self.setHandleStyle("1px", "100%", "10px", "#808080")

        Control.__init__(self, element, min_value, max_value, start_value,
                         step, **ka)

        self.addClickListener(self)
        self.addFocusListener(self)
        self.addMouseListener(self)

    def setHandleStyle(self, border, width, height, backgroundColor):
        if border is not None:
            DOM.setStyleAttribute(self.handle, "border", border)
        if width is not None:
            DOM.setStyleAttribute(self.handle, "width", width)
        if height is not None:
            DOM.setStyleAttribute(self.handle, "height", height)
        if backgroundColor is not None:
            DOM.setStyleAttribute(self.handle, "backgroundColor",
                                  backgroundColor)

    def onFocus(self, sender):
        self.addStyleName("gwt-VerticalSlider-focussed")

    def onLostFocus(self, sender):
        self.removeStyleName("gwt-VerticalSlider-focussed")
        self.dragging = False
        DOM.releaseCapture(self.getElement())

    def moveControl(self, mouse_x, mouse_y, first_move=False):
        handle_height = DOM.getIntAttribute(self.handle, "offsetHeight")
        widget_height = self.getOffsetHeight()
        height_range = widget_height - 10 # handle height is hard-coded
        relative_y = mouse_y - (handle_height / 2)
        if relative_y < 0:
            relative_y = 0
        if relative_y >= height_range:
            relative_y = height_range

        relative_y = height_range - relative_y # turn round (bottom to top)

        val_diff = self.max_value - self.min_value
        new_value = ((val_diff * relative_y) / height_range) + self.min_value
        new_value = self.processValue(new_value)

        self.setControlPos(new_value)
        self.setValue(new_value)

    def setControlPos(self, value):
        widget_height = self.getOffsetHeight()
        height_range = widget_height - 10 # handle height is hard-coded
        val_diff = self.max_value - self.min_value
        relative_y = height_range * (value - self.min_value) / val_diff

        # limit the position to be in the widget!
        if relative_y < 0:
            relative_y = 0
        if relative_y >= height_range:
            relative_y = height_range

        relative_y = height_range - relative_y # turn round (bottom to top)

        # move the handle
        DOM.setStyleAttribute(self.handle, "top", "%dpx" % relative_y)
        DOM.setStyleAttribute(self.handle, "position", "absolute")

Factory.registerClass('pyjamas.ui.VerticalSlider', 'VerticalSlider', VerticalSlider)
