# -*- coding: utf-8 -*-
from __future__ import division

import numpy as np

from scipy import integrate
from numpy import pi
from skmonaco import mcmiser

import basis_functions as bf

"""
This module contains routines for calculating potentials in 1D, 2D, 3D case.
This consists of both integrands and integrals.
"""


def b_pot_1d_cont(src, arg, R, h, sigma, basis_func):
    """
    Returns potential as a function of distance from the source.
    """
    resolution = 51
    x = np.array(np.linspace(src - 4*R, src + 4*R, resolution))
    potx = np.array([int_pot_1D(src, arg, current_pos, h, R, sigma, basis_func)
                    for current_pos in x]).flatten()

    pot = np.trapz(potx, x)

    return pot


def int_pot_1D(src, arg, curr_pos, h, R, sigma, basis_func):
    """
    Returns contribution of a single source as a function of distance
    """
    pot = (((arg - curr_pos)**2 + h**2)**0.5 - abs(arg - curr_pos))
    pot *= 1./(2 * sigma)

    # for this formula look at formula (8) from Pettersen et al., 2006
    pot *= basis_func(src, curr_pos, R)

    return pot


def b_pot_2d_cont(x, R, h, sigma, src_type):
    """
    Returns the value of the potential at point (x,0) generated
    by a basis source located at (0,0)
    """
    pot, err = integrate.dblquad(int_pot_2D, -R, R,
                                 lambda x: -R, lambda x: R,
                                 args=(x, R, h, src_type),
                                 epsrel=1e-2, epsabs=0)
    pot *= 1./(2.0*pi*sigma)
    return pot


def int_pot_2D(xp, yp, x, R, h, basis_func):
    """
    Returns contribution of a point xp,yp, belonging to a basis source
    support centered at (0,0) to the potential measured at (x,0),
    integrated over xp,yp gives the potential generated by a
    basis source element centered at (0,0) at point (x,0)

    **Parameters**

    xp, yp : floats
        coordinates of some point laying in the support of a
        basis element centered at (0,0)
    
    x : float
        coordinates of a point (x,0) at which we calculate the potential
    
    R : float
        radius of the basis element
    
    basis_func: callable
        axis-symmetric 2D basis function in the source space

    **Returns**

    pot : float
    """
    y = ((x-xp)**2 + yp**2)**(0.5)
    if y < 0.00001:
        y = 0.00001
    pot = np.arcsinh(h/y)

    pot *= basis_func(xp, yp, [0, 0], R)
    return pot


def b_pot_3d_cont(x, R, h, sigma, basis_func):
    """
    Returns the value of the potential at point (x,y,0) generated
    by a basis source located at (0,0,0)
    """
    pot, err = integrate.tplquad(int_pot_3D, -R, R,
                                 lambda x: -R, lambda x: R,
                                 lambda x, y: -R, lambda x, y: R,
                                 args=(x, R, h, basis_func))
    pot *= 1./(2.0*pi*sigma)  # TODO check the constant
    return pot


def int_pot_3D(xp, yp, zp, x, R, h, basis_func):
    """
    **Parameters**
    
    xp, yp, zp : floats
        coordinates of some point laying in the support of a
        basis element centered at (0,0,0)
    
    x : float
        coordinates of a point (x,0,0) at which we calculate the potential
    
    R : float
        radius of the basis element
    
    basis_func: callable
        axis-symmetric 3D basis function in the source space

    **Returns**

    pot : float
        contribution of a point xp,yp,zp belonging to a basis source
        support centered at (0,0,0) to the potential measured at (x,0,0)
        integrated over xp, yp, zp gives the potential generated by a
        basis source element centered at (0,0,0) at point (x,0,0)
    """
    y = ((x-xp)**2 + yp**2 + zp**2)**0.5
    if y < 0.00001:
        y = 0.00001
    y = 1.0/y

    y *= basis_func(xp, yp, zp, [0, 0, 0], R)
    return y


def b_pot_3d_mc(x, R, h, sigma, basis_func=bf.gauss_rescale_3D):
    """
    Calculate potential in the 3D case using Monte Carlo integration.
    It utilizes the MISER algorithm
    """
    pot, err = mcmiser(int_pot_3D_mc, npoints=1e5,
                       xl=[-R, -R, -R], xu=[R, R, R],
                       nprocs=4, args=(x, R, h, basis_func))
    pot *= 1./(4.0*pi*sigma)
    return pot


def int_pot_3D_mc(xyz, x, R, h, basis_func):
    """
    The same as int_pot_3D, just different input: x,y,z <-- xyz (tuple)
    """
    xp, yp, zp = xyz
    return int_pot_3D(xp, yp, zp, x, R, h, basis_func)


def b_pot_3d_analytic(x, R, h, sigma, basis):
    """
    Calculate 3D potential using analytic scheme assuming step
    spherical-symmetric source.
    This aims to reduce execution time for 3d reconstructions.

    **Parameters**
    
    x : float
        space point where we calculate potential

    R : float
        radius of sphere that contains sources

    sigma : float
        volumetric conductivity (scalar)
    """
    pot = 0
    C = 0.01
    if x > R:
        pot = C/(4*pi*sigma*x)
    else:
        pot = 3*C/(8*pi*sigma*R**3) * (R**2 - x**2 / 2)
    return pot
