# -*- coding: utf-8 -*-
#!/usr/bin/env python
"""
============================================================
Text log analysis (:mod:`pksci.scripts.analyze_txtlog`)
============================================================

.. currentmodule:: pksci.scripts.analyze_txtlog

This module allows one to query a text log file generated
with the :py:class:`~pksci.tools.datautils.TextLogGenerator` class.

.. seealso::

   :py:mod:`~pksci.scripts.txtlog2xlsx`
        command-line script for converting text logs to Excel xlsx files.

.. code-block:: python

   > analyze_txtlog --help
   usage: analyze_txtlog [-h] [--add-filter FIELD_KEY CMP_OP FIELD_VALUE]
                       [--output-field FIELD_KEY] [--print-fraction]
                       [--print-size] [--show-available-fields]
                       [--txtlog-src {None,LAMMPS-experiment}] [--verbose]
                       txtlogs [txtlogs ...]

   positional arguments:
   txtlogs

   optional arguments:
     -h, --help            show this help message and exit
     --add-filter FIELD_KEY CMP_OP FIELD_VALUE
     --output-field FIELD_KEY
     --print-fraction
     --print-size
     --show-available-fields
     --txtlog-src {None,LAMMPS-experiment}
     --verbose

.. autofunction:: analyze_txtlog

Notes
=====

It's possible that updates have removed and/or deprecated some/all
of the example code shown in the next section.
As always, view the script's help to see all available options before
filing a bug report::

    > analyze_txtlog --help

Note that the comparison operator needs to be escaped with a
backslash (\\) so that the shell interprets it as a string.

Also note that the default output will print *all* filtered data to
standard output and so I recommend specifying one or more ``--output-field``
command-line args, or else redirecting (``>>``) the output to a new file,
*i.e.*::

    > analyze_txtlog [--command-line-arg1 ARG1 ...] \\
            TXTLOG [TXTLOG ...] >> filtered_output.txt

or piping (``|``) the output through ``less -S``, *i.e.*::

    > analyze_txtlog [--command-line-arg1 ARG1 ...] \\
            TXTLOG [TXTLOG ...] | less -S
    [PAGER OUTPUT STREAM...]
    ...
    ...

where the ``-S`` flag is the short option for ``less``
option ``--chop-long-lines`` and causes lines longer than the screen width
to be chopped rather than wrapped.


Examples
========

Here are some examples of how I use the script to analyze ``Tdlogs``
from LAMMPS simulation output generated by the
:py:class:`~tasr.LAMMPS.experiments.TdProber` class.

Heuristic Log file analysis
---------------------------

::

    ➜  analyze_txtlog --show-available-fields heuristic_Tdlogs.txt

    ==========================================================================
    'heuristic_Tdlogs.txt' has the following list of recognized field keys:

    ['rundir',
     'Ch',
     'trial',
     'ion_ke_at_t0',
     'pka_ke_at_tko',
     'pka_atomid',
     'pka_dr_at_end',
     'defect_state_at_end',
     'pka_cn_at_end',
     'pka_group_cn_at_end',
     'pka_delta_etot_at_tko',
     'pka_dr_max',
     'pka_speed_at_tko',
     'pka_vx_at_tko',
     'pka_vy_at_tko',
     'pka_vz_at_tko',
     'pka_dx_at_tko',
     'pka_dy_at_tko',
     'pka_dz_at_tko',
     'pka_dr_at_tko',
     'pka_x_at_t0',
     'pka_y_at_t0',
     'pka_z_at_t0',
     'ion_x_at_t0',
     'ion_y_at_t0',
     'ion_z_at_t0',
     'ion_atomid',
     'ion_speed_at_t0',
     'ion_offset_at_t0',
     'ion_cn_at_end',
     'ion',
     'potential',
     'structure',
     'n',
     'm',
     'tube_length',
     'theta_at_t0',
     'phi_at_t0',
     'equilibration_run_steps',
     'dynamics_run_steps',
     'dr_cutoff',
     'model_info']
    ==========================================================================
    ➜  analyze_txtlog heuristic_Tdlogs.txt --add-filter pka_cn_at_end \!= 0 \\
            --print-fraction --print-size

    ==========================================================================
    analyzing txtlog: 'heuristic_Tdlogs.txt'

    data set size: 1944

    filtering field: 'pka_cn_at_end'
    data --> data[np.where(pka_cn_at_end != 0)]

    filtered data set size: 1013
    filtered data set fraction: 0.521
    ==========================================================================
    ➜  analyze_txtlog heuristic_Tdlogs.txt --add-filter pka_cn_at_end \== 0 \\
            --print-fraction --print-size

    ==========================================================================
    analyzing txtlog: 'heuristic_Tdlogs.txt'

    data set size: 1944

    filtering field: 'pka_cn_at_end'
    data --> data[np.where(pka_cn_at_end == 0)]

    filtered data set size: 931
    filtered data set fraction: 0.479
    ==========================================================================
    ➜  analyze_txtlog heuristic_Tdlogs.txt --add-filter pka_cn_at_end \== 1 \\
            --print-fraction --print-size

    ==========================================================================
    analyzing txtlog: 'heuristic_Tdlogs.txt'

    data set size: 1944

    filtering field: 'pka_cn_at_end'
    data --> data[np.where(pka_cn_at_end == 1)]

    filtered data set size: 884
    filtered data set fraction: 0.455
    ==========================================================================
    ➜  analyze_txtlog heuristic_Tdlogs.txt --add-filter pka_cn_at_end \== 2 \\
            --print-fraction --print-size

    ==========================================================================
    analyzing txtlog: 'heuristic_Tdlogs.txt'

    data set size: 1944

    filtering field: 'pka_cn_at_end'
    data --> data[np.where(pka_cn_at_end == 2)]

    filtered data set size: 99
    filtered data set fraction: 0.051
    ==========================================================================
    ➜  analyze_txtlog heuristic_Tdlogs.txt --add-filter pka_cn_at_end \== 3 \\
            --print-fraction --print-size

    ==========================================================================
    analyzing txtlog: 'heuristic_Tdlogs.txt'

    data set size: 1944

    filtering field: 'pka_cn_at_end'
    data --> data[np.where(pka_cn_at_end == 3)]

    filtered data set size: 22
    filtered data set fraction: 0.011
    ==========================================================================
    ➜  analyze_txtlog heuristic_Tdlogs.txt --add-filter pka_cn_at_end \> 3 \\
            --print-fraction --print-size

    ==========================================================================
    analyzing txtlog: 'heuristic_Tdlogs.txt'

    data set size: 1944

    filtering field: 'pka_cn_at_end'
    data --> data[np.where(pka_cn_at_end > 3)]

    filtered data set size: 8
    filtered data set fraction: 0.004
    ==========================================================================

PKA's with CN > 3??? Very interesting... Which runs? Which potentials? Which
irradiation models? What were the CN's? Let's discover!

::

    ➜  analyze_txtlog heuristic_Tdlogs.txt --add-filter pka_cn_at_end \> 3 \\
            --print-fraction --print-size --output-field rundir \\
            --output-field pka_cn_at_end --output-field model_info

    ==========================================================================
    analyzing txtlog: 'heuristic_Tdlogs.txt'

    data set size: 1944

    filtering field: 'pka_cn_at_end'
    data --> data[np.where(pka_cn_at_end > 3)]

    filtered data set size: 8
    filtered data set fraction: 0.004
    filtered data:

    output_fields:
    ['rundir', 'pka_cn_at_end', 'model_info']

    [['run_2013-10-01_070340', '5', 'REBO / C-'],
     ['run_2013-10-02_140927', '4', 'REBO / C-'],
     ['run_2013-10-02_161448', '6', 'REBO / C-'],
     ['run_2013-10-03_070502', '5', 'REBO / C-'],
     ['run_2013-10-03_095646', '5', 'REBO / C-'],
     ['run_2013-10-03_231847', '5', 'REBO / C-'],
     ['run_2013-10-04_122751', '4', 'REBO / e-'],
     ['run_2013-10-04_145135', '5', 'REBO / e-']]
    ==========================================================================


Brute-force, fixed-interval KE ramp, log file analysis
------------------------------------------------------

Let's say the name of the ``Tdlog`` file is
``Tdlog-BNC.tersoff.zbl.C-.txt`` and
we just want to see the data for which the PKA coordination number
is not equal to 0 at the end of the simulation and
only for cases where there was a defect at the end
of the simulation. Also, I'm only want to output the data set
sizes and fitlered fraction.

To filter out this data and print it to the terminal, run::

    ➜  analyze_txtlog Tdlog-BNC.tersoff.zbl_C-.txt \\
            --add-filter pka_cn_at_end \!= 0 \\
            --add-filter defect_state_at_end \== Defect \\
            --print-fraction --print-size

    ==========================================================================
    analyzing txtlog: 'Tdlog-BNC.tersoff.zbl_C-.txt'

    data set size: 160320

    filtering field: 'pka_cn_at_end'
    data --> data[np.where(pka_cn_at_end != 0)]

    filtering field: 'defect_state_at_end'
    data --> data[np.where(defect_state_at_end == Defect)]

    filtered data set size: 91612
    filtered data set fraction: 0.571
    ==========================================================================

    ➜  analyze_txtlog Tdlog-BNC.tersoff.zbl_C-.txt \\
            --add-filter pka_cn_at_end \== 0 \\
            --add-filter defect_state_at_end \== Defect \\
            --print-fraction --print-size

    ==========================================================================
    analyzing txtlog: 'Tdlog-BNC.tersoff.zbl_C-.txt'

    data set size: 160320

    filtering field: 'pka_cn_at_end'
    data --> data[np.where(pka_cn_at_end == 0)]

    filtering field: 'defect_state_at_end'
    data --> data[np.where(defect_state_at_end == Defect)]

    filtered data set size: 15609
    filtered data set fraction: 0.097
    ==========================================================================

    ➜  analyze_txtlog Tdlog-BNC.tersoff.zbl_C-.txt \\
            --add-filter ion_ke_at_t0 \< 50 --add-filter ion_ke_at_t0 \> 40 \\
            --add-filter defect_state_at_end \== Defect \\
            --print-fraction --print-size

    ==========================================================================
    analyzing txtlog: 'Tdlog-BNC.tersoff.zbl_C-.txt'

    data set size: 160320

    filtering field: 'ion_ke_at_t0'
    data --> data[np.where(ion_ke_at_t0 < 50)]

    filtering field: 'ion_ke_at_t0'
    data --> data[np.where(ion_ke_at_t0 > 40)]

    filtering field: 'defect_state_at_end'
    data --> data[np.where(defect_state_at_end == Defect)]

    filtered data set size: 61540
    filtered data set fraction: 0.384
    ==========================================================================
    ➜  analyze_txtlog Tdlog-BNC.tersoff.zbl_C-.txt \\
            --add-filter ion_ke_at_t0 \< 50 --add-filter ion_ke_at_t0 \> 40 \\
            --add-filter defect_state_at_end \== Defect \\
            --add-filter pka_cn_at_end \!= 0 --print-fraction --print-size

    ==========================================================================
    analyzing txtlog: 'Tdlog-BNC.tersoff.zbl_C-.txt'

    data set size: 160320

    filtering field: 'ion_ke_at_t0'
    data --> data[np.where(ion_ke_at_t0 < 50)]

    filtering field: 'ion_ke_at_t0'
    data --> data[np.where(ion_ke_at_t0 > 40)]

    filtering field: 'defect_state_at_end'
    data --> data[np.where(defect_state_at_end == Defect)]

    filtering field: 'pka_cn_at_end'
    data --> data[np.where(pka_cn_at_end != 0)]

    filtered data set size: 50745
    filtered data set fraction: 0.317
    ==========================================================================
    ➜  analyze_txtlog Tdlog-BNC.tersoff.zbl_C-.txt \\
            --add-filter ion_ke_at_t0 \< 50 --add-filter ion_ke_at_t0 \> 40 \\
            --add-filter defect_state_at_end \== Defect \\
            --add-filter pka_cn_at_end \== 0 --print-fraction --print-size

    ==========================================================================
    analyzing txtlog: 'Tdlog-BNC.tersoff.zbl_C-.txt'

    data set size: 160320

    filtering field: 'ion_ke_at_t0'
    data --> data[np.where(ion_ke_at_t0 < 50)]

    filtering field: 'ion_ke_at_t0'
    data --> data[np.where(ion_ke_at_t0 > 40)]

    filtering field: 'defect_state_at_end'
    data --> data[np.where(defect_state_at_end == Defect)]

    filtering field: 'pka_cn_at_end'
    data --> data[np.where(pka_cn_at_end == 0)]

    filtered data set size: 10795
    filtered data set fraction: 0.067
    ==========================================================================

"""
from __future__ import absolute_import, print_function, division

import argparse
import sys

from pprint import pprint

import numpy as np
np.set_printoptions(threshold=np.nan)

from ..tools.datautils import load_fixed_width_data
from ..tools import comparison_symbol_operator_mappings

__all__ = ['analyze_txtlog']


def argparser():
    parser = argparse.ArgumentParser()
    parser.add_argument('--add-filter', nargs=3, action='append',
                        metavar=('FIELD_KEY', 'CMP_OP', 'FIELD_VALUE'),
                        dest='field_filters',
                        help='populate a '
                        '``field_filters`` list with one or more calls to '
                        'this command-line argument. Each call to this '
                        'argument requires a *space* delimited sequence of 3 '
                        'parameters: FIELD_KEY CMP_OP FIELD_VALUE, where '
                        'FIELD_KEY is any one of the recognized '
                        'field key values (which can be determined using '
                        '``--show-available-fields`` command-line argument), '
                        'CMP_OP is a *comparison operator* specified by its '
                        'symbolic representation and can only be one of '
                        '{==,!=,<,<=,>,>=}. FIELD_VALUE is the FIELD_KEY '
                        '*value* used in conjunction with the CMP_OP '
                        'to return the elements of the data array for which '
                        'the following condition is ``True``: '
                        '``numpy.where(FIELD_KEY CMP_OP FIELD_VALUE)``. '
                        'For each text log file analyzed, the filtered data '
                        'stream will be the elements of the input data array '
                        'remaining after applying each field filter in the '
                        'list ``field_filters`` in sequence.')
    #parser.add_argument('--filter-field', default=None, action='append',
    parser.add_argument('--field-stats', default=None, action='append',
                        choices=('mean', 'std', 'var'),
                        help='compute statistics on filtered fields')
    parser.add_argument('--output-field', default=None, action='append',
                        metavar=('FIELD_KEY'), dest='output_fields',
                        help='The default behavior is to print all filtered '
                        'field values to sys.stdout. You can override this '
                        'behavior and specify which output field values '
                        'to show in the output stream by specifying the '
                        'corresponding output field key with one or more '
                        'calls to this argument. You can specify one output '
                        'field key per argument call. And each call will '
                        'append the given output field key to the '
                        '``output_fields`` list. The output field values '
                        'will be in the same sequence order as they '
                        'are found in ``output_fields``. '
                        '(default: %(default)s)')
    parser.add_argument('--print-fraction', action='store_true',
                        help='print fraction of data set remaining post '
                        'filtering. (i.e. ``np.size(filtered_data_set) '
                        '/ np.size(initial_data_set)``)')
    parser.add_argument('--print-size', action='store_true',
                        help='print size of initial and final '
                        '(i.e. filtered) data sets.')
    parser.add_argument('--show-available-fields', action='store_true',
                        help='print list of field keys that are recognized '
                        'in the ``txtlog-src`` generated ``txtlogs``.')
    parser.add_argument('--txtlog-src', default=None,
                        choices=(None, 'LAMMPS-experiment'),
                        help='Specify pksci (sub-)package from whence '
                        'these text logs cameth. Currently, the only source '
                        'of generated text logs is the LAMMPS experiments '
                        'sub-package. Therefore, this command argument is '
                        'not required and can be ``None``. '
                        '(default: %(default)s)')
    parser.add_argument('--verbose', action='store_true',
                        help='verbose output')
    parser.add_argument('txtlogs', nargs='+',
                        help='one or more text log '
                        'files generated from the **same** ``txtlog-src``.')

    return parser


def analyze_txtlog(txtlogs=[], field_filters=None, output_fields=None,
                   field_stats=None, txtlog_src=None, print_fraction=False,
                   print_size=False, show_available_fields=False,
                   verbose=False):
    """
    Analyze a text log file generated with the
    :py:class:`~pksci.tools.datautils.TextLogGenerator` class.


    Parameters
    ----------
    txtlogs : sequence
        list of one or more text log files generated from the
        **same** ``txtlog-src``.
    field_filters : sequence
        ``field_filters`` list with one or more calls to
        this command-line argument. Each call to this
        argument requires a *space* delimited sequence of 3
        parameters: FIELD_KEY CMP_OP FIELD_VALUE, where
        FIELD_KEY is any one of the recognized
        field key values (which can be determined using
        ``--show-available-fields`` command-line argument),
        CMP_OP is a *comparison operator* specified by its
        symbolic representation and can only be one of
        {==,!=,<,<=,>,>=}. FIELD_VALUE is the FIELD_KEY
        *value* used in conjunction with the CMP_OP
        to return the elements of the data array for which
        the following condition is ``True``:
        ``numpy.where(FIELD_KEY CMP_OP FIELD_VALUE)``.
        For each text log file analyzed, the filtered data
        stream will be the elements of the input data array
        remaining after applying each field filter in the
        list ``field_filters`` in sequence.
    output_fields : sequence, optional
        The default behavior is to print all filtered
        field values to sys.stdout. You can override this
        behavior and specify which output field values
        to show in the output stream by specifying the
        corresponding output field key with one or more
        calls to this argument. You can specify one output
        field key per argument call. And each call will
        append the given output field key to the
        ``output_fields`` list. The output field values
        will be in the same sequence order as they
        are found in ``output_fields``.
    txtlog_src : {None, 'LAMMPS-experiment'}, optional
        Specify pksci (sub-)package from whence these text logs cameth.
        Currently, the only source of generated text logs
        is the LAMMPS experiments sub-package. Therefore,
        this command argument is not required and can be
        ``None``.
    print_fraction : bool, optional
        print fraction of data set remaining post
        filtering. (i.e. ``np.size(filtered_data_set)
        / np.size(initial_data_set)``)
    print_size : bool, optional
        print size of initial and final (i.e. filtered) data sets.
    show_available_fields : bool, optional
        print list of field keys that are recognized
        in the ``txtlog-src`` generated ``txtlogs``.
    verbose : bool, optional
        verbose output

    """
    #print('\n' + 74 * '=')
    if txtlog_src is None or txtlog_src == 'LAMMPS-experiment':
        if verbose:
            print('\nLoading field key mappings for text log files '
                  'generated from\nLAMMPS simulations...')
        from tasr.LAMMPS.tools.datautils.log_fields import \
            log_field_key_name_mappings, log_field_name_key_mappings, \
            log_field_key_type_mappings
    else:
        print('There are no field key mappings for text \n'
              'log files generated from this source:\n{}'.format(txtlog_src))
        print('Exiting...')
        sys.exit(1)

    print()
    for findex, f in enumerate(txtlogs, start=1):
        try:
            print(74 * '=')
            field_names, data = \
                load_fixed_width_data(f, comments='--', names=True)
            field_keys = \
                [log_field_name_key_mappings[field_name]
                 for field_name in field_names]
            if show_available_fields:
                print('{!r} '.format(f), end='')
                if len(field_keys) != 0:
                    print('has the following list of recognized field keys:\n')
                    pprint(field_keys, width=70)
                else:
                    print('has no recognized field keys!')
            else:
                print('analyzing txtlog: {!r}\n'.format(f))
                filtered_data = data
                data_size = np.size(data, axis=0)
                if output_fields is not None:
                    for output_field in output_fields[:]:
                        if output_field not in field_keys:
                            output_fields.remove(output_field)
                if not output_fields:
                    output_fields = None
                else:
                    filtered_output_fields = []
                    for output_field in output_fields:
                        output_field_data = \
                            data[log_field_key_name_mappings[
                                output_field]]
                        filtered_output_fields.append(output_field_data)
                    filtered_output_fields = \
                        np.asarray(filtered_output_fields).T

                if print_size:
                    print('data set size: {}\n'.format(data_size))

                for filter_index, (field_key, cmp_symbol, value) in \
                        enumerate(field_filters, start=1):
                    if field_key in field_keys and cmp_symbol in \
                            comparison_symbol_operator_mappings.keys():
                        print('filtering field: {!r}\n'.format(field_key) +
                              'data --> data[np.where({} {} {})]\n'.format(
                                  field_key, cmp_symbol, value))
                        field_data = \
                            filtered_data[
                                log_field_key_name_mappings[field_key]]
                        field_type = log_field_key_type_mappings[field_key]
                        try:
                            cmp_op = \
                                comparison_symbol_operator_mappings[
                                    cmp_symbol]
                            filtered_indices = \
                                np.where(
                                    (cmp_op(field_data, field_type(value))))
                            filtered_data = filtered_data[filtered_indices]
                            if output_fields is not None:
                                filtered_output_fields = \
                                    filtered_output_fields[filtered_indices]
                        except KeyError as e:
                            print(e)
                            continue

                        filtered_data_size = np.size(filtered_data, axis=0)
                        filtered_data_fraction = filtered_data_size / data_size
                        if print_size:
                            print('filtered data set size: '
                                  '{}'.format(len(filtered_data)))
                        if print_fraction:
                            print('filtered data set fraction: '
                                  '{:.3f}'.format(filtered_data_fraction))
                        if filter_index != len(field_filters):
                            print()

                if output_fields is not None:
                    print('filtered data:\n')
                    print('output_fields:')
                    pprint(output_fields, width=74)
                    print()
                    pprint(filtered_output_fields.tolist(), width=74)
                if (not print_fraction and not print_size and
                        not output_fields) or verbose:
                    print('filtered data:\n')
                    pprint(filtered_data.tolist())
        except IOError as e:
            print(e)
        finally:
            if findex == len(txtlogs):
                print(74 * '=')


def main():

    args = argparser().parse_args()
    analyze_txtlog(**vars(args))

if __name__ == '__main__':
    sys.exit(main())
