import random
from abc import abstractmethod, ABCMeta
from faker import Factory


def fake_data(data_type):
    fake = Factory.create()
    try:
        return str(getattr(fake, data_type)())
    except AttributeError:
        print('cannot randomize data for {}. run "pylog list fake" '
              'to print a list of possible types.'.format(data_type))
        raise RuntimeError('cannot randomize data type {}'.format(data_type))


class BaseFormatter(object):

    __metaclass__ = ABCMeta

    @abstractmethod
    def __init__(self, config):
        return

    @abstractmethod
    def generate_data(self):
        return


class CustomFormatter(BaseFormatter):
    """generates log strings in a custom format"""
    def __init__(self, config):
        self.format = config['format']
        self.data = config['data']

    def generate_data(self):
        """returns a log string

        for every item in the format list, if an item in the data dict
        corresponds with it and the field's data equals "$RAND", use faker
        to fake an item for it. else, choose one item from the list randomly.
        if there no item in the data to correspond with the format, it will
        just append to format's field name to the log.

        .. code-block:: python

         example:
          'CustomFormatter': {
              'format': ['name', ' - ', 'level'],
              'data': {
                  'name': $RAND,
                  'level': ['ERROR', 'DEBUG', 'INFO', 'CRITICAL'],
             }
         }

        the output of the above example might be:

        .. code-block:: python

         Sally Fields - ERROR
         or
         Jason Banks - DEBUG
         or
         Danny Milwee - ERROR
         or
         ...
        """
        log = ''
        for field_name in self.format:
            for field, data in self.data.items():
                if field_name == field:
                    if data == '$RAND':
                        log += fake_data(field_name)
                    else:
                        log += random.choice(self.data[field_name])
            if field_name not in self.data.keys():
                log += field_name
        return log


class JsonFormatter(BaseFormatter):
    """generates log strings in json format"""
    def __init__(self, config):
        self.data = config['data']

    def generate_data(self):
        """returns a json string

        all fields in the data dict will be iterated over.
        if $RAND is set in one of the fields, random data will be generate_data
        for that field. If not, data will be chosen from the list.

        .. code-block:: python

         example:
         'JsonFormatter': {
             'data': {
                 'date_time': '$RAND',
                 'level': ['ERROR', 'DEBUG'],
                 'address': '$RAND',
             }
         },

        the output of the above example might be:

        .. code-block:: python

         {'date_time': '2006-11-05 13:31:09', 'name': 'Miss Nona Breitenberg DVM', 'level': 'ERROR'}  # NOQA
         or
         {'date_time': '1985-01-20 11:41:16', 'name': 'Almeda Lindgren', 'level': 'DEBUG'}  # NOQA
         or
         {'date_time': '1973-05-21 01:06:04', 'name': 'Jase Heaney', 'level': 'DEBUG'}  # NOQA
         or
         ...
        """
        log = {}
        for field, data in self.data.items():
            if data == '$RAND':
                log[field] = fake_data(field)
            else:
                log[field] = random.choice(data)
        return log
