# TODO: support additional apache formats (http://ossec-docs.readthedocs.org/en/latest/log_samples/apache/apache.html)  # NOQA
# TODO: maybe consolidate Json and Custom formatters to the same class, to let other custom formatters use them both for each type implemented  # NOQA

import random
from abc import abstractmethod, ABCMeta
from faker import Factory
from format_mappings import InHouseFaker
import json
import uuid


DEFAULT_CUSTOM_FORMAT = [
    'current_date_time', ' ', 'uuid', ' ', 'level', ': ',
    'module', ' - ', 'free_email'
],
DEFAULT_CUSTOM_DATA = {
    'current_date_time': '$RAND',
    'uuid': [str(uuid.uuid1()) for i in xrange(100)],
    'level': ['WARNING', 'ERROR', 'DEBUG', 'INFO', 'CRITICAL'],
    'module': ['module1', 'module2'],
    'free_email': '$RAND',
}


def fake_data(data_type):
    """returns fake data for the data type requested.

    will try and get fake data from the local format mappings.
    if fake data for the specific data type wasn't found, it will try and use
    fake-factory to fake the data.

    :param string data_type: the type of data to fake.
    :rtype: string
    """
    fake = Factory.create()
    try:
        return getattr(InHouseFaker(), 'default')(data_type)
    except KeyError:
        if hasattr(InHouseFaker, data_type):
            return getattr(InHouseFaker(), data_type)()
        elif hasattr(fake, data_type):
            return str(getattr(fake, data_type)())
        # except AttributeError:
        print('cannot randomize data for {}. run "pylog list fake" '
              'to print a list of possible types.'.format(data_type))
        raise RuntimeError('cannot randomize data type {}'.format(
            data_type))


class BaseFormatter(object):
    """base class for all formatters
    """
    __metaclass__ = ABCMeta

    @abstractmethod
    def __init__(self, config):
        return

    @abstractmethod
    def generate_data(self):
        return


class CustomFormatter(BaseFormatter):
    """returns a generated log string in a custom format

    this is also a formatter other formatters
    can rely on to generate application specific logs.
    see the ApacheAccessFormatter class for reference.

    for every item in the format list, if an item in the data dict
    corresponds with it and the field's data equals "$RAND", use faker
    to fake an item for it. else, choose one item from the list randomly.
    if there no item in the data to correspond with the format, it will
    just append to format's field name to the log.

    example:

    .. code-block:: python

     'CustomFormatter': {
         'format': ['name', ' - ', 'level'],
         'data': {
             'name': $RAND,
             'level': ['ERROR', 'DEBUG', 'INFO', 'CRITICAL'],
         }
     }

    the output of the above example might be:

    .. code-block:: python

     Sally Fields - ERROR
     or
     Jason Banks - DEBUG
     or
     Danny Milwee - ERROR
     or
     ...
    """
    def __init__(self, config):
        self.format = config.get('format', DEFAULT_CUSTOM_FORMAT)[0]
        self.data = config.get('data', DEFAULT_CUSTOM_DATA)

    def generate_data(self):
        log = ''
        for field_name in self.format:
            for field, data in self.data.items():
                if field_name == field:
                    if data == '$RAND':
                        log += fake_data(field_name)
                    else:
                        log += random.choice(self.data[field_name])
            if field_name not in self.data.keys():
                log += field_name
        return log


class JsonFormatter(BaseFormatter):
    """generates log strings in json format

    all fields in the data dict will be iterated over.
    if $RAND is set in one of the fields, random data will be generate_data
    for that field. If not, data will be chosen from the list.

    example:

    .. code-block:: python

     'JsonFormatter': {
         'data': {
             'date_time': '$RAND',
             'level': ['ERROR', 'DEBUG'],
             'address': '$RAND',
         }
     },

    the output of the above example might be:

    .. code-block:: python

     {'date_time': '2006-11-05 13:31:09', 'name': 'Miss Nona Breitenberg DVM', 'level': 'ERROR'}  # NOQA
     or
     {'date_time': '1985-01-20 11:41:16', 'name': 'Almeda Lindgren', 'level': 'DEBUG'}  # NOQA
     or
     {'date_time': '1973-05-21 01:06:04', 'name': 'Jase Heaney', 'level': 'DEBUG'}  # NOQA
     or
     ...
    """
    def __init__(self, config):
        self.data = config.get('data', DEFAULT_CUSTOM_DATA)

    def generate_data(self):
        log = {}
        for field, data in self.data.items():
            if data == '$RAND':
                log[field] = fake_data(field)
            else:
                log[field] = random.choice(data)
        return json.dumps(log)


class ApacheAccessFormatter(CustomFormatter):
    """returns an apache-access-log like string"""
    # 192.168.72.177 - - [22/Dec/2002:23:32:19 -0400] "GET /search.php HTTP/1.1" 400 1997 www.yahoo.com "-" "Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.1; ...)" "-"  # NOQA
    # 127.0.0.1 - frank [10/Oct/2000:13:55:36 -0700] "GET /apache_pb.gif HTTP/1.0" 200 2326  # NOQA
    # 192.168.2.20 - - [28/Jul/2006:10:27:10 -0300] "GET /cgi-bin/try/ HTTP/1.0" 200 3395  # NOQA
    # %{IPORHOST:clientip} %{USER:ident} %{USER:auth} \[%{HTTPDATE:timestamp}\] "(?:%{WORD:verb} %{NOTSPACE:request}(?: HTTP/%{NUMBER:httpversion})?|%{DATA:rawrequest})" %{NUMBER:response} (?:%{NUMBER:bytes}|-)  # NOQA
    def __init__(self, config):
        self.format = [
            'ipv4', ' - - [', 'current_day_of_month', '/',
            'current_month_name_short', '/',
            'current_year', ':', 'current_time', ' ',
            'current_time_zone_number', '] "', 'http_verbs',
            ' /', 'uri_path', ' ', 'http_versions', '" ',
            'http_error_codes', ' ', 'random_int'
        ]
        self.data = {
            'ipv4': config['data'].get('ipv4', '$RAND') if config else '$RAND',  # NOQA
            'current_day_of_month': config['data'].get('current_day_of_month', '$RAND') if config else '$RAND',  # NOQA
            'current_month_name_short': config['data'].get('current_month_name_short', '$RAND') if config else '$RAND',  # NOQA
            'current_year': config['data'].get('current_year', '$RAND') if config else '$RAND',  # NOQA
            'current_time': config['data'].get('current_time', '$RAND') if config else '$RAND',  # NOQA
            'current_time_zone_number': config['data'].get('current_time_zone_number', '$RAND') if config else '$RAND',  # NOQA
            'http_versions': config['data'].get('http_versions', '$RAND') if config else '$RAND',  # NOQA
            'http_verbs': config['data'].get('http_verbs', '$RAND') if config else '$RAND',  # NOQA
            'uri_path': config['data'].get('uri_path', '$RAND') if config else '$RAND',  # NOQA
            'http_error_codes': config['data'].get('http_error_codes', '$RAND') if config else '$RAND',  # NOQA
            'random_int': config['data'].get('random_int', '$RAND') if config else '$RAND',  # NOQA
        }


class ApacheErrorFormatter(CustomFormatter):
    """returns an apache-error-log like string"""
    # [Fri Dec 16 01:46:23 2005] [error] [client 1.2.3.4] Directory index forbidden by rule: /home/test/  # NOQA
    # [Mon Dec 19 23:02:01 2005] [error] [client 1.2.3.4] user test: authentication failure for "/~dcid/test1": Password Mismatch  # NOQA
    def __init__(self, config):
        self.format = [
            '[', 'current_day_of_week_short', ' ', 'current_month_name_short',
            ' ',
            'current_day_of_month', ' ', 'current_time', ' ', 'current_year',
            '] [',
            'syslog_error_levels_lower', '] [client ', 'ipv4', '] ',
            'catch_phrase'
        ]
        self.data = {
            'current_day_of_week_short': config['data'].get('current_day_of_week_short', '$RAND') if config else '$RAND',  # NOQA
            'ipv4': config['data'].get('ipv4', '$RAND') if config else '$RAND',  # NOQA
            'current_day_of_month': config['data'].get('current_day_of_month', '$RAND') if config else '$RAND',  # NOQA
            'current_month_name_short': config['data'].get('current_month_name_short', '$RAND') if config else '$RAND',  # NOQA
            'current_year': config['data'].get('current_year', '$RAND') if config else '$RAND',  # NOQA
            'current_time': config['data'].get('current_time', '$RAND') if config else '$RAND',  # NOQA
            'catch_phrase': config['data'].get('catch_phrase', '$RAND') if config else '$RAND',  # NOQA
            'syslog_error_levels_lower': config['data'].get('syslog_error_levels_lower', '$RAND') if config else '$RAND',  # NOQA
        }
