#!/usr/bin/env python
# from pylog import init_logger
import pika
import logging
from abc import abstractmethod, ABCMeta

# FILE DEFAULTS
DEFAULT_FILE_FORMAT = '%(asctime)s %(levelname)s - %(message)s'
DEFAULT_FILE_PATH = 'generated.log'
DEFAULT_MAXBYTES = 10000000
DEFAULT_BACKUPS = 20

# AMQP DEFAULTS
# queue name for packager events
DEFAULT_AMQP_QUEUE = 'myqueue'
# routing key..
DEFAULT_AMQP_ROUTING_KEY = 'myroutingkey'
# broker exchange
DEFAULT_AMQP_EXCHANGE = ''


class BaseTransport(object):

    __metaclass__ = ABCMeta

    @abstractmethod
    def __init__(self, config):
        return

    @abstractmethod
    def configure(self):
        return

    @abstractmethod
    def send(self, client, log):
        return


class UDPTransport(BaseTransport):
    def __init__(self, config):
        try:
            self.host = config['host']
            self.port = config['port']
        except KeyError as ex:
            raise('configuration not complete: {}'.format(ex))

    def configure(self):
        logger = logging.getLogger('pylog')
        handler = logging.handlers.DatagramHandler(self.host, self.port)
        logger.addHandler(handler)
        logger.setLevel(logging.DEBUG)
        return logger

    def send(self, client, log):
        # getattr(logger, level)(message)
        client.debug(log)

    def close(self):
        pass


class StreamTransport(BaseTransport):
    def __init__(self, config):
        pass

    def configure(self):
        logger = logging.getLogger('pylog')
        handler = logging.handlers.StreamHandler()
        logger.addHandler(handler)
        logger.setLevel(logging.DEBUG)
        return logger

    def send(self, client, log):
        client.debug(log)

    def close(self):
        pass


class FileTransport(BaseTransport):
    def __init__(self, config):
        self.file_path = config.get('file', DEFAULT_FILE_PATH)
        self.max_bytes = config.get('maxBytes', DEFAULT_MAXBYTES)
        self.backups_count = config.get('backupsCount', DEFAULT_BACKUPS)

    def configure(self):
        logger = logging.getLogger('pylog')
        handler = logging.handlers.RotatingFileHandler(
            self.file_path, maxBytes=self.max_bytes,
            backupCount=self.backups_count)
        logger.addHandler(handler)
        logger.setLevel(logging.DEBUG)
        return logger

    def send(self, client, log):
        client.debug(log)

    def close(self):
        pass


class AmqpTransport(BaseTransport):
    def __init__(self, config):
        try:
            self.host = config['host']
            self.queue = config['queue']
            self.exchange = config['exchange']
            self.routing_key = config['routing_key']
        except KeyError as ex:
            raise('configuration not complete: {}'.format(ex))

    def configure(self):
        if not self.host:
            raise RuntimeError('no host defined')
        try:
            self.connection = pika.BlockingConnection(
                pika.ConnectionParameters(self.host))
        except:
            return

        client = self.connection.channel()
        client.queue_declare(queue=self.queue, durable=True)
        return client

    def send(self, client, log):
        client.basic_publish(exchange=self.exchange,
                             routing_key=self.routing_key,
                             body=log,
                             properties=pika.BasicProperties(delivery_mode=2))

    # TODO: support connection closing
    def close(self):
        self.connection.close()
