#!/usr/bin/env python

"""
Classes for reading/manipulating/writing FEFF files.
http://leonardo.phys.washington.edu/feff/
XANES and EXAFS input files, and the xmu.dat, ldos.dat output files are
available, for non-spin case at this time.  FEFF input file has parameter tags,
potential definitions and atomic coordinates all in the feff.inp file. These
are each developed separately with the Header, FeffAtoms, FeffPot, and
FeffTags classes, then combined to produce the full feff.inp.
"""

from __future__ import division

__author__ = "Alan Dozier"
__copyright__ = "Copyright 2011, The Materials Project"
__version__ = "1.0.2"
__maintainer__ = "Alan Dozier"
__email__ = "adozier@uky.edu"
__status__ = "Beta"
__date__ = "Oct. 6, 2012"

import re
import itertools
import warnings
import numpy as np

from collections import defaultdict
from pymatgen.symmetry.finder import SymmetryFinder
from operator import itemgetter
from pymatgen.io.cifio import CifParser
from pymatgen.io.io_abc import FeffInput
from pymatgen.util.string_utils import str_aligned, str_delimited
from pymatgen.util.io_utils import zopen, clean_lines
from pymatgen.electronic_structure.core import Spin, Orbital
from pymatgen.electronic_structure.dos import CompleteDos, Dos
from pymatgen.core.lattice import Lattice
from pymatgen.core.structure import Structure


class Header(FeffInput):
    """
    Creates Header for feff.inp file generated by pymatgen.
    Has the following format::

        * This feff.inp file generated by pymatgen, www.materialsproject.org
        TITLE Source: CoO19128.cif
        TITLE Structure Summary: (Co2 O2)
        TITLE Reduced formula: CoO
        TITLE space group: P1,   space number: 1
        TITLE abc: 3.297078 3.297078 5.254213
        TITLE angles: 90.0 90.0 120.0
        TITLE sites: 4
        * 1 Co     0.666666     0.333332     0.496324
        * 2 Co     0.333333     0.666667     0.996324
        * 3 O     0.666666     0.333332     0.878676
        * 4 O     0.333333     0.666667     0.378675

    """

    def __init__(self, struct, comment=None):
        """
        Args:
            struct:
                Structure object, See pymatgen.core.structure.Structure.
            comment:
                comment for first header line
        """
        if struct.is_ordered:
            self._struct = struct
            self._site_symbols = []
            self._natoms = []
            sym = SymmetryFinder(struct)
            data = sym.get_symmetry_dataset()
            self._space_number = data["number"]
            self._space_group = data["international"]
            syms = [site.specie.symbol for site in struct]
            for (s, data) in itertools.groupby(syms):
                self._site_symbols.append(s)
                self._natoms.append(len(tuple(data)))
            self._comment = comment
        else:
            raise ValueError("Structure with partial occupancies cannot be "
                             "converted into atomic coordinates!")

    @staticmethod
    def from_cif_file(cif_file):

        """
        Static method to create Header object from cif_file

        Arg:
            cif_file:
                cif_file path and name
        Returns:
            Header Object

        """
        r = CifParser(cif_file)
        structure = r.get_structures()[0]

        return Header(structure)

    @property
    def structure_symmetry(self):
        """
        Returns space number and space group

        Returns:
            Space number and space group list
        """
        return [self._space_number, self._space_group]

    def set_source(self, source):
        """
        Property set method to set source string

        Arg:
            source:
                String identifying structure source
        """
        self.source = source
        return

    def get_source(self):
        """
        Property set method to return source string.
        """
        return self.source

    @property
    def site_symbols(self):
        """
        Symbols for each site in unit cell.
        """
        return self._site_symbols

    @property
    def struct(self):
        """
        Structure associated with the atomic coordinates.
        """
        return self._struct

    @property
    def space_group(self):
        """
        Returns Space Group symbol
        """
        return self._space_group

    @property
    def space_number(self):
        """
        Returns Space group number
        """
        return self._space_number

    @property
    def cif_file(self):
        """
        Strips path and returns cif file name
        """
        cif = self._cif_file.split("/")
        return cif[-1]

    def get_string(self):
        """
        Returns Header as a string
        """
        to_s = lambda x: "%0.6f" % x
        output = ["* This FEFF.inp file generated by pymatgen"]
        output.append("TITLE Source:  " + self.source)
        output.append("TITLE Structure Summary:  {}"
                      .format(self.struct.composition))
        output.append("TITLE Reduced formula:  {}"
                      .format(self.struct.composition.reduced_formula))
        output.append("TITLE space group: ({}), space number:  ({})"
                      .format(self.space_group, self.space_number))
        output.append("TITLE abc:{}"
                      .format(" ".join([to_s(i).rjust(10)
                                        for i in self.struct.lattice.abc])))
        output.append("TITLE angles:{}"
                      .format(" ".join([to_s(i).rjust(10)
                                        for i in self.struct.lattice.angles])))
        output.append("TITLE sites: {}".format(self.struct.num_sites))
        for i, site in enumerate(self.struct):
            output.append(" ".join(["*", str(i + 1), site.species_string,
                                    " ".join([to_s(j).rjust(12)
                                              for j in site.frac_coords])]))

        return "\n".join(output)

    @staticmethod
    def header_string_from_file(filename):
        """
        Reads Header string from file

        Args:
            filename:
                File name containing the Header data.

        Returns:
            Reads header string.
        """
        with zopen(filename, "r") as fobject:
            f = fobject.readlines()
            feff_header_str = []
            ln = 0
            try:
                feffpmg = f[0].find("pymatgen")
            except:
                feffpmg = 0

            if feffpmg > 0:
                nsites = int(f[7].split()[2])
                for line in f:
                    ln = ln + 1
                    if ln <= nsites + 8:
                        feff_header_str.append(line)
            else:
                end = 0
                for line in f:
                    if (line[0] == "*" or line[0] == "T") and end == 0:
                        feff_header_str.append(line.replace("\r", ""))
                    else:
                        end = 1

        return ''.join(feff_header_str)

    @staticmethod
    def from_string(header_str):
        """
        Reads Header string and returns Header object if header was
        generated by pymatgen.

        Args:
            header_str:
                pymatgen generated feff.inp header

        Returns:
            structure object.
        """

        lines = tuple(clean_lines(header_str.split("\n"), False))
        comment = lines[0]
        try:
            feffpmg = comment.find("pymatgen")
        except:
            feffpmg = 0

        if feffpmg > 0:
            source = lines[1].split()[2]
            natoms = int(lines[7].split()[2])
            basis_vec = lines[5].split()

            a = float(basis_vec[2])
            b = float(basis_vec[3])
            c = float(basis_vec[4])

            lengths = [a, b, c]
            basis_ang = lines[6].split()

            alpha = float(basis_ang[2])
            beta = float(basis_ang[3])
            gamma = float(basis_ang[4])
            angles = [alpha, beta, gamma]

            lattice = Lattice.from_lengths_and_angles(lengths, angles)
            atomic_symbols = []

            for i in xrange(8, 8 + natoms):
                atomic_symbols.append(lines[i].split()[2])

            # read the atomic coordinates
            coords = []

            for i in xrange(natoms):
                toks = lines[i + 8].split()
                coords.append([float(s) for s in toks[3:]])

            struct_fromfile = Structure(lattice, atomic_symbols, coords, False,
                                        False, False)
            struct_fromfile.compound = lines[3].split()[2]

            h = Header(struct_fromfile, comment)
            h.set_source(source)

            return h

        else:

            return 'Header not generated by pymatgen, cannot return header object'

    def __str__(self):
        """
        String representation of Header.
        """
        return self.get_string()

    def write_file(self, filename):
        """
        Writes Header into filename on disk
        """
        with open(filename, "w") as f:
            f.write(str(self) + "\n")


class FeffAtoms(FeffInput):
    """
    Object for representing atomic positions, placed in feff.inp file
    These are oredered as expanding shells.
    """

    def __init__(self, struct, central_atom, comment=None):
        """
        Args:
            struct:
                Structure object. See pymatgen.core.structure.Structure.
            central_atom:
                Symbol for absorbing atom
            comment:
                Optional comment line for atomic coordinates.
        """
        self._central_atom = central_atom
        if struct.is_ordered:
            self._struct = struct
            self._site_symbols = []
            self._natoms = []
            syms = [site.specie.symbol for site in struct]
            for (s, data) in itertools.groupby(syms):
                self._site_symbols.append(s)
                self._natoms.append(len(tuple(data)))
            self.comment = struct.formula if comment == None else comment
        else:
            raise ValueError("Structure with partial occupancies cannot be "
                             "converted into atomic coordinates!")
        unique_pot_atoms = []
        [unique_pot_atoms.append(i) for i in syms
         if not unique_pot_atoms.count(i)]
        self._pot_dict = {}
        i = 0
        for atom in unique_pot_atoms:
            i = i + 1
            self._pot_dict[atom] = i

    @property
    def central_atom(self):
        """
        Returns central atom
        """
        return self._central_atom

    @property
    def pot_dict(self):
        """
        returns dictionary for potential indexes
        """
        return self._pot_dict

    @property
    def site_symbols(self):
        """
        Symbols for each site atomic coordinate for Feff list.
        """
        return self._site_symbols

    @property
    def struct(self):
        """
        Structure associated with the atomic coordinates.
        """
        return self._struct

    @staticmethod
    def atoms_string_from_file(filename):
        """
        Reads atomic shells from file such as feff.inp or ATOMS file
        The lines are arranged as follows:

        x y z   ipot    Atom Symbol   Distance   Number

        with distance being the shell radius and ipot an integer identifying
        the potential used.

        Args:
            filename:
                file name containing atomic coord data.

        Returns:
            Atoms string.
        """
        with zopen(filename, "r") as fobject:
            f = fobject.readlines()
            coords = 0
            atoms_str = []

            for line in f:
                if coords == 0:
                    try:
                        find_atoms = line.find("ATOMS")
                        if find_atoms >= 0:
                            coords = 1
                    except:
                        coords = 0
                if coords == 1:
                    atoms_str.append(line.replace("\r", ""))

        return FeffAtoms.from_string(''.join(atoms_str))

    @staticmethod
    def from_string(data):
        """
        At the moment does nothing.

        From atoms string data generates atoms object
        """
        return data

    def get_string(self, radius=10.):
        """
        Returns a string representation of atomic shell coordinates to be used
        in the feff.inp file.

        Args:
            radius:
                Maximum atomic shell radius to include in atoms list

        Returns:
            String representation of Atomic Coordinate Shells.
        """
        nopts = len(self.struct.species)

        ptatoms = []
        for i in range(0, nopts):
            ptatoms.append(self.struct.species[i].symbol)

        index = ptatoms.index(self.central_atom)
        pt = self.struct.cart_coords[index]
        sphere = Structure.get_sites_in_sphere(self.struct, pt, radius)
        xshift = pt[0]
        yshift = pt[1]
        zshift = pt[2]
        end = len(sphere)
        row = []

        for i in range(0, end):

            atom = sphere[i][0]
            atm = re.sub(r"[^aA-zZ]+", "", atom.species_string)
            ipot = self.pot_dict[atm]
            x = atom.coords[0] - xshift
            y = atom.coords[1] - yshift
            z = atom.coords[2] - zshift
            distance = sphere[i][1]
            row.append(["{:f}".format(x), "{:f}".format(y), "{:f}".format(z),
                        ipot, atm, "{:f}".format(distance), i])

        row_sorted = sorted(row, key=itemgetter(5))
        row_sorted[0][3] = 0

        for i in range(0, end):
            row_sorted[i][6] = i

        row_sorted = str_aligned(row_sorted, ["*       x", "y", "z", "ipot",
                                              "Atom", "Distance", "Number"])
        atom_list = row_sorted.replace("--", "**")

        return ''.join(["ATOMS\n", atom_list, "\nEND\n"])

    def __str__(self):
        """
        String representation of Atoms file.
        """
        return self.get_string()

    def write_file(self, filename):
        """
        Write Atoms list to filename
        """
        with open(filename, "w") as f:
            f.write(str(self) + "\n")


"""**Non-exhaustive** list of valid Feff.inp tags"""

VALID_FEFF_TAGS = ("CONTROL", "PRINT", "ATOMS", "POTENTIALS", "RECIPROCAL",
                   "REAL", "MARKER", "LATTICE", "TITLE", "RMULTIPLIER",
                   "SGROUP", "COORDINATES", "EQUIVALENCE", "CIF", "CGRID",
                   "CFAVERAGE", "OVERLAP", "EXAFS", "XANES", "ELNES", "EXELFS",
                   "LDOS", "ELLIPTICITY", "MULTIPOLE", "POLARIZATION", "RHOZZP",
                   "DANES", "FPRIME", "NRIXS", "XES", "XNCD", "XMCD", "XNCD"
                   "CONTROL", "END", "KMESH", "PRINT", "EGRID", "DIMS",
                   "AFLOP", "EDGE", "COMPTON", "DANES", "FPRIME" "MDFF",
                   "HOLE", "COREHOLE", "S02", "CHBROAD", "EXCHANGE", "FOLP",
                   "NOHOLE", "RGRID", "SCF", "UNFREEZEF", "CHSHIFT", "DEBYE",
                   "INTERSTITIAL", "CHWIDTH", "EGAP", "EPS0", "EXTPOT",
                   "ION", "JUMPRM", "EXPOT", "SPIN", "LJMAX", "LDEC", "MPSE",
                   "PLASMON", "RPHASES", "RSIGMA", "PMBSE", "TDLDA", "FMS",
                   "DEBYA", "OPCONS", "PREP", "RESTART", "SCREEN", "SETE",
                   "STRFACTORS", "BANDSTRUCTURE", "RPATH", "NLEG", "PCRITERIA",
                   "SYMMETRY", "SS", "CRITERIA", "IORDER", "NSTAR", "ABSOLUTE",
                   "CORRECTIONS", "SIG2", "SIG3", "MBCONV", "SFCONV", "RCONV",
                   "SELF", "SFSE", "MAGIC")


class FeffTags(dict, FeffInput):
    """
    feff_tag object for reading and writing PARAMETER files
    """

    def __init__(self, params=dict()):
        """
        Creates a Feff_tag object.

        Args:
            params:
                A set of input parameters as a dictionary.
        """
        super(FeffTags, self).__init__()
        self.update(params)

    def __setitem__(self, key, val):
        """
        Add parameter-val pair to Feff_tag file.  Warns if parameter is not in
        list of valid Feff tags. Also cleans the parameter and val by stripping
        leading and trailing white spaces.
        """
        if key.strip().upper() not in VALID_FEFF_TAGS:
            warnings.warn(key.strip() + " not in VALID_FEFF_TAGS list")
        super(FeffTags, self).__setitem__(key.strip(),
                                          FeffTags.proc_val(key.strip(),
                                                            val.strip())
                                          if isinstance(val, basestring)
                                          else val)

    @property
    def to_dict(self):
        """
            Retruns dictionary of parameters from self object
        """

        return self

    @staticmethod
    def from_dict(d):
        """
            d:
                dictionary of Feff parameters and values
            returns:
                FeffTags object
        """
        i = FeffTags()
        for k, v in d.items():
            i[k] = v

        return i

    def get_string(self, sort_keys=True, pretty=True):
        """
        Returns a string representation of the Feff_tag file.  The reason why
        this method is different from the __str__ method is to provide options
        for pretty printing.

        Args:
            sort_keys:
                Set to True to sort the Feff parameters alphabetically.
                Defaults to False.
            pretty:
                Set to True for pretty aligned output. Defaults to False.
        """
        keys = self.keys()
        if sort_keys:
            keys = sorted(keys)
        lines = []
        for k in keys:
            if isinstance(self[k], list):
                lines.append([k, " ".join([str(i) for i in self[k]])])
            else:
                lines.append([k, self[k]])

        if pretty:
            return str_aligned(lines)
        else:
            return str_delimited(lines, None, "  ")

    def __str__(self):
        return self.get_string(sort_keys=False, pretty=True)

    def write_file(self, filename):
        """
        Write FeffTags to a Feff parameter tag file.

        Args:
            filename:
                filename to write to.
        """
        with open(filename, "w") as f:
            f.write(self.__str__() + "\n")

    @staticmethod
    def from_file(filename="feff.inp"):
        """
        Creates a Feff_tag dictionary from a PARAMETER or feff.inp file.

        Args:
            filename:
                Filename for file

        Returns:
            Feff_tag object
        """
        with zopen(filename, "r") as f:
            lines = list(clean_lines(f.readlines()))
        params = {}
        for line in lines:
            m = re.match("([A-Z]+\d*\d*)\s*(.*)", line)
            if m:
                key = m.group(1).strip()
                val = m.group(2).strip()
                val = FeffTags.proc_val(key, val)
                if key not in ("ATOMS", "POTENTIALS", "END", "TITLE"):
                    params[key] = val
        return FeffTags(params)

    @staticmethod
    def proc_val(key, val):
        """
        Static helper method to convert Feff parameters to proper types, e.g.
        integers, floats, lists, etc.

        Args:
            key:
                Feff parameter key
            val:
                Actual value of Feff parameter.
        """
        list_type_keys = VALID_FEFF_TAGS
        boolean_type_keys = ()
        float_type_keys = ("SCF", "EXCHANGE", "S02", "FMS", "XANES", "EXAFS",
                           "RPATH", "LDOS")
        int_type_keys = ("PRINT", "CONTROL")

        def smart_int_or_float(numstr):
            if numstr.find(".") != -1 or numstr.lower().find("e") != -1:
                return float(numstr)
            else:
                return int(numstr)
        try:
            if key in list_type_keys:
                output = list()
                toks = re.split("\s+", val)

                for tok in toks:
                    m = re.match("(\d+)\*([\d\.\-\+]+)", tok)
                    if m:
                        output.extend([smart_int_or_float(m.group(2))] *
                                      int(m.group(1)))
                    else:
                        output.append(smart_int_or_float(tok))
                return output
            if key in boolean_type_keys:
                m = re.search("^\W+([TtFf])", val)
                if m:
                    if m.group(1) == "T" or m.group(1) == "t":
                        return True
                    else:
                        return False
                raise ValueError(key + " should be a boolean type!")

            if key in float_type_keys:
                return float(val)

            if key in int_type_keys:
                return int(val)

        except:
            return val.capitalize()

        return val.capitalize()

    def diff(self, other):
        """
        Diff function.  Compares two PARAMETER files and indicates which
        parameters are the same and which are not. Useful for checking whether
        two runs were done using the same parameters.

        Args:
            other:
                The other PARAMETER dictionary to compare to.
        Returns:
            Dict of the format {"Same" : parameters_that_are_the_same,
            "Different": parameters_that_are_different} Note that the
            parameters are return as full dictionaries of values.
        """
        similar_param = {}
        different_param = {}
        for k1, v1 in self.items():
            if k1 not in other:
                different_param[k1] = {"FEFF_TAGS1": v1,
                                       "FEFF_TAGS2": "Default"}
            elif v1 != other[k1]:
                different_param[k1] = {"FEFF_TAGS1": v1,
                                       "FEFF_TAGS2": other[k1]}
            else:
                similar_param[k1] = v1
        for k2, v2 in other.items():
            if k2 not in similar_param and k2 not in different_param:
                if k2 not in self:
                    different_param[k2] = {"FEFF_TAGS1": "Default",
                                           "FEFF_TAGS2": v2}
        return {"Same": similar_param, "Different": different_param}

    def __add__(self, other):
        """
        Add all the values of another FeffTags object to this object
        Facilitates the use of "standard" FeffTags
        """
        params = {k: v for k, v in self.items()}
        for k, v in other.items():
            if k in self and v != self[k]:
                raise ValueError("FeffTags have conflicting values!")
            else:
                params[k] = v
        return FeffTags(params)


class FeffPot(FeffInput):
    """
    Object for representing Atomic Potenitals, placed in feff.inp file
    """

    def __init__(self, struct, central_atom, comment=None):
        """
        Args:
            struct:
                Structure object. See pymatgen.core.structure.Structure.
            central_atom:
                Absorbing atom symbol
            comment:
                Optional comment line for atomic coordinates.
        """
        self._central_atom = central_atom
        if struct.is_ordered:
            self._struct = struct
            self._site_symbols = []
            self._natoms = []
            syms = [site.specie.symbol for site in struct]
            for (s, data) in itertools.groupby(syms):
                self._site_symbols.append(s)
                self._natoms.append(len(tuple(data)))
            unique_pot_atoms = []
            [unique_pot_atoms.append(i) for i in syms
             if not unique_pot_atoms.count(i)]
            self._pot_dict = {}
            i = 0
            for atom in unique_pot_atoms:
                i = i + 1
                self._pot_dict[atom] = i

            self.comment = struct.formula if comment == None else comment
        else:

            raise ValueError("Structure with partial occupancies cannot be "
                             "converted into atomic coordinates!")

    @property
    def central_atom(self):
        """
        Returns Central absorbing atom
        """
        return self._central_atom

    @property
    def pot_dict(self):
        """
        Returns dictionary of potential indexes
        """
        return self._pot_dict

    @property
    def site_symbols(self):
        """
        Symbols for each site.
        """
        return self._site_symbols

    @property
    def struct(self):
        """
        Structure associated with the atomic coordinates.
        """
        return self._struct

    @staticmethod
    def pot_string_from_file(filename):
        """

        Reads Potential parameters from a feff.inp or FEFFPOT file.
        The lines are arranged as follows:

          ipot   Z   element   lmax1   lmax2   stoichometry   spinph

        Args:
            filename - file name containing potential data.

        Returns:
            FEFFPOT string.
        """
        with zopen(filename, "r") as f_object:
            f = f_object.readlines()
            ln = -1
            pot_str = ["POTENTIALS\n"]
            pot_tag = -1
            pot_data = 0
            pot_data_over = 1

            for line in f:
                if pot_data_over == 1:
                    ln = ln + 1
                    if pot_tag == -1:
                        try:
                            pot_tag = line.find("POTENTIALS")
                            ln = 0
                        except:
                            pot_tag = -1
                    if pot_tag >= 0 and ln > 0 and pot_data_over > 0:
                        try:
                            if int(line.split()[0]) == pot_data:
                                pot_data = pot_data + 1
                                pot_str.append(line.replace("\r", ""))
                        except:
                            if pot_data > 0:
                                pot_data_over = 0
        return ''.join(pot_str)

    @staticmethod
    def pot_dict_from_string(pot_data):
        """
        Creates atomic symbol/potential number dictionary
        forward and reverse

        Arg:
            pot_data:
                potential data in string format

        Returns:
            forward and reverse atom symbol and potential number dictionaries.
        """

        pot_dict = {}
        pot_dict_reverse = {}
        begin = 0
        ln = -1

        for line in pot_data.split("\n"):

            try:
                if begin == 0 and line.split()[0] == "0":
                    begin = begin + 1
                    ln = 0
                if begin == 1:
                    ln = ln + 1
                if ln > 0:
                    atom = line.split()[2]
                    index = int(line.split()[0])
                    pot_dict[atom] = index
                    pot_dict_reverse[index] = atom
            except:
                pass
        return pot_dict, pot_dict_reverse

    def get_string(self):
        """
        Returns a string representation of potential parameters to be used in
        the feff.inp file,
        determined from structure object.

                The lines are arranged as follows:

          ipot   Z   element   lmax1   lmax2   stoichometry   spinph

        Returns:
            String representation of Atomic Coordinate Shells.
        """

        noelements = len(self.struct.composition.items())
        nopts = len(self.struct.species)

        ptatoms = []
        for i in range(0, nopts):
            ptatoms.append(self.struct.species[i].symbol)

        index = ptatoms.index(self.central_atom)
        center = self.struct.species[index]
        cs = center.symbol
        cz = center.Z
        cindex = self.struct.composition[cs]
        stoic = self.struct.composition.items()[cindex][1]
        ipotrow = [[0, cz, cs, -1, -1, .0001, 0]]

        for i in range(0, noelements):
            center = self.struct.composition.items()[i][0]
            cs = center.symbol
            cz = center.Z
            ipot = self.pot_dict[cs]
            stoic = self.struct.composition.items()[i][1]
            ipotrow.append([ipot, cz, cs, -1, -1, stoic, 0])
        ipot_sorted = sorted(ipotrow, key=itemgetter(0))
        ipotrow = str_aligned(ipot_sorted, ["*ipot", "Z", "tag", "lmax1",
                                            "lmax2", "xnatph(stoichometry)",
                                            "spinph"])
        ipotlist = ipotrow.replace("--", "**")
        ipotlist = ''.join(["POTENTIALS\n", ipotlist])

        return ipotlist

    def __str__(self):
        """
        String representation of FEFFPOT file.
        """
        return self.get_string()

    def write_file(self, filename):
        """
        Write to a filename.

        Args:
            filename:
                filename and path to write potential file to.
        """
        with open(filename, "w") as f:
            f.write(str(self) + "\n")


class FeffLdos(object):
    """
    Parser for ldos files ldos01, ldos02, .....

    .. attribute pdos:

        List of list of pdos objects. Access as pdos[atomindex][orbitalindex]

    .. attribute charge:
        Charge on each ion as a tuple of dict, e.g., ({"p": 0.154,
        "s": 0.078, "d": 0.0, "tot": 0.232}, ...)

    .. attribute efermi:
        Fermi energy

    .. attribute energies:
        energies computed for ldos

    .. attribute ldos:
        ldos for site and orbital, in future will have spin

    .. attribute tdos:
        Total dos

    .. atttribute structure:
        Structure associated with DOS.
    """

    def __init__(self, filename1="feff.inp", filename2="ldos"):
        """
        Args:
            filename1:
                input file of run to obtain structure
            filename2:
                output ldos file of run to obtain dos info, etc.
        """
        self._input_filename = filename1
        self._ldos_filename = filename2
        self.header_str = Header.header_string_from_file(filename1)
        self.pot_string = FeffPot.pot_string_from_file(filename1)
        self._dicts = FeffPot.pot_dict_from_string(self.pot_string)
        self._pot_dict = self._dicts[0]
        self._pot_dict_reverse = self._dicts[1]
        with zopen(self.ldos_filename + "00.dat", "r") as fobject:
            f = fobject.readlines()
        self._efermi = float(f[0].split()[4])
        self._dos_ener = []
        self.ldos = self.ldos_from_file()
        for i in range(0, len(self.ldos[1])):
            self._dos_ener.append(self.ldos[1][i][0])

    @property
    def pot_dict(self):
        """Returns forward potential dictionary i.e., {atom:index}"""
        return self._pot_dict

    @property
    def pot_dict_reverse(self):
        """Returns reversed potential dictionary i.e., {index:atom}"""
        return self._pot_dict_reverse

    @property
    def structure(self):
        """Returns Input Structure from input file"""
        return Header.from_string(self.header_str).struct

    @property
    def input_filename(self):
        """returns name of file to read structure from"""
        return self._input_filename

    @property
    def ldos_filename(self):
        """Returns name of filename group to read ldos data, etc. from"""
        return self._ldos_filename

    @property
    def efermi(self):
        """Returns Fermi energy from Feff ldos00.dat file"""
        return self._efermi

    def ldos_from_file(self):
        """
        Reads in Feff ldos files into numpy array
        then returns ldos dictionary of sites and raw data

        """
        self.ldos = {}
        for i in range(1, len(self.pot_dict) + 1):

            if len(str(i)) == 1:
                self.ldos[i] = np.loadtxt(self.ldos_filename + "0" + str(i) +
                                          ".dat")

            else:
                self.ldos[i] = np.loadtxt(self.ldos_filename + str(i) + ".dat")

        return self.ldos

    @property
    def dos_energies(self):
        """Reads dos energies from first ldos file in to a list"""
        return self._dos_ener

    @property
    def pdos(self):
        """
        Returns constructed pdos from raw ldos data
        """
        all_pdos = []
        vorb = {"s": Orbital.s, "p": Orbital.py, "d": Orbital.dxy,
                "f": Orbital.f0}
        forb = {"s": 0, "p": 1, "d": 2, "f": 3}
        structure = self.structure
        nsites = structure.num_sites
        ldos = self.ldos_from_file()
        dlength = len(ldos[1])

        for i in range(nsites):
            pot_index = self.pot_dict[structure.species[i].symbol]
            all_pdos.append(defaultdict())
            for k, v in vorb.items():
                density = [ldos[pot_index][j][forb[k] + 1]
                           for j in range(dlength)]
                updos = density
                downdos = None
                if downdos:
                    all_pdos[-1][v] = {Spin.up: updos, Spin.down: downdos}
                else:
                    all_pdos[-1][v] = {Spin.up: updos}
        pdos = all_pdos

        return pdos

    @property
    def complete_dos(self):
        """
        Returns a complete dos object which incorporates the total dos and all
        projected dos.
        """
        pdos = self.pdos
        structure = self.structure
        vorb = {0: Orbital.s, 1: Orbital.py, 2: Orbital.dxy, 3: Orbital.f0}
        pdoss = {structure[i]: {v: pdos[i][v]
                                for v in vorb.values()}
                 for i in range(len(pdos))}
        return CompleteDos(structure, self.tdos, pdoss)

    @property
    def tdos(self):
        """
        Returns total DOS by summing all orbitals, spins and sites in unit
        cell
        """
        efermi = self.efermi
        dos_energies = self.dos_energies
        forb = {"s": 0, "p": 1, "d": 2, "f": 3}
        ldos = self.ldos_from_file()
        structure = self.structure
        nsites = structure.num_sites
        dlength = len(ldos[1])
        tdos = [0 for j in range(dlength)]
        for i in range(nsites):
            pot_index = self.pot_dict[structure.species[i].symbol]
            for v in forb.values():
                density = [ldos[pot_index][j][v + 1] for j in range(dlength)]
                for j in range(dlength):
                    tdos[j] = tdos[j] + density[j]
        tdos = {Spin.up: tdos}

        return Dos(efermi, dos_energies, tdos)

    def charge_transfer_from_file(self):
        """
        Returns tuple of dictionaries in order of potential sites
        ({"p": 0.154, "s": 0.078, "d": 0.0, "tot": 0.232}, ...)
        """
        self.cht = []
        for i in range(1, len(self.pot_dict) + 1):

            if len(str(i)) == 1:
                with zopen("{}0{}.dat".format(self.ldos_filename, i), "r") \
                        as fobject:
                    f = fobject.readlines()
                    tot = float(f[1].split()[4])
                    s = float(f[3].split()[2])
                    p = float(f[4].split()[2])
                    d = float(f[5].split()[2])
                    f1 = float(f[6].split()[2])
                self.cht.append({"s": s, "p": p, "d": d, "f": f1, "tot": tot})
            else:
                with zopen(self.ldos_filename + str(i) + ".dat", "r") as fid:
                    f = fid.readlines()
                    tot = float(f[1].split()[4])
                    s = float(f[3].split()[2])
                    p = float(f[4].split()[2])
                    d = float(f[5].split()[2])
                    f1 = float(f[6].split()[2])
                self.cht.append({"s": s, "p": p, "d": d, "f": f1, "tot": tot})

        return tuple(self.cht)


class Xmu(object):
    """
    Parser for data in xmu.dat
        Reads in data from xmu Feff file for plotting
        This file contains the absorption cross-sections
        for the single absorber and absorber in solid.

    Default attributes:

        xmu:
            Photon absorption cross section of absorber atom in material
        mu:
            Photon absorption cross section of single absorber atom
        Energies:
            Energies of data point
        Edge:
            Aborption Edge
        Absorbing atom:
           Species of absorbing atom
        Material:
            Formula of material
        Cif File:
            Cif file number
        Calculation:
            Type of Feff calculation performed

        to_dict:  creates a dictionary representation of data
    """
    def __init__(self, filename="xmu.dat", input_filename="feff.inp"):
        """
        Args:
            filename:
                filename and path for xmu.dat
            input_filename:
                filname and path of feff.inp input file
        """
        self._input_file = input_filename
        self._data = np.loadtxt(filename)

    @property
    def input_file(self):
        """returns filename of FEFF input file normally feff.inp"""
        return self._input_file

    @property
    def energies(self):
        """Returns energies for cross-section plots"""
        energies = []
        for i in range(len(self._data)):
            energy = self._data[i][0]
            energies[len(energies):] = [energy]
        return energies

    @property
    def across_section(self):
        """Returns absobtion cross-section of absorbing atom in solid"""
        across = []
        for i in range(len(self._data)):
            a = self._data[i][3]
            across[len(across):] = [a]
        return across

    @property
    def scross_section(self):
        """Returns aborption cross-section for absorbing atom"""
        scross = []
        for i in range(len(self._data)):
            s = self._data[i][4]
            scross[len(scross):] = [s]
        return scross

    @property
    def cif(self):
        """Returns ciffile identification from Header file"""

        header = Header.header_string_from_file(self.input_file)
        try:
            ciffile_name = header.splitlines()[1].split()[4]
        except:
            ciffile_name = 'structure not from cif file, other source'

        return ciffile_name

    @property
    def calc(self):
        """
        Returns type of Feff calculation, XANES or EXAFS from feff.inp file
        """
        tags = FeffTags.from_file(self.input_file)
        if "XANES" in tags:
            calc = "XANES"
        else:
            calc = "EXAFS"
        return calc

    @property
    def material_formula(self):
        """Returns chemical formula of material from feff.inp file"""
        header = Header.header_string_from_file(self.input_file)
        try:
            form = header.splitlines()[2].split()[3:]
        except:
            form = 'No formula provided'
        formula = str()
        for elements in form:
            formula = formula + str(elements)
        return formula

    @property
    def absorbing_atom(self):
        """Returns absorbing atom symbol from feff.inp file"""

        pots = FeffPot.pot_string_from_file(self.input_file)
        absorbing_atom = pots.splitlines()[1].split()[2]
        return absorbing_atom

    @property
    def edge(self):
        """Returns excitation edge from feff.inp file"""
        edge = FeffTags.from_file(self.input_file)["EDGE"]
        return edge

    @property
    def to_dict(self):
        """returns dictionay of xmu attributes"""
        xmu = {}
        xmu["energies"] = self.energies
        xmu["across"] = self.across_section
        xmu["scross"] = self.scross_section
        xmu["atom"] = self.absorbing_atom
        xmu["edge"] = self.edge
        xmu["cif"] = self.cif
        xmu["calc"] = self.calc
        xmu["formula"] = self.material_formula
        return xmu


class FeffParserError(Exception):
    """
    Exception class for Structure.
    Raised when the structure has problems, e.g., atoms that are too close.
    """

    def __init__(self, msg):
        self.msg = msg

    def __str__(self):
        return "FeffParserError : " + self.msg
