# Copyright 2009-2012 10gen, Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Tests for the son module."""

import unittest
import sys
import pickle
sys.path[0:0] = [""]

from nose.plugins.skip import SkipTest

from bson.py3compat import b
from bson.son import SON


class TestSON(unittest.TestCase):

    def setUp(self):
        pass

    def test_ordered_dict(self):
        a = SON()
        a["hello"] = "world"
        a["mike"] = "awesome"
        a["hello_"] = "mike"
        self.assertEqual(a.items(), [("hello", "world"),
                                     ("mike", "awesome"),
                                     ("hello_", "mike")])

        b = SON({"hello": "world"})
        self.assertEqual(b["hello"], "world")
        self.assertRaises(KeyError, lambda: b["goodbye"])

    def test_to_dict(self):
        a = SON()
        b = SON([("blah", SON())])
        c = SON([("blah", [SON()])])
        d = SON([("blah", {"foo": SON()})])
        self.assertEqual({}, a.to_dict())
        self.assertEqual({"blah": {}}, b.to_dict())
        self.assertEqual({"blah": [{}]}, c.to_dict())
        self.assertEqual({"blah": {"foo": {}}}, d.to_dict())
        self.assertEqual(dict, a.to_dict().__class__)
        self.assertEqual(dict, b.to_dict()["blah"].__class__)
        self.assertEqual(dict, c.to_dict()["blah"][0].__class__)
        self.assertEqual(dict, d.to_dict()["blah"]["foo"].__class__)

    def test_pickle(self):

        simple_son = SON([])
        complex_son = SON([('son', simple_son), ('list', [simple_son, simple_son])])

        for protocol in [0, 1, 2, -1]:
            pickled = pickle.loads(pickle.dumps(complex_son, protocol=protocol))
            self.assertEqual(pickled['son'], pickled['list'][0])
            self.assertEqual(pickled['son'], pickled['list'][1])

    def test_pickle_backwards_compatability(self):
        # For a full discussion see http://bugs.python.org/issue6137
        if sys.version.startswith('3.0'):
            raise SkipTest("Python 3.0.x can't unpickle "
                           "objects pickled in Python 2.x.")

        # This string was generated by pickling a SON object in pymongo
        # version 2.1.1
        pickled_with_2_1_1 = b(
            "ccopy_reg\n_reconstructor\np0\n(cbson.son\nSON\np1\n"
            "c__builtin__\ndict\np2\n(dp3\ntp4\nRp5\n(dp6\n"
            "S'_SON__keys'\np7\n(lp8\nsb."
        )
        son_2_1_1 = pickle.loads(pickled_with_2_1_1)
        self.assertEqual(son_2_1_1, SON([]))

if __name__ == "__main__":
    unittest.main()
