#-*- coding: UTF-8
from __future__ import absolute_import
from pymorphy.contrib import lastnames_ru
from .base import unittest2, MorphTestCase
from .dicts import morph_ru


class LastnameNormalFormTest(MorphTestCase):
    testcase = [
        (u'Титов', u'мр', u'Титов'),
        (u'Титовым', u'мр', u'Титов'),
        (u'Титова', u'мр', u'Титов'),
        (u'Титове', u'мр', u'Титов'),
        (u'Титова', u'жр', u'Титова'),
        (u'Титовой', u'жр', u'Титова'),
        (u'Титову', u'жр', u'Титова'),
        (u'Медведев', u'мр', u'Медведев'),
        (u'Медведевым', u'мр', u'Медведев'),
        (u'Медведеву', u'мр', u'Медведев'),
        (u'Медведева', u'мр', u'Медведев'),
        (u'Медведеве', u'мр', u'Медведев'),
        (u'Медведева', u'жр', u'Медведева'),
        (u'Медведевой', u'жр', u'Медведева'),
        (u'Медведеву', u'жр', u'Медведева'),
        (u'Янукович', u'мр', u'Янукович'),
        (u'Януковичу', u'мр', u'Янукович'),
        (u'Януковичем', u'мр', u'Янукович'),
        (u'Януковиче', u'мр', u'Янукович'),
        (u'Януковича', u'мр', u'Янукович'),
        (u'Янукович', u'жр', u'Янукович'),
        (u'Мутко', u'мр', u'Мутко'),
        (u'Мутко', u'жр', u'Мутко'),
        (u'Штирлиц', u'мр', u'Штирлиц'),
        (u'Штирлицем', u'мр', u'Штирлиц'),
        (u'Штирлицу', u'мр', u'Штирлиц'),
        (u'Штирлице', u'мр', u'Штирлиц'),
        (u'Штирлиц', u'жр', u'Штирлиц'),
        (u'Тихонов', u'мр', u'Тихонов'),
        (u'Тихонову', u'мр', u'Тихонов'),
        (u'Тихонове', u'мр', u'Тихонов'),
        (u'Тихоновым', u'мр', u'Тихонов'),
        (u'Тихонова', u'жр', u'Тихонова'),
        (u'Тихоновой', u'жр', u'Тихонова'),
        (u'Сысоев', u'мр', u'Сысоев'),
        (u'Сысоевым', u'мр', u'Сысоев'),
        (u'Сысоеву', u'мр', u'Сысоев'),
        (u'Сысоеве', u'мр', u'Сысоев'),
        (u'Сысоева', u'жр', u'Сысоева'),
        (u'Сысоевой', u'жр', u'Сысоева'),
        (u'Достоевский', u'мр', u'Достоевский'),
        (u'Достоевскому', u'мр', u'Достоевский'),
        (u'Достоевским', u'мр', u'Достоевский'),
        (u'Достоевском', u'мр', u'Достоевский'),
        (u'Достоевская', u'жр', u'Достоевская'),
        (u'Достоевской', u'жр', u'Достоевская'),
        (u'Достоевскую', u'жр', u'Достоевская'),
        (u'Табуретов', u'мр', u'Табуретов'),
        (u'Табуретову', u'мр', u'Табуретов'),
        (u'Табуретова', u'мр', u'Табуретов'),
        (u'Табуретовым', u'мр', u'Табуретов'),
        (u'Табуретове', u'мр', u'Табуретов'),
        (u'Табуретова', u'жр', u'Табуретова'),
        (u'Табуретову', u'жр', u'Табуретова'),
        (u'Табуретовой', u'жр', u'Табуретова'),
        (u'Крамской', u'мр', u'Крамский'),
        (u'Крамского', u'мр', u'Крамский'),
        (u'Крамскому', u'мр', u'Крамский'),
        (u'Крамском', u'мр', u'Крамский'),
        (u'Крамская', u'жр', u'Крамская'),
        (u'Крамской', u'жр', u'Крамская'),
        (u'Крамскую', u'жр', u'Крамская'),
        (u'Куцых', u'мр', u'Куцых'),
        (u'Куцых', u'жр', u'Куцых'),
        (u'Путин', u'мр', u'Путин'),
        (u'Путина', u'мр', u'Путин'),
        (u'Путину', u'мр', u'Путин'),
        (u'Путине', u'мр', u'Путин'),
        (u'Путиным', u'мр', u'Путин'),
        (u'Путина', u'жр', u'Путина'),
        (u'Путину', u'жр', u'Путина'),
        (u'Коваль', u'мр', u'Коваль'),
        (u'Ковалю', u'мр', u'Коваль'),
        (u'Ковалем', u'мр', u'Коваль'),
        (u'Ковале', u'мр', u'Коваль'),
        (u'Коваля', u'мр', u'Коваль'),
        (u'Коваль', u'жр', u'Коваль'),
        (u'Живаго', u'мр', u'Живаго'),
        (u'Живаго', u'жр', u'Живаго'),
        (u'Шевченко', u'мр', u'Шевченко'),
        (u'Шевченко', u'жр', u'Шевченко'),
        (u'Жук', u'мр', u'Жук'),
        (u'Жука', u'мр', u'Жук'),
        (u'Жуком', u'мр', u'Жук'),
        (u'Жуку', u'мр', u'Жук'),
        (u'Жуке', u'мр', u'Жук'),
        (u'Жук', u'жр', u'Жук'),
        (u'Козлов', u'мр', u'Козлов'),
        (u'Козлову', u'мр', u'Козлов'),
        (u'Козловым', u'мр', u'Козлов'),
        (u'Козлове', u'мр', u'Козлов'),
        (u'Козлова', u'жр', u'Козлова'),
        (u'Козлову', u'жр', u'Козлова'),
        (u'Козловой', u'жр', u'Козлова'),
        (u'Шоу', u'мр', u'Шоу'),
        (u'Шоу', u'жр', u'Шоу'),
        (u'Толстой', u'мр', u'Толстый'),
        (u'Толстым', u'мр', u'Толстый'),
        (u'Толстому', u'мр', u'Толстый'),
        (u'Толстого', u'мр', u'Толстый'),
        (u'Толстом', u'мр', u'Толстый'),
        (u'Толстая', u'жр', u'Толстая'),
        (u'Толстой', u'жр', u'Толстая'),
        (u'Толстую', u'жр', u'Толстая'),
        (u'Герцен', u'мр', u'Герцен'),
        (u'Герцена', u'мр', u'Герцен'),
        (u'Герценом', u'мр', u'Герцен'),
        (u'Герцену', u'мр', u'Герцен'),
        (u'Герцене', u'мр', u'Герцен'),
        (u'Герцен', u'жр', u'Герцен'),
        (u'Левитан', u'мр', u'Левитан'),
        (u'Левитану', u'мр', u'Левитан'),
        (u'Левитаном', u'мр', u'Левитан'),
        (u'Левитане', u'мр', u'Левитан'),
        (u'Левитан', u'жр', u'Левитан'),
        (u'Берия', u'мр', u'Берия'),
        (u'Берию', u'мр', u'Берия'),
        (u'Берии', u'мр', u'Берия'),
        (u'Берией', u'мр', u'Берия'),
        (u'Берия', u'жр', u'Берия'),
        (u'Берии', u'жр', u'Берия'),
        (u'Берию', u'жр', u'Берия'),
        (u'Гулиа', u'мр', u'Гулиа'),
        (u'Гулиа', u'жр', u'Гулиа'),
        (u'Зыков', u'мр', u'Зыков'),
        (u'Зыкова', u'мр', u'Зыков'),
        (u'Зыкову', u'мр', u'Зыков'),
        (u'Зыковым', u'мр', u'Зыков'),
        (u'Зыкове', u'мр', u'Зыков'),
        (u'Зыкова', u'жр', u'Зыкова'),
        (u'Зыковой', u'жр', u'Зыкова'),
        (u'Зыкову', u'жр', u'Зыкова'),
        (u'Ильвес', u'мр', u'Ильвес'),
        (u'Ильвеса', u'мр', u'Ильвес'),
        (u'Ильвесу', u'мр', u'Ильвес'),
        (u'Ильвеса', u'мр', u'Ильвес'),
        (u'Ильвесом', u'мр', u'Ильвес'),
        (u'Ильвесе', u'мр', u'Ильвес'),
        (u'Адамкус', u'мр', u'Адамкус'),
        (u'Адамкуса', u'мр', u'Адамкус'),
        (u'Адамкусу', u'мр', u'Адамкус'),
        (u'Адамкуса', u'мр', u'Адамкус'),
        (u'Адамкусом', u'мр', u'Адамкус'),
        (u'Адамкусе', u'мр', u'Адамкус'),
        (u'Годманис', u'мр', u'Годманис'),
        (u'Годманиса', u'мр', u'Годманис'),
        (u'Годманису', u'мр', u'Годманис'),
        (u'Годманиса', u'мр', u'Годманис'),
        (u'Годманисом', u'мр', u'Годманис'),
        (u'Годманисе', u'мр', u'Годманис'),
        (u'Акопян', u'мр', u'Акопян'),
        (u'Акопяна', u'мр', u'Акопян'),
        (u'Акопяну', u'мр', u'Акопян'),
        (u'Акопяна', u'мр', u'Акопян'),
        (u'Акопяном', u'мр', u'Акопян'),
        (u'Акопяне', u'мр', u'Акопян'),
        (u'Акопан', u'мр', u'Акопан'),
        (u'Акопане', u'мр', u'Акопан'),
        (u'Акопану', u'мр', u'Акопан'),
        (u'Акопане', u'мр', u'Акопан'),
        (u'Акопаном', u'мр', u'Акопан'),
        (u'Акопане', u'мр', u'Акопан'),
        (u'Акопянц', u'мр', u'Акопянц'),
        (u'Акопянца', u'мр', u'Акопянц'),
        (u'Акопянцу', u'мр', u'Акопянц'),
        (u'Акопянца', u'мр', u'Акопянц'),
        (u'Акопянцем', u'мр', u'Акопянц'),
        (u'Акопянце', u'мр', u'Акопянц'),
        (u'Кравчук', u'мр', u'Кравчук'),
        (u'Кравчука', u'мр', u'Кравчук'),
        (u'Кравчуку', u'мр', u'Кравчук'),
        (u'Кравчука', u'мр', u'Кравчук'),
        (u'Кравчуком', u'мр', u'Кравчук'),
        (u'Кравчуке', u'мр', u'Кравчук'),
        (u'Цапок', u'мр', u'Цапок'),
        (u'Цапка', u'мр', u'Цапок'),
        (u'Цапку', u'мр', u'Цапок'),
        (u'Цапком', u'мр', u'Цапок'),
        (u'Собчак', u'мр', u'Собчак'),
        (u'Собчака', u'мр', u'Собчак'),
        (u'Собчаку', u'мр', u'Собчак'),
        (u'Собчака', u'мр', u'Собчак'),
        (u'Собчаком', u'мр', u'Собчак'),
        (u'Собчаке', u'мр', u'Собчак'),
        (u'Петровичу', u'мр', u'Петрович'), # пример из документации
        (u'Кроссового', u'мр', u'Кроссовый'), # пример из документации
        (u'Махно', u'мр', u'Махно'),
        (u'Махно', u'жр', u'Махно'),
        (u'Бегунец', u'мр', u'Бегунец'),
        (u'Бегунца', u'мр', u'Бегунец'),
        (u'Бегунцу', u'мр', u'Бегунец'),
        (u'Бегунцом', u'мр', u'Бегунец'),
        (u'Бегунце', u'мр', u'Бегунец'),
        (u'Бегунец', u'жр', u'Бегунец'),
        (u'', u'мр', u''),
    ]

    def test_normal_form(self):
        for lastname, gender_tag, expected_lastname in self.testcase:
            res = lastnames_ru.normalize(morph_ru, lastname.upper(), gender_tag)
            self.assertEqualRu(res.capitalize(), expected_lastname)

    def test_improper_guess(self):
        res = lastnames_ru.normalize(morph_ru, u'Ильвесом'.upper(), u'мр')
        self.assertEqualRu(res.capitalize(), u'Ильвес')



class LastnameMisoperationsTest(unittest2.TestCase):
    testcase = [
        (u'Кроссовый', u'мр', u'Кроссов'),
        (u'Петровичу', u'жр', u'Петрович'),
    ]

    def test_misoperations(self):
        for word, gender_tag, wrong in self.testcase:
            res = lastnames_ru.normalize(morph_ru, word.upper(), gender_tag)
            self.assertNotEqual(res.capitalize(), wrong)


class LastnameInflectTest(unittest2.TestCase):
    testcase = [
        (u'Суворову', u'жр,дт', u'Суворовой'),
        (u'Суворову', u'мр,дт', u'Суворову'),
        (u'Суворов', u'им,ед', u'Суворов'),
    ]

    def test_inflect(self):
        for word, gram_form, expected in self.testcase:
            res = lastnames_ru.inflect(morph_ru, word.upper(), gram_form)
            self.assertEqual(res.capitalize(), expected)


class LastnameGraminfoTest(unittest2.TestCase):
    testcase = [
        (u'Суворову', (u'мр,дт', u'жр,вн',)),
        (u'Козловых', (u'мр-жр,мн,рд', u'мр-жр,мн,вн', u'мр,ед,им', u'жр,ед,им')),
    ]

    def test_get_graminfo(self):
        for word, expected in self.testcase:
            gram_info = lastnames_ru.get_graminfo(word.upper())

            expectations_met = 0
            for form in gram_info:
                form_tokens = [token.strip() for token in form.get('info', '').split(',')]
                for expected_form in expected:
                    expected_tokens = [token.strip() for token in expected_form.split(',')]

                    expectations_met += 1
                    for token in expected_tokens:
                        if token not in form_tokens:
                            expectations_met -= 1
                            break

            self.assertGreaterEqual(expectations_met, len(expected))


class LastnamePluralizeTest(unittest2.TestCase):

    def test_pluralize(self):
        testcase = [
            (u'Колобков', u'', u'Колобковы'),
            (u'Колобков', u'тв', u'Колобковыми'),
            (u'Колобкову', u'мр', u'Колобковым'), # исходная форма - 'дт', результат - 'дт'
            (u'Колобкову', u'жр', u'Колобковых'), # 'вн'
            (u'Достоевский', u'', u'Достоевские'),
            (u'Достоевскому', u'мр', u'Достоевским'),
            (u'Достоевский', u'тв', u'Достоевскими'),
            (u'Достоевской', u'жр', u'Достоевских'), # 'рд'
            (u'Цапок', u'', u'Цапки'),
            (u'Цапок', u'жр,тв', u'Цапками'),
            (u'Цапка', u'', u'Цапков'), # 'рд'
            (u'Цапке', u'', u'Цапках'), # 'пр'
            (u'Беглых', u'', u'Беглых'), # та же форма во мн. числе
            (u'Каменских', u'', u'Каменских'),
            (u'Мутко', u'', u'Мутко'),
            (u'Акопян', u'', u'Акопян'),
            (u'Гулиа', u'', u'Гулиа'),
        ]

        for lastname, gram_form, expected in testcase:
            pluralized = lastnames_ru.pluralize(morph_ru, lastname.upper(), gram_form)
#            print lastname, pluralized, expected
            self.assertEqual(pluralized, expected.upper())

    def test_pluralize_inflected(self):
        testcase = [
            (u'Попугаев', u'мр', 1, u'Попугаев'),
            (u'Попугаев', u'мр', 2, u'Попугаевых'),
            (u'Попугаев', u'мр', 5, u'Попугаевых'),
            (u'Попугаева', u'жр', 1, u'Попугаева'),
            (u'Попугаева', u'жр', 2, u'Попугаевых'),
            (u'Попугаева', u'жр', 5, u'Попугаевых'),
        ]

        for lastname, num, gender_tag, expected in testcase:
            pluralized = lastnames_ru.pluralize_inflected(morph_ru, lastname.upper(), num, gender_tag)
#            print lastname, pluralized, expected
            self.assertEqual(pluralized, expected.upper())

if __name__ == '__main__':
    # Упс, относительные импорты сломали возможность запускать скрипт отдельно.
    # А обычные мы использовать не можем, т.к. morph_tests не ставится
    # вместе с pymorphy.
    unittest2.main()
