# -*- coding: UTF-8 -*-
#
# The MIT License
# 
# Copyright (c) 2008 Felix Schwarz <felix.schwarz@oss.schwarz.eu>
# 
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
# 
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

from socket import socket
from unittest import TestCase

from pymta.command_parser import ParserImplementation
from pymta.session import SMTPSession


class CommandParsingTest(TestCase):

    def setUp(self):
        allowed_commands = SMTPSession(command_parser=None).get_all_allowed_internal_commands()
        self.parser = ParserImplementation(allowed_commands)
    
    def test_parse_command_without_arguments(self):
        self.assertEqual(('QUIT', None), self.parser.parse('QUIT'))
        self.assertEqual(('RSET', None), self.parser.parse('RSET'))
    
    def test_parse_helo(self):
        self.assertEqual(('HELO', 'foo.example.com'), 
                         self.parser.parse('HELO foo.example.com'))
        # This command is syntactically invalid but the validity of specific
        # commands should not be checked in the parser.
        self.assertEqual(('helo', 'foo example.com'), 
                         self.parser.parse('helo foo example.com'))
    
    def test_strip_parameters(self):
        self.assertEqual(('HELO', 'foo.example.com'), 
                         self.parser.parse('HELO   foo.example.com   '))
    
    def test_parse_commands_with_colons(self):
        self.assertEqual(('MAIL FROM', 'foo@example.com'), 
                        self.parser.parse('MAIL FROM: foo@example.com'))
        self.assertEqual(('MAIL FROM', 'foo@example.com'), 
                        self.parser.parse('MAIL FROM:foo@example.com'))
        self.assertEqual(('MAIL FROM', 'foo@example.com'), 
                        self.parser.parse('MAIL FROM:  foo@example.com   '))
        self.assertEqual(('RCPT TO', 'foo@example.com, bar@example.com'), 
                        self.parser.parse('RCPT TO:foo@example.com, bar@example.com'))
    
    def test_parse_auth_plain(self):
        self.assertEqual(('AUTH PLAIN', 'AGZvbwBiYXI='), 
                         self.parser.parse('AUTH PLAIN AGZvbwBiYXI='))


