#!/usr/bin/env python
# -*- coding: iso8859-1 -*-

# Copyright (C) 2005-6  Iigo Serna
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


u"""pynakotheka - (C) 2005-6, by Iigo Serna

pynakotheka is a simple python script which generates static HTML photo albums.
Released under GNU Public License, read COPYING for more details.

Usage:\tpynakotheka.py <options> [source_dir [target_dir]]

Arguments:
    source_dir       Location of the source photos, defaults to current dir
    target_dir       Directory where to save the gallery, defaults to current dir

Options:
    -h, --help       Show this text
    -q, --quiet      Don't show progress information messages, default SHOW
    -c, --color      Show information messages with colors, default NO COLOR

    -o, --copy-originals
                     Copy original images to the gallery, default NO
    -s, --thumbsize  Thumbnail size, defaults to 250
    -i, --imagesize  Image size, defaults to 640
    -d, --templates-dir
                     Path to templates directory, defaults to
                     "$PREFIX/share/pynakotheka/templates"
    -t, --template   Template to use, defaults to "default"

    --clean=all|images|html
                     Remove all files / images / html and style files
                     generated by the program
"""


__author__ = 'Iigo Serna'
__revision__ = '1.0.2'


import os
import os.path
import sys
import time
import datetime
import getopt
from stat import ST_CTIME
from htmlentitydefs import entitydefs
from xml.dom.minidom import parse

import Image
import EXIF
from Cheetah.Template import Template


######################################################################
##### Some defaults
######################################################################
DESCFILE = 'album.xml'
IMAGES_EXTS = ('.jpeg', '.jpg', '.png', '.gif')
CHUNKSIZE = 64 * 1024
MINITHUMB_SIZE = 150
TEMPLATES = ('view_albums.tmpl', 'view_thumbs.tmpl',
             'view_image.tmpl', 'view_tree.tmpl')
STYLE_FILES = ('style.css', 'background.jpeg', 'transparent_black.gif')

cfg = None


######################################################################
##### Utilities
######################################################################
mymap = {}
for k, v in entitydefs.items():
    mymap[v] = '&%s;' % k

def txt2html(txt):
    buf = ''
    for c in u2txt(txt):
        if c in mymap.keys():
            buf += mymap[c]
        else:
            buf += c
    return buf

def txt2link(txt):
    buf = ''
    for c in u2txt(txt2u(txt)):
        if c in mymap.keys():
            buf += '%%%x' % ord(c)
        else:
            buf += c
    return buf

def txt2u(buf):
    if type(buf) == type(unicode('')):
        return buf
    codecs_lst = ('ascii', 'latin-1', 'cp850')
    for c in codecs_lst:
        try:
            buf2 = unicode(buf, c)
        except UnicodeDecodeError:
            pass
        else:
            break
    else:
        buf2 = unicode(buf, c, 'replace')
    return buf2

def u2txt(buf):
#     tbl = maketrans('',
#                     'AAAAEEEEIIIIOOOOUUUUNCaaaaeeeeiiiioooouuuunc')
#     buf = translate(buf, tbl)
#     for c in '':
#         buf = buf.replace(c, '')
    buf = buf.encode('latin-1', 'ignore')
    return buf

def colorize(color, text):
    """Return colorized text"""
    col = '\033[0;3'
    if not color: return text
    elif color == 'black': return col + str(0) + 'm' + text + col + str(7) + 'm'
    elif color == 'red': return col + str(1) + 'm' + text + col + str(7) + 'm'
    elif color == 'green': return col + str(2) + 'm' + text + col + str(7) + 'm'
    elif color == 'yellow': return col + str(3) + 'm' + text + col + str(7)+ 'm'
    elif color == 'blue': return col + str(4) + 'm' + text + col + str(7) + 'm'
    elif color == 'magenta': return col + str(5) + 'm' + text + col + str(7)+'m'
    elif color == 'cyan': return col + str(6) + 'm' + text + col + str(7) + 'm'
    elif color == 'white': return col + str(7) + 'm' + text + col + str(7) + 'm'

def no_colorize(color, text):
    return text

def do_print(color, text):
    if type(text) == type(u''):
        text = text.encode('latin-1', 'replace')
    sys.stdout.write(colorize(color, text))
    sys.stdout.flush()
    
def print_info(text):
    do_print('white', text)

def print_important(text):
    do_print('red', text)

def print_warning(text):
    do_print('yellow', text)

def print_sect(text):
    do_print('blue', text)

def print_subsect(text):
    do_print('green', text)

def print_ok(text):
    do_print('white', text)

def copy_file(src, dest):
    infile = open(src, 'rb')
    outfile = open(dest, 'wb')
    buf = infile.read(CHUNKSIZE)
    while buf:
        outfile.write(buf)
        buf = infile.read(CHUNKSIZE)
    infile.close()
    outfile.close()


######################################################################
##### Directory class
######################################################################
class Dir:
    def __init__(self, basedir, path):
        self.relpath = path
        self.abspath = os.path.join(basedir, path)
        self.ctime = os.path.getctime(self.abspath)

    def __repr__(self):
        return u"""Dir(%s)""" % self.abspath

    def __parse_descfile(self, descfile):
        desc = { 'title': '', 'description': '', 'sample_photo' : '',
                 'omitted': False, 'omitted_files': [] }
        try:
            dom = parse(descfile)
        except:
            return desc
        album = dom.firstChild
        if album.nodeName != 'album':
            return desc
        for node in album.childNodes:
            if node.nodeType != node.ELEMENT_NODE:
                continue
            if node.nodeName == 'title':
                desc['title'] = node.firstChild.data.strip()
            if node.nodeName == 'description':
                desc['description'] = node.firstChild.data.strip()
            if node.nodeName == 'sample_photo':
                desc['sample_photo'] = node.firstChild.data.strip()
            if node.nodeName == 'omit_album':
                if node.firstChild.data.strip().upper() == 'TRUE':
                    desc['omitted'] = True
                else:
                    desc['omitted'] = False
            if node.nodeName == 'omitted_files':
                omitted_files = []
                for subnode in node.childNodes:
                    if subnode.nodeType != subnode.ELEMENT_NODE:
                        continue
                    if subnode.nodeName == 'omitted_file':
                        omitted_files.append(subnode.firstChild.data.strip())
                desc['omitted_files'] = omitted_files
        return desc

    def search_descfile(self):
        descfile = os.path.join(self.abspath, DESCFILE)
        if os.path.exists(descfile):
            return self.__parse_descfile(descfile)

    def search_subdirs(self):
        dirs = [f for f in os.listdir(self.abspath) \
                if os.path.isdir(os.path.join(self.abspath, f))]
        try:
            dirs.remove(os.pardir)
            dirs.remove(os.curdir)
        except ValueError:
            pass
        return dirs

    def search_images(self):
        files = [f for f in os.listdir(self.abspath) \
                 if os.path.isfile(os.path.join(self.abspath, f)) and \
                    os.path.splitext(f)[-1].lower() in IMAGES_EXTS]
        return files


######################################################################
##### Album
######################################################################
class Album:
    def __init__(self, parent, path):
        self.parent = parent   # parent album
        self.path = path       # relative path
        self.dirname = os.path.basename(path)  # directory name
        self.srcdir = None
        if self.parent:
            self.depth = self.parent.depth + 1
        else:
            self.depth = 0
        self.valid = False
        self.title = ''
        self.description = ''
        self.albums = []
        self.images = []
        self.num_albums = 0
        self.num_images = 0
        self.total_albums = 0
        self.total_images = 0
        self.index_type = 'subalbums'
        self.sample_photo_path = ''

    def __repr__(self):
        return """Album("%s", %d/%d subalbums, %d/%d images)""" % \
               (self.title, self.num_albums, self.total_albums,
                self.num_images, self.total_images)

    ##################################################
    # parse, validate, show, get_tree
    def cmp_albums(self, x, y):
        """sort albums. First albums containing subalbums, then by ctime"""
        if x.num_albums != 0 and y.num_albums == 0:
            return -1
        elif x.num_albums == 0 and y.num_albums != 0:
            return 1
        else:
            return cmp(x.srcdir.ctime, y.srcdir.ctime)

    def cmp_images(self, x, y):
        """sort by image EXIF date or by file ctime, else by file name"""
        if x.info.timestamp != y.info.timestamp:
            x, y = x.info.timestamp, y.info.timestamp
        else:
            x, y = x.basename.lower(), y.basename.lower()
        return cmp(x, y)

    def parse(self):
        if not cfg.quiet:
            print_subsect('  ' * self.depth + '  Processing: "%s"\n' % \
                          (self.path or os.path.basename(cfg.srcdir), ))
        # initialize source directory
        self.srcdir = Dir(cfg.srcdir, self.path)
        self.title = txt2u(os.path.basename(self.path))
        if not self.title:
            self.title = txt2u(os.path.basename(cfg.srcdir))
        # subalbums
        for diralbum in self.srcdir.search_subdirs():
            album = Album(self, os.path.join(self.path, diralbum))
            valid = album.parse()
            if valid:
                self.valid = True
            self.albums.append(album)
        # images
        images = [Photo(self, img) for img in self.srcdir.search_images()]
        self.images = [img for img in images if img.info != None]
        self.images.sort(cmp=self.cmp_images)
        # album description
        desc = self.srcdir.search_descfile()
        if desc:
            if desc['title']:
                self.title = desc['title']
            if desc['description']:
                self.description = desc['description']
            if desc['sample_photo']:
                abspath = os.path.join(cfg.srcdir, self.path,
                                       u2txt(desc['sample_photo']))
                basename, ext = os.path.splitext(u2txt(desc['sample_photo']))
                if os.path.isfile(abspath) and ext.lower() in IMAGES_EXTS:
                    img_name = basename + '-%d.jpeg' % MINITHUMB_SIZE
                    self.sample_photo_path = os.path.join(self.path, img_name)
            if desc['omitted']:
                self.valid = False
                if not cfg.quiet:
                    print_ok('  ' * self.depth + '  Album omitted\n')
                return self.valid
            if desc['omitted_files']:
                imgs = {}
                for img in self.images:
                    imgs[img.filename] = img
                for f in desc['omitted_files']:
                    f = u2txt(f)
                    if f in imgs.keys() and imgs[f].album == self:
                        self.images.remove(imgs[f])
        # end
        if self.images:
            self.valid = True
        if not cfg.quiet:
            if self.valid:
                print_ok('  ' * self.depth + '  Album: "' + self.title + '"\n')
            else:
                print_ok('  ' * self.depth + '  No valid Album\n')
        return self.valid

    def validate(self):
        """validate and sort subalbums"""
        to_delete = []
        num_albums = 0
        num_images = 0
        for subalbum in self.albums:
            if not subalbum.valid:
                if not cfg.quiet:
                    print_ok('  Removing "%s"\n' % subalbum.srcdir.relpath)
                # can't remove here because we'd alter working self.albums
                to_delete.append(subalbum)
            else:
                subalbum.validate()
                num_albums += subalbum.total_albums
                num_images += subalbum.total_images
        else:
            for subalbum in to_delete:
                self.albums.remove(subalbum)
        self.num_albums = len(self.albums)
        self.num_images = len(self.images)
        self.total_albums = num_albums + self.num_albums
        self.total_images = num_images + self.num_images
        if self.num_albums == 0: # and self.num_images > 0:
            self.index_type = 'thumbs'
        else:
            self.index_type = 'subalbums'
        # get sample photo
        if self.num_images > 0:
            if self.sample_photo_path == '':
                img_name = self.images[0].basename + '-%d.jpeg' % MINITHUMB_SIZE
                self.sample_photo_path = os.path.join(self.path, img_name)
        else:
            for subalbum in self.albums:
                if self.sample_photo_path == '' and \
                       subalbum.sample_photo_path != '':
                    self.sample_photo_path = subalbum.sample_photo_path
                    break
        # sort albums
#        self.albums.sort(cmp=lambda x, y: cmp(x.title.lower(), y.title.lower()))
        self.albums.sort(cmp=self.cmp_albums)

    def show(self):
        if not cfg.quiet:
            print_subsect('  ' * self.depth + '  %s\n' % unicode(self)) 
        for subalbum in self.albums:
            subalbum.show()

    def get_tree(self):
        url = self.srcdir.relpath + '/index.html'
        tree = [(self.depth, self.title, url , self.num_images)]
        for subalbum in self.albums:
            url = subalbum.srcdir.relpath + '/index.html'
            tree.extend(subalbum.get_tree())
        return tree

    ##################################################
    # create destination directories
    def create_target_dirs(self):
        try:
            os.makedirs(os.path.join(cfg.destdir, self.path))
        except OSError, err:
            if err.errno == 17: # File exists
                pass
            else:
                print_warning('ERROR: %s\n' % err)
                sys.exit(2)
        for subalbum in self.albums:
            subalbum.create_target_dirs()

    ##################################################
    # images
    def generate_images(self):
        if not self.images:
            if not cfg.quiet:
                print_subsect('\r' + '  ' * self.depth + '  Album: "%s" -> ' % \
                              (self.path or os.path.basename(cfg.srcdir), ))
                print_ok('0 images')
        to_delete = []
        for i, img in enumerate(self.images):
            if not cfg.quiet:
                print_subsect('\r' + '  ' * self.depth + '  Album: "%s" -> '% \
                              (self.path or os.path.basename(cfg.srcdir), ))
                print_ok('%d/%d images' % (i+1, self.num_images))
            st = img.generate_tn()
            if not st:
                to_delete.append(img)
            else:
                st = img.generate_photo()
                if not st:
                    to_delete.append(img)
        else:
            for img in to_delete:
                self.images.remove(img)
                self.num_images -= 1
                self.total_images -= 1
        if not cfg.quiet:
            print_ok('\n')
        # walk subalbums
        for subalbum in self.albums:
            subalbum.generate_images()

    def copy_original_images(self):
        if not self.images:
            if not cfg.quiet:
                print_subsect('\r' + '  ' * self.depth + '  Album: "%s" -> ' % \
                              (self.path or os.path.basename(cfg.srcdir), ))
                print_ok('0 images')
        for i, img in enumerate(self.images):
            if not cfg.quiet:
                print_subsect('\r' + '  ' * self.depth + '  Album: "%s" -> '% \
                              (self.path or os.path.basename(cfg.srcdir), ))
                print_ok('%d/%d images' % (i+1, self.num_images))
            st = img.copy_original_photo()
            if not st:
                self.images.remove(img)
                self.num_images -= 1
                self.total_images -= 1
        if not cfg.quiet:
            print_ok('\n')
        # walk subalbums
        for subalbum in self.albums:
            subalbum.copy_original_images()

    ##################################################
    # html
    def __generate_html_subalbums(self):
        if not cfg.quiet:
            print_ok('Album')
        albums_path = []
        node = self.parent
        while node != None:
            albums_path.append(node)
            node = node.parent
        albums_path.reverse()
        albums_file = os.path.join(cfg.destdir, self.path, 'view_albums.html')
        tmpl_file = os.path.join(cfg.tmpldir, 'view_albums.tmpl')
        html = Template(file=tmpl_file)
        html.txt2html = txt2html
        html.txt2link = txt2link
        html.album = self
        html.albums_path = albums_path
        html.timestamp = datetime.datetime.now().strftime('%A %d %B %Y %H:%M')
        htmlfile = open(albums_file, 'w')
        htmlfile.write(html.respond())
        htmlfile.close()

    def __generate_html_thumbs(self):
        if not cfg.quiet:
            print_ok(', Thumbnails')
        thumbs_file = os.path.join(cfg.destdir, self.path, 'view_thumbs.html')
        tmpl_file = os.path.join(cfg.tmpldir, 'view_thumbs.tmpl')
        html = Template(file=tmpl_file)
        html.txt2html = txt2html
        html.txt2link = txt2link
        html.album = self
        if self.num_albums == 0:
            html.back_albums_path = '../index.html'
        else:
            html.back_albums_path = 'index.html'
        html.tn_size = cfg.thumbsize
        html.img_size = cfg.imagesize
        html.timestamp = datetime.datetime.now().strftime('%A %d %B %Y %H:%M')
        htmlfile = open(thumbs_file, 'w')
        htmlfile.write(html.respond())
        htmlfile.close()

    def __generate_html_index(self):
        if self.index_type == 'thumbs':
            orig_file = os.path.join(cfg.destdir, self.path,
                                     'view_thumbs.html')
        elif self.index_type == 'subalbums':
            orig_file = os.path.join(cfg.destdir, self.path,
                                     'view_albums.html')
        else:
            raise ValueError, self.index_type
        index_file = os.path.join(cfg.destdir, self.path, 'index.html')
        copy_file(orig_file, index_file)

    def __generate_html_images(self):
        if not cfg.quiet:
            print_ok(', 0 Images')
        tmpl_file = os.path.join(cfg.tmpldir, 'view_image.tmpl')
        for i, im in enumerate(self.images):
            if not cfg.quiet:
                print_ok('\r' + '  ' * self.depth + '  Building html views: ' +
                         'Album, Thumbnails, %d/%d Images' % \
                         (i+1, self.num_images))
            html = Template(file=tmpl_file)
            html.txt2html = txt2html
            html.txt2link = txt2link
            html.width = cfg.imagesize
            html.copy_originals = cfg.originals
            html.album = self
            html.image = im
            html.info = im.info
            html.image_idx = i + 1
            if i == 0:
                html.prev_image = None
            else:
                html.prev_image = self.images[i-1]
            if i == self.num_images - 1:
                html.next_image = None
            else:
                html.next_image = self.images[i+1]
            html.timestamp = datetime.datetime.now().strftime('%A %d %B %Y %H:%M')
            htmlfile = open(im.html_file, 'w')
            htmlfile.write(html.respond())
            htmlfile.close()

    def generate_html(self):
        if not cfg.quiet:
            print_subsect('  ' * self.depth + '  Album: "%s":\n' % \
                          (self.path or os.path.basename(cfg.srcdir), ))
            print_ok('  ' * self.depth + '  Building html views: ')
        self.__generate_html_subalbums()
        self.__generate_html_thumbs()
        self.__generate_html_index()
        self.__generate_html_images()
        if not cfg.quiet:
            print_subsect('\n')
        for subalbum in self.albums:
            subalbum.generate_html()

    ##################################################
    # cleaning
    def clean_html(self):
        if not cfg.quiet:
            print_ok('  ' * (self.depth+1) + '  Album: "%s"\n' % \
                     (self.path or os.path.basename(cfg.srcdir), ))
        for subalbum in self.albums:
            subalbum.clean_html()
        try:
            os.remove(os.path.join(cfg.destdir, self.path, 'view_albums.html'))
        except OSError, err:
            print err
        try:
            os.remove(os.path.join(cfg.destdir, self.path, 'view_thumbs.html'))
        except OSError, err:
            print err
        try:
            os.remove(os.path.join(cfg.destdir, self.path, 'index.html'))
        except OSError, err:
            print err
        for im in self.images:
            try:
                os.remove(os.path.join(cfg.destdir, self.path, im.html_file))
            except OSError, err:
                print err
        try:
            os.rmdir(self.path)
        except OSError: # not empty
            pass

    def clean_images(self):
        if not cfg.quiet:
            print_ok('  ' * (self.depth+1) + '  Album: "%s"\n' % \
                     (self.path or os.path.basename(cfg.srcdir), ))
        for subalbum in self.albums:
            subalbum.clean_images()
        for img in self.images:
            try:
                os.remove(img.tn_file)
            except OSError, err:
                print err
            try:
                os.remove(img.tn100_file)
            except OSError, err:
                print err
            try:
                os.remove(img.img_file)
            except OSError, err:
                print err
            if cfg.originals:
                try:
                    os.remove(img.orig_file)
                except OSError, err:
                    print err
        try:
            os.rmdir(self.path)
        except OSError, err: # not empty
            pass


######################################################################
##### Photo
######################################################################
class Photo:
    def __init__(self, album, filename):
        self.album = album
        self.filename = filename
        self.basename = os.path.splitext(filename)[0]
        self.src_image = os.path.join(cfg.srcdir, self.album.path,
                                      self.filename)
        self.tn_file = os.path.join(cfg.destdir, self.album.path,
                                    self.basename + '-%d.jpeg' % cfg.thumbsize)
        self.tn100_file = os.path.join(cfg.destdir, self.album.path,
                                       self.basename + '-%d.jpeg' % \
                                       MINITHUMB_SIZE)
        self.img_file = os.path.join(cfg.destdir, self.album.path,
                                     self.basename + '-%d.jpeg' % cfg.imagesize)
        self.orig_file = os.path.join(cfg.destdir, self.album.path,
                                      self.filename)
        self.html_file = os.path.join(cfg.destdir, self.album.path,
                                      'view_%s.html' % self.basename)
        self.info = ImageInfo(self.src_image)
        st = self.info.fill_data()
        if not st:
            print_warning('ERROR:"%s" image not valid' % self.src_image)
            self.info = None

    def __repr__(self):
        return """Photo("%s" from Album "%s")""" % \
               (self.filename, self.album.title)

    def generate_tn(self):
        if os.path.exists(self.tn_file):
            return True
        try:
            size = self.info.size
            if size[0] > size[1]:
                new_size = (cfg.thumbsize, cfg.thumbsize)
                new_size2 = (MINITHUMB_SIZE, MINITHUMB_SIZE)
            else:
                value = int(cfg.thumbsize*float(size[0])/size[1])
                new_size = ((value, value))
                value2 = int(MINITHUMB_SIZE*float(size[0])/size[1])
                new_size2 = ((value2, value2))
            im = Image.open(self.src_image)
            if im.mode != "RGB":
                im = im.convert("RGB")
            im.thumbnail(new_size, Image.ANTIALIAS)
            im.save(self.tn_file, 'JPEG')
            im2 = im.copy()
            im2.thumbnail(new_size2, Image.BILINEAR)
            im2.save(self.tn100_file, 'JPEG')
        except IOError:
            print_warning('\nWARNING: cannot create thumbnail for "%s"\n' % \
                          self.src_image)
            return False
        else:
            return True

    def generate_photo(self):
        if os.path.exists(self.img_file):
            return True
        size = self.info.size
# don't copy image because dimensions could be rare and corrupt view style
#         if size[0] <= cfg.imagesize or size[1] <= cfg.imagesize:
#             try:
#                 copy_file(self.src_image, self.img_file)
#             except IOError:
#                 print_warning('\nWARNING: cannot create image for "%s"\n' % \
#                               self.src_image)
#                 return False
#             else:
#                 return True
#         else:
        try:
            if size[0] > size[1]:
                new_size = (int(cfg.imagesize),
                            int(cfg.imagesize*size[1]/float(size[0])))
            else:
                h = int(cfg.imagesize*size[0]/float(size[1]))
                new_size = (int(h*size[0]/float(size[1])), int(h))
            im = Image.open(self.src_image)
            if im.mode != "RGB":
                im = im.convert("RGB")
            im = im.resize(new_size)
            im.save(self.img_file, 'JPEG')
        except IOError:
            del(im)
            print_warning('\nWARNING: cannot create image for "%s"\n' % \
                          self.src_image)
            return False
        else:
            del(im)
            return True


    def copy_original_photo(self):
        if os.path.exists(self.orig_file):
            return True
        try:
            copy_file(self.src_image, self.orig_file)
        except IOError:
            print_warning('\nWARNING: cannot copy image "%s"\n' % \
                          self.src_image)
            return False
        else:
            return True


######################################################################
##### Image Information
######################################################################
class ImageInfo:
    MyExifDict = { 'timestamp': 'Image DateTime',
                   'orientation': 'Image Orientation',
                   'aperture': 'EXIF FNumber',
                   'exposure_time': 'EXIF ExposureTime',
                   'exposure_program': 'EXIF ExposureProgram',
                   'exposure_bias': 'EXIF ExposureBiasValue',
                   'focal_length': 'EXIF FocalLength',
                   'brightness': 'EXIF BrightnessValue',
                   'flash': 'EXIF Flash',
                   'ISOSpeedRatings': 'EXIF ISOSpeedRatings',
                   'camera_vendor': 'Image Make',
                   'camera_model': 'Image Model' }

    def __init__(self, fullpath):
        self.fullpath = fullpath
        self.filename = os.path.basename(fullpath)
        self.timestamp = None
        self.filesize = 0
        self.format = ''
        self.size = (0, 0)
        self.orientation = ''
        self.aperture = ''
        self.exposure_time = ''
        self.exposure_program = ''
        self.exposure_bias = ''
        self.focal_length = ''
        self.brightness = ''
        self.flash = ''
        self.isospeedratings = ''
        self.camera = ''

    def fill_data(self):
        try:
            self.filename = os.path.basename(self.fullpath)
            im = Image.open(self.fullpath)
            timestamp = os.stat(self.fullpath)[ST_CTIME]
            filesize = os.path.getsize(self.fullpath)
        except IOError, err:
            if not cfg.quiet:
                print_warning('ERROR:"%s" image not valid' % self.fullpath)
            return False
        self.format = im.format
        self.size = im.size
        self.filesize = int(filesize / 1024)
        info = EXIF.process_file(file(self.fullpath))
        for key in ImageInfo.MyExifDict.values():
            if not info.has_key(key):
                info[key] = ''
#                 if not cfg.quiet:
#                     print_ok('%s: missing %s\n' % (self.fullpath, key))
        self.timestamp = info['Image DateTime']
        if self.timestamp:
            try:
                timestamp = time.strptime(self.timestamp.printable,
                                          '%Y:%m:%d %H:%M:%S')
                timestamp = time.mktime(timestamp)
            except ValueError:
                pass
        timestamp = datetime.datetime.fromtimestamp(timestamp)
        self.timestamp = timestamp.strftime('%Y/%m/%d %H:%M:%S')    
        self.orientation = info['Image Orientation']
        try:
            ap = eval(info['EXIF FNumber'].printable + '.')
        except AttributeError:
            ap = 0
        self.aperture = 'f/' + str(round(float(ap), 2))
        self.exposure_time = info['EXIF ExposureTime']
        self.exposure_program = info['EXIF ExposureProgram']
        try:
            eb = eval(info['EXIF ExposureBiasValue'].printable + '.')
        except AttributeError:
            eb = 0
        self.exposure_bias = round(float(eb), 2)
        try:
            fl = eval(info['EXIF FocalLength'].printable + '.')
        except AttributeError:
            fl = 0
        self.focal_length = round(float(fl), 2)
        self.brightness = info['EXIF BrightnessValue']
        self.flash = info['EXIF Flash']
        self.isospeedratings = info['EXIF ISOSpeedRatings']
        self.camera = '%s %s' % (info['Image Make'], info['Image Model'])
        return True


######################################################################
##### pynakotheka
######################################################################
def generate_html_tree(gallery):
    if not cfg.quiet:
        print_ok('  Generating albums tree... ')
    tree_file = os.path.join(cfg.destdir, 'view_tree.html')
    tmpl_file = os.path.join(cfg.tmpldir, 'view_tree.tmpl')
    html = Template(file=tmpl_file)
    html.txt2html = txt2html
    html.txt2link = txt2link
    html.gallery = gallery
    html.tree = gallery.get_tree()[1:]  # don't include gallery root node
    html.timestamp = datetime.datetime.now().strftime('%A %d %B %Y %H:%M')
    htmlfile = open(tree_file, 'w')
    htmlfile.write(html.respond())
    htmlfile.close()
    if not cfg.quiet:
        print_warning('  OK\n')


def copy_style_files():
    if not cfg.quiet:
        print_ok('  Copying style files... ')
    for f in STYLE_FILES:
        print_ok('%s ' % f)
        src = os.path.join(cfg.tmpldir, f)
        dest = os.path.join(cfg.destdir, f)
        try:
            copy_file(src, dest)
        except OSError:
            if not cfg.quiet:
                print_warning('\nWARNING: cannot copy image "%s"\n' % \
                              os.path.basename(src))
    if not cfg.quiet:
        print_warning('  OK\n')


def pynakotheka():
    # show information
    if not cfg.quiet:
        t0 = time.time()
        print_important(u'\npynakotheka - (C) 2005-6, by Iigo Serna\n')
        print_ok('''
Source: %s
Target: %s
Thumbsize: %d
Imagesize: %d
Copy originals: %s
Template: %s
''' % (cfg.srcdir, cfg.destdir, cfg.thumbsize, cfg.imagesize,
       str(cfg.originals), cfg.template))
    # start
    gallery = Album(None, '')
    # parse directory structure
    if not cfg.quiet:
        print_sect('\nAnalyzing source directories structure:\n')
    gallery.parse()
    # validate albums
    if not cfg.quiet:
        print_sect('\nValidating albums:\n')
    gallery.validate()
    # show albums to build
    if not cfg.quiet:
        print_sect('\nAlbums to build:\n')
    gallery.show()
    # create target directories
    if not cfg.quiet:
        print_sect('\nCreating target directories structure... ')
    gallery.create_target_dirs()
    if not cfg.quiet:
        print_warning('OK\n')
    # generate images
    if not cfg.quiet:
        print_sect('\nGenerating images:\n')
    gallery.generate_images()
    # copy original images
    if cfg.originals:
        if not cfg.quiet:
            print_sect('\nCopying original images:\n')
        gallery.copy_original_images()
    # generate album pages
    if not cfg.quiet:
        print_sect('\nGenerating html pages:\n')
    copy_style_files()
    generate_html_tree(gallery)
    gallery.generate_html()
    # end
    if not cfg.quiet:
        t1 = time.time()
        delta = t1 - t0
        if delta > 60:
            elapsed = '%d\'%2.2d\"' % divmod(delta, 60)
        else:
            elapsed = '%d sec' % delta
        print_important('\nGallery succesfully created in %s in %s\n\n' % \
                        (cfg.destdir, elapsed))


######################################################################
##### cleaning
######################################################################
def clean_style_files():
    for f in STYLE_FILES:
        dest = os.path.join(cfg.destdir, f)
        try:
            os.remove(dest)
        except OSError, err:
            print err


def clean_tree_file():
    try:
        os.remove(os.path.join(cfg.destdir, 'view_tree.html'))
    except OSError, err:
        print err


def clean(delete_images=False, delete_html=False):
    global cfg

    t0 = time.time()
    # show information
    if not cfg.quiet:
        print_important(u'\npynakotheka - (C) 2005-6, by Iigo Serna\n')
        print_ok('''
Source: %s
Target: %s
Thumbsize: %d
Imagesize: %d
Copy originals: %s
Template: %s\n
''' % (cfg.srcdir, cfg.destdir, cfg.thumbsize, cfg.imagesize,
       str(cfg.originals), cfg.template))
    # start
    if not cfg.quiet:
        print_sect('Cleaning gallery:\n')
        print_subsect('  Analyzing source directories structure... ')
    verbose = cfg.quiet
    cfg.quiet = True
    gallery = Album(None, '')
    gallery.parse()
    gallery.validate()
    cfg.quiet = verbose
    if not cfg.quiet:
        print_warning('OK\n')
    if delete_images:
        if not cfg.quiet:
            print_subsect('  Removing all images generated by the program...\n')
        gallery.clean_images()
    if delete_html:
        if not cfg.quiet:
            print_subsect('  Removing all html and style files generated by the program...\n')
        clean_style_files()
        clean_tree_file()
        gallery.clean_html()
    # end
    if not cfg.quiet:
        t1 = time.time()
        delta = t1 - t0
        if delta > 60:
            elapsed = '%d\'%2.2d\"' % divmod(delta, 60)
        else:
            elapsed = '%d sec' % delta
        print_important('\nGallery cleaned in %s\n\n' % elapsed)


######################################################################
##### MAIN
######################################################################
class Config:
    def __init__(self):
        self.srcdir = os.getcwd()
        self.destdir = os.getcwd()
        self.thumbsize = 250
        self.imagesize = 640
        self.originals = False
        self.template = 'default'
        self.tmpldir = os.path.join(sys.prefix, 'share', 'pynakotheka',
                                    'templates')
        self.quiet = False
        self.color = False


def usage():
    print __doc__


def main():
    global cfg, colorize

    action = None
    cfg = Config()

    # parse arguments and options
    try:
        opts, args = getopt.getopt(sys.argv[1:], 'hqcs:i:od:t:',
                                   ['help', 'quiet', 'color',
                                    'thumbsize=', 'imagesize=',
                                    'copy-originals',
                                    'templates-dir=', 'template=',
                                    'clean='])
    except getopt.GetoptError:
        print 'ERROR: Invalid argument\n'
        usage()
        sys.exit(2)

    # options
    for o, a in opts:
        if o in ('-h', '--help'):
            usage()
            sys.exit(2)
        if o in ('-s', '--thumbsize'):
            try:
                cfg.thumbsize = int(a)
            except ValueError:
                print 'ERROR: thumbsize must be a number\n'
                usage()
                sys.exit(2)
        if o in ('-i', '--imagesize'):
            try:
                cfg.imagesize = int(a)
            except ValueError:
                print 'ERROR: imagesize must be a number\n'
                usage()
                sys.exit(2)
        if o in ('-o', '--copy-originals'):
            cfg.originals = True
        if o in ('-d', '--templates-dir'):
            if os.path.isabs(args[0]):
                cfg.tmpldir = a
            else:
                cfg.tmpldir = os.path.join(os.getcwd(), a)
        if o in ('-t', '--template'):
            cfg.template = a
        if o in ('-q', '--quiet'):
            cfg.quiet = True
        if o in ('-c', '--color'):
            cfg.color = True
        if o in ('--clean', ):
            action = 'clean=' + a

    # arguments
    if len(args) > 2:
        print 'ERROR: Invalid number of arguments\n'
        usage()
        sys.exit(2)
    elif len(args) == 1:
        if os.path.isabs(args[0]):
            cfg.srcdir = args[0]
        else:
            cfg.srcdir = os.path.join(os.getcwd(), args[0])
    elif len(args) == 2:
        if os.path.isabs(args[0]):
            cfg.srcdir = args[0]
        else:
            cfg.srcdir = os.path.join(os.getcwd(), args[0])
        if os.path.isabs(args[1]):
            cfg.destdir = args[1]
        else:
            cfg.destdir = os.path.join(os.getcwd(), args[1])

    # fix config values
    if cfg.srcdir.endswith(os.sep):
        cfg.srcdir = cfg.srcdir[:-1]
    if cfg.destdir.endswith(os.sep):
        cfg.destcdir = cfg.destdir[:-1]
    if not cfg.color:
        colorize = no_colorize
    # validate config values
    if not os.path.exists(cfg.srcdir) or not os.path.isdir(cfg.srcdir):
        print 'ERROR: "%s" is not a valid directory\n' % cfg.srcdir
        usage()
        sys.exit(2)
    tmpldir = os.path.join(cfg.tmpldir, cfg.template)
    if not action:
        if not os.path.exists(tmpldir) or not os.path.isdir(tmpldir):
            print 'ERROR: "%s" is not a valid template directory\n' % tmpldir
            usage()
            sys.exit(2)
        for filename in TEMPLATES:
            fullname = os.path.join(tmpldir, filename)
            if not os.path.exists(fullname) or not os.path.isfile(fullname):
                print 'ERROR: template "%s" hasn\'t needed "%s" file\n' % \
                      (cfg.template, filename)
                usage()
                sys.exit(2)
        cfg.tmpldir = tmpldir

    # actions are postponed until now to let app to get the arguments
    if action and action not in ('clean=all', 'clean=images', 'clean=html'):
        print 'ERROR: "%s" is not a valid clean target\n' % \
              action.split('=')[-1]
        usage()
        sys.exit(2)
    if action == 'clean=all':
        clean(delete_images=True, delete_html=True)
        sys.exit(0)
    if action == 'clean=images':
        clean(delete_images=True)
        sys.exit(0)
    if action == 'clean=html':
        clean(delete_html=True)
        sys.exit(0)
    elif not action:
        pynakotheka()
    else:
        raise ValueError, action


if __name__ == "__main__":
    main()
