#!/usr/bin/env python

import re
import sys

## class pyNet
## TODO:  add subnets() method, to return a list of networks that are subdivided
##            with the given mask argument
##        add first() method, to get the first usable address
##        add last() method, to get the last usable address
##        add compact() method (does nothing for v4)
##        add v4Tov6() method
##
##
## class pyIpHdr
##        add DSCPToDecimal() method
##
""" pynet.py - Manipulate IPv4 and IPv6 addresses efficiently
    Copyright (C) 2007 David Michael Pennington 

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

    If you need to contact the author, you can do so by emailing:
    mike [~at~] pennington [/dot\] net
    """

class pyNet(object):

   def __init__(self, input):
      """Process input, set the version, and internal variables"""
      ip, masklen, self.ver = self.parse_input( input )
      self.masklength = masklen
      self.ipINT = self.v4v6_to_INT( ip, self.ver )
      self.maskINT = self.masklen_to_INT( masklen, self.ver )


   def ip( self ):
      """Return the ip address"""
      return self.INT_to_v4v6( self.ipINT, self.ver )


   def network( self ):
      """Return the network address"""
      retval = self.INT_to_v4v6( ( self.ipINT & self.maskINT ), self.ver )
      return retval


   def cidr( self ):
      """Return a CIDR-style IP representation"""
      ip = self.INT_to_v4v6( self.ipINT, self.ver )
      masklen = self.INT_to_masklen( self.maskINT )
      return "%s/%s" % ( ip, masklen )


   def broadcast( self ):
      """Return the broadcast address"""
      # using my own binary_NOT function because I don't want a 2's compliment
      # TODO: incorporte Recipe 18.11 from the Python Cookbook, instead of 
      #    binary_NOT()
      host = long( self.binary_NOT( self.maskINT ), 2 )
      net = self.v4v6_to_INT( self.network(), self.ver )
      bcast_addr = long(net + host)
      retval = self.INT_to_v4v6( bcast_addr, self.ver )
      return retval


   def mask( self ):
      """Return a mask as either dotted-quad (ipv4) or colon-seperated string
      (ipv6)"""
      return self.INT_to_v4v6( self.maskINT, self.ver )


   def masklen( self ):
      """Return the mask length"""
      return self.masklength


   def bits( self ):
      """Return the width of the network in bits"""
      if self.ver == 4:
         retval = 32
      elif self.ver == 6:
         retval = 128
      return retval


   def wildcard( self ):
      """Return the wildcard representation of the netmask.  Useful in Cisco
      ACLs"""
      return self.INT_to_v4v6( long( self.binary_NOT( self.maskINT ), 2 ), 
             self.ver )


   def range( self ):
      """Return all addresses in the subnet, including network and broadcast"""
      return "%s-%s" % ( self.network(), self.broadcast() )


   def hosts( self ):
      """Return all USABLE addresses in the subnet.  Network and bcast 
      are excluded."""
      netINT = self.v4v6_to_INT( self.network(), self.ver )
      bcastINT = self.v4v6_to_INT( self.broadcast(), self.ver )
      if ( ( ( self.masklength < 31 ) & ( self.ver == 4 ) ) or 
         ( ( self.masklength < 127 ) & ( self.ver == 6 ) ) ):
         host_begin = self.INT_to_v4v6( netINT + 1L, self.ver )
         host_end   = self.INT_to_v4v6( bcastINT - 1L, self.ver )
      elif ( ( ( self.masklength == 31 ) & ( self.ver == 4 ) ) or 
         ( ( self.masklength == 127 ) & ( self.ver == 6 ) ) ):
	 # special case for Cisco's /31 netmask
         host_begin = self.INT_to_v4v6( netINT, self.ver )
         host_end   = self.INT_to_v4v6( bcastINT, self.ver )
      else:
	 # special case /32 netmask
         host_begin = self.INT_to_v4v6( netINT, self.ver )
         host_end   = host_begin
      return "%s-%s" % ( host_begin, host_end )


   def hostenum( self ):
      """Return a list of all USABLE addresses in the subnet.  Network and
      broadcast addresses are excluded."""
      ( begin, end ) = self.hosts().split("-")
      enum = []
      beginINT = self.v4v6_to_INT(  begin, self.ver  )
      endINT   = self.v4v6_to_INT(  end, self.ver  ) + 1L
      for ii in range( beginINT, endINT ):
         enum.append( self.INT_to_v4v6( ii, self.ver ) )
      return enum


   def rangeenum( self ):
      """Return a list of all addresses in the subnet.  Network and broadcast 
      addresses are included."""
      ( begin, end ) = self.range().split("-")
      enum = []
      beginINT = self.v4v6_to_INT(  begin, self.ver  )
      endINT   = self.v4v6_to_INT(  end, self.ver  ) + 1L
      for ii in range( beginINT, endINT ):
         enum.append( self.INT_to_v4v6( ii, self.ver ) )
      return enum


   def contains( self, arg2 ):
      """Return True if the self object contains the argument"""
      objA = self
      objB = pyNet( arg2 )
      objA_INT = objA.ipINT & objA.maskINT
      objB_INT = objB.ipINT & objA.maskINT
      if objA_INT == objB_INT:
         retval = True
      else:
         retval = False
      return retval


   def within( self, arg2 ):
      """Return True if the self object is within argument"""
      objA = self
      objB = pyNet( arg2 )
      objA_INT = objA.ipINT & objB.maskINT
      objB_INT = objB.ipINT & objB.maskINT
      if objA_INT == objB_INT:
         retval = True
      else:
         retval = False
      return retval

   
   def v4v6_to_INT(self, network, version ):
      """Convert user input to (long) INT variables: ipINT and maskINT"""
      if version == 4:
         width = 32
	 array = network.split( ".", 4 )
         ipINT = 0
         for ii in range( len( array ) ):
            ipINT = ipINT + long("0x100",16)**( len(array) - ii - 1 )*long( array[ii] )
      else:
         width = 128
         v6_expanded = False
         while v6_expanded == False:
            network = re.sub( "::", ":0:", network )
            if not re.search( "::", network ):
	       v6_expanded = True
         array = network.split( ":", 8 )
         ipINT = 0
         for ii in range( len( array ) ):
            ipINT = ipINT + long("0x10000",16)**( len(array) - ii - 1 )*long( array[ii], 16 )

      return ipINT


   def masklen_to_INT( self, masklen, version ):
      if version == 4:
         width = 32
      if version == 6:
         width = 128
      maskINT = long( ( 2**masklen - 1 ) << ( width - masklen ) )
      return maskINT


   def INT_to_masklen( self, intvar ):
      binstr = self.binary( intvar )
      masklen = len( re.sub("0", "", binstr ) )
      return masklen


   def INT_to_v4v6( self, intvar, version ):
      """Convert a long INT number into v4 or v6 notation"""
      retval = ""
      strvar = re.sub( "^0x([0-9a-fA-F]+)L$", "\g<1>", str( hex(intvar) ) ).lower()
      if version == 4:
         # pad with 0 nibbles in case there weren't enough
         strvar = re.sub( "\s", "0", "%8s" % strvar )
         oo = "[0-9a-f]{2}"
	 ii = "([0-9a-f]{1,2})(" + oo + ")(" + oo + ")(" + oo + ")$"
	 rr = re.search( ii, strvar )
         octets = [ rr.group(1), rr.group(2), rr.group(3), rr.group(4) ]
	 for ii in range( len( octets ) ):
	    octets[ii] = str( int( octets[ii], 16 ) )
	 retval = ".".join( octets )
      elif version == 6:
         # pad with 0 nibbles in case there weren't enough
         strvar = re.sub( "\s", "0", "%32s" % strvar )
         oo = "[0-9a-f]{4}"
	 ii = "([0-9a-f]{1,4})(%s)(%s)(%s)(%s)(%s)(%s)(%s)" % ( oo,oo,oo,oo,oo,oo,oo )
	 rr = re.search( ii, strvar )
         octets = [ rr.group(1), rr.group(2), rr.group(3), rr.group(4), rr.group(5), rr.group(6), rr.group(7), rr.group(8) ]
	 retval = ":".join( octets )
      else:
         print "pyNet: INT_to_v4v6: An undefined error occurred while processing %s" % intvar
      return retval


   def parse_input(self, input):
      """Ensure that user's network strings are formatted correctly.  Return 
      version"""
      ## TODO: validate that the IPv6 parsing regex's cover all valid combos
      if re.search( "^(\d{1,3}\.){3}\d{1,3}\/\d{1,3}$", input.lower() ):
         ip, masklen = re.split( "\/", input.lower(), 2 ) 
	 version = 4
      elif re.search( "^(\d{1,3}\.){3}\d{1,3}\/(\d{1,3}\.){3}\d{1,3}$", input.lower() ):
         ip, mask = re.split( "\/", input.lower(), 2 ) 
	 version = 4
	 masklen = self.INT_to_masklen( self.v4v6_to_INT( mask, version ) )
      elif re.search( "^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$", input.lower() ):
         ip, masklen = input, "32"
	 version = 4
      elif re.search( "^([0-9a-f]{0,4}:){0,7}([0-9a-f]){0,3}[0-9a-f]$", input.lower() ):
	 version = 6
         ip, masklen = input, "128"
      elif re.search( "^([0-9a-f]{0,4}:){0,7}([0-9a-f]){0,3}[0-9a-f]\/\d{1,3}$", input.lower() ):
	 version = 6
         ip, masklen = re.split( "\/", input.lower(), 2 )
      else:
         print "pyNet: string_to_INT: Invalid network address or mask: %s" % input
	 sys.exit(0)

      masklen = long( masklen )
      return ip, masklen, version

   def binary_NOT( self, n ):
      """Perform a binary NOT on the INT or LONG argument.  Return the binary
      representation as a string."""
      assert n >= 0
      bits = []
      while n:
         bits.append('10'[n&1])
         n >>= 1
      bits.reverse()
      return ''.join(bits) or '0'

   def binary( self, n ):
      """Perform a binary NOT on the INT or LONG argument.  Return the binary
      representation as a string."""
      assert n >= 0
      bits = []
      while n:
         bits.append('01'[n&1])
         n >>= 1
      bits.reverse()
      return ''.join(bits) or '0'


if __name__ == '__main__':
   print ""
