import threading
import multiprocessing
import logging

logger = logging.getLogger("pypacker")

QUEUE_STRATEGY_BLOCK		= 0
QUEUE_STRATEGY_OVERWRITE	= 1

class Timeout(Exception): pass
class SkipElement(Exception): pass

class SortedProducerConsumer(object):
	"""
	This is a sorted version of the producer/consumer concept:
	produced: 1, 2, 3 -> consumed: 2, 3, 1 -> created object by consumer (same order as producer) -> obj(1), obj(2), obj(3)
	"""
	def __init__(self, producer_callback, consumer_callback, max_size=500, timeout=5, queue_strategy=QUEUE_STRATEGY_BLOCK, **kwargs):
		"""
		max_size -- max size of consumable, produceable and temporary elements
		producer_callback -- the source (method) to be called to get a new element. Signature: callback(**kwargs).
			Raise StopIteration to stop this producer at any time.
		consumer_callback -- the consumer (method) which gets produced elemenets to be consumed as only parameter and creates
			new objects from it. This has to return something! Signature: callback(obj, **kwargs).
			Raise SkipElement to skip specific Elements.
		kwargs -- additional arguments to be given to producer/consumer (like self)
		timeout -- max seconds after read block until __next__ oder __iter__ raise a Timeout exception.
			This can be a block on producer or consumer.
		queue_strategy -- one of QUEUE_STRATEGY_BLOCK (block if queue is full) or QUEUE_STRATEGY_OVERWRITE (continue producing
			on full queue overwriting oldest entries).
		"""
		# incrementing id
		self._produced_cnt = 0
		# next id of the element to be returned
		self._consumer_id = 0
		self._max_size = max_size
		self._timeout = timeout
		self._producer_callback = producer_callback
		self._is_stopped = False
		self._kwargs_producer_consumer = kwargs
		# stores produced elements
		self._produced_queue = multiprocessing.Queue(maxsize=max_size)
		# stores consumed elements
		self._consumed_queue = multiprocessing.Queue(maxsize=max_size)
		# stores elements if returning gets out of order
		self._returnbuffer = {}
		# consumer process
		self._consumer = []
		self._producer_stopped = False

		logger.debug("amount of CPU: %d" % multiprocessing.cpu_count())
		# start consumers
		for i in range(multiprocessing.cpu_count()):
			logger.debug("starting consumer process No.%d" % (i+1))
			p = multiprocessing.Process(target=SortedProducerConsumer._consumer_cycle,\
				args=(consumer_callback, self._produced_queue, self._consumed_queue, kwargs))
			self._consumer.append(p)
			p.start()

		# start producer
		producer = threading.Thread(target=SortedProducerConsumer._producer_cycle,\
			args=(self, producer_callback, self._produced_queue, kwargs))
		producer.daemon = True
		producer.start()
		self._producer = producer

	def _producer_cycle(self, producer_callback, produced_queue, kwargs):
		cnt = 0

		while not self._is_stopped:
			#print("p")
			# this can block
			try:
				produced_element = producer_callback(kwargs)
			except StopIteration:
				# last element produced, stop it..NOW!
				logger.debug("producer is stopping now")
				self._producer_stopped = True
				break

			# this can block
			produced_queue.put((cnt, produced_element), block=True, timeout=None)
			# inform consumer about new element
			#if cnt % 10 == 0:
			#	logger.debug("current max id: %d" % cnt)
			cnt += 1
		
	def _consumer_cycle(consumer_callback, produced_queue, consumed_queue, kwargs):
		while True:
			#print("c")
			# this can block
			item = produced_queue.get(block=True, timeout=None)
			# this can block
			#logger.debug("calling callback")
			try:
				el = consumer_callback(item[1], kwargs)
				# this can block
				#logger.debug("putting into Queue: %s" % el)
				consumed_queue.put((item[0], el), block=True, timeout=None)
			except SkipElement:
				# just skip element
				pass


	def __next__(self):
		#print("n")
		_returnbuffer = self._returnbuffer
		# clear buffer first
		try:
			ret = _returnbuffer.pop(self._consumer_id)
			self._consumer_id += 1
			#logger.debug("returning buffered result: %s, len: %d" % (ret, len(_returnbuffer)))
			return ret
		except KeyError:
			pass

		# empty() is not reliable
		if self._producer_stopped and self._produced_queue.empty() and self._consumed_queue.empty():
			raise StopIteration

		#logger.debug("getting next element out of Queue")
		try:
			el = self._consumed_queue.get(block=True, timeout=self._timeout)

			# search for next id to be returned
			while el[0] != self._consumer_id:
				_returnbuffer[el[0]] = el[1]
				el = self._consumed_queue.get(block=True, timeout=self._timeout)

				if len(_returnbuffer) > self._max_size:
					raise Exception("Buffer full: could no retrieve next element..consumer too slow?")
			self._consumer_id += 1

			return el[1]
		except multiprocessing.Queue.Empty:
			raise Timeout("could not get next element")
		
	def __iter__(self):
		# run until StopIteration or Queue.Empty exception occurs
		while True:
			yield self.__next__()
		
	#def reset(self):
	#	"""
	#	Reset all buffers. All produced elements will be removed.
	#	"""
	#	# TODO: more efficient
	#	while not self._produced_queue.empty():
	#		self._produced_queue.get()
	#	while not self._consumed_queue.empty():
	#		self._consumed_queue.get()
	#	self._returnbuffer = {}

	def __is_stopped(self):
		return self._is_stopped

	is_stopped = property(__is_stopped)

	def stop(self):
		self._is_stopped = True

		self._produced_queue.close()
		self._consumed_queue.close()

		for p in self._consumer:
			p.terminate()
		self._returnbuffer = {}
