# -*- coding: utf-8 -*-

# PyParadox is a *nix launcher for Paradox games.
# Copyright (C) 2014  Ruben Bakker <rubykuby@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
from subprocess import Popen
from pyparadox.command import make_command
from pyparadox.Config import Config
from pyparadox.exceptions import PySideImportError, RunGameError
from pyparadox.finder import find_dlcs, find_mods
from pyparadox.gui.ConfigDlg import ConfigDlg
from pyparadox.gui.PluginListModel import PluginListModel
from pyparadox.Dlc import Dlc
from pyparadox.Mod import Mod
try:
    from PySide import QtCore, QtGui
    from PySide.QtCore import Signal, Slot
except ImportError:
    raise PySideImportError


class MainWindow(QtGui.QMainWindow):
    """Main window for the application."""

    def __init__(self, game, parent=None):
        super(MainWindow, self).__init__(parent)
        self.init_ui()

        self.config = Config(game, "PyParadox", "PyParadox")

        self.fill_dlcs()
        self.fill_mods()

        #filename = QtGui.QFileDialog.getExistingDirectory(self, self.tr("Open"), "/tmp")

        self.config_btn.clicked.connect(self.show_config_dlg)
        self.run_btn.clicked.connect(self.run_game)

    def init_ui(self):
        """Initialises all (empty) UI elements."""
        self.central_widget = QtGui.QWidget(self)
        self.setCentralWidget(self.central_widget)

        mods_layout = QtGui.QVBoxLayout()
        self.mods_lbl = QtGui.QLabel(self.tr("Mods:"))
        self.mods_list = QtGui.QListView()
        mods_layout.addWidget(self.mods_lbl)
        mods_layout.addWidget(self.mods_list)

        dlcs_layout = QtGui.QVBoxLayout()
        self.dlcs_lbl = QtGui.QLabel(self.tr("DLCs:"))
        self.dlcs_list = QtGui.QListView()
        dlcs_layout.addWidget(self.dlcs_lbl)
        dlcs_layout.addWidget(self.dlcs_list)

        plugins_layout = QtGui.QHBoxLayout()
        plugins_layout.addLayout(mods_layout)
        plugins_layout.addLayout(dlcs_layout)

        buttons_layout = QtGui.QHBoxLayout()
        self.config_btn = QtGui.QPushButton(self.tr("Configure"))
        self.run_btn = QtGui.QPushButton(self.tr("Run"))
        buttons_layout.addStretch()
        buttons_layout.addWidget(self.config_btn)
        buttons_layout.addWidget(self.run_btn)

        layout = QtGui.QVBoxLayout()
        #layout.addWidget(picture_of_game)
        layout.addLayout(plugins_layout)
        layout.addLayout(buttons_layout)

        self.central_widget.setLayout(layout)

    def fill_dlcs(self):
        """Fills the DLC ListView with found DLCs from the game folder and
        unselects DLCs based on PyParadox's configuration file.
        """
        dlcs = find_dlcs(os.path.join(self.config.game_path, "dlc"))
        for dlc in dlcs:
            if os.path.basename(dlc.filename) in self.config.excluded_dlcs:
                dlc.enabled = False
        self.dlcs_model = PluginListModel(dlcs)

        self.dlcs_list.setModel(self.dlcs_model)

    def fill_mods(self):
        """Fills the mods ListView with found mods from the mod folder and
        unselects mods based on PyParadox's configuration file.
        """
        mods = find_mods(self.config.mod_path)
        for mod in mods:
            if os.path.basename(mod.filename) not in self.config.mods:
                mod.enabled = False
        self.mods_model = PluginListModel(mods)

        self.mods_list.setModel(self.mods_model)

    @Slot()
    def show_config_dlg(self):
        config_dlg = ConfigDlg()
        config_dlg.game_path_edit.setText(self.config.game_path)
        config_dlg.mod_path_edit.setText(self.config.mod_path)
        config_dlg.binary_edit.setText(self.config.binary)
        if config_dlg.exec_():
            self.config.game_path = config_dlg.game_path_edit.text()
            self.config.mod_path  = config_dlg.mod_path_edit.text()
            self.config.binary    = config_dlg.binary_edit.text()

    @Slot()
    def run_game(self):
        excluded_dlcs = [os.path.basename(dlc.filename) for dlc in
                         self.dlcs_model.plugins if not dlc.enabled]
        mods          = [os.path.basename(mod.filename) for mod in
                         self.mods_model.plugins if mod.enabled]

        self.config.excluded_dlcs = excluded_dlcs
        self.config.mods          = mods

        command = make_command(self.config.game_path, self.config.binary,
                                self.config.excluded_dlcs, self.config.mods)
        try:
            Popen(command)
            self.close()
        except OSError:
            QtGui.QMessageBox.warning(self,
                                      self.tr("Error"),
                                      self.tr("Game could not launch."))
