# -*- coding: utf-8 -*-

# PyParadox is a *nix launcher for Paradox games.
# Copyright (C) 2014  Ruben Bakker <rubykuby@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import sys
import appdirs
try:
    import configparser
except ImportError:
    import ConfigParser as configparser


class Config(object):

    def __init__(self, game, author, program_name, filename=None):
        self.game         = game
        self.author       = author
        self.program_name = program_name
        # Sets filename to default path if none is given.
        if filename is None:
            self.filename = os.path.join(appdirs.user_data_dir(program_name,
                                                               author),
                                         "pyparadox.conf")
        # Changes filename to custom full file path if one is supplied.
        else:
            self.filename = filename

        if self.game == "eu4":
            self.full_game = "Europa Universalis IV"
        elif self.game == "ck2":
            self.full_game = "Crusader Kings II"
        else:
            raise Exception("Game doesn't exist")

        self.config = configparser.ConfigParser()
        self.create_config()
        self.config.read(self.filename)
        if not self.config.has_section(self.game):
            self.set_default_layout()

    def create_config(self):
        """Creates an empty config file if none exists. Does nothing
        otherwise.
        """
        try:
            os.makedirs(os.path.dirname(self.filename))
        except:
            if os.path.isdir(os.path.dirname(self.filename)):
                pass
            else:
                raise Exception("please report: ponies")  # This isn't supposed
                                                          # to happen.
        try:
            open(self.filename, 'a').close()
        except IOError as err:
            raise err

    def set_default_layout(self, refresh=True):
        """Creates a default game entry."""
        # No mods or DLCs selected by default.
        excluded_dlcs = ""
        mods = ""

        # Windows
        if sys.platform.startswith("win"):
            binary    = "{}.exe".format(self.game)
            game_path = os.path.join(os.environ("ProgramFiles(x86)"),
                                     "Paradox Interactive",
                                     self.full_game)
            mod_path  = os.path.join(os.path.expanduser("~"),
                                     "Documents",
                                     "Paradox Interactive",
                                     self.full_game,
                                     "mod")
        # OS X
        elif sys.platform == 'darwin':
            binary    = self.game
            game_path = os.path.join(appdirs.user_data_dir("Steam", ""),
                                     "SteamApps",
                                     "common",
                                     self.full_game)
            mod_path  = os.path.join(os.path.expanduser("~"),
                                     "Documents",
                                     "Paradox Interactive",
                                     self.full_game,
                                     "mod")
        # Linux
        else:
            binary    = self.game
            game_path = os.path.join(os.path.expanduser("~"),
                                     ".steam",
                                     "steam",
                                     "SteamApps",
                                     "common",
                                     self.full_game)
            mod_path  = os.path.join(os.path.expanduser("~"),
                                     ".paradoxinteractive",
                                     self.full_game,
                                     "mod")

        self.config.add_section(self.game)
        self.config.set(self.game, "game_path",     game_path)
        self.config.set(self.game, "mod_path",      mod_path)
        self.config.set(self.game, "binary",        binary)
        self.config.set(self.game, "excluded_dlcs", excluded_dlcs)
        self.config.set(self.game, "mods",          mods)

        if refresh:
            self.refresh_config_file()

    def refresh_config_file(self):
        with open(self.filename, 'w') as configfile:
            self.config.write(configfile)

    @property
    def game_path(self):
        return self.config.get(self.game, "game_path")

    @game_path.setter
    def game_path(self, path, refresh=True):
        self.config.set(self.game, "game_path", path)
        if refresh:
            self.refresh_config_file()

    @property
    def mod_path(self):
        return self.config.get(self.game, "mod_path")

    @mod_path.setter
    def mod_path(self, path, refresh=True):
        self.config.set(self.game, "mod_path", path)
        if refresh:
            self.refresh_config_file()

    @property
    def binary(self):
        return self.config.get(self.game, "binary")

    @binary.setter
    def binary(self, binary, refresh=True):
        self.config.set(self.game, "binary", binary)
        if refresh:
            self.refresh_config_file()

    @property
    def excluded_dlcs(self):
        dlcs = self.config.get(self.game, "excluded_dlcs").split(',')
        if len(dlcs) == 1 and dlcs[0] == "":
            return []
        else:
            return dlcs

    @excluded_dlcs.setter
    def excluded_dlcs(self, dlcs, refresh=True):
        self.config.set(self.game, "excluded_dlcs", ','.join(dlcs))
        if refresh:
            self.refresh_config_file()

    @property
    def mods(self):
        mods = self.config.get(self.game, "mods").split(',')
        if len(mods) == 1 and mods[0] == "":
            return []
        else:
            return mods

    @mods.setter
    def mods(self, mods, refresh=True):
        self.config.set(self.game, "mods", ','.join(mods))
        if refresh:
            self.refresh_config_file()
