# -*- coding: utf-8 -*-

# PyParadox is a *nix launcher for Paradox games.
# Copyright (C) 2014  Ruben Bakker <rubykuby@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import shutil
import tempfile
import unittest
try:
    import configparser
except ImportError:
    import ConfigParser as configparser
from pyparadox.Config import Config


class TestConfigGettersSetters(unittest.TestCase):

    @classmethod
    def setUpClass(cls):
        cls.tempdir = os.path.join(tempfile.gettempdir(), "testpyparadox")
        cls.filename = os.path.join(cls.tempdir, "pyparadox.conf")
        cls.parser = configparser.ConfigParser()

    def setUp(self):
        self.config = Config("ck2", "PyParadox", "PyParadox", self.filename)

    def tearDown(self):
        del self.config
        shutil.rmtree(self.tempdir)

    def test_game_path(self):
        self.config.game_path = "/tmp"
        self.parser.read(self.filename)
        self.assertEqual(self.parser.get("ck2", "game_path"), "/tmp")
        self.assertEqual(self.config.game_path, "/tmp")

    def test_mod_path(self):
        self.config.mod_path = "/tmp"
        self.parser.read(self.filename)
        self.assertEqual(self.parser.get("ck2", "mod_path"), "/tmp")
        self.assertEqual(self.config.mod_path, "/tmp")

    def test_binary(self):
        self.config.binary = "eu4"
        self.parser.read(self.filename)
        self.assertEqual(self.parser.get("ck2", "binary"), "eu4")
        self.assertEqual(self.config.binary, "eu4")

    def test_excluded_dlcs_none(self):
        self.config.excluded_dlcs = []
        self.parser.read(self.filename)
        self.assertEqual(self.parser.get("ck2", "excluded_dlcs"), "")
        self.assertEqual(self.config.excluded_dlcs, [])

    def test_excluded_dlcs_single(self):
        self.config.excluded_dlcs = ["dlc001.dlc"]
        self.parser.read(self.filename)
        self.assertEqual(self.parser.get("ck2", "excluded_dlcs"), "dlc001.dlc")
        self.assertEqual(self.config.excluded_dlcs, ["dlc001.dlc"])

    def test_excluded_dlcs_multiple(self):
        self.config.excluded_dlcs = ["dlc001.dlc", "dlc002.dlc"]
        self.parser.read(self.filename)
        self.assertEqual(self.parser.get("ck2", "excluded_dlcs"),
                         "dlc001.dlc,dlc002.dlc")
        self.assertEqual(self.config.excluded_dlcs, ["dlc001.dlc",
                                                     "dlc002.dlc"])

    def test_mods_none(self):
        self.config.mods = []
        self.parser.read(self.filename)
        self.assertEqual(self.parser.get("ck2", "mods"), "")
        self.assertEqual(self.config.mods, [])

    def test_mods_single(self):
        self.config.mods = ["better gender law mod.mod"]
        self.parser.read(self.filename)
        self.assertEqual(self.parser.get("ck2", "mods"),
                         "better gender law mod.mod")
        self.assertEqual(self.config.mods, ["better gender law mod.mod"])

    def test_mods_multiple(self):
        self.config.mods = ["better gender law mod.mod", "foo.mod"]
        self.parser.read(self.filename)
        self.assertEqual(self.parser.get("ck2", "mods"),
                         "better gender law mod.mod,foo.mod")
        self.assertEqual(self.config.mods, ["better gender law mod.mod",
                                            "foo.mod"])


class TestTwoConfigs(unittest.TestCase):

    @classmethod
    def setUpClass(cls):
        cls.tempdir = os.path.join(tempfile.gettempdir(), "testpyparadox")
        cls.filename = os.path.join(cls.tempdir, "pyparadox.conf")

    def tearDown(self):
        shutil.rmtree(self.tempdir)

    def test_two_configs_same_game(self):
        config_one = Config("ck2", "PyParadox", "PyParadox", self.filename)
        config_one.binary = "eu4"
        del config_one
        config_two = Config("ck2", "PyParadox", "PyParadox", self.filename)
        self.assertEqual(config_two.binary, "eu4")

    def test_two_configs_different_game(self):
        config_one = Config("ck2", "PyParadox", "PyParadox", self.filename)
        config_one.binary = "foo"
        del config_one
        config_two = Config("eu4", "PyParadox", "PyParadox", self.filename)
        self.assertNotEqual(config_two.binary, "foo")

if __name__ == "__main__":
    unittest.main()
