# -*- coding: utf-8 -*-

# PyParadox is a *nix launcher for Paradox games.
# Copyright (C) 2014  Ruben Bakker <rubykuby@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import sys
import appdirs
try:
    import configparser
except ImportError:
    import ConfigParser as configparser
from .exceptions import GameNotExistError


class Config(object):
    """Config holds getters and setters that translate to calls to a
    configparser.

    :ivar game: Short identifier of game.
    :type game: str.
    :ivar fileobj: File object that is read from/written to.
    :type fileobj: TextIOWrapper.
    :ivar refresh: Toggle for whether setters write to ``fileobj``.
    :type refresh: bool.
    """

    def __init__(self, game, fileobj, refresh=True):
        """Create a Config instance.

        :param game: Short identifier of game.
        :type game: str.
        :param fileobj: File object.
        :type fileobj: TextIOWrapper.
        :param refresh: Toggle for whether setters write to ``fileobj``.
        :type refresh: bool.
        """
        self.game = game
        self.fileobj = fileobj
        self.refresh = refresh

        if self.game == "eu4":
            self.full_game = "Europa Universalis IV"
        elif self.game == "ck2":
            self.full_game = "Crusader Kings II"
        else:
            raise GameNotExistError()

        self._config = configparser.ConfigParser()
        self._config.readfp(self.fileobj)
        if not self._config.has_section(self.game):
            self.set_default_layout()

    def set_default_layout(self):
        """Set a game entry in ``_config`` to some default values dependent on
        platform.
        """
        # No mods or DLCs selected by default.
        excluded_dlcs = ""
        mods = ""

        # Windows
        if sys.platform.startswith("win"):
            binary = "{}.exe".format(self.game)
            game_path = os.path.join(os.environ["ProgramFiles(x86)"],
                                     "Steam",
                                     "SteamApps",
                                     "common",
                                     self.full_game)
            mod_path = os.path.join(os.path.expanduser("~"),
                                    "Documents",
                                    "Paradox Interactive",
                                    self.full_game,
                                    "mod")
        # OS X
        elif sys.platform == 'darwin':
            binary = self.game
            game_path = os.path.join(appdirs.user_data_dir("Steam", ""),
                                     "SteamApps",
                                     "common",
                                     self.full_game)
            mod_path = os.path.join(os.path.expanduser("~"),
                                    "Documents",
                                    "Paradox Interactive",
                                    self.full_game,
                                    "mod")
        # Linux
        else:
            binary = self.game
            game_path = os.path.join(os.path.expanduser("~"),
                                     ".steam",
                                     "steam",
                                     "SteamApps",
                                     "common",
                                     self.full_game)
            mod_path = os.path.join(os.path.expanduser("~"),
                                    ".paradoxinteractive",
                                    self.full_game,
                                    "mod")

        try:
            self._config.add_section(self.game)
        except configparser.DuplicateSectionError:
            pass

        self.game_path = game_path
        self.mod_path = mod_path
        self.binary = binary
        self.excluded_dlcs = excluded_dlcs
        self.mods = mods

        if self.refresh:
            self.refresh_config_file()

    def refresh_config_file(self):
        """Write the state of ``_config`` to ``fileobj``."""
        self.fileobj.seek(0)
        self.fileobj.truncate()
        self._config.write(self.fileobj)

    @property
    def game_path(self):
        """Absolute path towards the root of the game directory.

        :getter: Read value from ``_config``.
        :setter: Write value to ``_config``.
        :type: str.
        """
        return self._config.get(self.game, "game_path")

    @game_path.setter
    def game_path(self, path):
        self._config.set(self.game, "game_path", path)
        if self.refresh:
            self.refresh_config_file()

    @property
    def mod_path(self):
        """Absolute path towards the mod directory.

        :getter: Read value from ``_config``.
        :setter: Write value to ``_config``.
        :type: str.
        """
        return self._config.get(self.game, "mod_path")

    @mod_path.setter
    def mod_path(self, path):
        self._config.set(self.game, "mod_path", path)
        if self.refresh:
            self.refresh_config_file()

    @property
    def binary(self):
        """Basename of the binary file.

        :getter: Read value from ``_config``.
        :setter: Write value to ``_config``.
        :type: str.
        """
        return self._config.get(self.game, "binary")

    @binary.setter
    def binary(self, binary):
        self._config.set(self.game, "binary", binary)
        if self.refresh:
            self.refresh_config_file()

    @property
    def excluded_dlcs(self):
        """List of basenames of ".dlc" files.

        :getter: Read value from ``_config``.
        :setter: Write value to ``_config``.
        :type: list.
        """
        dlcs = self._config.get(self.game, "excluded_dlcs").split(',')
        if len(dlcs) == 1 and dlcs[0] == "":
            return []
        else:
            return dlcs

    @excluded_dlcs.setter
    def excluded_dlcs(self, dlcs):
        self._config.set(self.game, "excluded_dlcs", ','.join(dlcs))
        if self.refresh:
            self.refresh_config_file()

    @property
    def mods(self):
        """List of basenames of ".mod" files.

        :getter: Read value from ``_config``.
        :setter: Write value to ``_config``.
        :type: list.
        """
        mods = self._config.get(self.game, "mods").split(',')
        if len(mods) == 1 and mods[0] == "":
            return []
        else:
            return mods

    @mods.setter
    def mods(self, mods):
        self._config.set(self.game, "mods", ','.join(mods))
        if self.refresh:
            self.refresh_config_file()
