# -*- coding: utf-8 -*-

# PyParadox is a *nix launcher for Paradox games.
# Copyright (C) 2014  Ruben Bakker <rubykuby@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import unittest
try:
    from unittest import mock
except ImportError:
    import mock
try:
    import configparser
except ImportError:
    import ConfigParser as configparser
from pyparadox.Config import Config
from pyparadox.exceptions import GameNotExistError


class TestConfig(unittest.TestCase):

    @mock.patch("pyparadox.Config.Config.set_default_layout", autospec=True)
    @mock.patch("pyparadox.Config.configparser", autospec=True)
    def test___init__(self, mock_configparser, mock_set_default_layout):
        config = Config("ck2", mock.MagicMock())

        # Make sure that a full_game was generated despite not being
        # instantiated with one.
        self.assertIsNotNone(config.full_game)

        # Make sure there's an internal ConfigParser.
        self.assertIsNotNone(config._config)

        # Make sure that refresh was generated despite not being
        # instantiated with one.
        self.assertTrue(config.refresh)

        # Assert that the configparser reads its file.
        config._config.readfp.assert_called_with(config.fileobj)

        # Assert that no default things were set.
        self.assertFalse(config.set_default_layout.called)

        # Assert that a default layout was set in case the ConfigParser
        # didn't have an internal section for the assigned game.
        config._config.has_section.return_value = False
        config = Config("ck2", mock.MagicMock())
        self.assertTrue(config.set_default_layout.called)

        # Pass a wrong game to the constructor and make sure that it
        # raises an exception.
        with self.assertRaises(GameNotExistError):
            config = Config("wrong_game", mock.MagicMock())

        # Make sure that no errors occured with eu4 either.
        config = Config("eu4", mock.MagicMock())
        self.assertTrue(config.full_game)

        # Assert that refresh was correctly set to False.
        config = Config("ck2", mock.MagicMock(), refresh=False)
        self.assertFalse(config.refresh)

    @mock.patch("pyparadox.Config.Config.mods", new_callable=mock.PropertyMock)
    @mock.patch("pyparadox.Config.Config.excluded_dlcs",
                new_callable=mock.PropertyMock)
    @mock.patch("pyparadox.Config.Config.binary",
                new_callable=mock.PropertyMock)
    @mock.patch("pyparadox.Config.Config.mod_path",
                new_callable=mock.PropertyMock)
    @mock.patch("pyparadox.Config.Config.game_path",
                new_callable=mock.PropertyMock)
    @mock.patch("pyparadox.Config.Config.refresh_config_file",
                new_callable=mock.PropertyMock)
    @mock.patch("pyparadox.Config.appdirs", autospec=True)
    @mock.patch("pyparadox.Config.os", autospec=True)
    @mock.patch("pyparadox.Config.sys", autospec=True)
    @mock.patch("pyparadox.Config.configparser", autospec=True)
    def test_set_default_layout(self, mock_configparser, mock_sys, mock_os,
                                mock_appdirs, mock_refresh_config_file,
                                mock_game_path, mock_mod_path, mock_binary,
                                mock_excluded_dlcs, mock_mods):
        config = Config("ck2", mock.MagicMock(), refresh=False)
        config.set_default_layout()

        # Assert that nothing was written to a file.
        self.assertFalse(config.refresh_config_file.called)

        config.refresh = True

        # Make sure that no crashes occur regardless of platform.
        mock_sys.platform = "windows"
        config.set_default_layout()

        mock_sys.platform = "darwin"
        config.set_default_layout()

        mock_sys.platform = "linux"
        config.set_default_layout()

        # Assert that values all values were passed on to setters.
        self.assertTrue(mock_game_path.called)
        self.assertTrue(mock_mod_path.called)
        self.assertTrue(mock_binary.called)
        self.assertTrue(mock_excluded_dlcs.called)
        self.assertTrue(mock_mods.called)

        # Assert that the values were written away to a file.
        self.assertTrue(config.refresh_config_file.called)

        # Assert that no errors occured in spite of the section existing.
        config._config.add_section.side_effect = \
            configparser.DuplicateSectionError("Section already existed")
        mock_configparser.DuplicateSectionError = \
            configparser.DuplicateSectionError
        config.set_default_layout()

    @mock.patch("pyparadox.Config.configparser", autospec=True)
    def test_refresh_config_file(self, mock_configparser):
        fileobj = mock.MagicMock()
        config = Config("ck2", fileobj)
        config.refresh_config_file()

        # Assert that the cursor was set to the start of the config file.
        config.fileobj.seek.assert_called_with(0)

        # Assert that the config file was written to.
        config._config.write.assert_called_with(fileobj)

    @mock.patch("pyparadox.Config.Config.refresh_config_file", autospec=True)
    @mock.patch("pyparadox.Config.configparser", autospec=True)
    def test_game_path(self, mock_configparser, mock_refresh_config_file):
        config = Config("ck2", mock.MagicMock())

        # Assert that the setter forwarded the right value to ``_config`` and
        # called ``refresh_config_file()``.
        config.game_path = "path"
        config._config.set.assert_called_with("ck2", "game_path", "path")
        self.assertTrue(config.refresh_config_file.called)

        # Assert that the setter did not call ``refresh_config_file()`` with
        # ``refresh`` set to False.
        config.refresh_config_file.reset_mock()
        config.refresh = False
        config.game_path = "path"
        self.assertFalse(config.refresh_config_file.called)

        # Assert that the getter called ``get`` of ``_config`` and got an
        # instance back from ConfigParser, which is a MagicMock.
        path = config.game_path
        config._config.get.assert_called_with("ck2", "game_path")
        self.assertIsInstance(path, mock.MagicMock)

    @mock.patch("pyparadox.Config.Config.refresh_config_file", autospec=True)
    @mock.patch("pyparadox.Config.configparser", autospec=True)
    def test_mod_path(self, mock_configparser, mock_refresh_config_file):
        config = Config("ck2", mock.MagicMock())

        # Assert that the setter forwarded the right value to ``_config`` and
        # called ``refresh_config_file()``.
        config.mod_path = "path"
        config._config.set.assert_called_with("ck2", "mod_path", "path")
        self.assertTrue(config.refresh_config_file.called)

        # Assert that the setter did not call ``refresh_config_file()`` with
        # ``refresh`` set to False.
        config.refresh_config_file.reset_mock()
        config.refresh = False
        config.mod_path = "path"
        self.assertFalse(config.refresh_config_file.called)

        # Assert that the getter called ``get`` of ``_config`` and got an
        # instance back from ConfigParser, which is a MagicMock.
        path = config.mod_path
        config._config.get.assert_called_with("ck2", "mod_path")
        self.assertIsInstance(path, mock.MagicMock)

    @mock.patch("pyparadox.Config.Config.refresh_config_file", autospec=True)
    @mock.patch("pyparadox.Config.configparser", autospec=True)
    def test_binary(self, mock_configparser, mock_refresh_config_file):
        config = Config("ck2", mock.MagicMock())

        # Assert that the setter forwarded the right value to ``_config`` and
        # called ``refresh_config_file()``.
        config.binary = "foo"
        config._config.set.assert_called_with("ck2", "binary", "foo")
        self.assertTrue(config.refresh_config_file.called)

        # Assert that the setter did not call ``refresh_config_file()`` with
        # ``refresh`` set to False.
        config.refresh_config_file.reset_mock()
        config.refresh = False
        config.binary = "foo"
        self.assertFalse(config.refresh_config_file.called)

        # Assert that the getter called ``get`` of ``_config`` and got an
        # instance back from ConfigParser, which is a MagicMock.
        binary = config.binary
        self.assertIsInstance(binary, mock.MagicMock)
        config._config.get.assert_called_with("ck2", "binary")

    @mock.patch("pyparadox.Config.Config.refresh_config_file", autospec=True)
    @mock.patch("pyparadox.Config.configparser", autospec=True)
    def test_excluded_dlcs(self, mock_configparser, mock_refresh_config_file):
        config = Config("ck2", mock.MagicMock())

        # Assert that the setter forwarded the right value to ``_config`` and
        # called ``refresh_config_file()``.
        config.excluded_dlcs = []
        config._config.set.assert_called_with("ck2", "excluded_dlcs", "")
        self.assertTrue(config.refresh_config_file.called)

        # Assert that the setter forwarded the right value to ``_config``.
        config.refresh_config_file.reset_mock()
        config.excluded_dlcs = ["foo.dlc", "bar.dlc"]
        config._config.set.assert_called_with("ck2", "excluded_dlcs",
                                              "foo.dlc,bar.dlc")

        # Assert that the setter did not call ``refresh_config_file()`` with
        # ``refresh`` set to False.
        config.refresh_config_file.reset_mock()
        config.refresh = False
        config.excluded_dlcs = []
        self.assertFalse(config.refresh_config_file.called)

        # Assert that the getter called ``get`` of ``_config`` and got an empty
        # list back from an empty string.
        config._config.get.return_value = ""
        dlcs = config.excluded_dlcs
        self.assertEqual(dlcs, [])
        config._config.get.assert_called_with("ck2", "excluded_dlcs")
        config._config.get.reset_mock()

        # Assert that the getter called ``get`` of ``_config`` and got a filled
        # list back from a comma-separated string.
        config._config.get.return_value = "foo.dlc,bar.dlc"
        dlcs = config.excluded_dlcs
        self.assertEqual(dlcs, ["foo.dlc", "bar.dlc"])
        config._config.get.assert_called_with("ck2", "excluded_dlcs")

    @mock.patch("pyparadox.Config.Config.refresh_config_file", autospec=True)
    @mock.patch("pyparadox.Config.configparser", autospec=True)
    def test_mods(self, mock_configparser, mock_refresh_config_file):
        config = Config("ck2", mock.MagicMock())

        # Assert that the setter forwarded the right value to ``_config`` and
        # called ``refresh_config_file()``.
        config.mods = []
        config._config.set.assert_called_with("ck2", "mods", "")
        self.assertTrue(config.refresh_config_file.called)

        # Assert that the setter forwarded the right value to ``_config``.
        config.refresh_config_file.reset_mock()
        config.mods = ["foo.mod", "bar.mod"]
        config._config.set.assert_called_with("ck2", "mods",
                                              "foo.mod,bar.mod")

        # Assert that the setter did not call ``refresh_config_file()`` with
        # ``refresh`` set to False.
        config.refresh_config_file.reset_mock()
        config.refresh = False
        config.mods = []
        self.assertFalse(config.refresh_config_file.called)

        # Assert that the getter called ``get`` of ``_config`` and got an empty
        # list back from an empty string.
        config._config.get.return_value = ""
        mods = config.mods
        self.assertEqual(mods, [])
        config._config.get.assert_called_with("ck2", "mods")
        config._config.get.reset_mock()

        # Assert that the getter called ``get`` of ``_config`` and got a filled
        # list back from a comma-separated string.
        config._config.get.return_value = "foo.mod,bar.mod"
        mods = config.mods
        self.assertEqual(mods, ["foo.mod", "bar.mod"])
        config._config.get.assert_called_with("ck2", "mods")
