# -*- coding: utf-8 -*-

# PyParadox is a *nix launcher for Paradox games.
# Copyright (C) 2014  Ruben Bakker <rubykuby@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import sys
import unittest
from pyparadox.command import make_command, execute_command


class TestMakeCommand(unittest.TestCase):

    @classmethod
    def setUpClass(cls):
        cls.game_path = os.path.join(os.path.expanduser("~"),
                                     ".steam",
                                     "steam",
                                     "SteamApps",
                                     "common",
                                     "Crusader Kings II")
        cls.binary = "ck2"

    def test_make_command(self):
        excluded_dlcs = ["dlc001.dlc"]
        mods = ["better gender law mod.mod"]
        self.assertEqual(make_command(self.game_path, self.binary,
                                      excluded_dlcs, mods),
                         [os.path.join(self.game_path, self.binary),
                          "-skiplauncher",
                          "-exclude_dlc={}".format(
                              os.path.join("dlc", excluded_dlcs[0])),
                          "-mod={}".format(os.path.join("mod", mods[0]))])

    def test_make_command_no_plugins(self):
        excluded_dlcs = []
        mods = []
        self.assertEqual(make_command(self.game_path, self.binary,
                                      excluded_dlcs, mods),
                         [os.path.join(self.game_path, self.binary),
                          "-skiplauncher"])


class TestExecuteCommand(unittest.TestCase):

    def test_execute_command_string(self):
        process = execute_command(os.path.join(os.path.dirname(__file__),
                                               "fake-game-folder",
                                               "ck2"))
        self.assertTrue(process)

    def test_execute_command_list(self):
        process = execute_command([os.path.join(os.path.dirname(__file__),
                                                "fake-game-folder",
                                                "ck2"),
                                   "-argument"])
        self.assertTrue(process)

    def test_execute_command_no_permission(self):
        if sys.version < '3':
            error = OSError
        else:
            error = PermissionError  # noqa
        with self.assertRaises(error):
            execute_command(os.path.dirname(__file__))

    def test_execute_command_not_found(self):
        if sys.version < '3':
            error = OSError
        else:
            error = FileNotFoundError  # noqa
        with self.assertRaises(error):
            execute_command(os.path.join(os.path.dirname(__file__),
                                         "fake-game-folder",
                                         "notagame"))

if __name__ == "__main__":
    unittest.main()
