# -*- coding: utf-8 -*-

# PyParadox is a *nix launcher for Paradox games.
# Copyright (C) 2014  Ruben Bakker <rubykuby@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import unittest
try:
    from unittest import mock
except ImportError:
    import mock
from pyparadox.exceptions import QtImportError
try:
    from pyparadox.main import _create_app, _get_config_path
    from pyparadox.main import _create_config_file, main
    from pyparadox.Qt import QtGui
    qt = True
except QtImportError:
    qt = False


@unittest.skipIf(not qt, "This test requires a Qt module")
class TestMain(unittest.TestCase):

    def setUp(self):
        self.mock_open = mock.mock_open()

    def tearDown(self):
        del self.mock_open

    @mock.patch("pyparadox.main.ctypes")
    @mock.patch("pyparadox.main.sys", autospec=True)
    @mock.patch("pyparadox.main.QtGui", autospec=True)
    def test__create_app(self, mock_QtGui, mock_sys, mock_ctypes):
        app = mock.MagicMock(spec=mock_QtGui.QApplication)
        mock_QtGui.QApplication.return_value = app

        # Assert a QApplication is returned.
        res = _create_app()
        self.assertEqual(app, res)

        # Assert a QApplication is initiated.
        mock_QtGui.QApplication.assert_called_with(mock_sys.argv)

        # Assert the correct attributes are set.
        self.assertTrue(app.setApplicationVersion.called)
        self.assertTrue(app.setApplicationName.called)
        self.assertTrue(app.setOrganizationName.called)
        self.assertIsNone(app.process)

    @mock.patch("pyparadox.main.appdirs", autospec=True)
    def test__get_config_path(self, mock_appdirs):
        mock_appdirs.user_data_dir.return_value = "path"
        app = mock.MagicMock()

        res = _get_config_path(app)

        # Assert that the output was correct and that the appdirs were called
        # correctly.
        self.assertEqual(res, "path/pyparadox.conf")
        mock_appdirs.user_data_dir.assert_called_with(app.applicationName(),
                                                      app.organizationName())

    @mock.patch("pyparadox.main.os", autospec=True)
    def test__create_config_file(self, mock_os):
        with mock.patch("pyparadox.main.open", create=True,
                        new=self.mock_open) as mock_open:
            config_path = "path/pyparadox.conf"
            mock_os.path.dirname.return_value = "path"

            _create_config_file(config_path)

            # Assert that the base directory was made and that the config file
            # was opened to create it.
            mock_os.makedirs.assert_called_with("path")
            mock_open.assert_called_with(config_path, "w")

            # Assert that the program still works even if the directory already
            # existed.
            mock_os.makedirs.side_effect = OSError
            _create_config_file(config_path)

    @mock.patch("pyparadox.main.QtGui", autospec=True)
    @mock.patch("pyparadox.main.os", autospec=True)
    @mock.patch("pyparadox.main.sys", autospec=True)
    @mock.patch("pyparadox.main._create_config_file", autospec=True)
    @mock.patch("pyparadox.main._get_config_path", autospec=True)
    @mock.patch("pyparadox.main._create_app", autospec=True)
    @mock.patch("pyparadox.main.Config", autospec=True)
    @mock.patch("pyparadox.main.MainWindow", autospec=True)
    def test_main(self, mock_MainWindow, mock_Config, mock__create_app,
                  mock__get_config_path, mock__create_config_file, mock_sys,
                  mock_os, mock_QtGui):
        with mock.patch("pyparadox.main.open", create=True,
                        new=self.mock_open) as mock_open:
            mock_os.path.isfile.return_value = True
            mock__get_config_path.return_value = "path"

            app = mock.MagicMock(spec=QtGui.QApplication)
            app.applicationName.return_value = "foo"
            app.organizationName.return_value = "bar"
            app.process = None
            mock__create_app.return_value = app

            config = mock.MagicMock(spec=mock_Config)
            mock_Config.return_value = config

            form = mock.MagicMock(spec=mock_MainWindow)
            mock_MainWindow.return_value = form

            main("ck2")

            # Assert that an app was created.
            self.assertTrue(mock__create_app.called)

            # Assert that a config path was generated.
            mock__get_config_path.assert_called_with(app)

            # Assert that NO config file was generated.
            self.assertFalse(mock__create_config_file.called)

            # Assert that the config file was opened.
            mock_open.assert_called_with("path", "r+")

            # Assert that some necessary objects were instantited.
            mock_Config.assert_called_with("ck2", mock_open())
            mock_MainWindow.assert_called_with(config)

            # Assert that the MainWindow is shown.
            self.assertTrue(form.show.called)

            # Assert that the application loop was entered.
            self.assertTrue(app.exec_.called)

            # Assert that the application cleanly exitted.
            self.assertTrue(mock_sys.exit.called)

            mock_os.path.isfile.return_value = False
            app.process = mock.MagicMock()

            main("ck2")

            # Assert that a config file was created.
            mock__create_config_file.assert_called_with("path")

            # Assert that process handling is done.
            self.assertTrue(app.exit.called)
            self.assertTrue(app.process.wait.called)
