# -*- coding: utf-8 -*-

# PyParadox is a *nix launcher for Paradox games.
# Copyright (C) 2014  Ruben Bakker <rubykuby@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from abc import ABCMeta
import re
from .exceptions import NameNotFoundError


class Plugin(object):
    """Represents a single plugin (DLC/mod).

    :ivar filename: Basename of the plugin file.
    :ivar name: User-friendly description of the plugin.
    :ivar enabled: State of plugin.
    """

    __metaclass__ = ABCMeta

    def __init__(self, filename, name, enabled=True):
        """Create a Plugin instance.

        :param filename: Basename of the plugin file.
        :type filename: str.
        :param name: User-friendly description of the plugin.
        :type name: str.
        :param enabled: State of plugin.
        :type enabled: bool.
        """
        self.filename = filename
        self.name = name
        self.enabled = enabled

    def __lt__(self, other):
        return self.name < other.name

    def __le__(self, other):
        return not self.name > other.name

    def __eq__(self, other):
        return self.name == other.name

    def __ne__(self, other):
        return self.name != other.name

    def __gt__(self, other):
        return self.name > other.name

    def __ge__(self, other):
        return not self.name < other.name

    @staticmethod
    def get_plugin_name(fileobj):
        """Get a user-friendly name from a file.  Search the file for
        "name = X" and return X.

        :param fileobj: File object.
        :type fileobj: TextIOWrapper.
        :returns: Name found inside the file.
        :rtype: str.
        :raises: NameNotFoundError.
        """
        try:
            return re.search('^name[ \t]*=[ \t]*"(.*)"', fileobj.read(),
                             re.MULTILINE).group(1)
        except AttributeError:
            # TODO: Log instance of plugin name not found.  Find a way to
            #       present it to user.
            raise NameNotFoundError()
