# -*- coding: utf-8 -*-

# PyParadox is a *nix launcher for Paradox games.
# Copyright (C) 2014  Ruben Bakker <rubykuby@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
from ..Qt import QtCore, QtGui


class ConfigDlg(QtGui.QDialog):
    """Dialog interface that allows editing of the configuration file.

    :ivar game_path_lbl: Label that describes ``game_path_edit``.
    :type game_path_lbl: QLabel.
    :ivar game_path_edit: LinEdit that contains path to the game folder.
    :type game_path_edit: QLineEdit.
    :ivar game_path_btn: Button that opens a file dialog.
    :type game_path_btn: QPushButton.
    :ivar mod_path_lbl: Label that describes ``mod_path_edit``.
    :type mod_path_lbl: QLabel.
    :ivar mod_path_edit: LinEdit that contains path to the mod folder.
    :type mod_path_edit: QLineEdit.
    :ivar mod_path_btn: Button that opens a file dialog.
    :type mod_path_btn: QPushButton.
    :ivar binary_lbl: Label that describes ``binary_edit``.
    :type binary_lbl: QLabel.
    :ivar binary_edit: LineEdit that contains the name of the game binary.
    :type binary_edit: QLineEdit.
    :ivar binary_btn: Button that opens a file dialog.
    :type binary_btn: QPushButton.
    :ivar button_box: Widget that contains buttons that accept or reject.
    :type button_box: QDialogButtonBox.
    """

    def __init__(self, parent=None):
        super(ConfigDlg, self).__init__()
        self.init_ui()

        self.button_box.button(QtGui.QDialogButtonBox.Save).setFocus()

        self.button_box.button(QtGui.QDialogButtonBox.Cancel).clicked.connect(
            self.reject)
        self.button_box.button(QtGui.QDialogButtonBox.Save).clicked.connect(
            self.accept)
        self.game_path_btn.clicked.connect(self.fill_game_path_edit)
        self.mod_path_btn.clicked.connect(self.fill_mod_path_edit)
        self.binary_btn.clicked.connect(self.fill_binary_edit)

    def init_ui(self):
        game_path_layout = QtGui.QHBoxLayout()
        self.game_path_lbl = QtGui.QLabel(self.tr("&Game path:"))
        self.game_path_edit = QtGui.QLineEdit()
        self.game_path_btn = QtGui.QPushButton(self.tr("Choose folder..."))
        self.game_path_lbl.setBuddy(self.game_path_edit)
        game_path_layout.addWidget(self.game_path_lbl)
        game_path_layout.addWidget(self.game_path_edit)
        game_path_layout.addWidget(self.game_path_btn)

        mod_path_layout = QtGui.QHBoxLayout()
        self.mod_path_lbl = QtGui.QLabel(self.tr("&Mod path:"))
        self.mod_path_edit = QtGui.QLineEdit()
        self.mod_path_btn = QtGui.QPushButton(self.tr("Choose folder..."))
        self.mod_path_lbl.setBuddy(self.mod_path_edit)
        mod_path_layout.addWidget(self.mod_path_lbl)
        mod_path_layout.addWidget(self.mod_path_edit)
        mod_path_layout.addWidget(self.mod_path_btn)

        binary_layout = QtGui.QHBoxLayout()
        self.binary_lbl = QtGui.QLabel(self.tr("&Binary:"))
        self.binary_edit = QtGui.QLineEdit()
        self.binary_btn = QtGui.QPushButton(self.tr("Choose binary..."))
        self.binary_lbl.setBuddy(self.binary_edit)
        binary_layout.addWidget(self.binary_lbl)
        binary_layout.addWidget(self.binary_edit)
        binary_layout.addWidget(self.binary_btn)

        buttons_layout = QtGui.QHBoxLayout()
        self.button_box = QtGui.QDialogButtonBox(
            QtGui.QDialogButtonBox.Cancel | QtGui.QDialogButtonBox.Save)
        buttons_layout.addStretch()
        buttons_layout.addWidget(self.button_box)

        layout = QtGui.QVBoxLayout()
        layout.addLayout(game_path_layout)
        layout.addLayout(mod_path_layout)
        layout.addLayout(binary_layout)
        layout.addLayout(buttons_layout)

        self.setLayout(layout)
        self.setWindowTitle(self.tr("Configure"))

    @QtCore.Slot()
    def fill_game_path_edit(self):
        old_path = self.game_path_edit.text()
        if not os.path.isdir(old_path):
            old_path = os.path.expanduser("~")
        path = QtGui.QFileDialog.getExistingDirectory(self,
                                                      self.tr("Choose folder"),
                                                      old_path)
        if path:
            self.game_path_edit.setText(path)

    @QtCore.Slot()
    def fill_mod_path_edit(self):
        old_path = self.mod_path_edit.text()
        if not os.path.isdir(old_path):
            old_path = os.path.expanduser("~")
        path = QtGui.QFileDialog.getExistingDirectory(self,
                                                      self.tr("Choose folder"),
                                                      old_path)
        if path:
            self.mod_path_edit.setText(path)

    @QtCore.Slot()
    def fill_binary_edit(self):
        game_path = self.game_path_edit.text()
        if not os.path.isdir(game_path):
            game_path = os.path.expanduser("~")
        binary = QtGui.QFileDialog.getOpenFileName(self,
                                                   self.tr("Choose binary"),
                                                   game_path)
        # Some weird conversion because getOpenFileName() returns a tuple with
        # two strings, where the first string is a full path.
        binary = os.path.basename(binary[0])
        if binary:
            self.binary_edit.setText(binary)
