# -*- coding: utf-8 -*-

# PyParadox is a *nix launcher for Paradox games.
# Copyright (C) 2014  Ruben Bakker <rubykuby@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import pkg_resources
from ..command import make_command, execute_command
from ..find import find_dlcs, find_mods
from ..Dlc import Dlc
from ..Mod import Mod
from ..Qt import QtCore, QtGui
from .ConfigDlg import ConfigDlg
from .PluginListModel import PluginListModel
from .. import __name__ as package_name


class MainWindow(QtGui.QMainWindow):
    """Main window for the application.

    :ivar central_widget: Widget that contains all widgets.
    :type central_widget: QWidget.
    :ivar game_lbl: Label that contains a picture of the game.
    :type game_lbl: QLabel.
    :ivar mods_lbl: Label that describes ``mods_list``.
    :type mods_lbl: QLabel.
    :ivar mods_list: View for a list of checkable mods.
    :type mods_list: QListView.
    :ivar mods_model: Model that holds a list of mods.
    :type mods_model: PluginListModel.
    :ivar dlcs_lbl: Label that describes ``dlcs_list``.
    :type dlcs_lbl: QLabel.
    :ivar dlcs_model: Model that holds a list of DLCs.
    :type dlcs_model: PluginListModel.
    :ivar dlcs_list: View for a list of checkable DLCs.
    :ivar config_btn: Button that leads to a configuration dialog.
    :type config_btn: QPushButton.
    :ivar run_btn: Button that starts the game.
    :type run_btn: QPushButton.
    :ivar config: Object that stores application settings.
    :type config: Config.
    """

    def __init__(self, config, parent=None):
        """Initialise the MainWindow.__class__

        :param config: Object that stores application settings.
        :type config: Config.
        :param parent: Parent QWidget.
        :type parent: QWidget.
        """
        super(MainWindow, self).__init__(parent)
        self.config = config

        self.init_ui()

        self.mods_model = None
        self.dlcs_model = None

        self.fill_dlcs()
        self.fill_mods()

        self.run_btn.setFocus()

        self.config_btn.clicked.connect(self.show_config_dlg)
        self.run_btn.clicked.connect(self.run_game)

    def init_ui(self):
        """Initialise all (empty) UI elements."""
        self.central_widget = QtGui.QWidget(self)
        self.setCentralWidget(self.central_widget)

        picture_layout = QtGui.QHBoxLayout()
        picture_layout.addStretch()
        self.game_lbl = QtGui.QLabel()
        # Gets file path to respective game from pkg_resources.  Sets it as the
        # pixmap of a label.
        self.game_lbl.setPixmap(pkg_resources.resource_filename(package_name,
                                "resources/{}.png".format(self.config.game)))
        picture_layout.addWidget(self.game_lbl)
        picture_layout.addStretch()

        mods_layout = QtGui.QVBoxLayout()
        self.mods_lbl = QtGui.QLabel(self.tr("&Mods:"))
        self.mods_list = QtGui.QListView()
        self.mods_lbl.setBuddy(self.mods_list)
        mods_layout.addWidget(self.mods_lbl)
        mods_layout.addWidget(self.mods_list)

        dlcs_layout = QtGui.QVBoxLayout()
        self.dlcs_lbl = QtGui.QLabel(self.tr("&DLCs:"))
        self.dlcs_list = QtGui.QListView()
        self.dlcs_lbl.setBuddy(self.dlcs_list)
        dlcs_layout.addWidget(self.dlcs_lbl)
        dlcs_layout.addWidget(self.dlcs_list)

        plugins_layout = QtGui.QHBoxLayout()
        plugins_layout.addLayout(mods_layout)
        plugins_layout.addLayout(dlcs_layout)

        buttons_layout = QtGui.QHBoxLayout()
        self.config_btn = QtGui.QPushButton(self.tr("&Configure"))
        self.run_btn = QtGui.QPushButton(self.tr("&Run"))
        buttons_layout.addStretch()
        buttons_layout.addWidget(self.config_btn)
        buttons_layout.addWidget(self.run_btn)

        layout = QtGui.QVBoxLayout()
        layout.addLayout(picture_layout)
        layout.addLayout(plugins_layout)
        layout.addLayout(buttons_layout)

        self.central_widget.setLayout(layout)
        self.setWindowTitle(QtGui.QApplication.applicationName())

    def fill_dlcs(self):
        """Fill the DLC ListView with found DLCs from the game folder and
        unselect DLCs based on PyParadox's configuration file.
        """
        dlcs = []
        # Find all DLCs inside the DLC folder.
        for i, dlc_path in enumerate(find_dlcs(os.path.join(
                self.config.game_path, "dlc"))):
            filename = os.path.basename(dlc_path)
            # Add `Dlc` objects to the `dlcs` list.
            with open(dlc_path) as fileobj:
                try:
                    plugin_name = Dlc.get_plugin_name(fileobj)
                except:
                    plugin_name = filename
                dlcs.append(Dlc(filename, plugin_name))

            # Disable all deselected DLCs.
            if dlcs[i].filename in self.config.excluded_dlcs:
                dlcs[i].enabled = False

        # Make a model and assign it to the View.
        self.dlcs_model = PluginListModel(dlcs)
        self.dlcs_list.setModel(self.dlcs_model)

    def fill_mods(self):
        """Fill the mods ListView with found mods from the mod folder and
        unselect mods based on PyParadox's configuration file.
        """
        mods = []
        # Find all mods inside the mod folder.
        for i, mod_path in enumerate(find_mods(self.config.mod_path)):
            filename = os.path.basename(mod_path)
            # Add `Mod` objects to the `mods` list.
            with open(mod_path) as fileobj:
                try:
                    plugin_name = Dlc.get_plugin_name(fileobj)
                except:
                    plugin_name = filename
                mods.append(Mod(filename, plugin_name))

            # Disable all unselected mods.
            if mods[i].filename not in self.config.mods:
                mods[i].enabled = False

        # Make a model and assign it to the view.
        self.mods_model = PluginListModel(mods)
        self.mods_list.setModel(self.mods_model)

    @QtCore.Slot()
    def show_config_dlg(self):
        """Create a dumb ``ConfigDlg``, write data to it to set it up, then
        retrieve its data and push it to ``config`` if it shut down
        affirmatively.
        """
        config_dlg = ConfigDlg()
        config_dlg.game_path_edit.setText(self.config.game_path)
        config_dlg.mod_path_edit.setText(self.config.mod_path)
        config_dlg.binary_edit.setText(self.config.binary)
        if config_dlg.exec_():
            self.config.game_path = config_dlg.game_path_edit.text()
            self.config.mod_path = config_dlg.mod_path_edit.text()
            self.config.binary = config_dlg.binary_edit.text()
            self.fill_dlcs()
            self.fill_mods()

    @QtCore.Slot()
    def run_game(self):
        """Extract data from ``mods_model`` and ``dlcs_model`` and write them
        to ``config``.  Use all data from ``config`` to create a command and
        try to execute it.  Assign the resulting process to the running
        ``QApplication`` and shut down.  If the process couldn't start, give
        a warning message.
        """
        excluded_dlcs = [dlc.filename for dlc in self.dlcs_model.plugins if not
                         dlc.enabled]
        mods = [mod.filename for mod in self.mods_model.plugins if mod.enabled]

        self.config.excluded_dlcs = excluded_dlcs
        self.config.mods = mods

        command = make_command(self.config.game_path, self.config.binary,
                               self.config.excluded_dlcs, self.config.mods)
        try:
            # Set process to a variable inside the QApplication.  This
            # variable can be accessed after the QMainWindow quits.
            QtGui.QApplication.process = execute_command(command)
            self.close()
        except OSError:
            msg_box = QtGui.QMessageBox(QtGui.QMessageBox.Warning,
                                        self.tr("Error"),
                                        self.tr("Game could not launch."))
            msg_box.setDetailedText(" ".join(command))
            msg_box.setStandardButtons(QtGui.QMessageBox.Ok)
            msg_box.setDefaultButton(QtGui.QMessageBox.Ok)
            msg_box.exec_()
