#!/usr/bin/env python
# -*- coding: utf-8 -*-

# PyParadox is a *nix launcher for Paradox games.
# Copyright (C) 2014  Ruben Bakker <rubykuby@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import sys
import ctypes
import pkg_resources
import appdirs
from . import __version__
from .Config import Config
from .Qt import QtGui
from .gui.MainWindow import MainWindow
from . import __name__ as package_name


def _create_app():
    """Create a QApplication instance with some variables set."""
    app = QtGui.QApplication(sys.argv)
    app.setApplicationVersion(__version__)
    app.setApplicationName(app.tr("PyParadox"))
    app.setOrganizationName(app.tr("PyParadox"))
    # The ``Popen`` process will be assigned to app.process as if it were a
    # global variable.  It's a bit of a hack, and there is probably a better
    # way to retrieve the process from ``main()``.
    app.process = None

    if sys.platform.startswith("win"):
        ctypes.windll.shell32.SetCurrentProcessExplicitAppUserModelID(
            app.applicationName())
    return app


def _get_config_path(app):
    """Get the path of the location of the configuration file.  The location
    differs per platform.

    :param app: The application to retrieve strings from
    :type app: QApplication.
    :returns: Absolute path to configuration file.
    :rtype: str.
    """
    return os.path.join(appdirs.user_data_dir(app.applicationName(),
                                              app.organizationName()),
                        "pyparadox.conf")


def _create_config_file(config_path):
    """Create an empty file in the assigned location, making directories as
    necessary.

    :param config_path: Absolute path towards the configuration file.
    :type confug_path: str.
    """
    try:
        # ``mkdir -p`` the directory that will contain the file.
        os.makedirs(os.path.dirname(config_path))
    except OSError:
        pass
    # Make an empty file.  If one already exists, it will be overwritten.
    open(config_path, "w").close()


def main(game):
    """Main entry for the program.  Create a MainWindow and start the Qt loop.
    When the application is shut down, test whether ``app.process`` exists as a
    process, and wait for the process to finish before cleanly exiting.

    :param game: Short identifier of the game.
    :type game: str.
    """

    app = _create_app()

    config_path = _get_config_path(app)
    if not os.path.isfile(config_path):
        _create_config_file(config_path)

    with open(config_path, "r+") as fileobj:
        config = Config(game, fileobj)

        form = MainWindow(config)
        form.setWindowIcon(QtGui.QIcon(pkg_resources.resource_filename(
                                       package_name, "resources/paradox.png")))
        form.show()

        app.exec_()

    if app.process is not None:
        process = app.process
        app.exit()
        process.wait()

    sys.exit()
